/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

#include <stdio.h>
#include "sim.h"	/* this must always be here */

/* constant state names */
#define STATIC 0
#define CHANGE 1

/* unit and site function type declarations */
int UFcolor();		/* defined here */

/*
   build constructs the network; its one argument is the name of a data file;
   the data file should have the following format:

       <number of regions>
       <region number> <region number>
	   .
	   .
       <region number> <region number>

  the first line is the number of regions;  the remaining lines specify
  pairs of regions which are adjacent.
*/
build(argc,argv)
    int argc;		/* number of arguments */
    char *argv[];	/* array of argument strings */
{
    int count,i,reg1,reg2,readstatus;
    FILE *infile;

    if(argc != 2){
	printf("Usage: build <desc file name>\n");
	return;
    }

    infile = fopen(argv[1],"r");	/* open data file */
    if(infile == NULL){
	printf("build: could not open %s\n",argv[1]);
	return;
    }

    fscanf(infile,"%d",&count);
    AllocateUnits(count*4);	/* make enough units for regions X 4 colors */

    DeclareSet("change");		/* declare a set name */
    DeclareState("Static",STATIC);	/* state name declaration */
    DeclareState("Change",CHANGE);	/* state name declaration */

    /* make the regions */
    for(i = 0;i < count;i++) region();

    /* make the borders */
    for(i=0;;i++){	/* infinite loop */
	readstatus = fscanf(infile,"%d %d",&reg1,&reg2);/* read two regions */
	if(readstatus != 2) break;	/* assume end of data - leave loop */
	border(reg1,reg2);		/* make the connection */
    }
    printf("%d regions with %d borders\n",count,i);
}

/*
   Make a region.  Four units are made - one for each color.
   The entire region is named as an array, with the color macros
   below specifying the appropriate indices.  They inhibit each other strongly.
*/

#define RED 0
#define BLUE 1
#define GREEN 2
#define WHITE 3
static char *colornames[] = {"red","blue","green","white"};

static int region()
{
    static int regionnum = 0;	/* which region - static means value remains
				   between calls; it is initialized at startup
				   to 0 */
    int i,j,first;
    char buf[15];

    /* make 4 consecutive units, each for a different color */
    first = makecolor(RED); /* remember index of first for name declaration*/
    makecolor(BLUE);
    makecolor(GREEN);
    makecolor(WHITE);

    /* name these as a single vector */
    (void) sprintf(buf,"region%1d",regionnum);	/* buf contains name */
    NameUnit(buf,VECTOR,first,4); 	/* vector of length 4 */

    /* make them inhibit each other */
    for(i = 0;i < 4;i++)
	for(j = i+1;j < 4;j++)
	    mux(first+i,first+j,-1000);
    return regionnum++;		/* returns region number */
}

/* mapunit takes a region number and color and returns that unit's index */
mapunit(region,color)
    int region,color;
{
    char buf[15];

    (void) sprintf(buf,"region%1d",region);
    return NameToInd(buf,color);
}

/* make a single unit representing a region color; return index of that unit */
static int makecolor(type)
    int  type;		/* a string containing color name */
{
    int index;

    index = MakeUnit(colornames[type],UFcolor,0,0,0,0,STATIC,STATIC);
    AddSite(index,"inhibit",SFweightedsum);
    SetFlag(index,NO_LINK_FUNC_FLAG);	/* weights don't change */
    return index;
}

/* unit1 and unit two are mutually exclusive */
static mux(unit1,unit2,weight)
    int unit1,unit2,weight;
{
    MakeLink(unit1,unit2,"inhibit",weight,0,NULL);
    MakeLink(unit2,unit1,"inhibit",weight,0,NULL);
}

/* make a border between region1 and region2 */
static border(region1,region2)
    int region1,region2;
{
    int i;

    mux(mapunit(region1,BLUE),mapunit(region2,BLUE),-100);
    mux(mapunit(region1,RED),mapunit(region2,RED),-100);
    mux(mapunit(region1,GREEN),mapunit(region2,GREEN),-100);
    mux(mapunit(region1,WHITE),mapunit(region2,WHITE),-100);
}

/*
   dice(inhibit) has value 1 with probability: (1000+inhibit)/1999;
   Note that if inhibit is <= -1000, this probability is zero.
*/
#define dice(inhibit) ((random()%1999) < (1000 + inhibit))

UFcolor(up)
    Unit *up;
{
    int inhibit;
    int oldpot;

    oldpot = up->potential;	/* remember old potential */
    inhibit = SiteValue("inhibit",up->sites);
    if(inhibit >= 0 || dice(inhibit)){	/* unit on */
	up->potential = 1000;
	up->output = 1000;
    }
    else{				/* unit off */
	up->potential = 0;
	up->output = 0;
    }

    /* change state and set membership if necessary */
    if(oldpot != up->potential){
	AddSet("change",UnitIndex(up));
	up->state = CHANGE;
    }
    else{
	RemSet("change",UnitIndex(up));
	up->state = STATIC;
    }
}
