static char rcsid[] = "$Header: nametable.c,v 1.3 87/11/03 17:13:04 goddard Exp $";
/*----------------------------------------------------------------------------

  The Rochester Connectionist Simulator.  Copyright Univesity of Rochester.
  Author: Nigel Goddard
  Date: May 1 1987

----------------------------------------------------------------------------*/

#include        <stdio.h>
#include      "Stupido:rochester:include:uniproc.h"

static NameDesc *name_table[HASH_SIZE]; /* Hash Table */
static int      no_names = 0;   /* actual number of names (for Units) */
static int      no_symbs = 0;   /* actual number of symbols in table */

/*********************************************************

 symbol storage

*********************************************************/
/*---------------------------------------------------------------------------
   puts a string in the hash table and returns a pointer to the entry;
   other entry fields could be modified after return.
----------------------------------------------------------------------------*/

static NameDesc * StoreSymbol(name)
char    *name;
{
        NameDesc        *ntmp;
        int     hash;
        char    *n;
        
        n = name;
        for (hash = 0; *n != '\0'; hash += *n++)
                ;
        hash %= HASH_SIZE;

        for (ntmp = name_table[hash]; ntmp != NULL; ntmp = ntmp->next)
          if(!strcmp(ntmp->name, name))
            return ntmp;        /* Already there */

        ntmp = (NameDesc *) si_malloc(sizeof(NameDesc));
        ntmp->name = (char *)si_malloc(strlen(name) + 1);
        strcpy(ntmp->name, name);
        ntmp->next = name_table[hash];
        ntmp->type = STRING_SYM;/* to distinguish from already existing */
        name_table[hash] = ntmp;
        ++no_symbs;

        return ntmp;
}

/*---------------------------------------------------------------------------
   returns a pointer to the entry for name if it exists, otherwise returns
   NULL.
----------------------------------------------------------------------------*/

static NameDesc * FindSymbol(name)
     char * name;

{
        NameDesc        *ntmp;
        int     hash;
        char    *n;
        
        if(!name)
                return (NameDesc *) NULL;
        n = name;
        for (hash = 0; *n != '\0'; hash += *n++)
                ;
        hash %= HASH_SIZE;

        for (ntmp = name_table[hash]; ntmp != NULL; ntmp = ntmp->next)
          if(!strcmp(ntmp->name, name))
            return ntmp;        /* Already there */
        return (NameDesc *) NULL;
}

/**********************************************
 
 read and write the network file

**********************************************/
/*---------------------------------------------------------------------------
  error check for use when setting up the unit to name mapping, below. 
----------------------------------------------------------------------------*/

static int errchk(index,limit,msg,name)
     int index, limit;
     char * msg,name;

{
  if (index >= limit)
    {
      LOGfprintf(stderr,
                 "error: attempting to load %s %s with index %d, out of range\n",
                 msg,name,index);
      return TRUE;
    }
  return FALSE;
}

/*---------------------------------------------------------------------------
  Called by NetLoad to set up the unit to name mapping when loading the
  network from a saved network file.  Also sets up Set and State mapping.
----------------------------------------------------------------------------*/

int si_LoadUnitNames()

  {
    int i,j,nc,no_sets,no_states;
    NameDesc * nt;

    for (i = 0; i < LastSet; i++)
      SetNames[i] = NULL;
    for (i = 0; i < NoStates; i++)
      StateNames[i] = NULL;
    for(nc = i = no_sets = no_states = 0; i < HASH_SIZE; i++) 
        for(nt = name_table[i]; nt != NULL; nt = nt->next)
          {
            if(nc >= no_symbs)
              {
                fprintf(stderr,
                        "error: wrong number of Names in name table!\n");
                return FALSE;
              }
            switch (nt->type)
              {
              case SCALAR:
                if (errchk(nt->index,NoUnits,"unit",nt->name)) return FALSE;
                UnitList[nt->index].name = nt->name;
                break;
              case VECTOR:
                if (errchk(nt->index,NoUnits,"unit",nt->name)) return FALSE;
                for (j = nt->index; j < (nt->index+nt->size); j++)
                  UnitList[j].name = nt->name;
                break;
              case ARRAY:
                if (errchk(nt->index,NoUnits,"unit",nt->name)) return FALSE;
                for (j = nt->index; j < nt->index+(nt->size*nt->length); j++)
                  UnitList[j].name = nt->name;
                break;
              case SET_SYM:
                if (errchk(nt->index,LastSet,"set",nt->name)) return FALSE;
                SetNames[nt->index] = nt->name;
                no_sets++;
                break;
              case STATE_SYM:
                if (errchk(nt->index,NoStates,"state",nt->name)) return FALSE;
                StateNames[nt->index] = nt->name;
                no_states++;
                break;
              case STRING_SYM:
              case FUNC_SYM:
              case TYPE_SYM:
              case SITE_SYM:
              default:
                break;
              }
          }
    if (NoSets != no_sets || StateCount != no_states)
      {
        LOGfprintf(stderr,
                   "error: wrong number of states or sets in nametable\n");
        return FALSE;
      }
    return TRUE;
  }    

/*---------------------------------------------------------------------------
  Called from NetSave to find out the number of strings to be saved
----------------------------------------------------------------------------*/

int si_GetStringCount()

  {
    register int nc;
    register int i;
    register NameDesc * nt;
    for(nc = i = 0; i < HASH_SIZE; i++) 
        for(nt = name_table[i]; nt != NULL; nt = nt->next)
          {
            if(nc >= no_symbs)
              {
                LOGfprintf(stderr, "wrong number of Names in name table!\n");
                return;
              }
            if (nt->type == SITE_SYM || nt->type == FUNC_SYM ||
                nt->type == TYPE_SYM)
               nc++;
          }
    return nc;
  }
    
/*---------------------------------------------------------------------------
  Called from NetSave to save the name table to file fp.
----------------------------------------------------------------------------*/

si_SaveNameTable(fp)
  FILE * fp;

  {
    register int sc;
    register int nc;
    register int i;
    register NameDesc * nt;

    printf("saving name table ...\n");
    fprintf(fp, "%d\n", no_symbs);
    for(sc = nc = i = 0; i < HASH_SIZE; i++) 
        for(nt = name_table[i]; nt != NULL; nt = nt->next)
          {
            if(nc >= no_symbs)
              {
                LOGfprintf(stderr, "wrong number of Names in name table!\n");
                return;
              }
            fprintf(fp, "%s %d %d %d %d\n",
                    nt->name,nt->type,nt->index,nt->size,nt->length);
            if (nt->type == SITE_SYM || nt->type == FUNC_SYM ||
                nt->type == TYPE_SYM) /* reset by si_ResetLengthField */
              nt->length = sc++; /* dump file uses this # instead of string */
            nc++;
          }
  }

si_ResetLengthField()

  {
    register int i;
    register NameDesc * nt;

    for (i = 0; i < HASH_SIZE; i++) 
      for(nt = name_table[i]; nt != NULL; nt = nt->next)
        if (nt->type == SITE_SYM || nt->type == FUNC_SYM ||
            nt->type == TYPE_SYM) /* set by si_SaveNameTable */
          nt->length = 0; /* dump file uses this field temporarily */
  }

/*---------------------------------------------------------------------------
  Recovers all unused entries in name table.
----------------------------------------------------------------------------*/
     
ScavengeNames()

{
  register int i;
  register NameDesc **ntpp;
  register NameDesc *nt;

  for (i = 0; i < HASH_SIZE; i++)
    {
      for (ntpp = &(name_table[i]);
           (*ntpp) != NULL;)
        if ((*ntpp)->type == STRING_SYM)
          {                     /* unused, so free */
            free((*ntpp)->name);
            (*ntpp) = (nt = (*ntpp))->next;
            free(nt);
            no_symbs--;
          }
        else                    /* move on to next */
          ntpp = &((*ntpp)->next);
    }
}

/*---------------------------------------------------------------------------
  Deletes all entries in name table.  If <all> is FALSE, then function names
  and state names are not deleted from the name table.
----------------------------------------------------------------------------*/

si_CleanNameTable(all)
        int all;

  {
    register int i;
    register NameDesc * nt;
    register NameDesc * ft;

    for(i = 0; i < HASH_SIZE; i++)
      {
        for (nt = name_table[i];
             nt != NULL && nt->type != FUNC_SYM && nt->type != DATA_SYM &&
             nt->type != CODE_SYM && nt->type != STATE_SYM;
             name_table[i] = nt)
          {                             /* free until first function */
            free (nt->name);
            nt = (ft = nt)->next;
            free (ft);
            no_symbs--;
          }
        if (nt != NULL)                 /* not at end yet */
          for (ft = nt->next; ft != NULL; ft = nt->next)
              if (ft->type != FUNC_SYM && ft->type != DATA_SYM &&
                  ft->type != CODE_SYM && ft->type != STATE_SYM)
                                /* ft is current, nt previous */
                {               /* free up ft */
                  free (ft->name);
                  nt->next = ft->next;
                  free (ft);
                  no_symbs--;
                }
              else                      /* a function,variable,codefile,state*/
                if (all)                /* delete anyway */
                  {                     /* free up ft */
                    free (ft->name);
                    nt->next = ft->next;
                    free (ft);
                    no_symbs--;
                  }
                else
                  nt = ft;              /* save function entry */
        if (all && nt != NULL)          /* delete final entry */
          {
            free (nt->name);
            free (nt);
            name_table[i] = NULL;
            no_symbs--;
          }

      }
  }

/*---------------------------------------------------------------------------
  utility to add or alter a name in the name table.  Overwrite is boolean.
----------------------------------------------------------------------------*/

static NameDesc * entername(name,type,index,size,length,overwrite)
     char *name;
     int type,index,size,length,overwrite;

{
  NameDesc * nptr;

  Guard();
  nptr = StoreSymbol(name);
  if (nptr != NULL)
    if (!overwrite && nptr->type != STRING_SYM) /* no overwrite,not free */
      if (strcmp(nptr->name,name) || nptr->type != type || /* redeclaration?*/
          nptr->index != index || nptr->size != size || nptr->length != length)
        {
          fprintf(stderr,
                  "Attempt to overwrite name table entry for %s failed\n",
                  name);
          return NULL;
        }
      else                      /* just redeclaration, all fields same */
        ;                       /* nptr is the right pointer */
    else
      {
        if (nptr->type==SCALAR || nptr->type==VECTOR || nptr->type==ARRAY)
          no_names++;
        nptr->type = type;
        nptr->index = index;
        nptr->size = size;
        nptr->length = length;
      }
  Release();
  return (nptr);
}

/*---------------------------------------------------------------------------
  EnterName will enter the { name} in the table.  { type} is used
  by the simulator to determine what type of name it is.  Type numbers 0
  through 8 are used by the simulator, and numbers 9 through 99 are
  reserved to the simulator.  Libraries should use numbers 100 through
  999, and user code numbers 1000 and up.  The three { data} fields
  are simply entered in the table.  If the name is successfully entered,
  the function returns the pointer to the stored name character string.
  Care should be taken not corrupt this character string.  EnterName
  will fail and return NULL if the name is already in the symbol table,
  unless all fields match those already in the table.
----------------------------------------------------------------------------*/

char * EnterName(name,type,index,size,length)
     char *name;
     int type,index,size,length;

{
  NameDesc * nptr;

  if ((nptr = entername(name,type,index,size,length,FALSE)) == NULL)
    return NULL;
  else
    return nptr->name;
}

/*---------------------------------------------------------------------------
  AlterName is just like EnterName, except that it never fails; if the name
  is already in the table, it is simply overwritten.  This function
  should be used with extreme care.
----------------------------------------------------------------------------*/

char * AlterName(name,type,index,size,length)
     char *name;
     int type,index,size,length;

{
  NameDesc * nptr;

  if ((nptr = entername(name,type,index,size,length,TRUE)) == NULL)
    return NULL;
  else
    return nptr->name;
}

/*---------------------------------------------------------------------------
  FindName looks up the name in the name table, and fills in the
  descriptor with the table entry contents.  It returns a pointer to the
  descriptor if the name was found, and NULL if not.
----------------------------------------------------------------------------*/

NameDesc * FindName(name,nte)
     char *name;
     NameDesc * nte;

{
  NameDesc * nptr;

  Guard();
  nptr = FindSymbol(name);
  Release();
  if (nptr != NULL)
    {
      *nte = *nptr;
      nte->next = NULL;                           /* can't follow chain */
      return (nte);
    }
  return (NULL);
}
