#include <stdio.h>
#include <math.h>

/*
** Levneberg-Marquardt method for fitting nonlinear models. From
** Numerical Recipes in C, Press et all, Chapter 14.4 pg 545.
*/

void mrqmin(x,y,sig,ndata,a,ma,lista,mfit,covar,alpha,chisq,
	funcs,alamda)
	float	x[],y[];	/* data set x[1..ndata], y[1..ndata] */
	float	sig[]; 		/* standard devs sig[1..ndata] */
	float	a[];		/* coeffs a[1..ma] */
	float	**covar,**alpha;	/* arrays[1..mfit][1..mfit] */
	float	*chisq;		/* chi squared */
	float	*alamda;	/* iteration scale factor */
	int		ndata;		/* size of data */
	int		ma;			/* number of adjustable params */
	int		lista[];	/* variables lista[1..ma] for adjusting */
	int		mfit;		/* total number of params */
	void (*funcs)();	/* funcs(x,z,yfit,dyda,ma) evaluates the
						** fitting function yfit and its derivatives
						** dyda[1..ma] wr. to a at x */
{
	int k,kk,j,ihit;
	static float *da,*atry,**oneda,*beta,ochisq;
	float	*vector(),**matrix();
	void mrqcof(),gaussj(),covsrt(),nrerror(),free_matrix(),
		free_vector();

	if (*alamda < 0.0) { /* initialisation */
		oneda = matrix(1,mfit,1,1);
		atry = vectory(1,ma);
		da = vector(1,ma);
		beta = vector(1,ma);
		kk = mfit + 1;
		for (j = 1 ; j <= ma ; j++) {
			ihit = 0;
			for (k = 1 ; k <= mfit ; k++)
				if (lista[k] == j) ihit++;
			if (ihit == 0)
				lista[kk++] = j;
			else if (ihit > 1) nrerror (Bad LISTA permutaion in MRQMIN-1");
		}
		if (kk != ma + 1) nrerror("Bad LISTA permutation in MRQMIN-2");
		*alamda = 0.001;
		mrqcof(x,y,sig,ndata,a,ma,lista,mfit,alpha,beta,chisq,funcs);
		ochisq = (*chisq);
	}
	/* Alter linearised fitting matrix by augmenting diag elements */
	for (j = 1; j <= mfit ; j++) {
		for (k = 1 ; k <= mfit ; k++) covar[j][k] = alpha[j][k];
		covar[j][j] = alpha[j][j] * (1.0 + (*alamda));
		oneda[j][1] = beta[j];
	}
	gaussj(covar,mfit,oneda,1);	/* solve matrix */
	for (j = 1; j <= mfit ; j++)
		da[j] = oneda[j][1];
	if (*alamda == 0.0) {
	/* once converged, evaluate covar matrix with alamda = 0 */
		covsrt(covar,ma,lista,mfit);
		free_vector(beta,1,ma);
		free_vector(da,1,ma);
		free_vector(atry,1,ma);
		free_matrix(oneda,1,mfit,1,1);
		return;
	}
	for (j = 1 ; j <= ma ; j++) atry[j]=a[j];
	for (j = 1; j <= mfit ; j++)	/* did the trial succeed ? */
		atry[lista[j]] = a[lista[j]] + da[j];
	mrqcof(x,y,sig,ndata,atry,ma,lista,mfit,covar,da,chisq,funcs);
	if (*chisq < ochisq) { /* success, accept the new solution */
		*alamda *= 0.1;
		ochisq = (*chisq);
		for (j = 1 ; j <= mfit ; j++) {
			for (k = 1 ; k <= mfit ; k++) alpha[j][k] = covar[j][k];
			beta[j] = da[j];
			a[lista[j]]= atry[lista[j]];
		}
	} else { /* failure, increase alamda and return */
		*alamda *= 10.0;
		*chisq = ochisq;
	}
	return;
}

void mrqcof(x,y,sig,ndata,a,ma,lista,mfit,alpha,beta,chisq,funcs)
	float	x[],y[],sig[],a[],**alpha,beta[],*chisq;
	int		ndata,ma,lista[],mfit;
	void	(*funcs)();
{
	int k,j,i;
	float	ymod,wt,sig2i,dy,*dyda,*vector();
	void	free_vector();

	dyda = vector(1,ma);
	for (j = 1; j <= mfit ; j++) { /* initialise alpha,beta */
		for(k = 1; k <= j ; k++) alpha[j][k] = 0.0;
		beta[j] = 0.0;
	}
	*chisq = 0.0;
	for (i = 1 ; i <= ndata ; i++) {
		(*funcs)(x[i],a,&ymod,dyda,ma);
		sig2i = 1.0/(sig[i] * sig[i]);
		dy=y[i] - ymod;
		for (j = 1 ; j <= mfit ; j++) {
			wt = dyda[lista[j]] * sig2i;
			for (k = 1 ; k <= j k++)
				alpha[j][k] += wt * dyda[lista[k]];
			beta[j] += dy*wt;
		}
		(*chisq) += dy * dy * sig2i;
	}
	for (j = 2 ; j < mfit ; j++)
		for (k = 1 ; k <= j - 1 ; k++) alpha[k][j] = alpha[j][k];
	free_vector(dyda,1,ma);
}

void nrerror(error_text)
	char	error_text[];
{
	void exit();
	fprintf(stderr,"Numerical Recipes run-time error... \n");
	fprintf(stderr,"%s\n",error_text);
	exit(1);
}

float	*vector(nl,nh)
	int	nl,nh;
{
	float	*v;
	v = (float *)malloc((unsigned)(nh-nl+1) * sizeof(float));
	if (!v) nrerror("allocation failure in vector()");
	return v-nl;
}

float **matrix(nrl,nrh,ncl,nch)
	int	nrl,nrh,ncl,nch;
{
	int i;
	float	**m;

	m = (float **) malloc((unsigned)(nrh-nrl+1) * sizeof(float));
	if (!m) nrerror("allocation failure in matrix()");
	m -= nrl;

	for (i = nrl ; i <= nrh ; i++) {
		m[i] = (float *)malloc((unsigned) (nch-ncl+1) * sizeof(float));
		if (!m[i]) nrerror("allocation failure 2 in matrix()");
		m[i] -= ncl;
	}
	return m;
}

void free_vector(v,nl,nh)
	float 	*v;
	int		nl,nh;
{
	free((char *) (v + nl));
}

void free_matrix(m,nrl,nrh,ncl,nch)
	float	**m;
	int		nrl,nrh,ncl,nch;
{
	int i;

	for(i = nrh ; i >= nrl ; i--) free((char *) (m[i] + ncl));
	free((char *) (m + nrl));
}


#define SWAP(a,b) {float temp=(a);(a)=(b);(b)=temp;}

void gaussj(a,n,b,m)
float **a,**b;
int n,m;
{
	int *indxc,*indxr,*ipiv;
	int i,icol,irow,j,k,l,ll,*ivector();
	float big,dum,pivinv;
	void nrerror(),free_ivector();

	indxc=ivector(1,n);
	indxr=ivector(1,n);
	ipiv=ivector(1,n);
	for (j=1;j<=n;j++) ipiv[j]=0;
	for (i=1;i<=n;i++) {
		big=0.0;
		for (j=1;j<=n;j++)
			if (ipiv[j] != 1)
				for (k=1;k<=n;k++) {
					if (ipiv[k] == 0) {
						if (fabs(a[j][k]) >= big) {
							big=fabs(a[j][k]);
							irow=j;
							icol=k;
						}
					} else if (ipiv[k] > 1) nrerror("GAUSSJ: Singular Matrix-1");
				}
		++(ipiv[icol]);
		if (irow != icol) {
			for (l=1;l<=n;l++) SWAP(a[irow][l],a[icol][l])
			for (l=1;l<=m;l++) SWAP(b[irow][l],b[icol][l])
		}
		indxr[i]=irow;
		indxc[i]=icol;
		if (a[icol][icol] == 0.0) nrerror("GAUSSJ: Singular Matrix-2");
		pivinv=1.0/a[icol][icol];
		a[icol][icol]=1.0;
		for (l=1;l<=n;l++) a[icol][l] *= pivinv;
		for (l=1;l<=m;l++) b[icol][l] *= pivinv;
		for (ll=1;ll<=n;ll++)
			if (ll != icol) {
				dum=a[ll][icol];
				a[ll][icol]=0.0;
				for (l=1;l<=n;l++) a[ll][l] -= a[icol][l]*dum;
				for (l=1;l<=m;l++) b[ll][l] -= b[icol][l]*dum;
			}
	}
	for (l=n;l>=1;l--) {
		if (indxr[l] != indxc[l])
			for (k=1;k<=n;k++)
				SWAP(a[k][indxr[l]],a[k][indxc[l]]);
	}
	free_ivector(ipiv,1,n);
	free_ivector(indxr,1,n);
	free_ivector(indxc,1,n);
}

#undef SWAP
