#include <stdio.h>
#include <setjmp.h>
#include "y.tab.h"
#include "parse.h"
#include "symtab.h"


#define	FALSE	0
#define	TRUE	(!FALSE)


extern char *strsave();

typedef struct _script_info
  {
    int		si_lineno;
    char	*si_script;
  } ScriptInfo;

/*
** eval.c
**
**	Parse tree evaluation module.  This source file uses defines
**	in the yacc program.  For this reason eval.c should be included
**	in the program section of the yacc source file.
*/


extern int errno;
extern char* TokenStr();
extern Symtab GlobalSymbols;
extern char *CurrentScriptName();

typedef struct _localvars
  {
    int			lv_nargs;
    int			lv_nlocals;
    Result		*lv_vars;
    struct _localvars	*lv_prev;
    char		*lv_script;
    char		*lv_function;
    int			lv_lineno;
  } LocalVars;
LocalVars	*CurLocals = NULL;


/*
** The following globals are used to implement the RETURN statement.
** When a function or a command is executed, setjmp is called to
** mark the start of function execution.  When RETURN is executed
** the return value (if present) is placed in ReturnResult and we
** longjmp back to the start of function node.
*/

jmp_buf		BreakBuf;
jmp_buf		ReturnBuf;
Result		ReturnResult;
static int	InLoop = 0;
static ScriptInfo ScriptLoc = { 0, "<stdin>" };



ScriptInfo *MakeScriptInfo()

{	/* MakeScriptInfo --- Return current script anem and line number */

	ScriptInfo	*si;

	si = (ScriptInfo *) malloc(sizeof(ScriptInfo));
	if (si == NULL)
	  {
	    yyerror("MakeScriptInfo: out of mem");
	    /* No Return */
	  }

	si->si_lineno = CurrentScriptLine();
	si->si_script = CurrentScriptName();
	if (si->si_script == NULL)
	    si->si_script = "<stdin>";
	else
	    si->si_script = (char *) strsave(si->si_script);

	return(si);

}	/* MakeScriptInfo */



/*
** The following functions are used by the simulator to retrieve
** the result type values to use in returning command values.
*/

int ArgListType()

{	/* ArgListType --- Return ARGLIST result type */

	return(ARGLIST);

}	/* ArgListType */



int IntType()

{	/* IntType --- Return INT result type */

	return(INT);

}	/* IntType */



int FloatType()

{	/* FloatType --- Return FLOAT result type */

    return(FLOAT);

}	/* FloatType */



int StrType()

{	/* StrType --- Return STR result type */

	return(STR);

}	/* StrType */



static int listmatches(ident, wild)

char	*ident;
char	*wild;

{	/* listmatches --- Return TRUE if the ident matches the SHELL wild */

	while (*wild != '\0')
	  {
	    switch (*wild)
	      {

	      case '?':
		if (*ident++ == '\0')
		    return(FALSE);
		break;

	      case '*':
		if (wild[1] == '?')
		  {
		    if (*ident++ == '\0')
			return(FALSE);
		    wild[1] = '*';
		  }
		else
		    while (*ident != wild[1] && *ident != '\0')
			ident++;
		break;

	      case '[':
		wild++;
		while (*wild != *ident)
		  {
		    if (*wild == ']' || *wild == '\0')
			return(FALSE);
		    wild++;
		  }

		while (*wild != ']')
		    wild++;
		ident++;
		break;

	      default:
		if (*ident++ != *wild)
		    return(FALSE);
		break;

	      }
	    wild++;
	  }

	return(*ident == '\0');

}	/* listmatches */



do_listglobals(argc, argv)

int	argc;
char	*argv[];

{	/* do_listglobals --- List global symbols and their types */

	extern int	optind;

	SymtabEnt	*se;
	int		first;
	int		listfunctions;
	int		listvariables;
	int		arg;
	int		c;

	listfunctions = TRUE;
	listvariables = TRUE;
	optind = 1;
	while ((c = getopt(argc, argv, "fv")) != EOF)
	    switch (c)
	      {

	      case 'f':
		listvariables = FALSE;
		listfunctions = TRUE;
		break;

	      case 'v':
		listvariables = TRUE;
		listfunctions = FALSE;
		break;

	      }

	first = TRUE;
	for (arg = optind; first || arg < argc; arg++)
	  {
	    se = GlobalSymbols.sym_entlist;
	    while (se != NULL)
	      {
		if (arg == argc || listmatches(se->sym_ident, argv[arg]))
		    switch (se->sym_val.r_type)
		      {

		      case INT:
			if (listvariables)
			    printf("int\t\t%s = %d\n", se->sym_ident, se->sym_val.r.r_int);
			break;

		      case FLOAT:
			if (listvariables)
			    printf("float\t\t%s = %0.15g\n", se->sym_ident,
							se->sym_val.r.r_float);
			break;

		      case STR:
			if (listvariables)
			    printf("str\t\t%s = \"%s\"\n", se->sym_ident,
							se->sym_val.r.r_str);
			break;

		      case FUNCTION:
			if (listfunctions)
			    printf("function\t%s\n", se->sym_ident);
			break;

		      }

		se = se->sym_next;
	      }

	    first = FALSE;
	  }

}	/* do_listglobals */



/*
** InControlStructure
**
**	Return TRUE if we are executing code in a loop.
*/

int InControlStructure()

{	/* InControlStructure --- Return TRUE if looping */

	return(InLoop != 0);

}	/* InControlStructure */



PushLocalVars(argc, argv, symtab)

int	argc;
char	*argv[];
Symtab	*symtab;

{	/* PushLocalVars --- Allocate a new frame of local variables */

	LocalVars	*localvars;
	Result		*vars, *rp;
	int		nlocals;
	int		i;

	localvars = (LocalVars *) calloc(1,sizeof(LocalVars));
	if (localvars == NULL)
	  {
	    perror("PushLocalVars");
	    sig_msg_restore_context(0, errno);
	  }
	localvars->lv_prev = CurLocals;
	CurLocals = localvars;

	if (symtab == NULL || symtab->sym_entlist == NULL ||
	    symtab->sym_entlist->sym_val.r_type != LOCREF)
	    nlocals = 0;
	else
	    nlocals = symtab->sym_entlist->sym_val.r.r_loc.l_offs + 1;
	nlocals += argc;

	localvars->lv_vars = (Result *) malloc(nlocals*sizeof(Result));
	if (localvars->lv_vars == NULL)
	  {
	    perror("PushLocalVars");
	    sig_msg_restore_context(0, errno);
	  }

	localvars->lv_nargs = argc;
	localvars->lv_nlocals = nlocals;
	for (i = 0, vars = localvars->lv_vars; i < nlocals; i++, vars++)
	  {
	    vars->r_type = STR;
	    if (i < argc)
	        vars->r.r_str = (char *) strsave(argv[i]);
	    else
	        vars->r.r_str = (char *) strsave("0");
	  }

#ifdef COMMENT
	if (*argv[0] == '\0')	/* FUNCTION CALL */
	  {
	    rp = SymtabLook(symtab, "*SCRIPT*");
	    localvars->lv_script = rp->r.r_str;
	    rp = SymtabLook(symtab, "*FUNCTION*");
	    localvars->lv_function = rp->r.r_str;
	  }
	else	/* SCRIPT CALL */
	  {
	    localvars->lv_script = argv[0];
	    localvars->lv_function = NULL;
	  }
#endif
	localvars->lv_lineno = 1;

}	/* PushLocalVars */



PopLocalVars()

{	/* PopLocalVars --- Deallocate local variable storage */

	if (CurLocals == NULL)
	    fprintf(stderr, "PopLocalVars: CurLocals == NULL\n");
	else
	  {
	    Result	*vars;
	    int		i;

	    for (i = 0, vars = CurLocals->lv_vars; i < CurLocals->lv_nlocals;
								    i++, vars++)
		if (vars->r_type == STR)
		    free(vars->r.r_str);

	    free(CurLocals->lv_vars);
	    free(CurLocals);
	    CurLocals = CurLocals->lv_prev;
	  }

}	/* PopLocalVars */



/*
** PTArgv
**
**	Return the arguments to a function or script by argument number
*/

char **PTArgv(arg)

int	arg;

{	/* PTArgv --- Return argument to a function or script */

	char	**argv;
	int	i;

	if (CurLocals == NULL)
	  {
	    argv = (char **) malloc(sizeof(char *));
	    if (argv == NULL)
		yyerror("PTArgv: Out of mem");
		/* No Return */

	    *argv = NULL;
	    return(argv);
	  }

	switch(arg)
	  {

	  case -1:	/* Return all args */
	    argv = (char **) malloc(sizeof(char *)*CurLocals->lv_nargs);
	    if (argv == NULL)
		yyerror("PTArgv: Out of mem");
		/* No Return */

	    for (i = 1; i < CurLocals->lv_nargs; i++)
		argv[i-1] = (char *) strsave(CurLocals->lv_vars[i].r.r_str);
	    argv[i-1] = NULL;
	    break;

	  case 0:	/* script or function name */
	    argv = (char **) malloc(sizeof(char *)*2);
	    if (argv == NULL)
		yyerror("PTArgv: Out of mem");
		/* No Return */

	    if (CurLocals->lv_function == NULL)
		argv[0] = (char *) strsave(CurLocals->lv_script);
	    else
		argv[0] = (char *) strsave(CurLocals->lv_function);
	    argv[1] = NULL;
	    break;

	  default:	/* Specific argument */
	    argv = (char **) malloc(sizeof(char *)*2);
	    if (argv == NULL)
		yyerror("PTArgv: Out of mem");
		/* No Return */

	    if (arg < 0 || arg >= CurLocals->lv_nargs)
		argv[0] = (char *) strsave("");
	    else
		argv[0] = (char *) strsave(CurLocals->lv_vars[arg].r.r_str);
	    argv[1] = NULL;
	    break;

	  }

	return(argv);

}	/* PTArgv */



/*
** PTArgc
**
**	Return the number of arguments passed to a function or script.
*/

int PTArgc()

{	/* PTArgc --- Return number of args */

	if (CurLocals == NULL)
	    return(0);
	else
	    return(CurLocals->lv_nargs-1);

}	/* PTArgc */



TraceScript()

{	/* TraceScript --- Provide backtrace of script and function calls */

	LocalVars	*lv;

	if (CurLocals == NULL)
	    return;

	lv = CurLocals;
	while (lv != NULL)
	  {
	    if (lv->lv_function != NULL)
		printf("function %s in <%s> line %d",
				lv->lv_function, lv->lv_script, lv->lv_lineno);
	    else
		printf("<%s> line %d", lv->lv_script, lv->lv_lineno);

	    lv = lv->lv_prev;
	    if (lv != NULL)
		printf(" <- ");
	  }
	printf("\n");

}	/* TraceScript */



SetLine(si)

ScriptInfo	*si;
/*
int		si;
*/

{	/* SetLine --- Set line number in the current local scope */

	if (CurLocals != NULL)
	  {
	    CurLocals->lv_lineno = si->si_lineno;
	    CurLocals->lv_script = si->si_script;
	  }
	ScriptLoc = *si;
	/*
	CurLocals->lv_lineno = si;
	*/

}	/* SetLine */



static Result *locref(offset)

int		offset;

{	/* locref --- Reference local variable */

	if (CurLocals == NULL)
	    yyerror("locref: NULL CurLocals");

	return(CurLocals->lv_vars + CurLocals->lv_nargs + offset);

}	/* locref */


static Result dollarref(offset)

int		offset;

{	/* dollarref --- Reference function argument */

	Result	r;

	if (CurLocals != NULL && offset < CurLocals->lv_nargs)
	    return(CurLocals->lv_vars[offset]);

	r.r_type = STR;
	r.r.r_str = (char *) strsave("");
	return(r);

}	/* dollarref */



CastToInt(rp)

Result	*rp;

{	/* CastToInt --- Cast value to integer */

	char	**argv;
	char	*mem;
	char    buf[100];
	int	i;

	switch(rp->r_type)
	  {

	  case ARGLIST:
	    argv = (char **) rp->r.r_str;
	    if (argv == NULL)
		yyerror("CastToInt: NULL Arglist\n");

	    mem = argv[0];
	    if (mem == NULL)
		yyerror("CastToInt: Empty Arglist\n");

	    if (sscanf(mem, "%d", &rp->r.r_int) != 1)
	      {
		Error();
		printf("CastToInt: Error casting '%s', using 0\n", mem);
		rp->r.r_int = 0;
	      }

	    for (i = 0; argv[i] != NULL; i++)
		free(argv[i]);
	    free(argv);
	    break;
	      
	  case STR:
	    mem = rp->r.r_str;
	    if (mem == NULL)
		yyerror("CastToInt: NULL string");
		/* No Return */
	    if (sscanf(rp->r.r_str, "%d", &rp->r.r_int) != 1)
	      {
		Error();
		printf("CastToInt: Error casting '%s', using 0\n",
					rp->r.r_str);
		rp->r.r_int = 0;
	      }
	    free(mem);
	    break;

	  case INT:
	    break;

	  case FLOAT:
	    rp->r.r_int = (int) rp->r.r_float;
	    break;

	  default:
	    sprintf(buf, "CastToInt: bad result type %s", TokenStr(rp->r_type));
	    yyerror(buf);
	    break;

	  }
	rp->r_type = INT;

}	/* CastToInt */



CastToFloat(rp)

Result	*rp;

{	/* CastToFloat --- Cast value to float */

	char	**argv;
	char	*mem;
	char    buf[100];
	int	i;

	switch(rp->r_type)
	  {

	  case ARGLIST:
	    argv = (char **) rp->r.r_str;
	    if (argv == NULL)
		yyerror("CastToFloat: NULL Arglist\n");

	    mem = argv[0];
	    if (mem == NULL)
		yyerror("CastToFloat: Empty Arglist\n");

	    if (sscanf(mem, "%lf", &rp->r.r_float) != 1)
	      {
		Error();
		printf("CastToFloat: Error casting '%s', using 0.0\n", mem);
		rp->r.r_float = 0.0;
	      }

	    for (i = 0; argv[i] != NULL; i++)
		free(argv[i]);
	    free(argv);
	    break;
	      
	  case STR:
	    mem = rp->r.r_str;
	    if (mem == NULL)
		yyerror("CastToFloat: NULL string");
		/* No Return */
	    if (sscanf(rp->r.r_str, "%lf", &rp->r.r_float) != 1)
	      {
		Error();
		printf("CastToFloat: Error casting '%s', using 0.0\n",
					rp->r.r_str);
		rp->r.r_float = 0.0;
	      }
	    free(mem);
	    break;

	  case INT:
	    rp->r.r_float = (double) rp->r.r_int;
	    break;

	  case FLOAT:
	    break;

	  default:
	    sprintf(buf,"CastToFloat: bad result type %s",TokenStr(rp->r_type));
	    yyerror(buf);
	    break;

	  }
	rp->r_type = FLOAT;

}	/* CastToFloat */



CastToStr(rp)

Result	*rp;

{	/* CastToStr --- Cast value to string */

	char	**argv;
	char	buf[100];
	char	*mem;
	int	i;

	switch(rp->r_type)
	  {

	  case ARGLIST:
	    argv = (char **) rp->r.r_str;
	    if (argv == NULL)
		yyerror("CastToInt: NULL Arglist\n");

	    mem = argv[0];
	    if (mem == NULL)
		yyerror("CastToInt: Empty Arglist\n");

	    rp->r.r_str = mem;
	    rp->r_type = STR;

	    for (i = 1; argv[i] != NULL; i++)
		free(argv[i]);
	    free(argv);
	    return;
	      
	  case STR:
	    if (rp->r.r_str == NULL)
		yyerror("CastToStr: NULL string");
		/* No Return */
	    return;

	  case INT:
	    sprintf(buf, "%d", rp->r.r_int);
	    break;

	  case FLOAT:
	    sprintf(buf, "%0.15g", rp->r.r_float);
	    break;

	  default:
	    sprintf(buf, "CastToStr: bad result type %s", TokenStr(rp->r_type));
	    yyerror(buf);
	    break;

	  }

	mem = (char *) malloc(strlen(buf)+1);
	if (mem == NULL)
	  {
	    perror("CastToStr");
	    sig_msg_restore_context(0, errno);
	  }

	strcpy(mem, buf);
	rp->r.r_str = mem;
	rp->r_type = STR;

}	/* CastToStr */



#define	DOOP(swop,rtyp,rfld,oper) case swop: \
				  r.r_type = rtyp; \
				  r.r.rfld = arg1 oper arg2; \
				  break

static Result intop(arg1, op, arg2)

int	arg1;
int	op;
int	arg2;

{	/* intop --- Do integer operation */

	Result	r;
	char    buf[100];

	switch (op)
	  {

	  case UMINUS:
	    r.r_type = INT;
	    r.r.r_int = -arg1;
	    break;

	  DOOP('*',INT,r_int,*);
	  DOOP('/',INT,r_int,/);
	  DOOP('%',INT,r_int,%);
	  DOOP('+',INT,r_int,+);
	  DOOP('-',INT,r_int,-);

	  case '~':
	    r.r_type = INT;
	    r.r.r_int = ~arg1;
	    break;

	  DOOP('|',INT,r_int,|);
	  DOOP('&',INT,r_int,&);
	  DOOP('^',INT,r_int,^);

	  DOOP(AND,INT,r_int,&&);
	  DOOP(OR,INT,r_int,||);

	  DOOP(LT,INT,r_int,<);
	  DOOP(LE,INT,r_int,<=);
	  DOOP(GT,INT,r_int,>);
	  DOOP(GE,INT,r_int,>=);
	  DOOP(EQ,INT,r_int,==);
	  DOOP(NE,INT,r_int,!=);
	  
	  case '!':
	    r.r_type = INT;
	    r.r.r_int = !arg1;
	    break;

	  default:
	    sprintf(buf, "intop: Unknown or illegal operator %s", TokenStr(op));
	    yyerror(buf);
	    break;;

	  }

	return(r);

}	/* intop */



static Result floatop(arg1, op, arg2)

float	arg1;
int	op;
float	arg2;

{	/* intop --- Do integer operation */

	Result	r;
	char    buf[100];

	switch (op)
	  {

	  case UMINUS:
	    r.r_type = FLOAT;
	    r.r.r_float = -arg1;
	    break;

	  DOOP('*',FLOAT,r_float,*);
	  DOOP('/',FLOAT,r_float,/);
	  DOOP('+',FLOAT,r_float,+);
	  DOOP('-',FLOAT,r_float,-);
	  DOOP(LT,INT,r_int,<);
	  DOOP(LE,INT,r_int,<=);
	  DOOP(GT,INT,r_int,>);
	  DOOP(GE,INT,r_int,>=);
	  DOOP(EQ,INT,r_int,==);
	  DOOP(NE,INT,r_int,!=);

	  default:
	    sprintf(buf,"floatop: Unknown or illegal operator %s",TokenStr(op));
	    yyerror(buf);
	    break;;

	  }

	return(r);

}	/* floatop */



static Result strop(arg1, op, arg2)

char	*arg1;
int	op;
char	*arg2;

{	/* strop --- Do string operation */

	Result	r;
	char    buf[100];

	r.r_type = INT;
	switch (op)
	  {

	  case '+':	/* Concatination */
	    r.r.r_str = (char *) malloc(strlen(arg1) + strlen(arg2) + 1);
	    if (r.r.r_str == NULL)
	      {
		perror("strop(+)");
		sig_msg_restore_context(0, errno);
		/* No return */
	      }

	    sprintf(r.r.r_str, "%s%s", arg1, arg2);
	    r.r_type = STR;
	    break;

	  case EQ:
	    r.r.r_int = strcmp(arg1, arg2) == 0;
	    break;

	  case NE:
	    r.r.r_int = strcmp(arg1, arg2) != 0;
	    break;

	  case GT:
	    r.r.r_int = strcmp(arg1, arg2) > 0;
	    break;

	  case GE:
	    r.r.r_int = strcmp(arg1, arg2) >= 0;
	    break;

	  case LT:
	    r.r.r_int = strcmp(arg1, arg2) < 0;
	    break;

	  case LE:
	    r.r.r_int = strcmp(arg1, arg2) <= 0;
	    break;

	  default:
	    sprintf(buf, "strop: Unknown or illegal operator %s", TokenStr(op));
	    yyerror(buf);
	    /* No Return */

	  }

	return(r);

}	/* strop */


promote(arg1, arg2)

Result	*arg1;
Result	*arg2;

{	/* promote --- Promote one or other arg if necessary */

	char	**argv;
	char    buf[100];
	int	i;

	if ((arg1->r_type == STR && arg1->r.r_str == NULL) ||
	    (arg2->r_type == STR && arg2->r.r_str == NULL))
	    yyerror("promote: NULL string");
	    /* No Return */

	if (arg1->r_type == ARGLIST)
	    CastToStr(arg1);

	if (arg2->r_type == ARGLIST)
	    CastToStr(arg2);

	if (arg2->r_type == 0 || arg1->r_type == arg2->r_type)
	    return;

	switch (arg1->r_type)
	  {

	  case STR:
	    switch (arg2->r_type)
	      {

	      case INT:
		free(arg1->r.r_str);
		arg1->r_type = INT;
		if (sscanf(arg1->r.r_str, "%d", &arg1->r.r_int) != 1)
		  {
		    Error();
		    printf("Error promoting '%s' to INT, using 0\n",
						arg1->r.r_str);
		    arg1->r.r_int = 0;
		  }
		break;

	      case FLOAT:
		free(arg1->r.r_str);
		arg1->r_type = FLOAT;
		if (sscanf(arg1->r.r_str, "%lf", &arg1->r.r_float) != 1)
		  {
		    Error();
		    printf("Error promoting '%s' to FLOAT, using 0\n",
						arg1->r.r_str);
		    arg1->r.r_float = 0.0;
		  }
		break;

	      default:
	        sprintf(buf,"promote: Unknown argument type %s",
				TokenStr(arg2->r_type));
		yyerror(buf);
	        /* No Return */

	      }
	    break;

	  case INT:
	    switch (arg2->r_type)
	      {

	      case STR:
		free(arg2->r.r_str);
		arg2->r_type = INT;
		if (sscanf(arg2->r.r_str, "%d", &arg2->r.r_int) != 1)
		  {
		    Error();
		    printf("Error promoting '%s' to INT, using 0\n",
						arg2->r.r_str);
		    arg2->r.r_int = 0;
		  }
		break;

	      case FLOAT:
		arg1->r_type = FLOAT;
		arg1->r.r_float = (double) arg1->r.r_int;
		break;

	      default:
	        sprintf(buf,"promote: Unknown argument type %s",
				TokenStr(arg2->r_type));
		yyerror(buf);
	        /* No Return */

	      }
	    break;

	  case FLOAT:
	    switch (arg2->r_type)
	      {

	      case STR:
		free(arg2->r.r_str);
		arg2->r_type = FLOAT;
		if (sscanf(arg2->r.r_str, "%lf", &arg2->r.r_float) != 1)
		  {
		    Error();
		    printf("Error promoting '%s' to FLOAT, using 0\n",
						arg2->r.r_str);
		    arg2->r.r_float = 0.0;
		  }
		break;

	      case INT:
		arg2->r_type = FLOAT;
		arg2->r.r_float = (double) arg2->r.r_int;
		break;

	      default:
	        sprintf(buf,"promote: Unknown argument type %s",
				TokenStr(arg2->r_type));
		yyerror(buf);
	        /* No Return */

	      }
	    break;

	  default:
	    sprintf(buf,"promote: Unknown argument type %s",
			TokenStr(arg1->r_type));
	    yyerror(buf);
	    /* No Return */

	  }

}	/* promote */



extern Result ExecuteCommand();

extern Result do_cmd();

char *do_cmd_args(arg, argc, argv)

ParseNode	*arg;
int		*argc;
char		*argv[];

{	/* do_cmd_args --- Handle arguments to a command */

	Result	r;
	char	**cargv;
	char	*argcp, *arg1, *arg2, *result;
	char    buf[1000];

	/*
	** Format argument list for command execution.
	*/

	if (arg == NULL)
	    return(NULL);
 
	switch (arg->pn_val.r_type)
	  {

	  case COMMAND:
	    r = do_cmd(arg);
	    switch (r.r_type)
	      {

	      case INT:
		sprintf(buf, "%d", r.r.r_int);
		return(strsave(buf));

	      case FLOAT:
		sprintf(buf, "%0.15g", r.r.r_float);
		return(strsave(buf));

	      case STR:
		if (r.r.r_str == NULL)
		    yyerror("do_cmd_args: NULL string");
		    /* No Return */

		return(strsave(r.r.r_str));

	      case ARGLIST:
		cargv = (char **) r.r.r_str;
		while (*cargv != NULL)
		  {
		    argv[*argc] = *cargv++;
		    *argc += 1;
		  }
		free(r.r.r_str);
		return(NULL);

	      }

	  case ARGLIST:
	    do_cmd_args(arg->pn_left, argc, argv);
	    argcp = do_cmd_args(arg->pn_right, argc, argv);
	    if (argcp != NULL)
	      {
		argv[*argc] = argcp;
		*argc += 1;
	      }
	    break;

	  case ARGUMENT:
	    arg1 = do_cmd_args(arg->pn_left, argc, argv);
	    arg2 = do_cmd_args(arg->pn_right, argc, argv);

	    if (arg1 == NULL) return(arg2);
	    if (arg2 == NULL) return(arg1);

	    result = (char *) malloc(strlen(arg1) + strlen(arg2) + 1);
	    if (result == NULL)
	      {
		perror("do_cmd_args");
		yyerror("Out of mem");
		/* No Return */
	      }

	    sprintf(result, "%s%s", arg1, arg2);
	    free(arg1);
	    free(arg2);

	    return(result);

	  case LITERAL:
	    return(strsave(arg->pn_val.r.r_str));

	  default:	/* expr or DOLLARARG */
	    r = PTEval(arg);
	    switch (r.r_type)
	      {

	      case INT:
	        sprintf(buf, "%d", r.r.r_int);
		return(strsave(buf));

	      case FLOAT:
	        sprintf(buf, "%0.15g", r.r.r_float);
		return(strsave(buf));

	      case STR:
		return(strsave(r.r.r_str));

	      default:
	        sprintf(buf, "do_cmd_args: Unknown return value type %s from PTEval", TokenStr(r.r_type));
		yyerror(buf);
		/* No Return */

	      }
	    break;

	  }

	return(NULL);

}	/* do_cmd_args */



static Result do_cmd(cmd)

ParseNode	*cmd;

{	/* do_cmd --- Call command and return result */

	int	i;
	int	argc;
	int	saveinloop;
	char	*argv[100];
	char	argbuf[1000];
	jmp_buf	save;
	Result	r;

	sprintf(argbuf, "%s", cmd->pn_val.r.r_str);
	argc = 1;
	argv[0] = argbuf;
	do_cmd_args(cmd->pn_left, &argc, argv);
	argv[argc] = NULL;

	bcopy(ReturnBuf, save, sizeof(jmp_buf));
	saveinloop = InLoop;
	if (setjmp(ReturnBuf) == 0)
	  {
	    r = ExecuteCommand(argc, argv);
	  }
	else
	    r = ReturnResult;
	InLoop = saveinloop;
	bcopy(save, ReturnBuf, sizeof(jmp_buf));

	for (i = 1; i < argc; i++)
	    free(argv[i]);

	return(r);

}	/* do_cmd */



do_include_args(script, args)

char		*script;
ParseNode	*args;

{	/* do_include_args --- Set up localvars for included script */

	int	i;
	int	argc;
	int	saveinloop;
	char	*argv[100];
	char	argbuf[1000];
	jmp_buf	save;
	Result	r;

	sprintf(argbuf, "%s", script);
	argc = 1;
	argv[0] = argbuf;
	do_cmd_args(args, &argc, argv);
	argv[argc] = NULL;

	bcopy(ReturnBuf, save, sizeof(jmp_buf));
	saveinloop = InLoop;
	if (setjmp(ReturnBuf) == 0)
	  {
	    PushLocalVars(argc, argv, NULL);
	  }
	InLoop = saveinloop;
	bcopy(save, ReturnBuf, sizeof(jmp_buf));

	for (i = 1; i < argc; i++)
	    free(argv[i]);

}	/* do_include_args */



/*
** ExecuteFunction
**
**	The simulator calls ExecuteFunction to execute a function or a
**	simulator command.  If the function doesn't exist then call
**	ExecuteCommand.
*/

ExecuteFunction(argc, argv)

int	argc;
char	*argv[];

{	/* ExecuteFunction --- Given argc/argv lookup function and execute */

	jmp_buf		save;
	int		saveinloop;
	Result		*rp;
	ParseNode	*func;

	if (argc < 1)
	    yyerror("ExecuteFunction: argc < 1!");
	    /* No Return */

	rp = SymtabLook(&GlobalSymbols, argv[0]);
	if (rp == NULL || rp->r_type != FUNCTION)
	    ExecuteCommand(argc, argv);
	else
	  {
	    func = (ParseNode *) rp->r.r_str;
	    PushLocalVars(argc, argv, func->pn_val.r.r_str);

	    bcopy(ReturnBuf, save, sizeof(jmp_buf));
	    saveinloop = InLoop;
	    InLoop++;
	    if (setjmp(ReturnBuf) == 0)
		PTEval(func);
	    InLoop = saveinloop;
	    bcopy(save, ReturnBuf, sizeof(jmp_buf));

	    PopLocalVars();
	  }

}	/* ExecuteFunction */



static Result do_func(func, args)

ParseNode	*func;
ParseNode	*args;

{	/* do_func --- Execute function */

	int	i;
	int	argc;
	char	*argv[100];
	char	argbuf[2];
	jmp_buf	save, savebrk;
	int	saveinloop;
	Result	r;

	if (func->pn_val.r.r_str == NULL)    /* extern function not defined */
	  {
	    r.r_type = INT;
	    r.r.r_int = 0;
	    return(r);
	  }

	argbuf[0] = '\0';
	argv[0] = argbuf;
	argc = 1;
	do_cmd_args(args, &argc, argv);
	PushLocalVars(argc, argv, func->pn_val.r.r_str);

	bcopy(ReturnBuf, save, sizeof(jmp_buf));
	bcopy(BreakBuf, savebrk, sizeof(jmp_buf));
	saveinloop = InLoop;
	InLoop++;
	if (setjmp(ReturnBuf) == 0)
	  {
	    PTEval(func);
	    r.r_type = INT;
	    r.r.r_int = 0;
	  }
	else
	    r = ReturnResult;
	InLoop = saveinloop;
	bcopy(save, ReturnBuf, sizeof(jmp_buf));
	bcopy(savebrk, BreakBuf, sizeof(jmp_buf));

	for (i = 1; i < argc; i++)
	    free(argv[i]);

	PopLocalVars();
	return(r);

}	/* do_func */



char *do_foreach_arg(arg, body, rp)

ParseNode	*arg;
ParseNode	*body;
Result		*rp;

{	/* do_foreach_arg --- execute foreach body for one argument */

	int	i;
	int	argc;
	char	argbuf[1000];
	Result	r;
	char	**cargv;
	char	*argcp, *arg1, *arg2, *result;
	char    buf[1000];

	/*
	** Format argument list for command execution.
	*/

	if (arg == NULL)
	    return(NULL);
 
	switch (arg->pn_val.r_type)
	  {

	  case COMMAND:
	    r = do_cmd(arg);
	    switch (r.r_type)
	      {

	      case INT:
		sprintf(buf, "%d", r.r.r_int);
		return(strsave(buf));

	      case FLOAT:
		sprintf(buf, "%0.15g", r.r.r_float);
		return(strsave(buf));

	      case STR:
		if (r.r.r_str == NULL)
		    yyerror("do_cmd_args: NULL string");
		    /* No Return */

		return(strsave(r.r.r_str));

	      case ARGLIST:
		cargv = (char **) r.r.r_str;
		while (*cargv != NULL)
		  {
		    free(rp->r.r_str);
		    rp->r.r_str = *cargv++;
		    PTEval(body);
		  }
		free(r.r.r_str);
		return(NULL);

	      }

	  case ARGLIST:
	    do_foreach_arg(arg->pn_left, body, rp);
	    argcp = do_foreach_arg(arg->pn_right, body, rp);
	    if (argcp != NULL)
	      {
		free(rp->r.r_str);
		rp->r.r_str = argcp;
		PTEval(body);
	      }
	    break;

	  case ARGUMENT:
	    arg1 = do_foreach_arg(arg->pn_left, body, rp);
	    arg2 = do_foreach_arg(arg->pn_right, body, rp);

	    if (arg1 == NULL) return(arg2);
	    if (arg2 == NULL) return(arg1);

	    result = (char *) malloc(strlen(arg1) + strlen(arg2) + 1);
	    if (result == NULL)
	      {
		perror("do_cmd_args");
		yyerror("Out of mem");
		/* No Return */
	      }

	    sprintf(result, "%s%s", arg1, arg2);
	    free(arg1);
	    free(arg2);

	    return(result);

	  case LITERAL:
	    return(strsave(arg->pn_val.r.r_str));

	  default:	/* expr or DOLLARARG */
	    r = PTEval(arg);
	    switch (r.r_type)
	      {

	      case INT:
	        sprintf(buf, "%d", r.r.r_int);
		return(strsave(buf));

	      case FLOAT:
	        sprintf(buf, "%0.15g", r.r.r_float);
		return(strsave(buf));

	      case STR:
		return(strsave(r.r.r_str));

	      default:
	        sprintf(buf, "do_cmd_args: Unknown return value type %s from PTEval", TokenStr(r.r_type));
		yyerror(buf);
		/* No Return */

	      }
	    break;

	  }

	return(NULL);

}	/* do_foreach_arg */



static do_foreach(arg, body, rp)

ParseNode	*arg;
ParseNode	*body;
Result		*rp;

{	/* do_foreach --- Execute the body for current argument */

	if (arg == NULL)
	    return;

	switch(arg->pn_val.r_type)
	  {

	  case ARGLIST:
	    do_foreach_arg(arg, body, rp);
	    break;

	  case ARGUMENT:
	    do_foreach_arg(arg, body, rp);
	    break;

	  default:
	    yyerror("do_foreach: bad parse node type");
	    /* No Return */

	  }


}	/* do_foreach */



Result PTEval(pn)

ParseNode	*pn;

{	/* PTEval --- Evaluate the current node of the parse tree */

	Result	arg1, arg2, r, *rp;
	jmp_buf	save;
	char    buf[100];

	r.r_type = 0;
	if (pn == NULL)
	    return(r);

	switch (pn->pn_val.r_type)
	  {

	  case UMINUS: case '*': case '/': case '%': case '+': case '-':
	  case LT: case LE: case GT: case GE: case EQ: case NE:
	  case '&': case '|': case '^': case '~':
	  case OR: case AND: case '!':

	    arg1 = PTEval(pn->pn_left);
	    arg2 = PTEval(pn->pn_right);
	    promote(&arg1, &arg2);
	    switch (arg1.r_type)
	      {

	      case INT:
		return(intop(arg1.r.r_int, pn->pn_val.r_type, arg2.r.r_int));

	      case FLOAT:
		return(floatop(arg1.r.r_float, pn->pn_val.r_type, arg2.r.r_float));

	      case STR:
		return(strop(arg1.r.r_str, pn->pn_val.r_type, arg2.r.r_str));

	      }
	    break;

	  case ICAST:
	    rp = (Result *) pn->pn_val.r.r_str;
	    if (rp->r_type == LOCREF)
	        rp = locref(rp->r.r_loc.l_offs);
	    CastToInt(rp);
	    break;

	  case FCAST:
	    rp = (Result *) pn->pn_val.r.r_str;
	    if (rp->r_type == LOCREF)
	        rp = locref(rp->r.r_loc.l_offs);
	    CastToFloat(rp);
	    break;

	  case SCAST:
	    rp = (Result *) pn->pn_val.r.r_str;
	    if (rp->r_type == LOCREF)
	        rp = locref(rp->r.r_loc.l_offs);
	    CastToStr(rp);
	    break;

	  case INTCONST:
	    r = pn->pn_val;
	    r.r_type = INT;
	    break;

	  case FLOATCONST:
	    r = pn->pn_val;
	    r.r_type = FLOAT;
	    break;

	  case STRCONST:
	    r.r.r_str = (char *) strsave(pn->pn_val.r.r_str);
	    r.r_type = STR;
	    break;

	  case INT:	/* Reference to a global integer variable */
	  case FLOAT:	/* Reference to a global float variable */
	    r = * (Result *)(pn->pn_val.r.r_str);
	    break;

	  case STR:	/* Reference to a global string variable */
	    r = * (Result *)(pn->pn_val.r.r_str);
	    r.r.r_str = (char *) strsave(r.r.r_str);
	    r.r_type = STR;
	    break;

	  case LOCREF:	/* Reference to a local variable */
	    rp = locref(pn->pn_val.r.r_loc.l_offs);
	    r = *rp;
	    if (r.r_type == STR)
		r.r.r_str = (char *) strsave(r.r.r_str);
	    break;

	  case DOLLARARG:
	    r = dollarref(pn->pn_val.r.r_int);
	    r.r.r_str = (char *) strsave(r.r.r_str);
	    break;

	  case '=':
	      {
		r = PTEval(pn->pn_left);
		rp = (Result *) pn->pn_val.r.r_str;
		if (rp->r_type == LOCREF)
		    rp = locref(rp->r.r_loc.l_offs);

	        switch (rp->r_type)
		  {

		  case INT:
		    CastToInt(&r);
		    break;

		  case FLOAT:
		    CastToFloat(&r);
		    break;

		  case STR:
		    CastToStr(&r);
		    break;

		  }
		rp->r = r.r;
	      }
	    break;

	  case FUNCTION:
	    r = do_func(pn->pn_val.r.r_str, pn->pn_left);
	    break;

	  case BREAK:
	    longjmp(BreakBuf, 1);
	    /* No Return */

	  case RETURN:
	    if (pn->pn_left == NULL)
	      {
		ReturnResult.r_type = INT;
		ReturnResult.r.r_int = 0;
	      }
	    else
		ReturnResult = PTEval(pn->pn_left);
	    longjmp(ReturnBuf, 1);
	    /* No Return */

	  case COMMAND:
	    r = do_cmd(pn);
	    break;

	  case INCLUDE:
	    ScriptLoc.si_lineno = 0;
	    ScriptLoc.si_script = pn->pn_val.r.r_str;
	    do_include_args(pn->pn_val.r.r_str, pn->pn_left);
	    break;

	  case ENDSCRIPT:
	    PopLocalVars();
	    break;

	  case SL:
	    PTEval(pn->pn_left);
	    SetLine(pn->pn_val.r.r_str);
	    PTEval(pn->pn_right);
	    break;

	  case IF:
	    SetLine(pn->pn_val.r.r_str);
	    r = PTEval(pn->pn_left);
	    if (r.r.r_int)
	        PTEval(pn->pn_right->pn_left);
	    else
	        PTEval(pn->pn_right->pn_right);
	    break;

	  case FOREACH:
	    rp = (Result *) pn->pn_val.r.r_str;
	    if (rp->r_type == LOCREF)
		rp = locref(rp->r.r_loc.l_offs);

	    bcopy(BreakBuf, save, sizeof(jmp_buf));
	    InLoop++;
	    if (setjmp(BreakBuf) == 0)
		do_foreach(pn->pn_left, pn->pn_right, rp);
	    InLoop--;
	    bcopy(save, BreakBuf, sizeof(jmp_buf));
	    break;

	  case WHILE:
	    SetLine(pn->pn_val.r.r_str);
	    r = PTEval(pn->pn_left);
	    bcopy(BreakBuf, save, sizeof(jmp_buf));
	    InLoop++;
	    if (setjmp(BreakBuf) == 0)
		while (r.r.r_int)
		  {
		    PTEval(pn->pn_right);
		    SetLine(pn->pn_val.r.r_str);
		    r = PTEval(pn->pn_left);
		  }
	    InLoop--;
	    bcopy(save, BreakBuf, sizeof(jmp_buf));
	    break;

	  default:
	    sprintf(buf, "PTEval: unknown parse tree node %s",
						TokenStr(pn->pn_val.r_type));
	    yyerror(buf);
	    /* No Return */

	  }

	return(r);

}	/* PTEval */



PTInit()

{	/* PTInit --- Initialize parse tree evaluation variables */

	InLoop = 0;
    	while (CurLocals != NULL)
	    PopLocalVars();

}	/* PTInit */

