/* Interface.h - abstract class interface to point-repulsion simulation
 *
 * $Id: Interface.h,v 1.3 96/02/11 21:27:41 leech Exp $
 *
 * Copyright (C) 1996, Jonathan P. Leech
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is". Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Log:	Interface.h,v $
 * Revision 1.3  96/02/11  21:27:41  leech
 * New -vrml option, better closest-pair stopping criterion, change
 * some option names.
 * 
 * Revision 1.2  92/10/27  22:50:05  leech
 * *** empty log message ***
 *
 * Revision 1.1  92/04/27  04:25:10  leech
 * Initial revision
 *
 */

#ifndef _jpl_interface_h
#define _jpl_interface_h

#include <boolean.h>
#include <limits.h>

class SphereState;

// This should be scoped in Interface, but g++ chokes on it.
enum TerminationCause {
    kConverged,
    kStopped,
    kQuit
};

// A simple-minded wrapper to handle GUI or text interfaces to the
//  simulation; derived classes returned by MakeInterface do the
//  actual work.
class Interface {
public:
    Interface(int interactive, int &ac, char **av);
    virtual ~Interface();

    // Should the simulation quit?
    virtual int Done() = 0;

    // Handle GUI events. Returns 1 if there was an event, 0 otherwise
    virtual int HandleEvent() = 0;

    // Redisplay state of simulation
    virtual void UpdatePoints() = 0;

    // Return the simulation model being driven
    SphereState *GetModel() const;

    // Run the simulation according to command line parameters
    void Run();

    // Clean up (write save files, etc.) after running simulation
    void Cleanup(long &iterations, TerminationCause &why);

    // Display help on stdout
    void Help(const char *program = "distribute");

    // Create a derived Interface for the particular GUI being used
    static Interface *MakeInterface(int &ac, char **av);

protected:
    SphereState *model;

    boolean Running() const { return running; }
    void RunStop(boolean stopgo) { running = stopgo; }

private:
    int trace;	    // Bitmask of values to be traced at each timestep
    boolean
	debug,	    // Flags for debugging & monitoring the simulation
	log,
	running;    // Is simulation running?
    const char
       *traceFmt;   // Format string to print trace results
    long
	step,	    // Current iteration
	maxStep;    // Stop here even if stop condition not satisfied
    const char
       *loadFile,   // File to load bodies from
       *saveFile,   // File to save bodies to
       *vrmlFile;   // File to save convex hull polygons to

    // Possible stopping conditions (when corresponding scalar drops
    //	below stopVal) and tracing values. k<Name> is an index,
    //	m<Name> is a bitmask for the format string options.
    enum {
	kKineticEnergy,
	kPotentialEnergy,
	kMaxVelocity,
	kClosestPair,
	kDistanceRatio,
	kUnset,		    // Flags when stop condition is not set by user
	kNeverStop
    } stopType;
    enum {
	mKineticEnergy = 1 << kKineticEnergy,
	mPotentialEnergy = 1 << kPotentialEnergy,
	mMaxVelocity = 1 << kMaxVelocity,
	mClosestPair = 1 << kClosestPair,
	mDistanceRatio = 1 << kDistanceRatio
    };
    double stopVal; // Value satisfying stop condition

    boolean NeedFmt(int index) const;
    int FmtMask(const char *);

    static int FmtMaskTable[UCHAR_MAX+1];
    static int FmtType(unsigned char c) { return FmtMaskTable[c]; }
    static void InitFmt();

    static char *HelpText[];
};

#endif /*_jpl_interface_h*/
