/* Morph3DSub.h header file for 3D image morphology subroutine Morph3DSub */

/* morph3d version 4.0  1 June 1993                */
/* 3D image morphology program                     */ 
/*                                                 */ 
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */
/* Nashville, TN 37235                             */
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */

/* This subroutine performs many (if not all) possible 3D morphological     */
/* operations on gray-level or binary image sequences. The operators        */
/* include hit-or-miss transforms, order-statistic filters,                 */
/* function-processing or set-processing erosion,  dilation, opening and    */
/* closing, minmax, maxmin, and isolated delete functions.                  */
/* The routine will process either time sequences of 2D images or true 3D   */
/* datasets that have been organized as a set of 2D images (voxel planes).  */

/* For a nice tutorial on image morphology, see                             */
/* Haralick, R.M, S.R. Sternberg, and X. Zhuang, "Image analysis using      */
/* mathematical morphology," IEEE Trans. PAMI, vol. 9, No. 4, July 1987,    */
/* pp. 532-550.                                                             */
/* or                                                                       */
/* Maragos, P. and R. Shaffer,                                              */
/* "Morphological systems for multidimensional Signal Processing",          */
/* Proc. IEEE, vol. 78, no. 4, April 1990.                                  */
/* or                                                                       */
/* For additional info on the hit-or-miss transform, see J. Serra, Image    */
/* Analysis and Mathematical Morphology, Academic Press, London, 1982.      */

/*

To use:

void Morph3DSub( MOp, InputImage, InStruct, OutputImage, OutStruct, 
                 Ix, Iy, ImgType, NZPad,LThresh, UThresh, 
                 SE, AutoSE, sX, sY, sZ, sV, sorgx, sorgy, sorgz, SEType, SorF, 
                 Rank ) 

   int MOp;                Morphological operation to perform 

   int (*InputImage)();    address of image input function  
   int *InStruct;          address of structure to pass to InputImage 
   int (*OutputImage)();   address of image output function 
   int *OutStruct;         address of structure to pass to OutputImage 
   int Ix, Iy;             image horizontal, vertical dimensions 
   int ImgType;            image type (gray-level or binary) 
   int NZPad ;             flag.  F => zeropadding of input 
   int LThresh;            lower binary threshold value 
   int UThresh;            upper binary threshold value 

   int *SE;                SE in plane-row-major order 
   int sX,sY,sZ,sV;        SE x, y, and z support dims and max gray-lev (v) 
   int sorgx,sorgy,sorgz;  SE origin coordinates 
   int SEType;             Binary SE or gray-level SE 
   int SorF;               Set operation or Function operation 
   int Rank;               rank for rank filter 



Detailed explanation of parameters:


    MOp     Morphological Operation. This integer parameter can have the
            following values:

            mnemon  hex       dec  action
            
            ERODE   0x0001      1  erosion
            DILATE  0x0002      2  dilation
            OPEN    0x0004      4  opening
            CLOSE   0x0008      8  closing
            RANK    0x0010     16  rank filter (order statistic filter)
            LUM     0x0011     17  Lower-Upper-Middle filter
            LUMSMO  0x0012     18  LUM smoothing filter
            LUMSHA  0x0014     19  LUM sharpening filter
            MAXMIN  0x0020     32  maximum of minima from nbhd. in each image
            MINMAX  0x0040     64  minimum of maxima from nbhd. in each image
            
            Moreover, the erode and dilate functions can be "negated" using:

            NOTFLG  0x0080  128 "not" flag for binary erodes and dilates

            Let I = original image; E = eroded image; D = dilated image.
            The following are valid parameters:
 
            mnemon            hex       dec      action
            
            ERODE  | NOTFLG   0x0081    129      I && !E (pixelwise)
            DILATE | NOTFLG   0x0082    130      D && !I (pixelwise)

            ERODE | NOTFLG with a binary hit-or-miss structuring element will 
            delete in a binary image sequence, white features with the shape 
            of the "hit" portion of the SE. (e.g. one can easily devise a SE to 
            delete isolated white pixels). With a gray-level SE it will delete 
            the "interiors" from sets of white pixels in a binary sequence.
            DILATE | NOTFLG with a binary hit-or-miss structuring element will 
            delete in a binary image seq., black features with the shape of the 
            "hit" portion of the SE. (e.g. one can easily devise a SE to delete 
            isolated black pixels). With a gray-level SE it will delete the 
            "interiors" from sets of black pixels in a binary image sequence.

            LUM is the lower-upper-middle filter as defined by Hardie, R. C.,
            and C. G. Boncelet, "LUM filters: A class of rank-order-based 
            filters for smoothing and sharpening," IEEE Trans. Signal 
            Processing, vol. SP-41, No. 3, March 1993.            
            LUMSMO is the LUM smoothing filter defined therein, and LUMSHA is 
            the LUM sharpening filter.
            These filters compare the center pixel in a neighborhood defined by 
            an SE to upper and lower order statistics in the neighborhood.
            Depending on the ordering either the center pixel or one of the 
            order stats is output. 
            Note that we define order statistics opposite Hardie and Boncelet
            Whereas OS(1) is the minimum for them OS(1) is the maximum in
            these routines.   

            Each of these operations occurs over a 3D data set within a 3D
            neighborhood (nbhd) defined by the 3D shape of the structuring
            element (SE). The definitions are exactly the same as their 2D
            counterparts. (Mathematical morphology is defined in n dimesions.)
            The exceptions are maxmin and minmax which have no counterparts
            in 2D. These operators were designed for use with time sequences.
            Consider the 3D SE to be a stack of z 2D SE's. The 2D SE's trace
            nbhds in adjacent sequential images. Maxmin finds the min in each 
            2D nbhd and takes the maximum of these. Minmax takes the smallest 
            maximum from among the 2D nbhds. The idea behind minmax is this:
            Applied to a time sequence, a 3D structuring element demarcates
            an area (a nbhd) in successive images. (The nbhd in an image
            may or may not be the same the nbhds in succesive images.)
            If a bright, moving particle stays within the successive nbhds
            the result of the minmax will be a bright pixel. If, on the other
            hand, the particle moves out of one of the nbhds, the result will
            be a darker pixel. Thus, when used with minmax, an appropriately 
            shaped SE can act as a velocity filter. Maxmin is the same with
            with respect to dark particles.


    InputImage(), InStruct, OutputImage(), OutStruct

            These are pointers to user supplied IO routines and data structures 
            that Morph3DSub uses to read and write images. These are described
            below.


    Ix      Image horizontal dimension. Number of pixels per line (or
            number of columns per row) in an image from the sequence.

    Iy      Image vertical dimension. Number of lines (or rows) in an image.

            All images in the input sequence must have the same xy
            dimensions.


    ImgType Image type: This specifies whether the input image sequence is 
            to be treated like a gray level or binary sequnce. The values in
            a binary image are zero and not-zero (it does not matter
            which of the values in [BLACK,WHITE]).
            Parameter values:

            mnemon    hex       dec        meaning
            
            GRAIMG    0x0000      0        gray-level image sequence
            BINIMG    0x0200    512        binary image sequence


    NZPad   If zero (FALSE) Morph3DSub will extend the size of each input
            image internally and create a border of zeros around it in the 
            x and y dimensions. If nonzero (TRUE) Morph3DSub will not do the 
            zero padding.

            There are image border effects with each NZPAD option. The effects
            differ depending on the option.
            With the option FALSE, Morph3DSub does what it can to "color in" 
            each image near its perimeter.
            With this option TRUE, each output image has a border of zeros 
            inside it the width and height of the SE. That is, the 
            transformed area of each output image is smaller than the actual
            image dimensions. This is, in a sense, a more accurate result than
            the zero padded default. To zero pad the input permits the program
            to transform the border region, but it does this on the assumption
            that the original scene was black outside the image sequence. This, 
            of course, is almost never true. Thus, the border region is 
            inaccurately transformed. Use this switch if accuracy is more 
            important than having an image sequence that is "colored in" out 
            to the boundary.

            Notes: 

                This is all internal to Morph3DSub. The input and
                output images are all Ix by Iy independent of this flag.

                Morph3DSub _always_ zero pads in the z-dimension. That is it
                always catenates enough zero images to the beginning and and
                of the sequence so that the number of images output equals
                the number of images input.


    LThresh lower binary threshold.
    UTHresh upper binary thrshold.

            These two parameters will extract binary images from grayscale 
            images via thresholding if: UThresh is non zero AND SorF is SET. 
            Then all pixels with grey-levels in [Lthresh,UThresh] will be set 
            to WHITE, all others to BLACK. 
            Thresholding occurs before any morphology. This makes it possible 
            to specify binary morphology on a graylevel image sequence. If 
            UThresh > 0 AND LThresh > Uthresh, Morph3DSub aborts with an error. 
            If UThresh is zero no thresholding is performed.
            Note that one can use Morph3DSub to perform a simple threshold on an
            image by specifying MOp = ERODE, ImgType = GRAIMG, AutoSE = AUTO
            sX = 1, sY = 1, sZ = 1, sV=1 and the appropriate LThresh and 
            UThresh.


    SE, sX, sY, sZ, sV, sorgx, sorgy, sorgz

            This is a pointer to the structuring element (SE). Conceptually,
            the SE fits in a rectangular box with dimensions sX, sY, sZ in 
            3-space. Morph3DSub treats the SE like it is a collection of sZ
            rectangles, each with sX elements in the horizontal dimension
            and sY in the vertical. The origin of the SE is specified by
            (sorgx,sorgy,sorgz) where (0,0,0) refers to the front (i.e.
            first rectangle) upper lefthand corner of the box. The origin
            must lie within the enclosing box or Morph3DSub aborts. The SE
            is stored as a list of ints in plane-row-major order. That is
            the first int is the front upper lefthand corner of the SE
            and the last int is the back lower righthand corner of the SE.
            Sequentially through the int list is the first row of the first
            plane of the SE, followed by the second row of the first plane,
            etc., until the first plane is specified, then comes the
            second plane in row-major order, and so on. 

            sZ is a particularly important parameter. Morph3DSub uses this
            value as the z depth of its 3D image buffer.

            The 3D SE scans the image sequence in 3D moving window fashion.
            That is, the SE moves systematically over the entire 3D image
            so that the origin of the SE coincides with each pixel in the
            image exactly once. The way I have set up the program, for each
            position of the moving window scan, consecutive planes of the 3D 
            SE lie in consecutive 2D images from the sequence. To wit, SE 
            planes and image (voxel) planes coincide. At any point in the 
            scan, the output image is in the same sequential position as 
            the image that contains the SE origin.
            
            All seven of the above parameters can be supplied to Morph3DSub
            by the companion routine GetSE(). The functionality of GetSE
            was made separate from MorphSub3D because a calling program
            may need to know the structure of the SE (in particular its 
            z-dimension size) to set up image IO. (E.g.,so it knows which 
            output image corresponds to which input image).

            A more complete description of the structure of the SE is given
            below.


    SEType  specifies whether the structuring element is to be interpreted
            as a gray-level SE or a binary SE. (See below for the distinction.) 
            Its possible values are:

            mnemon    hex      dec        meaning

            GRASE    0x0000      0        gray-level SE
            BINSE    0x0100    256        binary SE


    SorF    Specifies whether the morphological operation is to be of the
            "set" type or "function" type. (See below for the 
            distinction.) Its possible values are:

            mnemon    hex        dec   meaning

            SET      0x0000       0    set operation
            FUNCT    0x0400    1024    function operation


    Rank    This is meaningful only if Mop == RANK. Then the parameter 
            indicates the order of the filter. The rank option of Morph3DSub
            is actually an order statistic filter.  To compute the order,
            Morph3DSub counts the number of pixels in the 3D support of
            the SE. In a gray-level SE that is all values greater than
            or equal to zero. We did not define a hit-or-miss rank filter. 
            Therefore, to compute the support of a binary SE, Morph3DSub 
            changes the zeros in a binary SE into -1's (DON'T CARES). 
            Then all the rank routine considers the support to be all SE 
            elements >= ZERO.

            If Rank == 0, Morph3DSub applies a median filter. If Rank == 1 
            it does a dilation. If Rank == support of SE, then Morph3DSub
            does an erosion. Otherwise, Morph3DSub does an order statistic
            filter of order Rank.



    User supplied I/O routines:

            Routine Morph3DSub operates on a 3D image as a sequence of 2D 
            images. The sequence can be either a time sequence of 2D images 
            or "plane slices" of voxels. Because there are few workstations 
            that can load a large 3D image into memory all at once, Morph3DSub 
            scrolls through the sequence keeping only the minimum number in 
            memory at once. (The minimum number is SZ, the z-dimension of the 
            SE.) 
            
            I wanted Morph3DSub to be independent of any particular image file 
            format. This meant that Morph3DSub could not easily do its own file 
            I/O. However, because of the size of 3D images, the 3D morphology 
            program needs to read images successively from files.
            Thus, I designed Morph3DSub to "call back" to its host program 
            for the next image in the input sequence. Likewise, it calls back
            to output an image. 
            This approach lets a user write a host program to do file I/O
            on any format he likes and call Morph3DSub to operate on them.
            To do this, a user needs to write, along with the host program,
            an image input routine and an image output routine. The user
            passes the addresses of these two "call back procedures" to 
            Morph3DSub during its function call. If ImageIn() is the input 
            procedure and ImageOut() the output procedure that the user has 
            written, Then the host program calls Morph3DSub() like this: 

               Morph3DSub( MorphOp, ImageIn, InParams, ImageOut, OutParams, ...

            InParams and OutParams are pointers to structures that contain
            all the data necessary for ImageIn() and ImageOut() to work, 
            yet about which Morph3DSub() does not need to know.

            Morph3DSub receives the address of the image input routine in
            variable InputImage. It gets the address of the output routine in 
            OutputImage. The local input and output data structure pointers
            are passed in InStruct and OutStruct, respectively.
            Whenever Morph3DSub needs a new image, it calls

               IOError = InputImage( In, NumInReqs, &eof, InStruct );

            Whenever Morph3DSub is ready to output an image, it calls

               IOError = OutputImage( Out, NumOutReqs, OutStruct );

            Morph3DSub expects the call back procedures to have the following
            structure:

                int ImageIn( In, NumReqs, eof, InStruct )
                   byte *In;
                   int NumReqs;
                   int *eof;
                   struct IOS *InStruct;
                {
                1. Figure out name of next input file;

                2. Read next image file and set *eof = FALSE;
                    (If there was an error in the read, return with error.)
                3. If the file could not be read be cause there are no more 
                    images set *eof = TRUE and return with no error; 
                4. Decode image and extract luminance information from image 
                    (lumi. extraction not necessary if image is grayscale);
                5. Copy grayscale image in row major order, one byte per pixel,
                    _TO_ consecutive locations starting at In;
                6. Return with no error.
                }

            and

                int ImageOut( Out, NumReqs, OutStruct )
                   byte *Out;
                   int NumReqs;
                   struct IOS *OutStruct;
                {
                1. Figure out name of next output file;

                2. Copy grayscale image in row major order, one byte per pixel,
                    _FROM_ consecutive locations starting at Out;
                3. Construct output image in appropriate format;

                4. Write image to file;

                5. If write was sucessful, return with no error.
                    (Else return with error.)
                }

            The names of the two callback procedures are defined by the
            user. For simplicity of description, I will assume they
            have the names ImageIn() and ImageOut().

            The first argument, In, of the ImageIn is a pointer to an image 
            array. Morph3DSub allocates this array and passes its address to 
            the ImageIn through this variable. ImageIn reads and decodes an 
            image file. It extracts the luminence component and writes the 
            gray-level pixels (one byte per pixel) in row-major order 
            starting at the address in In.

            NumReqs is the number of input requests that have been made by
            Morph3DSub. When ImageIn is called the first time NumReqs == 1.
            ImageIn, may or may not use this information.

            The third argument, eof, is a pointer to a flag in Morph3DSub.
            ImageIn must write a zero there whenever it inputs an image.
            If there are no more images in the input sequence, ImageIn
            must write a nonzero number there.

            InStruct is a pointer to a structure that includes information
            necessary for ImageIn to work but is not directly needed by 
            Morph3DSub.  InStruct is ImageIn's link back to the host program. 
            For example, InStruct my contain image header and colormap 
            information, filename tags, or information to be conveyed to
            ImageOut.

            The first argument, Out, of the ImageOut is a pointer to an image 
            array. Morph3DSub allocates this array and passes its address to 
            the ImageOut through this variable. ImageOut creates an output
            file incorporating the image in this array.

            NumReqs is the number of output requests that have been made by
            Morph3DSub. When ImageOut is called the first time NumReqs == 1.
            The output routine, ImageOut, may or may not use this information.

            OutStruct is a pointer to a structure that includes information
            necessary for ImageOut to work but is not directly needed by 
            Morph3DSub.  OutStruct is ImageOut's link back to the host program. 
            For example, OutStruct my contain image header and colormap 
            information, filename tags, or information to be conveyed to
            ImageIn. It can be convenient to have OutStruct == InStruct.

            Both callback procedures return a functional value. A zero
            returned indicates normal completion of the routine. A nonzero
            value indicates an I/O error that should abort the entire
            operation.

            If you want to write a host program for Morph3DSub, I highly
            recommend that you look at the example I/O routines InputImage() 
            and OutputImage() in morph3d.c
            morph3d.c is a sun rasterfile based host program for Morph3DSub.


    Structuring element specifications:

            A companion routine, GetSE, loads or creates structuring elements
            for Morph3DSub. It returns a pointer to the SE list. Its calling
            sequence is defined as follows:

            int *GetSE( SEName, AutoSE, SEType, SorF, sX, sY, sZ, sV, 
                        sorgx, sorgy, sorgz )
               char *SEName;                filename of SE file or NULL
               int AutoSE;                  nonzero ==> create SE internally
               int *SEType;                 binary or graylevel
               int *SorF;                   set or function
               int *sX,*sY,*sZ,*sV;         encl box size and max gray level
               int *sorgx,*sorgy,*sorgz;    origin cdts.
            {
            body of routine
            }

            SEName: 

            This is the filename of a SE file. If the environment variable
            SEPATH is defined, the value of it is prepended to the filename
            prior to opening the file.
            If this value is NULL, GetSE() assumes that it is being asked
            to roll its own SE.

            A structuring element file is an ASCII file of integers 
            separated by spaces. The first three numbers, x, y, z are the 
            horizontal, vertical, and depth dimensions in pixels of the 
            smallest 3D box that will cover the structuring element. 
            Numbers x, y and z must be > 0.  The next three numbers, i, j, k 
            are the horizontal, vertical, and depth coordinates, respectively, 
            of the SE origin.  
            IMPORTANT: The origin is expected to be in the covering box.
            If not, MorphSub3D aborts.
            The front upper left hand corner of the box has coordinates 
            (0,0,0); the back lower right is (x-1,y-1z-1). Following the 
            first six integers are x*y*z integers separated by spaces. 
            These numbers are the SE elements. Their interpretation depends 
            on the morphological operation being performed.

            Negative SE elements are ALWAYS treated as logical DON'T CAREs. 
            That is, when the operation is in progress, image pixels under 
            negative SE elements are ignored. Thus, the support of the SE 
            is limited to those elments that are nonnegative. This permits 
            the creation of odd-shaped and multiply connected SE's or the
            placement of the SE origin outside the body of the SE.
            If ImgType is binary, (i.e. pixels grouped as zero and not 
            zero), and if SEType is binary, then the SE is used to perform 
            a hit-or-miss transform. In this case, zero SE elements cover 
            the "miss" support and positive (nonzero) elements cover the 
            "hit" support. The actual gray-levels are ignored.

            If ImgType is binary, and SEType is gray then the nonnegative 
            (both zero and greater than zero) SE elements determine the 
            support of a "hit-only" transform.  That is, the nonnegative 
            suport is used as a standard set-type SE for set (binary) 
            morphology. (Of course, the other gray-level info is ignored.)
            Note: if ImgType is binary, then a set operation is performed
            by default (SorF is ignored).

            The interpretation of the SE elements for ImgType gray depends
            on the flags SEType and SorF:

            SEType  SorF    action

            binary  set     Function-set morphology on support of strictly
                            greater than zero SE elements.
            binary  funct   Same as above.

            gray    set     Function-set morphology on support of nonnegative  
                            (greater than or equal to zero) SE elements.
            gray    funct   Function-function morphology on support of 
                            nonnegative SE elements.

            AutoSE:  

            If this value is zero an SE file is read in. If it is nonzero,
            it tells GetSE() to either use one of the predefined (canned) SEs
            or to create an SE. Here are the possible values for AutoSE
            and their implications.


            mnemon      dec      meaning

            PLUS        1        use 3 by 3 by 3 3D "+" shaped SE
            S3X3X3      2        use 3 by 3 by 3 cube SE
            S5X5X5      3        use 5 by 5 by 5 quasi disk SE
            CONE        4        make a conical SE
            CYLINDER    5        make cylindrical SE
            SPHERE      6        make spherical SE

            The first three SEs are canned SEs. That is they exist as
            data structures in the compiled code of Morph3DSub. The
            other three SEs are built by Morph3DSub to user specification.
            When any of the six are requested SEType is set to gray (SEType 
            == GRASE). If one of the canned SEs is chosen a set operation is 
            performed (SorF == SET) overriding the input parameter.
            If CONE, CYLINDER, or SPHERE is chosen a set operation is 
            performed if sV == 0, and a function operation is performed if
            sV != 0.

            PLUS: This SE has a voxel at its origin and another attached
            to each face of the origin voxel for as total of 7 voxel elements.
            It looks like a "+" in orthographic projection from any x, y or
            z, direction.

            S3X3X3: This is simply a 3x3x3 cube of voxels with origin at
            the center.

            S5X5X5: Extend each face of the 3x3x3 cube outward by one layer
            of voxels and you get this SE. I looks like a fat, short-armed
            3D "+".

            CONE: GetSE makes an elliptical cone as follows: The 
            origin is a single voxel at the center of the SE. The axis
            of the cone is parallel the z-axis (it is perpendicular to 
            the image/voxel planes). Each end of the cone is an sX by sY 
            elliptical disk (circular disk, if sX = sY). sX, sY, and sZ 
            must be odd and greater than or equal to 1. 
            Each end of the cone is (sZ-1)/2 image planes away from the 
            plane of the origin. For example, if sZ = 3, the ends of the 
            cone are in the planes adjacent to the origin plane.  If sZ=5 
            there is one image plane between the end plane and the origin 
            plane at each end. The slope of the cone in the xz plane is 
            sX/sZ and in the yz plane the slope is sY/sZ.
            SE planes between the origin and the ends are ellipical disks
            with semimajor and semiminor axes determined by the slopes.
            The cone is actually an hourglass in shape. This shape can
            track pixels in a time sequence that have a maximum velocity
            in (x,y) given by (sX/sZ,sY/sZ). 
            If a graylevel value, sV > 0, is requested, the voxels along the
            cone axis are given value sV. The graylevels assigned to other
            voxels decrease as the square root of the radius to zero at the 
            edges of the cone.

            CYLINDER: GetSE makes an elliptical cylinder as follows:
            First, it makes an elliptical disk with a diameter of sX pixels 
            horizontally and sY pixels vertically.  
            sX and sY must be odd and greater than or equal to 1. 
            sV is the gray level of the center voxel. If sV > 0 is requested
            the gray level of the voxels decrease as the square root of the 
            radius to zero at the edge of the disk. 
            Then, the disk is copied sZ times. The origin of the SE is
            placed at its very center (center of the disk at sZ/2).
            A cylindrical SE can track shapes in a time sequence larger than 
            its own cross-section that are not moving over sZ frames.
            A cylinder of diameter 1 can track stationary pixels.

            SPHERE: This is a spheroid of dimensions sX by sY by sZ It is 
            made by GetSE in a fashion analogous to the construction of a 
            single identiacal to one slice of the cylinder. If sV is > 0,
            sV is the value of the center pixel and the gray-values decrease
            as the square root of the radius to zero at the edges.
            The graylevel sphere will approximate densities in a 3D voxel
            image.
            
            Note that when AutoSE != 0, the values of SEType and SorF are
            ignored.
*/


/* includes */

#include <ctype.h>
#include <malloc.h>            
#include <math.h>
#include <memory.h>
#include <stdio.h>    
#include <stdlib.h>
#include <strings.h>


/* defs */

#define SFMASK  0x0400     /* set / function mask */
#define SET     0x0000     /* set operation */
#define FUNCT   0x0400     /* function operation */

#define IMMASK  0x0200     /* image type bit of MorphOp */
#define GRAIMG  0x0000     /* gray-level image flag */
#define BINIMG  0x0200     /* binary image flag */

#define SEMASK  0x0100     /* structuring element (SE) type bit */
#define GRASE   0x0000     /* gray-level SE */
#define BINSE   0x0100     /* binary SE */

#define OPMASK  0x007F     /* morph operation type bits of MorphOp */
#define LLMASK  0x001F     /* low-level morph ops mask */
#define LUMASK  0x0007     /* LUM bits mask */
#define ERODE   0x0001     /* erode flag  */
#define DILATE  0x0002     /* dilate flag */
#define OPEN    0x0004     /* open flag   */
#define CLOSE   0x0008     /* close flag  */
#define RANK    0x0010     /* rank filter flag */
#define LUM     0x0011     /* Lower-Upper-Middle filter */
#define LUMSMO  0x0012     /* LUM smoothing filter */
#define LUMSHA  0x0014     /* LUM sharpening filter */
#define MAXMIN  0x0020     /* maxmin flag */
#define MINMAX  0x0040     /* minmax flag */
#define NOTFLG  0x0080     /* "not" flag for binary erodes and dilates */

#define SEPATH  "SEPATH"   /* structuring element directory env. var */
#define INPATH  "IMGPATHI" /* input image directory environment var. */
#define OUTPATH "IMGPATHO" /* output image directory environment var. */
#define FNLGTH  128        /* maximum filename length */
#define IXLGTH    3        /* file index length (must be strictly < 10) */
#define SXLGTH    8        /* maximum number of chars in file name suffix */
#define FTLGTH    8        /* max length of format string for output proc */

#define PLUS      1        /* use 3 by 3 by 3 3D "+" shaped SE */
#define S3X3X3    2        /* use 3 by 3 by 3 cube SE */
#define S5X5X5    3        /* use 5 by 5 by 5 quasi sphere SE */
#define CONE      4        /* make a conical SE */
#define CYLINDER  5        /* make cylindrical SE */
#define SPHERE    6        /* make spherical SE */

#define BYTEPX    1        /* pixels are 1 byte long */
#define WORDPX    2        /* pixels are 2 bytes long */

#define BLACK     0
#define WHITE   255
#define TRUE      1
#define FALSE     0
#define ZERO      0
#define NSIZE   256
#define ERROR    -1
#define NOERR     0

#define MIN(a,b) ((a<b)?(a):(b))
#define MAX(a,b) ((a>b)?(a):(b))

/* typedefs */

#ifndef BYTE
typedef unsigned char byte;
#define BYTE 1
#endif

#ifndef WORD
typedef short int word;
#define WORD
#endif

/* global functions */

extern byte **AllocBuf();
extern void   DeAllocBuf();
extern void   BlockMove();
extern void   OptErr();
extern void   ScrollBuffer();
extern int    SetUpOp();
extern void   ThresholdImg();

extern void BinBinErode();
extern void BinGrayErode();
extern void GrayBinErode();
extern void GrayGraySetErode();
extern void GrayGrayFctErode();
extern void BinBinDilate();
extern void BinGrayDilate();
extern void GrayBinDilate();
extern void GrayGraySetDilate();
extern void GrayGrayFctDilate();

extern void BinSetMaxMin();
extern void GraySetMaxMin();
extern void GrayFctMaxMin();
extern void BinSetMinMax();
extern void GraySetMinMax();
extern void GrayFctMinMax();

extern int  GetSupport();
extern void BinRankFilt();
extern void GrayRankFilt();
extern void LUMfilt();
extern void LUMsmooth();
extern void LUMsharp();


/* global variables */
extern int SEplus[];    /* internal 3x3x3 "+" shaped kernel */
extern int SE3x3x3[];   /* internal 3x3x3 square kernel */
extern int SE5x5x5[];   /* internal 5x5x5 quasi disk kernel */
