 /*
  * Khoros: $Id: ldfiltlp.c,v 1.3 1992/03/20 23:26:17 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: ldfiltlp.c,v 1.3 1992/03/20 23:26:17 dkhoros Exp $";
#endif

 /*
  * $Log: ldfiltlp.c,v $
 * Revision 1.3  1992/03/20  23:26:17  dkhoros
 * VirtualPatch5
 *
  */

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1992, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1992 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: ldfiltlp.c
 >>>>
 >>>>      Program Name: dfiltlp
 >>>>
 >>>> Date Last Updated: Mon Mar  9 20:19:11 1992 
 >>>>
 >>>>          Routines: ldfiltlp - the library call for dfiltlp
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* second order filters: six coefficients */
#define ARCSINH(x) (log((double)x + sqrt((double)x * (double)x + 1.0)))
#define ARCCOSH(x) (log((double)x + sqrt((double)x * (double)x - 1.0)))
#include "vpoly.h"
#define MAXSTAGES 100
static int chbilow(),chbiilow(),btwlow();
/* -library_includes_end */


/****************************************************************
*
* Routine Name: ldfiltlp - library call for dfiltlp
*
* Purpose:
*    
*    1D lowpass frequency domain filter design
*    
*    

* Input:
*    
*    f1             digital cutoff frequency in hertz
*    
*    f2             digital rejection frequency in hertz
*    
*    tolc           magnitude gain at the passband edge.   for  Cheby-
*                   chev  I,  it  also  specifies the magnitude of the
*                   ripple.
*    
*    tolr           magnitude gain at the stopband edge.   for  Cheby-
*                   chev  II,  it  also specifies the magnitude of the
*                   ripple.
*    
*    sfreq          sampling frequency of the system.
*    
*    class          filter class:  0 specifies a Butterworth, 1 speci-
*                   fies  a  Chebychev  I, and 2 specifies a Chebychev
*                   II.
*    
*    

* Output:
*    
*    poly           pointer to a poly_struct that contains a  descrip-
*                   tion of the polynomial
*    
*    Return Value:  1 on success, 0 on failure.
*    
*    

*
* Written By: Jeremy Worley
*    
*    Jeremy Worley 05 Mar 1992 11:06 MST
*              Fixed up implicit declarations to cmul() and cdiv()  in
*              btwlow().
*    
*    Jeremy Worley 09 Mar 1992 19:54 MST
*              Added checks of the return values of some internal rou-
*              tines.
*    
*    

****************************************************************/


/* -library_def */
int ldfiltlp(poly,f1,f2,tolc,tolr,sfreq,class)
    int class;
    float f1,f2,sfreq,tolc,tolr;
    struct poly_struct *poly;
/* -library_def_end */

/* -library_code */
{
    char *program = "ldfiltlp";
    float *a,*b,*c,*d,*e;
    int i,offset=0,stages;
    float wc,wr,epsilon,lambda;

/*
** check for user stupidity
*/

    if(poly==NULL){
       fprintf(stderr,"%s:  polynomial structure must not be NULL\n",program);
       return(0);
    }

    if(f1>=f2){
       fprintf(stderr,"%s:  f1 must be less than f2.\n",program);
       return(0);
    }

    if(f1>0.5*sfreq || f2>0.5*sfreq){
       fprintf(stderr,"%s:  critical frequencies for the filter ",program);
       fprintf(stderr,"must be less than one half of the sampling \n");
       fprintf(stderr,"frequency.\n");
       return(0);
    }

    if(tolr>=1.0 || tolr<=0.0){
       fprintf(stderr,"%s:  rejection tolerance is out of range.\n",program);
       fprintf(stderr,"legal range is between 0.0 and 1.0\n");
       return(0);
    }

    if(tolc>=1.0 || tolc<=0.0){
       fprintf(stderr,"%s:  cutoff tolerance is out of range.\n",program);
       fprintf(stderr,"legal range is between 0.0 and 1.0\n");
       return(0);
    }

    if(tolr>=tolc){
       fprintf(stderr,"%s:  for a lowpass filter, the rejection tolerance.\n",
               program);
       fprintf(stderr,"must be less than the cutoff tolerance.\n");
       return(0);
    }

/*
** allocate memory for junk passed into low level filter routines 
*/

    a = (float *)malloc(MAXSTAGES*sizeof(float));
    if(a==NULL){
       fprintf(stderr,"%s:  [1] memory allocation failed.\n",program);
       return(0);
    }

    b = (float *)malloc(MAXSTAGES*sizeof(float));
    if(b==NULL){
       fprintf(stderr,"%s:  [2] memory allocation failed.\n",program);
       return(0);
    }

    c = (float *)malloc(MAXSTAGES*sizeof(float));
    if(c==NULL){
       fprintf(stderr,"%s:  [3] memory allocation failed.\n",program);
       return(0);
    }

    d = (float *)malloc(MAXSTAGES*sizeof(float));
    if(d==NULL){
       fprintf(stderr,"%s:  [4] memory allocation failed.\n",program);
       return(0);
    }

    e = (float *)malloc(MAXSTAGES*sizeof(float));
    if(e==NULL){
       fprintf(stderr,"%s:  [5] memory allocation failed.\n",program);
       return(0);
    }

/*
*/

   epsilon = sqrt((1.0 - tolc*tolc)/(tolc*tolc));
   lambda = sqrt((1.0 - tolr*tolr)/(tolr*tolr));
   wc = tan((double)XV_PI*f1/sfreq);
   wr = tan((double)XV_PI*f2/sfreq);

/*
** design the filter
*/
    
    switch(class){
       case 0 :  if(!btwlow(wc,wr,epsilon,lambda,a,b,c,d,e,&stages)){
                    fprintf(stderr,"%s: failed call to btwlow.\n",program);
                 }
                 break;
       case 1 :  if(!chbilow(wc,wr,epsilon,lambda,a,b,c,d,e,&stages)){
                    fprintf(stderr,"%s: failed call to chbilow.\n",program);
                 }
                 break;
       case 2 :  if(!chbiilow(wc,wr,epsilon,lambda,a,b,c,d,e,&stages)){
                    fprintf(stderr,"%s: failed call to chbiilow.\n",program);
                 }
                 break;
    }

/*
** now rearrange data into a format that writepoly() can understand
**
** the following code segment assumes that the polynomial for a stage is:
**
**    H(z)  = (a*z^2 + b*z + c)/(z^2 + d*z +e)
**
*/

    for(i=0;i<stages;i++){
        poly->terms[offset].coef = a[i]; poly->terms[offset].expon = 0.0;
        poly->terms[offset+1].coef = b[i]; poly->terms[offset+1].expon = -1.0;
        poly->terms[offset+2].coef = c[i]; poly->terms[offset+2].expon = -2.0;
        poly->terms[offset+3].coef = 1.0;  poly->terms[offset+3].expon = 0.0;
        poly->terms[offset+4].coef = d[i]; poly->terms[offset+4].expon = -1.0;
        poly->terms[offset+5].coef = e[i]; poly->terms[offset+5].expon = -2.0;
        poly->nterms[2*i] = 3; poly->nterms[2*i+1] = 3;
        offset += 6;
    }
    for(i=0;i<stages*6;i++){
        poly->terms[i].delay = 0;
        poly->terms[i].type = _STDTERM;
        if((poly->terms[i].varname = (char *)malloc(2*sizeof(char)))==NULL){
           fprintf(stderr,"%s:  malloc error after filter design\n",program);
           return(0);
        }
        strcpy(poly->terms[i].varname,"z");
    }
    strcpy(poly->indep_var,"z");
    strcpy(poly->func_name,"H");
    poly->stages = stages;

    return(1);
}

/***********************************************************************
*
*  Routine Name: btwlow() 
*
*          Date: Fri Sep  7 15:35:49 MDT 1990
*        
*       Purpose: computes coefficients for a low pass butterworth filter
*                made up of 'stages' stages of 2nd order filters. 
*                The function used for one stage is:  
*
*                       H(z) = (a*z^2 + b*z + c)/(z^2 + d*z + e)
*
*         Input: stages  - number of second order stages
*                wc      - analog cutoff frequency
*
*        Output: a - f   - coefficients as explained above
*
*    Written By: Jeremy Worley
* 
* Modifications:
*
***********************************************************************/

static int btwlow(wc,wr,epsilon,lambda,a,b,c,d,e,nstages)
    int *nstages;
    float wc,wr,epsilon,lambda,*a,*b,*c,*d,*e;
{
    int n,order,stages;
    float ws,beta,zr,zi,sr,si;
    char *program = "btwlow";
    int cmul(), cdiv();

    order = log(lambda/epsilon)/log(wr/wc) + 1;
    if((float)(order/2)!=(float)order/2.0)order+=1;

    stages = order/2;
    *nstages = stages;

    if(stages>MAXSTAGES){
       fprintf(stderr,"%s: Number of stages exceeds maximum (%d > %d).\n",
               program,stages,MAXSTAGES);
       return(0);
    }

    ws    = wc*wc;

    for(n=0;n<stages;n++){
        beta = ((float)(2.0*(n+1) - 1.0 + order)/(float)(2.0*order))*XV_PI;
        cmul(&sr,&si,wc,(float)0.0,(float)cos(beta),(float)sin(beta));
        cdiv(&zr,&zi,(float)(1.0+sr),si,(float)(1.0-sr),-si);

/*
** calculate coefficients.  note that a (and hence b,c) is multiplied by
** wc^stages, which is necessary to bring the gain to 1.0.
*/

        a[n] = ws/(1.0 - 2.0*sr + ws);
        b[n] = 2.0*a[n];
        c[n] = a[n];
        d[n] = -2.0*zr;
        e[n] = zr*zr + zi*zi;
    }

    return(1);
}

/***********************************************************************
*
*  Routine Name: chbilow() 
*
*          Date: Fri Sep  7 15:35:49 MDT 1990
*        
*       Purpose: computes coefficients for a low pass chebychev I filter
*                made up of 'stages' stages of 2nd order filters. 
*                The function used for one stage is:  
*
*                       H(z) = (a*z^2 + b*z + c)/(z^2 + d*z + e)
*
*         Input: stages  - number of second order stages
*                wc      - analog cutoff frequency
*
*        Output: a - f   - coefficients as explained above
*
*    Written By: Jeremy Worley
* 
* Modifications:
*
***********************************************************************/

static int chbilow(wc,wr,epsilon,lambda,a,b,c,d,e,nstages)
    float wc,*a,*b,*c,*d,*e,wr,epsilon,lambda;
    int *nstages;
{
    int n,order,stages;
    float gain,alpha,bn,xn,ynn,rn,qn,temp;
    char *program = "chbilow";

/*
** calculate a suitable order for the filter.  
*/

    order = (int)(ARCCOSH(lambda/epsilon)/ARCCOSH(wr/wc)) + 1;
    if((float)(order/2)!=(float)order/2.0)order+=1;

    stages = order/2;
    *nstages = stages;

    if(stages>MAXSTAGES){
       fprintf(stderr,"%s: Number of stages exceeds maximum (%d > %d).\n",
               program,stages,MAXSTAGES);
       return(0);
    }

    alpha = ARCSINH(1.0/epsilon)/stages;

    for(n=0;n<stages;n++){
        bn = XV_PI*(((float)(2.0*(n+1)-1.0+stages))/(2.0*(float)stages));
        xn = sinh(alpha)*cos(bn);
        ynn = cosh(alpha)*sin(bn);
        rn = wc*wc*(xn*xn + ynn*ynn);
        qn = 2.0*wc*xn;

        gain = 1.0/(rn - qn +1.0);

        a[n] = rn*gain;
        b[n] = 2.0*a[n];
        c[n] = a[n];
        d[n] = 2.0*(rn - 1.0)*gain;
        e[n] = (rn +qn +1.0)*gain;
 
        if(n==0){
           temp = 1.0 + epsilon*epsilon;
           a[n] /= temp;
           b[n] /= temp;
           c[n] /= temp;
        } 
    }
    
    return(1);
}

/***********************************************************************
*
*  Routine Name: chbiilow() 
*
*          Date: Fri Sep  7 15:35:49 MDT 1990
*        
*       Purpose: computes coefficients for a low pass chebychev II filter
*                made up of 'stages' stages of 2nd order filters. 
*                The function used for one stage is:  
*
*                       H(z) = (a*z^2 + b*z + c)/(z^2 + d*z + e)
*
*         Input: stages  - number of second order stages
*                wc      - analog cutoff frequency
*                wr      - analog rejection frequency
*
*        Output: a - f   - coefficients as explained above
*
*    Written By: Jeremy Worley
* 
* Modifications:
*
***********************************************************************/

static int chbiilow(wc,wr,epsilon,lambda,a,b,c,d,e,nstages)
    int *nstages;
    float wc,wr,epsilon,lambda,*a,*b,*c,*d,*e;
{
    int n,order,stages;
    float epslnhat,gain,alpha,bn,xn,ynn,rn,vn,sbn,wrs;
    char *program = "chbiilow";

    order = (int)(ARCCOSH(lambda/epsilon)/ARCCOSH(wr/wc)) + 1;
    if((float)(order/2)!=(float)order/2.0)order+=1;

    stages = order/2;
    *nstages = stages;

    if(stages>MAXSTAGES){
       fprintf(stderr,"%s: Number of stages exceeds maximum (%d > %d).\n",
               program,stages,MAXSTAGES);
       return(0);
    }

    vn = cosh((float)stages*ARCCOSH(wr/wc));
    epslnhat = 1.0/(epsilon*vn);
    alpha = ARCSINH(1.0/epslnhat)/stages;

    for(n=0; n<stages; n++){
        bn = XV_PI*(((float)(2.0*(n+1)-1.0+stages))/(2.0*(float)stages));
        xn = sinh(alpha)*cos(bn);
        ynn = cosh(alpha)*sin(bn);
        rn = xn*xn + ynn*ynn;

        sbn = sin(bn)*sin(bn);
        wrs = wr*wr;

        gain = 1.0/(rn - 2.0*wr*xn + wr*wr);

        a[n] = (sbn + wrs)*gain;
        b[n] = 2.0*(wrs - sbn)*gain;
        c[n] = a[n];
        d[n] = 2.0*(wrs - rn)*gain;
        e[n] = (rn + 2.0*wr*xn + wrs)*gain;
    }

    return(1);
}
/* -library_code_end */
