 /*
  * Khoros: $Id: Stream.c,v 1.3 1992/03/20 23:38:38 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: Stream.c,v 1.3 1992/03/20 23:38:38 dkhoros Exp $";
#endif

 /*
  * $Log: Stream.c,v $
 * Revision 1.3  1992/03/20  23:38:38  dkhoros
 * VirtualPatch5
 *
  */ 


/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>								<<<<
   >>>>	    file name: Stream.c					<<<<
   >>>>								<<<<
   >>>>   description: Khoros pipe (FIFO) stream routines	<<<<
   >>>>								<<<<
   >>>>      routines: stream_tempnam				<<<<
   >>>>      	       stream_open				<<<<
   >>>>		       stream_close				<<<<
   >>>>		       stream_read				<<<<
   >>>>		       stream_write				<<<<
   >>>>		       stream_lseek				<<<<
   >>>>		       stream_tell				<<<<
   >>>>		       stream_access				<<<<
   >>>>		       stream_unlink				<<<<
   >>>>		       stream_lock				<<<<
   >>>>		       stream_descriptors			<<<<
   >>>>								<<<<
   >>>> modifications:						<<<<
   >>>>								<<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


#ifdef KSTREAM

#include "internals.h"	
#include "Stream.h"	



/**************************************************************
*
* MODULE NAME: stream_tempnam
*
*     PURPOSE: This function initializes a tempnam for a "stream"
*	       type transport.  Given a request with a filename of
*	       the following syntax:
*
*			"stream=XXXXXXXXXX"
*
*
*       INPUT: 
*
*        
*      OUTPUT:  depending whether we sucessfully created the template
*      		returns 0 or -1 if an error occurs
*
* CALLED FROM:  internal routine called from ktempnam()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_tempnam(identifier, template, result)

char *identifier;
char *template;
char *result;
{
	char	*temp, filename[LENGTH];


	(void) strcpy(filename, template);
	if ((temp = vtempnam(filename)) == NULL)
	   return(-1);

	/*
	 *  Create the FIFO stream if it doesn't already exist.
	 */
	if (mknod(temp, S_IFIFO | 0666, 0) < 0)
	{
	   if (errno != EEXIST)
	   {
	      perror("stream_tempnam:  Create Failure for stream....");
	      return(-1);
	   }
	}

	if (identifier == NULL)
	   (void) sprintf(result, "stream=%s", temp);
	else
	   (void) sprintf(result,"%s=%s", identifier, temp);

	kfree(temp);
	return(0);
}



/**************************************************************
*
* MODULE NAME: stream_open
*
*     PURPOSE: This function opens a "stream" connection.  It is the
*	       internal driver to open a stream, which is called by
*	       the user when they call kopen().  kopen() calls when
*	       it gets an open request with a filename of the following
*	       syntax:
*
*			"stream=filename"
*
*
*       INPUT: 
*
*        
*      OUTPUT:  returns 0 or -1 if an error occurs
*
* CALLED FROM:  internal routine called from kopen()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_open(path, flags, mode, file)

char *path;
int  flags;
int  mode;
kfile *file;
{
	int	     fid, stream_mode;
	ResourceStruct *resources;


	/*
	 *  Create the FIFO stream if it doesn't already exist.
	 */
	if (!flags & O_CREAT)
	   stream_mode = 0666;
	else
	   stream_mode = mode & 0777;

	if (mknod(path, S_IFIFO | stream_mode, 0) < 0)
	{
	   if (errno != EEXIST)
	   {
	      perror("stream_open:  Create Failure for stream....");
	      return(-1);
	   }
	}

	if ((fid = open(path, flags, mode)) < 0)
	{
	   perror("stream_open:  Open failure....");
	   return(-1);
	}

	if (!(resources = (ResourceStruct *) calloc(1,sizeof(ResourceStruct))))
	{
	   (void) fprintf(stderr,"stream_open:  Not enough memory....\n\n");
	   (void) fprintf(stderr,"  Unable to malloc (%d) bytes for the khoros \
stream structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}

	resources->fid = fid;
	file->resources = (caddr_t) resources;
	return(0);
}



/**************************************************************
*
* MODULE NAME: stream_close
*
*     PURPOSE: This function closes a "stream".  It is the internal driver
*	       to close a data stream, which is called by the user
*	       when they call kclose().  kclose() calls the "stream" internal
*	       drivers by the nature of the transport, which is dictated
*	       by the initial kopen().
*
*
*       INPUT:  file - the kfile structure.
*
*      OUTPUT:  returns whether we were able to close the stream
*
* CALLED FROM:  internal routine called from kclose()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_close(file)

kfile *file;
{
	int	     fid, status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call close() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	status = close(fid);
	return(status);
}



/**************************************************************
*
* MODULE NAME: stream_read
*
*     PURPOSE: This function reads a "stream".  It is the internal driver
*	       to read data from a stream, which is called by the user
*	       when they call kread().  kread() calls the "stream" internal
*	       drivers by the nature of the transport, which is dictated
*	       by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		ptr    - the pointer to store the data into.
*		nbytes - the number of bytes to read.
*
*      OUTPUT:  returns the number of bytes read from the file
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_read(file, ptr, nbytes)

kfile *file;
char  *ptr;
int   nbytes;
{
	int	     fid, count;
	int	     numread = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call read() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	do
	{
	   count = MIN(nbytes - numread, DefaultPageSize);
	   count = read(fid, (char *) (ptr + numread), count);
	   numread += count;
	}
	while (numread < nbytes && count > 0);

	return(numread);
}



/**************************************************************
*
* MODULE NAME: stream_write
*
*     PURPOSE: This function writes to a "stream".  It is the internal driver
*	       to write data from the supplied data array to a stream, which
*	       is called by the user when they call kwrite().  kwrite() calls
*	       the "stream" internal drivers by the nature of the transport,
*	       which is dictated by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		ptr    - the pointer to write the data from.
*		nbytes - the number of bytes to read.
*
*      OUTPUT:  returns the number of bytes written to the kfile
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_write(file, ptr, nbytes)

kfile *file;
char  *ptr;
int   nbytes;
{
	int	     fid, count;
	int	     numwrite = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call write() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	do
	{
	    count = MIN(nbytes - numwrite, DefaultPageSize);
	    count = write(fid, (char *) (ptr + numwrite), count);
	    numwrite += count;
	} while (numwrite < nbytes && count > 0);

	return(numwrite);
}



/**************************************************************
*
* MODULE NAME: stream_lseek
*
*     PURPOSE: This function is used to do a "lseek".  It is the internal
*	       driver to rewind to a specific point so that the data can be
*	       skipped or re-read.  This is called when the user calls
*	       klseek().  klseek() calls the "stream" internal drivers by the
*	       nature of the transport, which is dictated by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		offset - the offset in which to seek
*		whence - the control of how the offset will be applied
*
*      OUTPUT:  returns the number of bytes written to the kfile
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_lseek(file, offset, whence)

kfile *file;
int   offset;
int   whence;
{
	int	     fid, pos;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call lseek() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	pos = lseek(fid, offset, whence);
	return(pos);
}



/**************************************************************
*
* MODULE NAME: stream_tell
*
*     PURPOSE: This function is used to do a "lseek".  It is the internal
*	       driver to rewind to a specific point so that the data can be
*	       skipped or re-read.  This is called when the user calls
*	       ktell().  ktell() calls the "stream" internal drivers by the
*	       nature of the transport, which is dictated by the kopen().
*
*
*       INPUT:  file   - the kfile structure.
*		offset - the offset in which to seek
*		whence - the control of how the offset will be applied
*
*      OUTPUT:  returns the number of bytes written to the kfile
*
* CALLED FROM:  internal routine called from kread()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_tell(file)

kfile *file;
{
	int	     fid, pos;
	ResourceStruct *resources = (ResourceStruct *) file->resources;

	/*
	 *  Simply call tell() since "kfile" is just a regular file
	 *  descriptor.
	 */
	fid = resources->fid;
	pos = tell(fid);
	return(pos);
}



/**************************************************************
*
* MODULE NAME: stream_access
*
*     PURPOSE: This function is used to do see if a file is accessable.
*	       This is called when the user calls kaccess().  kaccess()
*	       calls the "stream" internal drivers by the nature of the
*	       transport, which is dictated by the transport_identifier().
*
*
*       INPUT:  path     - the initial path.
*		filename - the filename that they wish to access
*		mode     - the access mode information
*
*      OUTPUT:  returns the result
*
* CALLED FROM:  internal routine called from kaccess()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_access(path, filename, mode)

char *path;
char *filename;
int  mode;
{
	int	   status;

	/*
	 *  Simply call access() since "filename" is just a stream file
	 */
	status = access(filename, mode);
	return(status);
}



/**************************************************************
*
* MODULE NAME: stream_unlink
*
*     PURPOSE: This function initializes a tempnam for a "stream"
*	       type transport.  Given a request with a filename of
*	       the following syntax:
*
*			"stream=XXXXXXXXXX"
*
*       INPUT: 
*
*        
*      OUTPUT:  returns 0 or -1 depending whether we
*		sucessfully created the template
*
* CALLED FROM:  internal routine called from kunlink()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_unlink(identifier, filename)

char *identifier;
char *filename;
{
	int	status;


	status = unlink(filename);
	return(status);
}



/**************************************************************
*
* MODULE NAME: stream_lock
*
*     PURPOSE: This function locks a "stream" transport.  Given
*	       a request with a filename of the following syntax:
*
*			"stream=filename"
*
*       INPUT: 
*
*        
*      OUTPUT:  returns 0 or -1 depending whether we
*		sucessfully created the template
*
* CALLED FROM:  internal routine called from kflock()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_lock(file, operation)

kfile *file;
int   operation;
{
	int	status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	status = flock(resources->fid, operation);
	return(status);
}



/**************************************************************
*
* MODULE NAME: stream_descriptors
*
*     PURPOSE: This function shouldn't exist, but there is occasion
*	       when the user wants to know the actual file descriptor
*	       associated with the file descriptor.
*
*       INPUT: file structure
*
*        
*      OUTPUT:  inum - the input file descriptor
*
* CALLED FROM:  internal routine called from kdescriptors()
*
*  WRITTEN BY:  Mark Young
*
**************************************************************/


int stream_descriptors(file, inum, onum)

kfile *file;
int   *inum, *onum;
{
	int	status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	if (inum != NULL) *inum = resources->fid;
	if (onum != NULL) *onum = resources->fid;
	return(0);
}

#endif  /* KSTREAM */
/* don`t add after the endif */
