/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/********************************************************************
*
*  File: storeseg.c (storage.c substitute)
*
*  Purpose:  File defining details of storage implementation.
*            All machine-dependent gory details should be here
*            (purely private details) or in storeseg.h
*            (for inclusion in other source files that need to know).
*            
*      This version has element ids implemented as longs.
*      This version for segmented architectures.
*/

#include "include.h"

/* individual dimension block pointer arrays, handy for debugging */
char **vbase;
char **ebase;
char **fbase;
char **bbase;
char **febase;

/* unified block references, indexed by element type */
char **base[NUMELEMENTS];

#define NULLID 0L 

struct web web = {
{ { VERTEX, NULL, 0, NULLID, NULLID, 0, NULLID },
 { EDGE  , NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACET , NULL, 0, NULLID, NULLID, 0, NULLID },
 { BODY  , NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACETEDGE, NULL, 0, NULLID, NULLID, 0, NULLID } },
 SOAPFILM,LINEAR
    };

#undef NULLID 

element_id
  NULLID      =  0xF << TYPESHIFT, 
  NULLVERTEX  =  VERTEX  << TYPESHIFT, 
  NULLEDGE    =  EDGE  << TYPESHIFT,
  NULLFACET   =  FACET  << TYPESHIFT,
  NULLBODY    =  BODY  << TYPESHIFT,  
  NULLFACETEDGE =  FACETEDGE << TYPESHIFT;

#ifdef TCPP
struct element *elptr(element_id id)
#else
struct element *elptr(id)
element_id id;
#endif
{
  int type = id_type(id);
  return (struct element *)(base[type][(id&BLOCKMASK)>>BLOCKSHIFT]
						   + (id & OFFSETMASK));
}

struct vertex *verptr(element_id id)
{ return (struct vertex *)elptr(id); }

void extend(type,number)
int type;
int number;
{
   char *newblock = NULL;
   element_id id;
   struct element *new_el;
   int k;
   int offset;
   int block;

   if ( base[type] == NULL )  
     {  /* first time around */
       char **ptrblock;
       ptrblock = (char **)mycalloc(MAXBLOCK,sizeof(char *));
       base[type] = ptrblock;
       web.skel[type].base = ptrblock;
       switch ( type )
         {
           case VERTEX: vbase = ptrblock; break;
           case EDGE:   ebase = ptrblock; break;
           case FACET:  fbase = ptrblock; break;
           case BODY:   bbase = ptrblock; break;
           case FACETEDGE: febase = ptrblock; break;
         }
     }
  
  for ( ; number > 0 ; number -= BATCHSIZE )
  {
   if ( web.skel[type].blocks >= MAXBLOCK )
     { sprintf(errmsg,"Too many blocks allocated, type = %d\n",type);
       error(errmsg,UNRECOVERABLE);
     }

   newblock = (char *)mycalloc(BATCHSIZE,sizes[type]);
   if ( !newblock )
     { sprintf(errmsg,"Out of memory.\n");
       error(errmsg,UNRECOVERABLE);
     }

   block = web.skel[type].blocks++;
   web.skel[type].base[block] = newblock;

   /* add to freelist */
   offset = 0;
   id = ((long)type << TYPESHIFT) + VALIDMASK + 
                                    ((long)block << BLOCKSHIFT) + offset;
   web.skel[type].free = id;
   for ( k = 0 ; k < BATCHSIZE ; k++ )
     { 
       new_el = (struct element *)(newblock + offset);
       offset += sizes[type];
       id = ((long)type << TYPESHIFT) + VALIDMASK + 
	  ((long)block << BLOCKSHIFT) + offset;
       new_el->forechain = id;
       new_el->ordinal = web.skel[type].max_ord++;  
     }
   new_el->forechain = NULLID;
   web.skel[type].maxcount += BATCHSIZE;
   }

}

element_id new_element(type)
int type;
{
  element_id newid;
  struct element *new_el,*last;
  ORDTYPE ord;
  
  newid = web.skel[type].free;
  
  if ( !valid_id(newid) ) /* free list empty */
   {
     extend(type,BATCHSIZE);
     newid = web.skel[type].free;
   }
  new_el = elptr(newid);

  /* remove from free chain */
  web.skel[type].free = new_el->forechain;

  /* clean out old info */
  ord = new_el->ordinal;
  memset((char *)new_el,0,sizes[type]);
  new_el->ordinal = ord;

  /* add to end of in-use chain */
  new_el->forechain = NULLID;
  new_el->backchain = web.skel[type].last;
  if ( valid_id(web.skel[type].last) )
    {
      last = elptr(web.skel[type].last);    /* find end of in-use chain */
      last->forechain = newid;
    }
  else
    {      
      web.skel[type].used = newid;
    }
  web.skel[type].last = newid;

  new_el->attr = ALLOCATED | NEWELEMENT;
  web.skel[type].count++;  
  return newid;
}

void free_element(id)
element_id id;
{
  struct element *ptr;
  int type = id_type(id);

/* printf("freeing %X  type %d ordinal %d\n",id,type,ordinal(id)); */

  if ( !valid_id(id) )
   {
     sprintf(errmsg,"Trying to free invalid id %08X \n",id);
     error(errmsg,RECOVERABLE);
   }
   
  ptr = elptr(id);
  if ( !(ptr->attr & ALLOCATED) )
   {
     sprintf(errmsg,"Trying to free unallocated element id %08X \n",id);
     error(errmsg,RECOVERABLE);
   }
  ptr->attr = 0;  /* clear attributes */

  /* remove from in-use list */
  if ( valid_id(ptr->forechain) )
    elptr(ptr->forechain)->backchain = ptr->backchain;
  else
    web.skel[type].last = ptr->backchain;

  if ( valid_id(ptr->backchain) )
    elptr(ptr->backchain)->forechain = ptr->forechain;
  else
    web.skel[type].used = ptr->forechain;

   /* add to discard list */
   /* save forechain for generators */
   ptr->backchain = web.skel[type].discard;
   web.skel[type].discard = id & (TYPEMASK | VALIDMASK | OFFSETMASK);
                         /* clear old bits, keep only needed */

  web.skel[type].count--; 

}

int generate_all(type,idptr)  /* re-entrant */
int type;
element_id *idptr;
{
  struct element *ptr;

  if ( !valid_id(*idptr) ) /* first time */
    { *idptr = web.skel[type].used;
      if ( !valid_id(*idptr) ) return 0;
    }
  else
    { ptr = (struct element *)(base[type][(*idptr&BLOCKMASK)>>BLOCKSHIFT]
				+ (*idptr&OFFSETMASK));
      do
	{ /* may have to get off discard list */
          *idptr = ptr->forechain;
          if ( !valid_id(*idptr) ) return 0;
          ptr = (struct element *)(base[type][(*idptr&BLOCKMASK)>>BLOCKSHIFT]
				+ (*idptr&OFFSETMASK));
         }
      while ( !(ptr->attr & ALLOCATED) );
    }
  return 1;
}

void memory_report()
{
   long mem = 0;
   int k;

   mem = 0;
   for ( k = 0 ; k < NUMELEMENTS ; k++ )
     mem += web.skel[k].count*sizes[k];

   sprintf(msg,"Vertices: %ld Edges: %ld Facets: %ld  Facetedges: %ld Memory: %ld\n",
     web.skel[0].count,web.skel[1].count,web.skel[2].count,web.skel[4].count,
     mem);
   outstring(msg);
}

int ordinal(id)
element_id id;
{
  int type = id_type(id);
  if (!valid_id(id)) return 0;
  return  BATCHSIZE*((id&BLOCKMASK)>>BLOCKSHIFT) + (id&OFFSETMASK)/sizes[type];
}

void reset_skeleton()
{
  int i,j;
  struct boundary *bdry;
  struct surf_energy *surfen;
  struct constraint *con;
  struct quantity *quan;

  if ( gauss1Dpt ) { free((char *)gauss1Dpt); gauss1Dpt = NULL; }
  if ( gauss1Dwt ) { free((char *)gauss1Dwt); gauss1Dwt = NULL; }

  /* deallocate boundary specs */
  for ( i = 0, bdry = web.boundaries ; i < BDRYMAX ; i++, bdry++ )
    if ( bdry->coordf[0] )
      {
        for ( j = 0 ; j < web.sdim ; j++ )
	  free_expr(bdry->coordf[j]);
	free((char *)bdry->coordf[0]);
      }

  /* deallocate constraint specs */
  for ( i = 0, con = web.constraints ; i < CONSTRMAX ; i++, con++ )
    constraint_free(con);

  /* deallocate surface energy specs */
  for ( i = 0, surfen = web.surfen ; i < web.surfen_count ; i++, surfen++ )
    { if ( surfen->envect[0] )
        { for ( j = 0 ; j < web.sdim ; j++ )
             free_expr(surfen->envect[j]);
          free((char *)web.surfen[i].envect[0]);
	}
    }

  /* deallocate quantity specs */
  for ( i = 0, quan = web.quants ; i < web.quantity_count ; i++, quan++ )
    { if ( quan->attr & IN_USE )
        { for ( j = 0 ; j < web.sdim ; j++ )
             free_expr(quan->quanvect[j]);
          free((char *)quan->quanvect[0]);
	}
    }

  /* deallocate metric expressions */
  if ( web.metric_flag )
    for ( i = 0 ; i < web.sdim ; i++ )
      for ( j = 0 ; j < web.sdim ; j++ )
	free_expr(&web.metric[i][j]);

  if ( web.skel[0].base )
   for ( i = 0 ; i < NUMELEMENTS ; i++ )
    {
#ifdef MAXBLOCK
      for ( j = 0 ; j < web.skel[i].blocks ; j++ )  
      free((char *)web.skel[i].base[j]);  
#endif
      free((char *)web.skel[i].base);
    }
  memset((char *)base,0,sizeof(base));
  vbase = ebase = fbase = bbase = febase = NULL;

  /* free parameters */
  if ( web.params ) free((char *)web.params);

  if ( web.inverse_periods )  free_matrix(web.inverse_periods);

  memset((char *)&web,0,sizeof(web));  /* total clean out */

}

/*******************************************************************
*
*  function: vgrad_init()
*
*  purpose:  allocates storage for vertex volume gradients
*
*/

volgrad *vgradbase;   /* allocated list of structures */
volgrad **vgptrbase;   /* allocated list of chain start pointers */
int      vgradtop;    /* number of first free  structure */
long      vgradmax;    /* number allocated */

void vgrad_init(qfixed)
int qfixed; /* how many fixed quantities */
{
  long allocsize;

  allocsize = (long)web.skel[VERTEX].maxcount*sizeof(volgrad  *);
  if ( allocsize < MAXALLOC )
   vgptrbase = (volgrad **)temp_calloc(web.skel[VERTEX].maxcount,
                                                   sizeof(volgrad *));
  if ( vgptrbase == NULL )
    {
      sprintf(errmsg,"Cannot allocate %d volgrad pointers of size %d\n",
        web.skel[VERTEX].maxcount, sizeof(volgrad  *));
      error(errmsg,RECOVERABLE);
    }

  if ( web.simplex_flag )  /* may be gross overestimate, but safe */
    vgradmax = web.skel[VERTEX].count*web.skel[BODY].count;
  else if ( web.dimension == STRING )
    vgradmax = 2*web.skel[EDGE].count;
  else /* SOAPFILM */
    /* could have factor of 2 on facets if could guarantee bodies
       next to an edge are separated by a facet, which we can't
       if we permit incomplete bodies. 
     */
    vgradmax = web.skel[FACETEDGE].count - web.skel[FACET].count
             + web.skel[BODY].count + qfixed*web.skel[VERTEX].count + 10;

  if ( web.modeltype == QUADRATIC )
    vgradmax += web.skel[FACETEDGE].count;  /* for midpoints */

  allocsize = vgradmax*sizeof(struct volgrad);
  if ( allocsize < MAXALLOC )
    vgradbase = (struct volgrad *)temp_calloc(vgradmax,sizeof(struct volgrad));
  if ( vgradbase == NULL )
    {
      sprintf(errmsg,"Cannot allocate %ld volgrad structures of size %d\n",
        vgradmax,sizeof(struct volgrad));
      vgrad_end();
      error(errmsg,RECOVERABLE);
    }
  
  vgradtop = 0;

}

void vgrad_end()
{
  if ( vgradbase ) farfree((char *)vgradbase); vgradbase = NULL;
  if ( vgptrbase ) free((char *)vgptrbase); vgptrbase = NULL;
  vgradmax = 0;
}

#ifdef TCPP
volgrad *get_vertex_vgrad(vertex_id v_id)
#else
volgrad *get_vertex_vgrad(v_id)
vertex_id v_id;
#endif
{
  return vgptrbase[ordinal(v_id)];
}

#ifdef TCPP
volgrad *get_next_vgrad(volgrad *vgptr)
#else
volgrad *get_next_vgrad(vgptr)
volgrad *vgptr;
#endif
{
  if ( vgptr ) vgptr = vgptr->chain;
  return vgptr;
}

volgrad  *new_vgrad()
{

  if ( vgradtop >= vgradmax ) 
  /* error */
    {
      vgrad_end();
      sprintf(errmsg,"Too many volgrad structures requested: %d\n", vgradtop);
      error(errmsg,RECOVERABLE);
    }

  return (volgrad *)((struct volgrad huge *)vgradbase + vgradtop++);
}

#ifdef TCPP      
volgrad *get_bv_vgrad(body_id b_id,vertex_id v_id)
#else
volgrad *get_bv_vgrad(b_id,v_id)
body_id b_id;
vertex_id v_id;
#endif
{
  volgrad  *vgptr;

  vgptr = get_vertex_vgrad(v_id);
  while ( vgptr )
   if ( vgptr->b_id == b_id ) break;
   else vgptr = vgptr->chain;

  return vgptr;   /* null if not found */
}
      
#ifdef TCPP      
volgrad *get_bv_new_vgrad(body_id b_id,vertex_id v_id)
#else
volgrad *get_bv_new_vgrad(b_id,v_id)
body_id b_id;
vertex_id v_id;
#endif
{
  volgrad  **ptrptr;  /* pointer to pointer so can update if need be */

  ptrptr = vgptrbase + ordinal(v_id);
  while ( *ptrptr )
   if ( (*ptrptr)->b_id == b_id ) return *ptrptr;
   else ptrptr = &(*ptrptr)->chain;

  /* need to get new structure */
  *ptrptr = new_vgrad();
  (*ptrptr)->b_id = b_id;

  return *ptrptr;
}

void free_discards()
{ int type;

  for ( type = 0 ; type < NUMELEMENTS ; type++ )
    { element_id id = web.skel[type].discard;
      while ( valid_id(id) )
	{ struct element *ptr = elptr(id);
	  web.skel[type].discard = ptr->backchain;
	  ptr->forechain = web.skel[type].free;
	  ptr->backchain = NULLID;
	  web.skel[type].free = id;
	  id = web.skel[type].discard;
	}
    }
}
