/*
    UIT misc file
    Copyright (c) 1993, 1994 Tudor Hulubei & Andrei Pitis

This file is part of UIT (UNIX Interactive Tools)

UIT is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation; either version 2, or (at your option) any later version.

UIT is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details .

You should have received a copy of the GNU General Public License along with
UIT; see the file COPYING.  If not, write to the Free Software Foundation,
675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

#include "xmalloc.h"
#include "config.h"
#include "misc.h"


static char CONFIGFILE_PREFIX[] = "/.uitrc.";
static char lib[] = LIB_DIRECTORY;

char *tty_name;
int  tty_name_len;
char *login_name;
int  login_name_len;


void configuration_help(char *configfile)
{
    fprintf(stderr, "\n%s: cannot open configuration file '%s'.\n",
	    program, configfile);
    fprintf(stderr, "%s: See the manual page for details.\n", program);
    fprintf(stderr, "%s: If the TERM environment variable is, let's say,\n",
	    program);
    fprintf(stderr, "%s: vt102, your configuration file name should be\n",
	    program);
    fprintf(stderr, "%s: '.uitrc.vt102'. Try modifying '.uitrc.console' ...\n",
	    program);
}


void configuration_check(void)
{
    int config_ok;

    char *configfile = xmalloc(strlen(home) + 1 + strlen(CONFIGFILE_PREFIX) +
    			       strlen(tty_type) + 1);
    strcpy(configfile, home);
    strcat(configfile, CONFIGFILE_PREFIX);
    strcat(configfile, tty_type);

    config_ok = configuration_init(configfile);

    if (!config_ok)
    {
	free(configfile);
	configfile = xmalloc(strlen(lib) + 1 + strlen(CONFIGFILE_PREFIX) +
    			     strlen(tty_type) + 1);
	strcpy(configfile, lib);
	strcat(configfile, CONFIGFILE_PREFIX);
	strcat(configfile, tty_type);

	config_ok = configuration_init(configfile);
    }

    if (!config_ok)
    {
	configuration_help(configfile);
	exit(1);
    }
}


void use_section(char *section)
{
    if (configuration_section(section) == -1)
    {
	fprintf(stderr,"%s: can't find section %s in the configuration file.\n",
		program, section);
	exit(1);
    }
}


int get_int_var(char *var_name, char *options[],
		int options_no, int default_value)
{
    int i;
    char *data;

    configuration_getvarinfo(var_name, &data, 1, DO_SEEK);

    if (data)
    {
	for (i = 0; i < options_no; i++)
	    if (strcmp(data, options[i]) == 0)
		break;

	if (i == options_no)
	    fprintf(stderr, "%s: invalid %s (%s).\n", program, var_name, data);
	else
	    return i;
    }

    return default_value;
}


int get_flag_var(char *var_name, int default_value)
{
    char *data;

    configuration_getvarinfo(var_name, &data, 1, DO_SEEK);

    if (data)
    {
	if (strcmp(data, "ON")  == 0) return 1;
	if (strcmp(data, "OFF") == 0) return 0;
	fprintf(stderr, "%s: invalid %s (%s).\n", program, var_name, data);
	return default_value;
    }

    return default_value;
}


char *get_string_var(char *var_name, char *default_value)
{
    char *data;

    configuration_getvarinfo(var_name, &data, 1, DO_SEEK);

    if (data)
    {
	char *string = xmalloc(strlen(data) + 1);
	return strcpy(string, data);
    }

    return default_value;
}


void get_colorset_var(int *charset, char *colorset_name[], int fields_no)
{
    int i, index;
    char *data;

    for (i = 0; i < fields_no; i++)
    {
	configuration_getvarinfo(colorset_name[i], &data, 1, DO_SEEK);

	if (data)
	{
	    index = tty_getcolorindex(data);
	    if (index == -1)
		fprintf(stderr, "%s: invalid %s (%s).\n",
			program, colorset_name[i], data);
	    else
		charset[i] = index;
	}
    }
}


char *clear_path(char *path)
{
    char *cpath = path;
    char *opath = path;

    if (*opath == 0) return path;
    if (*opath == '/') *cpath++ = *opath++;

    while (*opath)
    {
	while (*opath == '/' || 
	       (*opath == '.' && (*(opath + 1) == '/' || *(opath + 1) == 0)))
	    opath++;
	while (*opath && *opath != '/') *cpath++ = *opath++;
	if (*opath) *cpath++ = '/';
    }
    if (*(cpath - 1) == '/' && cpath - path > 1) cpath--;
    *cpath = 0;
    return path;
}


int my_readlink(char *filename)
{
    int size = 100;

    while (1)
    {
	char *buffer = xmalloc(size);
	int  nchars  = readlink(filename, buffer, size);

	if (nchars < size)
	{
	    free(buffer);
	    return nchars;
	}

	free(buffer);
	size *= 2;
    }
}


void get_tty_name(void)
{
    if ((tty_name = ttyname(1)) == NULL)
    {
	fprintf(stderr, "%s: can't get terminal name.\n", program);
	exit(1);
    }

    tty_name_len = strlen(tty_name);
}


void get_login_name(void)
{
    struct passwd *pwd;
    int euid = geteuid();

    if ((pwd = getpwuid(euid)) == NULL)
    {
	fprintf(stderr, "%s: OOOPS, I can't get your user name (euid = %d) !\n", program, euid);
	fprintf(stderr, "%s: Your account no longer exists or you are a SYSTEM CRACKER !! :-)\n", program);
	fprintf(stderr, "%s: Correct the problem and try again.\n", program, program);
	exit(1);
    }

    login_name = xmalloc(strlen(pwd->pw_name) + 1);
    strcpy(login_name, pwd->pw_name);
    login_name_len = strlen(login_name);
}
