/**********************************************************************/
/*                                                                    */
/*	CRISP - Programmable editor                                   */
/*	===========================                                   */
/*                                                                    */
/*  File:          abbrev.cr                                          */
/*  Author:        P. D. Fox                                          */
/*  Created:       8 May 1991                      		      */
/*                                                                    */
/*  Copyright (c) 1990, 1991 Paul Fox                                 */
/*                All Rights Reserved.                                */
/*                                                                    */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*  Description:  Abbreviation mode handling                          */
/*                                                                    */
/**********************************************************************/

/* SCCS ID: %Z% %M% %R%.%L% */

# include	"crisp.h"

/**********************************************************************/
/*   Current keymap for abbreviations keyboard.			      */
/**********************************************************************/
int	ab_keymap;

/**********************************************************************/
/*   Abbreviations currently defined are stored in this list.	      */
/**********************************************************************/
list	abbrev_list;

void
main()
{
	keyboard_push();
	ab_keymap = inq_keyboard();
	assign_to_key("<Space>", "check_abbrev");
	keyboard_pop(1);

	autoload("objects", "default_word_left");	
}
/**********************************************************************/
/*   Macro  called  to  define  a  new  abbreviation  and to turn on  */
/*   abbreviation mode.						      */
/**********************************************************************/
void
abbrev(~string)
{	string	abbr, expr, w;
	int	i;

	if (get_parm(0, abbr, "Abbreviation string: ") <= 0 || abbr == "")
		return;
	/***********************************************/
	/*   If  the  abbreviation  was  typed on the  */
	/*   command  line,  and  it consists of more  */
	/*   than  one  word,  then  these words will  */
	/*   be  passed  as  separate arguments so we  */
	/*   need  to  string  them all back together  */
	/*   again.				       */
	/***********************************************/
	for (i = 1; ; i++) {
		if (get_parm(i, w) <= 0)
			break;
		expr += w + " ";
		}
	if (i == 1) {
		if (get_parm(NULL, expr, "Expansion string: ") <= 0)
			return;
		}
		
	/***********************************************/
	/*   See  if  we  already have definition for  */
	/*   this entry.			       */
	/***********************************************/
	i = re_search(NULL, "<" + quote_regexp(abbr) + ":", abbrev_list);
	if (i < 0)
		abbrev_list += abbr + ":" + trim(expr);
	else
		abbrev_list[i] = abbr + ":" + trim(expr);
	use_local_keyboard(ab_keymap);
}
/**********************************************************************/
/*   Macro to load up a set of abbreviations from a file.	      */
/**********************************************************************/
void
load_abbrev_file(string arg)
{	int	curbuf, buf;
	string	s;
	
	curbuf = inq_buffer();
	buf = create_buffer("-abbrev-", arg, TRUE);
	set_buffer(buf);
	top_of_buffer();
	while (1) {
		if ((s = ltrim(trim(read()))) == "")
			break;
		/***********************************************/
		/*   Allow  for  comments  and  ignore  lines  */
		/*   without ':' in them.		       */
		/***********************************************/
		if (substr(s, 1, 1) != "#" && index(s, ":") > 0)
			abbrev_list += s;
		down();
		}
	set_buffer(curbuf);
	delete_buffer(buf);
	assign_to_key("<Space>", "check_abbrev");
}
/**********************************************************************/
/*   Macro  called  when  <Space>  hit  in buffer with abbreviations  */
/*   active.  Check  previous  word  to  see  if  we  have  a  valid  */
/*   abbreviation.						      */
/**********************************************************************/
void
check_abbrev()
{	string	word, new_word;
	int	i = -1;
	
	save_position();
	drop_anchor(MK_NORMAL);
	default_word_left();
	word = trim(read(inq_mark_size() - 1));
	if (word != "") {
		word = quote_regexp(word) + ":";
		i = re_search(NULL, word, abbrev_list);
		}
	if (i < 0) {
		raise_anchor();
		restore_position();
	        self_insert(' '); 
		}
	else {
		/***********************************************/
		/*   Delete  abbrev  without  using  scrap so  */
		/*   we don't corrupt it.		       */
		/***********************************************/
		delete_char(inq_mark_size() - 1);
		raise_anchor();
		restore_position(0);
		new_word = substr(abbrev_list[i], strlen(word) + 1); 
		insert(new_word);
		}
}
