/*
 * @(#)DirClassMetaDataReader.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;


/**
 * Knows how to operate the repository for the meta-data.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 16, 2002
 */
public class DirClassMetaDataReader implements IClassMetaDataReader
{
    private DirMetaDataIO store;
    private IAnalysisModule am;
    private AnalysisModuleSet ams;
    
    DirClassMetaDataReader( IAnalysisModule module, DirMetaDataIO data )
            throws IOException
    {
        if (data == null || module == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        this.store = data;
        this.am = module;
        this.ams = data.getAnalysisModuleSet();
    }
    
    
    /**
     * Returns the class record for the class with this particular signature.
     */
    public ClassRecord readClass( String classSignature )
            throws IOException
    {
        if (classSignature == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        checkClose();
        return this.store.getClassRecord( this.ams, this.am, classSignature );
    }
    
    
    /**
     * Returns all classes known for this particular data set.
     */
    public String[] getClassSignatures()
            throws IOException
    {
        checkClose();
        return this.store.getClassSigsForAnalysisModule( this.am );
    }
    
    
    /**
     * Closes this reader to prevent further access.
     */
    public void close()
            throws IOException
    {
        checkClose();
        // do not close the store!!
        this.store = null;
    }
    
    
    private void checkClose()
            throws IOException
    {
        if (this.store == null)
        {
            throw new IOException( "Reader has already been closed." );
        }
    }
    
    
    
    protected void finalize() throws Throwable
    {
        super.finalize();
        
        if (this.store != null)
        {
            throw new IllegalStateException("Did not close reader.");
        }
    }
}

