/*
 *  linux/include/asm-arm/arch-ep93xx/memory.h
 *
 *  ******************************************************
 *	*    CONFUSED?  Read Documentation/IO-mapping.txt	 *
 *  ******************************************************
 *
 *
 *  Copyright (C) 1999 ARM Limited
 *  Copyright (C) 2002-2003 Cirrus Logic Corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef __ASM_ARCH_MMU_H
#define __ASM_ARCH_MMU_H

/*
 * For EP93xx, SDRAM can be discontiguous, in a set number of blocks
 * of equal size and (usually) equal spacing.  The 9301 spacing isn't equal.
 *
 * SDRAM_START is the physical address of the start of SDRAM.
 * SDRAM_NUMBER_OF_BLOCKS = # of blocks of SDRAM.
 * Each block is of size SDRAM_BLOCK_SIZE and starts at a boundary
 * of SDRAM_BLOCK_START_BOUNDARY.
 *
 * So memory blocks are at:
 *  SDRAM_START
 *  SDRAM_START + SDRAM_BLOCK_START_BOUNDARY
 *  SDRAM_START + (SDRAM_BLOCK_START_BOUNDARY * 2)
 *  SDRAM_START + (SDRAM_BLOCK_START_BOUNDARY * 3)
 *  so on
 */

#ifndef CONFIG_DISCONTIGMEM

/*
 * Single 32Meg block of physical memory physically located at 0 .
 */
#ifdef CONFIG_MACH_IPD
#define SDRAM_START                         0xc0000000
#define SDRAM_NUMBER_OF_BLOCKS              1
#define SDRAM_BLOCK_SIZE                    0x01000000
#define SDRAM_BLOCK_START_BOUNDARY          0x00000000
#else
#define SDRAM_START                         0x00000000
#define SDRAM_NUMBER_OF_BLOCKS              1
#define SDRAM_BLOCK_SIZE                    0x02000000
#define SDRAM_BLOCK_START_BOUNDARY          0x00000000
#endif

#else /* CONFIG_DISCONTIGMEM */

#ifdef CONFIG_ARCH_EP9301

/*
 * The 9301 memory map doesn't have regular gaps because two
 * address pins aren't connected - see asm-arm/mach-ep93xx/arch.c to
 * see how it is.
 */
#define SDRAM_START                         0x00000000
#define SDRAM_NUMBER_OF_BLOCKS              4
#define SDRAM_BLOCK_SIZE                    0x00800000
#define SDRAM_BLOCK_START_BOUNDARY          0x01000000

#else /* CONFIG_ARCH_EP9312 or CONFIG_ARCH_EP9315 */

/* 
 * 2 32Meg blocks that are located physically at 0 and 64Meg. 
 */
#ifdef CONFIG_MACH_IPD
#define SDRAM_START                         0xc0000000
#define SDRAM_NUMBER_OF_BLOCKS              1
#define SDRAM_BLOCK_SIZE                    0x01000000
#define SDRAM_BLOCK_START_BOUNDARY          0x00000000
#else
#define SDRAM_START                         0x00000000
#define SDRAM_NUMBER_OF_BLOCKS              2
#define SDRAM_BLOCK_SIZE                    0x02000000
#define SDRAM_BLOCK_START_BOUNDARY          0x04000000
#endif

#endif

/*
 * Here we are assuming EP93xx is configured to have two 32MB SDRAM 
 * areas with 32MB of empty space between them.  So use 24 for the node 
 * max shift to get 64MB node sizes.
 */
#define NODE_MAX_MEM_SHIFT	26
#define NODE_MAX_MEM_SIZE	(1<<NODE_MAX_MEM_SHIFT)

#endif /* CONFIG_DISCONTIGMEM */


/*
 * MEM_SIZE and PHYS_OFFSET are used to set size of SDRAM for
 * initial page table in arch/arm/kernel/setup.c
 * For ep93xx, PHYS_OFFSET is set to be SDRAM_START.
 */
#define MEM_SIZE                            (SDRAM_BLOCK_SIZE)

/*
 * If memory is not discontiguous, this is #defined	in 
 * arch/arm/mm/init.c to be 1.
 */
#ifdef CONFIG_DISCONTIGMEM
#define NR_NODES 							(SDRAM_NUMBER_OF_BLOCKS)
#endif

/*
 * Where to load the ramdisk (virtual address, not physical) and how 
 * big to make it. (used in arch/arm/kernel/setup.c
 * In both cases, when redboot loads the ramdisk image to 0x01000000,
 * the processor will find it because the linux map is funny.
 */
#ifdef CONFIG_ARCH_EP9301
#define RAMDISK_START_VIRT      (0xC4000000)
#else
#define RAMDISK_START_VIRT      (0xC1000000)
#endif

/*
 * The ramdisk size comes from a make menuconfig option.
 */
#define RAMDISK_SIZE            ((CONFIG_BLK_DEV_RAM_SIZE)<<10)
 
/*
 * Task size: 2GB (from 0 to base of IO in virtual space)
 */
#define TASK_SIZE		(0x80000000UL)
#define TASK_SIZE_26	(0x04000000UL)

/*
 * This decides where the kernel will search for a free chunk of vm
 * space during mmap's.
 */
#define TASK_UNMAPPED_BASE (TASK_SIZE / 3)

/*
 * Page offset: 3GB (start of kernel memory in virtual space)
 * Phys offset: 0   (start of kernel memory in physical space)
 */
#define PAGE_OFFSET		(0xC0000000UL)
#define PHYS_OFFSET		(SDRAM_START)

/*
 * We take advantage of the fact that physical and virtual address can be the
 * same.  The NUMA code is handling the large holes that might exist between
 * all memory banks.
 */
#define __virt_to_phys__is_a_macro
#define __virt_to_phys(vpage) ((vpage) - PAGE_OFFSET + PHYS_OFFSET)

#define __phys_to_virt__is_a_macro
#define __phys_to_virt(ppage) ((ppage) + PAGE_OFFSET - PHYS_OFFSET) 

/*
 * Virtual view <-> DMA view memory address translations
 * virt_to_bus: Used to translate the virtual address to an
 *              address suitable to be passed to set_dma_addr
 * bus_to_virt: Used to convert an address for DMA operations
 *              to an address that the kernel can use.
 */
#define __virt_to_bus__is_a_macro
#define __virt_to_bus(x)	 __virt_to_phys(x)

#define __bus_to_virt__is_a_macro
#define __bus_to_virt(x)	 __phys_to_virt(x)


/*
 * Note that this file is included by include/asm-arm/memory.h so 
 * the macros in this file have to play nice with those.
 */
#ifdef CONFIG_DISCONTIGMEM

/*
 * Given a kernel address, find the home node of the underlying memory.
 */
#define KVADDR_TO_NID(addr) \
		(((unsigned long)(addr) - PAGE_OFFSET) >> NODE_MAX_MEM_SHIFT)

/*
 * Given a page frame number, convert it to a node id.
 */
#define PFN_TO_NID(pfn) \
	(((pfn) - PHYS_PFN_OFFSET) >> (NODE_MAX_MEM_SHIFT - PAGE_SHIFT))

/*
 * Given a kaddr, ADDR_TO_MAPBASE finds the owning node of the memory
 * and returns the mem_map of that node.
 */
#define ADDR_TO_MAPBASE(kaddr) NODE_MEM_MAP(KVADDR_TO_NID((unsigned long)(kaddr)))

#define PFN_TO_MAPBASE(pfn)	NODE_MEM_MAP(PFN_TO_NID(pfn))

/*
 * Given a kaddr, LOCAL_MAR_NR finds the owning node of the memory
 * and returns the index corresponding to the appropriate page in the
 * node's mem_map.
 */
#define LOCAL_MAP_NR(kaddr) \
	(((unsigned long)(kaddr) & (NODE_MAX_MEM_SIZE - 1)) >> PAGE_SHIFT)

#endif /* CONFIG_DISCONTIGMEM */

#endif

