/*
 * called.p - kernel and demon processes for passive open
 *
 * Written 1996 by Werner Almesberger, EPFL-LRC
 */


#define QSIZE 10	/* kernel to demon queue size - we actually only have
			   very few messages in flight */
#define CONNS 2		/* limit the number of connections we try to set up in
			   order to avoid state explosion */

mtype = { bind,accept,reject,listen,indicate,close,okay,error };

chan d = [QSIZE] of { mtype,chan };	/* demon's input queue (to demux) */
chan k = [0] of { mtype,chan };		/* kernel's input port (to demux) */
chan d_c[CONNS] = [0] of { mtype };	/* demon: demux to connections */
chan k_c[CONNS] = [0] of { mtype };	/* kernel: demux to connections */
chan d_l = [0] of { mtype };		/* demon: demux to listener */
chan k_l = [0] of { mtype };		/* kernel: demux to listener */
chan k_lq = [CONNS] of { mtype };	/* kernel's listen queue */


/*
 * k_conn accepts an incoming connection, "uses" it, and finally closes it or
 * accepts closing from the demon.
 */

proctype k_conn(chan in;chan out)
{
    if
    :: in?error			/* demon says "no" (e.g. it was dropped) */
    :: d!close(out) ->		/* we get impatient and abort */
	if
	:: in?okay ->
	    in?close
	:: in?error
	:: in?close
	fi
    :: in?okay ->		/* connection is ready now */
	if
	:: d!close(out) ->
	    in?close
	:: in?close ->
	    d!close(out)
	fi
    fi
}


/*
 * k_listen registers a listening socket, waits for incoming connection
 * indications, and handles them.
 */

proctype k_listen(chan in)
{
    byte conn;			/* count the connection to select the right */
				/*   port. We're cheating a bit here, but */
    conn = 0;			/*   doing it like the real implementation */
    do				/*   gets messy. */
    :: d!bind(d_l) ->		/* binding is optional, but a good idea */
	if
	:: in?okay
	:: in?error
	fi
    :: d!listen(d_l) -> break	/* request to start listening */
    od;
    if
    :: in?error			/* demon says "no" */
    :: in?okay ->		/* demon agrees */
	do
	:: k_lq?indicate ->	/* an incoming connection */
	    if
	    :: d!accept(d_c[conn]) ->	/* accept */
	        run k_conn(k_c[conn],d_c[conn])
	    :: d!reject(d_c[conn])	/* refuse */
	    fi;
	    conn++
	:: d!close(d_l) ->	/* we get tired and shut down the socket */
	    in?close;		/* wait for the confirmation */
	    do
	    :: nempty(k_lq) ->	/* drain the listen queue */
		k_lq?indicate;
		d!reject(d_c[conn]); /* NB: this goes to the connection proc */
		conn++
	    :: empty(k_lq) -> goto done
	    od;
	od
    fi;
done:
    skip			/* this is the null statement of Promela */
}


/*
 * d_conn handles incoming connections after they have been indicated to the
 * kernel.
 */

proctype d_conn(chan in;chan out)
{
    if
    :: in?reject		/* kernel refuses the connection */
    :: in?accept ->		/* kernel accepts it ... */
	if
	:: k!error(out) ->	/* ... but we refuse */
	    end: in?close	/* catch one stray close */
	:: in?close ->		/* kernel closes before completion */
	    k!close(out)
	:: k!okay(out) ->	/* the connection is ready */
	    if
	    :: k!close(out) ->
		in?close
	    :: in?close ->
		k!close(out)
	    fi
	fi
    fi
}


/*
 * d_listen accepts listen requests from the kernel, sends indications for
 * incoming connections, and launches the processes to handle them.
 */

proctype d_listen(chan in)
{
    byte conn;			/* limit the number of connections */

    conn = 0;
    do
    :: in?bind ->		/* kernel wants to bind */
	if
	:: k!okay(k_l)
	:: k!error(k_l)
	fi
    :: in?listen -> break	/* kernel wants to listen */
    od;
    if
    :: k!error(k_l)		/* demon refuses, e.g. invalid SAP */
    :: k!okay(k_l) ->		/* demon accepts */
	do
	:: (conn < CONNS) ->	/* indicate arrival of a new call */
	    k!indicate(k_l);
	    run d_conn(d_c[conn],k_c[conn]);
	    conn++
	:: in?close->		/* kernel closes the listening socket */
	    k!close(k_l);
	    break
	od
    fi
}


/*
 * demux are the two processes that distribute incoming messages to the
 * *_listen or *_conn processes at either side. In the kernel, also the
 * listen queue is served by demux. (The demon never receives an indicate
 * message, so nothing bad will happen.)
 */

proctype demux(chan in;chan out)
{
    mtype msg;
    chan dst;

end:
    do
    :: in?msg(dst) ->
	if
	:: (msg != indicate) -> dst!msg
	:: (msg == indicate) -> k_lq!msg
	fi
    od
}


init
{
    run d_listen(d_l);
    run demux(d,k);
    run demux(k,d);
    run k_listen(k_l)
}
