/* calibrate.c: default delay calibration
 *
 * Excised from init/main.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  GK 2/5/95  -  Changed to support mounting root fs via NFS
 *  Added initrd & change_root: Werner Almesberger & Hans Lermen, Feb '96
 *  Moan early if gcc is old, avoiding bogus kernels - Paul Gortmaker, May '96
 *  Simplified starting of init:  Michael A. Griffith <grif@acm.org> 
 *
 */

#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/init.h>


/* This is the number of bits of precision for the loops_per_jiffy.  Each
   bit takes on average 1.5/HZ seconds.  This (like the original) is a little
   better than 1% */
#define LPS_PREC 8

void __init calibrate_delay(void)
{
	unsigned long ticks, loopbit;
	int lps_precision = LPS_PREC;

#ifdef FIXED_BOGOMIPS
	int bogus;

/* FIXED_BOGOMIPS converted to __delay units.  */
#define FIXED_LOOPS_PER_JIFFY	(unsigned long)(FIXED_BOGOMIPS * (500000 / HZ))

/* The maximum error in FIXED_LOOPS_PER_JIFFY that we will tolerate.  */
#define FIXED_LPJ_TOLERANCE	(unsigned long)(FIXED_LOOPS_PER_JIFFY * 0.10)

	/* Make sure fixed delay - T% is zero ticks.  */
	ticks = jiffies;
	while (ticks == jiffies) /* Synchronize with start of tick */
		/* nothing */;
	ticks = jiffies;
	__delay(FIXED_LOOPS_PER_JIFFY - FIXED_LPJ_TOLERANCE);
	bogus = (jiffies != ticks);

	if (! bogus) {
		/* Make sure fixed delay + T% is one tick.  The delay here
		   is very short because we're actually continuing timing from
		   the tick synchronization above (we don't resynchronize).  */
		__delay(2 * FIXED_LPJ_TOLERANCE);
		bogus = (jiffies != ticks + 1);
	}
	
	if (! bogus) {
		/* Use the precomputed value.  */
		loops_per_jiffy = FIXED_LOOPS_PER_JIFFY;
		printk("Delay loop constant: %lu.%02lu BogoMIPS (precomputed)\n",
		       (unsigned long)FIXED_BOGOMIPS,
		       ((unsigned long)(FIXED_BOGOMIPS * 100)) % 100);
		return;
	} else {
		printk("Precomputed BogoMIPS value (%lu.%02lu) inaccurate!\n",
		       (unsigned long)FIXED_BOGOMIPS,
		       ((unsigned long)(FIXED_BOGOMIPS * 100)) % 100);
		/* ... and fall through to normal bogomips calculation.  */
	}
#endif /* FIXED_BOGOMIPS */

	loops_per_jiffy = (1<<12);

	printk("Calibrating delay loop... ");
	while (loops_per_jiffy <<= 1) {
		/* wait for "start of" clock tick */
		ticks = jiffies;
		while (ticks == jiffies)
			/* nothing */;
		/* Go .. */
		ticks = jiffies;
		__delay(loops_per_jiffy);
		ticks = jiffies - ticks;
		if (ticks)
			break;
	}

/* Do a binary approximation to get loops_per_jiffy set to equal one clock
   (up to lps_precision bits) */
	loops_per_jiffy >>= 1;
	loopbit = loops_per_jiffy;
	while ( lps_precision-- && (loopbit >>= 1) ) {
		loops_per_jiffy |= loopbit;
		ticks = jiffies;
		while (ticks == jiffies);
		ticks = jiffies;
		__delay(loops_per_jiffy);
		if (jiffies != ticks)	/* longer than 1 tick */
			loops_per_jiffy &= ~loopbit;
	}

/* Round the value and print it */	
	printk("%lu.%02lu BogoMIPS\n",
		loops_per_jiffy/(500000/HZ),
		(loops_per_jiffy/(5000/HZ)) % 100);
}
