/* $XConsortium: UtkasciiSrc.c,v 1.55 91/07/25 18:09:27 rws Exp $ */

/*
 * Copyright 1989 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL M.I.T.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Chris Peterson, MIT X Consortium.
 *
 * Much code taken from X11R3 String and Disk Sources.
 */

/*
 * UtkasciiSrc.c - UtkasciiSrc object. (For use with the text widget).
 *
 */
/*
 *
$Log: UtkasciiSrc.c,v $
 * Revision 1.3  1993/06/16  16:41:30  wade
 * never found weird AIX mpi problem; changed index file instead
 *
 * Revision 1.2  1992/12/05  17:15:51  larose
 * added rcs logging.
 *
 *
 */


#include <X11/IntrinsicP.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <X11/StringDefs.h>
#include <X11/Xos.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xmu/CharSet.h>

#include "UtkasciiTextP.h" /* for Widget Classes. */
#include "UtkasciiSrcP.h"


/****************************************************************
 *
 * Full class record constant
 *
 ****************************************************************/

/* Private Data */

static int magic_value = MAGIC_VALUE;

#define offset(field) XtOffsetOf(UtkasciiSrcRec, utkascii_src.field)

static XtResource resources[] = {
    {XtNstring, XtCString, XtRString, sizeof (char *),
       offset(string), XtRString, NULL},
    {XtNtype, XtCType, XtRUtkasciiType, sizeof (XawUtkasciiType),
       offset(type), XtRImmediate, (XtPointer)XawUtkasciiString},
    {XtNdataCompression, XtCDataCompression, XtRBoolean, sizeof (Boolean),
       offset(data_compression), XtRImmediate, (XtPointer) TRUE},
    {XtNpieceSize, XtCPieceSize, XtRInt, sizeof (XawTextPosition),
       offset(piece_size), XtRImmediate, (XtPointer) BUFSIZ},
    {XtNcallback, XtCCallback, XtRCallback, sizeof(XtPointer), 
       offset(callback), XtRCallback, (XtPointer)NULL},
    {XtNgetAllButton, XtCGetAllButton, XtRBoolean, sizeof (Boolean),
       offset(get_all_button), XtRImmediate, (XtPointer) FALSE},
    {XtNuseStringInPlace, XtCUseStringInPlace, XtRBoolean, sizeof (Boolean),
       offset(use_string_in_place), XtRImmediate, (XtPointer) FALSE},
    {XtNlength, XtCLength, XtRInt, sizeof (int),
       offset(utkascii_length), XtRInt, (XtPointer) &magic_value},

#ifdef ASCII_DISK
    {XtNfile, XtCFile, XtRString, sizeof (String),
       offset(filename), XtRString, NULL},
#endif /* ASCII_DISK */
};
#undef offset

static XawTextPosition Scan(), Search(), ReadText();
static int ReplaceText();
static Piece * FindPiece(), * AllocNewPiece();
static FILE * InitStringOrFile();
static void FreeAllPieces(), RemovePiece(), BreakPiece(), LoadPieces();
static void RemoveOldStringOrFile(),  CvtStringToUtkasciiType();
static void ClassInitialize(), Initialize(), Destroy(), GetValuesHook();
static String MyStrncpy(), StorePiecesInString();
static Boolean SetValues(), WriteToFile();
extern int errno, sys_nerr;
extern char* sys_errlist[];

#define superclass		(&textSrcClassRec)
UtkasciiSrcClassRec utkasciiSrcClassRec = {
  {
/* core_class fields */	
    /* superclass	  	*/	(WidgetClass) superclass,
    /* class_name	  	*/	"UtkasciiSrc",
    /* widget_size	  	*/	sizeof(UtkasciiSrcRec),
    /* class_initialize   	*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited       	*/	FALSE,
    /* initialize	  	*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize		  	*/	NULL,
    /* actions		  	*/	NULL,
    /* num_actions	  	*/	0,
    /* resources	  	*/	resources,
    /* num_resources	  	*/	XtNumber(resources),
    /* xrm_class	  	*/	NULLQUARK,
    /* compress_motion	  	*/	FALSE,
    /* compress_exposure  	*/	FALSE,
    /* compress_enterleave	*/	FALSE,
    /* visible_interest	  	*/	FALSE,
    /* destroy		  	*/	Destroy,
    /* resize		  	*/	NULL,
    /* expose		  	*/	NULL,
    /* set_values	  	*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	NULL,
    /* get_values_hook		*/	GetValuesHook,
    /* accept_focus	 	*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private   	*/	NULL,
    /* tm_table		   	*/	NULL,
    /* query_geometry		*/	NULL,
    /* display_accelerator	*/	NULL,
    /* extension		*/	NULL
  },
/* textSrc_class fields */
  {
    /* Read                     */      ReadText,
    /* Replace                  */      ReplaceText,
    /* Scan                     */      Scan,
    /* Search                   */      Search,
    /* SetSelection             */      XtInheritSetSelection,
    /* ConvertSelection         */      XtInheritConvertSelection
  },
/* utkasciiSrc_class fields */
  {
    /* Keep the compiler happy */       NULL
  }
};

WidgetClass utkasciiSrcObjectClass = (WidgetClass)&utkasciiSrcClassRec;

/************************************************************
 *
 * Semi-Public Interfaces.
 *
 ************************************************************/

/*      Function Name: ClassInitialize
 *      Description: Class Initialize routine, called only once.
 *      Arguments: none.
 *      Returns: none.
 */

static void
ClassInitialize()
{
  XawInitializeWidgetSet();
  XtAddConverter( XtRString, XtRUtkasciiType, CvtStringToUtkasciiType,
		 NULL, (Cardinal) 0);
}

/*      Function Name: Initialize
 *      Description: Initializes the simple menu widget
 *      Arguments: request - the widget requested by the argument list.
 *                 new     - the new widget with both resource and non
 *                           resource values.
 *      Returns: none.
 */

/* ARGSUSED */
static void
Initialize(request, new)
Widget request, new;
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) new;
  FILE * file;

/*
 * Set correct flags (override resources) depending upon widget class.
 */

#ifdef ASCII_DISK
  if (XtIsSubclass(XtParent(new), utkasciiDiskWidgetClass)) {
    src->utkascii_src.type = XawUtkasciiFile;
    src->utkascii_src.string = src->utkascii_src.filename;
  }
#endif

#ifdef ASCII_STRING
  if (XtIsSubclass(XtParent(new), utkasciiStringWidgetClass)) {
    src->utkascii_src.use_string_in_place = TRUE;
    src->utkascii_src.type = XawUtkasciiString;
  }
#endif

  src->utkascii_src.changes = FALSE;
  src->utkascii_src.allocated_string = FALSE;

  file = InitStringOrFile(src, src->utkascii_src.type == XawUtkasciiFile);
  LoadPieces(src, file, NULL);

  if (file != NULL) fclose(file);
}

/*	Function Name: ReadText
 *	Description: This function reads the source.
 *	Arguments: w - the UtkasciiSource widget.
 *                 pos - position of the text to retreive.
 * RETURNED        text - text block that will contain returned text.
 *                 length - maximum number of characters to read.
 *	Returns: The number of characters read into the buffer.
 */

static XawTextPosition
ReadText(w, pos, text, length)
Widget w;
XawTextPosition pos;
XawTextBlock *text;	
int length;		
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;
  XawTextPosition count, start;
  Piece * piece = FindPiece(src, pos, &start);
    
  text->firstPos = pos;
  text->ptr = piece->text + (pos - start);
  count = piece->used - (pos - start);
  text->length = (length > count) ? count : length;
  return(pos + text->length);
}

/*	Function Name: ReplaceText.
 *	Description: Replaces a block of text with new text.
 *	Arguments: w - the UtkasciiSource widget.
 *                 startPos, endPos - ends of text that will be removed.
 *                 text - new text to be inserted into buffer at startPos.
 *	Returns: XawEditError or XawEditDone.
 */

/*ARGSUSED*/
static int 
ReplaceText (w, startPos, endPos, text)
Widget w;
XawTextPosition startPos, endPos;
XawTextBlock *text;
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;
  Piece *start_piece, *end_piece, *temp_piece;
  XawTextPosition start_first, end_first;
  int length, firstPos;

/*
 * Editing a read only source is not allowed.
 */

  if (src->text_src.edit_mode == XawtextRead) 
    return(XawEditError);

  start_piece = FindPiece(src, startPos, &start_first);
  end_piece = FindPiece(src, endPos, &end_first);

  src->utkascii_src.changes = TRUE; /* We have changed the buffer. */

/* 
 * Remove Old Stuff. 
 */

  if (start_piece != end_piece) {
    temp_piece = start_piece->next;

/*
 * If empty and not the only piece then remove it. 
 */

    if ( ((start_piece->used = startPos - start_first) == 0) &&
	 !((start_piece->next == NULL) && (start_piece->prev == NULL)) )
      RemovePiece(src, start_piece);

    while (temp_piece != end_piece) {
      temp_piece = temp_piece->next;
      RemovePiece(src, temp_piece->prev);
    }
    end_piece->used -= endPos - end_first;
    if (end_piece->used != 0)
      MyStrncpy(end_piece->text, (end_piece->text + endPos - end_first),
		(int) end_piece->used);
  }
  else {			/* We are fully in one piece. */
    if ( (start_piece->used -= endPos - startPos) == 0) {
      if ( !((start_piece->next == NULL) && (start_piece->prev == NULL)) )
	RemovePiece(src, start_piece);
    }
    else {
      MyStrncpy(start_piece->text + (startPos - start_first),
		start_piece->text + (endPos - start_first),
		(int) (start_piece->used - (startPos - start_first)) );
      if ( src->utkascii_src.use_string_in_place && 
	   ((src->utkascii_src.length - (endPos - startPos)) < 
	    (src->utkascii_src.piece_size - 1)) ) 
	start_piece->text[src->utkascii_src.length - (endPos - startPos)] = '\0';
    }
  }

  src->utkascii_src.length += -(endPos - startPos) + text->length;

  if ( text->length != 0) {

    /* 
     * Put in the New Stuff.
     */
    
    start_piece = FindPiece(src, startPos, &start_first);
    
    length = text->length;
    firstPos = text->firstPos;
    
    while (length > 0) {
      char * ptr;
      int fill;
      
      if (src->utkascii_src.use_string_in_place) {
	if (start_piece->used == (src->utkascii_src.piece_size - 1)) {
	  /*
	   * If we are in utkascii string emulation mode. Then the
	   *  string is not allowed to grow.
	   */
	  start_piece->used = src->utkascii_src.length = 
	                                         src->utkascii_src.piece_size - 1;
	  start_piece->text[src->utkascii_src.length] = '\0';
	  return(XawEditError);
	}
      }


      if (start_piece->used == src->utkascii_src.piece_size) {
	BreakPiece(src, start_piece);
	start_piece = FindPiece(src, startPos, &start_first);
      }

      fill = Min((int)(src->utkascii_src.piece_size - start_piece->used), length);
      
      ptr = start_piece->text + (startPos - start_first);
      MyStrncpy(ptr + fill, ptr, 
		(int) start_piece->used - (startPos - start_first));
      strncpy(ptr, text->ptr + firstPos, fill);
      
      startPos += fill;
      firstPos += fill;
      start_piece->used += fill;
      length -= fill;
    }
  }

  if (src->utkascii_src.use_string_in_place)
    start_piece->text[start_piece->used] = '\0';

  XtCallCallbacks(w, XtNcallback, NULL); /* Call callbacks, we have changed 
					    the buffer. */

  return(XawEditDone);
}

/*	Function Name: Scan
 *	Description: Scans the text source for the number and type
 *                   of item specified.
 *	Arguments: w - the UtkasciiSource widget.
 *                 position - the position to start scanning.
 *                 type - type of thing to scan for.
 *                 dir - direction to scan.
 *                 count - which occurance if this thing to search for.
 *                 include - whether or not to include the character found in
 *                           the position that is returned. 
 *	Returns: the position of the item found.
 *
 * Note: While there are only 'n' characters in the file there are n+1 
 *       possible cursor positions (one before the first character and
 *       one after the last character.
 */

static 
XawTextPosition 
Scan (w, position, type, dir, count, include)
Widget                w;
XawTextPosition       position;
XawTextScanType       type;
XawTextScanDirection  dir;
int     	      count;
Boolean	              include;
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;
  register int inc;
  Piece * piece;
  XawTextPosition first, first_eol_position;
  register char * ptr;

  if (type == XawstAll) {	/* Optomize this common case. */
    if (dir == XawsdRight)
      return(src->utkascii_src.length);
    return(0);			/* else. */
  }

  if (position > src->utkascii_src.length)
    position = src->utkascii_src.length;

  if ( dir == XawsdRight ) {
    if (position == src->utkascii_src.length)
/*
 * Scanning right from src->utkascii_src.length???
 */
      return(src->utkascii_src.length);
    inc = 1;
  }
  else {
    if (position == 0)
      return(0);		/* Scanning left from 0??? */
    inc = -1;
    position--;
  }

  piece = FindPiece(src, position, &first);

/*
 * If the buffer is empty then return 0. 
 */

  if ( piece->used == 0 ) return(0); 

  ptr = (position - first) + piece->text;

  switch (type) {
  case XawstEOL: 
  case XawstParagraph: 
  case XawstWhiteSpace: 
    for ( ; count > 0 ; count-- ) {
      Boolean non_space = FALSE, first_eol = TRUE;
      while (TRUE) {
	register unsigned char c = *ptr;

	ptr += inc;
	position += inc;
	
	if (type == XawstWhiteSpace) {
	  if (isspace(c)) {
	    if (non_space) 
	      break;
	  }
	  else
	    non_space = TRUE;
	}
	else if (type == XawstEOL) {
	  if (c == '\n') break;
	}
	else { /* XawstParagraph */
	  if (first_eol) {
	    if (c == '\n') {
	      first_eol_position = position;
	      first_eol = FALSE;
	    }
	  }
	  else
	    if ( c == '\n') 
	      break;
	    else if ( !isspace(c) )
	      first_eol = TRUE;
	}
	      

	if ( ptr < piece->text ) {
	  piece = piece->prev;
	  if (piece == NULL)	/* Begining of text. */
	    return(0);
	  ptr = piece->text + piece->used - 1;
	}
	else if ( ptr >= (piece->text + piece->used) ) {
	  piece = piece->next;
	  if (piece == NULL)	/* End of text. */
	    return(src->utkascii_src.length);
	  ptr = piece->text;
	}
      }
    }
    if (!include) {
      if ( type == XawstParagraph)
	position = first_eol_position;
      position -= inc;
    }
    break;
  case XawstPositions: 
    position += count * inc;
    break;
/*  case XawstAll:		---- handled in special code above */
  }

  if ( dir == XawsdLeft )
    position++;

  if (position >= src->utkascii_src.length)
    return(src->utkascii_src.length);
  if (position < 0)
    return(0);

  return(position);
}

/*	Function Name: Search
 *	Description: Searchs the text source for the text block passed
 *	Arguments: w - the UtkasciiSource Widget.
 *                 position - the position to start scanning.
 *                 dir - direction to scan.
 *                 text - the text block to search for.
 *	Returns: the position of the item found.
 */

static XawTextPosition 
Search(w, position, dir, text)
Widget                w;
XawTextPosition       position;
XawTextScanDirection  dir;
XawTextBlock *        text;
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;
  register int inc, count = 0;
  register char * ptr;
  Piece * piece;
  char * buf;
  XawTextPosition first;

  if ( dir == XawsdRight )
    inc = 1;
  else {
    inc = -1;
    if (position == 0)
      return(XawTextSearchError);	/* scanning left from 0??? */
    position--;
  }

  buf = XtMalloc(sizeof(unsigned char) * text->length);
  strncpy(buf, (text->ptr + text->firstPos), text->length);
  piece = FindPiece(src, position, &first);
  ptr = (position - first) + piece->text;

  while (TRUE) {
    if (*ptr == ((dir == XawsdRight) ? *(buf + count) 
		                     : *(buf + text->length - count - 1)) ) {
      if (count == (text->length - 1))
	break;
      else
	count++;
    }
    else {
      if (count != 0) {
	position -=inc * count;
	ptr -= inc * count;
      }
      count = 0;
    }

    ptr += inc;
    position += inc;
    
    while ( ptr < piece->text ) {
      piece = piece->prev;
      if (piece == NULL) {	/* Begining of text. */
	XtFree(buf);
	return(XawTextSearchError);
      }
      ptr = piece->text + piece->used - 1;
    }
   
    while ( ptr >= (piece->text + piece->used) ) {
      piece = piece->next;
      if (piece == NULL) {	/* End of text. */
	XtFree(buf);
	return(XawTextSearchError);
      }
      ptr = piece->text;
    }
  }

  XtFree(buf);
  if (dir == XawsdLeft)
    return(position);
  return(position - (text->length - 1));
}

/*	Function Name: SetValues
 *	Description: Sets the values for the UtkasciiSource.
 *	Arguments: current - current state of the widget.
 *                 request - what was requested.
 *                 new - what the widget will become.
 *	Returns: True if redisplay is needed.
 */

/* ARGSUSED */
static Boolean
SetValues(current, request, new, args, num_args)
Widget current, request, new;
ArgList args;
Cardinal * num_args;
{
  UtkasciiSrcObject src =      (UtkasciiSrcObject) new;
  UtkasciiSrcObject old_src = (UtkasciiSrcObject) current;
  Boolean total_reset = FALSE, string_set = FALSE;
  FILE * file;
  int i;

  if ( old_src->utkascii_src.use_string_in_place != 
       src->utkascii_src.use_string_in_place ) {
      XtAppWarning( XtWidgetToApplicationContext(new),
	   "UtkasciiSrc: The XtNuseStringInPlace resource may not be changed.");
       src->utkascii_src.use_string_in_place = 
	   old_src->utkascii_src.use_string_in_place;
  }

  for (i = 0; i < *num_args ; i++ ) 
      if (streq(args[i].name, XtNstring)) {
	  string_set = TRUE;
	  break;
      }
  
  if ( string_set || (old_src->utkascii_src.type != src->utkascii_src.type) ) {
    RemoveOldStringOrFile(old_src, string_set); /* remove old info. */
    file = InitStringOrFile(src, string_set);	/* Init new info. */
    LoadPieces(src, file, NULL);    /* load new info into internal buffers. */
    if (file != NULL) fclose(file);
    XawTextSetSource( XtParent(new), new, 0);   /* Tell text widget
						   what happened. */
    total_reset = TRUE;
  }

  if ( old_src->utkascii_src.utkascii_length != src->utkascii_src.utkascii_length ) 
      src->utkascii_src.piece_size = src->utkascii_src.utkascii_length;

  if ( !total_reset && 
      (old_src->utkascii_src.piece_size != src->utkascii_src.piece_size) ) {
      String string = StorePiecesInString(old_src);
      FreeAllPieces(old_src);
      LoadPieces(src, NULL, string);
      XtFree(string);
  }

  return(FALSE);
}

/*	Function Name: GetValuesHook
 *	Description: This is a get values hook routine that sets the
 *                   values specific to the utkascii source.
 *	Arguments: w - the UtkasciiSource Widget.
 *                 args - the argument list.
 *                 num_args - the number of args.
 *	Returns: none.
 */

static void
GetValuesHook(w, args, num_args)
Widget w;
ArgList args;
Cardinal * num_args;
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;
  register int i;

  if (src->utkascii_src.type == XawUtkasciiString) {
    for (i = 0; i < *num_args ; i++ ) 
      if (streq(args[i].name, XtNstring)) {
	  if (src->utkascii_src.use_string_in_place) {
	      *((char **) args[i].value) = src->utkascii_src.first_piece->text;
	  }
	  else {
	      if (XawUtkasciiSave(w))	/* If save sucessful. */
		  *((char **) args[i].value) = src->utkascii_src.string;
	  }
	break;
      }
  }
}    

/*	Function Name: Destroy
 *	Description: Destroys an utkascii source (frees all data)
 *	Arguments: src - the Utkascii source Widget to free.
 *	Returns: none.
 */

static void 
Destroy (w)
Widget w;
{
  RemoveOldStringOrFile((UtkasciiSrcObject) w, True);
}

/************************************************************
 *
 * Public routines 
 *
 ************************************************************/

/*	Function Name: XawUtkasciiSourceFreeString
 *	Description: Frees the string returned by a get values call
 *                   on the string when the source is of type string.
 *	Arguments: w - the UtkasciiSrc widget.
 *	Returns: none.
 */

void
#if NeedFunctionPrototypes
XawUtkasciiSourceFreeString(Widget w)
#else
XawUtkasciiSourceFreeString(w)
Widget w;
#endif
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;

  if (src->utkascii_src.allocated_string && src->utkascii_src.type != XawUtkasciiFile) {
    src->utkascii_src.allocated_string = FALSE;
    XtFree(src->utkascii_src.string);
    src->utkascii_src.string = NULL;
  }
}

/*	Function Name: XawUtkasciiSave
 *	Description: Saves all the pieces into a file or string as required.
 *	Arguments: w - the utkasciiSrc Widget.
 *	Returns: TRUE if the save was successful.
 */

Boolean
#if NeedFunctionPrototypes
XawUtkasciiSave(Widget w)
#else
XawUtkasciiSave(w)
Widget w;
#endif
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;

/*
 * If using the string in place then there is no need to play games
 * to get the internal info into a readable string.
 */

  if (src->utkascii_src.use_string_in_place) 
    return(TRUE);

  if (src->utkascii_src.type == XawUtkasciiFile) {
    char * string;

    if (!src->utkascii_src.changes) 		/* No changes to save. */
      return(TRUE);

    string = StorePiecesInString(src);

    if (WriteToFile(string, src->utkascii_src.string) == FALSE) {
      XtFree(string);
      return(FALSE);
    }
    XtFree(string);
  }
  else {			/* This is a string widget. */
    if (src->utkascii_src.allocated_string == TRUE) 
      XtFree(src->utkascii_src.string);
    else
      src->utkascii_src.allocated_string = TRUE;
    
    src->utkascii_src.string = StorePiecesInString(src);
  }
  src->utkascii_src.changes = FALSE;
  return(TRUE);
}

/*	Function Name: XawUtkasciiSaveAsFile
 *	Description: Save the current buffer as a file.
 *	Arguments: w - the UtkasciiSrc widget.
 *                 name - name of the file to save this file into.
 *	Returns: True if the save was sucessful.
 */

Boolean
#if NeedFunctionPrototypes
XawUtkasciiSaveAsFile(Widget w, _Xconst char* name)
#else
XawUtkasciiSaveAsFile(w, name)
Widget w;
String name;
#endif
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) w;
  String string;
  Boolean ret;

  string = StorePiecesInString(src); 

  ret = WriteToFile(string, name);
  XtFree(string);
  return(ret);
}

/*	Function Name: XawUtkasciiSourceChanged
 *	Description: Returns true if the source has changed since last saved.
 *	Arguments: w - the utkascii source widget.
 *	Returns: a Boolean (see description).
 */

Boolean 
#if NeedFunctionPrototypes
XawUtkasciiSourceChanged(Widget w)
#else
XawUtkasciiSourceChanged(w)
Widget w;
#endif
{
  return( ((UtkasciiSrcObject) w)->utkascii_src.changes );
}
  
/************************************************************
 *
 * Private Functions.
 *
 ************************************************************/

static void
RemoveOldStringOrFile(src, checkString) 
UtkasciiSrcObject src;
Boolean checkString;
{
  FreeAllPieces(src);

  if (checkString && src->utkascii_src.allocated_string) {
    XtFree(src->utkascii_src.string);
    src->utkascii_src.allocated_string = False;
    src->utkascii_src.string = NULL;
  }
}

/*	Function Name: WriteToFile
 *	Description: Write the string specified to the begining of the file
 *                   specified.
 *	Arguments: string - string to write.
 *                 name - the name of the file
 *	Returns: returns TRUE if sucessful, FALSE otherwise.
 */

static Boolean
WriteToFile(string, name)
String string, name;
{
  int fd;
  
  if ( ((fd = creat(name, 0666)) == -1 ) ||
       (write(fd, string, sizeof(unsigned char) * strlen(string)) == -1) )
    return(FALSE);

  if ( close(fd) == -1 ) 
    return(FALSE);

  return(TRUE);
}

/*	Function Name: StorePiecesInString
 *	Description: store the pieces in memory into a standard utkascii string.
 *	Arguments: data - the utkascii pointer data.
 *	Returns: none.
 */

static String
StorePiecesInString(src)
UtkasciiSrcObject src;
{
  String string;
  XawTextPosition first;
  Piece * piece;

  string = XtMalloc(sizeof(unsigned char) * src->utkascii_src.length + 1);
  
  for (first = 0, piece = src->utkascii_src.first_piece ; piece != NULL; 
       first += piece->used, piece = piece->next) 
    strncpy(string + first, piece->text, piece->used);

  string[src->utkascii_src.length] = '\0';	/* NULL terminate this sucker. */

/*
 * This will refill all pieces to capacity. 
 */

  if (src->utkascii_src.data_compression) {	
    FreeAllPieces(src);
    LoadPieces(src, NULL, string);
  }

  return(string);
}

/*	Function Name: InitStringOrFile.
 *	Description: Initializes the string or file.
 *	Arguments: src - the UtkasciiSource.
 *	Returns: none - May exit though.
 */

static FILE *
InitStringOrFile(src, newString)
UtkasciiSrcObject src;
Boolean newString;
{
    char * open_mode;
    FILE * file;
    char fileName[TMPSIZ];

    if (src->utkascii_src.type == XawUtkasciiString) {

	if (src->utkascii_src.string == NULL)
	    src->utkascii_src.length = 0;

	else if (! src->utkascii_src.use_string_in_place) {
	    src->utkascii_src.string = XtNewString(src->utkascii_src.string);
	    src->utkascii_src.allocated_string = True;
	    src->utkascii_src.length = strlen(src->utkascii_src.string);
	}

	if (src->utkascii_src.use_string_in_place) {
	    src->utkascii_src.length = strlen(src->utkascii_src.string);
	    /* In case the length resource is incorrectly set */
	    if (src->utkascii_src.length > src->utkascii_src.utkascii_length)
		src->utkascii_src.utkascii_length = src->utkascii_src.length;

	    if (src->utkascii_src.utkascii_length == MAGIC_VALUE) 
		src->utkascii_src.piece_size = src->utkascii_src.length;
	    else
		src->utkascii_src.piece_size = src->utkascii_src.utkascii_length + 1;
	}
		
	return(NULL);
    }

/*
 * type is XawUtkasciiFile.
 */
    
    src->utkascii_src.is_tempfile = FALSE;

    switch (src->text_src.edit_mode) {
    case XawtextRead:
	if (src->utkascii_src.string == NULL)
	    XtErrorMsg("NoFile", "utkasciiSourceCreate", "XawError",
		     "Creating a read only disk widget and no file specified.",
		       NULL, 0);
	open_mode = "r";
	break;
    case XawtextAppend:
    case XawtextEdit:
	if (src->utkascii_src.string == NULL) {
	    src->utkascii_src.string = fileName;
	    (void) tmpnam(src->utkascii_src.string);
	    src->utkascii_src.is_tempfile = TRUE;
	    open_mode = "w";
	} else
	    open_mode = "r+";
	break;
    default:
	XtErrorMsg("badMode", "utkasciiSourceCreate", "XawError",
		"Bad editMode for utkascii source; must be Read, Append or Edit.",
		   NULL, NULL);
    }

    /* Allocate new memory for the temp filename, because it is held in
     * a stack variable, not static memory.  This widget does not need
     * to keep the private state field is_tempfile -- it is only accessed
     * in this routine, and its former setting is unused.
     */
    if (newString || src->utkascii_src.is_tempfile) {
	src->utkascii_src.string = XtNewString(src->utkascii_src.string);
	src->utkascii_src.allocated_string = TRUE;
    }
    
    if (!src->utkascii_src.is_tempfile) {
	if ((file = fopen(src->utkascii_src.string, open_mode)) != 0) {
	    (void) fseek(file, 0L, 2);
	    src->utkascii_src.length = ftell(file);
	    return file;
	} else {
	    String params[2];
	    Cardinal num_params = 2;
	    char msg[11];
	    
	    params[0] = src->utkascii_src.string;
	    if (errno <= sys_nerr)
		params[1] = sys_errlist[errno];
	    else {
		sprintf(msg, "errno=%.4d", errno);
		params[1] = msg;
	    }
	    XtAppWarningMsg(XtWidgetToApplicationContext((Widget)src),
			    "openError", "utkasciiSourceCreate", "XawWarning",
			    "Cannot open file %s; %s", params, &num_params);
	}
    } 
    src->utkascii_src.length = 0;
    return((FILE *)NULL);
}


static void
LoadPieces(src, file, string)
UtkasciiSrcObject src;
FILE * file;
char * string;
{
  char *local_str, *ptr;
  register Piece * piece = NULL;
  XawTextPosition left;
  char line[1024];
  long offset;
  int tabcount;
  register int i,j;

  if (string == NULL) {
    if (src->utkascii_src.type == XawUtkasciiFile) {
      local_str = XtMalloc((src->utkascii_src.length + 200) *sizeof(unsigned char));
      if (src->utkascii_src.length != 0) {
		fseek(file, 0L, 0);

		/* eat first 2 lines */
		fgets(line, 1020, file);
		offset = strlen(line);
		fgets(line, 1020, file);
		offset += strlen(line);

		src->utkascii_src.length -= offset;

		if (src->utkascii_src.get_all_button) {
			strcpy(local_str, GET_ALL_PR);
			offset = strlen(GET_ALL_PR);
		} else {
			offset = 0;
		}

		if ( fread(local_str + (unsigned char)offset, 
			   sizeof(unsigned char),
			   src->utkascii_src.length, file) != src->utkascii_src.length ) 
			XtErrorMsg("readError", "utkasciiSourceCreate", "XawError",
								 "fread returned error.", NULL, NULL);

		src->utkascii_src.length += offset;

		/* extract tabs */
		/* must be same as detab method in UtkasciiText.c:detab()	*/
		for (tabcount=i=0; i < src->utkascii_src.length; i++)
			if (local_str[i] == '\t')
				tabcount++;
		ptr = XtMalloc((src->utkascii_src.length + (tabcount*8)) *sizeof(unsigned char));
		for (j=i=0; i < src->utkascii_src.length; j++,i++) {
			if (local_str[i] == '\t') {
				ptr[j] = ' '; j++;
				ptr[j] = ' '; j++;
				ptr[j] = ' '; j++;
				ptr[j] = ' '; j++;
				ptr[j] = ' ';
			} else {
				ptr[j] = local_str[i];
			}
		}
		XtFree(local_str);
		local_str = ptr;
		src->utkascii_src.length += j-i;
      }
      local_str[src->utkascii_src.length] = '\0';
    }
    else
      local_str = src->utkascii_src.string;
  }
  else
    local_str = string;

/*
 * If we are using teh string in place then set the other fields as follows:
 *
 * piece_size = length;
 * piece->used = src->utkascii_src.length;
 */
  
  if (src->utkascii_src.use_string_in_place) {
    piece = AllocNewPiece(src, piece);
    piece->used = Min(src->utkascii_src.length, src->utkascii_src.piece_size);
    piece->text = src->utkascii_src.string;
    return;
  }

  ptr = local_str;
  left = src->utkascii_src.length;

  do {
    piece = AllocNewPiece(src, piece);

    piece->text = XtMalloc(src->utkascii_src.piece_size * sizeof(unsigned char));
    piece->used = Min(left, src->utkascii_src.piece_size);
    if (piece->used != 0)
      strncpy(piece->text, ptr, piece->used);

    left -= piece->used;
    ptr += piece->used;
  } while (left > 0);

  if ( (src->utkascii_src.type == XawUtkasciiFile) && (string == NULL) )
    XtFree(local_str);
}

/*	Function Name: AllocNewPiece
 *	Description: Allocates a new piece of memory.
 *	Arguments: src - The UtkasciiSrc Widget.
 *                 prev - the piece just before this one, or NULL.
 *	Returns: the allocated piece.
 */

static Piece *
AllocNewPiece(src, prev)
UtkasciiSrcObject src;
Piece * prev;
{
  Piece * piece;
  piece = (Piece*)XtMalloc((sizeof(Piece))+1);

  if (prev == NULL) {
    src->utkascii_src.first_piece = piece;
    piece->next = NULL;
  }
  else {
    if (prev->next != NULL)
      (prev->next)->prev = piece;
    piece->next = prev->next;
    prev->next = piece;
  }
  
  piece->prev = prev;

  return(piece);
}

/*	Function Name: FreeAllPieces
 *	Description: Frees all the pieces
 *	Arguments: src - The UtkasciiSrc Widget.
 *	Returns: none.
 */

static void 
FreeAllPieces(src)
UtkasciiSrcObject src;
{
  Piece * next, * first = src->utkascii_src.first_piece;

  if (first->prev != NULL)
    printf("Programmer Botch in FreeAllPieces, there may be a memory leak.\n");

  for ( ; first != NULL ; first = next ) {
    next = first->next;
    RemovePiece(src, first);
  }
}
  
/*	Function Name: RemovePiece
 *	Description: Removes a piece from the list.
 *	Arguments: 
 *                 piece - the piece to remove.
 *	Returns: none.
 */

static void
RemovePiece(src, piece)
UtkasciiSrcObject src;
Piece * piece;
{
  if (piece->prev == NULL)
    src->utkascii_src.first_piece = piece->next;
  else
    (piece->prev)->next = piece->next;

  if (piece->next != NULL)
    (piece->next)->prev = piece->prev;

  if (!src->utkascii_src.use_string_in_place)
    XtFree(piece->text);

  XtFree((char *)piece);

}

/*	Function Name: FindPiece
 *	Description: Finds the piece containing the position indicated.
 *	Arguments: src - The UtkasciiSrc Widget.
 *                 position - the position that we are searching for.
 * RETURNED        first - the position of the first character in this piece.
 *	Returns: piece - the piece that contains this position.
 */

static Piece *
FindPiece(src, position, first)
UtkasciiSrcObject src;
XawTextPosition position, *first;
{
  Piece * old_piece, * piece = src->utkascii_src.first_piece;
  XawTextPosition temp;

  for ( temp = 0 ; piece != NULL ; temp += piece->used, piece = piece->next ) {
    *first = temp;
    old_piece = piece;

    if ((temp + piece->used) > position) 
      return(piece);
  }
  return(old_piece);	  /* if we run off the end the return the last piece */
}
    
/*	Function Name: MyStrncpy
 *	Description: Just like string copy, but slower and will always
 *                   work on overlapping strings.
 *	Arguments: (same as strncpy) - s1, s2 - strings to copy (2->1).
 *                  n - the number of chars to copy.
 *	Returns: s1.
 */

static String
MyStrncpy(s1, s2, n)
char * s1, * s2;
int n;
{
  char * temp = XtMalloc(sizeof(unsigned char) * n);

  strncpy(temp, s2, n);		/* Saber has a bug that causes it to generate*/
  strncpy(s1, temp, n);		/* a bogus warning message here (CDP 6/32/89)*/
  XtFree(temp);
  return(s1);
}
  
/*	Function Name: BreakPiece
 *	Description: Breaks a full piece into two new pieces.
 *	Arguments: src - The UtkasciiSrc Widget.
 *                 piece - the piece to break.
 *	Returns: none.
 */

#define HALF_PIECE (src->utkascii_src.piece_size/2)

static void
BreakPiece(src, piece)
UtkasciiSrcObject src;
Piece * piece;
{
  Piece * new = AllocNewPiece(src, piece);
  
  new->text = XtMalloc(src->utkascii_src.piece_size * sizeof(unsigned char));
  strncpy(new->text, piece->text + HALF_PIECE,
	  src->utkascii_src.piece_size - HALF_PIECE);
  piece->used = HALF_PIECE;
  new->used = src->utkascii_src.piece_size - HALF_PIECE; 
}

/* ARGSUSED */
static void
CvtStringToUtkasciiType(args, num_args, fromVal, toVal)
XrmValuePtr *args;		/* unused */
Cardinal	*num_args;	/* unused */
XrmValuePtr	fromVal;
XrmValuePtr	toVal;
{
  static XawUtkasciiType type;
  static XrmQuark  XtQEstring = NULLQUARK;
  static XrmQuark  XtQEfile;
  XrmQuark q;
  char lowerName[BUFSIZ];

  if (XtQEstring == NULLQUARK) {
/* from R5 - R4 doesn't like it
    XtQEstring = XrmPermStringToQuark(XtEstring);
    XtQEfile   = XrmPermStringToQuark(XtEfile);
*/
    XtQEstring = XrmStringToQuark(XtEstring);
    XtQEfile   = XrmStringToQuark(XtEfile);
  }

  XmuCopyISOLatin1Lowered(lowerName, (char *) fromVal->addr);
  q = XrmStringToQuark(lowerName);

  if (q == XtQEstring) type = XawUtkasciiString;
  if (q == XtQEfile)  type = XawUtkasciiFile;

  (*toVal).size = sizeof(XawUtkasciiType);
  (*toVal).addr = (char*) &type;
  return;
}

#if (defined(ASCII_STRING) || defined(ASCII_DISK))
#  include <X11/Xaw/Cardinals.h>
#endif

#ifdef ASCII_STRING
/************************************************************
 *
 * Compatability functions.
 *
 ************************************************************/
 
/*	Function Name: UtkasciiStringSourceCreate
 *	Description: Creates a string source.
 *	Arguments: parent - the widget that will own this source.
 *                 args, num_args - the argument list.
 *	Returns: a pointer to the new text source.
 */

Widget
UtkXawStringSourceCreate(parent, args, num_args)
Widget parent;
ArgList args;
Cardinal num_args;
{
  XawTextSource src;
  ArgList utkascii_args;
  Arg temp[2];

  XtSetArg(temp[0], XtNtype, XawUtkasciiString);
  XtSetArg(temp[1], XtNuseStringInPlace, TRUE);
  utkascii_args = XtMergeArgLists(temp, TWO, args, num_args);

  src = XtCreateWidget("genericUtkasciiString", utkasciiSrcObjectClass, parent,
		       utkascii_args, num_args + TWO);
  XtFree((char *)utkascii_args);
  return(src);
}

/*
 * This is hacked up to try to emulate old functionality, it
 * may not work, as I have not old code to test it on.
 *
 * Chris D. Peterson  8/31/89.
 */

void 
UtkXawTextSetLastPos (w, lastPos)
Widget w;
XawTextPosition lastPos;
{
  UtkasciiSrcObject src = (UtkasciiSrcObject) XawTextGetSource(w);

  src->utkascii_src.piece_size = lastPos;
}
#endif /* ASCII_STRING */

#ifdef ASCII_DISK
/*	Function Name: UtkasciiDiskSourceCreate
 *	Description: Creates a disk source.
 *	Arguments: parent - the widget that will own this source.
 *                 args, num_args - the argument list.
 *	Returns: a pointer to the new text source.
 */

Widget
UtkXawDiskSourceCreate(parent, args, num_args)
Widget parent;
ArgList args;
Cardinal num_args;
{
  XawTextSource src;
  ArgList utkascii_args;
  Arg temp[1];
  register int i;

  XtSetArg(temp[0], XtNtype, XawUtkasciiFile);
  utkascii_args = XtMergeArgLists(temp, ONE, args, num_args);
  num_args++;

  for (i = 0; i < num_args; i++) 
    if (streq(utkascii_args[i].name, XtNfile) || 
	          streq(utkascii_args[i].name, XtCFile)) 
      utkascii_args[i].name = XtNstring;

  src = XtCreateWidget("genericUtkasciiDisk", utkasciiSrcObjectClass, parent,
		       utkascii_args, num_args);
  XtFree((char *)utkascii_args);
  return(src);
}
#endif /* ASCII_DISK */

String XawUtkGetText(w)
UtkasciiWidget w;
{
	UtkasciiSrcObject src = (UtkasciiSrcObject)(w->text.source);
	
	return StorePiecesInString(src);
}
