/*-
 * This file is provided under a dual BSD/GPLv2 license.  When using or
 * redistributing this file, you may do so under either license.
 *
 * GPL LICENSE SUMMARY
 *
 * Copyright(c) 2008 - 2011 Intel Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 * The full GNU General Public License is included in this distribution
 * in the file called LICENSE.GPL.
 *
 * BSD LICENSE
 *
 * Copyright(c) 2008 - 2011 Intel Corporation. All rights reserved.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: stable/8/sys/dev/isci/scil/scu_registers.h 231688 2012-02-14 15:56:01Z jimharris $
 */
#ifndef _SCU_REGISTERS_H_
#define _SCU_REGISTERS_H_

/**
 * @file
 *
 * @brief This file contains the constants and structures for the SCU memory
 * mapped registers.
 */
#ifdef __cplusplus
extern "C" {
#endif

#include <dev/isci/scil/sci_types.h>
#include <dev/isci/scil/scu_viit_data.h>



// Generate a value for an SCU register
#define SCU_GEN_VALUE(name, value) \
   (((U32)(value) << name ## _SHIFT) & (name ## _MASK))

// Generate a bit value for an SCU register
// Make sure that the register MASK is just a single bit
#define SCU_GEN_BIT(name) \
      SCU_GEN_VALUE(name, ((U32)1))

#define SCU_SET_BIT(name, reg_value) \
   ((reg_value) | SCU_GEN_BIT(name))

#define SCU_CLEAR_BIT(name, reg_value) \
   ((reg_value) $ ~(SCU_GEN_BIT(name)))

//*****************************************************************************
// Unions for bitfield definitions of SCU Registers
// SMU Post Context Port
//*****************************************************************************
#define SMU_POST_CONTEXT_PORT_CONTEXT_INDEX_SHIFT         (0UL)
#define SMU_POST_CONTEXT_PORT_CONTEXT_INDEX_MASK          (0x00000FFFUL)
#define SMU_POST_CONTEXT_PORT_LOGICAL_PORT_INDEX_SHIFT    (12UL)
#define SMU_POST_CONTEXT_PORT_LOGICAL_PORT_INDEX_MASK     (0x0000F000UL)
#define SMU_POST_CONTEXT_PORT_PROTOCOL_ENGINE_SHIFT       (16UL)
#define SMU_POST_CONTEXT_PORT_PROTOCOL_ENGINE_MASK        (0x00030000UL)
#define SMU_POST_CONTEXT_PORT_COMMAND_CONTEXT_SHIFT       (18UL)
#define SMU_POST_CONTEXT_PORT_COMMAND_CONTEXT_MASK        (0x00FC0000UL)
#define SMU_POST_CONTEXT_PORT_RESERVED_MASK               (0xFF000000UL)

#define SMU_PCP_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_POST_CONTEXT_PORT_##name, value)

//*****************************************************************************
#define SMU_INTERRUPT_STATUS_COMPLETION_SHIFT       (31UL)
#define SMU_INTERRUPT_STATUS_COMPLETION_MASK        (0x80000000UL)
#define SMU_INTERRUPT_STATUS_QUEUE_SUSPEND_SHIFT    (1UL)
#define SMU_INTERRUPT_STATUS_QUEUE_SUSPEND_MASK     (0x00000002UL)
#define SMU_INTERRUPT_STATUS_QUEUE_ERROR_SHIFT      (0UL)
#define SMU_INTERRUPT_STATUS_QUEUE_ERROR_MASK       (0x00000001UL)
#define SMU_INTERRUPT_STATUS_RESERVED_MASK          (0x7FFFFFFCUL)

#define SMU_ISR_GEN_BIT(name) \
    SCU_GEN_BIT(SMU_INTERRUPT_STATUS_##name)

#define SMU_ISR_QUEUE_ERROR   SMU_ISR_GEN_BIT(QUEUE_ERROR)
#define SMU_ISR_QUEUE_SUSPEND SMU_ISR_GEN_BIT(QUEUE_SUSPEND)
#define SMU_ISR_COMPLETION    SMU_ISR_GEN_BIT(COMPLETION)

//*****************************************************************************
#define SMU_INTERRUPT_MASK_COMPLETION_SHIFT         (31UL)
#define SMU_INTERRUPT_MASK_COMPLETION_MASK          (0x80000000UL)
#define SMU_INTERRUPT_MASK_QUEUE_SUSPEND_SHIFT      (1UL)
#define SMU_INTERRUPT_MASK_QUEUE_SUSPEND_MASK       (0x00000002UL)
#define SMU_INTERRUPT_MASK_QUEUE_ERROR_SHIFT        (0UL)
#define SMU_INTERRUPT_MASK_QUEUE_ERROR_MASK         (0x00000001UL)
#define SMU_INTERRUPT_MASK_RESERVED_MASK            (0x7FFFFFFCUL)

#define SMU_IMR_GEN_BIT(name) \
    SCU_GEN_BIT(SMU_INTERRUPT_MASK_##name)

#define SMU_IMR_QUEUE_ERROR   SMU_IMR_GEN_BIT(QUEUE_ERROR)
#define SMU_IMR_QUEUE_SUSPEND SMU_IMR_GEN_BIT(QUEUE_SUSPEND)
#define SMU_IMR_COMPLETION    SMU_IMR_GEN_BIT(COMPLETION)

//*****************************************************************************
#define SMU_INTERRUPT_COALESCING_CONTROL_TIMER_SHIFT    (0UL)
#define SMU_INTERRUPT_COALESCING_CONTROL_TIMER_MASK     (0x0000001FUL)
#define SMU_INTERRUPT_COALESCING_CONTROL_NUMBER_SHIFT   (8UL)
#define SMU_INTERRUPT_COALESCING_CONTROL_NUMBER_MASK    (0x0000FF00UL)
#define SMU_INTERRUPT_COALESCING_CONTROL_RESERVED_MASK  (0xFFFF00E0UL)

#define SMU_ICC_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_INTERRUPT_COALESCING_CONTROL_##name, value)

//*****************************************************************************
#define SMU_TASK_CONTEXT_RANGE_START_SHIFT      (0UL)
#define SMU_TASK_CONTEXT_RANGE_START_MASK       (0x00000FFFUL)
#define SMU_TASK_CONTEXT_RANGE_ENDING_SHIFT     (16UL)
#define SMU_TASK_CONTEXT_RANGE_ENDING_MASK      (0x0FFF0000UL)
#define SMU_TASK_CONTEXT_RANGE_ENABLE_SHIFT     (31UL)
#define SMU_TASK_CONTEXT_RANGE_ENABLE_MASK      (0x80000000UL)
#define SMU_TASK_CONTEXT_RANGE_RESERVED_MASK    (0x7000F000UL)

#define SMU_TCR_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_TASK_CONTEXT_RANGE_##name, value)

#define SMU_TCR_GEN_BIT(name, value) \
    SCU_GEN_BIT(SMU_TASK_CONTEXT_RANGE_##name)

//*****************************************************************************

#define SMU_COMPLETION_QUEUE_PUT_POINTER_SHIFT          (0UL)
#define SMU_COMPLETION_QUEUE_PUT_POINTER_MASK           (0x00003FFFUL)
#define SMU_COMPLETION_QUEUE_PUT_CYCLE_BIT_SHIFT        (15UL)
#define SMU_COMPLETION_QUEUE_PUT_CYCLE_BIT_MASK         (0x00008000UL)
#define SMU_COMPLETION_QUEUE_PUT_EVENT_POINTER_SHIFT    (16UL)
#define SMU_COMPLETION_QUEUE_PUT_EVENT_POINTER_MASK     (0x03FF0000UL)
#define SMU_COMPLETION_QUEUE_PUT_EVENT_CYCLE_BIT_SHIFT  (26UL)
#define SMU_COMPLETION_QUEUE_PUT_EVENT_CYCLE_BIT_MASK   (0x04000000UL)
#define SMU_COMPLETION_QUEUE_PUT_RESERVED_MASK          (0xF8004000UL)

#define SMU_CQPR_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_COMPLETION_QUEUE_PUT_##name, value)

#define SMU_CQPR_GEN_BIT(name) \
    SCU_GEN_BIT(SMU_COMPLETION_QUEUE_PUT_##name)

//*****************************************************************************

#define SMU_COMPLETION_QUEUE_GET_POINTER_SHIFT          (0UL)
#define SMU_COMPLETION_QUEUE_GET_POINTER_MASK           (0x00003FFFUL)
#define SMU_COMPLETION_QUEUE_GET_CYCLE_BIT_SHIFT        (15UL)
#define SMU_COMPLETION_QUEUE_GET_CYCLE_BIT_MASK         (0x00008000UL)
#define SMU_COMPLETION_QUEUE_GET_EVENT_POINTER_SHIFT    (16UL)
#define SMU_COMPLETION_QUEUE_GET_EVENT_POINTER_MASK     (0x03FF0000UL)
#define SMU_COMPLETION_QUEUE_GET_EVENT_CYCLE_BIT_SHIFT  (26UL)
#define SMU_COMPLETION_QUEUE_GET_EVENT_CYCLE_BIT_MASK   (0x04000000UL)
#define SMU_COMPLETION_QUEUE_GET_ENABLE_SHIFT           (30UL)
#define SMU_COMPLETION_QUEUE_GET_ENABLE_MASK            (0x40000000UL)
#define SMU_COMPLETION_QUEUE_GET_EVENT_ENABLE_SHIFT     (31UL)
#define SMU_COMPLETION_QUEUE_GET_EVENT_ENABLE_MASK      (0x80000000UL)
#define SMU_COMPLETION_QUEUE_GET_RESERVED_MASK          (0x38004000UL)

#define SMU_CQGR_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_COMPLETION_QUEUE_GET_##name, value)

#define SMU_CQGR_GEN_BIT(name) \
    SCU_GEN_BIT(SMU_COMPLETION_QUEUE_GET_##name)

#define SMU_CQGR_CYCLE_BIT \
    SMU_CQGR_GEN_BIT(CYCLE_BIT)

#define SMU_CQGR_EVENT_CYCLE_BIT \
    SMU_CQGR_GEN_BIT(EVENT_CYCLE_BIT)

#define SMU_CQGR_GET_POINTER_SET(value) \
    SMU_CQGR_GEN_VAL(POINTER, value)


//*****************************************************************************
#define SMU_COMPLETION_QUEUE_CONTROL_QUEUE_LIMIT_SHIFT  (0UL)
#define SMU_COMPLETION_QUEUE_CONTROL_QUEUE_LIMIT_MASK   (0x00003FFFUL)
#define SMU_COMPLETION_QUEUE_CONTROL_EVENT_LIMIT_SHIFT  (16UL)
#define SMU_COMPLETION_QUEUE_CONTROL_EVENT_LIMIT_MASK   (0x03FF0000UL)
#define SMU_COMPLETION_QUEUE_CONTROL_RESERVED_MASK      (0xFC00C000UL)

#define SMU_CQC_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_COMPLETION_QUEUE_CONTROL_##name, value)

#define SMU_CQC_QUEUE_LIMIT_SET(value) \
    SMU_CQC_GEN_VAL(QUEUE_LIMIT, value)

#define SMU_CQC_EVENT_LIMIT_SET(value) \
    SMU_CQC_GEN_VAL(EVENT_LIMIT, value)


//*****************************************************************************
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_TC_SHIFT    (0UL)
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_TC_MASK     (0x00000FFFUL)
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_LP_SHIFT    (12UL)
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_LP_MASK     (0x00007000UL)
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_RNC_SHIFT   (15UL)
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_RNC_MASK    (0x07FF8000UL)
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_PEG_SHIFT   (27UL)
#define SMU_DEVICE_CONTEXT_CAPACITY_MAX_PEG_MASK    (0x08000000UL)
#define SMU_DEVICE_CONTEXT_CAPACITY_RESERVED_MASK   (0xF0000000UL)

#define SMU_DCC_GEN_VAL(name, value) \
   SCU_GEN_VALUE(SMU_DEVICE_CONTEXT_CAPACITY_##name, value)

#define SMU_DCC_GET_MAX_PEG(value) \
   ( \
         ((U32)((value) & SMU_DEVICE_CONTEXT_CAPACITY_MAX_PEG_MASK)) \
      >> SMU_DEVICE_CONTEXT_CAPACITY_MAX_LP_SHIFT \
   )

#define SMU_DCC_GET_MAX_LP(value) \
   ( \
         ((U32)((value) & SMU_DEVICE_CONTEXT_CAPACITY_MAX_LP_MASK)) \
      >> SMU_DEVICE_CONTEXT_CAPACITY_MAX_LP_SHIFT \
   )

#define SMU_DCC_GET_MAX_TC(value) \
   ( \
         ((U32)((value) & SMU_DEVICE_CONTEXT_CAPACITY_MAX_TC_MASK)) \
      >> SMU_DEVICE_CONTEXT_CAPACITY_MAX_TC_SHIFT \
   )

#define SMU_DCC_GET_MAX_RNC(value) \
   ( \
        ((U32)((value) & SMU_DEVICE_CONTEXT_CAPACITY_MAX_RNC_MASK)) \
     >> SMU_DEVICE_CONTEXT_CAPACITY_MAX_RNC_SHIFT \
   )

//*****************************************************************************
#define SMU_CLOCK_GATING_CONTROL_IDLE_ENABLE_SHIFT    (0UL)
#define SMU_CLOCK_GATING_CONTROL_IDLE_ENABLE_MASK     (0x00000001UL)
#define SMU_CLOCK_GATING_CONTROL_XCLK_ENABLE_SHIFT    (1UL)
#define SMU_CLOCK_GATING_CONTROL_XCLK_ENABLE_MASK     (0x00000002UL)
#define SMU_CLOCK_GATING_CONTROL_TXCLK_ENABLE_SHIFT   (2UL)
#define SMU_CLOCK_GATING_CONTROL_TXCLK_ENABLE_MASK    (0x00000004UL)
#define SMU_CLOCK_GATING_CONTROL_REGCLK_ENABLE_SHIFT  (3UL)
#define SMU_CLOCK_GATING_CONTROL_REGCLK_ENABLE_MASK   (0x00000008UL)
#define SMU_CLOCK_GATING_CONTROL_IDLE_TIMEOUT_SHIFT   (16UL)
#define SMU_CLOCK_GATING_CONTROL_IDLE_TIMEOUT_MASK    (0x000F0000UL)
#define SMU_CLOCK_GATING_CONTROL_FORCE_IDLE_SHIFT     (31UL)
#define SMU_CLOCK_GATING_CONTROL_FORCE_IDLE_MASK      (0x80000000UL)
#define SMU_CLOCK_GATING_CONTROL_RESERVED_MASK        (0x7FF0FFF0UL)

#define SMU_CGUCR_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_CLOCK_GATING_CONTROL_##name, value)

#define SMU_CGUCR_GEN_BIT(name) \
    SCU_GEN_BIT(SMU_CLOCK_GATING_CONTROL_##name)

// --------------------------------------------------------------------------

#define SMU_CONTROL_STATUS_TASK_CONTEXT_RANGE_ENABLE_SHIFT      (0UL)
#define SMU_CONTROL_STATUS_TASK_CONTEXT_RANGE_ENABLE_MASK       (0x00000001UL)
#define SMU_CONTROL_STATUS_COMPLETION_BYTE_SWAP_ENABLE_SHIFT    (1UL)
#define SMU_CONTROL_STATUS_COMPLETION_BYTE_SWAP_ENABLE_MASK     (0x00000002UL)
#define SMU_CONTROL_STATUS_CONTEXT_RAM_INIT_COMPLETED_SHIFT     (16UL)
#define SMU_CONTROL_STATUS_CONTEXT_RAM_INIT_COMPLETED_MASK      (0x00010000UL)
#define SMU_CONTROL_STATUS_SCHEDULER_RAM_INIT_COMPLETED_SHIFT   (17UL)
#define SMU_CONTROL_STATUS_SCHEDULER_RAM_INIT_COMPLETED_MASK    (0x00020000UL)
#define SMU_CONTROL_STATUS_RESERVED_MASK                        (0xFFFCFFFCUL)

#define SMU_SMUCSR_GEN_BIT(name) \
   SCU_GEN_BIT(SMU_CONTROL_STATUS_##name)

#define SMU_SMUCSR_SCHEDULER_RAM_INIT_COMPLETED \
   (SMU_SMUCSR_GEN_BIT(SCHEDULER_RAM_INIT_COMPLETED))

#define SMU_SMUCSR_CONTEXT_RAM_INIT_COMPLETED   \
   (SMU_SMUCSR_GEN_BIT(CONTEXT_RAM_INIT_COMPLETED))

#define SCU_RAM_INIT_COMPLETED \
   ( \
       SMU_SMUCSR_CONTEXT_RAM_INIT_COMPLETED \
     | SMU_SMUCSR_SCHEDULER_RAM_INIT_COMPLETED \
   )

// --------------------------------------------------------------------------

#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE0_SHIFT  (0UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE0_MASK   (0x00000001UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE1_SHIFT  (1UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE1_MASK   (0x00000002UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE2_SHIFT  (2UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE2_MASK   (0x00000004UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE3_SHIFT  (3UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_PE3_MASK   (0x00000008UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE0_SHIFT  (8UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE0_MASK   (0x00000100UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE1_SHIFT  (9UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE1_MASK   (0x00000200UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE2_SHIFT  (10UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE2_MASK   (0x00000400UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE3_SHIFT  (11UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_PE3_MASK   (0x00000800UL)

#define SMU_RESET_PROTOCOL_ENGINE(peg, pe) \
    ((1UL << (pe)) << ((peg) * 8UL))

#define SMU_RESET_PEG_PROTOCOL_ENGINES(peg) \
    ( \
        SMU_RESET_PROTOCOL_ENGINE(peg, 0) \
      | SMU_RESET_PROTOCOL_ENGINE(peg, 1) \
      | SMU_RESET_PROTOCOL_ENGINE(peg, 2) \
      | SMU_RESET_PROTOCOL_ENGINE(peg, 3) \
    )

#define SMU_RESET_ALL_PROTOCOL_ENGINES() \
    ( \
        SMU_RESET_PEG_PROTOCOL_ENGINES(0) \
      | SMU_RESET_PEG_PROTOCOL_ENGINES(1) \
    )

#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG0_LP0_SHIFT  (16UL)
#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG0_LP0_MASK   (0x00010000UL)
#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG0_LP2_SHIFT  (17UL)
#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG0_LP2_MASK   (0x00020000UL)
#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG1_LP0_SHIFT  (18UL)
#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG1_LP0_MASK   (0x00040000UL)
#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG1_LP2_SHIFT  (19UL)
#define SMU_SOFTRESET_CONTROL_RESET_WIDE_PORT_PEG1_LP2_MASK   (0x00080000UL)

#define SMU_RESET_WIDE_PORT_QUEUE(peg, wide_port) \
    ((1UL << ((wide_port) / 2)) << ((peg) * 2UL) << 16UL)

#define SMU_SOFTRESET_CONTROL_RESET_PEG0_SHIFT      (20UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG0_MASK       (0x00100000UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_SHIFT      (21UL)
#define SMU_SOFTRESET_CONTROL_RESET_PEG1_MASK       (0x00200000UL)
#define SMU_SOFTRESET_CONTROL_RESET_SCU_SHIFT       (22UL)
#define SMU_SOFTRESET_CONTROL_RESET_SCU_MASK        (0x00400000UL)

// It seems to make sense that if you are going to reset the protocol
// engine group that you would also reset all of the protocol engines
#define SMU_RESET_PROTOCOL_ENGINE_GROUP(peg) \
    ( \
        (1UL << ((peg) + 20)) \
      | SMU_RESET_WIDE_PORT_QUEUE(peg, 0) \
      | SMU_RESET_WIDE_PORT_QUEUE(peg, 1) \
      | SMU_RESET_PEG_PROTOCOL_ENGINES(peg) \
    )

#define SMU_RESET_ALL_PROTOCOL_ENGINE_GROUPS() \
    ( \
        SMU_RESET_PROTOCOL_ENGINE_GROUP(0) \
      | SMU_RESET_PROTOCOL_ENGINE_GROUP(1) \
    )

#define SMU_RESET_SCU()  (0xFFFFFFFF)



//*****************************************************************************
#define SMU_TASK_CONTEXT_ASSIGNMENT_STARTING_SHIFT              (0UL)
#define SMU_TASK_CONTEXT_ASSIGNMENT_STARTING_MASK               (0x00000FFFUL)
#define SMU_TASK_CONTEXT_ASSIGNMENT_ENDING_SHIFT                (16UL)
#define SMU_TASK_CONTEXT_ASSIGNMENT_ENDING_MASK                 (0x0FFF0000UL)
#define SMU_TASK_CONTEXT_ASSIGNMENT_RANGE_CHECK_ENABLE_SHIFT    (31UL)
#define SMU_TASK_CONTEXT_ASSIGNMENT_RANGE_CHECK_ENABLE_MASK     (0x80000000UL)
#define SMU_TASK_CONTEXT_ASSIGNMENT_RESERVED_MASK               (0x7000F000UL)

#define SMU_TCA_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SMU_TASK_CONTEXT_ASSIGNMENT_##name, value)

#define SMU_TCA_GEN_BIT(name) \
    SCU_GEN_BIT(SMU_TASK_CONTEXT_ASSIGNMENT_##name)

//*****************************************************************************
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_CONTROL_QUEUE_SIZE_SHIFT   (0UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_CONTROL_QUEUE_SIZE_MASK    (0x00000FFFUL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_CONTROL_RESERVED_MASK      (0xFFFFF000UL)

#define SCU_UFQC_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SDMA_UNSOLICITED_FRAME_QUEUE_CONTROL_##name, value)

#define SCU_UFQC_QUEUE_SIZE_SET(value) \
    SCU_UFQC_GEN_VAL(QUEUE_SIZE, value)

//*****************************************************************************
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_PUT_POINTER_SHIFT      (0UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_PUT_POINTER_MASK       (0x00000FFFUL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_PUT_CYCLE_BIT_SHIFT    (12UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_PUT_CYCLE_BIT_MASK     (0x00001000UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_PUT_RESERVED_MASK      (0xFFFFE000UL)

#define SCU_UFQPP_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SDMA_UNSOLICITED_FRAME_QUEUE_PUT_##name, value)

#define SCU_UFQPP_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SDMA_UNSOLICITED_FRAME_QUEUE_PUT_##name)

//*****************************************************************************
//* SDMA Registers
//*****************************************************************************
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_POINTER_SHIFT      (0UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_POINTER_MASK       (0x00000FFFUL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_CYCLE_BIT_SHIFT    (12UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_CYCLE_BIT_MASK     (12UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_ENABLE_BIT_SHIFT   (31UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_ENABLE_BIT_MASK    (0x80000000UL)
#define SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_RESERVED_MASK      (0x7FFFE000UL)

#define SCU_UFQGP_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_##name, value)

#define SCU_UFQGP_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SDMA_UNSOLICITED_FRAME_QUEUE_GET_##name)

#define SCU_UFQGP_CYCLE_BIT(value) \
    SCU_UFQGP_GEN_BIT(CYCLE_BIT, value)

#define SCU_UFQGP_GET_POINTER(value) \
    SCU_UFQGP_GEN_VALUE(POINTER, value)

#define SCU_UFQGP_ENABLE(value) \
   (SCU_UFQGP_GEN_BIT(ENABLE) | value)

#define SCU_UFQGP_DISABLE(value) \
   (~SCU_UFQGP_GEN_BIT(ENABLE) & value)

#define SCU_UFQGP_VALUE(bit, value) \
    (SCU_UFQGP_CYCLE_BIT(bit) | SCU_UFQGP_GET_POINTER(value))

//*****************************************************************************
#define SCU_PDMA_CONFIGURATION_ADDRESS_MODIFIER_SHIFT                               (0UL)
#define SCU_PDMA_CONFIGURATION_ADDRESS_MODIFIER_MASK                                (0x0000FFFFUL)
#define SCU_PDMA_CONFIGURATION_PCI_RELAXED_ORDERING_ENABLE_SHIFT                    (16UL)
#define SCU_PDMA_CONFIGURATION_PCI_RELAXED_ORDERING_ENABLE_MASK                     (0x00010000UL)
#define SCU_PDMA_CONFIGURATION_PCI_NO_SNOOP_ENABLE_SHIFT                            (17UL)
#define SCU_PDMA_CONFIGURATION_PCI_NO_SNOOP_ENABLE_MASK                             (0x00020000UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_BYTE_SWAP_SHIFT                   (18UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_BYTE_SWAP_MASK                    (0x00040000UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_XPI_SGL_FETCH_SHIFT               (19UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_XPI_SGL_FETCH_MASK                (0x00080000UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_XPI_RX_HEADER_RAM_WRITE_SHIFT     (20UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_XPI_RX_HEADER_RAM_WRITE_MASK      (0x00100000UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_XPI_UF_ADDRESS_FETCH_SHIFT        (21UL)
#define SCU_PDMA_CONFIGURATION_BIG_ENDIAN_CONTROL_XPI_UF_ADDRESS_FETCH_MASK         (0x00200000UL)
#define SCU_PDMA_CONFIGURATION_ADDRESS_MODIFIER_SELECT_SHIFT                        (22UL)
#define SCU_PDMA_CONFIGURATION_ADDRESS_MODIFIER_SELECT_MASK                         (0x00400000UL)
#define SCU_PDMA_CONFIGURATION_RESERVED_MASK                                        (0xFF800000UL)

#define SCU_PDMACR_GEN_VALUE(name, value) \
    SCU_GEN_VALUE(SCU_PDMA_CONFIGURATION_##name, value)

#define SCU_PDMACR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_PDMA_CONFIGURATION_##name)

#define SCU_PDMACR_BE_GEN_BIT(name) \
    SCU_PCMACR_GEN_BIT(BIG_ENDIAN_CONTROL_##name)

//*****************************************************************************
#define SCU_CDMA_CONFIGURATION_PCI_RELAXED_ORDERING_ENABLE_SHIFT                    (8UL)
#define SCU_CDMA_CONFIGURATION_PCI_RELAXED_ORDERING_ENABLE_MASK                     (0x00000100UL)

#define SCU_CDMACR_GEN_BIT(name) \
   SCU_GEN_BIT(SCU_CDMA_CONFIGURATION_##name)

//*****************************************************************************
//* SCU Link Layer Registers
//*****************************************************************************
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_TIMEOUT_SHIFT             (0UL)
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_TIMEOUT_MASK              (0x000000FFUL)
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_LOCK_TIME_SHIFT           (8UL)
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_LOCK_TIME_MASK            (0x0000FF00UL)
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_RATE_CHANGE_DELAY_SHIFT   (16UL)
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_RATE_CHANGE_DELAY_MASK    (0x00FF0000UL)
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_DWORD_SYNC_TIMEOUT_SHIFT  (24UL)
#define SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_DWORD_SYNC_TIMEOUT_MASK   (0xFF000000UL)
#define SCU_LINK_LAYER_SPEED_NECGOIATION_TIMER_VALUES_REQUIRED_MASK             (0x00000000UL)
#define SCU_LINK_LAYER_SPEED_NECGOIATION_TIMER_VALUES_DEFAULT_MASK              (0x7D00676FUL)
#define SCU_LINK_LAYER_SPEED_NECGOIATION_TIMER_VALUES_RESERVED_MASK             (0x00FF0000UL)

#define SCU_SAS_SPDTOV_GEN_VALUE(name, value) \
    SCU_GEN_VALUE(SCU_LINK_LAYER_SPEED_NEGOTIATION_TIMER_VALUES_##name, value)


#define SCU_LINK_STATUS_DWORD_SYNC_AQUIRED_SHIFT            (2UL)
#define SCU_LINK_STATUS_DWORD_SYNC_AQUIRED_MASK             (0x00000004UL)
#define SCU_LINK_STATUS_TRANSMIT_PORT_SELECTION_DONE_SHIFT  (4UL)
#define SCU_LINK_STATUS_TRANSMIT_PORT_SELECTION_DONE_MASK   (0x00000010UL)
#define SCU_LINK_STATUS_RECEIVER_CREDIT_EXHAUSTED_SHIFT     (5UL)
#define SCU_LINK_STATUS_RECEIVER_CREDIT_EXHAUSTED_MASK      (0x00000020UL)
#define SCU_LINK_STATUS_RESERVED_MASK                       (0xFFFFFFCDUL)

#define SCU_SAS_LLSTA_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_LINK_STATUS_##name)


// TODO: Where is the SATA_PSELTOV register?

//*****************************************************************************
//* SCU SAS Maximum Arbitration Wait Time Timeout Register
//*****************************************************************************
#define SCU_SAS_MAX_ARBITRATION_WAIT_TIME_TIMEOUT_VALUE_SHIFT       (0UL)
#define SCU_SAS_MAX_ARBITRATION_WAIT_TIME_TIMEOUT_VALUE_MASK        (0x00007FFFUL)
#define SCU_SAS_MAX_ARBITRATION_WAIT_TIME_TIMEOUT_SCALE_SHIFT       (15UL)
#define SCU_SAS_MAX_ARBITRATION_WAIT_TIME_TIMEOUT_SCALE_MASK        (0x00008000UL)

#define SCU_SAS_MAWTTOV_GEN_VALUE(name, value) \
    SCU_GEN_VALUE(SCU_SAS_MAX_ARBITRATION_WAIT_TIME_TIMEOUT_##name, value)

#define SCU_SAS_MAWTTOV_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SAS_MAX_ARBITRATION_WAIT_TIME_TIMEOUT_##name)


// TODO: Where is the SAS_LNKTOV regsiter?
// TODO: Where is the SAS_PHYTOV register?

#define SCU_SAS_TRANSMIT_IDENTIFICATION_SMP_TARGET_SHIFT            (1UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_SMP_TARGET_MASK             (0x00000002UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_STP_TARGET_SHIFT            (2UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_STP_TARGET_MASK             (0x00000004UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_SSP_TARGET_SHIFT            (3UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_SSP_TARGET_MASK             (0x00000008UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_DA_SATA_HOST_SHIFT          (8UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_DA_SATA_HOST_MASK           (0x00000100UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_SMP_INITIATOR_SHIFT         (9UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_SMP_INITIATOR_MASK          (0x00000200UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_STP_INITIATOR_SHIFT         (10UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_STP_INITIATOR_MASK          (0x00000400UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_SSP_INITIATOR_SHIFT         (11UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_SSP_INITIATOR_MASK          (0x00000800UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_REASON_CODE_SHIFT           (16UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_REASON_CODE_MASK            (0x000F0000UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_ADDRESS_FRAME_TYPE_SHIFT    (24UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_ADDRESS_FRAME_TYPE_MASK     (0x0F000000UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_DEVICE_TYPE_SHIFT           (28UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_DEVICE_TYPE_MASK            (0x70000000UL)
#define SCU_SAS_TRANSMIT_IDENTIFICATION_RESERVED_MASK               (0x80F0F1F1UL)

#define SCU_SAS_TIID_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SAS_TRANSMIT_IDENTIFICATION_##name, value)

#define SCU_SAS_TIID_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SAS_TRANSMIT_IDENTIFICATION_##name)

// SAS Identify Frame PHY Identifier Register
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_BREAK_REPLY_CAPABLE_SHIFT      (16UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_BREAK_REPLY_CAPABLE_MASK       (0x00010000UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_REQUESTED_INSIDE_ZPSDS_SHIFT   (17UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_REQUESTED_INSIDE_ZPSDS_MASK    (0x00020000UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_INSIDE_ZPSDS_PERSISTENT_SHIFT  (18UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_INSIDE_ZPSDS_PERSISTENT_MASK   (0x00040000UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_ID_SHIFT                       (24UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_ID_MASK                        (0xFF000000UL)
#define SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_RESERVED_MASK                  (0x00F800FFUL)

#define SCU_SAS_TIPID_GEN_VALUE(name, value) \
    SCU_GEN_VALUE(SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_##name, value)

#define SCU_SAS_TIPID_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_LINK_LAYER_IDENTIFY_FRAME_PHY_IDENTIFIER_##name)


#define SCU_SAS_PHY_CONFIGURATION_TX_PARITY_CHECK_SHIFT                     (4UL)
#define SCU_SAS_PHY_CONFIGURATION_TX_PARITY_CHECK_MASK                      (0x00000010UL)
#define SCU_SAS_PHY_CONFIGURATION_TX_BAD_CRC_SHIFT                          (6UL)
#define SCU_SAS_PHY_CONFIGURATION_TX_BAD_CRC_MASK                           (0x00000040UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_SCRAMBLER_SHIFT                   (7UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_SCRAMBLER_MASK                    (0x00000080UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_DESCRAMBLER_SHIFT                 (8UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_DESCRAMBLER_MASK                  (0x00000100UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_CREDIT_INSERTION_SHIFT            (9UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_CREDIT_INSERTION_MASK             (0x00000200UL)
#define SCU_SAS_PHY_CONFIGURATION_SUSPEND_PROTOCOL_ENGINE_SHIFT             (11UL)
#define SCU_SAS_PHY_CONFIGURATION_SUSPEND_PROTOCOL_ENGINE_MASK              (0x00000800UL)
#define SCU_SAS_PHY_CONFIGURATION_SATA_SPINUP_HOLD_SHIFT                    (12UL)
#define SCU_SAS_PHY_CONFIGURATION_SATA_SPINUP_HOLD_MASK                     (0x00001000UL)
#define SCU_SAS_PHY_CONFIGURATION_TRANSMIT_PORT_SELECTION_SIGNAL_SHIFT      (13UL)
#define SCU_SAS_PHY_CONFIGURATION_TRANSMIT_PORT_SELECTION_SIGNAL_MASK       (0x00002000UL)
#define SCU_SAS_PHY_CONFIGURATION_HARD_RESET_SHIFT                          (14UL)
#define SCU_SAS_PHY_CONFIGURATION_HARD_RESET_MASK                           (0x00004000UL)
#define SCU_SAS_PHY_CONFIGURATION_OOB_ENABLE_SHIFT                          (15UL)
#define SCU_SAS_PHY_CONFIGURATION_OOB_ENABLE_MASK                           (0x00008000UL)
#define SCU_SAS_PHY_CONFIGURATION_ENABLE_FRAME_TX_INSERT_ALIGN_SHIFT        (23UL)
#define SCU_SAS_PHY_CONFIGURATION_ENABLE_FRAME_TX_INSERT_ALIGN_MASK         (0x00800000UL)
#define SCU_SAS_PHY_CONFIGURATION_FORWARD_IDENTIFY_FRAME_SHIFT              (27UL)
#define SCU_SAS_PHY_CONFIGURATION_FORWARD_IDENTIFY_FRAME_MASK               (0x08000000UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_BYTE_TRANSPOSE_STP_FRAME_SHIFT    (28UL)
#define SCU_SAS_PHY_CONFIGURATION_DISABLE_BYTE_TRANSPOSE_STP_FRAME_MASK     (0x10000000UL)
#define SCU_SAS_PHY_CONFIGURATION_OOB_RESET_SHIFT                           (29UL)
#define SCU_SAS_PHY_CONFIGURATION_OOB_RESET_MASK                            (0x20000000UL)
#define SCU_SAS_PHY_CONFIGURATION_THREE_IAF_ENABLE_SHIFT                    (30UL)
#define SCU_SAS_PHY_CONFIGURATION_THREE_IAF_ENABLE_MASK                     (0x40000000UL)
#define SCU_SAS_PHY_CONFIGURATION_OOB_ALIGN0_ENABLE_SHIFT                   (31UL)
#define SCU_SAS_PHY_CONFIGURATION_OOB_ALIGN0_ENABLE_MASK                    (0x80000000UL)
#define SCU_SAS_PHY_CONFIGURATION_REQUIRED_MASK                             (0x0100000FUL)
#define SCU_SAS_PHY_CONFIGURATION_DEFAULT_MASK                              (0x4180100FUL)
#define SCU_SAS_PHY_CONFIGURATION_RESERVED_MASK                             (0x00000000UL)

#define SCU_SAS_PCFG_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SAS_PHY_CONFIGURATION_##name)

#define SCU_LINK_LAYER_ALIGN_INSERTION_FREQUENCY_GENERAL_SHIFT      (0UL)
#define SCU_LINK_LAYER_ALIGN_INSERTION_FREQUENCY_GENERAL_MASK       (0x000007FFUL)
#define SCU_LINK_LAYER_ALIGN_INSERTION_FREQUENCY_CONNECTED_SHIFT    (16UL)
#define SCU_LINK_LAYER_ALIGN_INSERTION_FREQUENCY_CONNECTED_MASK     (0x00ff0000UL)

#define SCU_ALIGN_INSERTION_FREQUENCY_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_LINK_LAYER_ALIGN_INSERTION_FREQUENCY_##name, value)

#define SCU_LINK_LAYER_ENABLE_SPINUP_CONTROL_COUNT_SHIFT    (0UL)
#define SCU_LINK_LAYER_ENABLE_SPINUP_CONTROL_COUNT_MASK     (0x0003FFFFUL)
#define SCU_LINK_LAYER_ENABLE_SPINUP_CONTROL_ENABLE_SHIFT   (31UL)
#define SCU_LINK_LAYER_ENABLE_SPINUP_CONTROL_ENABLE_MASK    (0x80000000UL)
#define SCU_LINK_LAYER_ENABLE_SPINUP_CONTROL_RESERVED_MASK  (0x7FFC0000UL)

#define SCU_ENSPINUP_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_LINK_LAYER_ENABLE_SPINUP_CONTROL_##name, value)

#define SCU_ENSPINUP_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_LINK_LAYER_ENABLE_SPINUP_CONTROL_##name)


#define SCU_LINK_LAYER_PHY_CAPABILITIES_TXSSCTYPE_SHIFT     (1UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_TXSSCTYPE_MASK      (0x00000002UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_RLLRATE_SHIFT       (4UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_RLLRATE_MASK        (0x000000F0UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SWO15GBPS_SHIFT     (8UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SWO15GBPS_MASK      (0x00000100UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SW15GBPS_SHIFT      (9UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SW15GBPS_MASK       (0x00000201UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SWO30GBPS_SHIFT     (10UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SWO30GBPS_MASK      (0x00000401UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SW30GBPS_SHIFT      (11UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SW30GBPS_MASK       (0x00000801UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SWO60GBPS_SHIFT     (12UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SWO60GBPS_MASK      (0x00001001UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SW60GBPS_SHIFT      (13UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_SW60GBPS_MASK       (0x00002001UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_EVEN_PARITY_SHIFT   (31UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_EVEN_PARITY_MASK    (0x80000000UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_DEFAULT_MASK        (0x00003F01UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_REQUIRED_MASK       (0x00000001UL)
#define SCU_LINK_LAYER_PHY_CAPABILITIES_RESERVED_MASK       (0x7FFFC00DUL)

#define SCU_SAS_PHYCAP_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_LINK_LAYER_PHY_CAPABILITIES_##name, value)

#define SCU_SAS_PHYCAP_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_LINK_LAYER_PHY_CAPABILITIES_##name)


#define SCU_LINK_LAYER_PHY_SOURCE_ZONE_GROUP_CONTROL_VIRTUAL_EXPANDER_PHY_ZONE_GROUP_SHIFT  (0UL)
#define SCU_LINK_LAYER_PHY_SOURCE_ZONE_GROUP_CONTROL_VIRTUAL_EXPANDER_PHY_ZONE_GROUP_MASK   (0x000000FFUL)
#define SCU_LINK_LAYER_PHY_SOURCE_ZONE_GROUP_CONTROL_INSIDE_SOURCE_ZONE_GROUP_SHIFT         (31UL)
#define SCU_LINK_LAYER_PHY_SOURCE_ZONE_GROUP_CONTROL_INSIDE_SOURCE_ZONE_GROUP_MASK          (0x80000000UL)
#define SCU_LINK_LAYER_PHY_SOURCE_ZONE_GROUP_CONTROL_RESERVED_MASK                          (0x7FFFFF00UL)

#define SCU_PSZGCR_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_LINK_LAYER_PHY_SOURCE_ZONE_GROUP_CONTROL_##name, value)

#define SCU_PSZGCR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_LINK_LAYER_PHY_SOURCE_ZONE_GROUP_CONTROL_##name)

#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE0_LOCKED_SHIFT        (1UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE0_LOCKED_MASK         (0x00000002UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE0_UPDATING_SHIFT      (2UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE0_UPDATING_MASK       (0x00000004UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE1_LOCKED_SHIFT        (4UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE1_LOCKED_MASK         (0x00000010UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE1_UPDATING_SHIFT      (5UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZONE1_UPDATING_MASK       (0x00000020UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE0_SHIFT (16UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE0_MASK  (0x00030000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE0_SHIFT      (19UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE0_MASK       (0x00080000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE1_SHIFT (20UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE1_MASK  (0x00300000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE1_SHIFT      (23UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE1_MASK       (0x00800000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE2_SHIFT (24UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE2_MASK  (0x03000000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE2_SHIFT      (27UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE2_MASK       (0x08000000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE3_SHIFT (28UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_ZPT_ASSOCIATION_PE3_MASK  (0x30000000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE3_SHIFT      (31UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_AIP_ENABLE_PE3_MASK       (0x80000000UL)
#define SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_RESERVED_MASK             (0x4444FFC9UL)

#define SCU_PEG_SCUVZECR_GEN_VAL(name, val) \
    SCU_GEN_VALUE(SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_##name, val)

#define SCU_PEG_SCUVZECR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_PROTOCOL_ENGINE_GROUP_VIRTUAL_ZONING_EXPANDER_CONTROL_##name)


//*****************************************************************************
//* Port Task Scheduler registers shift and mask values
//*****************************************************************************
#define SCU_PTSG_CONTROL_IT_NEXUS_TIMEOUT_SHIFT     (0UL)
#define SCU_PTSG_CONTROL_IT_NEXUS_TIMEOUT_MASK      (0x0000FFFFUL)
#define SCU_PTSG_CONTROL_TASK_TIMEOUT_SHIFT         (16UL)
#define SCU_PTSG_CONTROL_TASK_TIMEOUT_MASK          (0x00FF0000UL)
#define SCU_PTSG_CONTROL_PTSG_ENABLE_SHIFT          (24UL)
#define SCU_PTSG_CONTROL_PTSG_ENABLE_MASK           (0x01000000UL)
#define SCU_PTSG_CONTROL_ETM_ENABLE_SHIFT           (25UL)
#define SCU_PTSG_CONTROL_ETM_ENABLE_MASK            (0x02000000UL)
#define SCU_PTSG_CONTROL_DEFAULT_MASK               (0x00020002UL)
#define SCU_PTSG_CONTROL_REQUIRED_MASK              (0x00000000UL)
#define SCU_PTSG_CONTROL_RESERVED_MASK              (0xFC000000UL)

#define SCU_PTSGCR_GEN_VAL(name, val) \
    SCU_GEN_VALUE(SCU_PTSG_CONTROL_##name, val)

#define SCU_PTSGCR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_PTSG_CONTROL_##name)


//*****************************************************************************
#define SCU_PTSG_REAL_TIME_CLOCK_SHIFT          (0UL)
#define SCU_PTSG_REAL_TIME_CLOCK_MASK           (0x0000FFFFUL)
#define SCU_PTSG_REAL_TIME_CLOCK_RESERVED_MASK  (0xFFFF0000UL)

#define SCU_RTCR_GEN_VAL(name, val) \
    SCU_GEN_VALUE(SCU_PTSG_##name, val)


#define SCU_PTSG_REAL_TIME_CLOCK_CONTROL_PRESCALER_VALUE_SHIFT  (0UL)
#define SCU_PTSG_REAL_TIME_CLOCK_CONTROL_PRESCALER_VALUE_MASK   (0x00FFFFFFUL)
#define SCU_PTSG_REAL_TIME_CLOCK_CONTROL_RESERVED_MASK          (0xFF000000UL)

#define SCU_RTCCR_GEN_VAL(name, val) \
   SCU_GEN_VALUE(SCU_PTSG_REAL_TIME_CLOCK_CONTROL_##name, val)


#define SCU_PTSG_PORT_TASK_SCHEDULER_CONTROL_SUSPEND_SHIFT  (0UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_CONTROL_SUSPEND_MASK   (0x00000001UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_CONTROL_ENABLE_SHIFT   (1UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_CONTROL_ENABLE_MASK    (0x00000002UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_CONTROL_RESERVED_MASK  (0xFFFFFFFCUL)

#define SCU_PTSxCR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_PTSG_PORT_TASK_SCHEDULER_CONTROL_##name)


#define SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_NEXT_RN_VALID_SHIFT             (0UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_NEXT_RN_VALID_MASK              (0x00000001UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_ACTIVE_RNSC_LIST_VALID_SHIFT    (1UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_ACTIVE_RNSC_LIST_VALID_MASK     (0x00000002UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_PTS_SUSPENDED_SHIFT             (2UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_PTS_SUSPENDED_MASK              (0x00000004UL)
#define SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_RESERVED_MASK                   (0xFFFFFFF8UL)

#define SCU_PTSxSR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_PTSG_PORT_TASK_SCHEDULER_STATUS_##name)


//*****************************************************************************
//* SGPIO Register shift and mask values
//*****************************************************************************
#define SCU_SGPIO_CONTROL_SGPIO_ENABLE_SHIFT                    (0UL)
#define SCU_SGPIO_CONTROL_SGPIO_ENABLE_MASK                     (0x00000001UL)
#define SCU_SGPIO_CONTROL_SGPIO_SERIAL_CLOCK_SELECT_SHIFT       (1UL)
#define SCU_SGPIO_CONTROL_SGPIO_SERIAL_CLOCK_SELECT_MASK        (0x00000002UL)
#define SCU_SGPIO_CONTROL_SGPIO_SERIAL_SHIFT_WIDTH_SELECT_SHIFT (2UL)
#define SCU_SGPIO_CONTROL_SGPIO_SERIAL_SHIFT_WIDTH_SELECT_MASK  (0x00000004UL)
#define SCU_SGPIO_CONTROL_SGPIO_TEST_BIT_SHIFT                  (15UL)
#define SCU_SGPIO_CONTROL_SGPIO_TEST_BIT_MASK                   (0x00008000UL)
#define SCU_SGPIO_CONTROL_SGPIO_RESERVED_MASK                   (0xFFFF7FF8UL)

#define SCU_SGICRx_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SGPIO_CONTROL_SGPIO_##name)

#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R0_SHIFT      (0UL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R0_MASK       (0x0000000FUL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R1_SHIFT      (4UL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R1_MASK       (0x000000F0UL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R2_SHIFT      (8UL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R2_MASK       (0x00000F00UL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R3_SHIFT      (12UL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_R3_MASK       (0x0000F000UL)
#define SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_RESERVED_MASK (0xFFFF0000UL)

#define SCU_SGPBRx_GEN_VAL(name, valueUL) \
    SCU_GEN_VALUE(SCU_SGPIO_PROGRAMMABLE_BLINK_REGISTER_##name, value)

#define SCU_SGPIO_START_DRIVE_LOWER_R0_SHIFT        (0UL)
#define SCU_SGPIO_START_DRIVE_LOWER_R0_MASK         (0x00000003UL)
#define SCU_SGPIO_START_DRIVE_LOWER_R1_SHIFT        (4UL)
#define SCU_SGPIO_START_DRIVE_LOWER_R1_MASK         (0x00000030UL)
#define SCU_SGPIO_START_DRIVE_LOWER_R2_SHIFT        (8UL)
#define SCU_SGPIO_START_DRIVE_LOWER_R2_MASK         (0x00000300UL)
#define SCU_SGPIO_START_DRIVE_LOWER_R3_SHIFT        (12UL)
#define SCU_SGPIO_START_DRIVE_LOWER_R3_MASK         (0x00003000UL)
#define SCU_SGPIO_START_DRIVE_LOWER_RESERVED_MASK   (0xFFFF8888UL)

#define SCU_SGSDLRx_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SGPIO_START_DRIVE_LOWER_##name, value)

#define SCU_SGPIO_START_DRIVE_UPPER_R0_SHIFT        (0UL)
#define SCU_SGPIO_START_DRIVE_UPPER_R0_MASK         (0x00000003UL)
#define SCU_SGPIO_START_DRIVE_UPPER_R1_SHIFT        (4UL)
#define SCU_SGPIO_START_DRIVE_UPPER_R1_MASK         (0x00000030UL)
#define SCU_SGPIO_START_DRIVE_UPPER_R2_SHIFT        (8UL)
#define SCU_SGPIO_START_DRIVE_UPPER_R2_MASK         (0x00000300UL)
#define SCU_SGPIO_START_DRIVE_UPPER_R3_SHIFT        (12UL)
#define SCU_SGPIO_START_DRIVE_UPPER_R3_MASK         (0x00003000UL)
#define SCU_SGPIO_START_DRIVE_UPPER_RESERVED_MASK   (0xFFFF8888UL)

#define SCU_SGSDURx_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SGPIO_START_DRIVE_LOWER_##name, value)

#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D0_SHIFT      (0UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D0_MASK       (0x00000003UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D1_SHIFT      (4UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D1_MASK       (0x00000030UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D2_SHIFT      (8UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D2_MASK       (0x00000300UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D3_SHIFT      (12UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_D3_MASK       (0x00003000UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_RESERVED_MASK (0xFFFF8888UL)

#define SCU_SGSIDLRx_GEN_VAL(name, valueUL) \
    SCU_GEN_VALUE(SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_##name, value)

#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D0_SHIFT      (0UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D0_MASK       (0x00000003UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D1_SHIFT      (4UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D1_MASK       (0x00000030UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D2_SHIFT      (8UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D2_MASK       (0x00000300UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D3_SHIFT      (12UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_D3_MASK       (0x00003000UL)
#define SCU_SGPIO_SERIAL_INPUT_DATA_UPPER_RESERVED_MASK (0xFFFF8888UL)

#define SCU_SGSIDURx_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SGPIO_SERIAL_INPUT_DATA_LOWER_##name, value)

#define SCU_SGPIO_VENDOR_SPECIFIC_CODE_SHIFT            (0UL)
#define SCU_SGPIO_VENDOR_SPECIFIC_CODE_MASK             (0x0000000FUL)
#define SCU_SGPIO_VENDOR_SPECIFIC_CODE_RESERVED_MASK    (0xFFFFFFF0UL)

#define SCU_SGVSCR_GEN_VAL(value) \
    SCU_GEN_VALUE(SCU_SGPIO_VENDOR_SPECIFIC_CODE##name, value)

#define SCU_SGPIO_OUPUT_DATA_SELECT_INPUT_DATA0_SHIFT           (0UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INPUT_DATA0_MASK            (0x00000003UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INVERT_INPUT_DATA0_SHIFT    (2UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INVERT_INPUT_DATA0_MASK     (0x00000004UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_JOG_ENABLE_DATA0_SHIFT      (3UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_JOG_ENABLE_DATA0_MASK       (0x00000008UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INPUT_DATA1_SHIFT           (4UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INPUT_DATA1_MASK            (0x00000030UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INVERT_INPUT_DATA1_SHIFT    (6UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INVERT_INPUT_DATA1_MASK     (0x00000040UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_JOG_ENABLE_DATA1_SHIFT      (7UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_JOG_ENABLE_DATA1_MASK       (0x00000080UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INPUT_DATA2_SHIFT           (8UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INPUT_DATA2_MASK            (0x00000300UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INVERT_INPUT_DATA2_SHIFT    (10UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_INVERT_INPUT_DATA2_MASK     (0x00000400UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_JOG_ENABLE_DATA2_SHIFT      (11UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_JOG_ENABLE_DATA2_MASK       (0x00000800UL)
#define SCU_SGPIO_OUPUT_DATA_SELECT_RESERVED_MASK               (0xFFFFF000UL)

#define SCU_SGODSR_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SGPIO_OUPUT_DATA_SELECT_##name, value)

#define SCU_SGODSR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SGPIO_OUPUT_DATA_SELECT_##name)

#ifdef ARLINGTON_BUILD
typedef char LEX_REGISTERS_T;
#endif

//*****************************************************************************
//* SMU Registers
//*****************************************************************************

// ----------------------------------------------------------------------------
// SMU Registers
// These registers are based off of BAR0
//
// To calculate the offset for other functions use
//       BAR0 + FN# * SystemPageSize * 2
//
// The TCA is only accessable from FN#0 (Physical Function) and each
// is programmed by (BAR0 + SCU_SMU_TCA_OFFSET + (FN# * 0x04)) or
//    TCA0 for FN#0 is at BAR0 + 0x0400
//    TCA1 for FN#1 is at BAR0 + 0x0404
//    etc.
// ----------------------------------------------------------------------------
// Accessable to all FN#s
#define SCU_SMU_PCP_OFFSET          0x0000
#define SCU_SMU_AMR_OFFSET          0x0004
#define SCU_SMU_ISR_OFFSET          0x0010
#define SCU_SMU_IMR_OFFSET          0x0014
#define SCU_SMU_ICC_OFFSET          0x0018
#define SCU_SMU_HTTLBAR_OFFSET      0x0020
#define SCU_SMU_HTTUBAR_OFFSET      0x0024
#define SCU_SMU_TCR_OFFSET          0x0028
#define SCU_SMU_CQLBAR_OFFSET       0x0030
#define SCU_SMU_CQUBAR_OFFSET       0x0034
#define SCU_SMU_CQPR_OFFSET         0x0040
#define SCU_SMU_CQGR_OFFSET         0x0044
#define SCU_SMU_CQC_OFFSET          0x0048
// Accessable to FN#0 only
#define SCU_SMU_RNCLBAR_OFFSET      0x0080
#define SCU_SMU_RNCUBAR_OFFSET      0x0084
#define SCU_SMU_DCC_OFFSET          0x0090
#define SCU_SMU_DFC_OFFSET          0x0094
#define SCU_SMU_SMUCSR_OFFSET       0x0098
#define SCU_SMU_SCUSRCR_OFFSET      0x009C
#define SCU_SMU_SMAW_OFFSET         0x00A0
#define SCU_SMU_SMDW_OFFSET         0x00A4
// Accessable to FN#0 only
#define SCU_SMU_TCA_OFFSET          0x0400
// Accessable to all FN#s
#define SCU_SMU_MT_MLAR0_OFFSET     0x2000
#define SCU_SMU_MT_MUAR0_OFFSET     0x2004
#define SCU_SMU_MT_MDR0_OFFSET      0x2008
#define SCU_SMU_MT_VCR0_OFFSET      0x200C
#define SCU_SMU_MT_MLAR1_OFFSET     0x2010
#define SCU_SMU_MT_MUAR1_OFFSET     0x2014
#define SCU_SMU_MT_MDR1_OFFSET      0x2018
#define SCU_SMU_MT_VCR1_OFFSET      0x201C
#define SCU_SMU_MPBA_OFFSET         0x3000

/**
 * @struct SMU_REGISTERS
 *
 * @brief These are the SMU registers
 *        See SCU SMU Specification on how this register space is used.
 */
typedef struct SMU_REGISTERS
{
// 0x0000 PCP
   U32   post_context_port;
// 0x0004 AMR
   U32   address_modifier;
   U32   reserved_08;
   U32   reserved_0C;
// 0x0010 ISR
   U32   interrupt_status;
// 0x0014 IMR
   U32   interrupt_mask;
// 0x0018 ICC
   U32   interrupt_coalesce_control;
   U32   reserved_1C;
// 0x0020 HTTLBAR
   U32   host_task_table_lower;
// 0x0024 HTTUBAR
   U32   host_task_table_upper;
// 0x0028 TCR
   U32   task_context_range;
   U32   reserved_2C;
// 0x0030 CQLBAR
   U32   completion_queue_lower;
// 0x0034 CQUBAR
   U32   completion_queue_upper;
   U32   reserved_38;
   U32   reserved_3C;
// 0x0040 CQPR
   U32   completion_queue_put;
// 0x0044 CQGR
   U32   completion_queue_get;
// 0x0048 CQC
   U32   completion_queue_control;
   U32   reserved_4C;
   U32   reserved_5x[4];
   U32   reserved_6x[4];
   U32   reserved_7x[4];
// Accessable to FN#0 only
// 0x0080 RNCLBAR
   U32   remote_node_context_lower;
// 0x0084 RNCUBAR
   U32   remote_node_context_upper;
   U32   reserved_88;
   U32   reserved_8C;
// 0x0090 DCC
   U32   device_context_capacity;
// 0x0094 DFC
   U32   device_function_capacity;
// 0x0098 SMUCSR
   U32   control_status;
// 0x009C SCUSRCR
   U32   soft_reset_control;
// 0x00A0 SMAW
   U32   mmr_address_window;
// 0x00A4 SMDW
   U32   mmr_data_window;
// 0x00A8 CGUCR
   U32   clock_gating_control;
// 0x00AC CGUPC
   U32   clock_gating_performance;
// A whole bunch of reserved space
   U32   reserved_Bx[4];
   U32   reserved_Cx[4];
   U32   reserved_Dx[4];
   U32   reserved_Ex[4];
   U32   reserved_Fx[4];
   U32   reserved_1xx[64];
   U32   reserved_2xx[64];
   U32   reserved_3xx[64];
// Accessable to FN#0 only
// 0x0400 TCA
   U32   task_context_assignment[256];
// MSI-X registers not included
} SMU_REGISTERS_T;

//*****************************************************************************
// SDMA Registers
//*****************************************************************************
#define SCU_SDMA_BASE               0x6000
#define SCU_SDMA_PUFATLHAR_OFFSET   0x0000
#define SCU_SDMA_PUFATUHAR_OFFSET   0x0004
#define SCU_SDMA_UFLHBAR_OFFSET     0x0008
#define SCU_SDMA_UFUHBAR_OFFSET     0x000C
#define SCU_SDMA_UFQC_OFFSET        0x0010
#define SCU_SDMA_UFQPP_OFFSET       0x0014
#define SCU_SDMA_UFQGP_OFFSET       0x0018
#define SCU_SDMA_PDMACR_OFFSET      0x001C
#define SCU_SDMA_CDMACR_OFFSET      0x0080

/**
 * @struct SCU_SDMA_REGISTERS
 *
 * @brief These are the SCU SDMA Registers
 *        See SCU SDMA specification on how these registers are used.
 */
typedef struct SCU_SDMA_REGISTERS
{
// 0x0000 PUFATLHAR
   U32   uf_address_table_lower;
// 0x0004 PUFATUHAR
   U32   uf_address_table_upper;
// 0x0008 UFLHBAR
   U32   uf_header_base_address_lower;
// 0x000C UFUHBAR
   U32   uf_header_base_address_upper;
// 0x0010 UFQC
   U32   unsolicited_frame_queue_control;
// 0x0014 UFQPP
   U32   unsolicited_frame_put_pointer;
// 0x0018 UFQGP
   U32   unsolicited_frame_get_pointer;
// 0x001C PDMACR
   U32   pdma_configuration;
// Reserved until offset 0x80
   U32   reserved_0020_007C[0x18];
// 0x0080 CDMACR
   U32   cdma_configuration;
// Remainder SDMA register space
   U32   reserved_0084_0400[0xDF];

} SCU_SDMA_REGISTERS_T;

//*****************************************************************************
//* SCU Link Registers
//*****************************************************************************
#define SCU_PEG0_OFFSET    0x0000
#define SCU_PEG1_OFFSET    0x8000

#define SCU_TL0_OFFSET     0x0000
#define SCU_TL1_OFFSET     0x0400
#define SCU_TL2_OFFSET     0x0800
#define SCU_TL3_OFFSET     0x0C00

#define SCU_LL_OFFSET      0x0080
#define SCU_LL0_OFFSET     (SCU_TL0_OFFSET + SCU_LL_OFFSET)
#define SCU_LL1_OFFSET     (SCU_TL1_OFFSET + SCU_LL_OFFSET)
#define SCU_LL2_OFFSET     (SCU_TL2_OFFSET + SCU_LL_OFFSET)
#define SCU_LL3_OFFSET     (SCU_TL3_OFFSET + SCU_LL_OFFSET)

// Transport Layer Offsets (PEG + TL)
#define SCU_TLCR_OFFSET         0x0000
#define SCU_TLADTR_OFFSET       0x0004
#define SCU_TLTTMR_OFFSET       0x0008
#define SCU_TLEECR0_OFFSET      0x000C
#define SCU_STPTLDARNI_OFFSET   0x0010


#define SCU_TLCR_HASH_SAS_CHECKING_ENABLE_SHIFT    (0UL)
#define SCU_TLCR_HASH_SAS_CHECKING_ENABLE_MASK     (0x00000001UL)
#define SCU_TLCR_CLEAR_TCI_NCQ_MAPPING_TABLE_SHIFT (1UL)
#define SCU_TLCR_CLEAR_TCI_NCQ_MAPPING_TABLE_MASK  (0x00000002UL)
#define SCU_TLCR_STP_WRITE_DATA_PREFETCH_SHIFT     (3UL)
#define SCU_TLCR_STP_WRITE_DATA_PREFETCH_MASK      (0x00000008UL)
#define SCU_TLCR_CMD_NAK_STATUS_CODE_SHIFT         (4UL)
#define SCU_TLCR_CMD_NAK_STATUS_CODE_MASK          (0x00000010UL)
#define SCU_TLCR_RESERVED_MASK                     (0xFFFFFFEBUL)

#define SCU_TLCR_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_TLCR_##name)

/**
 * @struct SCU_TRANSPORT_LAYER_REGISTERS
 *
 * @brief These are the SCU Transport Layer registers see SSPTL spec for how
 *        they are used.
 */
typedef struct SCU_TRANSPORT_LAYER_REGISTERS
{
   // 0x0000 TLCR
   U32 control;
   // 0x0004 TLADTR
   U32 arbitration_delay_timer;
   // 0x0008 TLTTMR
   U32 timer_test_mode;
   // 0x000C reserved
   U32 reserved_0C;
   // 0x0010 STPTLDARNI
   U32   stp_rni;
   // 0x0014 TLFEWPORCTRL
   U32 tlfe_wpo_read_control;
   // 0x0018 TLFEWPORDATA
   U32 tlfe_wpo_read_data;
   // 0x001C RXTLSSCSR1
   U32 rxtl_single_step_control_status_1;
   // 0x0020 RXTLSSCSR2
   U32 rxtl_single_step_control_status_2;
   // 0x0024 AWTRDDCR
   U32 tlfe_awt_retry_delay_debug_control;
   // Remainder of TL memory space
   U32 reserved_0028_007F[0x16];

} SCU_TRANSPORT_LAYER_REGISTERS_T;

// Protocol Engine Group Registers
#define SCU_SCUVZECRx_OFFSET        0x1080

// Link Layer Offsets (PEG + TL + LL)
#define SCU_SAS_SPDTOV_OFFSET       0x0000
#define SCU_SAS_LLSTA_OFFSET        0x0004
#define SCU_SATA_PSELTOV_OFFSET     0x0008
#define SCU_SAS_TIMETOV_OFFSET      0x0010
#define SCU_SAS_LOSTOT_OFFSET       0x0014
#define SCU_SAS_LNKTOV_OFFSET       0x0018
#define SCU_SAS_PHYTOV_OFFSET       0x001C
#define SCU_SAS_AFERCNT_OFFSET      0x0020
#define SCU_SAS_WERCNT_OFFSET       0x0024
#define SCU_SAS_TIID_OFFSET         0x0028
#define SCU_SAS_TIDNH_OFFSET        0x002C
#define SCU_SAS_TIDNL_OFFSET        0x0030
#define SCU_SAS_TISSAH_OFFSET       0x0034
#define SCU_SAS_TISSAL_OFFSET       0x0038
#define SCU_SAS_TIPID_OFFSET        0x003C
#define SCU_SAS_TIRES2_OFFSET       0x0040
#define SCU_SAS_ADRSTA_OFFSET       0x0044
#define SCU_SAS_MAWTTOV_OFFSET      0x0048
#define SCU_SAS_ECENCR_OFFSET       0x0050
#define SCU_SAS_FRPLDFIL_OFFSET     0x0054
#define SCU_SAS_RFCNT_OFFSET        0x0060
#define SCU_SAS_TFCNT_OFFSET        0x0064
#define SCU_SAS_RFDCNT_OFFSET       0x0068
#define SCU_SAS_TFDCNT_OFFSET       0x006C
#define SCU_SAS_LERCNT_OFFSET       0x0070
#define SCU_SAS_RDISERRCNT_OFFSET   0x0074
#define SCU_SAS_CRERCNT_OFFSET      0x0078
#define SCU_STPCTL_OFFSET           0x007C
#define SCU_SAS_PCFG_OFFSET         0x0080
#define SCU_SAS_CLKSM_OFFSET        0x0084
#define SCU_SAS_TXCOMWAKE_OFFSET    0x0088
#define SCU_SAS_TXCOMINIT_OFFSET    0x008C
#define SCU_SAS_TXCOMSAS_OFFSET     0x0090
#define SCU_SAS_COMINIT_OFFSET      0x0094
#define SCU_SAS_COMWAKE_OFFSET      0x0098
#define SCU_SAS_COMSAS_OFFSET       0x009C
#define SCU_SAS_SFERCNT_OFFSET      0x00A0
#define SCU_SAS_CDFERCNT_OFFSET     0x00A4
#define SCU_SAS_DNFERCNT_OFFSET     0x00A8
#define SCU_SAS_PRSTERCNT_OFFSET    0x00AC
#define SCU_SAS_CNTCTL_OFFSET       0x00B0
#define SCU_SAS_SSPTOV_OFFSET       0x00B4
#define SCU_FTCTL_OFFSET            0x00B8
#define SCU_FRCTL_OFFSET            0x00BC
#define SCU_FTWMRK_OFFSET           0x00C0
#define SCU_ENSPINUP_OFFSET         0x00C4
#define SCU_SAS_TRNTOV_OFFSET       0x00C8
#define SCU_SAS_PHYCAP_OFFSET       0x00CC
#define SCU_SAS_PHYCTL_OFFSET       0x00D0
#define SCU_SAS_LLCTL_OFFSET        0x00D8
#define SCU_AFE_XCVRCR_OFFSET       0x00DC
#define SCU_AFE_LUTCR_OFFSET        0x00E0

#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_ALIGN_DETECTION_SHIFT          (0UL)
#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_ALIGN_DETECTION_MASK           (0x000000FFUL)
#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_HOT_PLUG_SHIFT                 (8UL)
#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_HOT_PLUG_MASK                  (0x0000FF00UL)
#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_COMSAS_DETECTION_SHIFT         (16UL)
#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_COMSAS_DETECTION_MASK          (0x00FF0000UL)
#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_RATE_CHANGE_SHIFT              (24UL)
#define SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_RATE_CHANGE_MASK               (0xFF000000UL)

#define SCU_SAS_PHYTOV_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SAS_PHY_TIMER_TIMEOUT_VALUES_##name, value)

#define SCU_SAS_LINK_LAYER_CONTROL_MAX_LINK_RATE_SHIFT                  (0UL)
#define SCU_SAS_LINK_LAYER_CONTROL_MAX_LINK_RATE_MASK                   (0x00000003UL)
#define SCU_SAS_LINK_LAYER_CONTROL_MAX_LINK_RATE_GEN1                   (0UL)
#define SCU_SAS_LINK_LAYER_CONTROL_MAX_LINK_RATE_GEN2                   (1UL)
#define SCU_SAS_LINK_LAYER_CONTROL_MAX_LINK_RATE_GEN3                   (2UL)
#define SCU_SAS_LINK_LAYER_CONTROL_BROADCAST_PRIMITIVE_SHIFT            (2UL)
#define SCU_SAS_LINK_LAYER_CONTROL_BROADCAST_PRIMITIVE_MASK             (0x000003FCUL)
#define SCU_SAS_LINK_LAYER_CONTROL_CLOSE_NO_ACTIVE_TASK_DISABLE_SHIFT   (16UL)
#define SCU_SAS_LINK_LAYER_CONTROL_CLOSE_NO_ACTIVE_TASK_DISABLE_MASK    (0x00010000UL)
#define SCU_SAS_LINK_LAYER_CONTROL_CLOSE_NO_OUTBOUND_TASK_DISABLE_SHIFT (17UL)
#define SCU_SAS_LINK_LAYER_CONTROL_CLOSE_NO_OUTBOUND_TASK_DISABLE_MASK  (0x00020000UL)
#define SCU_SAS_LINK_LAYER_CONTROL_NO_OUTBOUND_TASK_TIMEOUT_SHIFT       (24UL)
#define SCU_SAS_LINK_LAYER_CONTROL_NO_OUTBOUND_TASK_TIMEOUT_MASK        (0xFF000000UL)
#define SCU_SAS_LINK_LAYER_CONTROL_RESERVED                             (0x00FCFC00UL)

#define SCU_SAS_LLCTL_GEN_VAL(name, value) \
    SCU_GEN_VALUE(SCU_SAS_LINK_LAYER_CONTROL_##name, value)

#define SCU_SAS_LLCTL_GEN_BIT(name) \
    SCU_GEN_BIT(SCU_SAS_LINK_LAYER_CONTROL_##name)

//#define SCU_FRXHECR_DCNT_OFFSET      0x00B0
#define SCU_PSZGCR_OFFSET           0x00E4
#define SCU_SAS_RECPHYCAP_OFFSET    0x00E8
//#define SCU_TX_LUTSEL_OFFSET         0x00B8

#define SCU_SAS_PTxC_OFFSET         0x00D4 // Same offset as SAS_TCTSTM

// This macro define the DWORD SYNC ACQUIRED bit in link layer status register.
#define SCU_SAS_LLSTA_DWORD_SYNCA_BIT   0x4

/**
 * @struct SCU_LINK_LAYER_REGISTERS
 *
 * @brief SCU Link Layer Registers
 *        See the SCU SSLL Specification on how these registers are used.
 */
typedef struct SCU_LINK_LAYER_REGISTERS
{
// 0x0000 SAS_SPDTOV
   U32   speed_negotiation_timers;
// 0x0004 SAS_LLSTA
   U32   link_layer_status;
// 0x0008 SATA_PSELTOV
   U32   port_selector_timeout;
   U32   reserved0C;
// 0x0010 SAS_TIMETOV
   U32   timeout_unit_value;
// 0x0014 SAS_RCDTOV
   U32   rcd_timeout;
// 0x0018 SAS_LNKTOV
   U32   link_timer_timeouts;
// 0x001C SAS_PHYTOV
   U32   phy_timer_timeout_values;
// 0x0020 SAS_AFERCNT
   U32   received_address_frame_error_counter;
// 0x0024 SAS_WERCNT
   U32   invalid_dword_counter;
// 0x0028 SAS_TIID
   U32   transmit_identification;
// 0x002C SAS_TIDNH
   U32   sas_device_name_high;
// 0x0030 SAS_TIDNL
   U32   sas_device_name_low;
// 0x0034 SAS_TISSAH
   U32   source_sas_address_high;
// 0x0038 SAS_TISSAL
   U32   source_sas_address_low;
// 0x003C SAS_TIPID
   U32   identify_frame_phy_id;
// 0x0040 SAS_TIRES2
   U32   identify_frame_reserved;
// 0x0044 SAS_ADRSTA
   U32   received_address_frame;
// 0x0048 SAS_MAWTTOV
   U32   maximum_arbitration_wait_timer_timeout;
// 0x004C SAS_PTxC
   U32   transmit_primitive;
// 0x0050 SAS_ECENCR
   U32   error_counter_event_notification_control;
// 0x0054 SAS_FRPLDFIL
   U32   frxq_payload_fill_threshold;
// 0x0058 SAS_LLHANG_TOT
   U32   link_layer_hang_detection_timeout;
   U32   reserved_5C;
// 0x0060 SAS_RFCNT
   U32   received_frame_count;
// 0x0064 SAS_TFCNT
   U32   transmit_frame_count;
// 0x0068 SAS_RFDCNT
   U32   received_dword_count;
// 0x006C SAS_TFDCNT
   U32   transmit_dword_count;
// 0x0070 SAS_LERCNT
   U32   loss_of_sync_error_count;
// 0x0074 SAS_RDISERRCNT
   U32   running_disparity_error_count;
// 0x0078 SAS_CRERCNT
   U32   received_frame_crc_error_count;
// 0x007C STPCTL
   U32   stp_control;
// 0x0080 SAS_PCFG
   U32   phy_configuration;
// 0x0084 SAS_CLKSM
   U32   clock_skew_management;
// 0x0088 SAS_TXCOMWAKE
   U32   transmit_comwake_signal;
// 0x008C SAS_TXCOMINIT
   U32   transmit_cominit_signal;
// 0x0090 SAS_TXCOMSAS
   U32   transmit_comsas_signal;
// 0x0094 SAS_COMINIT
   U32   cominit_control;
// 0x0098 SAS_COMWAKE
   U32   comwake_control;
// 0x009C SAS_COMSAS
   U32   comsas_control;
// 0x00A0 SAS_SFERCNT
   U32   received_short_frame_count;
// 0x00A4 SAS_CDFERCNT
   U32   received_frame_without_credit_count;
// 0x00A8 SAS_DNFERCNT
   U32   received_frame_after_done_count;
// 0x00AC SAS_PRSTERCNT
   U32   phy_reset_problem_count;
// 0x00B0 SAS_CNTCTL
   U32   counter_control;
// 0x00B4 SAS_SSPTOV
   U32   ssp_timer_timeout_values;
// 0x00B8 FTCTL
   U32   ftx_control;
// 0x00BC FRCTL
   U32   frx_control;
// 0x00C0 FTWMRK
   U32   ftx_watermark;
// 0x00C4 ENSPINUP
   U32   notify_enable_spinup_control;
// 0x00C8 SAS_TRNTOV
   U32   sas_training_sequence_timer_values;
// 0x00CC SAS_PHYCAP
   U32   phy_capabilities;
// 0x00D0 SAS_PHYCTL
   U32   phy_control;
   U32   reserved_d4;
// 0x00D8 LLCTL
   U32   link_layer_control;
// 0x00DC AFE_XCVRCR
   U32   afe_xcvr_control;
// 0x00E0 AFE_LUTCR
   U32   afe_lookup_table_control;
// 0x00E4 PSZGCR
   U32   phy_source_zone_group_control;
// 0x00E8 SAS_RECPHYCAP
   U32   receive_phycap;
   U32   reserved_ec;
// 0x00F0 SNAFERXRSTCTL
   U32   speed_negotiation_afe_rx_reset_control;
// 0x00F4 SAS_SSIPMCTL
   U32   power_management_control;
// 0x00F8 SAS_PSPREQ_PRIM
   U32   sas_pm_partial_request_primitive;
// 0x00FC SAS_PSSREQ_PRIM
   U32   sas_pm_slumber_request_primitive;
// 0x0100 SAS_PPSACK_PRIM
   U32   sas_pm_ack_primitive_register;
// 0x0104 SAS_PSNAK_PRIM
   U32   sas_pm_nak_primitive_register;
// 0x0108 SAS_SSIPMTOV
   U32   sas_primitive_timeout;
   U32   reserved_10c;
// 0x0110 - 0x011C PLAPRDCTRLxREG
   U32   pla_product_control[4];
// 0x0120 PLAPRDSUMREG
   U32   pla_product_sum;
// 0x0124 PLACONTROLREG
   U32   pla_control;
// Remainder of memory space 896 bytes
   U32   reserved_0128_037f[0x96];

} SCU_LINK_LAYER_REGISTERS_T;

// 0x00D4 // Same offset as SAS_TCTSTM SAS_PTxC
//   U32   primitive_transmit_control;

// ----------------------------------------------------------------------------
// SGPIO
// ----------------------------------------------------------------------------
#define SCU_SGPIO_OFFSET         0x1400

//#define SCU_SGPIO_OFFSET         0x6000   // later moves to 0x1400 see HSD 652625
#define SCU_SGPIO_SGICR_OFFSET   0x0000
#define SCU_SGPIO_SGPBR_OFFSET   0x0004
#define SCU_SGPIO_SGSDLR_OFFSET  0x0008
#define SCU_SGPIO_SGSDUR_OFFSET  0x000C
#define SCU_SGPIO_SGSIDLR_OFFSET 0x0010
#define SCU_SGPIO_SGSIDUR_OFFSET 0x0014
#define SCU_SGPIO_SGVSCR_OFFSET  0x0018
// Address from 0x0820 to 0x083C
#define SCU_SGPIO_SGODSR_OFFSET  0x0020

/**
 * @struct SCU_SGPIO_REGISTERS
 *
 * @brief SCU SGPIO Registers
 *        See the SCU SGPIO Specification on how these registers are used.
 */
typedef struct SCU_SGPIO_REGISTERS
{
// 0x0000 SGPIO_SGICR
   U32   interface_control;
// 0x0004 SGPIO_SGPBR
   U32   blink_rate;
// 0x0008 SGPIO_SGSDLR
   U32   start_drive_lower;
// 0x000C SGPIO_SGSDUR
   U32   start_drive_upper;
// 0x0010 SGPIO_SGSIDLR
   U32   serial_input_lower;
// 0x0014 SGPIO_SGSIDUR
   U32   serial_input_upper;
// 0x0018 SGPIO_SGVSCR
   U32   vendor_specific_code;
// 0x001C Reserved
   U32   reserved_001C;
// 0x0020 SGPIO_SGODSR
   U32   output_data_select[8];
// Remainder of memory space 256 bytes
   U32   reserved_1444_14ff[0x30];

} SCU_SGPIO_REGISTERS_T;

//*****************************************************************************
//* Defines for VIIT entry offsets
//* Access additional entries by SCU_VIIT_BASE + index * 0x10
//*****************************************************************************
#define     SCU_VIIT_BASE     0x1c00

struct SCU_VIIT_REGISTERS
{
   U32 registers[256];
};

//*****************************************************************************
//* SCU PORT TASK SCHEDULER REGISTERS
//*****************************************************************************

#define SCU_PTSG_BASE               0x1000

#define SCU_PTSG_PTSGCR_OFFSET      0x0000
#define SCU_PTSG_RTCR_OFFSET        0x0004
#define SCU_PTSG_RTCCR_OFFSET       0x0008
#define SCU_PTSG_PTS0CR_OFFSET      0x0010
#define SCU_PTSG_PTS0SR_OFFSET      0x0014
#define SCU_PTSG_PTS1CR_OFFSET      0x0018
#define SCU_PTSG_PTS1SR_OFFSET      0x001C
#define SCU_PTSG_PTS2CR_OFFSET      0x0020
#define SCU_PTSG_PTS2SR_OFFSET      0x0024
#define SCU_PTSG_PTS3CR_OFFSET      0x0028
#define SCU_PTSG_PTS3SR_OFFSET      0x002C
#define SCU_PTSG_PCSPE0CR_OFFSET    0x0030
#define SCU_PTSG_PCSPE1CR_OFFSET    0x0034
#define SCU_PTSG_PCSPE2CR_OFFSET    0x0038
#define SCU_PTSG_PCSPE3CR_OFFSET    0x003C
#define SCU_PTSG_ETMTSCCR_OFFSET    0x0040
#define SCU_PTSG_ETMRNSCCR_OFFSET   0x0044

/**
 * @struct SCU_PORT_TASK_SCHEDULER_REGISTERS
 *
 * @brief These are the control/stats pairs for each Port Task Scheduler.
 *        See the SCU SCHED Specification on how these registers are used.
 */
typedef struct SCU_PORT_TASK_SCHEDULER_REGISTERS
{
   U32   control;
   U32   status;
} SCU_PORT_TASK_SCHEDULER_REGISTERS_T;

typedef U32 SCU_PORT_PE_CONFIGURATION_REGISTER_T;

/**
 * @struct SCU_PORT_TASK_SCHEDULER_GROUP_REGISTERS
 *
 * @brief These are the PORT Task Scheduler registers
 *        See the SCU SCHED Specification on how these registers are used.
 */
typedef struct SCU_PORT_TASK_SCHEDULER_GROUP_REGISTERS
{
// 0x0000 PTSGCR
   U32   control;
// 0x0004 RTCR
   U32   real_time_clock;
// 0x0008 RTCCR
   U32   real_time_clock_control;
// 0x000C
   U32   reserved_0C;
// 0x0010 PTS0CR
// 0x0014 PTS0SR
// 0x0018 PTS1CR
// 0x001C PTS1SR
// 0x0020 PTS2CR
// 0x0024 PTS2SR
// 0x0028 PTS3CR
// 0x002C PTS3SR
   SCU_PORT_TASK_SCHEDULER_REGISTERS_T port[4];
// 0x0030 PCSPE0CR
// 0x0034 PCSPE1CR
// 0x0038 PCSPE2CR
// 0x003C PCSPE3CR
   SCU_PORT_PE_CONFIGURATION_REGISTER_T protocol_engine[4];
// 0x0040 ETMTSCCR
   U32   tc_scanning_interval_control;
// 0x0044 ETMRNSCCR
   U32   rnc_scanning_interval_control;
// Remainder of memory space 128 bytes
   U32   reserved_1048_107f[0x0E];

} SCU_PORT_TASK_SCHEDULER_GROUP_REGISTERS_T;

#define SCU_PTSG_SCUVZECR_OFFSET        0x003C

//*****************************************************************************
//* AFE REGISTERS
//*****************************************************************************
#define SCU_AFE_MMR_BASE                  0xE000

#if defined(ARLINGTON_BUILD)
#define SCU_AFE_PLL_CTL_OFFSET            0x0000
#define SCU_AFE_RXPI_CTL_OFFSET           0x0004
#define SCU_AFE_MBIAS_CTL0_OFFSET         0x000C
#define SCU_AFE_MBIAS_CTL1_OFFSET         0x0010
#define SCU_AFE_COMM_STA_OFFSET           0x0020
#define SCU_AFE_RXPI_STA_OFFSET           0x0024
#define SCU_AFE_XCVR0_CTL0_OFFSET         0x0040
#define SCU_AFE_XCVR1_CTL0_OFFSET         0x0044
#define SCU_AFE_XCVR2_CTL0_OFFSET         0x0048
#define SCU_AFE_XCVR3_CTL0_OFFSET         0x004C
#define SCU_AFE_XCVR0_CTL1_OFFSET         0x0050
#define SCU_AFE_XCVR1_CTL1_OFFSET         0x0054
#define SCU_AFE_XCVR2_CTL1_OFFSET         0x0058
#define SCU_AFE_XCVR3_CTL1_OFFSET         0x005C
#define SCU_AFE_XCVR0_RXEQ_CTL_OFFSET     0x0060
#define SCU_AFE_XCVR1_RXEQ_CTL_OFFSET     0x0064
#define SCU_AFE_XCVR2_RXEQ_CTL_OFFSET     0x0068
#define SCU_AFE_XCVR3_RXEQ_CTL_OFFSET     0x006C
#define SCU_AFE_XCVR0_CDR_STA_OFFSET      0x0080
#define SCU_AFE_XCVR1_CDR_STA_OFFSET      0x0084
#define SCU_AFE_XCVR2_CDR_STA_OFFSET      0x0088
#define SCU_AFE_XCVR3_CDR_STA_OFFSET      0x008C
#define SCU_AFE_XCVR0_RXEQ_STA0_OFFSET    0x0090
#define SCU_AFE_XCVR1_RXEQ_STA0_OFFSET    0x0094
#define SCU_AFE_XCVR2_RXEQ_STA0_OFFSET    0x0098
#define SCU_AFE_XCVR3_RXEQ_STA0_OFFSET    0x009C
#define SCU_AFE_XCVR0_RXEQ_STA1_OFFSET    0x00A0
#define SCU_AFE_XCVR1_RXEQ_STA1_OFFSET    0x00A4
#define SCU_AFE_XCVR2_RXEQ_STA1_OFFSET    0x00A8
#define SCU_AFE_XCVR3_RXEQ_STA1_OFFSET    0x00AC
#define SCU_AFE_DFX_MSTR_CTL_OFFSET       0x0104
#define SCU_AFE_NTL_CTL_OFFSET            0x010C
#define SCU_AFE_DFX_XCVR_STA_CLR_OFFSET   0x0120
#define SCU_AFE_NTL_STA_OFFSET            0x0124
#define SCU_AFE_DFX_XCVR0_STA0_OFFSET     0x0130
#define SCU_AFE_DFX_XCVR1_STA0_OFFSET     0x0134
#define SCU_AFE_DFX_XCVR2_STA0_OFFSET     0x0138
#define SCU_AFE_DFX_XCVR3_STA0_OFFSET     0x013C
#define SCU_AFE_DFX_XCVR0_STA1_OFFSET     0x0140
#define SCU_AFE_DFX_XCVR1_STA1_OFFSET     0x0144
#define SCU_AFE_DFX_XCVR2_STA1_OFFSET     0x0148
#define SCU_AFE_DFX_XCVR3_STA1_OFFSET     0x014C
#define SCU_AFE_DFX_MON_CTL_OFFSET        0x0150

#define SCU_AFE_DFX_RX_CTL0_AFE0_XCVR0_OFFSET       0x0180
#define SCU_AFE_DFX_RX_CTL0_AFE0_XCVR1_OFFSET       0x0184
#define SCU_AFE_DFX_RX_CTL0_AFE0_XCVR2_OFFSET       0x0188
#define SCU_AFE_DFX_RX_CTL0_AFE0_XCVR3_OFFSET       0x018C
#define SCU_AFE_DFX_RX_CTL0_AFE1_XCVR0_OFFSET       0x0980
#define SCU_AFE_DFX_RX_CTL0_AFE1_XCVR1_OFFSET       0x0984
#define SCU_AFE_DFX_RX_CTL0_AFE1_XCVR2_OFFSET       0x0988
#define SCU_AFE_DFX_RX_CTL0_AFE1_XCVR3_OFFSET       0x098C

#define SCU_AFE_DFX_RX_CTL1_AFE0_XCVR0_OFFSET       0x0190
#define SCU_AFE_DFX_RX_CTL1_AFE0_XCVR1_OFFSET       0x0194
#define SCU_AFE_DFX_RX_CTL1_AFE0_XCVR2_OFFSET       0x0198
#define SCU_AFE_DFX_RX_CTL1_AFE0_XCVR3_OFFSET       0x019C
#define SCU_AFE_DFX_RX_CTL1_AFE1_XCVR0_OFFSET       0x0990
#define SCU_AFE_DFX_RX_CTL1_AFE1_XCVR1_OFFSET       0x0994
#define SCU_AFE_DFX_RX_CTL1_AFE1_XCVR2_OFFSET       0x0998
#define SCU_AFE_DFX_RX_CTL1_AFE1_XCVR3_OFFSET       0x099C

#define SCU_AFE_PLL_DFX_CTL_OFFSET        0x01C0

#define SCU_AFE_XCVR0_DFX_DATA_OFFSET     0x0200   // [0:0F]
#define SCU_AFE_XCVR0_CC_OFFSET           0x0240
#define SCU_AFE_XCVR0_DFX_IR_OFFSET       0x0250   // [0:1F]

#define SCU_AFE_XCVR1_DFX_DATA_OFFSET     0x0300   // [0:0F]
#define SCU_AFE_XCVR1_CC_OFFSET           0x0340
#define SCU_AFE_XCVR1_DFX_IR_OFFSET       0x0350   // [0:1F]

#define SCU_AFE_XCVR2_DFX_DATA_OFFSET     0x0400   // [0:0F]
#define SCU_AFE_XCVR2_CC_OFFSET           0x0440
#define SCU_AFE_XCVR2_DFX_IR_OFFSET       0x0450   // [0:1F]

#define SCU_AFE_XCVR3_DFX_DATA_OFFSET     0x0500   // [0:0F]
#define SCU_AFE_XCVR3_CC_OFFSET           0x0540
#define SCU_AFE_XCVR3_DFX_IR_OFFSET       0x0550   // [0:1F]
#else // defined(ARLINGTON_BUILD)

#endif // defined(ARLINGTON_BUILD)

/**
 * @struct SCU_AFE_TRANSCEIVER
 *
 * @brief AFE Transceiver Registers
 *        See SCU AFE Specification for use of these registers.
 *
 * @note For ARLINGTON_BUILD see the SCU AFE specification.
 * @note For PLEASANT_RIDGE_BUILD build see the Uaoa AFE specification.
 */
#if defined(ARLINGTON_BUILD)
   struct SCU_AFE_TRANSCEIVER
   {
   // 0x00
      U32 afe_transceiver_dfx_data[0x10];
   // 0x40
      U32 afe_transceiver_dpg_cycle_control;
   // 0x44 - 0x4c
      U32 reserved_0044_004c[3];
   // 0x50
      U32 afe_transceiver_dfx_instruction[0x20];
   // 0xd0 - 0xfc
      U32 reserved_00d0_00fc[0x0C];
   };

#elif    defined(PLEASANT_RIDGE_BUILD) \
      || defined(PBG_HBA_A0_BUILD) \
      || defined(PBG_HBA_A2_BUILD) \
      || defined(PBG_HBA_BETA_BUILD) \
      || defined(PBG_BUILD)

   // AFE 0 is at offset 0x0800
   // AFE 1 is at offset 0x0900
   // AFE 2 is at offset 0x0a00
   // AFE 3 is at offset 0x0b00
   struct SCU_AFE_TRANSCEIVER
   {
   // 0x0000 AFE_XCVR_CTRL0
      U32 afe_xcvr_control0;
   // 0x0004 AFE_XCVR_CTRL1
      U32 afe_xcvr_control1;
   // 0x0008
      U32 reserved_0008;
   // 0x000c afe_dfx_rx_control0
      U32 afe_dfx_rx_control0;
   // 0x0010 AFE_DFX_RX_CTRL1
      U32 afe_dfx_rx_control1;
   // 0x0014
      U32 reserved_0014;
   // 0x0018 AFE_DFX_RX_STS0
      U32 afe_dfx_rx_status0;
   // 0x001c AFE_DFX_RX_STS1
      U32 afe_dfx_rx_status1;
   // 0x0020
      U32 reserved_0020;
   // 0x0024 AFE_TX_CTRL
      U32 afe_tx_control;
   // 0x0028 AFE_TX_AMP_CTRL0
      U32 afe_tx_amp_control0;
   // 0x002c AFE_TX_AMP_CTRL1
      U32 afe_tx_amp_control1;
   // 0x0030 AFE_TX_AMP_CTRL2
      U32 afe_tx_amp_control2;
   // 0x0034 AFE_TX_AMP_CTRL3
      U32 afe_tx_amp_control3;
   // 0x0038 afe_tx_ssc_control
      U32 afe_tx_ssc_control;
   // 0x003c
      U32 reserved_003c;
   // 0x0040 AFE_RX_SSC_CTRL0
      U32 afe_rx_ssc_control0;
   // 0x0044 AFE_RX_SSC_CTRL1
      U32 afe_rx_ssc_control1;
   // 0x0048 AFE_RX_SSC_CTRL2
      U32 afe_rx_ssc_control2;
   // 0x004c AFE_RX_EQ_STS0
      U32 afe_rx_eq_status0;
   // 0x0050 AFE_RX_EQ_STS1
      U32 afe_rx_eq_status1;
   // 0x0054 AFE_RX_CDR_STS
      U32 afe_rx_cdr_status;
   // 0x0058
      U32 reserved_0058;
   // 0x005c AFE_CHAN_CTRL
      U32 afe_channel_control;
   // 0x0060-0x006c
      U32 reserved_0060_006c[0x04];
   // 0x0070 AFE_XCVR_EC_STS0
      U32 afe_xcvr_error_capture_status0;
   // 0x0074 AFE_XCVR_EC_STS1
      U32 afe_xcvr_error_capture_status1;
   // 0x0078 AFE_XCVR_EC_STS2
      U32 afe_xcvr_error_capture_status2;
   // 0x007c afe_xcvr_ec_status3
      U32 afe_xcvr_error_capture_status3;
   // 0x0080 AFE_XCVR_EC_STS4
      U32 afe_xcvr_error_capture_status4;
   // 0x0084 AFE_XCVR_EC_STS5
      U32 afe_xcvr_error_capture_status5;
   // 0x0088-0x00fc
      U32 reserved_008c_00fc[0x1e];
   };
#else // !defined(PLEASANT_RIDGE_BUILD) && !defined(ARLINGTON_BUILD)
   #error "Target platform not defined."
#endif // defined(PLEASANT_RIDGE_BUILD) || defined(ARLINGTON_BUILD)

/**
 * @struct SCU_AFE_REGISTERS
 *
 * @brief AFE Regsiters
 *        See SCU AFE Specification for use of these registers.
 */
#if defined(ARLINGTON_BUILD)
   typedef struct SCU_AFE_REGISTERS
   {
   // 0x0000
      U32 afe_pll_control;
   // 0x0004
      U32 afe_phase_interplator_control;
   // 0x0008
      U32 reservd_0008;
   // 0x000C
      U32 afe_bias_control[2];
   // 0x0014 - 0x001c
      U32 reserved_0014_001c[3];
   // 0x0020
      U32 afe_common_status;
   // 0x0024
      U32 afe_phase_interpolator_status;
   // 0x0028 - 0x003C
      U32 reserved_0028_003c[6];
   // 0x0040
      U32 afe_transceiver_control0[4];
   // 0x0050
      U32 afe_transceiver_control1[4];
   // 0x0060
      U32 afe_transceiver_equalization_control[4];
   // 0x0070 - 0x007c
      U32 reserved_0070_007c[4];
   // 0x0080
      U32 afe_transceiver_cdr_status[4];
   // 0x0090
      U32 afe_transceiver_rx_equaliation_status_register0[4];
   // 0x00A0
      U32 afe_transceiver_rx_equaliation_status_register1[4];
   // 0x00B0 - 0x0100
      U32 reserved_00b0_0100[0x15];
   // 0x0104
      U32 afe_dfx_master_control;
   // 0x0108
      U32 reserved_0108;
   // 0x010c
      U32 afe_no_touch_leakage_control;
   // 0x0110 - 0x011C
      U32 reserved_0110_011c[4];
   // 0x0120
      U32 afe_dfx_transceiver_status_clear;
   // 0x0124
      U32 afe_no_touch_leakage_status;
   // 0x0128 - 0x012c
      U32 reserved_0128_012c[2];
   // 0x0130
      U32 afe_dfx_transceiver_status_register0[4];
   // 0x0140
      U32 afe_dfx_transceiver_status_register1[4];
   // 0x0150
      U32 afe_dfx_transmit_monitor_control;
   // 0x0154 - 0x017c
      U32 reserved_0154_017C[0x0B];
   // 0x0180
      U32 afe_dfx_receive_control_register0[4];
   // 0x0190
      U32 afe_dfx_receive_control_register1[4];
   // 0x1A0
      U32 afe_dfx_transmit_control_register[4];
   // 0x01B0 - 0x01BC
      U32 reserved_01b0_01bc[4];
   // 0x01C0
      U32 afe_pll_dfx_control;
   // 0x01c4 - 0x01fc
      U32 reserved_01c4_01fc[0x0F];
   // 0x0200 - 0x05fc
      struct SCU_AFE_TRANSCEIVER afe_transceiver[4];

   // 0x0600 - 0x06FC
      U32 reserved_0600_06FC[0x40];

   // 0x0700
      struct SCU_AFE_TRANSCEIVER afe_all_transceiver;

      U32 reserved_0800_2000[0x600];

   } SCU_AFE_REGISTERS_T;

#elif    defined(PLEASANT_RIDGE_BUILD) \
      || defined(PBG_HBA_A0_BUILD) \
      || defined(PBG_HBA_A2_BUILD) \
      || defined(PBG_HBA_BETA_BUILD) \
      || defined(PBG_BUILD)

   /* Uaoa AFE registers */
   typedef struct SCU_AFE_REGISTERS
   {
   // 0Xe000 AFE_BIAS_CTRL
      U32 afe_bias_control;
      U32 reserved_0004;
   // 0x0008 AFE_PLL_CTRL0
      U32 afe_pll_control0;
   // 0x000c AFE_PLL_CTRL1
      U32 afe_pll_control1;
   // 0x0010 AFE_PLL_CTRL2
      U32 afe_pll_control2;
   // 0x0014 AFE_CB_STS
      U32 afe_common_block_status;
   // 0x0018-0x007c
      U32 reserved_18_7c[0x1a];
   // 0x0080 AFE_PMSN_MCTRL0
      U32 afe_pmsn_master_control0;
   // 0x0084 AFE_PMSN_MCTRL1
      U32 afe_pmsn_master_control1;
   // 0x0088 AFE_PMSN_MCTRL2
      U32 afe_pmsn_master_control2;
   // 0x008C-0x00fc
      U32 reserved_008c_00fc[0x1D];
   // 0x0100 AFE_DFX_MST_CTRL0
      U32 afe_dfx_master_control0;
   // 0x0104 AFE_DFX_MST_CTRL1
      U32 afe_dfx_master_control1;
   // 0x0108 AFE_DFX_DCL_CTRL
      U32 afe_dfx_dcl_control;
   // 0x010c AFE_DFX_DMON_CTRL
      U32 afe_dfx_digital_monitor_control;
   // 0x0110 AFE_DFX_AMONP_CTRL
      U32 afe_dfx_analog_p_monitor_control;
   // 0x0114 AFE_DFX_AMONN_CTRL
      U32 afe_dfx_analog_n_monitor_control;
   // 0x0118 AFE_DFX_NTL_STS
      U32 afe_dfx_ntl_status;
   // 0x011c AFE_DFX_FIFO_STS0
      U32 afe_dfx_fifo_status0;
   // 0x0120 AFE_DFX_FIFO_STS1
      U32 afe_dfx_fifo_status1;
   // 0x0124 AFE_DFX_MPAT_CTRL
      U32 afe_dfx_master_pattern_control;
   // 0x0128 AFE_DFX_P0_CTRL
      U32 afe_dfx_p0_control;
   // 0x012c-0x01a8 AFE_DFX_P0_DRx
      U32 afe_dfx_p0_data[32];
   // 0x01ac
      U32 reserved_01ac;
   // 0x01b0-0x020c AFE_DFX_P0_IRx
      U32 afe_dfx_p0_instruction[24];
   // 0x0210
      U32 reserved_0210;
   // 0x0214 AFE_DFX_P1_CTRL
      U32 afe_dfx_p1_control;
   // 0x0218-0x245 AFE_DFX_P1_DRx
      U32 afe_dfx_p1_data[16];
   // 0x0258-0x029c
      U32 reserved_0258_029c[0x12];
   // 0x02a0-0x02bc AFE_DFX_P1_IRx
      U32 afe_dfx_p1_instruction[8];
   // 0x02c0-0x2fc
      U32 reserved_02c0_02fc[0x10];
   // 0x0300 AFE_DFX_TX_PMSN_CTRL
      U32 afe_dfx_tx_pmsn_control;
   // 0x0304 AFE_DFX_RX_PMSN_CTRL
      U32 afe_dfx_rx_pmsn_control;
      U32 reserved_0308;
   // 0x030c AFE_DFX_NOA_CTRL0
      U32 afe_dfx_noa_control0;
   // 0x0310 AFE_DFX_NOA_CTRL1
      U32 afe_dfx_noa_control1;
   // 0x0314 AFE_DFX_NOA_CTRL2
      U32 afe_dfx_noa_control2;
   // 0x0318 AFE_DFX_NOA_CTRL3
      U32 afe_dfx_noa_control3;
   // 0x031c AFE_DFX_NOA_CTRL4
      U32 afe_dfx_noa_control4;
   // 0x0320 AFE_DFX_NOA_CTRL5
      U32 afe_dfx_noa_control5;
   // 0x0324 AFE_DFX_NOA_CTRL6
      U32 afe_dfx_noa_control6;
   // 0x0328 AFE_DFX_NOA_CTRL7
      U32 afe_dfx_noa_control7;
   // 0x032c-0x07fc
      U32 reserved_032c_07fc[0x135];

   // 0x0800-0x0bfc
      struct SCU_AFE_TRANSCEIVER scu_afe_xcvr[4];

   // 0x0c00-0x0ffc
      U32 reserved_0c00_0ffc[0x0100];
   } SCU_AFE_REGISTERS_T;
#else // !defined(PBG_HBA_BUILD) && defined(PLEASANT_RIDGE_BUILD) && !defined(ARLINGTON_BUILD)
   #error "Target platform not defined."
#endif // defined(PBG_HBA_BUILD) || defined(PLEASANT_RIDGE_BUILD) || defined(ARLINGTON_BUILD)


struct SCU_PROTOCOL_ENGINE_GROUP_REGISTERS
{
   U32 table[0xE0];
};


struct SCU_VIIT_IIT
{
   U32 table[256];
};

/**
 * @brief Placeholder for the ZONE Partition Table information
 *        ZONING will not be included in the 1.1 release.
 *
 */
struct SCU_ZONE_PARTITION_TABLE
{
   U32 table[2048];
};

/**
 * @brief CRAM register. MMR base address for CRAMC is 0x6400
 *        relative to SCUBAR.
 *
 */
struct SCU_COMPLETION_RAM
{
   U32 sram_base_address_0;         //0x0000
   U32 sram_upper_base_address_0;   //0x0004
   U32 sram_ecc_control_0;          //0x0008
   U32 sram_ecc_log_0;              //0x000c
   U32 sram_ecc_addrress_0;         //0x0010
   U32 sram_ecc_context_address_0;  //0x0014
   U32 sram_ecc_test_0;             //0x0018
   U32 sram_parity_control_and_status_0;   //0x001C
   U32 sram_parity_address_0;              //0x0020
   U32 sram_parity_upper_address_0;        //0x0024
   U32 sram_parity_context_0;              //0x0028
   U32 sram_memory_controller_interrupt_status_0;   //0x002C
   U32 sram_mcu_read_arbiter_control_0;    //0x0030
   U32 sram_mcu_write_arbiter_control_0;   //0x0034
   U32 smcu_error_event_counter_0_0;       //0x0038

   //Remainder CRAM register space
   U32 reserved_003C_0200[113];
};

/**
 * @brief FBRAM registers. MMR base address for FBRAM is
 *        0x6600 relative to SCUBAR.
 */
struct SCU_FRAME_BUFFER_RAM
{
   U32 sram_base_address_1;         //0x0000
   U32 sram_upper_base_address_1;   //0x0004
   U32 sram_ecc_control_1;          //0x0008
   U32 sram_ecc_log_1;              //0x000c
   U32 sram_ecc_addrress_1;         //0x0010
   U32 sram_ecc_context_address_1;  //0x0014
   U32 sram_ecc_test_1;             //0x0018
   U32 sram_parity_control_and_status_1;   //0x001C
   U32 sram_parity_address_1;              //0x0020
   U32 sram_parity_upper_address_1;        //0x0024
   U32 sram_parity_context_1;              //0x0028
   U32 sram_memory_controller_interrupt_status_1;   //0x002C
   U32 sram_mcu_read_arbiter_control_1;    //0x0030
   U32 sram_mcu_write_arbiter_control_1;   //0x0034
   U32 smcu_error_event_counter_0_1;       //0x0038

   //Remainder of FBRAM register space
   U32 reserved_003C_0200[113];
};

#define SCU_SCRATCH_RAM_SIZE_IN_DWORDS  256

/**
* @brief Placeholder for the scratch RAM registers.
*
*/
struct SCU_SCRATCH_RAM
{
    U32 ram[SCU_SCRATCH_RAM_SIZE_IN_DWORDS];
};

/**
 * @brief Placeholder since I am not yet sure what these registers are here
 *        for.
 *
 */
struct NOA_PROTOCOL_ENGINE_PARTITION
{
   U32 reserved[64];
};

/**
 * @brief Placeholder since I am not yet sure what these registers are here
 *        for.
 *
 */
struct NOA_HUB_PARTITION
{
   U32 reserved[64];
};

/**
 * @brief Placeholder since I am not yet sure what these registers are here
 *        for.
 *
 */
struct NOA_HOST_INTERFACE_PARTITION
{
   U32 reserved[64];
};

/**
 * @struct TRANSPORT_LINK_LAYER_PAIR
 *
 * @brief The SCU Hardware pairs up the TL registers with the LL registers
 *        so we must place them adjcent to make the array of registers in
 *        the PEG.
 *
 */
struct TRANSPORT_LINK_LAYER_PAIR
{
   struct SCU_TRANSPORT_LAYER_REGISTERS tl;
   struct SCU_LINK_LAYER_REGISTERS      ll;
};

/**
 * @struct SCU_PEG_REGISTERS
 *
 * @brief SCU Protocol Engine Memory mapped register space.  These
 *        registers are unique to each protocol engine group.  There can be
 *        at most two PEG for a single SCU part.
 *
 */
struct SCU_PEG_REGISTERS
{
   struct TRANSPORT_LINK_LAYER_PAIR                pe[4];
   struct SCU_PORT_TASK_SCHEDULER_GROUP_REGISTERS  ptsg;
   struct SCU_PROTOCOL_ENGINE_GROUP_REGISTERS      peg;
   struct SCU_SGPIO_REGISTERS                      sgpio;
   U32    reserved_01500_1BFF[0x1C0];
   struct SCU_VIIT_ENTRY                           viit[64];
   struct SCU_ZONE_PARTITION_TABLE                 zpt0;
   struct SCU_ZONE_PARTITION_TABLE                 zpt1;
};

/**
 * @struct SCU_REGISTERS
 *
 * @brief SCU regsiters including both PEG registers if we turn on that
 *        compile option.
 *        All of these registers are in the memory mapped space returned
 *        from BAR1.
 *        See SCU SMU Specification for how these registers are mapped.
 *
 */
typedef struct SCU_REGISTERS
{
   // 0x0000 - PEG 0
   struct SCU_PEG_REGISTERS              peg0;

   // 0x6000 - SDMA and Miscellaneous
   struct SCU_SDMA_REGISTERS             sdma;
   struct SCU_COMPLETION_RAM             cram;
   struct SCU_FRAME_BUFFER_RAM           fbram;
   U32    reserved_6800_69FF[0x80];
   struct NOA_PROTOCOL_ENGINE_PARTITION  noa_pe;
   struct NOA_HUB_PARTITION              noa_hub;
   struct NOA_HOST_INTERFACE_PARTITION   noa_if;
   U32    reserved_6d00_7fff[0x4c0];

   // 0x8000 - PEG 1
   struct SCU_PEG_REGISTERS              peg1;

   // 0xE000 - AFE Registers
   struct SCU_AFE_REGISTERS              afe;

} SCU_REGISTERS_T;

#ifdef __cplusplus
}
#endif

#endif   // _SCU_REGISTERS_HEADER_
