;;; cal-xemacs.el --- calendar functions for menu bar and popup menu support
;;; Original file is cal-menu.el.

;; Copyright (C) 1994, 1995, 2001, 2002, 2003, 2004, 2005, 2006
;;   Free Software Foundation, Inc.

;; Author: Edward M. Reingold <reingold@cs.uiuc.edu>
;;	Lara Rios <lrios@coewl.cen.uiuc.edu>
;; Maintainer: Glenn Morris <rgm@gnu.org>
;; Keywords: calendar
;; Human-Keywords: calendar, popup menus, menu bar

;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
;; Boston, MA 02110-1301, USA.

;;; Synched up with: FSF Emacs 22.1 CVS 2006-09-15
;;; The Emacs counterpart is named cal-menu.el

;;; Commentary:

;; This collection of functions implements menu bar and popup menu support for
;; calendar.el.

;; Comments, corrections, and improvements should be sent to
;;  Edward M. Reingold               Department of Computer Science
;;  (217) 333-6733                   University of Illinois at Urbana-Champaign
;;  reingold@cs.uiuc.edu             1304 West Springfield Avenue
;;                                   Urbana, Illinois 61801

;;; Code:
(eval-when-compile
  (require 'cal-tex)
  (require 'cal-compat)
  (require 'holidays))

(defvar displayed-month)
(defvar displayed-year)

;; Don't require calendar because calendar requires us.
;; (eval-when-compile (require 'calendar))
(defvar calendar-mode-map)

(require 'easymenu)

;; XEmacs change
;; not available until 21.5
(unless (fboundp 'display-popup-menus-p)
  (defun display-popup-menus-p (&optional display)
    "Return non-nil if popup menus are supported on DISPLAY.
DISPLAY can be a frame, a device, a console, or nil (meaning the selected
frame).  Support for popup menus requires that the mouse be available."
    (and
     (memq (framep-on-display display) '(x ns gtk mswindows))
     (display-mouse-p display)))
  )


(defconst calendar-popup-menu-3
  '("Calendar"
    ["Scroll forward" scroll-calendar-left-three-months t]
    ["Scroll backward" scroll-calendar-right-three-months t]
    ["Mark diary entries" mark-diary-entries t]
    ["List holidays" list-calendar-holidays t]
    ["Mark holidays" mark-calendar-holidays t]
    ["Unmark" calendar-unmark t]
    ["Lunar phases" calendar-phases-of-moon t]
    ["Show diary" diary-show-all-entries t]
    ["Exit calendar" exit-calendar t]
    ))

(defun calendar-popup-menu-3 (e)
  (interactive "@e")
  (popup-menu calendar-popup-menu-3))
(define-key calendar-mode-map 'button3 'calendar-popup-menu-3)

(define-key calendar-mode-map 'button2 'calendar-mouse-2-date-menu)



(defconst calendar-scroll-menu
  '("Scroll"
    ["Forward 1 Month" scroll-calendar-left t]
    ["Forward 3 Months" scroll-calendar-left-three-months t]
    ["Forward 1 Year" (scroll-calendar-left-three-months 4) t]
    ["Backward 1 Month" scroll-calendar-right t]
    ["Backward 3 Months" scroll-calendar-right-three-months t]
    ["Backward 1 Year" (scroll-calendar-right-three-months 4) t]))

(defconst calendar-goto-menu
  '("Goto"
    ["Today" calendar-goto-today t]
    ["Beginning of week" calendar-beginning-of-week (calendar-cursor-to-date)]
    ["End of week" calendar-end-of-week (calendar-cursor-to-date)]
    ["Beginning of month" calendar-beginning-of-month (calendar-cursor-to-date)]
    ["End of month" calendar-end-of-month (calendar-cursor-to-date)]
    ["Beginning of year" calendar-beginning-of-year (calendar-cursor-to-date)]
    ["End of year" calendar-end-of-year (calendar-cursor-to-date)]
    ["Day of Year" calendar-goto-day-of-year]
    ["Other date" calendar-goto-date t]
    ["ISO date" calendar-goto-iso-date t]
    ["ISO week" calendar-goto-iso-week t]
    ["Astronomical date" calendar-goto-astro-day-number t]
    ["Hebrew date" calendar-goto-hebrew-date t]
    ["Persian date" calendar-goto-persian-date t]
    ["Baha'i date" calendar-goto-bahai-date t]
    ["Islamic date" calendar-goto-islamic-date t]
    ["Julian date" calendar-goto-julian-date t]
    ["Chinese date" calendar-goto-chinese-date t]
    ["Coptic date" calendar-goto-coptic-date t]
    ["Ethiopic date" calendar-goto-ethiopic-date t]
    ("Mayan date"
     ["Next Tzolkin" calendar-next-tzolkin-date t]
     ["Previous Tzolkin" calendar-previous-tzolkin-date t]
     ["Next Haab" calendar-next-haab-date t]
     ["Previous Haab" calendar-previous-haab-date t]
     ["Next Round" calendar-next-calendar-round-date t]
     ["Previous Round" calendar-previous-calendar-round-date t])
    ["French date" calendar-goto-french-date t]))

(defconst calendar-holidays-menu
  '("Holidays"
    ["One day" calendar-cursor-holidays (calendar-cursor-to-date)]
    ["3 months" list-calendar-holidays t]
    ["Year" cal-menu-list-holidays-year t]
    ["Following year"  cal-menu-list-holidays-following-year]
    ["Previous year"  cal-menu-list-holidays-previous-year]
    ["Mark" mark-calendar-holidays t]
    ["Unmark" calendar-unmark t]))

(defconst calendar-diary-menu
  '("Diary"
    ["Other File" view-other-diary-entries (calendar-cursor-to-date)]
    ["Cursor Date" diary-view-entries (calendar-cursor-to-date)]
    ["Mark All" mark-diary-entries t]
    ["Show All" diary-show-all-entries t]
    ["Insert Diary Entry"insert-diary-entry t]
    ["Insert Weekly" insert-weekly-diary-entry (calendar-cursor-to-date)]
    ["Insert Monthly" insert-monthly-diary-entry (calendar-cursor-to-date)]
    ["Insert Yearly" insert-yearly-diary-entry (calendar-cursor-to-date)]
    ["Insert Anniversary" insert-anniversary-diary-entry (calendar-cursor-to-date)]
    ["Insert Block" insert-block-diary-entry (calendar-cursor-to-date)]
    ["Insert Cyclic" insert-cyclic-diary-entry (calendar-cursor-to-date)]
    ["Insert Baha'i" calendar-mouse-insert-bahai-diary-entry (calendar-cursor-to-date)]
    ["Insert Islamic" calendar-mouse-insert-islamic-diary-entry (calendar-cursor-to-date)]
    ["Insert Hebrew" calendar-mouse-insert-hebrew-diary-entry (calendar-cursor-to-date)]))

(defun calendar-add-menus ()
  (set-buffer-menubar (copy-sequence current-menubar))
  (if (assoc "Calendar" current-menubar)
      nil
    (add-submenu nil '("Calendar"))
    (if (not (assoc "Scroll" current-menubar))
	(add-submenu '("Calendar") calendar-scroll-menu))
    (if (not (assoc "Goto" current-menubar))
	(add-submenu '("Calendar") calendar-goto-menu))
    (if (not (assoc "Holidays" current-menubar))
	(add-submenu '("Calendar") calendar-holidays-menu))
    (if (not (assoc "Diary" current-menubar))
	(add-submenu '("Calendar") calendar-diary-menu))
    (if (not (assoc "Moon" current-menubar))
	(add-menu-button '("Calendar") ["Moon" calendar-phases-of-moon t]))))

(defun calendar-flatten (list)
  "Flatten LIST eliminating sublists structure; result is a list of atoms.
This is the same as the preorder list of leaves in a rooted forest."
  (if (atom list)
      (list list)
    (if (cdr list)
        (append (calendar-flatten (car list)) (calendar-flatten (cdr list)))
      (calendar-flatten (car list)))))

(defun cal-menu-list-holidays-year ()
  "Display a list of the holidays of the selected date's year."
  (interactive)
  (let ((year (extract-calendar-year (calendar-cursor-to-date))))
    (list-holidays year year)))

(defun cal-menu-list-holidays-following-year ()
  "Display a list of the holidays of the following year."
  (interactive)
  (let ((year (1+ (extract-calendar-year (calendar-cursor-to-date)))))
    (list-holidays year year)))

(defun cal-menu-list-holidays-previous-year ()
  "Display a list of the holidays of the previous year."
  (interactive)
  (let ((year (1- (extract-calendar-year (calendar-cursor-to-date)))))
    (list-holidays year year)))

(defun cal-menu-update ()
  ;; Update the holiday part of calendar menu bar for the current display.
;;  (interactive)
  (condition-case nil
      (if (eq major-mode 'calendar-mode)
          (let ((l))
            ;; Show 11 years--5 before, 5 after year of middle month
            (dotimes (i 11)
              (let ((y (+ displayed-year -5 i)))
                (push (vector (format "For Year %s" y)
                              (list (list 'lambda 'nil '(interactive)
                                          (list 'list-holidays y y)))
                              t)
                      l)))
            (setq l (cons ["Mark Holidays" mark-calendar-holidays t]
                          (cons ["Unmark Calendar" calendar-unmark t]
                                (cons "--" l))))

            (easy-menu-change '("Calendar") "Holidays" (nreverse l))
            
            (add-menu-button '("Calendar" "Holidays") 
                             "---"
                             (format "For Year %s" (- displayed-year 5)))
	    (add-menu-button '("Calendar" "Holidays")
                             (vector
                              (format "For Today (%s)"
				   (calendar-date-string (calendar-current-date) t t))
			   'cal-menu-today-holidays t)  
                             "---")
            (let ((title
                   (let ((my1 (calendar-increment-month -1))
                         (my2 (calendar-increment-month 1)))
                     (if (= (cdr my1) (cdr my2))
                         (format "%s-%s, %d"
                                 (calendar-month-name (car my1) 'abbrev)
                                 (calendar-month-name (car my2) 'abbrev)
                                 (cdr my2))
                       (format "%s, %d-%s, %d"
                               (calendar-month-name (car my1) 'abbrev)
                               (cdr my1)
                               (calendar-month-name (car my2) 'abbrev)
                               (cdr my2))))))
              (add-menu-button '("Calendar" "Holidays")
                                (vector
                                 (format "For Window (%s)" title )
                                 'list-calendar-holidays t)
                                "---"))
            (let ((date (calendar-cursor-to-date)))
              (if date
		  (add-menu-button '("Calendar" "Holidays")
                                   (vector
                                    (format "For Cursor Date (%s)"
                                            (calendar-date-string date t t))
				 'calendar-cursor-holidays)
                                   "---")))
	    ))
    ;; Try to avoid entering infinite beep mode in case of errors.
    (error (ding))))

(defun calendar-event-to-date (&optional error)
  "Date of last event.
If event is not on a specific date, signals an error if optional parameter
ERROR is t, otherwise just returns nil."
  (save-excursion
    ;; we keep a copy of the last button press event.  This makes it easier
    ;; to mimic the Emacs version.  Otherwise, on the next call, we get
    ;; a misc-user-event when a menu item is selected.  Then we lose the
    ;; buffer and point information
    (if (button-event-p last-input-event)
	(progn
	  (setq last-calendar-button-event (allocate-event))
	  (copy-event last-input-event last-calendar-button-event))
      )
    ;;  Emacs has this.
    ;;(set-buffer (window-buffer (posn-window (event-start last-input-event))))
    ;;(goto-char (posn-point (event-start last-input-event)))
    ;; I think this does the same thing - jmiller
    (set-buffer (event-buffer last-calendar-button-event))
    (goto-char (event-point last-calendar-button-event))
    (calendar-cursor-to-date error)))

(defun calendar-mouse-insert-hebrew-diary-entry (event)
  "Pop up menu to insert a Hebrew-date diary entry."
  (interactive "e")
  (let ((menu (list (format "Hebrew insert menu - %s"
			    (calendar-hebrew-date-string
			     (calendar-cursor-to-date)))
		    "-----"
		    ["One time" insert-hebrew-diary-entry t]
		    ["Monthly" insert-monthly-hebrew-diary-entry t]
		    ["Yearly" insert-yearly-hebrew-diary-entry t])))
    (popup-menu menu)))

(defun calendar-mouse-insert-islamic-diary-entry (event)
  "Pop up menu to insert an Islamic-date diary entry."
  (interactive "e")
  (let ((menu (list (format "Islamic insert menu - %s"
			    (calendar-islamic-date-string
			     (calendar-cursor-to-date)))
		    "-----"
		    ["One time" insert-islamic-diary-entry t]
		    ["Monthly" insert-monthly-islamic-diary-entry t]
		    ["Yearly" insert-yearly-islamic-diary-entry t])))
    (popup-menu menu)))

(defun calendar-mouse-insert-bahai-diary-entry (event)
  "Pop up menu to insert an Baha'i-date diary entry."
  (interactive "e")
  (let ((menu (list (format "Baha'i insert menu - %s"
			    (calendar-bahai-date-string
			     (calendar-cursor-to-date)))
		    "-----"
		    ["One time" insert-bahai-diary-entry t]
		    ["Monthly" insert-monthly-bahai-diary-entry t]
		    ["Yearly" insert-yearly-bahai-diary-entry t])))
    (popup-menu menu)))

(defun calendar-mouse-sunrise/sunset ()
  "Show sunrise/sunset times for mouse-selected date."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (calendar-sunrise-sunset)))

(defun cal-menu-today-holidays ()
  "Show holidays for today's date."
  (interactive)
  (save-excursion
    (calendar-cursor-to-date (calendar-current-date))
    (calendar-cursor-holidays)))

(autoload 'check-calendar-holidays "holidays")
(autoload 'diary-list-entries "diary-lib")

(defun calendar-mouse-holidays (&optional event)
  "Pop up menu of holidays for mouse selected date."
  (interactive "e")
  (save-excursion
    (let* ((date (calendar-event-to-date))
	   (l (check-calendar-holidays date))
	   (menu
	    (cons
	     (format "Holidays for %s" (calendar-date-string date))
	     (if l l '("None")))))
      (popup-menu menu))))

(defun calendar-mouse-view-diary-entries (&optional date diary event)
  "Pop up menu of diary entries for mouse-selected date.
Use optional DATE and alternative file DIARY.

Any holidays are shown if `holidays-in-diary-buffer' is t."
  (interactive)
 ; (save-excursion
  (let* ((date (if date date (calendar-event-to-date)))
         (diary-file (if diary diary diary-file))
         (diary-list-include-blanks nil)
         (diary-display-hook 'ignore)
         (diary-entries
          (mapcar (lambda (x) (split-string (car (cdr x)) "\^M\\|\n"))
                  (diary-list-entries date 1 'list-only)))
         (holidays (if holidays-in-diary-buffer
                       (mapcar '(lambda (x) (list x))
                               (check-calendar-holidays date))))
         (title (concat "Diary entries "
                        (if diary (format "from %s " diary) "")
                        "for "
                        (calendar-date-string date)))
         (menu
          (append
           (list title)
           (if holidays
               (mapcar '(lambda (x) (concat "     " (car x)))
                       holidays))
           (if holidays
               (list  "--:shadowEtchedIn" "--:shadowEtchedIn"))
           
           (if diary-entries
               (mapcar 'concat (calendar-flatten diary-entries))
             '("None")))))
    (popup-menu menu)))

(defun calendar-mouse-view-other-diary-entries ()
  "Pop up menu of diary entries from alternative file on mouse-selected date."
  (interactive)
  (calendar-mouse-view-diary-entries
   (calendar-event-to-date)
   (read-file-name "Enter diary file name: " default-directory nil t)))

(defun calendar-mouse-insert-diary-entry ()
  "Insert diary entry for mouse-selected date."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (insert-diary-entry nil)))

(defun calendar-mouse-set-mark ()
  "Mark the date under the cursor."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (calendar-set-mark nil)))

(defun cal-tex-mouse-day ()
  "Make a buffer with LaTeX commands for the day mouse is on."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-day nil)))

(defun cal-tex-mouse-week ()
  "One page calendar for week indicated by cursor.
Holidays are included if `cal-tex-holidays' is t."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-week nil)))

(defun cal-tex-mouse-week2 ()
  "Make a buffer with LaTeX commands for the week cursor is on.
The printed output will be on two pages."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-week2 nil)))

(defun cal-tex-mouse-week-iso ()
  "One page calendar for week indicated by cursor.
Holidays are included if `cal-tex-holidays' is t."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-week-iso nil)))

(defun cal-tex-mouse-week-monday ()
  "One page calendar for week indicated by cursor."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-week-monday nil)))

(defun cal-tex-mouse-filofax-daily ()
  "Day-per-page Filofax calendar for week indicated by cursor."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-filofax-daily nil)))

(defun cal-tex-mouse-filofax-2week ()
  "One page Filofax calendar for week indicated by cursor."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-filofax-2week nil)))

(defun cal-tex-mouse-filofax-week ()
  "Two page Filofax calendar for week indicated by cursor."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-filofax-week nil)))

(defun cal-tex-mouse-month ()
  "Make a buffer with LaTeX commands for the month cursor is on.
Calendar is condensed onto one page."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-month nil)))

(defun cal-tex-mouse-month-landscape ()
  "Make a buffer with LaTeX commands for the month cursor is on.
The output is in landscape format, one month to a page."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-month-landscape nil)))

(defun cal-tex-mouse-year ()
  "Make a buffer with LaTeX commands for the year cursor is on."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-year nil)))

(defun cal-tex-mouse-filofax-year ()
  "Make a buffer with LaTeX commands for Filofax calendar of year cursor is on."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-filofax-year nil)))

(defun cal-tex-mouse-year-landscape ()
  "Make a buffer with LaTeX commands for the year cursor is on."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (cal-tex-cursor-year-landscape nil)))

(defun calendar-mouse-print-dates (&optional event)
  "Pop up menu of equivalent dates to mouse selected date."
  (interactive "e")
  (let* ((date (calendar-event-to-date))
	 (menu (list (format "Date Menu - %s (Gregorian)"
			     (calendar-date-string date))
		     "-----"
		     (calendar-day-of-year-string date)
		     (format "ISO date: %s" (calendar-iso-date-string date))
		     (format "Julian date: %s"
			     (calendar-julian-date-string date))
		     (format "Astronomical (Julian) date (before noon): %s"
			     (calendar-astro-date-string date))
		     (format "Fixed (RD) date: %s"
			     (calendar-absolute-from-gregorian date))
		     (format "Hebrew date (before sunset): %s"
			     (calendar-hebrew-date-string date))
		     (format "Persian date: %s"
			     (calendar-persian-date-string date))
		     (format "Baha'i date: %s"
			     (calendar-bahai-date-string date))
		     (let ((i (calendar-islamic-date-string date)))
		       (if (not (string-equal i ""))
			   (format "Islamic date (before sunset): %s" i)))
		     (format "Chinese date: %s"
			     (calendar-chinese-date-string date))
		     (let ((c (calendar-coptic-date-string date)))
		       (if (not (string-equal c ""))
			   (format "Coptic date: %s" c)))
		     (let ((e (calendar-ethiopic-date-string date)))
		       (if (not (string-equal e ""))
			   (format "Ethiopic date: %s" e)))
		     (let ((f (calendar-french-date-string date)))
		       (if (not (string-equal f ""))
			   (format "French Revolutionary date: %s" f)))
		     (format "Mayan date: %s"
			     (calendar-mayan-date-string date))
		     )))
    (popup-menu menu)
    ))



(defun calendar-mouse-chinese-date ()
  "Show Chinese equivalent for mouse-selected date."
  (interactive)
  (save-excursion
    (calendar-mouse-goto-date (calendar-event-to-date))
    (calendar-print-chinese-date)))

(defun calendar-mouse-goto-date (date)
  (set-buffer (event-buffer  last-calendar-button-event))
  (calendar-goto-date date))

(defun calendar-mouse-2-date-menu (event)
  (interactive "@e")
  (let* ((date (calendar-event-to-date))
	(menu (list (format "Menu - %s" (calendar-date-string date) t t)
	       "-----"
	       ["Holidays" calendar-mouse-holidays t]
	       ["Mark date" calendar-mouse-set-mark t]
	       ["Sunrise/sunset" calendar-mouse-sunrise/sunset t]
	       ["Other calendars" calendar-mouse-print-dates t]
	       ["Prepare LaTeX Buffer" calendar-mouse-cal-tex-menu t]
	       ["Diary entries" calendar-mouse-view-diary-entries t]
	       ["Insert diary entry" calendar-mouse-insert-diary-entry t]
	       ["Other Diary file entries"
		calendar-mouse-view-other-diary-entries
		(calendar-cursor-to-date)]
	       )))
    (popup-menu menu)))

(defun calendar-mouse-cal-tex-menu (event)
  "Pop up submenu for Mouse-2 for cal-tex commands for selected date in the calendar window."
  (interactive "e")
  (let* ((date (calendar-event-to-date t))
	 (menu (list (calendar-date-string date t nil)
		     "-----"
		     ["Daily (1 page)"    cal-tex-mouse-day  ]
		     ["Weekly (1 page)"   cal-tex-mouse-week ]
		     ["Weekly (2 pages)"  cal-tex-mouse-week2]
		     ["Weekly (other style; 1 page)" cal-tex-mouse-week-iso]
		     ["Weekly (yet another style; 1 page)"
		      cal-tex-mouse-week-monday]
		     ["Monthly" cal-tex-mouse-month]
		     ["Monthly (landscape)"  cal-tex-mouse-month-landscape]
		     ["Yearly"  cal-tex-mouse-year]
		     ["Yearly (landscape)"  cal-tex-mouse-year-landscape]
		     ["Filofax styles" cal-tex-mouse-filofax]
		     )))
    (popup-menu menu)
    ))

(defun cal-tex-mouse-filofax (event)
  "Pop up sub-submenu for Mouse-2 for Filofax cal-tex commands for selected date."
  (interactive "e")
  (let* ((date (calendar-event-to-date t))
	 (menu (list (calendar-date-string date t nil)
		     "-----"
		     ["Filofax Daily (One-day-per-page)"
		      cal-tex-mouse-filofax-daily]
		     ["Filofax Weekly (2-weeks-at-a-glance)"
		      cal-tex-mouse-filofax-2week]
		     ["Filofax Weekly (week-at-a-glance)"
		      cal-tex-mouse-filofax-week]
		     ["Filofax Yearly"  cal-tex-mouse-filofax-year]
		     )))
    (popup-menu menu)
    ))

(run-hooks 'cal-xemacs-load-hook)

(provide 'cal-xemacs)

;; arch-tag: aa81cf73-ce89-48a4-97ec-9ef861e87fe9
;;; cal-menu.el ends here
