;;; x-menubar.el --- Menubar and popup-menu support for X.

;; Copyright (C) 1991, 1992, 1993, 1994, 1995 Free Software Foundation, Inc.
;; Copyright (C) 1995 Tinker Systems and INS Engineering Corp.

;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with Xmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(defconst default-menubar
  (purecopy-menubar
   '(
     ("File"
      :filter file-menu-filter
      ["Open..."		find-file		t]
      ["Open in New Frame..."	find-file-other-frame	t]
      ["Insert File..." 	insert-file		t]
      ["View File..."		view-file		t]
      "------"
      ["Save"			save-buffer		t  nil]
      ["Save As..."		write-file		t]
      ["Save Some Buffers"	save-some-buffers	t]
      "-----"
      ["Print Buffer"		lpr-buffer		t  nil]
      ["Pretty-Print Buffer"	ps-print-buffer-with-faces t  nil]
      "-----"
      ["New Frame"		make-frame		t]
      ["Delete Frame"		delete-frame		t]
      "-----"
      ["Split Window"		split-window-vertically t]
      ["Un-Split (Keep This)"	delete-other-windows	(not (one-window-p t))]
      ["Un-Split (Keep Others)"	delete-window		(not (one-window-p t))]
      "-----"
      ["Revert Buffer"		revert-buffer		 t  nil]
      ["Delete Buffer"		kill-this-buffer	 t  nil]
      "-----"
      ["Exit XEmacs"		save-buffers-kill-emacs	t]
      )

     ("Edit"
      :filter edit-menu-filter
      ["Undo"			advertised-undo		   t]
      ["Cut"			x-kill-primary-selection   t]
      ["Copy"			x-copy-primary-selection   t]
      ["Paste"			x-yank-clipboard-selection t]
      ["Clear"			x-delete-primary-selection t]
      "----"
      ["Search..."		isearch-forward		t]
      ["Search Backward..."	isearch-backward	t]
      ["Replace..."		query-replace		t]
      "----"
      ["Search (Regexp)..."	isearch-forward-regexp	t]
      ["Search Backward (Regexp)..." isearch-backward-regexp t]
      ["Replace (Regexp)..."	query-replace-regexp	t]
      "----"
      ["Goto Line..."		goto-line		t]
      ["What Line"		what-line		t]
      "----"
      ["Start Macro Recording"	start-kbd-macro	      (not defining-kbd-macro)]
      ["End Macro Recording"	end-kbd-macro		defining-kbd-macro]
      ["Execute Last Macro"	call-last-kbd-macro	last-kbd-macro]
      )

     ("Apps"
      ["Read Mail (VM)..."	vm			t]
      ["Read Mail (MH)..."	(mh-rmail t)		t]
      ["Send mail..."		mail			t]
      ["Usenet News"		gnus			t]
      ["Browse the Web"		w3			t]
      ["Gopher"			gopher			t]
      ["Hyperbole..."		hyperbole		t]
      "----"
      ["Spell-Check Buffer"	ispell-buffer		t]
      ["Emulate VI"		viper-mode		t]
      "----"
      ("Calendar"
       ["3-Month Calendar"	calendar		t]
       ["Diary"			diary			t]
       ["Holidays"		holidays		t]
       ;; we're all pagans at heart ...
       ["Phases of the Moon"	phases-of-moon		t]
       ["Sunrise/Sunset"	sunrise-sunset		t]
       )
      ("Games"
       ["Quote from Zippy"	yow			t]
       ["Psychoanalyst"		doctor			t]
       ["Psychoanalyze Zippy!"	psychoanalyze-pinhead	t]
       ["Random Flames"		flame			t]
       ["Dunnet (Adventure)"	dunnet			t]
       ["Towers of Hanoi"	hanoi			t]
       ["Game of Life"		life			t]
       ["Multiplication Puzzle"	mpuz			t]
       )
      )

     ("Options"
      ["Read Only" (toggle-read-only)
       :style toggle :selected buffer-read-only]
      ("Editing Options"
       ["Overstrike" (progn
		       (overwrite-mode current-prefix-arg)
		       (setq-default overwrite-mode overwrite-mode))
	:style toggle :selected overwrite-mode]
       ["Case Sensitive Search" (progn
				  (setq case-fold-search (not case-fold-search))
				  (setq-default case-fold-search
						case-fold-search))
	:style toggle :selected (not case-fold-search)]
       ["Case Sensitive Replace" (setq case-replace (not case-replace))
	:style toggle :selected (not case-replace)]
       ["Auto Delete Selection" (if (memq 'pending-delete-pre-hook
					  pre-command-hook)
				    (pending-delete-off nil)
				  (pending-delete-on nil))
	:style toggle
	:selected (memq 'pending-delete-pre-hook pre-command-hook)]
       ["Active Regions" (setq zmacs-regions (not zmacs-regions))
	:style toggle :selected zmacs-regions]
       ["Mouse Paste At Text Cursor" (setq mouse-yank-at-point
					   (not mouse-yank-at-point))
	:style toggle :selected mouse-yank-at-point]
       )
      ("General Options"
       ["Teach Extended Commands" (setq teach-extended-commands-p
					(not teach-extended-commands-p))
	:style toggle :selected teach-extended-commands-p]
       ["Debug On Error" (setq debug-on-error (not debug-on-error))
	:style toggle :selected debug-on-error]
       ["Debug On Quit" (setq debug-on-quit (not debug-on-quit))
	:style toggle :selected debug-on-quit]
       )
;     ["Line Numbers" (line-number-mode nil)
;      :style toggle :selected line-number-mode]
      "-----"
      ("Syntax Highlighting" 
       ["None" (progn
		 (font-lock-mode 0)
		 (fast-lock-mode 0))
	:style radio :selected (null font-lock-mode)]
       ["Fonts" (progn (require 'font-lock)
		       (font-lock-use-default-fonts)
		       (setq font-lock-use-fonts t
			     font-lock-use-colors nil)
		       (font-lock-mode 1))
	:style radio
	:selected (and font-lock-mode
		       font-lock-use-fonts)]
       ["Colors" (progn (require 'font-lock)
			(font-lock-use-default-colors)
			(setq font-lock-use-colors t 
			      font-lock-use-fonts nil)
			(font-lock-mode 1))
	:style radio
	:selected (and font-lock-mode
		       font-lock-use-colors)]
       "-----"
       ["Less" (progn (require 'font-lock)
		      (font-lock-use-default-minimal-decoration)
		      (setq font-lock-use-maximal-decoration nil)
		      (font-lock-mode 0)
		      (font-lock-mode 1))
	:style radio
	:selected (and font-lock-mode
		       (not font-lock-use-maximal-decoration))]
       ["More" (progn (require 'font-lock)
		      (font-lock-use-default-maximal-decoration)
		      (setq font-lock-use-maximal-decoration t)
		      (font-lock-mode 0)
		      (font-lock-mode 1))
	:style radio
	:selected (and font-lock-mode
		       font-lock-use-maximal-decoration)]
       "-----"
       ["Fast" (progn (require 'fast-lock)
		      (if fast-lock-mode
			  (progn
			    (fast-lock-mode 0)
			    ;; this shouldn't be necessary so there has to
			    ;; be a redisplay bug lurking somewhere (or
			    ;; possibly another event handler bug)
			    (redraw-modeline))
			(if font-lock-mode
			    (progn
			      (fast-lock-mode 1)
			      (redraw-modeline)))))
	:active font-lock-mode
	:style toggle
	:selected fast-lock-mode]
       ["Auto-Fontify" (if (not (featurep 'font-lock))
			   (progn
			     (setq font-lock-auto-fontify t)
			     (require 'font-lock))
			 (setq font-lock-auto-fontify
			       (not font-lock-auto-fontify)))
	:style toggle
	:selected (and (featurep 'font-lock) font-lock-auto-fontify)]
       )
      ("Paren Highlighting"
       ["None" (paren-set-mode -1)
	:style radio :selected (not paren-mode)]
       ["Blinking Paren" (paren-set-mode 'blink-paren)
	:style radio :selected (eq paren-mode 'blink-paren)]
       ["Steady Paren" (paren-set-mode 'paren)
	:style radio :selected (eq paren-mode 'paren)]
       ["Expression" (paren-set-mode 'sexp)
	:style radio :selected (eq paren-mode 'sexp)]
;;;       ["Nested Shading" (paren-set-mode 'nested)
;;;        :style radio :selected (eq paren-mode 'nested)]
       )
      "-----"
      ("Frame Appearance"
       ["Toolbar" (if (featurep 'toolbar)
		      (if (specifier-instance default-toolbar)
			  (remove-specifier default-toolbar)
			(set-specifier default-toolbar initial-toolbar-spec)))
	:style toggle :selected
	(and (featurep 'toolbar) (specifier-instance default-toolbar))]
       ["Scrollbars" (if (= (specifier-instance scrollbar-width) 0)
			 (progn
			   (set-specifier scrollbar-width 15)
			   (set-specifier scrollbar-height 15))
		       (set-specifier scrollbar-width 0)
		       (set-specifier scrollbar-height 0))
	:style toggle :selected (> (specifier-instance scrollbar-width) 0)]
       ["3D Modeline"
	(progn
	  (if (zerop (specifier-instance modeline-shadow-thickness))
	      (set-specifier modeline-shadow-thickness 2)
	    (set-specifier modeline-shadow-thickness 0))
	  (redraw-modeline t))
	:style toggle :selected
	(let ((thickness
	       (specifier-instance modeline-shadow-thickness)))
	  (and (integerp thickness)
	       (> thickness 0)))]
       ["Truncate Lines" (progn
			   (setq truncate-lines (not truncate-lines))
			   (setq-default truncate-lines truncate-lines))
	:style toggle :selected truncate-lines]
       ["Bar Cursor" (progn
		       (setq bar-cursor (not bar-cursor))
		       (force-cursor-redisplay))
	:style toggle :selected bar-cursor]
       )
      ("Menubar Appearance"
       ["Buffers Menu Length..."
	(progn
	  (setq buffers-menu-max-size
		(read-number
		 "Enter number of buffers to display (or 0 for unlimited): "))
	  (if (eq buffers-menu-max-size 0) (setq buffers-menu-max-size nil)))
	t]
       ["Buffers Sub-Menus" (setq complex-buffers-menu-p
				  (not complex-buffers-menu-p))
	:style toggle :selected complex-buffers-menu-p]
       ["Frame-Local Font Menu" (setq font-menu-this-frame-only-p
				    (not font-menu-this-frame-only-p))
	:style toggle :selected font-menu-this-frame-only-p]
       ["Ignore Scaled Fonts" (setq font-menu-ignore-scaled-fonts
				    (not font-menu-ignore-scaled-fonts))
	:style toggle :selected font-menu-ignore-scaled-fonts]
       )
      "-----"
      ["Edit Faces..." edit-faces t]
      ("Font"   :filter font-menu-family-constructor)
      ("Size"	:filter font-menu-size-constructor)
      ("Weight"	:filter font-menu-weight-constructor)
      "-----"
      ["Save Options" save-options-menu-settings t]
      )
     
     ("Buffers"
      :filter buffers-menu-filter
      ["List All Buffers" list-buffers t]
      "--"
      )
     
     ("Tools"
      ["Grep..."		grep			t]
      ["Compile..."		compile			t]
      ["Shell Command..."	shell-command		t]
      ["Shell Command on Region..."
				shell-command-on-region (region-exists-p)]
      ["Debug (GDB)..."		gdb			t]
      ["Debug (DBX)..."		dbx			t]
      "-----"
      ("Compare"
       ["Two Files ..."  ediff-files t]
       ["Two Buffers ..." ediff-buffers t]
       ["Three Files ..."  ediff-files3 t]
       ["Three Buffers ..." ediff-buffers3 t]
       "---"
       ["Two Directories ..." ediff-directories t]
       ["Three Directories ..." ediff-directories3 t]
       "---"
       ["File with Revision ..."  ediff-revision t]
       "---"
       ["Windows Word-by-word ..." ediff-windows-wordwise t]
       ["Windows Line-by-line ..." ediff-windows-linewise t]
       "---"
       ["Regions Word-by-word ..." ediff-regions-wordwise t]
       ["Regions Line-by-line ..." ediff-regions-linewise t])
      ("Merge"
       ["Files ..."  ediff-merge-files t]
       ["Files with Ancestor ..." ediff-merge-files-with-ancestor t]
       ["Buffers ..."  ediff-merge-buffers t]
       ["Buffers with Ancestor ..."
	ediff-merge-buffers-with-ancestor t]
       "---"
       ["Directories ..."  ediff-merge-directories t]
       ["Directories with Ancestor ..."
	ediff-merge-directories-with-ancestor t]
       "---"
       ["Revisions ..."  ediff-merge-revisions t]
       ["Revisions with Ancestor ..."
	ediff-merge-revisions-with-ancestor t])
      ("Apply Patch"
       ["To a file..."	    	ediff-patch-file    	 t]
       ["To a buffer..."	ediff-patch-buffer	 t])
      ["OO-Browser..."		oobr			t]
      ("Tags"
       ["Find..."		find-tag		t]
       ["Find Other Window..."	find-tag-other-window	t]
       ["Tags Search..."	tags-search		t]
       ["Tags Replace..."	tags-query-replace	t]
       "-----"
       ["Continue"		tags-loop-continue	t]
       ["Pop stack"		pop-tag-mark		t]
       ["Apropos..."		tags-apropos		t]))

     nil		; the partition: menus after this are flushright

     ("Help"
      ["About XEmacs..."	about-xemacs		t]
      "-----"
      ["XEmacs WWW Page"	xemacs-www-page		t]
      ["XEmacs FAQ via WWW"	xemacs-www-faq		t]
      ["XEmacs Tutorial"	help-with-tutorial	t]
      ["XEmacs News"		view-emacs-news		t]
      "-----"
      ["Info (Detailed Docs)"	info			t]
      ["Package Browser"	finder-by-keyword	t]
      ["Describe Mode"		describe-mode		t]
      ["Hyper Apropos..."	hyper-apropos		t]
      ["Full Apropos..."	apropos			t]
      ["Super Apropos..."	super-apropos		t]
      "-----"
      ["List Keybindings"	describe-bindings	t]
      ["List Mouse Bindings"	describe-pointer	t]
      ["Describe Key..."	describe-key		t]
      ["Describe Function..."	describe-function	t]
      ["Describe Variable..."	describe-variable	t]
      "-----"
      ["Unix Manual..."		manual-entry		t]
      )
     )))


;;; Add Load Init button to menubar when starting up with -q
(defun maybe-add-init-button ()
  ;; by Stig@hackvan.com
  (if init-file-user
      nil
    (add-menu-button nil
		     ["Load .emacs"
		      (progn (delete-menu-item '("Load .emacs"))
			     (load-user-init-file (user-login-name)))
		      t]
		     "Help")
    ))

(add-hook 'before-init-hook 'maybe-add-init-button)


;;; The File and Edit menus

(defvar put-buffer-names-in-file-menu t)

;; The sensitivity part of this function could be done by just adding forms
;; to evaluate to the menu items themselves; that would be marginally less
;; efficient but not perceptibly so (I think).  But in order to change the
;; names of the Undo menu item and the various things on the File menu item,
;; we need to use a hook.

(defun file-menu-filter (menu-items)
  "Incrementally update the file menu.
This function changes the arguments and sensitivity of these File menu items:

  Delete Buffer  has the name of the current buffer appended to it.
  Print Buffer   has the name of the current buffer appended to it.
  Pretty-Print Buffer
		 has the name of the current buffer appended to it.
  Save           has the name of the current buffer appended to it, and is
                 sensitive only when the current buffer is modified.
  Revert Buffer  has the name of the current buffer appended to it, and is
                 sensitive only when the current buffer has a file.
  Delete Frame   sensitive only when there is more than one frame.

The name of the current buffer is only appended to the menu items if
`put-buffer-names-in-file-menu' is non-nil.  This behavior is the default."
  (let* ((bufname (buffer-name))
	 (result menu-items)		; save pointer to start of menu.
	 name
	 item)
    ;; the contents of the menu items in the file menu are destructively
    ;; modified so that there is as little consing as possible.  This is okay.
    ;; As soon as the result is returned, it is converted to widget_values
    ;; inside lwlib and the lisp menu-items can be safely modified again. 
    (while (setq item (pop menu-items))
      (if (vectorp item)
	  (progn
	    (setq name (aref item 0))
	    (and put-buffer-names-in-file-menu
		 (member name '("Save" "Revert Buffer" "Print Buffer"
				"Pretty-Print Buffer" "Delete Buffer"))
		 (>= 4 (length item))
		 (aset item 3 bufname))
	    (and (string= "Save" name)
		 (aset item 2 (buffer-modified-p)))
	    (and (string= "Revert Buffer" name)
		 (aset item 2 (not (not (or buffer-file-name
					    revert-buffer-function)))))
	    (and (string= "Delete Frame" name)
		 (aset item 2 (not (eq (next-frame) (selected-frame)))))
	    )))
    result))

(defun edit-menu-filter (menu-items)
  "For use as an incremental menu construction filter.
This function changes the sensitivity of these Edit menu items:

  Cut    sensitive only when emacs owns the primary X Selection.
  Copy   sensitive only when emacs owns the primary X Selection.
  Clear  sensitive only when emacs owns the primary X Selection.
  Paste  sensitive only when there is an owner for the X Clipboard Selection.
  Undo   sensitive only when there is undo information.
         While in the midst of an undo, this is changed to \"Undo More\"."
  (let* (item
	name
	(result menu-items)		; save pointer to head of list
	(x-dev (eq 'x (device-type (selected-device))))
	(emacs-owns-selection-p (and x-dev (x-selection-owner-p)))
	(clipboard-exists-p (and x-dev (x-selection-exists-p 'CLIPBOARD)))
;;;       undo-available undoing-more
;;;       (undo-info-available (not (null (and (not (eq t buffer-undo-list))
;;;                                 (if (eq last-command 'undo)
;;;                                     (setq undoing-more
;;;                                           (and (boundp 'pending-undo-list)
;;;                                          pending-undo-list)
;;;                                   buffer-undo-list))))))
	undo-name undo-state
	)
    ;; As with file-menu-filter, menu-items are destructively modified.
    ;; This is OK.
    (while (setq item (pop menu-items))
      (if (vectorp item)
	  (progn
	    (setq name (aref item 0))
	    (and (member name '("Cut" "Copy" "Clear"))
		 (aset item 2 emacs-owns-selection-p))
	    (and (string= name "Paste")
		 (aset item 2 clipboard-exists-p))
	    (and (member name '("Undo" "Undo More"))
		 (progn
		   ;; we could also do this with the third field of the item.
		   (if (eq last-command 'undo)
		       (setq undo-name "Undo More"
			     undo-state (not (null (and (boundp 'pending-undo-list)
							pending-undo-list))))
		     (setq undo-name "Undo"
			   undo-state (and (not (eq buffer-undo-list t))
					   (not (null
						 (or buffer-undo-list
						     (and (boundp 'pending-undo-list)
							  pending-undo-list)))))))
		   (if buffer-read-only (setq undo-state nil))
		   (aset item 0 undo-name)
		   (aset item 2 undo-state)
		   ))
      )))
    result))


;;; The Buffers menu

;; this version is too slow
(defun slow-format-buffers-menu-line (buffer)
  "Returns a string to represent the given buffer in the Buffer menu.
nil means the buffer shouldn't be listed.  You can redefine this."
  (if (string-match "\\` " (buffer-name buffer))
      nil
    (save-excursion
     (set-buffer buffer)
     (let ((size (buffer-size)))
       (format "%s%s %-19s %6s %-15s %s"
	       (if (buffer-modified-p) "*" " ")
	       (if buffer-read-only "%" " ")
	       (buffer-name)
	       size
	       mode-name
	       (or (buffer-file-name) ""))))))

(defun format-buffers-menu-line (buffer)
  "Returns a string to represent the given buffer in the Buffer menu.
nil means the buffer shouldn't be listed.  You can redefine this."
  (if (string-match "\\` " (setq buffer (buffer-name buffer)))
      nil
    buffer))

(defvar buffers-menu-max-size 20
  "*Maximum number of entries which may appear on the \"Buffers\" menu.
If this is 10, then only the ten most-recently-selected buffers will be
shown.  If this is nil, then all buffers will be shown.  Setting this to
a large number or nil will slow down menu responsiveness.")

(defvar complex-buffers-menu-p nil
  "*If true, the buffers menu will contain several commands, as submenus
of each buffer line.  If this is false, then there will be only one command:
select that buffer.")

(defvar buffers-menu-switch-to-buffer-function 'switch-to-buffer
  "*The function to call to select a buffer from the buffers menu.
`switch-to-buffer' is a good choice, as is `pop-to-buffer'.")


(defun buffer-menu-save-buffer (buffer)
  (save-excursion
    (set-buffer buffer)
    (save-buffer)))

(defun buffer-menu-write-file (buffer)
  (save-excursion
    (set-buffer buffer)
    (write-file (read-file-name
		 (format "Write %s to file: "
			 (buffer-name (current-buffer)))))))

(defsubst build-buffers-menu-internal (buffers)
  (let (name line)
    (mapcar
     (if complex-buffers-menu-p
	 #'(lambda (buffer)
	     (if (setq line (format-buffers-menu-line buffer))
		 (delq nil
		   (list line
		       (vector "Switch to Buffer"
			       (list buffers-menu-switch-to-buffer-function
				     (setq name (buffer-name buffer)))
			       t)
		       (if (eq buffers-menu-switch-to-buffer-function
			       'switch-to-buffer)
			   (vector "Switch to Buffer, Other Frame"
				   (list 'switch-to-buffer-other-frame
					 (setq name (buffer-name buffer)))
				   t)
			 nil)
		       (if (and (buffer-modified-p buffer)
				(buffer-file-name buffer))
			   (vector "Save Buffer"
				   (list 'buffer-menu-save-buffer name) t)
			     ["Save Buffer" nil nil]
			   )
		       (vector "Save As..."
			       (list 'buffer-menu-write-file name) t)
		       (vector "Delete Buffer" (list 'kill-buffer name) t)))))
       #'(lambda (buffer)
	   (if (setq line (format-buffers-menu-line buffer))
	       (vector line
		       (list buffers-menu-switch-to-buffer-function
			     (buffer-name buffer))
		       t))))
     buffers)))

(defun buffers-menu-filter (menu)
  "This is the menu filter for the top-level buffers \"Buffers\" menu.
It dynamically creates a list of buffers to use as the contents of the menu.
Only the most-recently-used few buffers will be listed on the menu, for
efficiency reasons.  You can control how many buffers will be shown by
setting `buffers-menu-max-size'.  You can control the text of the menu
items by redefining the function `format-buffers-menu-line'."
  (let ((buffers (buffer-list)))
    (and (integerp buffers-menu-max-size)
	 (> buffers-menu-max-size 1)
	 (> (length buffers) buffers-menu-max-size)
	 ;; shorten list of buffers
	 (setcdr (nthcdr buffers-menu-max-size buffers) nil))
      (setq buffers (delq nil (build-buffers-menu-internal buffers)))
      (append menu buffers)
      ))



;;; The Options menu

(defconst options-menu-saved-forms
  ;; This is really quite a kludge, but it gets the job done.
  (purecopy
   '(overwrite-mode			; #### - does this WORK???
     teach-extended-commands-p
     bar-cursor
     debug-on-error
     debug-on-quit
     complex-buffers-menu-p
     font-menu-ignore-scaled-fonts
     font-menu-this-frame-only-p
     buffers-menu-max-size
     case-fold-search
     case-replace
     zmacs-regions
     truncate-lines
     mouse-yank-at-point
     ;; We only save global settings since the others will belong to
     ;; objects which only exist during this session.
     (if (featurep 'toolbar)
	 `(if (and (featurep 'toolbar) (eq (device-type (selected-device)) 'x))
	      (progn
		(remove-specifier default-toolbar 'global)
		(add-spec-list-to-specifier
		 default-toolbar
		 ',(specifier-spec-list default-toolbar 'global)))))
     ;; #### - scrollbars need to modify frame properties... I'll fix it later.
     `(add-spec-list-to-specifier
       modeline-shadow-thickness
       ',(specifier-spec-list modeline-shadow-thickness 'global))
     (if paren-mode
 	 `(progn (require 'paren) (paren-set-mode ',paren-mode)))
     (if (memq 'pending-delete-pre-hook pre-command-hook)
	 '(require 'pending-del))
     ;; Setting this in lisp conflicts with X resources.  Bad move.  --Stig 
     ;; (list 'set-face-font ''default (face-font-name 'default))
     ;; (list 'set-face-font ''modeline (face-font-name 'modeline))
     font-lock-auto-fontify
     font-lock-use-fonts
     font-lock-use-colors
     font-lock-use-maximal-decoration
     font-lock-mode-enable-list
     font-lock-mode-disable-list
     ;; #### - these structures are clearly broken.  There's no way to ever
     ;; un-require font-lock or fast-lock via the menus.  --Stig
     (if (featurep 'font-lock)
	 '(require 'font-lock))
     (if (featurep 'fast-lock)
	 '(require 'fast-lock))
     (if (and (boundp 'font-lock-mode-hook)
	      (memq 'turn-on-fast-lock font-lock-mode-hook))
	 '(add-hook 'font-lock-mode-hook 'turn-on-fast-lock)
       '(remove-hook 'font-lock-mode-hook 'turn-on-fast-lock))
     (if (featurep 'font-lock)
	 (cons 'progn
	       (apply 'nconc
		      (mapcar
		       #'(lambda (face)
			   (delq nil
				 (list
				  (if (face-foreground-instance face)
				      (list 'set-face-foreground
					    (list 'quote face)
					    (color-instance-name
					     (face-foreground-instance face))))
				  (if (face-background-instance face)
				      (list 'set-face-background
					    (list 'quote face)
					    (color-instance-name
					     (face-background-instance face))))
				  (if (face-font-instance face)
				      (list 'set-face-font (list 'quote face)
					    (face-font-name face)))
				  )))
		       '(font-lock-comment-face font-lock-string-face
						font-lock-doc-string-face
						font-lock-function-name-face
						font-lock-keyword-face
						font-lock-type-face))))
       )))
  "The variables to save; or forms to evaluate to get forms to write out.")


(defun save-options-menu-settings ()
  "Saves the current settings of the `Options' menu to your `.emacs' file."
  (interactive)
  (let ((output-buffer (find-file-noselect
			(expand-file-name
			 (concat "~" init-file-user "/.emacs"))))
	output-marker)
    (save-excursion
      (set-buffer output-buffer)
      ;;
      ;; Find and delete the previously saved data, and position to write.
      ;;
      (goto-char (point-min))
      (if (re-search-forward "^;; Options Menu Settings *\n" nil 'move)
	  (let ((p (match-beginning 0)))
	    (goto-char p)
	    (or (re-search-forward
		 "^;; End of Options Menu Settings *\\(\n\\|\\'\\)"
		 nil t)
		(error "can't find END of saved state in .emacs"))
	    (delete-region p (match-end 0)))
	(goto-char (point-max))
	(insert "\n"))
      (setq output-marker (point-marker))

      ;; run with current-buffer unchanged so that variables are evaluated in
      ;; the current context, instead of in the context of the ".emacs" buffer.
      (let ((print-readably t)
	    (print-escape-newlines t)
	    (standard-output output-marker))
	(princ ";; Options Menu Settings\n")
	(princ ";; =====================\n")
	(princ "(cond\n")
	(princ " ((and (string-match \"XEmacs\" emacs-version)\n")
	(princ "       (boundp 'emacs-major-version)\n")
	(princ "       (= emacs-major-version 19)\n")
	(princ "       (>= emacs-minor-version 12))\n")
	(mapcar #'(lambda (var)
		    (princ "  ")
		    (if (symbolp var)
			(prin1 (list 'setq-default var
				     (let ((val (symbol-value var)))
				       (if (or (memq val '(t nil))
					       (not (symbolp val)))
					   val
					 (list 'quote val)))))
		      (setq var (eval var))
		      (cond ((eq (car-safe var) 'progn)
			     (while (setq var (cdr var))
			       (prin1 (car var))
			       (princ "\n")
			       (if (cdr var) (princ "  "))
			       ))
			    (var
			     (prin1 var))))
		    (if var (princ "\n")))
		options-menu-saved-forms)
	(princ "  ))\n")
	(princ ";; ============================\n")
	(princ ";; End of Options Menu Settings\n")
	))
    (set-marker output-marker nil)
    (save-excursion
      (set-buffer output-buffer)
      (save-buffer))
    ))


(set-menubar default-menubar)


;;; Popup menus.

(defconst default-popup-menu
  '("XEmacs Commands"
    ["Undo"		advertised-undo		t]
    ["Cut"		x-kill-primary-selection   t]
    ["Copy"		x-copy-primary-selection   t]
    ["Paste"		x-yank-clipboard-selection t]
    "-----"
    ["Select Block"	mark-paragraph 		t]
    ["Split Window"	(split-window)		t]
    ["Unsplit Window" 	delete-other-windows	t]
    ))

(defvar global-popup-menu nil
  "The global popup menu.  This is present in all modes.
See the function `popup-menu' for a description of menu syntax.")

(defvar mode-popup-menu nil
  "The mode-specific popup menu.  Automatically buffer local.
This is appended to the default items in `global-popup-menu'.
See the function `popup-menu' for a description of menu syntax.")
(make-variable-buffer-local 'mode-popup-menu)

;; In an effort to avoid massive menu clutter, this mostly worthless menu is
;; superceded by any local popup menu...
(setq-default mode-popup-menu default-popup-menu)

(defvar activate-popup-menu-hook nil
  "Function or functions run before a mode-specific popup menu is made visible.
These functions are called with no arguments, and should interrogate and
modify the value of `global-popup-menu' or `mode-popup-menu' as desired.
Note: this hook is only run if you use `popup-mode-menu' for activating the
global and mode-specific commands; if you have your own binding for button3,
this hook won't be run.")

(defun popup-mode-menu ()
  "Pop up a menu of global and mode-specific commands.
The menu is computed by combining `global-popup-menu' and `mode-popup-menu'."
  (interactive "@_")
  (run-hooks 'activate-popup-menu-hook)
  (popup-menu
   (cond ((and global-popup-menu mode-popup-menu)
	  (check-menu-syntax mode-popup-menu)
	  (let ((title (car mode-popup-menu))
		(items (cdr mode-popup-menu)))
	    (append global-popup-menu
		    '("---" "---")
		    (if popup-menu-titles (list title))
		    (if popup-menu-titles '("---" "---"))
		    items)))
	 (t
	  (or mode-popup-menu
	      global-popup-menu
	      (error "No menu here."))))))

(defun popup-buffer-menu (event) 
  "Pop up a copy of the Buffers menu (from the menubar) where the mouse is clicked."
  ;; by Stig@hackvan.com
  (interactive "e")
  (let ((window (and (event-over-text-area-p event) (event-window event)))
	bmenu)
    (or window
	(error "Pointer must be in a normal window"))
    (select-window window)
    (setq bmenu (assoc "Buffers" (or current-menubar default-menubar)))
    (popup-menu bmenu)))

(defun popup-menubar-menu (event) 
  "Pop up a copy of menu that also appears in the menubar"
  ;; by Stig@hackvan.com
  (interactive "e")
  (let ((window (and (event-over-text-area-p event) (event-window event)))
	popup-menubar)
    (or window
	(error "Pointer must be in a normal window"))
    (select-window window)
    (and current-menubar (run-hooks 'activate-menubar-hook))
    ;; ##### Instead of having to copy this just to safely get rid of
    ;; any nil what we should really do is fix up the internal menubar
    ;; code to just ignore nil if generating a popup menu
    (setq popup-menubar (delete nil (copy-sequence (or current-menubar
						       default-menubar))))
    (popup-menu (cons "Menubar Menu" popup-menubar))
    ))

(global-set-key 'button3 'popup-mode-menu)
;; shift button3 and shift button2 are reserved for Hyperbole
(global-set-key '(meta control button3) 'popup-buffer-menu)
(global-set-key '(meta shift button3) 'popup-menubar-menu)

;; Here's a test of the cool new menu features (from Stig).

;(setq mode-popup-menu
;      '("Test Popup Menu"
;        :filter cdr
;        ["this item won't appear because of the menu filter" ding t]
;        "--:singleLine"
;        "singleLine"
;        "--:doubleLine"
;        "doubleLine"
;        "--:singleDashedLine"
;        "singleDashedLine"
;        "--:doubleDashedLine"
;        "doubleDashedLine"
;        "--:noLine"
;        "noLine"
;        "--:shadowEtchedIn"
;        "shadowEtchedIn"
;        "--:shadowEtchedOut"
;        "shadowEtchedOut"
;        "--:shadowDoubleEtchedIn"
;        "shadowDoubleEtchedIn"
;        "--:shadowDoubleEtchedOut"
;        "shadowDoubleEtchedOut"
;        "--:shadowEtchedInDash"
;        "shadowEtchedInDash"
;        "--:shadowEtchedOutDash"
;        "shadowEtchedOutDash"
;        "--:shadowDoubleEtchedInDash"
;        "shadowDoubleEtchedInDash"
;        "--:shadowDoubleEtchedOutDash"
;        "shadowDoubleEtchedOutDash"
;        ))


(provide 'x-menubar)

;;; x-menubar.el ends here.
