/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bsevalueblock.h"
#include	"bsesample.h"
#include	"bsemem.h"
#include	<string.h>
#include	<time.h>
#include	<fcntl.h>
#include	<unistd.h>



/* --- signals --- */
enum
{
  SIGNAL_CHANGED,
  SIGNAL_LAST
};
typedef void (*SignalChanged)		(GtkObject	*object,
					 gulong		 m_time,
					 gpointer	 func_data);
static	void sample_handle_signal_changed(BseSample	*sample,
					  gulong	 m_time);
static	void marshal_signal_changed	(GtkObject	*object,
					 GtkSignalFunc	 func,
					 gpointer	 func_data,
					 GtkArg		*args)
{
  register SignalChanged sfunc = (SignalChanged) func;
  (* sfunc) (object, GTK_VALUE_ULONG (args[0]), func_data);
}

/* --- prototypes --- */
static void	sample_init			  (BseSample	    *sample);
static void	sample_class_init		  (BseSampleClass   *class);
static void	sample_finalize			  (GtkObject	    *object);


/* --- variables --- */
static GtkObjectClass   *parent_class = NULL;
static guint             sample_signals[SIGNAL_LAST] = { 0 };
static BseSampleClass	*bse_sample_class = NULL;
static GMemChunk	*sample_hentry_chunks = NULL;
static	GScannerConfig	sample_cache_scanner_config =
{
  (
   " \t\n"
   )			/* cset_skip_characters */,
  (
   G_CSET_a_2_z
   "_"
   G_CSET_A_2_Z
   )			/* cset_identifier_first */,
  (
   G_CSET_a_2_z
   "-+_0123456789"
   G_CSET_A_2_Z
   G_CSET_LATINS
   G_CSET_LATINC
   )			/* cset_identifier_nth */,
  ( ";\n" )		/* cpair_comment_single */,
  
  FALSE			/* case_sensitive */,
  
  TRUE			/* skip_comment_multi */,
  TRUE			/* skip_comment_single */,
  FALSE			/* scan_comment_multi */,
  TRUE			/* scan_identifier */,
  FALSE			/* scan_identifier_1char */,
  FALSE			/* scan_identifier_NULL */,
  TRUE			/* scan_symbols */,
  TRUE			/* scan_binary */,
  TRUE			/* scan_octal */,
  TRUE			/* scan_float */,
  TRUE			/* scan_hex */,
  FALSE			/* scan_hex_dollar */,
  TRUE			/* scan_string_sq */,
  TRUE			/* scan_string_dq */,
  TRUE			/* numbers_2_int */,
  FALSE			/* int_2_float */,
  FALSE			/* identifier_2_string */,
  TRUE			/* char_2_token */,
  FALSE			/* symbol_2_token */,
};


/* --- functions --- */
GtkType
bse_sample_get_type (void)
{
  static GtkType sample_type = 0;

  if (!sample_type)
    {
      GtkTypeInfo sample_info =
      {
	"BseSample",
	sizeof (BseSample),
	sizeof (BseSampleClass),
	(GtkClassInitFunc) sample_class_init,
	(GtkObjectInitFunc) sample_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      sample_type = gtk_type_unique (GTK_TYPE_OBJECT, &sample_info);
      gtk_type_set_chunk_alloc (sample_type, 16);
    }

  return sample_type;
}

static void
sample_init (BseSample *sample)
{
  register guint i;
  BseMunk init_munk = { BSE_NOTE_VOID, 0, 0, 0, NULL, NULL };
  
  /* initial refcount is 1
   */
  gtk_object_ref (GTK_OBJECT (sample));
  gtk_object_sink (GTK_OBJECT (sample));

  sample->type = 0;
  sample->name = NULL;
  sample->blurb = NULL;
  sample->file_name = NULL;
  sample->type_blurb = NULL;
  sample->creation_time = 0;
  sample->modification_time = 0;
  sample->last_modification_time = 0;
  sample->n_channels = 0;
  sample->recording_freq = 0;

  for (i = 0; i < BSE_MAX_SAMPLE_MUNKS; i++)
    sample->munk[i] = init_munk;
}

static void
sample_class_init (BseSampleClass   *class)
{
  GtkObjectClass *object_class;

  bse_sample_class = class;
  parent_class = gtk_type_class (GTK_TYPE_OBJECT);
  object_class = (GtkObjectClass*) class;

  class->sample_hash_table = g_hash_table_new (g_str_hash, g_str_equal);

  sample_signals[SIGNAL_CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (BseSampleClass, changed),
		    marshal_signal_changed,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_ULONG);
  gtk_object_class_add_signals (object_class, sample_signals, SIGNAL_LAST);

  object_class->finalize = sample_finalize;

  class->cpair_comment_single = g_strdup (";\n");

  class->changed = sample_handle_signal_changed;

  sample_hentry_chunks = g_mem_chunk_create (BseSampleHashEntry, 32, G_ALLOC_AND_FREE);
}

BseSample*
bse_sample_new (BseSampleType  type,
		const gchar    *sample_name,
		const gchar    *file_name)
{
  register BseSample *sample;
  register guint i;
  register guint init_note = 0;
  BseSampleHashEntry *hentry;
  
  g_return_val_if_fail (type < BSE_SAMPLE_LAST, NULL);
  g_return_val_if_fail (sample_name != NULL, NULL);

  if (!bse_sample_class)
    gtk_type_class (BSE_TYPE_SAMPLE);

  hentry = g_hash_table_lookup (bse_sample_class->sample_hash_table, (gpointer) sample_name);
  if (hentry && hentry->sample)
    {
      g_warning ("BseSample: sample \"%s\" already exists.", sample_name);
      return NULL;
    }
  
  sample = gtk_type_new (BSE_TYPE_SAMPLE);
  
  sample->type = type;
  sample->name = NULL;
  sample->blurb = g_strdup ("");
  sample->file_name = NULL;
  sample->type_blurb = g_strdup ("");
  sample->creation_time = time (NULL);
  sample->modification_time = sample->creation_time;
  sample->last_modification_time = sample->modification_time;
  sample->n_channels = 0;
  sample->recording_freq = 0;
  
  switch (sample->type)
    {
    case  BSE_SAMPLE_EFFECT_MUNKS:
      init_note = BSE_SAMPLE_DFL_BASE_NOTE;
      break;
    case  BSE_SAMPLE_NOTE_MUNKS:
      init_note = BSE_NOTE_VOID;
      break;
    default:
      g_assert_not_reached ();
      break;
    }
  
  for (i = 0; i < BSE_MAX_SAMPLE_MUNKS; i++)
    {
      sample->munk[i].recording_note = init_note;
      sample->munk[i].n_values = 0;
      sample->munk[i].loop_begin = 0;
      sample->munk[i].loop_end = 0;
      sample->munk[i].values = NULL;
      sample->munk[i].block = NULL;
    }

  /* this will invoke bse_sample_changed() for us
   */
  bse_sample_set_name (sample, sample_name);
  bse_sample_set_file_name (sample, file_name);
  
  return sample;
}

void
sample_finalize (GtkObject *object)
{
  BseSample *sample;
  register   guint i;
  register BseSampleHashEntry *hentry;
  
  sample = BSE_SAMPLE (object);

  for (i = 0; i < BSE_MAX_SAMPLE_MUNKS; i++)
    if (sample->munk[i].block)
      bse_value_block_unref (sample->munk[i].block);
  
  g_dataset_destroy (sample);

  hentry = g_hash_table_lookup (bse_sample_class->sample_hash_table, sample->name);
  g_assert (hentry != NULL && hentry->sample == sample); /* FIXME */
  hentry->sample = NULL;
  sample->name = NULL;
  sample->file_name = NULL;

  /* don't g_free (sample->name); */
  g_free (sample->blurb);
  /* don't g_free (sample->file_name); */
  g_free (sample->type_blurb);

  GTK_OBJECT_CLASS (parent_class)->finalize (object);
}

BseSampleHashEntry*
bse_sample_lookup_entry (const gchar	  *sample_name)
{
  g_return_val_if_fail (sample_name != NULL, NULL);

  if (!bse_sample_class)
    return NULL;

  return g_hash_table_lookup (bse_sample_class->sample_hash_table, (gpointer) sample_name);
}

BseSample*
bse_sample_lookup (const gchar	  *sample_name)
{
  BseSampleHashEntry *hentry;

  g_return_val_if_fail (sample_name != NULL, NULL);

  if (!bse_sample_class)
    return NULL;

  hentry = g_hash_table_lookup (bse_sample_class->sample_hash_table, (gpointer) sample_name);

  return hentry ? hentry->sample : NULL;
}

gchar*
bse_sample_lookup_path (const gchar	  *sample_name)
{
  BseSampleHashEntry *hentry;

  g_return_val_if_fail (sample_name != NULL, NULL);

  if (!bse_sample_class)
    return NULL;

  hentry = g_hash_table_lookup (bse_sample_class->sample_hash_table, (gpointer) sample_name);

  return hentry ? hentry->sample_path : NULL;
}

void
bse_sample_set_creation (BseSample *sample,
			 gulong     c_time)
{
  g_return_if_fail (sample != NULL);
  g_return_if_fail (c_time >= 631148400);

  sample->creation_time = c_time;

  bse_sample_set_modification (sample, c_time);
}

void
bse_sample_set_modification (BseSample *sample,
			     gulong     m_time)
{
  g_return_if_fail (sample != NULL);
  g_return_if_fail (m_time >= sample->creation_time);

  sample->modification_time = m_time;

  gtk_signal_emit (GTK_OBJECT (sample), sample_signals[SIGNAL_CHANGED], m_time);
}

void
bse_sample_set_params (BseSample      *sample,
		       guint	      n_channels,
		       float	      recording_freq)
{
  g_return_if_fail (sample != NULL);
  g_return_if_fail (n_channels > 0 && n_channels <= BSE_MAX_SAMPLE_CHANNELS);
  g_return_if_fail (recording_freq > 0 && recording_freq < BSE_MAX_MIX_FREQ);
  
  sample->n_channels = n_channels;
  sample->recording_freq = recording_freq;

  bse_sample_changed (sample);
}

static inline void
sample_set_munk_i (BseSample	   *sample,
		   guint	    munk_index,
		   guint	    recording_note,
		   BseValueBlock   *block,
		   guint	    block_offset,
		   guint	    n_values,
		   guint	    loop_begin,
		   guint	    loop_end)
{
  register BseMunk *munk;
  
  g_return_if_fail (sample != NULL);
  g_return_if_fail (munk_index < BSE_MAX_SAMPLE_MUNKS);
  if (sample->type == BSE_SAMPLE_EFFECT_MUNKS)
    g_return_if_fail (recording_note == BSE_NOTE_VOID);
  else if (sample->type == BSE_SAMPLE_NOTE_MUNKS)
    g_return_if_fail (recording_note >= BSE_MIN_NOTE && recording_note <= BSE_MAX_NOTE);
  else
    g_assert_not_reached ();
  g_return_if_fail (block != NULL);
  g_return_if_fail (block_offset * sample->n_channels <= block->n_values);
  g_return_if_fail (n_values != 0);
  g_return_if_fail (n_values * sample->n_channels <= block->n_values);
  g_return_if_fail (loop_begin <= loop_end);
  g_return_if_fail (loop_end < n_values - block_offset);
  if (n_values < block->n_values)
    {
      g_return_if_fail (loop_end == 0);
      g_return_if_fail (block->loop_end == 0);
    }
  if (block->identifier)
    {
      /* we got a non-zero block here
       */
      g_return_if_fail (loop_begin == block->loop_begin * sample->n_channels);
      g_return_if_fail (loop_end == block->loop_end * sample->n_channels);
    }
  
  munk = &sample->munk[munk_index];
  
  if (munk->block)
    bse_value_block_unref (munk->block);
  
  munk->recording_note = recording_note;
  munk->block = block;
  munk->n_values = n_values;
  munk->loop_begin = loop_begin;
  munk->loop_end = loop_end;
  munk->values = block->values + block_offset * sample->n_channels;
  bse_value_block_ref (munk->block);
}

void
bse_sample_set_munk (BseSample	    *sample,
		     guint	    munk_index,
		     guint	    recording_note,
		     BseValueBlock  *block,
		     guint	    block_offset,
		     guint	    n_values,
		     guint	    loop_begin,
		     guint	    loop_end)
{
  sample_set_munk_i (sample, munk_index, recording_note,
		     block, block_offset, n_values,
		     loop_begin, loop_end);

  bse_sample_changed (sample);
}

void
bse_sample_set_munks (BseSample	     *sample,
		      guint	     first_munk,
		      guint	     last_munk,
		      guint	     recording_note,
		      BseValueBlock  *block,
		      guint	     block_offset,
		      guint	     n_values,
		      guint	     loop_begin,
		      guint	     loop_end)
{
  register guint i;
  
  g_return_if_fail (first_munk <= last_munk && last_munk < BSE_MAX_SAMPLE_MUNKS);
  
  for (i = first_munk; i <= last_munk; i++)
    sample_set_munk_i (sample, i, recording_note, block, block_offset, n_values, loop_begin, loop_end);

  bse_sample_changed (sample);
}

void
bse_sample_fillup_munks (BseSample	*sample)
{
  register guint i;
  register BseMunk *munk;
  
  g_return_if_fail (sample != NULL);
  
  /* fillup empty munks
   */
  munk = NULL;
  for (i = 0; i < BSE_MAX_SAMPLE_MUNKS; i++)
    {
      if (sample->munk[i].block)
	{
	  munk = &sample->munk[i];
	  break;
	}
    }
  
  if (!munk)
    {
      g_warning ("BseSample: attempt to fillup \"%s\", with empty BseMunks",
		 sample->name);
      return;
    }
  
  for (i = 0; i < BSE_MAX_SAMPLE_MUNKS; i++)
    {
      if (!sample->munk[i].block)
	bse_sample_set_munk (sample, i,
			     munk->recording_note,
			     munk->block,
			     munk->values - munk->block->values,
			     munk->n_values,
			     munk->loop_begin,
			     munk->loop_end);
      else if (sample->munk[i].block != munk->block)
	munk = &sample->munk[i];
    }

  bse_sample_changed (sample);
}

void
bse_sample_changed (BseSample *sample)
{
  gulong t;

  g_return_if_fail (sample != NULL);

  t = time (NULL);

  gtk_signal_emit (GTK_OBJECT (sample), sample_signals[SIGNAL_CHANGED], t);
}

static void
sample_handle_signal_changed (BseSample *sample,
			      gulong     m_time)
{
  sample->last_modification_time = MAX (sample->modification_time, m_time);
}

void
bse_sample_set_name (BseSample	  *sample,
		     const gchar  *sample_name)
{
  BseSampleHashEntry *tmp_hentry, *hentry;

  g_return_if_fail (sample != NULL);
  g_return_if_fail (sample_name != NULL);

  hentry = g_hash_table_lookup (bse_sample_class->sample_hash_table, (gpointer) sample_name);
  if (hentry && hentry->sample)
    {
      g_warning ("BseSample: sample name \"%s\" already exists", sample_name);
      return;
    }
  if (sample->name)
    {
      tmp_hentry = g_hash_table_lookup (bse_sample_class->sample_hash_table, sample->name);
      g_hash_table_remove (bse_sample_class->sample_hash_table, tmp_hentry->sample_name);
    }
  else
    tmp_hentry = NULL;

  if (!hentry)
    {
      if (tmp_hentry)
	{
	  hentry = tmp_hentry;
	  g_free (hentry->sample_name);
	}
      else
	{
	  hentry = g_chunk_new (BseSampleHashEntry, sample_hentry_chunks);
	  hentry->sample = sample;
	  hentry->sample_path = sample->file_name;
	}
      hentry->sample_name = g_strdup (sample_name);
      g_hash_table_insert (bse_sample_class->sample_hash_table, hentry->sample_name, hentry);
    }
  else 
    {
      if (tmp_hentry)
	{
	  g_free (tmp_hentry->sample_name);
	  g_free (hentry->sample_path);
	  hentry->sample_path = tmp_hentry->sample_path;
	  g_chunk_free (tmp_hentry, sample_hentry_chunks);
	}
      else
	{
	  g_free (sample->file_name);
	  sample->file_name = hentry->sample_path;
	}
      hentry->sample = sample;
    }
  sample->name = hentry->sample_name;
  
  bse_sample_changed (sample);
}

void
bse_sample_set_file_name (BseSample	 *sample,
			  const gchar	 *file_name)
{
  BseSampleHashEntry *hentry;

  g_return_if_fail (sample != NULL);

  hentry = g_hash_table_lookup (bse_sample_class->sample_hash_table, sample->name);
  g_assert (hentry && hentry->sample == sample);

  g_free (hentry->sample_path);
  hentry->sample_path = g_strdup (file_name);
  sample->file_name = hentry->sample_path;
  
  bse_sample_changed (sample);
}

static void
bse_sample_ht_foreach (gpointer key,
		       gpointer value,
		       gpointer user_data)
{
  GList **list_p;
  BseSampleHashEntry *hentry;
  
  list_p = user_data;
  hentry = value;

  if (hentry->sample)
    *list_p = g_list_prepend (*list_p, hentry->sample);
}

GList*
bse_sample_list_all (void)
{
  GList *list;
  
  if (!bse_sample_class)
    return NULL;

  list = NULL;
  g_hash_table_foreach (bse_sample_class->sample_hash_table, bse_sample_ht_foreach, &list);
  
  return list;
}

void
bse_sample_set_blurb (BseSample	     *sample,
		      const gchar    *blurb)
{
  g_return_if_fail (sample != NULL);
  g_return_if_fail (blurb != NULL);
  
  if (!blurb)
    blurb = "";

  g_free (sample->blurb);
  sample->blurb = g_strdup (blurb);
  
  bse_sample_changed (sample);
}

gchar*
bse_sample_get_blurb (BseSample	     *sample)
{
  g_return_val_if_fail (sample != NULL, NULL);
  
  return sample->blurb;
}

gchar*
bse_sample_get_name (BseSample	     *sample)
{
  g_return_val_if_fail (sample != NULL, NULL);
  
  return sample->name;
}

gchar*
bse_sample_get_file_name (BseSample	 *sample)
{
  g_return_val_if_fail (sample != NULL, NULL);
  
  return sample->file_name;
}

void
bse_sample_set_type_blurb (BseSample	  *sample,
			   const gchar	  *type_blurb)
{
  g_return_if_fail (sample != NULL);
  g_return_if_fail (type_blurb != NULL);
  
  if (!type_blurb)
    type_blurb = "";

  g_free (sample->type_blurb);
  sample->type_blurb = g_strdup (type_blurb);
  
  bse_sample_changed (sample);
}

gchar*
bse_sample_get_type_blurb (BseSample	  *sample)
{
  g_return_val_if_fail (sample != NULL, NULL);
  
  return sample->type_blurb;
}

typedef struct
{
  BsePrintFunc         print_func;
  gpointer             func_data;
  guint                loaded_only : 1;
  guint                path_length;
  const gchar         *path;
} DumpLimiterData;

static void
bse_sample_cache_foreach (gpointer hash_key,
			  gpointer value,
			  gpointer user_data)
{
  BseSampleHashEntry *hentry;
  DumpLimiterData *data;
  gchar *string;
  gchar comment_prefix[2] = "\000\000";

  hentry = value;
  data = user_data;

  if ((data->path && strncmp (hentry->sample_path, data->path, data->path_length)) ||
      (data->loaded_only && !hentry->sample))
    return;

  comment_prefix[0] = bse_sample_class->cpair_comment_single[0];
  
  string = g_strconcat ("(sample-path \"",
			hash_key,
			"\" \"",
			hentry->sample_path ? hentry->sample_path : "",
			"\")",
			NULL);

  data->print_func (data->func_data, string);

  g_free (string);
}

void
bse_sample_cache_dump_rc (const gchar    *prefix_match,
			  gboolean        loaded_only,
			  BsePrintFunc    print_func,
			  gpointer        func_data)
{
  DumpLimiterData data;

  g_return_if_fail (print_func != NULL);

  if (!bse_sample_class)
    gtk_type_class (BSE_TYPE_SAMPLE);

  data.print_func = print_func;
  data.func_data = func_data;
  data.loaded_only = (loaded_only != FALSE);
  data.path_length = prefix_match ? strlen (prefix_match) : 0;
  data.path = prefix_match;

  g_hash_table_foreach (bse_sample_class->sample_hash_table, bse_sample_cache_foreach, &data);
}

static guint
bse_sample_cache_parse_path (GScanner            *scanner,
			     BseSampleClass      *class)
{
  BseSampleHashEntry *hentry;
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_STRING)
    return G_TOKEN_STRING;

  g_scanner_peek_next_token (scanner);
  if (scanner->next_token != G_TOKEN_STRING)
    {
      g_scanner_get_next_token (scanner);
      return G_TOKEN_STRING;
    }

  hentry = g_hash_table_lookup (class->sample_hash_table, scanner->value.v_string);
  if (!hentry)
    {
      hentry = g_chunk_new (BseSampleHashEntry, sample_hentry_chunks);
      hentry->sample_name = g_strdup (scanner->value.v_string);
      hentry->sample = NULL;
      hentry->sample_path = NULL;

      g_hash_table_insert (class->sample_hash_table, hentry->sample_name, hentry);
    }

  g_scanner_get_next_token (scanner);

  if (hentry->sample)
    bse_sample_set_file_name (hentry->sample, scanner->value.v_string);
  else
    {
      g_free (hentry->sample_path);
      hentry->sample_path = g_strdup (scanner->value.v_string);
    }

  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';
  else
    return G_TOKEN_NONE;
}

static void
bse_sample_cache_parse_statement (GScanner            *scanner,
				  BseSampleClass      *class)
{
  guint expected_token;
  
  g_scanner_get_next_token (scanner);
  
  if (scanner->token == G_TOKEN_SYMBOL)
    {
      guint (*parser_func) (GScanner*, BseSampleClass*);

      parser_func = scanner->value.v_symbol;

      /* check whether this is a BseSampleClass symbol...
       */
      if (parser_func == bse_sample_cache_parse_path)
	expected_token = parser_func (scanner, class);
      else
	expected_token = G_TOKEN_SYMBOL;
    }
  else
    expected_token = G_TOKEN_SYMBOL;

  /* skip rest of statement on errrors
   */
  if (expected_token != G_TOKEN_NONE)
    {
      register guint level;

      level = 1;
      if (scanner->token == ')')
	level--;
      if (scanner->token == '(')
	level++;
      
      while (!g_scanner_eof (scanner) && level > 0)
	{
	  g_scanner_get_next_token (scanner);
	  
	  if (scanner->token == '(')
	    level++;
	  else if (scanner->token == ')')
	    level--;
	}
    }
}

void
bse_sample_cache_parse_rc (const gchar    *file_name)
{
  gint fd;
  GScanner *scanner;

  g_return_if_fail (file_name != NULL);

  fd = open (file_name, O_RDONLY);
  if (fd < 0)
    return;

  scanner = g_scanner_new (&sample_cache_scanner_config);

  g_scanner_input_file (scanner, fd);

  bse_sample_cache_parse_rc_scanner (scanner);

  g_scanner_destroy (scanner);

  close (fd);
}

void
bse_sample_cache_parse_rc_string (const gchar    *rc_string)
{
  GScanner *scanner;

  g_return_if_fail (rc_string != NULL);

  scanner = g_scanner_new (&sample_cache_scanner_config);

  g_scanner_input_text (scanner, rc_string, strlen (rc_string));

  bse_sample_cache_parse_rc_scanner (scanner);

  g_scanner_destroy (scanner);
}

void
bse_sample_cache_parse_rc_scanner (GScanner      *scanner)
{
  gchar *orig_cpair_comment_single;

  g_return_if_fail (scanner != NULL);

  if (!bse_sample_class)
    gtk_type_class (BSE_TYPE_SAMPLE);

  g_scanner_add_symbol (scanner, "sample-path", bse_sample_cache_parse_path);
  orig_cpair_comment_single = scanner->config->cpair_comment_single;
  scanner->config->cpair_comment_single = bse_sample_class->cpair_comment_single;

  g_scanner_peek_next_token (scanner);

  while (scanner->next_token == '(')
    {
      g_scanner_get_next_token (scanner);

      bse_sample_cache_parse_statement (scanner, bse_sample_class);

      g_scanner_peek_next_token (scanner);
    }

  scanner->config->cpair_comment_single = orig_cpair_comment_single;
  g_scanner_remove_symbol (scanner, "sample-path");
}
