/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bseparser.h"
#include	"bseparserpriv.h"
#include	"bsevalueblock.h"
#include	"bsepattern.h"
#include	"bseinstrument.h"
#include	"bsesong.h"
#include	"bseprivate.h"
#include	<string.h>
#include	<unistd.h>
#include	<stdio.h>
#include	<fcntl.h>
#include	<time.h>


/* --- scanner configuration --- */
static	GScannerConfig	bse_scanner_config_template =
{
  (
   " \t\n"
   )			/* cset_skip_characters */,
  (
   G_CSET_a_2_z
   "_"
   G_CSET_A_2_Z
   )			/* cset_identifier_first */,
  (
   G_CSET_a_2_z
   "-+_0123456789"
   G_CSET_A_2_Z
   G_CSET_LATINS
   G_CSET_LATINC
   )			/* cset_identifier_nth */,
  ( ";\n" )		/* cpair_comment_single */,
  
  FALSE			/* case_sensitive */,
  
  TRUE			/* skip_comment_multi */,
  TRUE			/* skip_comment_single */,
  FALSE			/* scan_comment_multi */,
  TRUE			/* scan_identifier */,
  FALSE			/* scan_identifier_1char */,
  FALSE			/* scan_identifier_NULL */,
  TRUE			/* scan_symbols */,
  TRUE			/* scan_binary */,
  TRUE			/* scan_octal */,
  TRUE			/* scan_float */,
  TRUE			/* scan_hex */,
  FALSE			/* scan_hex_dollar */,
  TRUE			/* scan_string_sq */,
  TRUE			/* scan_string_dq */,
  TRUE			/* numbers_2_int */,
  FALSE			/* int_2_float */,
  FALSE			/* identifier_2_string */,
  TRUE			/* char_2_token */,
  FALSE			/* symbol_2_token */,
};


/* --- keys --- */
static const gchar	*key_munk_queue = "bse-munk-queue";
static guint		 KEYID_MUNK_QUEUE = 0;
static const gchar	*key_note_shift = "bse-note-shift";
static guint		 KEYID_NOTE_SHIFT = 0;


/* --- sample & song parser fucntions --- */
#include	"bseparsersample.c"
#include	"bseparsersong.c"


/* --- parser tables --- */
static	BseParserSymbol		root_symbols[] =
{
  /* root keywords
   */
  { "value-block",		parse_value_block,	JOB_ID_NONE },
  { "note-munks-sample",	parse_sample,		JOB_ID_NOTE_MUNKS },
  { "effect-munks-sample",	parse_sample,		JOB_ID_EFFECT_MUNKS },
  { "binary-appendix",		parse_binary_appendix,	JOB_ID_NONE },
  { "song",			parse_song,		JOB_ID_NONE },
};
static	BseParserEntry		symbol_table[] =
{
  /* table entries from bseparsersample.c
   */
  {
    root_symbols,
    sizeof (root_symbols) / sizeof (root_symbols[0])
  },
  {
    sample_binary_appendix_symbols,
    sizeof (sample_binary_appendix_symbols) / sizeof (sample_binary_appendix_symbols[0])
  },
  {
    sample_value_block_symbols,
    sizeof (sample_value_block_symbols) / sizeof (sample_value_block_symbols[0])
  },
  {
    sample_sample_symbols,
    sizeof (sample_sample_symbols) / sizeof (sample_sample_symbols[0])
  },
  {
    sample_munk_symbols,
    sizeof (sample_munk_symbols) / sizeof (sample_munk_symbols[0])
  },
  /* table entries from bseparsersong.c
   */
  {
    song_song_symbols,
    sizeof (song_song_symbols) / sizeof (song_song_symbols[0])
  },
  {
    song_instrument_symbols,
    sizeof (song_instrument_symbols) / sizeof (song_instrument_symbols[0])
  },
  {
    song_pattern_symbols,
    sizeof (song_pattern_symbols) / sizeof (song_pattern_symbols[0])
  },
};
static guint symbol_table_length = sizeof (symbol_table) / sizeof (symbol_table[0]);

/* --- functions --- */
/* identify a BSE data file
 * enforce a string of type
 * "(BSE-Data V1 " 10*<oct-value> ");" <anything> "\n"
 * then check the version numbering
 * magic entry:
 * #------------------------------------------------------------------------------
 * # BSE Data - Tim Janik <timj@gtk.org>
 * 0	   string	   (BSE-Data\040	BSE Library, musical data
 * >10	   string	   =V1\040		- version 1
 */
BseErrorType
bse_parser_identify_fd (gint		fd,
			BseIoDataFlags *flags_p)
{
  gchar chunk[64];
  register gchar *expect;
  register guint len;
  register guint i;
  guint flags;

  if (flags_p)
    *flags_p = BSE_IO_DATA_NONE;

  g_return_val_if_fail (fd >= 0, BSE_ERROR_INTERNAL);
  
  /* read "(BSE-Data V1 "
   */
  expect = "(BSE-Data V1 ";
  len = strlen (expect);
  chunk[len] = 0;
  if (read (fd, chunk, len) != len)
    return BSE_ERROR_FILE_TOO_SHORT;
  if (!g_str_equal (chunk, expect))
    return BSE_ERROR_HEADER_CORRUPT;

  /* read octal values
   */
  if (read (fd, chunk, 10) != 10)
    return BSE_ERROR_FILE_TOO_SHORT;
  chunk[10] = 0;

  /* check octal values
   */
  flags = 0;
  for (i = 0; i < 10; i++)
    {
      if (chunk[i] >= '0' || chunk[i] <= '7')
	flags |= (chunk[i] - '0') << (9 - i);
      else
	return BSE_ERROR_HEADER_CORRUPT;
    }
  
  /* read ");"
   */
  expect = ");";
  len = strlen (expect);
  chunk[len] = 0;
  if (read (fd, chunk, len) != len)
    return BSE_ERROR_FILE_TOO_SHORT;
  if (!g_str_equal (chunk, expect))
    return BSE_ERROR_HEADER_CORRUPT;
  
  /* skip to newline
   */
  do
    {
      if (read (fd, chunk, 1) != 1)
	return BSE_ERROR_FILE_TOO_SHORT;
    }
  while (chunk[0] != '\n');

  if (flags_p)
    *flags_p = flags;
  
  return BSE_ERROR_NONE;
}

GScanner*
bse_parser_setup (const gchar	*input_name,
		  gint		 in_fd,
		  guint		 line_offs,
		  guint		 position_offs,
		  gchar		*input_text,
		  guint		 input_text_len)
{
  GScanner *scanner = NULL;

  g_return_val_if_fail (input_name != NULL, NULL);
  g_return_val_if_fail (in_fd >= 0 || input_text_len >= 0, NULL);
  if (input_text_len)
    g_return_val_if_fail (input_text != NULL, NULL);
  
  if (!scanner)
    {
      BseParserData *pdata;
      guint i;
      
      scanner = g_scanner_new (&bse_scanner_config_template);

      pdata = g_new0 (BseParserData, 1);
      scanner->user_data = pdata;

      PDATA (scanner)->valid_song_names = NULL;
      PDATA (scanner)->valid_sample_names = NULL;
      PDATA (scanner)->pblock_list = NULL;
      PDATA (scanner)->binary_pblock_list = NULL;
      PDATA (scanner)->new_songs = NULL;
      PDATA (scanner)->new_samples = NULL;
      PDATA (scanner)->pattern_cntr_row = 0;
      PDATA (scanner)->pattern_cntr_channel = 0;
      PDATA (scanner)->in_appendix_definition = FALSE;
      PDATA (scanner)->parse_all_songs = TRUE;
      PDATA (scanner)->parse_all_samples = TRUE;
      PDATA (scanner)->skip_song_dups = FALSE;
      PDATA (scanner)->skip_sample_dups = FALSE;
      
      /* setup the symbol tables
       */
      for (i = 0; i < symbol_table_length; i++)
	{
	  register guint n;
	  
	  for (n = 0; n < symbol_table[i].n_symbols; n++)
	    {
	      register GList *list;
	      
	      symbol_table[i].symbols[n].group = i + 1;
	      list = g_scanner_lookup_symbol (scanner, symbol_table[i].symbols[n].name);
	      list = g_list_prepend (list, &symbol_table[i].symbols[n]);
	      g_scanner_add_symbol (scanner,
				    symbol_table[i].symbols[n].name,
				    list);
	    }
	}
      
      if (!KEYID_MUNK_QUEUE)
	{
	  KEYID_MUNK_QUEUE = g_dataset_force_id (key_munk_queue);
	  KEYID_NOTE_SHIFT = g_dataset_force_id (key_note_shift);
	}
    }
  
  /* setup input (file/memory)
   */
  if (input_text)
    g_scanner_input_text (scanner, input_text, input_text_len);
  else
    g_scanner_input_file (scanner, in_fd);
  scanner->line = line_offs;
  scanner->position = position_offs;
  scanner->input_name = input_name;
  scanner->max_parse_errors = 1;
  scanner->parse_errors = 0;

  return scanner;
}

BseErrorType
bse_parser_parse (GScanner	 *scanner,
		  BseIoDataFlags  read_mask,
		  GSList	 *sample_names,
		  GSList	 *song_names)
{
  g_return_val_if_fail (scanner != NULL, ~0);
  g_return_val_if_fail (PDATA (scanner)->valid_song_names == NULL, ~0);
  g_return_val_if_fail (PDATA (scanner)->valid_sample_names == NULL, ~0);
  g_return_val_if_fail (PDATA (scanner)->pblock_list == NULL, ~0);
  g_return_val_if_fail (PDATA (scanner)->binary_pblock_list == NULL, ~0);
  g_return_val_if_fail (PDATA (scanner)->new_songs == NULL, ~0);
  g_return_val_if_fail (PDATA (scanner)->new_samples == NULL, ~0);

  PDATA (scanner)->valid_song_names = song_names;
  PDATA (scanner)->valid_sample_names = sample_names;
  PDATA (scanner)->parse_all_songs = (read_mask & BSE_IO_DATA_SONGS) != 0;
  PDATA (scanner)->parse_all_samples = (read_mask & BSE_IO_DATA_SAMPLES) != 0;
  PDATA (scanner)->skip_song_dups = FALSE;
  PDATA (scanner)->skip_sample_dups = TRUE;
  
  while (!bse_parser_should_abort (scanner) &&
	 g_scanner_peek_next_token (scanner) != G_TOKEN_EOF)
    {
      g_scanner_get_next_token (scanner);
      
      if (scanner->token == '(')
	{
	  register guint last_expected_token;
	  
	  last_expected_token = parser_parse_symbol (scanner, root_symbols[0].group, NULL);
	  
	  if (last_expected_token != G_TOKEN_NONE &&
	      !bse_parser_should_abort (scanner))
	    {
	      bse_parser_unexp_token_skip (scanner, last_expected_token);
	      
	      if (scanner->token != ')')
		parser_skip_statement (scanner, scanner->token == '(' ? 2 : 1);
	    }
	}
      else
	{
	  if (scanner->token == G_TOKEN_SYMBOL)
	    {
	      register GList	       *symbol_list;
	      register BseParserSymbol *symbol;
	      
	      symbol_list = scanner->value.v_symbol;
	      symbol = (BseParserSymbol*) symbol_list->data;
	      
	      bse_parser_unexp_symbol (scanner, '(', symbol->name);
	    }
	  else if (!bse_parser_should_abort (scanner))
	    bse_parser_unexp_token (scanner, '(');
	}
    }

  PDATA (scanner)->valid_song_names = NULL;
  PDATA (scanner)->valid_sample_names = NULL;

  if (scanner->parse_errors)
    return BSE_ERROR_DATA_CORRUPT;
  else
    return BSE_ERROR_NONE;
}

BseErrorType
bse_parser_read_binary_appendices (GScanner *scanner)
{
  GSList *slist;
  BseErrorType load_error;

  g_return_val_if_fail (scanner != NULL, BSE_ERROR_INTERNAL);
  g_return_val_if_fail (scanner->input_fd >= 0, BSE_ERROR_INTERNAL);

  /* read binary appendices
   */
  load_error = BSE_ERROR_NONE;
  slist = PDATA (scanner)->binary_pblock_list;
  while (slist && !load_error)
    {
      register BseParserBlock *pblock;

      pblock = slist->data;
      slist = slist->next;

      switch (pblock->block_type)
	{
	case  BSE_BLOCK_BINARY_APPENDIX:
	  printf ("bsebswread.c: (%s): [%u] read %u appended bytes, %s endian\n",
		  scanner->input_name,
		  pblock->nth_appendix,
		  pblock->byte_size * pblock->n_values,
		  pblock->big_endian ? "big" : "little");
	  load_error = read_block (scanner->input_fd, pblock);
	  break;

	case  BSE_BLOCK_ZERO:
	  /* FIXME: fillin */
	  printf ("bseparser.c: creating block with %d zeros\n",
		  pblock->byte_size * pblock->n_values);
	  break;

	default:
	  g_assert_not_reached ();
	}
    }

  return load_error;
}

void
bse_parser_complete_samples (GScanner *scanner)
{
  GSList *slist;

  g_return_if_fail (scanner != NULL);

  /* apply sample munks and fillup the samples
   */
  for (slist = PDATA (scanner)->new_samples; slist; slist = slist->next)
    {
      register BseSample *sample;
      register BseParserMunk *pmunk;
      
      sample = slist->data;
      
      pmunk = g_dataset_id_get_data (sample, KEYID_MUNK_QUEUE);
      g_dataset_id_remove_data (sample, KEYID_MUNK_QUEUE);
      while (pmunk)
	{
	  BseParserMunk *next;
	  
	  bse_sample_set_munk (sample,
			       pmunk->note,
			       pmunk->pblock->recording_note,
			       pmunk->pblock->vblock,
			       pmunk->block_offset,
			       pmunk->n_values,
			       pmunk->loop_begin,
			       pmunk->loop_end);
	  next = pmunk->next;
	  g_free (pmunk);
	  pmunk = next;
	}
      
      bse_sample_fillup_munks (sample);
    }
}

void
bse_parser_complete_songs (GScanner *scanner)
{
  GSList *slist;

  g_return_if_fail (scanner != NULL);

  /* apply sample munks and fillup the samples
   */
  for (slist = PDATA (scanner)->new_songs; slist; slist = slist->next)
    {
      register BseSong *song;
      
      song = slist->data;
      
      bse_song_set_modification (song, song->modification_time);
    }
}

GSList*
bse_parser_get_song_list (GScanner	 *scanner)
{
  g_return_val_if_fail (scanner != NULL, NULL);

  return PDATA (scanner)->new_songs;
}

GSList*
bse_parser_get_sample_list (GScanner	   *scanner)
{
  g_return_val_if_fail (scanner != NULL, NULL);

  return PDATA (scanner)->new_samples;
}

void
bse_parser_shutdown (GScanner *scanner)
{
  GSList *slist;

  g_return_if_fail (scanner != NULL);

  /* free all parser blocks
   */
  g_slist_free (PDATA (scanner)->binary_pblock_list);
  PDATA (scanner)->binary_pblock_list = NULL;
  for (slist = PDATA (scanner)->pblock_list; slist; slist = slist->next)
    {
      register BseParserBlock *pblock;
      
      pblock = slist->data;

      if (pblock->vblock)
	bse_value_block_unref (pblock->vblock);
      g_free (pblock->identifier);
      g_free (pblock);
    }
  g_slist_free (PDATA (scanner)->pblock_list);
  PDATA (scanner)->pblock_list = NULL;

  /* free new songs
   */
  for (slist = PDATA (scanner)->new_songs; slist; slist = slist->next)
    {
      register BseSong *song;

      song = slist->data;

      bse_song_unref (song);
    }
  g_slist_free (PDATA (scanner)->new_songs);
  PDATA (scanner)->new_songs = NULL;

  /* free new samples
   */
  for (slist = PDATA (scanner)->new_samples; slist; slist = slist->next)
    {
      register BseSample *sample;
      register BseParserMunk *pmunk;
      
      sample = slist->data;

      /* free samples munk queue
       */
      pmunk = g_dataset_id_get_data (sample, KEYID_MUNK_QUEUE);
      g_dataset_id_remove_data (sample, KEYID_MUNK_QUEUE);
      while (pmunk)
	{
	  BseParserMunk *next;

	  next = pmunk->next;
	  g_free (pmunk);
	  pmunk = next;
	}

      bse_sample_unref (sample);
    }
  g_slist_free (PDATA (scanner)->new_samples);
  PDATA (scanner)->new_samples = NULL;

  /* free the scanner
   */
  g_free (scanner->user_data);
  scanner->user_data = NULL;
  g_scanner_destroy (scanner);
}

static gchar*
parser_check_song_name (GScanner *scanner,
			const gchar *template)
{
  GSList *slist;
  gchar *name;
  guint i;

  g_return_val_if_fail (scanner != NULL, NULL);
  g_return_val_if_fail (template != NULL, NULL);

  for (slist = PDATA (scanner)->valid_song_names; slist; slist = slist->next)
    {
      gchar *test_name;

      test_name = slist->data;
      if (g_str_equal ((gchar*) template, test_name))
	break;
    }
  if (!slist && !PDATA (scanner)->parse_all_songs)
    return NULL;

  if (!bse_song_lookup (template))
    return g_strdup (template);
  else if (PDATA (scanner)->skip_song_dups)
    return NULL;

  i = 1;
  name = NULL;
  do
    {
      gchar buffer[32];

      g_free (name);
      sprintf (buffer, "_%u", i++);
      name = g_strconcat (template, buffer, NULL);
      bse_song_name_make_valid (name);
    }
  while (bse_song_lookup (name));

  return name;
}

static gchar*
parser_check_sample_name (GScanner *scanner,
			  const gchar *template)
{
  GSList *slist;
  gchar *name;
  guint i;
  
  g_return_val_if_fail (scanner != NULL, NULL);
  g_return_val_if_fail (template != NULL, NULL);
  
  for (slist = PDATA (scanner)->valid_sample_names; slist; slist = slist->next)
    {
      gchar *test_name;

      test_name = slist->data;
      if (g_str_equal ((gchar*) template, test_name))
	break;
    }
  if (!slist && !PDATA (scanner)->parse_all_samples)
    return NULL;

  if (!bse_sample_lookup (template))
    return g_strdup (template);
  else if (PDATA (scanner)->skip_sample_dups)
    return NULL;

  i = 1;
  name = NULL;
  do
    {
      gchar buffer[32];

      g_free (name);
      sprintf (buffer, "_%u", i++);
      name = g_strconcat (template, buffer, NULL);
      bse_sample_name_make_valid (name);
    }
  while (bse_sample_lookup (name));

  return name;
}

static void
parser_skip_statement (GScanner	 *scanner,
		       guint	 starting_level)
{
  register guint level;
  
  level = starting_level;
  while (!bse_parser_should_abort (scanner) &&
	 level > 0)
    {
      g_scanner_get_next_token (scanner);
      
      if (scanner->token == G_TOKEN_EOF)
	bse_parser_unexp_token (scanner, ')');
      else if (scanner->token == '(')
	level++;
      else if (scanner->token == ')')
	level--;
    }
}

/* parse a symbol of a special group, if found
 * run it's associated parsing function
 */
static guint
parser_parse_symbol (GScanner	   *scanner,
		     guint	    symbol_group,
		     gpointer	    struct_data)
{
  register guint expected_token;
  
  expected_token = G_TOKEN_SYMBOL;
  g_scanner_get_next_token (scanner);
  
  if (scanner->token == expected_token)
    {
      register GList	       *symbol_list;
      register BseParserSymbol *symbol = NULL;
      
      symbol_list = scanner->value.v_symbol;
      while (symbol_list)
	{
	  symbol = (BseParserSymbol*) symbol_list->data;
	  
	  if (symbol->group == symbol_group)
	    break;
	  
	  symbol_list = symbol_list->next;
	}
      
      
      if (!symbol_list)
	{
	  bse_parser_unexp_symbol_skip (scanner, G_TOKEN_SYMBOL, symbol->name);
	  
	  parser_skip_statement (scanner, 1);
	  expected_token = G_TOKEN_NONE;
	}
      else
	{
	  if (symbol->func)
	    expected_token = (* symbol->func) (scanner, symbol, symbol->job_id, struct_data);
	  else
	    {
	      parser_skip_statement (scanner, 1);
	      expected_token = G_TOKEN_NONE;
	    }
	}
    }
  
  return expected_token;
}

/* read until next ')', additional '(' <symbol> constructs are accepted
 * and are dispatched via parser_parse_symbol ().
 */
static guint
parser_parse_rest (GScanner	       *scanner,
		   BseParserSymbol     *symbol_group_member,
		   gpointer	       struct_data)
{
  register guint expected_token;
  register guint symbol_group;

  symbol_group = symbol_group_member->group;
  
  expected_token = ')';
  while (!bse_parser_should_abort (scanner) &&
	 expected_token != G_TOKEN_NONE)
    {
      g_scanner_get_next_token (scanner);
      
      if (scanner->token == expected_token)
	expected_token = G_TOKEN_NONE;
      else if (symbol_group && scanner->token == '(')
	{
	  register guint last_expected_token;
	  
	  last_expected_token = parser_parse_symbol (scanner, symbol_group, struct_data);
	  
	  if (last_expected_token != G_TOKEN_NONE)
	    {
	      bse_parser_unexp_token (scanner, last_expected_token);
	      
	      if (scanner->token != ')')
		parser_skip_statement (scanner, scanner->token == '(' ? 2 : 1);
	    }
	}
      else
	{
	  bse_parser_unexp_token (scanner, expected_token);
	}
    }
  
  if (bse_parser_should_abort (scanner))
    return G_TOKEN_NONE;
  else
    return expected_token;
}

/* read a string by concatenating subsequent parsed strings.
 * the string is assigned to (Anything*) struct_data via the
 * job_id.
 */
static guint
parse_string (GScanner		*scanner,
	      BseParserSymbol	*symbol,
	      guint		job_id,
	      gpointer		struct_data)
{
  register gchar *description;
  register GList *strings, *list;
  register guint total_len;
  
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  
  switch (job_id)
    {
      register BseSample *sample;
      register BsePattern *pattern;
      register BseInstrument *instrument;
      register BseSong *song;
      
    case  JOB_ID_SAMPLE_BLURB:
      sample = struct_data;
      description = bse_sample_get_blurb (sample);
      break;
      
    case  JOB_ID_SAMPLE_TYPE_BLURB:
      sample = struct_data;
      description = bse_sample_get_type_blurb (sample);
      break;
      
    case  JOB_ID_PATTERN_BLURB:
      pattern = struct_data;
      description = bse_pattern_get_blurb (pattern);
      break;

    case  JOB_ID_PATTERN_NAME:
      pattern = struct_data;
      description = bse_pattern_get_name (pattern);
      break;
      
    case  JOB_ID_INSTRUMENT_BLURB:
      instrument = struct_data;
      description = bse_instrument_get_blurb (instrument);
      break;

    case  JOB_ID_INSTRUMENT_NAME:
      instrument = struct_data;
      description = bse_instrument_get_name (instrument);
      break;

    case  JOB_ID_INSTRUMENT_DSAMPLE_PATH:
      instrument = struct_data;
      description = instrument->deferred_sample_path;
      break;
      
    case  JOB_ID_SONG_BLURB:
      song = struct_data;
      description = bse_song_get_blurb (song);
      break;

    case  JOB_ID_SONG_AUTHOR:
      song = struct_data;
      description = bse_song_get_author (song);
      break;

    case  JOB_ID_SONG_COPYRIGHT:
      song = struct_data;
      description = bse_song_get_copyright (song);
      break;

    case  JOB_ID_SONG_BSE_VER:
      song = struct_data;
      description = NULL;
      break;
	
    default:
      g_assert_not_reached ();
      description = NULL;
      break;
    }
  
  if (description)
    {
      strings = g_list_append (NULL, g_strdup (description));
      total_len = strlen (description) + 1;
    }
  else
    {
      strings = NULL;
      total_len = 0;
    }
  
  g_scanner_get_next_token (scanner);
  while (scanner->token == G_TOKEN_STRING)
    {
      strings = g_list_append (strings, g_strdup (scanner->value.v_string));
      total_len += strlen (scanner->value.v_string) + 1;
      
      g_scanner_get_next_token (scanner);
    }
  
  if (total_len)
    {
      description = g_new (gchar, total_len);
      *description = 0;
    }
  else
    description = NULL;
  
  for (list = strings; list; list = list->next)
    {
      if (description)
	{
	  strcat (description, list->data);
	  if (list->next)
	    strcat (description, " ");
	}
      g_free (list->data);
    }
  g_list_free (strings);
  
  switch (job_id)
    {
      register BseSample *sample;
      register BsePattern *pattern;
      register BseInstrument *instrument;
      register BseSong *song;
      
    case	JOB_ID_SAMPLE_BLURB:
      sample = struct_data;
      bse_sample_set_blurb (sample, description);
      break;
      
    case	JOB_ID_SAMPLE_TYPE_BLURB:
      sample = struct_data;
      bse_sample_set_type_blurb (sample, description);
      break;
      
    case  JOB_ID_PATTERN_BLURB:
      pattern = struct_data;
      bse_pattern_set_blurb (pattern, description);
      break;

    case  JOB_ID_PATTERN_NAME:
      pattern = struct_data;
      bse_pattern_set_name (pattern, description);
      break;
      
    case  JOB_ID_INSTRUMENT_BLURB:
      instrument = struct_data;
      bse_instrument_set_blurb (instrument, description);
      break;

    case  JOB_ID_INSTRUMENT_NAME:
      instrument = struct_data;
      bse_instrument_set_name (instrument, description);
      break;

    case  JOB_ID_INSTRUMENT_DSAMPLE_PATH:
      instrument = struct_data;
      g_free (instrument->deferred_sample_path);
      instrument->deferred_sample_path = g_strdup (description);
      break;

    case  JOB_ID_SONG_BLURB:
      song = struct_data;
      bse_song_set_blurb (song, description);
      break;

    case  JOB_ID_SONG_AUTHOR:
      song = struct_data;
      bse_song_set_author (song, description);
      break;

    case  JOB_ID_SONG_COPYRIGHT:
      song = struct_data;
      bse_song_set_copyright (song, description);
      break;

    case  JOB_ID_SONG_BSE_VER:
      song = struct_data;
      g_free (song->bse_version);
      song->bse_version = g_strdup (description);
      break;

    default:
      g_assert_not_reached ();
      break;
    }

  g_free (description);
  
  if (scanner->token == ')')
    return G_TOKEN_NONE;
  else
    return G_TOKEN_STRING;
}

static guint
parser_scan_note (GScanner *scanner,
		  guint	   *note,
		  gchar	   **err_string)
{
  gchar *string;

  if (err_string)
    *err_string = NULL;
  g_return_val_if_fail (note != NULL, G_TOKEN_ERROR);
  *note = BSE_NOTE_VOID;

  g_scanner_get_next_token (scanner);
  if (scanner->token == '#')
    g_scanner_get_next_token (scanner);
  
  if (scanner->token >= 'A' && scanner->token <= 'z')
    {
      string = g_new (gchar, 2);
      string[0] = scanner->token;
      string[1] = 0;
    }
  else if (scanner->token == G_TOKEN_IDENTIFIER)
    string = g_strdup (scanner->value.v_identifier);
  else
    return G_TOKEN_IDENTIFIER;

  *note = bse_string_2_note (string);
  if (*note == BSE_NOTE_UNPARSABLE)
    {
      if (err_string)
	*err_string = g_strdup (string);
      *note = BSE_NOTE_VOID;
    }

  g_free (string);

  return G_TOKEN_NONE;
}

static guint
parse_note (GScanner		*scanner,
	    BseParserSymbol	*symbol,
	    guint		job_id,
	    gpointer		struct_data)
{
  guint expected_token;
  guint note;
  gchar *err_string;

  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  
  expected_token = parser_scan_note (scanner, &note, &err_string);
  if (expected_token != G_TOKEN_NONE)
    return expected_token;
  
  switch (job_id)
    {
      register BseParserBlock *pblock;
      
    case  JOB_ID_REC_NOTE:
      pblock = struct_data;
      if (err_string || note == BSE_NOTE_VOID)
	{
	  g_scanner_warn (scanner, "invalid recording note definition `%s'",
			  err_string ? err_string : "void");
	  g_free (err_string);
	  note = BSE_KAMMER_NOTE;
	}
      pblock->recording_note = note;
      break;
      
    default:
      g_assert_not_reached ();
      break;
    }
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';
  
  return G_TOKEN_NONE;
}

/* toggle a flag in (Anything*) struct_data determined by the
 * job_id.
 */
static guint
parse_flag (GScanner	      *scanner,
	    BseParserSymbol   *symbol,
	    guint	      job_id,
	    gpointer	      struct_data)
{
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  
  switch (job_id)
    {
      register BseParserBlock *pblock;
      register BseInstrument *instrument;
      
    case  JOB_ID_LITTLE_ENDIAN:
      pblock = struct_data;
      pblock->big_endian = FALSE;
      break;
      
    case  JOB_ID_BIG_ENDIAN:
      pblock = struct_data;
      pblock->big_endian = TRUE;
      break;

    case  JOB_ID_INSTRUMENT_INTERPOLATION_ON:
      instrument = struct_data;
      instrument->interpolation = TRUE;
      break;

    case  JOB_ID_INSTRUMENT_INTERPOLATION_OFF:
      instrument = struct_data;
      instrument->interpolation = FALSE;
      break;

    case  JOB_ID_INSTRUMENT_POLYPHONY_ON:
      instrument = struct_data;
      instrument->polyphony = TRUE;
      break;

    case  JOB_ID_INSTRUMENT_POLYPHONY_OFF:
      instrument = struct_data;
      instrument->polyphony = FALSE;
      break;

    default:
      g_assert_not_reached ();
    }
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';
  
  return G_TOKEN_NONE;
}

/* read a number with optional '-' prefix.
 * the number is assigned to (Anything*) struct_data via the
 * job_id.
 */
static guint
parse_number (GScanner		*scanner,
	      BseParserSymbol	*symbol,
	      guint		job_id,
	      gpointer		struct_data)
{
  register gdouble number;
  register gboolean negate;
  
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);
  
  negate = FALSE;
  g_scanner_get_next_token (scanner);
  while (scanner->token == '-')
    {
      negate = !negate;
      g_scanner_get_next_token (scanner);
    }
  if (scanner->token == G_TOKEN_FLOAT)
    number = scanner->value.v_float;
  else if (scanner->token == G_TOKEN_INT)
    number = scanner->value.v_int;
  else
    return G_TOKEN_INT;
  
  if (negate)
    number = - number;
  
  switch (job_id)
    {
      register BseParserBlock *pblock;
      register BseSample *sample;
      register BseInstrument *instrument;
      register BseSong *song;
      
    case  JOB_ID_OCTAVE_SHIFT:
      sample = struct_data;
      g_dataset_id_set_data (sample,
			     KEYID_NOTE_SHIFT,
			     (gpointer) ((gint) CLAMP (number * 12,
						       - BSE_MAX_NOTE / 2,
						       BSE_MAX_NOTE / 2)));
      break;
      
    case  JOB_ID_BLOCK_BYTE_SIZE:
      pblock = struct_data;
      pblock->byte_size = number;
      break;

    case  JOB_ID_BLOCK_N_VALUES:
      pblock = struct_data;
      pblock->n_values = number;
      break;
      
    case  JOB_ID_SAMPLE_N_CHANNELS:
      sample = struct_data;
      if (g_dataset_id_get_data (sample, KEYID_MUNK_QUEUE))
	g_scanner_error (scanner, "n-channels definition is preceed by munk definitions");
      else
	sample->n_channels = CLAMP (number, 1, BSE_MAX_SAMPLE_CHANNELS);
      break;
      
    case  JOB_ID_SAMPLE_REC_FREQ:
      sample = struct_data;
      sample->recording_freq = CLAMP (number, BSE_MIN_MIX_FREQ, BSE_MAX_MIX_FREQ);
      break;

    case  JOB_ID_SONG_BPM:
      song = struct_data;
      bse_song_set_bpm (song, CLAMP (number, BSE_MIN_BPM, BSE_MAX_BPM));
      break;
      
    case  JOB_ID_SONG_VOLUME:
      song = struct_data;
      bse_song_set_volume (song, CLAMP (number, BSE_MIN_VOLUME, BSE_MAX_VOLUME));
      break;
      
    case  JOB_ID_INSTRUMENT_VOLUME:
      instrument = struct_data;
      instrument->volume = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_BALANCE:
      instrument = struct_data;
      instrument->balance = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;
      
    case  JOB_ID_INSTRUMENT_TRANSPOSE:
      instrument = struct_data;
      instrument->transpose = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;
      
    case  JOB_ID_INSTRUMENT_FINE_TUNE:
      instrument = struct_data;
      instrument->fine_tune = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_DELAY_TIME:
      instrument = struct_data;
      instrument->delay_time = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_ATTACK_TIME:
      instrument = struct_data;
      instrument->attack_time = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_ATTACK_LEVEL:
      instrument = struct_data;
      instrument->attack_level = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_DECAY_TIME:
      instrument = struct_data;
      instrument->decay_time = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_SUSTAIN_LEVEL:
      instrument = struct_data;
      instrument->sustain_level = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_SUSTAIN_TIME:
      instrument = struct_data;
      instrument->sustain_time = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_RELEASE_LEVEL:
      instrument = struct_data;
      instrument->release_level = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;

    case  JOB_ID_INSTRUMENT_RELEASE_TIME:
      instrument = struct_data;
      instrument->release_time = number;
      bse_song_instrument_changed (instrument->song, instrument);
      break;
      
    default:
      g_assert_not_reached ();
      break;
    }
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';
  
  return G_TOKEN_NONE;
}

static guint
parse_date (GScanner	       *scanner,
	    BseParserSymbol    *symbol,
	    guint		job_id,
	    gpointer		struct_data)
{
  const guint n_formats = 9;
  guint year[n_formats];
  guint month[n_formats];
  guint day[n_formats];
  guint hour[n_formats];
  guint minute[n_formats];
  guint second[n_formats];
  gboolean success[n_formats];
  gboolean garbage[n_formats];
  gboolean finished;
  gchar *string;
  time_t ttime;
  guint i;
  
  g_return_val_if_fail (struct_data != NULL, G_TOKEN_ERROR);

  /* ok, this is tricky, we will support several date formats, where
   * all of them are encoded literally in strings. we make several attempts
   * to match such a string and pick the best one. if we acquire a full match
   * before all match possibilities have passed, we skip outstanding match
   * attempts. we do not use strptime, since it carries the locale(7) junk
   * that doesn't do anything usefull for the purpose of generic file parsing
   * and it doesn't give us the smallest clue whether the source string was
   * (not) valid in any meaningfull sense.
   * rules:
   * - years need to be specified by 4 digits
   * - date _and_ time need to be specified
   * - seconds are always optional
   *
   * the following formats are currently implemented:
   * "yyyy-mm-dd hh:mm:ss"
   * "mm/dd/yyyy hh:mm:ss"
   * "dd.mm.yyyy hh:mm:ss"
   * "hh:mm:ss yyyy-mm-dd"
   * "hh:mm:ss mm/dd/yyyy"
   * "hh:mm:ss dd.mm.yyyy"
   * "hh:mm yyyy-mm-dd"
   * "hh:mm mm/dd/yyyy"
   * "hh:mm dd.mm.yyyy"
   */
  
  g_scanner_get_next_token (scanner);
  if (scanner->token != G_TOKEN_STRING)
    return G_TOKEN_STRING;
  
  string = scanner->value.v_string;
  
  for (i = 0; i < n_formats; i++)
    {
      year[i] = month[i] = day[i] = 0;
      hour[i] = minute[i] = second[i] = 0;
      success[i] = garbage[i] = FALSE;
    }
  
  finished = FALSE;
  i = 0;
  
#define DATE_CHECK(index)	(year[(index)] >= 1990 &&	\
				 month[(index)] >= 1 &&		\
				 month[(index)] <= 12 &&	\
				 day[(index)] >= 1 &&		\
				 day[(index)] <= 31 &&		\
				 hour[(index)] >= 0 &&		\
				 hour[(index)] <= 23 &&		\
				 minute[(index)] >= 0 &&	\
				 minute[(index)] <= 59 &&	\
				 second[(index)] >= 0 &&	\
				 second[(index)] <= 61)
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "yyyy-mm-dd hh:mm:ss" e.g. "1998-04-16 23:59:59"
       */
      second[i] = 0;
      n_values = sscanf (string,
			 "%u-%u-%u %u:%u:%u%c",
			 &year[i], &month[i], &day[i],
			 &hour[i], &minute[i], &second[i],
			 &end_char);
      success[i] = n_values >= 5;
      garbage[i] = n_values > 6;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "mm/dd/yyyy hh:mm:ss" e.g. "04/16/1998 23:59:59"
       */
      second[i] = 0;
      n_values = sscanf (string,
			 "%u/%u/%u %u:%u:%u%c",
			 &month[i], &day[i], &year[i],
			 &hour[i], &minute[i], &second[i],
			 &end_char);
      success[i] = n_values >= 5;
      garbage[i] = n_values > 6;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "dd.mm.yyyy hh:mm:ss" e.g. "16.4.1998 23:59:59"
       */
      second[i] = 0;
      n_values = sscanf (string,
			 "%u.%u.%u %u:%u:%u%c",
			 &day[i], &month[i], &year[i],
			 &hour[i], &minute[i], &second[i],
			 &end_char);
      success[i] = n_values >= 5;
      garbage[i] = n_values > 6;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "hh:mm:ss yyyy-mm-dd" e.g. "23:59:59 1998-04-16"
       */
      n_values = sscanf (string,
			 "%u:%u:%u %u-%u-%u%c",
			 &hour[i], &minute[i], &second[i],
			 &year[i], &month[i], &day[i],
			 &end_char);
      success[i] = n_values >= 6;
      garbage[i] = n_values > 6;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "hh:mm yyyy-mm-dd" e.g. "23:59 1998-04-16"
       */
      second[i] = 0;
      n_values = sscanf (string,
			 "%u:%u %u-%u-%u%c",
			 &hour[i], &minute[i],
			 &year[i], &month[i], &day[i],
			 &end_char);
      success[i] = n_values >= 5;
      garbage[i] = n_values > 5;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "hh:mm:ss mm/dd/yyyy" e.g. "23:59:59 04/16/1998"
       */
      n_values = sscanf (string,
			 "%u:%u:%u %u/%u/%u%c",
			 &hour[i], &minute[i], &second[i],
			 &month[i], &day[i], &year[i],
			 &end_char);
      success[i] = n_values >= 6;
      garbage[i] = n_values > 6;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "hh:mm mm/dd/yyyy" e.g. "23:59 04/16/1998"
       */
      second[i] = 0;
      n_values = sscanf (string,
			 "%u:%u %u/%u/%u%c",
			 &hour[i], &minute[i],
			 &month[i], &day[i], &year[i],
			 &end_char);
      success[i] = n_values >= 5;
      garbage[i] = n_values > 5;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "hh:mm:ss dd.mm.yyyy" e.g. "23:59:59 16.4.1998"
       */
      n_values = sscanf (string,
			 "%u:%u:%u %u.%u.%u%c",
			 &hour[i], &minute[i], &second[i],
			 &day[i], &month[i], &year[i],
			 &end_char);
      success[i] = n_values >= 6;
      garbage[i] = n_values > 6;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
  if (!finished)
    {
      gint n_values;
      gchar end_char = 0;
      
      /* parse "hh:mm dd.mm.yyyy" e.g. "23:59:59 16.4.1998"
       */
      second[i] = 0;
      n_values = sscanf (string,
			 "%u:%u %u.%u.%u%c",
			 &hour[i], &minute[i],
			 &day[i], &month[i], &year[i],
			 &end_char);
      success[i] = n_values >= 5;
      garbage[i] = n_values > 5;
      finished = success[i] && !garbage[i] && DATE_CHECK (i);
      i++;
    }
#undef	DATE_CHECK
  
  /* ok, try to find out the best/first match if any
   */
  if (finished)
    i--;
  else
    {
      for (i = 0; i < n_formats - 1; i++)
	if (success[i])
	  break;
    }
  
  if (!success[i])
    {
      g_scanner_error (scanner, "unable to match date string `%s'", string);
      ttime = 0;
    }
  else
    {
      struct tm tm_data = { 0 };
      
      if (garbage[i])
	g_scanner_warn (scanner, "encountered garbage at end of date string `%s'", string);
      if (year[i] < 1990)
	{
	  g_scanner_warn (scanner, "year `%u' out of bounds in date string `%s'", year[i], string);
	  year[i] = 1990;
	}
      if (month[i] < 1 || month[i] > 12)
	{
	  g_scanner_warn (scanner, "month `%u' out of bounds in date string `%s'", month[i], string);
	  month[i] = CLAMP (month[i], 1, 12);
	}
      if (day[i] < 1 || day[i] > 31)
	{
	  g_scanner_warn (scanner, "day `%u' out of bounds in date string `%s'", day[i], string);
	  month[i] = CLAMP (day[i], 1, 31);
	}
      hour[i] = CLAMP (hour[i], 0, 23);
      minute[i] = CLAMP (minute[i], 0, 59);
      second[i] = CLAMP (second[i], 0, 61);
      
      tm_data.tm_sec = second[i];
      tm_data.tm_min = minute[i];
      tm_data.tm_hour = hour[i];
      tm_data.tm_mday = day[i];
      tm_data.tm_mon = month[i] - 1;
      tm_data.tm_year = year[i] - 1900;
      tm_data.tm_wday = 0;
      tm_data.tm_yday = 0;
      tm_data.tm_isdst = -1;
      
      ttime = mktime (&tm_data);
      
      /* printf ("DEBUG: year(%u) month(%u) day(%u) hour(%u) minute(%u) second(%u)\n",
       * year[i], month[i], day[i], hour[i], minute[i], second[i]);
       *
       * printf ("timeparser: (%s) secs=%lu, <%s>\n",
       * string,
       * ttime == -1 ? 0 : ttime, ctime (&ttime));
       */
    }
  
  if (ttime < 631148400) /* limit ttime to 1.1.1990 */
    ttime = 631148400;
  
  switch (job_id)
    {
      register BseSong *song;
      register BseSample *sample;

    case  JOB_ID_SONG_CREATED:
      song = struct_data;
      bse_song_set_creation (song, ttime);
      break;

    case  JOB_ID_SAMPLE_CREATED:
      sample = struct_data;
      bse_sample_set_creation (sample, ttime);
      break;

    case  JOB_ID_SONG_MODIFIED:
      song = struct_data;
      bse_song_set_modification (song, ttime);
      break;

    case  JOB_ID_SAMPLE_MODIFIED:
      sample = struct_data;
      bse_sample_set_modification (sample, ttime);
      break;

    default:
      g_assert_not_reached ();
      break;
    }

  g_scanner_get_next_token (scanner);
  if (scanner->token != ')')
    return ')';
  
  return G_TOKEN_NONE;
}
