/* BEAST - Bedevilled Audio System
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include "bstsongshell.h"
#include "bstshell.h"
#include "bstpatternview.h"
#include "bstpatterneditor.h"
#include "bstinstrumenteditor.h"
#include "gtkitemfactory.h"
#include "bstrcargs.h"
#include <gtk/gtksignal.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>



/* --- defines --- */

/* --- signals --- */
enum
{
  LAST_SIGNAL
};


/* --- prototypes --- */
static void	bst_song_shell_class_init	(BstSongShellClass	*klass);
static void	bst_song_shell_init		(BstSongShell		*pe);
static void	bst_song_shell_shutdown		(GtkObject		*object);
static void	bst_song_shell_destroy		(GtkObject		*object);
static void	bst_song_shell_ticker		(BstSongShell		*song_shell,
						 BseSong		*song,
						 BsePattern		*pattern,
						 guint			 row);
static void	bst_adjustments_value_changed	(GtkAdjustment		*adjustment,
						 BstSongShell		*song_shell);
static void	bst_song_shell_clist_select_row (GtkCList		*clist,
						 gint			 row,
						 gint			 column,
						 GdkEventButton		*event,
						 BstSongShell		*song_shell);
static gint	bst_song_shell_delete_event	(GtkWidget		*widget,
						 GdkEventAny		*event);


/* --- static variables --- */
static GtkWindowClass	*parent_class = NULL;
static guint		 song_shell_signals[LAST_SIGNAL] = { 0 };
static const gchar*	 key_song_shell = "BstSongShell";
static guint		 key_song_shell_id = 0;
static const gchar	*class_rc_string =
"#style'bst-BstSongShellClass-style'"
"{"
"bg[NORMAL]={1.,1.,1.}"
"base[NORMAL]={1.,0.,0.}"
"}"
"widget_class'*BstSongShell'style'bst-BstSongShellClass-style'"
"\n"
;
static gchar	*bst_song_shell_factories_path = "<BstSongShell>";

/* --- menus --- */
#define SSC_OP(song_shell_class_op)	\
  ((GtkItemFactoryCallback) bst_song_shell_clop), \
  (BST_SONG_SHELL_CLASS_OP_ ## song_shell_class_op)
#define SS_OP(song_shell_op)	\
  ((GtkItemFactoryCallback) bst_song_shell_operation), \
  (BST_SONG_SHELL_OP_ ## song_shell_op)

/* pattern popup, class basis
 */
static GtkItemFactoryEntry pview_popup_entries[] =
{
  { "/Pattern Menu",	"",		SSC_OP (NONE),		"<Title>" },
  { "/---",		"",		SSC_OP (NONE),		"<Separator>" },
  { "/New Pattern",	"<control>N",	SSC_OP (PATTERN_NEW),	"<Item>" },
  { "/New View",	"<control>P",	SSC_OP (PATTERN_VIEW),	"<Item>" },
};
static guint n_pview_popup_entries = (sizeof (pview_popup_entries) /
				      sizeof (pview_popup_entries[0]));

/* per song shell menubar
 */
static GtkItemFactoryEntry menubar_entries[] =
{
  { "/Song/New",	"",	   SS_OP (SHELL_NEW),	      "<Item>" },
  { "/Song/Open",	"<ctrl>O", SS_OP (SHELL_OPEN),	      "<Item>" },
  { "/Song/Save",	"<ctrl>S", SS_OP (SONG_SAVE),	      "<Item>" },
  { "/Song/Save As",	"",	   SS_OP (SONG_SAVE_AS),      "<Item>" },
  { "/Song/Preferences...", "",	   SS_OP (SHELL_PREFERENCES), "<Item>" },
  { "/Song/-----",	"",	   SS_OP (NONE),	      "<Separator>" },
  { "/Song/Dialogs/Pattern Play List...", "<ctrl>L", SS_OP (PATTERN_LIST), "<Item>" },
  { "/Song/Dialogs/Instrument List...", "<ctrl>I", SS_OP (INSTRUMENT_LIST), "<Item>" },
  { "/Song/-----",	"",	   SS_OP (NONE),	      "<Separator>" },
  { "/Song/Close",	"<ctrl>W", SS_OP (SONG_CLOSE),	      "<Item>" },
  { "/Song/Quit",	"<ctrl>Q", SS_OP (SHELL_QUIT),	      "<Item>" },
  { "/Help",		"",	   NULL, 0,		      "<LastBranch>" },
  { "/Help/About...",	"",	   SS_OP (SHELL_ABOUT),	      "<Item>" },
};
static guint n_menubar_entries = (sizeof (menubar_entries) /
				  sizeof (menubar_entries[0]));
										 

/* --- functions --- */
GtkType
bst_song_shell_get_type (void)
{
  static GtkType song_shell_type = 0;

  if (!song_shell_type)
    {
      GtkTypeInfo song_shell_info =
      {
	"BstSongShell",
	sizeof (BstSongShell),
	sizeof (BstSongShellClass),
	(GtkClassInitFunc) bst_song_shell_class_init,
	(GtkObjectInitFunc) bst_song_shell_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

	song_shell_type = gtk_type_unique (gtk_window_get_type (), &song_shell_info);
    }

  return song_shell_type;
}

static void
bst_song_shell_class_init (BstSongShellClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;

  if (!parent_class)
    parent_class = gtk_type_class (gtk_window_get_type ());

  key_song_shell_id = g_dataset_force_id (key_song_shell);

  // gtk_object_class_add_signals (object_class, song_shell_signals, LAST_SIGNAL);

  object_class->shutdown = bst_song_shell_shutdown;
  object_class->destroy = bst_song_shell_destroy;

  widget_class->delete_event = bst_song_shell_delete_event;

  class->factories_path = bst_song_shell_factories_path;
  class->pview_popup_factory = NULL;

  gtk_rc_parse_string (class_rc_string);
}

static void
bst_song_shell_shutdown (GtkObject		*object)
{
  BstSongShellClass *class;
  BstSongShell *song_shell;

  g_return_if_fail (object != NULL);

  song_shell = BST_SONG_SHELL (object);
  class = BST_SONG_SHELL_CLASS (GTK_OBJECT (song_shell)->klass);
  
  if (song_shell->is_playing)
    bst_song_shell_play_stop (song_shell);

  while (song_shell->pview_list)
    gtk_widget_destroy (song_shell->pview_list->data);

  song_shell->last_pattern = NULL;
  g_dataset_id_remove_data (song_shell->song, key_song_shell_id);
  bse_song_unref (song_shell->song);
  song_shell->song = NULL;

  GTK_OBJECT_CLASS (parent_class)->shutdown (object);
}

static void
bst_song_shell_destroy (GtkObject		*object)
{
  BstSongShell *song_shell;

  g_return_if_fail (object != NULL);

  song_shell = BST_SONG_SHELL (object);

  gtk_object_unref (BST_SONG_SHELL_CLASS (GTK_OBJECT (song_shell)->klass)->pview_popup_factory);

  gtk_object_unref (song_shell->menubar_factory);
  song_shell->menubar_factory = NULL;

  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static gint
bst_song_shell_delete_event (GtkWidget		    *widget,
			     GdkEventAny	    *event)
{
  bst_song_shell_operation (BST_SONG_SHELL (widget), BST_SONG_SHELL_OP_DELETE);

  return TRUE;
}

static void
bst_song_shell_init (BstSongShell *song_shell)
{
  BstSongShellClass *class;
  GtkWidget *main_vbox;
  GtkWidget *vbox;
  GtkWidget *vbox1;
  GtkWidget *hbox;
  GtkWidget *frame;
  GtkWidget *hscale;
  GtkWidget *spinner;
  gchar *clist_titles[16];
  guint i;

  class = BST_SONG_SHELL_CLASS (GTK_OBJECT (song_shell)->klass);

  song_shell->song = NULL;
  song_shell->current_pattern_guid = 0;

  song_shell->is_playing = FALSE;

  song_shell->pview_list = NULL;

  /* Pattern view popup
   */
  if (!class->pview_popup_factory)
    {
      class->pview_popup_factory =
	gtk_item_factory_new (gtk_menu_get_type (),
			      bst_song_shell_factories_path,
			      NULL);
      gtk_signal_connect (GTK_OBJECT (class->pview_popup_factory),
			  "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &class->pview_popup_factory);
      gtk_item_factory_create_items (GTK_ITEM_FACTORY (class->pview_popup_factory),
				     n_pview_popup_entries,
				     pview_popup_entries,
				     class);
    }
  else
    gtk_object_ref (class->pview_popup_factory);
  gtk_window_add_accelerator_table (GTK_WINDOW (song_shell),
				    GTK_ITEM_FACTORY (class->pview_popup_factory)->table);

  /* GUI
   */
  gtk_widget_set (GTK_WIDGET (song_shell),
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::title", "BSE-Song",
		  "GtkWindow::window_position", GTK_WIN_POS_NONE,
		  "GtkWindow::allow_shrink", FALSE,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", song_shell,
		    "GtkWidget::visible", TRUE,
		    NULL);

  /* MenuBar
   */
  song_shell->menubar_factory = gtk_item_factory_new (gtk_menu_bar_get_type (),
						      bst_song_shell_factories_path,
						      NULL);
  gtk_window_add_accelerator_table (GTK_WINDOW (song_shell),
				    GTK_ITEM_FACTORY (song_shell->menubar_factory)->table);
  gtk_item_factory_create_items (GTK_ITEM_FACTORY (song_shell->menubar_factory),
				 n_menubar_entries,
				 menubar_entries,
				 song_shell);
  gtk_box_pack_start (GTK_BOX (main_vbox),
		      GTK_ITEM_FACTORY (song_shell->menubar_factory)->widget,
		      FALSE,
		      TRUE,
		      0);
  gtk_widget_show (GTK_ITEM_FACTORY (song_shell->menubar_factory)->widget);

  vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::parent", main_vbox,
		    NULL);

  /* Song name
   */
  song_shell->name_entry =
    gtk_widget_new (gtk_entry_get_type (),
		    "GtkWidget::visible", TRUE,
		    //		    "GtkObject::object_signal::activate", gtk_window_activate_default, editor,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &song_shell->name_entry,
		    NULL);
  gtk_box_pack_start (GTK_BOX (vbox), song_shell->name_entry, FALSE, TRUE, 0);

  /* Bpm
   */
  frame =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::label", "Bpm",
		    "GtkFrame::label_xalign", 0.5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, TRUE, 0);
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", frame,
		    "GtkWidget::visible", TRUE,
		    NULL);
  song_shell->adj_bpm = (GtkAdjustment*)
    gtk_adjustment_new (bse_default_bpm,
			BSE_MIN_BPM, BSE_MAX_BPM,
			1,
			5, 0);
  gtk_signal_connect (GTK_OBJECT (song_shell->adj_bpm),
		      "value_changed",
		      GTK_SIGNAL_FUNC (bst_adjustments_value_changed),
		      song_shell);
  hscale =
    gtk_hscale_new (song_shell->adj_bpm);
  gtk_widget_set (hscale,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::can_focus", FALSE,
		  "GtkWidget::parent", hbox,
		  NULL);
  gtk_scale_set_digits (GTK_SCALE (hscale), 0);
  gtk_scale_set_draw_value (GTK_SCALE (hscale), FALSE);
  spinner =
    gtk_widget_new (gtk_spin_button_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, TRUE, 0);
  gtk_spin_button_construct (GTK_SPIN_BUTTON (spinner),
			     song_shell->adj_bpm,
			     0, 0);

  /* Master Volume
   */
  frame =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::label", "Master Volume",
		    "GtkFrame::label_xalign", 0.5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, TRUE, 0);
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", frame,
		    "GtkWidget::visible", TRUE,
		    NULL);
  song_shell->adj_volume = (GtkAdjustment*)
    gtk_adjustment_new (BSE_DFL_MASTER_VOLUME,
			BSE_MIN_VOLUME, BSE_MAX_VOLUME,
			1,
			5, 0);
  gtk_signal_connect (GTK_OBJECT (song_shell->adj_volume),
		      "value_changed",
		      GTK_SIGNAL_FUNC (bst_adjustments_value_changed),
		      song_shell);
  hscale =
    gtk_hscale_new (song_shell->adj_volume);
  gtk_widget_set (hscale,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::can_focus", FALSE,
		  "GtkWidget::parent", hbox,
		  NULL);
  gtk_scale_set_digits (GTK_SCALE (hscale), 0);
  gtk_scale_set_draw_value (GTK_SCALE (hscale), FALSE);
  spinner =
    gtk_widget_new (gtk_spin_button_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), spinner, FALSE, TRUE, 0);
  gtk_spin_button_construct (GTK_SPIN_BUTTON (spinner),
			     song_shell->adj_volume,
			     0, 0);

  /* Output
   */
  frame =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::label", "Output",
		    "GtkFrame::label_xalign", 0.5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, TRUE, 0);
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", frame,
		    "GtkWidget::visible", TRUE,
		    NULL);
  song_shell->play_button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Play",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::clicked", bst_song_shell_play_start, song_shell,
		    NULL);
  song_shell->play_indicator =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "Stop",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  song_shell->stop_button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Stop",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::clicked", bst_song_shell_play_stop, song_shell,
		    NULL);
  
  /* patterns
   */
  frame =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::label", "Patterns",
		    "GtkFrame::label_xalign", 0.5,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::parent", vbox,
		    NULL);
  vbox1 =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", frame,
		    "GtkWidget::visible", TRUE,
		    NULL);
  clist_titles[0] = "Guid";
  clist_titles[1] = "Blurb";
  song_shell->pattern_clist =
    gtk_clist_new_with_titles (2, clist_titles);
  gtk_widget_set (song_shell->pattern_clist,
		  "GtkWidget::height", 100,
		  "GtkObject::signal::select_row", bst_song_shell_clist_select_row, song_shell,
		  "GtkWidget::parent", vbox1,
		  "GtkWidget::visible", TRUE,
		  NULL);
  gtk_clist_set_policy (GTK_CLIST (song_shell->pattern_clist), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_clist_set_selection_mode (GTK_CLIST (song_shell->pattern_clist), GTK_SELECTION_BROWSE);
  gtk_clist_column_titles_passive (GTK_CLIST (song_shell->pattern_clist));
  i = 0;
  gtk_clist_set_column_width (GTK_CLIST (song_shell->pattern_clist),
			      i,
			      gdk_string_width (song_shell->pattern_clist->style->font, clist_titles[i]));

  /* dialogs
   */
  song_shell->pattern_list = NULL;
  song_shell->instrument_editor = NULL;

  /* update values
   */
  bst_song_shell_ticker (song_shell, NULL, NULL, 0);
}

static gint
auto_pattern_view (gpointer data)
{
  BstSongShell *song_shell = data;

  if (song_shell->current_pattern_guid)
    bst_song_shell_new_pattern_view (song_shell, song_shell->current_pattern_guid);
  gtk_object_unref (GTK_OBJECT (song_shell));

  return FALSE;
}

static gint
auto_instrument_list (gpointer data)
{
  BstSongShell *song_shell = data;

  bst_song_shell_operation (song_shell, BST_SONG_SHELL_OP_INSTRUMENT_LIST);
  gtk_object_unref (GTK_OBJECT (song_shell));

  return FALSE;
}


GtkWidget*
bst_song_shell_new (BseSong *song)
{
  GtkWidget *widget;

  g_return_val_if_fail (song != NULL, NULL);
  g_return_val_if_fail (g_dataset_id_get_data (song, key_song_shell_id) == NULL, NULL);

  widget = gtk_type_new (bst_song_shell_get_type ());

  BST_SONG_SHELL (widget)->song = song;
  bse_song_ref (song);
  g_dataset_id_set_data (song, key_song_shell_id, widget);

  gtk_window_set_title (GTK_WINDOW (widget), song->name);

  BST_SONG_SHELL (widget)->current_pattern_guid = 0;
  bst_song_shell_update (BST_SONG_SHELL (widget));

  if (BST_RCVAL_AUTO_PATTERN_VIEW)
    {
      gtk_object_ref (GTK_OBJECT (widget));
      gtk_idle_add (auto_pattern_view, widget);
    }

  if (BST_RCVAL_AUTO_INSTRUMENT_LIST)
    {
      gtk_object_ref (GTK_OBJECT (widget));
      gtk_idle_add (auto_instrument_list, widget);
    }

  return widget;
}

void
bst_song_shell_update (BstSongShell *song_shell)
{
  GList *list;
  
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));

  /* update song name
   */
  gtk_entry_set_text (GTK_ENTRY (song_shell->name_entry), song_shell->song->name);

  /* sliders
   */
  gtk_adjustment_set_value (song_shell->adj_bpm, song_shell->song->bpm);
  gtk_adjustment_set_value (song_shell->adj_volume, song_shell->song->master_volume);

  /* update pattern list
   */
  gtk_clist_freeze (GTK_CLIST (song_shell->pattern_clist));
  gtk_clist_clear (GTK_CLIST (song_shell->pattern_clist));
  for (list = song_shell->song->patterns; list; list = list->next)
    {
      gchar *text[16] = { NULL, };
      gchar buffer[16];
      BsePattern *pattern;
      
      pattern = list->data;

      sprintf (buffer, "%03u", bse_pattern_get_guid (pattern));
      text[0] = buffer;
      text[1] = (gchar*) bse_pattern_get_blurb (pattern);
      gtk_clist_append (GTK_CLIST (song_shell->pattern_clist), text);
    }
  gtk_clist_thaw (GTK_CLIST (song_shell->pattern_clist));
}

void
bst_song_shell_sequencer_tick (BstSongShell *song_shell)
{
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));

}

static void
bst_song_shell_delete_view (GtkWidget	 *pv,
			    BstSongShell *song_shell)
{
  g_return_if_fail (pv != NULL);
  g_return_if_fail (BST_IS_PATTERN_VIEW (pv));
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));

  song_shell->pview_list = g_list_remove (song_shell->pview_list, pv);
}

GtkWidget*
bst_song_shell_new_pattern_view (BstSongShell	*song_shell,
				 guint		 pattern_guid)
{
  GtkWidget *pv;
  BsePattern *pattern;

  g_return_val_if_fail (song_shell != NULL, NULL);
  g_return_val_if_fail (BST_IS_SONG_SHELL (song_shell), NULL);
  g_return_val_if_fail (pattern_guid > 0, NULL);

  pattern = bse_song_get_pattern (song_shell->song, pattern_guid);

  if (pattern)
    {
      pv = bst_pattern_view_new (pattern);
      gtk_widget_set (pv,
		      "GtkObject::signal::destroy", bst_song_shell_delete_view, song_shell,
		      NULL);

      song_shell->pview_list = g_list_prepend (song_shell->pview_list, pv);

      gtk_widget_show (pv);
    }
  else
    pv = NULL;

  return pv;
}

static void
bst_song_shell_ticker (BstSongShell   *song_shell,
		       BseSong	      *song,
		       BsePattern     *pattern,
		       guint	       row)
{
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));

  /* song might be NULL as well
   */

  if (!pattern)
    song_shell->is_playing = FALSE;

  gtk_label_set (GTK_LABEL (song_shell->play_indicator),
		 song_shell->is_playing ? "Playing" : "Stopped");
  gtk_widget_set_sensitive (song_shell->play_button, !song_shell->is_playing);
  gtk_widget_set_sensitive (song_shell->stop_button, song_shell->is_playing);
}

void
bst_song_shell_play_start (BstSongShell	  *song_shell)
{
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));

  if (song_shell->is_playing)
    return;

  bse_main_play_song_with_ticks (song_shell->song,
				 (BseSequencerTickCB) bst_song_shell_ticker,
				 song_shell);
  song_shell->is_playing = TRUE;
  gtk_widget_set_sensitive (song_shell->play_button, FALSE);
  gtk_widget_set_sensitive (song_shell->stop_button, TRUE);
}

void
bst_song_shell_play_stop (BstSongShell	 *song_shell)
{
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));

  if (!song_shell->is_playing)
    return;

  bse_main_quit_song (song_shell->song);
}

static void
bst_adjustments_value_changed (GtkAdjustment	   *adjustment,
			       BstSongShell	   *song_shell)
{
  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (GTK_IS_ADJUSTMENT (adjustment));
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));

  bse_song_set_bpm (song_shell->song, song_shell->adj_bpm->value);
  bse_song_set_volume (song_shell->song, song_shell->adj_volume->value);
}

BstSongShell*
bst_song_shell_from_song (BseSong	 *song)
{
  BstSongShell *song_shell;

  g_return_val_if_fail (song != NULL, NULL);

  song_shell = g_dataset_id_get_data (song, key_song_shell_id);
  if (song_shell)
    {
      g_return_val_if_fail (song_shell != NULL, NULL);
      g_return_val_if_fail (BST_IS_SONG_SHELL (song_shell), NULL);
    }

  return song_shell;
}

static void
bst_song_shell_clist_select_row (GtkCList	*clist,
				 gint		 row,
				 gint		 column,
				 GdkEventButton *event,
				 BstSongShell	*song_shell)
{
  g_return_if_fail (clist != NULL);
  g_return_if_fail (GTK_IS_CLIST (clist));
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));
  
  song_shell->current_pattern_guid = MAX (0, row) + 1;
  
  if (row >= 0 &&
      event &&
      event->type == GDK_BUTTON_PRESS &&
      event->button == 3)
    {
      BstSongShellClass *class;

      class = BST_SONG_SHELL_CLASS (GTK_OBJECT (song_shell)->klass);

      gtk_item_factory_popup (GTK_ITEM_FACTORY (class->pview_popup_factory),
			      song_shell,
			      event->x_root,
			      event->y_root,
			      event->button,
			      event->time);
    }
}

static void
bst_song_shell_dialog_instrument_editor (BstSongShell *song_shell)
{
  BstInstrumentEditor *instrument_editor;

  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));
  g_return_if_fail (song_shell->instrument_editor == NULL);
  
  song_shell->instrument_editor = bst_instrument_editor_new (song_shell->song);

  instrument_editor = BST_INSTRUMENT_EDITOR (song_shell->instrument_editor);

  gtk_signal_connect (GTK_OBJECT (instrument_editor),
		      "destroy",
		      GTK_SIGNAL_FUNC (gtk_widget_destroyed),
		      &song_shell->instrument_editor);
  gtk_signal_connect_object_while_alive (GTK_OBJECT (song_shell),
					 "destroy",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (instrument_editor));
}

void
bst_song_shell_clop (BstSongShellClass		*class,
		     BstSongShellClassOps	 ssc_op)
{
  GtkWidget *widget;
  BstSongShell *song_shell;

  g_return_if_fail (class != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL_CLASS (class));
  g_return_if_fail (ssc_op < BST_SONG_SHELL_CLASS_OP_LAST);

  widget = gtk_get_event_widget (gtk_get_current_event ());
  g_return_if_fail (widget != NULL);
  widget = gtk_widget_get_toplevel (widget);
  g_return_if_fail (widget != NULL);

  /* if we got called from the popup, find out about the invoking shell
   */
  if (GTK_IS_MENU (widget))
    widget = gtk_object_get_user_data (GTK_OBJECT (widget));

  g_return_if_fail (BST_IS_SONG_SHELL (widget));

  song_shell = BST_SONG_SHELL (widget);

  gtk_widget_ref (GTK_WIDGET (song_shell));

  switch (ssc_op)
    {
      BsePattern *pattern;

    case  BST_SONG_SHELL_CLASS_OP_NONE:
      break;

      /* pattern operations
       */
    case  BST_SONG_SHELL_CLASS_OP_PATTERN_NEW:
      pattern = bse_song_add_pattern (song_shell->song);
      bst_song_shell_update (song_shell);
      // bst_song_shell_new_pattern_view (song_shell, bse_pattern_get_guid (pattern));
      break;
    case  BST_SONG_SHELL_CLASS_OP_PATTERN_VIEW:
      g_return_if_fail (song_shell->current_pattern_guid > 0);
      bst_song_shell_new_pattern_view (song_shell, song_shell->current_pattern_guid);
      break;

    default:
      g_warning ("BstSongShellClassOps: operation `%d' unhandled\n", ssc_op);
      break;
    }

  gtk_widget_unref (GTK_WIDGET (song_shell));
}

void
bst_song_shell_operation (BstSongShell	  *song_shell,
			  BstSongShellOps  ss_op)
{
  g_return_if_fail (song_shell != NULL);
  g_return_if_fail (BST_IS_SONG_SHELL (song_shell));
  g_return_if_fail (ss_op < BST_SONG_SHELL_OP_LAST);

  gtk_widget_ref (GTK_WIDGET (song_shell));

  switch (ss_op)
    {
    case  BST_SONG_SHELL_OP_NONE:
      break;

      /* song operations
       */
    case  BST_SONG_SHELL_OP_SONG_SAVE:
      printf ("FIXME: unimplemented\n");
      break;
    case  BST_SONG_SHELL_OP_SONG_SAVE_AS:
      printf ("DEBUG: test saving to ./tmp-song.bse\n");
      bse_io_save_song_test ("./tmp-song.bse", song_shell->song);
      break;
    case  BST_SONG_SHELL_OP_SONG_CLOSE:
      bst_song_shell_operation (song_shell, BST_SONG_SHELL_OP_DELETE);
      break;
	
      /* Dialogs
       */
    case  BST_SONG_SHELL_OP_PATTERN_LIST:
      printf ("FIXME: unimplemented\n");
      break;
    case  BST_SONG_SHELL_OP_INSTRUMENT_LIST:
      if (!song_shell->instrument_editor)
	bst_song_shell_dialog_instrument_editor (song_shell);
      gtk_widget_show (song_shell->instrument_editor);
      gdk_window_raise (song_shell->instrument_editor->window);
      break;

      /* operations forwarded to BstShell
       */
    case  BST_SONG_SHELL_OP_SHELL_NEW:
      bst_shell_operation (bst_shell_get (), BST_SHELL_OP_SONG_NEW);
      break;
    case  BST_SONG_SHELL_OP_SHELL_OPEN:
      bst_shell_operation (bst_shell_get (), BST_SHELL_OP_OPEN_SONG);
      break;
    case  BST_SONG_SHELL_OP_SHELL_PREFERENCES:
      bst_shell_operation (bst_shell_get (), BST_SHELL_OP_PREFERENCES);
      break;
    case  BST_SONG_SHELL_OP_SHELL_QUIT:
      bst_shell_operation (bst_shell_get (), BST_SHELL_OP_DELETE);
      break;
    case  BST_SONG_SHELL_OP_SHELL_ABOUT:
      bst_shell_operation (bst_shell_get (), BST_SHELL_OP_ABOUT);
      break;
      
      /* misc
       */
    case  BST_SONG_SHELL_OP_DELETE:
      gtk_widget_destroy (GTK_WIDGET (song_shell));
      break;

    default:
      g_warning ("BstSongShellOps: operation `%d' unhandled\n", ss_op);
      break;
    }

  gtk_widget_unref (GTK_WIDGET (song_shell));
}
