#!/usr/bin/env python

from gnuradio import gr
from gnuradio import usrp
from gnuradio.eng_option import eng_option
from optparse import OptionParser


class siggen (object):
    __slots__ = ['interp', 'waveform_type', 'waveform_ampl',
                 'waveform_freq', 'waveform_offset', 'fg', 'usrp',
                 'siggen', 'noisegen', 'src', 'file_sink' ]

    def __init__ (self):
        # controllable values
        self.interp = 64
        self.waveform_type = gr.GR_SIN_WAVE
        self.waveform_ampl = 16000
        self.waveform_freq = 100.12345e3
        self.waveform_offset = 0
        self._instantiate_blocks ()
        self.set_waveform_type (self.waveform_type)

    def start (self):
        self.fg.start ()

    def stop (self):
        self.fg.stop ()
        
    def usb_freq (self):
        return self.usrp.dac_freq() / self.interp

    def usb_throughput (self):
        return self.usb_freq () * 4
        
    def set_waveform_type (self, type):
        '''
        valid waveform types are: gr.GR_SIN_WAVE, gr.GR_CONST_WAVE,
        gr.GR_UNIFORM and gr.GR_GAUSSIAN
        '''
        self._configure_graph (type)
        self.waveform_type = type

    def set_waveform_ampl (self, ampl):
        self.waveform_ampl = ampl
        self.siggen.set_amplitude (ampl)
        self.noisegen.set_amplitude (ampl)

    def set_waveform_freq (self, freq):
        self.waveform_freq = freq
        self.siggen.set_frequency (freq)
        
    def set_waveform_offset (self, offset):
        self.waveform_offset = offset
        self.siggen.set_offset (offset)

    def set_interpolator (self, interp):
        self.interp = interp
        self.siggen.set_sampling_freq (self.usb_freq ())
        self.usrp.set_interp_rate (interp)

    def set_duc_freq (self, freq):
        self.usrp.set_tx_freq (0, freq)
        
    def _instantiate_blocks (self):
        self.fg = gr.flow_graph ()
        self.src = None
        self.usrp = usrp.sink_c (0, self.interp)
        
        self.siggen = gr.sig_source_c (self.usb_freq (),
                                       gr.GR_SIN_WAVE,
                                       self.waveform_freq,
                                       self.waveform_ampl,
                                       self.waveform_offset)

        self.noisegen = gr.noise_source_c (gr.GR_UNIFORM,
                                           self.waveform_ampl)

        # self.file_sink = gr.file_sink (gr.sizeof_gr_complex, "siggen.dat")

    def _configure_graph (self, type):
        was_running = self.fg.is_running ()
        if was_running:
            self.fg.stop ()
        self.fg.disconnect_all ()
        if type == gr.GR_SIN_WAVE or type == gr.GR_CONST_WAVE:
            self.fg.connect (self.siggen, self.usrp)
            # self.fg.connect (self.siggen, self.file_sink)
            self.siggen.set_waveform (type)
            self.src = self.siggen
        elif type == gr.GR_UNIFORM or type == gr.GR_GAUSSIAN:
            self.fg.connect (self.noisegen, self.usrp)
            self.noisegen.set_type (type)
            self.src = self.noisegen
        else:
            raise ValueError, type
        if was_running:
            self.fg.start ()


def main ():
    parser = OptionParser (option_class=eng_option)
    parser.add_option ("--sine", dest="type", action="store_const", const=gr.GR_SIN_WAVE,
                       help="generate a complex sinusoid [default]", default=gr.GR_SIN_WAVE)
    parser.add_option ("--const", dest="type", action="store_const", const=gr.GR_CONST_WAVE, 
                       help="generate a constant output")
    parser.add_option ("--gaussian", dest="type", action="store_const", const=gr.GR_GAUSSIAN,
                       help="generate Gaussian random output")
    parser.add_option ("--uniform", dest="type", action="store_const", const=gr.GR_UNIFORM,
                       help="generate Uniform random output")

    parser.add_option ("-i", "--interp", type="int", default=64,
                       help="set fgpa interpolation rate to INTERP")
    parser.add_option ("-f", "--freq", type="eng_float", default=100e3,
                       help="set waveform frequency to FREQ")
    parser.add_option ("-a", "--amplitude", type="eng_float", default=16e3,
                       help="set waveform amplitude to AMPLITUDE", metavar="AMPL")
    parser.add_option ("-o", "--offset", type="eng_float", default=0,
                       help="set waveform offset to OFFSET")
    parser.add_option ("-c", "--duc-freq", type="eng_float", default=0,
                       help="set Tx DUC frequency to FREQ", metavar="FREQ")
    parser.add_option ("-n", "--nchannels", type="int", default=1,
                       help="set number of output channels to NCHANNELS")
    parser.add_option ("-m", "--mux", type="intx", default=0x98,
                       help="set output mux register")
    (options, args) = parser.parse_args ()

    sg = siggen ()
    sg.set_interpolator (options.interp)
    sg.set_waveform_type (options.type)
    sg.set_waveform_freq (options.freq)
    sg.set_waveform_ampl (options.amplitude)
    sg.set_waveform_offset (options.offset)
    sg.set_duc_freq (options.duc_freq)

    sg.usrp.set_nchannels (options.nchannels)
    sg.usrp.set_mux (options.mux)

    sg.start ()
    raw_input ('Press Enter to quit: ')
    sg.stop ()

if __name__ == '__main__':
    main ()
