/* really simplistic program to join multiple tcpdump files together */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>

#define TCPDUMP_MAGIC 0xa1b2c3d4
#define VERSION_MAJOR 2
#define VERSION_MINOR 2

/*
 * THE first record in the file contains saved values for some
 * of the flags used in the printout phases of tcpdump.
 * Many fields here are longs so compilers won't insert unwanted
 * padding; these files need to be interchangeable across architectures.
 */
struct file_header {
	u_long magic;
	u_short version_major;
	u_short version_minor;
	long thiszone;		/* gmt to local correction */
	u_long sigfigs;		/* accuracy of timestamps */
	u_long snaplen;		/* max length saved portion of each pkt */
	u_long linktype;
};

long Thiszone;
u_long Precision, Snaplen, Linktype;

/*
 * Each packet in the dump file is prepended with this generic header.
 * This gets around the problem of different headers for different
 * packet interfaces.
 */
struct packet_header {
	struct timeval ts;	/* time stamp */
	u_long len;		/* length this packet (off wire) */
	u_long caplen;		/* length of portion present */
};

/* true if the contents of the savefile being read are byte-swapped */
int sf_swapped;

#define	SWAPLONG(y) \
((((y)&0xff)<<24) | (((y)&0xff00)<<8) | (((y)&0xff0000)>>8) | (((y)>>24)&0xff))
#define	SWAPSHORT(y) \
	( (((y)&0xff)<<8) | (((y)&0xff00)>>8) )

FILE *sf_readfile;
FILE *sf_writefile;
int Firstfile=1;

static int sf_write_header(void);
static void swap_hdr(struct file_header *hp);
void sf_read_init(char *fname);
int sf_next_packet(struct packet_header *, u_char *, int );
void sf_write_init(char *fname);
void sf_write(u_char *, struct timeval *, int, int );
void sf_write_end(void);

main( int argc, char *argv[] )
{
  int src = 1;
  int dest = argc;
  struct packet_header phdr;
  u_char buf[4096];
  int buflen = 4096;

  argc--;
  if( argc <= 1 ) {
    fprintf(stderr,"Usage: join_dumps [src_files...] destination\n");
    exit(1);
  }
  printf("destination is %s\n",argv[dest-1]);
  while( --argc ) {
    printf("source is %s\n",argv[src]);
    sf_read_init(argv[src]);
    if( Firstfile) { 
      sf_write_init(argv[dest-1]);
      Firstfile=0;
    }
    while( sf_next_packet(&phdr, buf, buflen)==0 )  {
      sf_write(buf, &phdr.ts, phdr.len, phdr.caplen);
    }
    src++;
  }
  sf_write_end();
}
    
static int
sf_write_header()
{
  struct file_header hdr;
  
  hdr.magic = TCPDUMP_MAGIC;
  hdr.version_major = VERSION_MAJOR;
  hdr.version_minor = VERSION_MINOR;
  
  hdr.thiszone = Thiszone;
  hdr.snaplen = Snaplen;
  hdr.sigfigs = Precision;
  hdr.linktype = Linktype;
  
  if (fwrite((char *)&hdr, sizeof(hdr), 1, sf_writefile) != 1)
    return -1;
  
  return 0;
}

static void
swap_hdr(struct file_header *hp)
{
  hp->version_major = SWAPSHORT(hp->version_major);
  hp->version_minor = SWAPSHORT(hp->version_minor);
  hp->thiszone = SWAPLONG(hp->thiszone);
  hp->sigfigs = SWAPLONG(hp->sigfigs);
  hp->snaplen = SWAPLONG(hp->snaplen);
  hp->linktype = SWAPLONG(hp->linktype);
}

void
sf_read_init(char *fname)
{
  register FILE *fp;
  struct file_header hdr;
  
  fp = fopen(fname, "r");
  if (fp == 0) {
    (void) fprintf(stderr, "tcpdump: fopen: ");
    perror(fname);
    exit(1);
  }

  if (fread((char *)&hdr, sizeof(hdr), 1, fp) != 1) {
    (void) fprintf(stderr, "tcpdump: fread: ");
    perror(fname);
    exit(1);
  }

  if (hdr.magic != TCPDUMP_MAGIC) {
    if (SWAPLONG(hdr.magic) != TCPDUMP_MAGIC) {
      fprintf(stderr," ERROR: Magic Header not found on file %s\n",fname);
      exit(1);
    }
    sf_swapped = 1;
    swap_hdr(&hdr);
  }

  if (hdr.version_major < VERSION_MAJOR) {
      fprintf(stderr," ERROR: Bad version on file %s\n",fname);
      exit(1);
    }

  if( Firstfile ) {
    Thiszone = hdr.thiszone;
    Snaplen = hdr.snaplen;
    Linktype = hdr.linktype;
    Precision = hdr.sigfigs;
  } else {
    if( Snaplen != hdr.snaplen ) {
      fprintf(stderr,"ERROR: Capture length changes from %d to %d in file %s\n",
	      Snaplen, hdr.snaplen,fname);
      exit(1);
    }
    if( Linktype != hdr.linktype ) {
      fprintf(stderr,"ERROR: Link type changes from %d to %d in file %s\n",
	      Linktype,hdr.linktype,fname);
      exit(1);
    }
    if( Precision != hdr.sigfigs ) {
      fprintf(stderr,"ERROR: Precision changes from %d to %d in file %s\n",
	      Precision,hdr.sigfigs,fname);
      exit(1);
    }
  }
  
  sf_readfile = fp;
}

/*
 * Read sf_readfile and return the next packet.  Return the header in hdr 
 * and the contents in buf.  Return 0 on success, SFERR_EOF if there were 
 * no more packets, and SFERR_TRUNC if a partial packet was encountered.
 */
int
sf_next_packet(hdr, buf, buflen)
	struct packet_header *hdr;
	u_char *buf;
	int buflen;
{
	FILE *fp = sf_readfile;

	/* read the stamp */
	if (fread((char *)hdr, sizeof(struct packet_header), 1, fp) != 1) {
		/* probably an EOF, though could be a truncated packet */
		return 1;
	}

	if (sf_swapped) {
		/* these were written in opposite byte order */
		hdr->caplen = SWAPLONG(hdr->caplen);
		hdr->len = SWAPLONG(hdr->len);
		hdr->ts.tv_sec = SWAPLONG(hdr->ts.tv_sec);
		hdr->ts.tv_usec = SWAPLONG(hdr->ts.tv_usec);
	}

	if (hdr->caplen > buflen)
		return 1;

	/* read the packet itself */
	if (fread((char *)buf, hdr->caplen, 1, fp) != 1)
		return 1;
	return 0;
}

/*
 * Initialize so that sf_write() will output to the file named 'fname'.
 */
void
sf_write_init(char *fname)
{
  sf_writefile = fopen(fname, "w");
  if (sf_writefile == 0) {
    (void) fprintf(stderr, "tcpdump: fopen: ");
    perror(fname);
    exit(1);
  }
  (void)sf_write_header();
}

/*
 * Output a packet to the intialized dump file.
 */
void
sf_write(sp, tvp, length, caplen)
     u_char *sp;
     struct timeval *tvp;
     int length;
     int caplen;
{
  struct packet_header h;
  
  h.ts.tv_sec = tvp->tv_sec;
  h.ts.tv_usec = tvp->tv_usec;
  h.len = length;
  h.caplen = caplen;
  printf("len=%d caplen=%d\n",h.len,h.caplen);  
  (void)fwrite((char *)&h, sizeof(struct packet_header), 1, sf_writefile);
  (void)fwrite((char *)sp, caplen, 1, sf_writefile);
}

void sf_write_end()
{
  fclose(sf_writefile);
}













