/*
   fn.c: functions for adding and deleting functions from the symbol table.
   Support for signal handlers is also found here.
*/

#include <signal.h>
#include "rc.h"
#include "sigmsgs.h"

static void fn_handler(int);

static Node *handlers[NUMOFSIGNALS], ignore_function, null_function;
static bool runexit = FALSE;

/*
   Set signals to default values for rc. This means that interactive
   shells ignore SIGQUIT, etc.
*/

extern void inithandler() {
	int i;
	null_function.type = nBody;
	null_function.u[0].p = null_function.u[1].p = NULL;
	ignore_function = null_function;
	if (!dashdee) {
		rc_signal(SIGQUIT, SIG_IGN);
		handlers[SIGQUIT] = &null_function;
		rc_signal(SIGTERM, SIG_IGN);
		handlers[SIGTERM] = &null_function;
	}
	for (i = 1; i < NUMOFSIGNALS; i++) {
		if ((interactive && handlers[i] == &null_function)
#ifdef NOSIGCLD
		|| i == SIGCLD
#endif
		)
			continue;
		if (sighandlers[i] == SIG_IGN)
			fnassign(signals[i][0], NULL); /* ignore incoming ignored signals */
	}
	if (interactive || sighandlers[SIGINT] != SIG_IGN)
		fnrm("sigint");
}

/* only run this in a child process! resets signals to their default values */

extern void setsigdefaults(bool sysvbackground) {
	int i;
	/*
	   General housekeeping: (setsigdefaults happens after fork(),
	   so it's a convenient place to clean up)
	*/
	/*interactive = FALSE; don't need to do this? */
	closefds();	/* close open file descriptors like history, scripts, etc. */
	/* Restore signals to SIG_DFL */
	for (i = 1; i < NUMOFSIGNALS; i++)
		if (sysvbackground && (i == SIGINT || i == SIGQUIT || i == SIGTERM)) { /* old-style backgrounding */
			handlers[i] = &ignore_function;
			rc_signal(i, SIG_IGN);
		} else if ((i == SIGINT || handlers[i] != NULL) && handlers[i] != &ignore_function) { /* generic signal-restoring code */
			handlers[i] = NULL;
			rc_signal(i, SIG_DFL);
		}
	runexit = FALSE; /* No sigexit on subshells */
}

/* rc's exit. if runexit is set, run the sigexit function. */

extern void rc_exit(int stat) {
	static char *sigexit[2] = {
		"sigexit",
		NULL
	};
	if (runexit) {
		runexit = FALSE;
		funcall(sigexit);
		stat = getstatus();
	}
	exit(stat);
}

/* The signal handler for all functions. calls walk() */

static void fn_handler(int s) {
	List *dollarzero;
	Estack e;
	if (s < 1 || s >= NUMOFSIGNALS)
		panic("unknown signal");
	dollarzero = nnew(List);
	dollarzero->w = signals[s][0];
	dollarzero->n = NULL;
	varassign("*", dollarzero, TRUE);
	except(eVarstack, "*", &e);
	walk(handlers[s], TRUE);
	varrm("*", TRUE);
	unexcept(); /* eVarstack */
}

/*
   Assign a function in Node form. Check to see if the function is also
   a signal, and set the signal vectors appropriately.
*/

extern void fnassign(char *name, Node *def) {
	Node *newdef = treecpy(def == NULL ? &null_function : def, ealloc); /* important to do the treecopy first */
	Function *new = get_fn_place(name);
	int i;
	new->def = newdef;
	new->extdef = NULL;
	if (strncmp(name, "sig", conststrlen("sig")) == 0) { /* slight optimization */
#ifdef NOSIGCLD /* System V machines treat SIGCLD very specially */
		if (streq(name, "sigcld"))
			rc_error("can't trap SIGCLD");
#endif
		if (streq(name, "sigexit"))
			runexit = TRUE;
		for (i = 1; i < NUMOFSIGNALS; i++) /* zero is a bogus signal */
			if (streq(signals[i][0], name)) {
				if (def != NULL) {
					handlers[i] = newdef;
					rc_signal(i, fn_handler);
				} else {
					handlers[i] = &ignore_function;
					rc_signal(i, SIG_IGN);
				}
				break;
			}
	}
}

/* Assign a function from the environment. Store just the external representation */

extern void fnassign_string(char *extdef) {
	char *name = get_name(extdef+3); /* +3 to skip over "fn_" */
	Function *new;
	if (name == NULL)
		return;
	new = get_fn_place(name);
	new->def = NULL;
	new->extdef = ecpy(extdef);
}

/* Return a function in Node form, evaluating an entry from the environment if necessary */

extern Node *fnlookup(char *name) {
	Function *look = lookup_fn(name);
	Node *ret;
	if (look == NULL)
		return NULL; /* not found */
	if (look->def != NULL)
		return look->def;
	if (look->extdef == NULL) /* function was set to null, e.g., fn foo {} */
		return &null_function;
	ret = parse_fn(name, look->extdef);
	if (ret == NULL) {
		efree(look->extdef);
		look->extdef = NULL;
		return &null_function;
	} else {
		return look->def = treecpy(ret, ealloc); /* Need to take it out of nalloc space */
	}
}

/* Return a function in string form (used by makeenv) */

extern char *fnlookup_string(char *name) {
	Function *look = lookup_fn(name);

	if (look == NULL)
		return NULL;
	if (look->extdef != NULL)
		return look->extdef;
	return look->extdef = fun2str(name, look->def);
}

/*
   Remove a function from the symbol table. If it also defines a signal
   handler, restore the signal handler to its default value.
*/

extern void fnrm(char *name) {
	int i;
	for (i = 1; i < NUMOFSIGNALS; i++) /* signal 0 unused */
		if (streq(signals[i][0], name)) {
			handlers[i] = NULL;
			if (i == SIGINT)
				rc_signal(i, sigint); /* restore default signal handler for rc */
			else if ((i == SIGQUIT || i == SIGTERM) && !dashdee) {
				handlers[i] = &null_function;
				rc_signal(i, SIG_IGN);	 /* ditto */
			} else {
				rc_signal(i, SIG_DFL);
			}
		}
	if (streq(name, "sigexit"))
		runexit = FALSE;
	delete_fn(name);
}

extern void whatare_all_signals() {
	int i;
	for (i = 1; i < NUMOFSIGNALS; i++)
		if (*signals[i][0] != '\0')
			if (handlers[i] == NULL)
				fprint(1, "fn %s\n", signals[i][0]);
			else
				fprint(1, "fn %S {%T}\n", signals[i][0], handlers[i]);
}

extern void prettyprint_fn(int fd, char *name, Node *n) {
	fprint(fd, "fn %S {%T}\n", name, n);
}
