#include <setjmp.h>
#include <signal.h>
#include "rc.h"
#include "jbwrap.h"

/*
   A return goes back stack frames to the last return. A break does
   not. A signal goes to the last interactive level. (see below)
*/

static Estack *estack;

/* add an exception to the input stack. */

extern void except(ecodes e, void *jb, Estack *ex) {
	ex->prev = estack;
	estack = ex;
	switch (estack->e = e) {
	case eArena:
		estack->b = newblock();
		break;
	case eError:
	case eBreak:
	case eReturn:
		estack->interactive = interactive;
		estack->jb = (Jbwrap *) jb;
		break;
	case eVarstack:
		estack->name = (char *) jb;
		break;
	}
}

/* remove an exception, restore last interactive value */

extern void unexcept() {
	if (estack->e == eError)
		interactive = estack->interactive;
	else if (estack->e == eArena)
		restoreblock(estack->b);
	estack = estack->prev;
}

/*
   Raise an exception. The rules are pretty complicated: you can return
   from a loop inside a function, but you can't break from a function
   inside of a loop. On errors, rc_raise() goes back to the LAST
   INTERACTIVE stack frame. If no such frame exists, then rc_raise()
   exits the shell.  This is what happens, say, when there is a syntax
   error in a noninteractive shell script. While traversing the
   exception stack backwards, rc_raise() also removes input sources
   (closing file-descriptors, etc.) and pops instances of variables
   that have been pushed onto the variable stack (e.g., for a function
   call (for $*) or a local assignment).
*/

extern void rc_raise(ecodes e) {
	if (e == eError && rc_pid != getpid())
		exit(1); /* child processes exit on an error/signal */
	for (; estack != NULL; estack = estack->prev)
		if (estack->e != e) {
			if (e == eBreak && estack->e != eArena)
				rc_error("break outside of loop");
			else if (e == eReturn && estack->e == eError) /* can return from loops inside functions */
				rc_error("return outside of function");
			if (estack->e == eVarstack)
				varrm(estack->name, TRUE);
			else if (estack->e == eArena)
				restoreblock(estack->b);
		} else {
			if (e == eError && !estack->interactive) {
				popinput();
			} else {
				Jbwrap *j = estack->jb;

				interactive = estack->interactive;
				estack = estack->prev;
				longjmp(j->j, 1);
			}
		}
	rc_exit(1); /* top of exception stack */
}

/* exception handlers */

extern void rc_error(char *s) {
	pr_error(s);
	set(FALSE);
	redirq = NULL;
	cond = FALSE; /* no longer inside conditional */
	empty_fifoq();
	rc_raise(eError);
}

extern void sigint(int s) {
	if (s != SIGINT)
		panic("s != SIGINT in sigint catcher");
	/* this is the newline you see when you hit ^C while typing a command */
	fprint(2, "\n");
	redirq = NULL;
	cond = FALSE;
	empty_fifoq();
	rc_raise(eError);
}
