/* Copyright 1992 by Simmule R. Turner

   Permission is granted to make and distribute VERBATIM copies
   of this software provided the above copyright notice and this
   permission notice are preserved in all copies.

   ALL other rights are reserved.

   NO WARRANTY is provided with this software. */


static char *_scid_ = "$Header: editline.c, v0.95.1 3/8/92 12:27:45  ++SrT$";


#include "readline.h"
#include "editline.h"
#include "support.x"


char *readline(prompt)
char *prompt;
{   char *line;


    if (el_line == NULL)
    {   if ((el_line = malloc((SIZE_T) MEM_INC)) == NULL)
            return (NULL);
        el_length = MEM_INC;
    }

    SetupTerminal();
    if (prompt)
        el_prompt = prompt;
    else
        el_prompt = "";
    bputs(el_prompt);
    if ((line = editinput()) != NULL)
    {   bputs("\r\n");
        bflush();
    }
    ResetTerminal();
    return (line);
}

static char *editinput()
{   int c;
    int status;


    el_arg = NO_ARG;
    el_oldpoint = el_point = el_mark = el_end = 0;
    el_line[0] = '\0';

    while (GET(c) != '\0')
    {   switch (status = SpecialCharacter(c))
        {   case GOT_EOF:
                return (NULL);
            case REPOSITION:
                reposition_dot();
            case DONTMOVE:
                continue;
            case NOT_SPECIAL:
                break;
        }

        if ((status = emacs(c)) == REPOSITION)
            reposition_dot();
        else if (status == GOT_LINE)
            break;
    }
    return (el_line);
}

static int emacs(c)
int c;
{   int status;


    status = (c < ' ') ? e_map[c].function() : self_insert_command(c); 

    if (el_pushedchar < 0)
        el_arg = NO_ARG;
    return (status);
}

static int self_insert_command(c)
int c;
{   char *p;
    int status = DONTMOVE;


    if (el_arg == NO_ARG)
    {   char str[2];


        str[0] = c;
        str[1] = '\0';
        status = insert_string(str);
    }
    else if ((p = malloc((SIZE_T) (el_arg+1))) != NULL)
    {   int i;


        for (i=0; i<el_arg; ++i)
            p[i] = c;
        p[el_arg] = '\0';
        status = insert_string(p);
        free(p);
    }

    return (status);
}

static int beginning_of_line()
{   if (el_point)
    {   el_point = 0;
        return (REPOSITION);
    }
    return (DONTMOVE);
}

static int backward_char()
{   int i=0;


    do
    {   if (el_point)
            left(REPOSITION);
        else
            break;
    }   while (++i < el_arg);

    return (DONTMOVE);
}

static int up_letter()
{   return (case_string((el_arg == NO_ARG) ? 1 : el_arg, TO_UPPER));
}

static int delete_char()
{   return (delete_string((el_arg == NO_ARG) ? 1 : el_arg));
}

static int end_of_line()
{   if (el_point != el_end)
    {   el_point = el_end;
        return (REPOSITION);
    }
    return (DONTMOVE);
}

static int forward_char()
{   int i=0;


    do
    {   if (el_point < el_end)
            right(REPOSITION);
        else
            break;
    }   while (++i < el_arg);
    return (DONTMOVE);
}

static int backward_delete_char()
{   int i=0;


    do
    {   if (el_point)
            left(REPOSITION);
        else
            break;
    }   while (++i < el_arg);

    return (delete_string(i));
}

static int c_complete()
{   char *str;
    char *word;
    int unique;


    str = complete(word = find_word(), &unique);
    if (word && *word)
        free(word);
    if (str && *str)
    {   int status = insert_string(str);


        if (!unique)
            (void) ring_bell();
        free(str);
        return (status);
    }
    return (ring_bell());
}

static int accept_line()
{   el_line[el_end] = '\0';
    return (GOT_LINE);
}

static int kill_line()
{   if (el_arg != NO_ARG)
    {   if (el_arg < el_point)
        {   int t = el_point;


            el_point = el_arg;
            reposition_dot();
            (void) delete_string(t - el_point);
        }
        else if (el_arg > el_point)
        {   right(REPOSITION);
            (void) delete_string(el_arg - el_point - 1);
        }
        return (REPOSITION);
    }

    save_yank(el_point, (SIZE_T) (el_end - el_point));
    el_line[el_point] = '\0';
    ceol();
    el_end = el_point;
    return (DONTMOVE);
}

static int redisplay_line()
{   bputs("\r\n");
    bputs(el_prompt);
    print_string(el_line);
    return (REPOSITION);
}

static int h_next()
{   return (do_history(next_history));
}

static int h_previous()
{   return (do_history(previous_history));
}

static int h_search()
{   char *str;
    char *old_prompt;
    static int in_search = 0;


    if (in_search)
        return (ring_bell());

    old_prompt = el_prompt;
    clear_line();
    ++in_search;
    bputs(el_prompt = "Search: ");
    str = search_history(editinput(), (el_arg == 0) ? 1: (-1));
    clear_line();
    el_prompt = old_prompt;
    bputs(el_prompt);
    --in_search;
    return(do_insert_history(str));
}

static int transpose_chars()
{   if (el_point)
    {   char c;


        if (el_point == el_end)
            left(REPOSITION);
        c = el_line[el_point - 1]; 
        left(DONTMOVE);
        el_line[el_point - 1] = el_line[el_point];
        output_char(el_line[el_point - 1]);
        el_line[el_point++] = c;
        output_char(c);
    }
    return (DONTMOVE);
}

static int quote()
{   int c;


    GET(c);
    return (self_insert_command(c));
}

static int wipe()
{   if (el_mark > el_end)
        return (ring_bell());

    if (el_point > el_mark)
    {   int t = el_point;


        el_point = el_mark;
        el_mark = t;
        reposition_dot();
    }

    return (delete_string(el_mark - el_point));
}

static int exchange_mark()
{   int c;


    GET(c);
    if (c != CTL('X'))
        return (ring_bell());

    if ((c = el_mark) <= el_end)
    {   el_mark = el_point;
        el_point = c;
        return (REPOSITION);
    }
    return (DONTMOVE); 
}

static int yank()
{   if (el_yank && *el_yank)
        return (insert_string(el_yank));
    return (DONTMOVE);
}

static int meta()
{   int c;
    int i;
    int in_keymap;
    int move = DONTMOVE;


    GET(c);
#ifdef ANSI_ARROWS
    if ((move = arrow(c)) != (-1))
        return (move);
#endif

    if (isdigit(c))
    {   el_arg = c - '0';
        while (GET(c) != '\0' && isdigit(c))
            el_arg = el_arg * 10 + c - '0';
        UNGET(c);
        return (DONTMOVE);
    }

    if (isupper(c))
        return (do_macro(c));
    i=0;
    in_keymap=0;
    el_oldpoint = el_point;
    while (meta_map[i].key_code)
    {   if (meta_map[i].key_code == c)
        {   in_keymap = 1;
            move = meta_map[i].function();
            break;
        }
        ++i;
    }

    if (!in_keymap)
        return (ring_bell());
    return (move);
}

static int move_to_char()
{   int c;
    int i;


    GET(c);
    for (i=el_point+1; i<el_end; i++)
        if (el_line[i] == c)
        {   el_point = i;
            return (REPOSITION);
        }
    return (DONTMOVE);
}

static int backward_kill_word()
{   (void) backward_word();
    if (el_oldpoint != el_point)
        return (delete_string(el_oldpoint - el_point));
    return (DONTMOVE);
}

static int set_mark()
{   el_mark = el_point;
    return (DONTMOVE);
}

static int last_argument()
{   char **argv;
    char *line;
    char *p;
    int argc;
    int status = DONTMOVE;


    if ((p = newest_history()) == NULL)
        return (ring_bell());

    if ((line = malloc(strlen(p)+1)) == NULL)
        return (DONTMOVE);
    else
       (void) memmove(line, p, strlen(p)+1);

    argc = mkargv(line, &argv);

    if (el_arg != NO_ARG)
    {   if (el_arg < argc)
            status = insert_string(argv[el_arg]);
        else
            status = ring_bell();
    }
    else if (argc)
        status = insert_string(argv[argc-1]);

    if (argc)
        free(argv);
    free(line);
    return (status);
}

static int h_old()
{   return (do_history(oldest_history));
}

static int h_new()
{   return (do_history(newest_history));
}

static int c_list_possible()
{   char **argp;
    char *word;
    int argc;


    argc = list_possible(word = find_word(), &argp);
    if (word && *word)
        free(word);
    if (argc)
    {   columns(argc, argp);
        while (argc--)
             free(argp[argc]);
        free(argp);
        return (REPOSITION);
    }
    return (ring_bell());
}

static int backward_word()
{   int i=0;


    do
    {   while (el_point && !isalnum(el_line[el_point-1]))
            left(REPOSITION);

        while (el_point &&
               el_line[el_point-1] != ' ' && isalnum(el_line[el_point-1]))
            left(REPOSITION);

        if (el_point == 0)
            break;
    }   while (++i < el_arg);

    return (DONTMOVE);
}

static int kill_word()
{   (void) do_forward(DONTMOVE);
    if (el_oldpoint != el_point)
    {   int nchars = el_point - el_oldpoint;


        el_point = el_oldpoint;
        return (delete_string(nchars));
    }
    return (DONTMOVE);
}

static int forward_word()
{   return (do_forward(REPOSITION));
}

static int case_down_word()
{   return (do_case(TO_LOWER));
}

static int case_up_word()
{   return (do_case(TO_UPPER));
}

static int version()
{   bputs("\r\n");
    bputs(_scid_);
    return (redisplay_line());
}

static int push_region()
{   if (el_mark > el_end)
        return (ring_bell());

    if (el_point > el_mark)
        save_yank(el_mark, (SIZE_T) (el_point - el_mark));
    else
        save_yank(el_point, (SIZE_T) (el_mark - el_point));

    return (DONTMOVE);
}

static int ring_bell()
{   bputc('\07');
    bflush();
    return (DONTMOVE);
}

static int insert_string(str)
char *str;
{   SIZE_T len = strlen(str);


/* BUG: we need to calculate length of str properly, and scroll display */
    if ((strlen(el_prompt) + el_end + len) > (SCREEN_WIDTH - 2))
        return (ring_bell());

    if ((el_end + len) >= el_length)
    {   char *ptr;


        if ((ptr = malloc(el_length + len + MEM_INC)) == NULL)
            return (DONTMOVE);
        if (el_length)
        {   (void) memmove(ptr, el_line, el_length);
            free(el_line);
        }
        el_line = ptr;
        el_length += (len + MEM_INC);
    }

    (void) memmove(&el_line[el_point+len],
                   &el_line[el_point], (SIZE_T) (el_end-el_point));
    (void) memmove(&el_line[el_point], str, len);
    el_end += len;
    el_line[el_end] = '\0';
    print_string(&el_line[el_point]);
    el_point += len;

    if (el_point == el_end)
        return (DONTMOVE);
    return (REPOSITION);
}

static int delete_string(nchars)
int nchars;
{   if (nchars > 0)
    {   if ((el_point+nchars) > el_end)
        {   if ((nchars = el_end - el_point) <= 0)
                return (DONTMOVE);
        }

        if (nchars > 1)
            save_yank(el_point, (SIZE_T) nchars);

        (void) memmove(&el_line[el_point],
                       &el_line[el_point+nchars],
                       (SIZE_T) (el_end-(el_point+nchars)+1));
        ceol();
        el_end -= nchars;
        print_string(&el_line[el_point]);
        return (REPOSITION);
    }
    return (ring_bell());
}
