
/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 */



/* Exports:
 *    AlexInfoLStat()      -   sort of like an lstat but uses a .alex.info
 *    AlexInfoInOpen()     -   Use macro ALEXINFOINOPEN
 *    AlexInfoInNext()     -   read lines from a .alex.info
 *    AlexInfoCompatStat() - silly old compatability routine
 *    AlexInfoMTime()      -   given Inode returns MTime if in cache
 */

#include "alexincs.h"
#include "alex.h"

#define MAXTOKENS  11
#define MAXTLEN   200 


int CallAlexOnDirForFile(FullPath, UidStr)
char *FullPath, *UidStr;
{
    char DirPath[MAXPATH];

    PathToDir(FullPath, DirPath);
 
    return(CheckFileCache(DirPath, UidStr));
}

int ClearActiveAlexInfo(AAIPtr)
struct ActiveAlexInfo *AAIPtr;
{
    if (AAIPtr == NULL) {
        Log("ClearActiveAlexInfo got a NULL pointer");
        return(AFAIL);
    }

    AAIPtr->File=NULL;                                          /* clear */
    AAIPtr->HasAlexError=0;
    AAIPtr->NeedAnUpdate=0;
    AAIPtr->PartsInHostName=0;
    AAIPtr->PartsInDirPath=0;
    AAIPtr->SizeOk=0;
    AAIPtr->Version=0;
    AAIPtr->UpdateDate=0;
    AAIPtr->NewestDate=0;

    return(AOK);
}


/* Recursion means calling CheckFileCache
 */
extern int AlexInfoInOpen(AlexInfoPath, AAIPtr, UidStr, RecursionOk, srcfile, srcline)
char *AlexInfoPath;
struct ActiveAlexInfo *AAIPtr;
char   *UidStr;
int    RecursionOk;
char   *srcfile;
int    srcline;
{
    struct stat buf;
    int Status;
    char *NameOfFileToOpen;

    Status=ClearActiveAlexInfo(AAIPtr);
    if (Status != AOK) {
        return(AFAIL);
    }
                                        /* /usr2/alex-cache/edu/berkeley/.alex.info is 4 */ 
    AAIPtr->PartsInDirPath = CountOccurrences(AlexInfoPath, '/') -CACHEDIRPARTS -1;
    if (AAIPtr->PartsInDirPath < 0) {
        AAIPtr->PartsInDirPath = 0;
    }
    LogN("AlexInfoInOpen PartsInDirPath ", AAIPtr->PartsInDirPath);

                          /* ROOTALEXINFOSTR could be something like /usr2/.alex.info  */
    if (streql(AlexInfoPath, ROOTALEXINFOSTR) || streql(AlexInfoPath,    "/.alex.info")) {
        Log2("AlexInfoInOpen had ", AlexInfoPath);
        NameOfFileToOpen=ROOTALEXINFO;
        AAIPtr->NeedAnUpdate = -1;            /* never try to update root */
    } else {
        NameOfFileToOpen=AlexInfoPath;
    } 

    Log2("AlexInfoInOpen with ", NameOfFileToOpen);

    (void) strcpy(AAIPtr->OpenAlexInfoPath,  NameOfFileToOpen);


#ifdef DEVNULLBROKEN                              /* just define this if /dev/null does not work */
    if (streql(AlexInfoPath, "/dev/null")) {
        Log("AlexInfoInOpen with /dev/null done ");
        return(AOK);                              /* reads will fail which is all we need */
    }
#endif

    if ((lstat(NameOfFileToOpen, &buf) != 0) ||                          /* try lstat */
                 (RecursionOk && (buf.st_mtime < MINTIMEOKALEXINFO))) {  /* code to cleanup bad .alex.infos */
        Log2("\n\n AlexInfoInOpen did not find this .alex.info ", NameOfFileToOpen);
        if (!RecursionOk) {
            return(AFAIL);
        } else {
            Status=CallAlexOnDirForFile(NameOfFileToOpen, UidStr);  /* ends in /.alex.info */
            if (Status==AWORKING) {
                LogT("AlexInfoInOpen done ", Status);
                return(Status);
            }
            if (lstat(NameOfFileToOpen, &buf) != 0) {           /* try lstat again */
                return(AFAIL);
            } else {
                Log2("\n\n AlexInfoInOpen got Alex to make ", NameOfFileToOpen);
            }
        }
    }

    Log2("AlexInfoInOpen has good stat for ", NameOfFileToOpen);

    AAIPtr->File=afopen(NameOfFileToOpen,"r", srcfile, srcline);  /* caller resposible for close */
    if (AAIPtr->File==NULL) {
        Log2("AlexInfoInOpen could not open ", NameOfFileToOpen);
        return(AFAIL);
    }

    AAIPtr->Version=ALEXINFOVERSION;         /* default to current version */
    Log("AlexInfoInOpen done AOK");
    return(AOK);
}

/* for unfsd we sort of emulating opendir
 */
extern int AlexOpenDir(DirPath, AAIPtr, UidStr)
char *DirPath;
struct ActiveAlexInfo *AAIPtr;
char *UidStr;
{
    int Result;

    char AlexInfoPath[MAXPATH];

    (void) strcpy(AlexInfoPath, DirPath);
    (void) SimplifyPath(AlexInfoPath);
    if (streql(AlexInfoPath, "/")) {
        (void) strcat(AlexInfoPath, ALEXINFO);
    } else {
        (void) strcat(AlexInfoPath, SLASHALEXINFO);
    }

    Result=ALEXINFOINOPEN(AlexInfoPath, AAIPtr, UidStr, RECURSIONOK);

    LogT("AlexOpenDir done ", Result); 
    return(Result);
}

extern int AlexInfoInClose(AAIPtr)
struct ActiveAlexInfo *AAIPtr;
{
    int Result;

    if (AAIPtr->File == NULL) {
        Result=AOK;                       /* might be done by AlexInfoInNext in error cases */
    } else {
        if (AFCLOSE(AAIPtr->File) == 0) {
            Result=AOK;
        } else {
            Result=AFAIL;
        }
        AAIPtr->File=NULL;
    }
   
    LogT("AlexInfoInClose returning ", Result); 
    return(Result);
}


#define MAXTOKENINPARSEDLINE 6

int ParseOneLine(Line, Current, Version)
char *Line;
struct ParsedDir *Current;
int Version;
{
    char Tokens[MAXTOKENS][MAXTLEN];
    int NumTokens, NumScanned, NoLinkSize, T;
    int Result, GoodType, ShouldHaveSymLink;

    NumTokens=LineToTokens(Line, Tokens);


    if (NumTokens < (MAXTOKENINPARSEDLINE -1)) {
        ToLog(DBERROR, "ParseOneLine ERROR BUG need more than %d tokens, but input is %s\n", 
                        NumTokens, Line);
        return(AFAIL);
    }

    switch (Version) {
       case 3:
       case 4:                       /* version 3 and 4 are the same in data part */
            NumScanned=1;  (void) strcpy(Current->Name, Tokens[0]);
            if (StringToUnsigned(Tokens[1],  (unsigned *) &Current->Type) == AOK) NumScanned++;
            if (StringToUnsigned(Tokens[2],  &Current->Size) == AOK) NumScanned++;
            if (StringToUnsigned(Tokens[3],  &Current->Date) == AOK) NumScanned++;
            if (StringToUnsigned(Tokens[4],  &Current->Inode) == AOK) NumScanned++;
            if (NumTokens==MAXTOKENINPARSEDLINE) {
                 NumScanned ++; (void) strcpy(Current->SymLink, Tokens[MAXTOKENINPARSEDLINE -1]);
#ifdef EXECUTABLE
                /* if the symlink is "-1", it's faked and we should mark the file executable */
                if (streql(Current->SymLink,"-1")) {
                     Current->Executable = 1;
                     strcpy(Current->SymLink,"");
                     --NumScanned;
                }
#endif
            }
            NoLinkSize=MAXTOKENINPARSEDLINE -1;
            break;

       default:
            ToLog(DBERROR, "ParseOneLine ERROR BUG unknown Version %d\n", Version);
            return(AFAIL);
    }

    T=Current->Type;
    ShouldHaveSymLink = (T==ALINK) && (T=HOSTALIAS);
    GoodType = (T==AFILE) || (T==ADIR) || (T==ALINK) || (T==AHOST) || 
               (T==HOSTALIAS) || (T==AERRORMESSAGE) || (T==ADOMAIN) ;

    if (GoodType &&                                                         /* if type is ok    */
        ( ((NumScanned==NoLinkSize)   && (!ShouldHaveSymLink)) ||           /* and right # tokens */
          ((NumScanned==NoLinkSize+1) &&  (ShouldHaveSymLink))   )  ) {
        Result=AOK;                                                         /* then all is well  */
    } else {
        ToLog(DBERROR, "ParseOneLine ERROR BUG NumTokens=%d NumScanned= %d, Type= %s\n", 
                       NumTokens, NumScanned, ATypeToString(Current->Type));
        Result=AFAIL;
    }

    return(Result);
}


struct DirCacheEntry {
    unsigned int Date;           /* MTime */
    int    Inode;  
};

#define DIRINODECACHESIZE 50000

struct DirCacheEntry DirInodeCache[DIRINODECACHESIZE];


extern unsigned int AlexInfoMTime(Inode)
unsigned int Inode;
{
   unsigned int MTime;
   int Index;

   Index = Inode % DIRINODECACHESIZE;

   if (DirInodeCache[Index].Inode == Inode) {
       MTime = DirInodeCache[Index].Date;
       LogN("AlexInfoMTime hit on ", Inode);
   } else {
       MTime = 0;
   }

   return(MTime);
}



/*  Read next directory item from AAIPtr 
 *  Returns:
 *        AOK   - fine
 *        AEOF  - end of file
 *        AFAIL - did not work.
 *         
 */
extern int AlexInfoInNext(AAIPtr, Current)
struct ActiveAlexInfo *AAIPtr;
struct ParsedDir *Current;
{
    char line[MAXPATH+MAXPATH];
    int NeedALine, Status, Result, Index;
    double Now, Age, OkSlop, SecsSinceCheck;

    if (AAIPtr->File == NULL) {
        Log("AlexInfoInNext ERROR called with no open file");
        return(AFAIL);
    }

    ClearParsedDirEntry(Current);
    line[0]=0;
    NeedALine=1;
    Result=AFAIL;
    while (NeedALine) {
        NeedALine=0;
        if(fgets(line, MAXPATH+MAXPATH, AAIPtr->File ) == NULL) {
            return(AEOF);                                           /* that all folks */
        }

        if (line[0] == METACHAR) {
            NeedALine=1;                                            /* not a data line */
            Status=AOK;
            switch (line[1]) {
                case VERSCHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->Version)); 
                    break; 

                case DATECHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->UpdateDate)); 
                    break; 

                case NEWCHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->NewestDate)); 
                    break; 

                case SIZECHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->SizeOk)); 
                    break; 

                case HOSTPARTSCHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->PartsInHostName)); 
                    break; 

                case ERRORCHAR:
                    AAIPtr->HasAlexError = 1; 
                    break; 

                case UPDATECHAR:
                    if (AAIPtr->NeedAnUpdate >= 0) AAIPtr->NeedAnUpdate = 1; 
                    break; 

                default:
                    Status=AFAIL;
                    ToLog(DBERROR, "AlexInfoInNext ERROR BUG unknown meta-data type %c\n", line[1]);
            }
            
            if (Status != AOK) {
                ToLog(DBERROR, "AlexInfoInNext ERROR BUG meta-data problems Status= %s\n", 
                         ATypeToString(Status));
            }
                
        } else if ((strncmp(line, OLDVERSIONTOKEN, strlen(OLDVERSIONTOKEN)) == 0) && 
             (CountOccurrences(line, ' ') == 1)) {                   /* one ping only  */
            NeedALine=1;                                            /* in either case ignore */
            if (StringToUnsigned(&line[strlen(OLDVERSIONTOKEN)], 
                                   (unsigned *) &(AAIPtr->Version)) != AOK) {
                ToLog(DBERROR, "AlexInfoInNext ERROR BUG StringToUnsigned of version %s\n", line);
            }
        }
    }
  

    Result=ParseOneLine(line, Current, AAIPtr->Version); 
    if (Result != AOK) {
        Current->Stale = 1;                            /* all are stale         */
        AAIPtr->HasAlexError=1;                        /* there is an error  */   
        AAIPtr->NeedAnUpdate=1;                        /* there is an error  */   
        ToLog(DBERROR, "AlexInfoInNext ERROR BUG could not read V= %d Name= %s Line= %s\n", 
                                AAIPtr->Version, Current->Name, line);
        return(Result);
    } 

    switch (Current->Type) {
        case ADIR:    if (Current->Name[0] != '.') {
                          Index = Current->Inode % DIRINODECACHESIZE;
                          DirInodeCache[Index].Inode = Current->Inode;
                          DirInodeCache[Index].Date  = Current->Date;  
                      }
                      /* fall through - no break */
        case AERRORMESSAGE:
        case AFILE:   Current->PartsInHostName = AAIPtr->PartsInHostName; 
                      break;

        case ADOMAIN: Current->PartsInHostName = -1;
                      break;

        case ALINK:
        case HOSTALIAS:
        case AHOST:   Current->PartsInHostName = AAIPtr->PartsInDirPath+1;
                      break;

        default:      LogT("AlexInfoInNext ERROR BUG funny type when getting PartsInHostName", Current->Type);
    }

    Current->Stale = 0;                            /* not known to be stale so far */
    if (AAIPtr->NeedAnUpdate >=0) {
        /* Now = TimeInSeconds(); */             /* do not need exact time */
        Now = RecentTime;
        /* Age = Now - Current->Date; */         /* real for this file */
        Age = Now - AAIPtr->NewestDate;          /* but we are doing it on a directory level now XXXX */
        if (Age < 60) {
            Age=60;
        }

        SecsSinceCheck= Now - AAIPtr->UpdateDate;   /* time since since we looked */

        if (SecsSinceCheck < 0) {
            if (SecsSinceCheck > -(5 * AMINUTE)) {     /* overlook a small oddity :-) */
                SecsSinceCheck=0;
            } else {
                ToLog(DBERROR, "AlexInfoInNext ERROR BUG in future SSC= %d Now= %d UD= %d Age= %d", 
                         (int) SecsSinceCheck, (int) Now, (int) AAIPtr->UpdateDate, (int) Age);
                SecsSinceCheck=1000;
                Current->Stale = 1;                       /* an update might help (or loop) */
            }
        }
  
        OkSlop = Age / OKINCONSISTENCY;  /* reasonable inconsistency of 10 percent */
        if (OkSlop < MINCONSSLOP) {
            OkSlop = MINCONSSLOP;
        }

        if ((Current->Type == AERRORMESSAGE) || (AAIPtr->HasAlexError)) {
            AAIPtr->HasAlexError=1;                        /* there is an error      */
            if ((SecsSinceCheck > SECSTILLRETRY) &&
                ((Current->Stale != 1) || (AAIPtr->NeedAnUpdate != 1)))  {
                if (SecsSinceWrite(AAIPtr->OpenAlexInfoPath) > SECSTILLRETRY) {
                    Current->Stale = 1;                                /* really astale XXXX  */
                    AAIPtr->NeedAnUpdate=1;                         /* time for a retry    */
                    ToLog(DBALL, "KnownStaleDir %s\n", AAIPtr->OpenAlexInfoPath);
                }
            }
        } else {
            if ((SecsSinceCheck > OkSlop) || (SecsSinceCheck > MAXSECSSINCECHECK)) {
                Current->Stale = 1;                            /* not within spec for consistent */
                AAIPtr->NeedAnUpdate=1;                        /* directory level now XXXXX */
            } 
        }
    }

/* Something like the following does not work because it would only cause the directory to be 
 *  updated and not for the host alais to be checked.
 *
 *   if ((Current->Type == HOSTALIAS) && (SecsSinceCheck > TRUSTHOSTALIAS)) {
 *       Current->Stale=1;
 *   }
 */

    if (AAIPtr->NeedAnUpdate > 0) {
        Current->Stale = 1;             /* all are stale */
    }

    if (AAIPtr->UpdateDate == 0) {    /* if "!D 0" in .alex.info never change - for RootAlexInfo */
        AAIPtr->NeedAnUpdate = 0;
        Current->Stale = 0;            
    }


    /* LogN("AlexInfoInNext sees SecsSinceCheck ", (int) SecsSinceCheck);
     * LogN("AlexInfoInNext sees OkSlop ", (int) OkSlop); 
     */

                                         /* if Stale we would like to return AUPDATE */
                                         /* problem is DirToInfo may be calling us on old things */
    return(Result);
}


extern int SimpleType(Type)
int Type;
{
    int Result;

    switch(Type) {
        case ADOMAIN:
        case AHOST:
        case ADIR:   Result=ADIR; 
                     break;

        case AERRORMESSAGE:
        case AFILE:  Result= AFILE; 
                     break;

        case HOSTALIAS:
        case ALINK:  Result= ALINK;
                     break;

        default:     Result=Type;     
                     break;
   }
   return(Result);
}

#ifdef EXECUTABLE
int ATypeToStatMode(AType,IsExecutable)
int AType;
int IsExecutable;
#else
int ATypeToStatMode(AType)
int AType;
#endif
{
    int Result, Type;

   /* Log2("ATypeToStatMode is called with ", ATypeToString(AType)); */

    Type=SimpleType(AType);

    switch (Type) {
        case ADIR:   Result= S_IFDIR + 0755;
                     break;

        case AFILE:
#ifdef EXECUTABLE
                     if (IsExecutable) {
                         Result= S_IFREG + 0755;
                     } else {
#endif
                         Result= S_IFREG + 0644;
#ifdef EXECUTABLE
                     }
#endif
                     break;

        case ALINK:  Result= S_IFLNK + 0777;
                     break;

        default:     ToLog(DBERROR, "ATypeToStatMode ERROR BUG %d\n", AType);
                     Result= S_IFDIR + 0444;
    }

    return(Result);
}

StandardStat(StatBuf)
struct stat *StatBuf;
{
    StatBuf->st_dev=1;
    StatBuf->st_nlink=1;
    StatBuf->st_rdev=1; 
    StatBuf->st_uid=ALEXUIDVAR; 
    StatBuf->st_gid=ALEXGIDVAR;
    StatBuf->st_blksize=1024; 
/*    StatBuf->st_spare1=0;
    StatBuf->st_spare2=0;
    StatBuf->st_spare3=0;
    StatBuf->st_spare4[0]=0;
    StatBuf->st_spare4[1]=0;
 */
}

int ParsedDirToStat(Current, StatBuf)
struct ParsedDir *Current;
struct stat *StatBuf;
{


    if (Current->Inode != 0) {
        StatBuf->st_ino=Current->Inode;
    } else {
        ToLog(DBERROR, "ParsedDirToStat ERROR BUG why does this happen? %d\n", (int) InodeNext);
        StatBuf->st_ino=InodeNext++;
    }

#ifdef EXECUTABLE
    StatBuf->st_mode=ATypeToStatMode(Current->Type,Current->Executable);
#else
    StatBuf->st_mode=ATypeToStatMode(Current->Type);
#endif

    StatBuf->st_size=Current->Size; 
    StatBuf->st_blocks= 1 + (Current->Size / 512);

    StatBuf->st_atime=Current->Date;
    StatBuf->st_mtime=Current->Date;
    StatBuf->st_ctime=Current->Date;

    StandardStat(StatBuf);
}



/* Currently the StatCache module does not help and is not used.
 * It causes many copies and strcmp()s.
 * Wish AlexInfoLStat was given Inode number when known, we could compare 
 * much faster with that and get rid of the strcmp time at least.        XXXXX  */

#ifdef USESTATCACHE
/************************* StatCache module  ***************************
 *   This module only the following entries:
 *      ClearStatCache()
 *      AddToStatCache()
 *      CheckStatCache()
 */

#define STATCACHESIZE 2       /* We need at least 2 */

struct StatCacheEntry {
    struct ParsedDir Current;
    char Path[MAXPATH];
} StatCache[STATCACHESIZE];

int    StatCacheNextIndex=0;                            /* where next will go */
int    StatCacheNumInCache=0;

/*****/

ClearStatCache()
{
    StatCacheNumInCache=0;
    StatCacheNextIndex=0;
}

AddToStatCache(Path, PtrCurrent)
char *Path;
struct ParsedDir *PtrCurrent;
{
    (void) strcpy(StatCache[StatCacheNextIndex].Path, Path);
    CopyParsedDir(PtrCurrent, &StatCache[StatCacheNextIndex].Current);

    StatCacheNextIndex = (StatCacheNextIndex + 1) % STATCACHESIZE;
    if (StatCacheNumInCache < STATCACHESIZE) {
        StatCacheNumInCache++;
    }
}

int CheckStatCache(Path, PtrCurrent)
char *Path;
struct ParsedDir *PtrCurrent;
{
    int i, MatchStatus;

    MatchStatus = AFAIL;
    for (i=0; MatchStatus == AFAIL && i<STATCACHESIZE && i<StatCacheNumInCache; i++) {
        if (streql(StatCache[i].Path, Path)) {
           MatchStatus = AOK;
           CopyParsedDir(&StatCache[i].Current, PtrCurrent);
        }
    }
    return(MatchStatus);
}

/************************* End of StatCache module  ***************************/
#endif


int InvalidateStatCache;                           /* outsiders can change this */

/*  It is important that for the case where stats are in order that
 *  we end up only reading the .alex.info once.  Cache open file
 *  and position in that file.
 *
 *  Recursion means calling Alex
 *
 *  Given a "FullPathArg" of /usr2/alexsrvr/alex-cache/edu/cmu/cs
 *  This looks up "cs" in /usr2/alexsrvr/alex-cache/edu/cmu/.alex.info
 *
 *  RecursionOk means we can call CheckFileCache 
 *
 */
extern int AlexInfoLStat(FullPathArg, Result, UidStr, RecursionOk)
char *FullPathArg;
struct ParsedDir *Result;
char   *UidStr;
int    RecursionOk;
{
    char   FullPath[MAXPATH];
    static char CurrentOpenAlexInfo[MAXPATH];
    static char KnownBadAlexInfo[MAXPATH];
    static struct ActiveAlexInfo SAAI;              /* Static AAI           */
    struct ActiveAlexInfo AAI;                      /* our own AAI           */
    static int SomethingOpen=0;
    static struct ParsedDir Current;
    static int Status=AFAIL;                        /* status of Current now */
    int TmpStatus;
    char AlexInfoPath[MAXPATH], FileName[MAXPATH];
    struct stat s2;                                 

    AlexAssert(FullPathArg != NULL);
    AlexAssert(FullPathArg[0] != 0);

    strcpy(FullPath, FullPathArg);
    (void) SimplifyPath(FullPath);

    PathToDir(FullPathArg, AlexInfoPath);
    (void) SimplifyPath(AlexInfoPath);
    (void) strcat(AlexInfoPath, SLASHALEXINFO);

    Log2("AlexInfoLStat with ", AlexInfoPath);
    LogN(UidStr, RecursionOk);

    (void) PathToFileName(FullPath, FileName);

    if ((!InvalidateStatCache) && (streql(AlexInfoPath, KnownBadAlexInfo))) {
        Log2("AlexInfoLStat  using info failure cache ", FullPath);
        return(AFAIL);
    } 

    if (!InvalidateStatCache) {
#ifdef USESTATCACHE
        if (CheckStatCache(FullPath, Result) == AOK) {            /* note copies to Result if AOK */
            Log2("AlexInfoLStat got a cache hit ", FullPath);
            return(AOK);
        }
#else   
        if ((Status == AOK) &&  streql(AlexInfoPath, CurrentOpenAlexInfo) &&
            (streql(FileName, Current.Name)) && !Current.Stale && SomethingOpen) {    
            Log2("AlexInfoLStat cache hit", FullPath);
            CopyParsedDir(&Current, Result);
            return(Status);
        }
#endif
    } else { /* InvalidateStatCache */
#ifdef USESTATCACHE
        ClearStatCache();
#endif

        if (SomethingOpen) {
            (void) AlexInfoInClose(&SAAI);
            SomethingOpen=0;
        }
        KnownBadAlexInfo[0]=0;
        CurrentOpenAlexInfo[0]=0;

        InvalidateStatCache=0;                  /* we are making a new entry not using cache */
    }

    if (FileName[0] == 0) {
        Log("AlexInfoLStat NULL filename - adding .");
        strcat(FileName, ".");
    }

    (void) ClearParsedDirEntry(&Current);

    Status=AOK;
    if ((!SomethingOpen) || (!streql(AlexInfoPath, CurrentOpenAlexInfo))) {
        CurrentOpenAlexInfo[0]=0;
        if (Status == AOK) {   
            Status=ALEXINFOINOPEN(AlexInfoPath, &AAI, UidStr, RecursionOk);
            if (Status == AWORKING) {                                        /* do the open */
                return(Status);
            }
            if (Status != AOK) {                      
                (void) strcpy(KnownBadAlexInfo, AlexInfoPath);
                (void) AlexInfoInClose(&AAI);                                /* should not need  */
                Status=AFAIL;
            } else { 
                if (SomethingOpen) {
                    Status=AlexInfoInClose(&SAAI);                           
                    SomethingOpen=0;
                }
                Status=AOK;
                SAAI=AAI;                                                    /* copy to static   */
                SomethingOpen=1;                                             /* new active one */
                (void) strcpy(CurrentOpenAlexInfo, AlexInfoPath);
            }
        }
    }

    if ((Status != AOK) && (RecursionOk)) {
        return(AFAIL);
    }


    if (Status==AOK) {                                   /* if AOK then AAI has right file open */
        Status=AlexInfoInNext(&SAAI, &Current);                           /* first try next item */
        if ((Status != AOK) || (!streql(Current.Name, FileName))) {
            Log2("AlexInfoLStat going back to begining of file looking for ", FileName);
            if (fseek(SAAI.File, (long) 0, 0)) {
                Log2("AlexInfoLStat ERROR fseek failed", FileName);
                (void) AlexInfoInClose(&SAAI);
                SomethingOpen=0;
                Status=AFAIL;                                              /* going back failed  */
            } else {
                Status=AOK;
                while((Status==AOK) && !streql(Current.Name, FileName)) {  /* goes at least once */
                    Status=AlexInfoInNext(&SAAI, &Current);                /* get new Current    */
                }
            }
        }
    }

    if ((Status != AOK) || !streql(Current.Name, FileName)) {      /* if still not there */
        Log2("AlexInfoLStat found No such file as ", FileName);    /* no such file */
        Result->PartsInHostName=SAAI.PartsInHostName;              /* what host we checked - alex.c uses */
        (void) AlexInfoInClose(&SAAI);
        SomethingOpen=0;
        if ( RecursionOk && Current.Stale ) {
            Status=AFAIL;                                          /* will cause to CheckFileCache() */
        } else {
            Status=NOSUCHFILE;
        }
        LogT("AlexInfoLStat ", Status);
        CopyParsedDir(&Current, Result);                           /* might want to know if was stale */
        return(Status);
    }

    /* At this point Status == AOK */

    if ( ! SAAI.SizeOk) {                                           /* if could not parse sizes */
        if (lstat(FullPath, &s2) == 0) {                            /* if we have file in cache */
            Current.Size=s2.st_size;                                /* use size of one in cache */
        }
        Current.SizeTrusted=0;
    } else {
        Current.SizeTrusted=1;                                 /* size comes from good .alex.dir */
    }

              

    if ((Status==AOK) && (Current.SizeTrusted) && (!Current.Stale)) {
#ifdef USESTATCACHE
        AddToStatCache(FullPath, &Current);
#endif
    } else {
        if ((Status != AWORKING) && RecursionOk && Current.Stale) {
             Status=AFAIL;                  /* really AOK, but just so we don't use Current as cache */
             TmpStatus=CallAlexOnDirForFile(FullPath, UidStr);  
             Status=AFAIL;                  /* really AOK, but just so we don't use Current as cache */
             if ((TmpStatus != NOSUCHFILE) && (TmpStatus != AFAIL)) {
                 return(ARETRY);            /* make caller retry */
             } else {
                 return(AFAIL);
             }
        }
        Status=AFAIL;        /* really AOK, but just so we don't use Current as cache */
    } 

    CopyParsedDir(&Current, Result);       

    ToLog(DBALL, "AlexInfoLStat %s %s %d\n", FullPath, ATypeToString(Current.Type), Current.Inode); 
    return(AOK);
}



/* For compatability we fill in a regular stat struct 
 * For incompatability we use Alex types not regular return codes for lstat 
 *
 * Recursion means calling Alex
 */
extern int AlexInfoCompatStat(FullPath, StatBuf, Current, UidStr, RecursionOk)
char        *FullPath;
struct stat *StatBuf;
struct ParsedDir *Current;
char          *UidStr;
int          RecursionOk;
{
    int  Status;
    struct ParsedDir CurrentBuf;
    char LocalPath[MAXPATH], LastOfPath[MAXPATH];

    Log2("AlexInfoCompatStat", FullPath);

    if (UidStr==NULL) {
        UidStr=LastUidStr;
    }

    if (Current == NULL) {
        Current = &CurrentBuf;
    }

    (void) strcpy(LocalPath, FullPath);
    Status=SimplifyPath(LocalPath);
    PathToFileName(LocalPath, LastOfPath);

    if (streql(LastOfPath, ALEXUPDATE)) {
        if (!RecursionOk) {
            ToLog(DBERROR, "AlexInfoCompatStat ERROR BUG recursion needed for .alex.update %s\n",
                                                             LocalPath);
        }
        Status=CheckFileCache(LocalPath, UidStr);            /* don't need DirForFile since .alex.update */
        if (Status == AWORKING) {
            return(Status);
        } else {
            return(AFAIL);
        }
    }

    Status=AlexInfoLStat(LocalPath, Current, UidStr, RecursionOk);
    if (Status == ARETRY) {
        Log("AlexInfoCompatStat doing one retry");
        Status=AlexInfoLStat(LocalPath, Current, UidStr, RecursionOk);
        if (Status == ARETRY) {
            Status = AWORKING;
        }
    }
                 /* if recursion is Ok and helps AlexInfoLStat will do it so we don't need to here */
    if (Status == AOK) {
        LogN("AlexInfoCompatStat has current inode ", (int) Current->Inode);
        if ((debugLog != NULL) && (DEBUGLEVEL >= 10)) {
            OneLineOut(debugLog, Current);
        }
        (void) ParsedDirToStat(Current, StatBuf);
        LogN("AlexInfoCompatStat has stat inode ", (int) StatBuf->st_ino);
        errno=0;
    } else {
        ToLog(DBALL, "AlexInfoCompatStat returning %s for %s\n", ATypeToString(Status), FullPath);
    }
    
    return(Status);
}



