/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1986. */

/* generic routines for B texts, lists and tables */

#include "b.h"
#include "bint.h"
#include "feat.h"
#include "bobj.h"
#include "i1tlt.h"

#define SIZE_TLT	MESS(300, "in #t, t is not a text list or table")

#define SIZE2_TLT	MESS(301, "in e#t, t is not a text list or table")
#define SIZE2_CHAR	MESS(302, "in e#t, t is a text, but e is not a character")

#define MIN_TLT		MESS(303, "in min t, t is not a text list or table")
#define MIN_EMPTY	MESS(304, "in min t, t is empty")

#define MAX_TLT		MESS(305, "in max t, t is not a text list or table")
#define MAX_EMPTY	MESS(306, "in max t, t is empty")

#define MIN2_TLT	MESS(307, "in e min t, t is not a text list or table")
#define MIN2_EMPTY	MESS(308, "in e min t, t is empty")
#define MIN2_CHAR	MESS(309, "in e min t, t is a text, but e is not a character")
#define MIN2_ELEM	MESS(310, "in e min t, no element of t exceeds e")

#define MAX2_TLT	MESS(311, "in e max t, t is not a text list or table")
#define MAX2_EMPTY	MESS(312, "in e max t, t is empty")
#define MAX2_CHAR	MESS(313, "in e max t, t is a text, but e is not a character")
#define MAX2_ELEM	MESS(314, "in e max t, no element of t is less than e")

#define ITEM_TLT	MESS(315, "in t item n, t is not a text list or table")
#define ITEM_EMPTY	MESS(316, "in t item n, t is empty")
#define ITEM_NUM	MESS(317, "in t item n, n is not a number")
#define ITEM_INT	MESS(318, "in t item n, n is not an integer")
#define ITEM_L_BND	MESS(319, "in t item n, n is < 1")
#define ITEM_U_BND	MESS(320, "in t item n, n exceeds #t")

#ifdef B_COMPAT

#define THOF_TLT	MESS(321, "in n th'of t, t is not a text list or table")
#define THOF_EMPTY	MESS(322, "in n th'of t, t is empty")
#define THOF_NUM	MESS(323, "in n th'of t, n is not a number")
#define THOF_INT	MESS(324, "in n th'of t, n is not an integer")
#define THOF_L_BND	MESS(325, "in n th'of t, n is < 1")
#define THOF_U_BND	MESS(326, "in n th'of t, n exceeds #t")

#endif /* B_COMPAT */

extern bool comp_ok;

Visible value mk_elt() { return grab(ELT, 0); }

Visible value size(x) value x; { /* monadic # operator */
	intlet n= 0;
	if (Is_range(x))
		return rangesize(Lwb(x), Upb(x));
	else if (!Is_tlt(x)) 
		interr(SIZE_TLT);
	else
		n= Length(x);
	return mk_integer((int) n);
}

#define Lisent(tp,k) (*(tp+(k)))

Visible value size2(v, t) value v, t; { /* Dyadic # operator */
	intlet len, n= 0, k; value *tp= Ats(t);
	if (!Is_tlt(t)) {
		interr(SIZE2_TLT);
		return mk_integer((int) n);
	}
	len= Length(t);
	switch (Type(t)) {
	case Tex:
		{string cp= (string)tp; char c;
			if (Type(v) != Tex || Length(v) != 1)
				interr(SIZE2_CHAR);
			else {
				c= *Str(v);
				for (k= 0; k < len; k++) if (*cp++ == c) n++;
			}
		} break;
	case ELT:
		break;
	case Lis:
		{intlet lo= -1, mi, xx, mm, hi= len; relation c;
		bins:	if (hi-lo < 2) break;
			mi= (lo+hi)/2;
			if ((c= compare(v, Lisent(tp,mi))) == 0) goto some;
			if (!comp_ok) break;
			if (c < 0) hi= mi; else lo= mi;
			goto bins;
		some:	xx= mi;
			while (xx-lo > 1) {
				mm= (lo+xx)/2;
				if (compare(v, Lisent(tp,mm)) == 0) xx= mm;
				else lo= mm;
			}
			xx= mi;
			while (hi-xx > 1) {
				mm= (xx+hi)/2;
				if (compare(v, Lisent(tp,mm)) == 0) xx= mm;
				else hi= mm;
			}
			n= hi-lo-1;
		} break;
	case Ran:
		if (compare(Lwb(t), v) <= 0
		    &&
		    comp_ok
		    &&
		    compare(v, Upb(t)) <= 0
		)
			n= 1;
		else
			n= 0;
		break;
	case Tab:
		for (k= 0; k < len; k++) {
			if (compare(v, Dts(*tp++)) == 0) n++;
			if (!comp_ok) { n= 0; break; }
		}
		break;
	default:
		syserr(MESS(327, "size2() on non tlt value"));
		break;
	}
	return mk_integer((int) n);
}

Hidden bool less(r) relation r;    { return r<0; }
Hidden bool greater(r) relation r; { return r>0; }

Hidden value mm1(t, rel) value t; bool (*rel)(); {
	intlet len= Length(t), k; value m, *tp= Ats(t);
	switch (Type(t)) {
	case Tex:
		{string cp= (string) tp; char mc= '\0', mm[2];
			for (k= 0; k < len; k++) {
				if (mc == '\0' || ((*rel)(*cp < mc ? -1 : (*cp > mc ? 1 : 0))))
					mc= *cp;
				cp++;
			}
			mm[0]= mc; mm[1]= '\0';
			m= mk_text(mm);
		} break;
	case Lis:
		if ((*rel)(-1)) /*min*/ m= copy(*Ats(t));
		else m= copy(*(Ats(t)+len-1));
		break;
	case Ran:
		if ((*rel)(-1)) /*min*/ m= copy(Lwb(t));
		else m= copy(Upb(t));
		break;
	case Tab:
		{value dm= Vnil;
			for (k= 0; k < len; k++) {
				if (dm == Vnil)
					dm= Dts(*tp);
				else {
					relation c= compare(Dts(*tp), dm);
					if (!comp_ok) 
						return Vnil;
					if ((*rel)(c))
						dm= Dts(*tp);
				}
				tp++;
			}
			m= copy(dm);
		} break;
	default:
		syserr(MESS(328, "mm1() on non tlt value"));
	}
	return m;
}

Hidden value mm2(v, t, rel) value v, t; bool (*rel)(); {
	intlet len= Length(t), k; value m= Vnil, *tp= Ats(t);
	switch (Type(t)) {
	case Tex:
		{string cp= (string) tp; char c, mc= '\0', mm[2];
			c= *Str(v);
			for (k= 0; k < len; k++) {
				if ((*rel)(c < *cp ? -1 : c > *cp ? 1 : 0)) {
					if (mc == '\0' || (*rel)(*cp < mc ? -1 : *cp>mc ? 1 : 0))
						mc= *cp;
				}
				cp++;
			}
			if (mc != '\0') {
				mm[0]= mc; mm[1]= '\0';
				m= mk_text(mm);
			}
		} break;
	case Lis:
		{intlet lim1, mid, lim2; relation c;
			if ((*rel)(-1)) { /*min*/
				lim1= 0; lim2= len-1;
			} else {
				lim2= 0; lim1= len-1;
			}
			c= compare(v, Lisent(tp, lim2));
			if (!comp_ok) return Vnil;
			if (!(*rel)(c)) break;
			if (len == 1 || (*rel)(compare(v, Lisent(tp,lim1)))) {
				m= copy(Lisent(tp,lim1));
				break;
			}
			/* v rel tp[lim2] && !(v rel tp[lim1]) */
			while (abs(lim2-lim1) > 1) {
				mid= (lim1+lim2)/2;
				if ((*rel)(compare(v, Lisent(tp,mid)))) lim2= mid;
				else lim1= mid;
			}
			m= copy(Lisent(tp,lim2));
		} break;
	case Ran:
		{relation c= compare(v, Lwb(t));
			if (!comp_ok)
				return Vnil;
			if ((*rel)(-1)) {
				/* min2 */
				if (c < 0)
					m= copy(Lwb(t));
				else if (compare(v, Upb(t)) < 0) {
					if (integral(v))
						m= sum(v, one);
					else
						m= ceilf(v);
				}
				else
					m= Vnil;
			}
			else {
				/* max2 */
				if (c <= 0)
					m= Vnil;
				else if (compare(v, Upb(t)) <= 0) {
					if (integral(v))
						m= diff(v, one);
					else
						m= floorf(v);
				}
				else
					m= copy(Upb(t));
			}
		} break;
	case Tab:
		{value dm= Vnil; relation c;
			for (k= 0; k < len; k++) {
				c= compare(v, Dts(*tp));
				if (!comp_ok) return Vnil;
				if ((*rel)(c)) {
					if (dm == Vnil ||
						(*rel)(compare(Dts(*tp), dm)))
						dm= Dts(*tp);
				}
				tp++;
			}
			if (dm != Vnil) m= copy(dm);
		} break;
	default:
		syserr(MESS(329, "mm2() on non tlt value"));
		break;
	}
	return m;
}

Visible value min1(t) value t; { /* Monadic min */
	value m= Vnil;
	if (!Is_tlt(t))
		interr(MIN_TLT);
	else if (Length(t) == 0)
		interr(MIN_EMPTY);
	else m= mm1(t, less);
	return m;
}

Visible value min2(v, t) value v, t; {
	value m= Vnil;
	if (!Is_tlt(t))
		interr(MIN2_TLT);
	else if (Length(t) == 0)
		interr(MIN2_EMPTY);
	else if (Is_text(t)) {
		if (!Is_text(v) || Length(v) != 1)
			interr(MIN2_CHAR);
	}
	if (still_ok) {
		m= mm2(v, t, less);
		if (m == Vnil && still_ok)
			interr(MIN2_ELEM);
	}
	return m;
}

Visible value max1(t) value t; {
	value m= Vnil;
	if (!Is_tlt(t))
		interr(MAX_TLT);
	else if (Length(t) == 0)
		interr(MAX_EMPTY);
	else m= mm1(t, greater);
	return m;
}

Visible value max2(v, t) value v, t; {
	value m= Vnil;
	if (!Is_tlt(t))
		interr(MAX2_TLT);
	else if (Length(t) == 0)
		interr(MAX2_EMPTY);
	else if (Is_text(t)) {
		if (!Is_text(v) || Length(v) != 1)
			interr(MAX2_CHAR);
	}
	if (still_ok) {
		m= mm2(v, t, greater);
		if (m == Vnil && still_ok)
			interr(MAX2_ELEM);
	}
	return m;
}

Visible value item(t, n) value t, n; {
	value w= Vnil;
	int m;
	if (!Is_tlt(t))
		interr(ITEM_TLT);
	else if (!Is_number(n) || !integral(n))
		interr(ITEM_INT);
	else if (empty(t))
		interr(ITEM_EMPTY);
	else if (Is_range(t)) {
		value r;
		r= rangesize(Lwb(t), Upb(t));
		if (compare(n, zero) <= 0)
			interr(ITEM_L_BND);
		else if (compare(r, n) < 0)
			interr(ITEM_U_BND);
		else {
			release(r);
			r= sum(n, Lwb(t));
			w= diff(r, one);
		}
		release(r);
	}
	else {
		m= intval(n);
		if (m <= 0)
			interr(ITEM_L_BND);
		else if (m > Length(t))
			interr(ITEM_U_BND);
		else w= thof(m, t);
	} 
	return w;
}

#ifdef B_COMPAT

Visible value th_of(n, t) value n, t; {
	value w= Vnil;
	int m;
	if (!Is_tlt(t))
		interr(THOF_TLT);
	else if (!Is_number(n) || !integral(n))
		interr(THOF_INT);
	else if (empty(t))
		interr(THOF_EMPTY);
	else if (Is_range(t)) {
		value r;
		r= rangesize(Lwb(t), Upb(t));
		if (compare(n, zero) <= 0)
			interr(THOF_L_BND);
		else if (compare(r, n) < 0)
			interr(THOF_U_BND);
		else {
			release(r);
			r= sum(n, Lwb(t));
			w= diff(r, one);
		}
		release(r);
	}
	else {
		m= intval(n);
		if (m <= 0)
			interr(THOF_L_BND);
		else if (m > Length(t))
			interr(THOF_U_BND);
		else w= thof(m, t);
	} 
	return w;
}

#endif /* B_COMPAT */

Visible value thof(n, t) int n; value t; {
	value w= Vnil; value r;
	switch (Type(t)) {
		case Tex:
			{char ww[2];
				ww[0]= *(Str(t)+n-1); ww[1]= '\0';
				w= mk_text(ww);
			} break;
		case Lis:
			w= copy(*(Ats(t)+n-1));
			break;
		case Ran:
			r= sum(w= mk_integer(n), Lwb(t));
			release(w);
			w= diff(r, one);
			release(r);
			break;
		case Tab:
			w= copy(Dts(*(Ats(t)+n-1)));
			break;
		default:
			syserr(MESS(330, "thof() on non tlt value"));
			break;
	}
	return w;
}

Visible bool found_ok= Yes;

Visible bool found(elem, v, probe, where)
	value (*elem)(), v, probe; intlet *where;
	/* think of elem(v,lo-1) as -Infinity and elem(v,hi+1) as +Infinity.
	   found and where at the end satisfy:
	   SELECT:
	       SOME k IN {lo..hi} HAS probe = elem(v,k):
		   found = Yes AND where = k
	       ELSE: found = No AND elem(v,where-1) < probe < elem(v,where).
	*/
{relation c; intlet lo=0, hi= Length(v)-1;
	found_ok= Yes;
	if (lo > hi) { *where= lo; return No; }
	if ((c= compare(probe, (*elem)(v, lo))) == 0) {*where= lo; return Yes; }
	if (!comp_ok || c < 0) { found_ok= comp_ok; *where=lo; return No; }
	if (lo == hi) { *where=hi+1; return No; }
	if ((c= compare(probe, (*elem)(v, hi))) == 0) { *where=hi; return Yes; }
	if (!comp_ok || c > 0) { found_ok= comp_ok; *where=hi+1; return No; }
	/* elem(lo) < probe < elem(hi) */
	while (hi-lo > 1) {
		if ((c= compare(probe, (*elem)(v, (lo+hi)/2))) == 0) {
			*where= (lo+hi)/2; return Yes;
		}
		if (!comp_ok) { found_ok= comp_ok; *where= lo; return No; }
		if (c < 0) hi= (lo+hi)/2; else lo= (lo+hi)/2;
	}
	*where= hi; return No;
}

Visible bool in(v, t) value v, t; {
	intlet where, k, len; value *tp= Ats(t);
	switch (Type(t)) {
	case Tex:
		return strchr((string) tp, *Str(v)) != 0;
	case ELT:
		return No;
	case Lis:
		return found(list_elem, t, v, &where);
	case Ran:
		return (integral(v)
			&&
			compare(Lwb(t), v) <= 0
			&&
			compare(v, Upb(t)) <= 0);
	case Tab:
		len= Length(t);
		for (k= 0; k < len; k++) {
			if (compare(v, Dts(*tp++)) == 0) return Yes;
			if (!comp_ok) return No;
		}
		return No;
	default:
		syserr(MESS(331, "in() on non tlt value"));
		return No;
	}
}

Visible bool empty(v) value v; {
	switch (Type(v)) {
	case Tex:
	case Lis:
	case Ran:
	case Tab:
	case ELT:
		return (Length(v) == 0);
	default:
		syserr(MESS(332, "empty() on non tlt value"));
		return (No);
	}
}
