/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1989. */

/* abckeys -- create a key definitions file interactively */

#include "b.h"
#include "bfil.h"
#include "bmem.h"
#include "feat.h"
#include "keys.h"
#include "getc.h"
#include "trm.h"
#include "release.h"
#include "keydef.h"

char *getenv();

Visible bool intrflag= No; /* not used; only definition needed here */
#ifdef SIGNAL
#include <signal.h>
#ifdef SIGTSTP
Visible bool suspflag= No; /* idem */
#endif
#endif
Visible bool in_vtrm= No;
Visible bool raw_newline= No;

Visible Procedure immexit(status) int status; {
	endprocess(status);
}

#ifndef NDEBUG
Visible bool dflag= No;
#endif

Visible FILE *errfile= stderr;

#ifdef VTRMTRACE
Visible FILE *vtrmfp= NULL;
	/* -V vtrmfile: trace typechecker on vtrmfile; abc only */
#endif

extern int errcount; /* Number of errors detected in key definitions */

extern string intr_char;
#ifdef CANSUSPEND
extern string susp_char;
#endif

/******************************************************************/

#define SNULL ((string) NULL)

/*
 * definitions in deftab[0..nharddefs-1] are determined in ?1keys.c;
 * hardcoded, read in from termcap, and/or taken from tty-chars
 */

Visible int nharddefs;

/*
 * definitions in deftab[nharddefs..nfiledefs-1] come from current keysfile
 * (read in e1getc.c)
 */

Hidden int nfiledefs;

/*
 * The new definitions the user supplies in this program are keep()ed
 * in deftab[nfiledefs..ndefs-1]
 */


/* 
 * The table can than be written to the new keydefinitions file:
 * first the definitions from the old keydefinitions file
 * that are still valid, in [nharddefs.. nfiledefs-1],
 * then the new ones, in [nfiledefs..ndefs-1].
 */

typedef struct oper {
	int code;		/* returned by inchar */
	string name;		/* operation name */
	int allowed;		/* may process */
	string descr;		/* long description */
} operation;

Hidden operation oplist[]= {
	{WIDEN,		S_WIDEN,	0, "Widen focus"},
	{EXTEND,	S_EXTEND,	0, "Extend focus"},
	{FIRST,		S_FIRST,	0, "Focus to first contained item"},
	{LAST,		S_LAST,		0, "Focus to last contained item"},
	{PREVIOUS,	S_PREVIOUS,	0, "Focus to previous item"},
	{NEXT,		S_NEXT,		0, "Focus to next item"},
	{UPLINE,	S_UPLINE,	0, "Focus to whole line above"},
	{DOWNLINE,	S_DOWNLINE,	0, "Focus to whole line below"},
	{UPARROW,	S_UPARROW,	0, "Make hole, move up"},
	{DOWNARROW,	S_DOWNARROW,	0, "Make hole, move down"},
	{LEFTARROW,	S_LEFTARROW,	0, "Make hole, move left"},
	{RITEARROW,	S_RITEARROW,	0, "Make hole, move right"},
	{GOTO,		S_GOTO,		0, "New focus at cursor position"},
	{ACCEPT,	S_ACCEPT,	0, "Accept suggestion, goto hole"},
	{NEWLINE,	S_NEWLINE,	0, "New line, or decrease indent"},
	{UNDO,		S_UNDO,		0, "Undo effect of last key pressed"},
	{REDO,		S_REDO,		0, "Redo last UNDOne key"},
	{COPY,		S_COPY,		0, "Copy focus to/from buffer"},
	{DELETE,	S_DELETE,	0, "Delete focus (to buffer if empty)"},
	{RECORD,	S_RECORD,	0, "Start/stop recording keystrokes"},
	{PLAYBACK,	S_PLAYBACK,	0, "Play back recorded keystrokes"},
	{REDRAW,	S_LOOK,		0, "Redisplay the screen"},
	{HELP,		S_HELP,		0, "Display summary of keys"},
	{EXIT,		S_EXIT,		0, "Finish unit or execute command"},
	{CANCEL,	S_INTERRUPT,	0, "Interrupt a computation"},
	{SUSPEND,	S_SUSPEND,	0, "Suspend the process"},
	{IGNORE,	S_IGNORE,	0, "Unbind this key sequence"},
	{TERMINIT,	S_TERMINIT,	0, "string to be sent to the screen at startup"},
	{TERMDONE,	S_TERMDONE,	0, "string to be sent to the screen upon exit"},
	/* last entry, op->name == SNULL : */
	{0, 		SNULL, 		0, SNULL} 
};

#define ONULL ((operation *) NULL)

Hidden operation *findoperation(name) string name; {
	operation *op;

	for (op= oplist; op->name != SNULL; op++) {
		if (strcmp(op->name, name) == 0)
			return op;
	}
	return ONULL;
}

Visible Procedure confirm_operation(code, name) int code; string name; {
	operation *op;

	for (op= oplist; op->name != SNULL; op++) {
		if (code == op->code) {
			op->allowed= 1;
			op->name= name; /* to be sure */
		}
	}
}

#define Inchar() 	(cvchar(trminput()))

#define Printable(c)	(isascii(c) && (isprint(c) || (c) == ' '))
#define CRLF(c)		(Creturn(c) || Clinefeed(c))
#define Creturn(c)	((c) == '\r')
#define Clinefeed(c)	((c) == '\n')
#define Cbackspace(c)	((c) == '\b')
#define Ctab(c)		((c) == '\t')
#define Cspace(c)	((c) == ' ')

#define Empty(d)	(strlen(d) == 0)
#define Val(d)		((d) != SNULL && !Empty(d))

#define Equal(s1, s2)	(strcmp(s1, s2) == 0)

/****************************************************************************/

Hidden string newfile= SNULL;	/* name for new keydefinitions file */

main(argc, argv) int argc; char *argv[]; {
	string arg0= argv[0];
	string cp;
	int c;

	cp= strrchr(arg0, DELIM);
	if (cp)
		arg0= cp+1;

	initfmt();

	if (argc != 1) /* no arguments allowed */
		usage(arg0);

	init();
	
	checking();
	
	process();
	
	fini();
	
	exit(0);
}

/****************************************************************************/

/* immediate exit */

Hidden Procedure usage(name) string name; {
	putSstr(errfile, "*** Usage: %s\n", name);
	exit(1);
}

Hidden Procedure endprocess(status) int status; {
	fini_term();
	exit(status);
}

Visible Procedure syserr(s) string s; {
	putSstr(errfile, "*** System error: %s\n", s);
	endprocess(-1);
}

Visible Procedure memexh() {
	static bool beenhere= No;
	if (beenhere) endprocess(-1);
	beenhere= Yes;
	putstr(errfile, "*** Sorry, memory exhausted\n");
	endprocess(-1);
}

/****************************************************************************/

Hidden Procedure init() {
#ifdef MEMTRACE
	initmem();
#endif

	initmess();
	initfile();
	initkeys();		/* fills deftab and ndefs in e1getc.c */
	nfiledefs= ndefs;
	
	init_newfile();
	init_ignore();
	init_strings();
	init_term();
	init_bindings();
	init_buffers();
}

Hidden Procedure fini() {
#ifdef MEMTRACE
	fini_buffers();
#endif
	fini_term();
}


/****************************************************************************/

Hidden Procedure checking() {
	if (!Val(intr_char)) {
		putdata(E_INTERRUPT, 0);
		endprocess(1);
	}
}

/****************************************************************************/

#define DNULL (tabent *) NULL

Hidden tabent *finddefentry(code) int code;  {
	tabent *d;

	for (d= deftab+ndefs-1; d >= deftab; d--) {
		if (code == d->code)
			return d;
	}
	return DNULL;
}

Hidden tabent *terminit= DNULL;
Hidden tabent *termdone= DNULL;

Hidden Procedure init_strings() {
	terminit= finddefentry(TERMINIT);
	termdone= finddefentry(TERMDONE);
}

/* Output a string to the terminal */

Hidden Procedure outstring(str) string str; {
	fputs(str, stdout);
	putnewline(stdout);
	fflush(stdout);
}

Hidden bool inisended= No;

Hidden Procedure sendinistring() {
	if (terminit != DNULL && Val(terminit->def)) {
		outstring(terminit->def);
		redrawscreen();
		inisended= Yes;
	}
	else clearwindow();
}

Hidden Procedure sendendstring() {
	if (!inisended)
		return;
	if (termdone != DNULL && Val(termdone->def)) {
		outstring(termdone->def);
	}
}

/****************************************************************************/

/* screen stuff */

Hidden struct screen {
	int yfirst, ylast;
	int width;
	int y, x;
} win;

Hidden Procedure init_term() {
	int height, width, flags;
	int err;

	err= trmstart(&height, &width, &flags);
	if (err != TE_OK) {
		if (err <= TE_DUMB)
			putstr(errfile,
"*** Bad $TERM or termcap, or dumb terminal\n");
		else if (err == TE_BADSCREEN)
			putstr(errfile,
"*** Bad SCREEN environment\n");
		else
			putstr(errfile,
"*** Cannot reach keyboard or screen\n");

		exit(1);
	}
	in_vtrm= Yes;
	raw_newline= Yes;
	win.yfirst= 0;
	win.ylast= height-1;
	win.width= width-1;
	win.y= win.yfirst;
	win.x= 0;
	
#define MINWIDTH 75
#define MINHEIGHT 24

	if (width < MINWIDTH || height < MINHEIGHT) {
		put2Dstr(errfile,
"*** Sorry, too small screen size; needed at least %dx%d; giving up\n",
		MINHEIGHT, MINWIDTH);
		endprocess(1);
	}

	if (errcount != 0) /* errors found reading definitions */
		asktocontinue(win.ylast);
#ifdef DUMPKEYS
	if (dflag && errcount == 0) 
		asktocontinue(win.ylast);
#endif
	clearscreen(); 
}

/* 
 * clearing the screen is done by scrolling instead of putting empty data
 * because there are systems (MSDOS, ANSI) where the latter leaves rubbish
 * on the screen
 */
 
Hidden Procedure clearscreen() {
	trmscrollup(0, win.ylast, win.ylast + 1);
}

Hidden int hlp_yfirst;
Hidden int hlp_nlines;

#define Upd_bindings() putbindings(hlp_yfirst)

Hidden Procedure init_bindings() {
	setup_bindings(win.width, &hlp_nlines);
}

Hidden int nscrolls= 0;

Hidden Procedure set_windows(yfirst) int yfirst; {
	hlp_yfirst= yfirst;
	win.yfirst= hlp_yfirst + hlp_nlines + 1;
	win.y= win.yfirst;
	win.x= 0;
	nscrolls= 0;
}

Hidden Procedure clearwindow() {
	trmputdata(win.yfirst, win.ylast, 0, "");
	win.y= win.yfirst;
	win.x= 0;
	nscrolls= 0;
	trmsync(win.y, win.x);
}

Hidden Procedure redrawscreen() {
	bind_all_changed();
	clearscreen();
	set_windows(0);
	Upd_bindings();
}

Hidden Procedure fini_term() {
	if (in_vtrm) {
#ifdef MEMTRACE
		fini_bindings();
#endif
		nextline();
		sendendstring();
		trmend();
	}
	in_vtrm= No;
}

/* TODO: indent > width-1 */

#define Too_width(data, bound) (strlen(data) > (bound))

Hidden Procedure putdata(data, indent) string data; int indent; {
	static string buf= SNULL;
	int width= win.width;
	int len;
	string q;

	if (data == SNULL)
		return;
	if (buf == SNULL)
		buf= (string) getmem((unsigned) width+1);

	if (indent == 0 && strlen(data) > 0 && win.x > 0)
		nextline();

	while (Too_width(data, width-indent)) {
		q= data + width-1-indent;
		while (q - data > 0 && *q != ' ')
			--q;
		len= q - data;
		if (len > 0 && len < width-indent)
			++len;
		else
			len= width-indent;
		strncpy(buf, data, len);
		buf[len]= '\0';
		data+= len;
		trmputdata(win.y, win.y, indent, buf);
		nextline();
		indent= 0;
	}
	trmputdata(win.y, win.y, indent, data);
	win.x= indent+strlen(data);
	trmsync(win.y, win.x);
}

#define CONTINUE_GIVEN (nscrolls == 1)

Hidden Procedure nextline() {
	if (win.y == win.ylast-1) {
		if (nscrolls == 0 || nscrolls == (win.ylast - win.yfirst)) {
			asktocontinue(win.ylast);
			nscrolls= 0;
		}
		trmscrollup(win.yfirst, win.ylast, 1);
		nscrolls++;
	}
	else {
		win.y++;
		nscrolls= 0;
	}
	trmsync(win.y, win.x= 0);
}

#define SOBIT 0200
#define MAXBUFFER 81

Hidden string mkstandout(data) string data; {
	static char buffer[MAXBUFFER];
	string cp;
	
	strcpy(buffer, data);
	for (cp= buffer; *cp; cp++)
		*cp |= SOBIT;

	return (string) buffer;
}

#define CONTINUE_PROMPT "Press [SPACE] to continue "

Hidden Procedure asktocontinue(y) int y; {
	int c;
	string data= mkstandout(CONTINUE_PROMPT);

	trmputdata(y, y, 0, data);
		/*
		 * putdata() isn't called to avoid a call of nextline();
		 * there is no harm in that if the data can fit on one line
		 */
	trmsync(y, strlen(data));
	for (;;) {
		c= Inchar();
		if (Cspace(c) || c == EOF)
			break;
		trmbell();
	}
	trmputdata(y, y, 0, "");
}

/****************************************************************************/

/* buffer stuff */

Hidden char fmtbuf[BUFSIZ];	/* to make formatted messages */

Hidden bufadm definpbuf;	/* to save definitions from input */
Hidden bufadm repinpbuf;	/* to save representations from input */
Hidden bufadm reprbuf;		/* to save reprs from defs */

Hidden Procedure init_buffers() {
	bufinit(&definpbuf);
	bufinit(&repinpbuf);
	bufinit(&reprbuf);
}

#ifdef MEMTRACE

Hidden Procedure fini_buffers() {
	buffree(&definpbuf);
	buffree(&repinpbuf);
	buffree(&reprbuf);
}

#endif

Hidden string getbuf(bp) bufadm *bp; {
	bufpush(bp, '\0');
	return (string) bp->buf;
}

/****************************************************************************/

#ifndef NULL_EXTENDED

#define MAXAVAILABLE 100

Hidden int available[MAXAVAILABLE];	/* save chars from trmavail() */
Hidden int navailable= 0;		/* nr of available chars */
Hidden int iavailable= 0;		/* next available character */

/*
 * attempt to recognize key sequences using trmavail();
 * it works if the user presses the keys one after another not too fast;
 * be careful: if trmavail() isn't implemented it still has to work!
 * returns -1 for EOF, 0 for extended chars, >0 for 'normal' chars.
 */

Hidden int inchar() {
	int c;
	
	if (iavailable != navailable) {		/* char in buffer */
		c= available[iavailable++];
		if (iavailable == navailable)
			iavailable= navailable= 0;
		return c;
	}

	c= Inchar();	/* returns -1 or >0 */

	while (c != EOF && trmavail() == 1) {
		available[navailable++]= c;
		c= Inchar();
	}
	if (navailable == 0)			/* no char available */
		return c;
	else {
		available[navailable++]= c;
		return 0;
	}
}

Hidden string findrepr(def) string def; {
	tabent *d;
	string findoldrepr();
	string rep;

	for (d= deftab+ndefs-1; d >= deftab; d--) {
		if (Val(d->def) && Equal(d->def, def) && Val(d->rep))
			return d->rep;
	}
	return findoldrepr(def);
}

/*
 * try to find a representation for thw whole sequence in the buffer
 */

Hidden bool knownkeysequence(key, rep) string *key, *rep; {
	string pkey;
	int n;

	if (navailable < 2)			/* no sequence */
		return No;

	/* make sequence */
	*key= pkey= (string) getmem((unsigned) (navailable+1));
	for (n= 0; n < navailable; n++)
		*pkey++= available[n];
	*pkey= '\0';

	if ((*rep= findrepr(*key)) != SNULL) {
		iavailable= navailable= 0; 	/* empty buffer */
		return Yes;
	}
	freemem((ptr) *key);
	return No;
}

#endif /* ! NULL_EXTENDED */

/****************************************************************************/

/*
 * get a key sequence from input, delimited by \r (or \n)
 * if you want that delimiter in your binding,
 * enclose the entire binding with single or double quotes
 */

#define NEW_KEY	"Press new key(s) for %s (%s)"

#define Quote(c) ((c) == '\"' || (c) == '\'')

Hidden string ask_definition(op, prepr) operation *op; string *prepr; {
	int c;
	string def;
	string repr;
	bufadm *dp= &definpbuf;
	bufadm *rp= &reprbuf;
	char quot_repr[20];
	bool quoting= No;
	bool first= Yes;

	sprintf(fmtbuf, NEW_KEY, op->name, op->descr);
	putdata(fmtbuf, 0);
	nextline();

	bufreinit(dp);
	bufreinit(rp);

	for (;; first= No) {

#ifdef NULL_EXTENDED

		c= Inchar();
		
#else /* ! NULL_EXTENDED */

		c= inchar();
		if (c == 0) { /* there are chars in the buffer */
			if (knownkeysequence(&def, &repr)) {
				savputrepr(rp, repr);	/* save and put repr */
				bufcpy(dp, def);	/* save key */
				freemem((ptr) def);
				continue;
			}
			else c= inchar(); /* get char out of buffer */
					  /* note: c != 0 */
		}

#endif /* ! NULL_EXTENDED */

		if (c == EOF)
			break;
		if (Eok(c)) {		/* end of key sequence */
			if (!quoting)
				break;
			if (Equal(repr, quot_repr)) {
					/* pop quote from key buffer: */	
				--(dp->ptr);
					/* pop quote from rep buffer: */
				rp->ptr-= strlen(repr) + 1;
				break;
			}
		}
		if (first && Quote(c)) {
			quoting= Yes;
			repr= reprchar(c);
			strcpy(quot_repr, repr);
			putdata(repr, win.x);	/* no save */
			putdata(" ", win.x);	
			repr= "";		/* to prevent equality above */
		}
		else {
			repr= reprchar(c);
			savputrepr(rp, repr);	/* save and put repr */
			bufpush(dp, c);		/* save key */
		}
	}
	*prepr= getbuf(rp);

	return getbuf(dp);
}

/* save and put the representation */

Hidden Procedure savputrepr(rp, repr) bufadm *rp; string repr; {
	if (strlen(repr) > 0) {
		/* save */
		if (rp->ptr != rp->buf) /* not the first time */
			bufpush(rp, ' '); 
		bufcpy(rp, repr);

		/* put */
		putdata(repr, win.x);
		putdata(" ", win.x);
	}
}

Hidden string new_definition(op, prepr) operation *op; string *prepr; {
	string def;

	if (op == ONULL)
		return SNULL;
	for (;;) {
		def= ask_definition(op, prepr);
		if (op->code < 0) /* string-valued */
			return def;
		if (!illegal(def))
			return def;
	}
}

Hidden bool illegal(def) string def; {
	if (Empty(def))
		return No;
	if  (Printable(*def)) {
		sprintf(fmtbuf, E_ILLEGAL, *def);
		putdata(fmtbuf, 0);
		return Yes;
	}
	for (; *def; def++) {
		if (is_spchar(*def)) {
			putdata(E_SPCHAR, 0);
			return Yes;
		}
	}
	return No;
}

/****************************************************************************/

/*
 * getinput() reads characters from input delimited by \r or \n 
 */
 
Hidden string getinput(bp)  bufadm *bp; {
	int c;
	char echo[2];

	echo[1]= '\0';
	bufreinit(bp);
	for (;;) {
		c= Inchar();
		if (c == EOF || CRLF(c))
			break;

		if (Cbackspace(c)) {
			if (bp->ptr == bp->buf)		/* no chars */
				trmbell();
			else {
				if (win.x == 0) {	/* begin of line */
					--win.y;
					win.x= win.width;
				}
				putdata("", --win.x);
				--(bp->ptr);	/* pop character from buffer */
			}
		}
		else if (Printable(c)) {
			echo[0]= c;
			putdata(echo, win.x);
			bufpush(bp, c);
		}
		else trmbell();
	}
	return getbuf(bp);
}

/****************************************************************************/

#define ALPHA_REP "Enter an alpha-numeric representation for this definition"

#define DFLT_REP " [default %s] "

Hidden string ask_representation(dfltrep) string dfltrep; {
	int len= strlen(DFLT_REP) + strlen(dfltrep);
	char *dflt= (char *) getmem((unsigned) (len+1));
	/* we don't use fmtbuf, because the 'dfltrep' can be very long */

	putdata(ALPHA_REP, 0);
	sprintf(dflt, DFLT_REP, dfltrep);
	putdata(dflt, 0);
	freemem((ptr) dflt);
	return getinput(&repinpbuf);
}

Hidden string new_representation(dfltrep, def) string dfltrep, def; {
	string repr;

	for (;;) {
		repr= ask_representation(dfltrep);

		if (Empty(repr)) /* accept default */
			return dfltrep;
		if (unlawful(repr) || rep_in_use(repr, def))
			continue; 
		return repr;
	}
}

Hidden string representation(def) string def; {
	bufadm *rp= &reprbuf;
	string repr;

	bufreinit(rp);

	for (; *def; def++) {
		repr= reprchar(*def);
		if (strlen(repr) > 0) {
			bufcpy(rp, repr);
			if (*(def+1) != '\0') {
				bufpush(rp, ' ');
			}
		}
	}
	return getbuf(rp);
}

Hidden bool unlawful(rep) string rep; {
	for (; *rep; rep++) {
		if (!Printable(*rep)) {
			putdata(E_UNLAWFUL, 0);
			return Yes;
		}
	}

	return No;
}

Hidden bool rep_in_use(rep, def) string rep, def; {
	tabent *d;

	for (d= deftab; d < deftab+ndefs; d++) {
		if (Val(d->rep) && Equal(rep, d->rep)
		    &&
		    Val(d->def) && !Equal(def, d->def)
		    &&
		    d->code != DELBIND
		   ) {
			sprintf(fmtbuf, E_IN_USE, d->name);
			putdata(fmtbuf, 0); 
			return Yes;
		}
	}
	return No;
}

/****************************************************************************/

Hidden Procedure keep(code, name, def, rep) int code; string name, def, rep; {
	if (ndefs == MAXDEFS) {
		putdata(E_TOO_MANY, 0);
		return;
	}
	undefine(code, def);
	deftab[ndefs].code= code;
	deftab[ndefs].name= name;
	deftab[ndefs].def= (string) savestr(def);
	deftab[ndefs].rep= (string) savestr(rep);
	ndefs++;
}

Hidden Procedure store(code, name, def, rep) int code; string name, def, rep; {
	tabent *d;

	if (code > 0) {
		keep(code, name, def, rep);
	}
	else {	/* code < 0; string-valued entry */
		/* find the place matching name to replace definition */
	        for (d= deftab; d < deftab+ndefs; ++d) {
			if (code == d->code) {
	                       	d->def= (string) savestr(def);
	                       	d->rep= (string) savestr(rep);
	                       	break;
			}
		}
	}
	bind_changed(code);
}

/****************************************************************************/

#define I_OP_PROMPT "Enter operation [? for help]: "
#define OP_PROMPT   "Enter operation: "

Hidden string ask_name(prompt) string prompt; {
	putdata(prompt, 0);
	return getinput(&definpbuf);
}

Hidden Procedure print_heading() {
	sprintf(fmtbuf, ABC_RELEASE, RELEASE);
	putdata(fmtbuf, 0);
	nextline();
	putdata(COPYRIGHT, 0);
	nextline();
	putdata(HEADING, 0);
	nextline();
	nextline();
}

Hidden Procedure process() {
	operation *op;
	string name;
	bool show;
	bool del;
	bool first= Yes;
	int ysave;

	print_heading();

	ysave= win.y;

	set_windows(win.y);
	Upd_bindings();

	for (;;) {
		if (first) {
			name= ask_name(I_OP_PROMPT);
			scrolloff_heading(ysave);
			first= No;
		}
		else {
			setpromptline();
			name= ask_name(OP_PROMPT);
		}
		if (Empty(name))
			continue;
		if (Equal(name, "?")) {
			help();
			continue;
		}
		show= *name == '=';
		del= *name == '-';
		if (show || del) name++;

		if (is_quit(name)) {
			if (!del)
				putkeydefs();
			break;
		}
		else if (is_init(name)) {
			nextline();
			sendinistring();
			continue;
		}

		sprintf(fmtbuf, "[%s]", name);
		op= findoperation(fmtbuf);

		if (op == ONULL || !op->allowed) {
			putdata(E_UNKNOWN, 0);
			continue;
		}
		if (!show && spec_operation(op)) {
			sprintf(fmtbuf, E_NOTALLOWED, name);
			putdata(fmtbuf, 0);
			continue;
		}

		if (show)
			showbindings(op);
		else if (del)
			delbindings(op);
		else
			definebinding(op);
	}
}

Hidden bool is_quit(name) string name; {
	if (Equal(name, "q") || Equal(name, "quit"))
		return Yes;
	return No;
}

Hidden bool is_init(name) string name; {
	if (Equal(name, "init"))
		return Yes;
	return No;
}

Hidden bool spec_operation(op) operation *op; {
	if (op->code == CANCEL || op->code == SUSPEND)
		return Yes;
	return No;
}

Hidden Procedure scrolloff_heading(n) int n; {
	int y= win.y, x= win.x;		/* save old values */

	trmscrollup(0, win.ylast, n);
	set_windows(0);
	win.y= y - n;
	win.x= x;
}

Hidden Procedure setpromptline() {
	if (win.y != win.yfirst || win.x > 0) {
		if (win.x > 0)
			nextline();
		if (!CONTINUE_GIVEN)
			nextline();
		if (CONTINUE_GIVEN)
			clearwindow();
	}
}

/****************************************************************************/

Hidden Procedure definebinding(op) operation *op; {
	string def, rep;

	clearwindow();
	def= new_definition(op, &rep);
	if (!Val(def))
		return;

#ifndef KNOWN_KEYBOARD
	rep= new_representation(rep, def);
#else
	if (op->code == TERMINIT || op->code == TERMDONE)
		rep= new_representation(rep, def);
#endif

	store(op->code, op->name, def, rep);
	Upd_bindings();
}

#define SHOW_PROMPT "Showing the bindings for %s (%s):"

Hidden Procedure showbindings(op) operation *op; {
	tabent *d;

	clearwindow();
	sprintf(fmtbuf, SHOW_PROMPT, op->name, op->descr);
	putdata(fmtbuf, 0);

	for (d= deftab+ndefs-1; d >= deftab; d--) {
		if (d->code != op->code || !Val(d->def) || !Val(d->rep))
			continue;
		putdata(d->rep, 0);
	}
}

Hidden Procedure delbindings(op) operation *op; {
	tabent *d;

	for (d= deftab; d < deftab+ndefs; d++) {
		if (d->code == op->code && Val(d->def)) {
			store(DELBIND, S_IGNORE, d->def, d->rep);
			d->def= d->rep= SNULL;
			bind_changed(d->code);
		}
	}
	Upd_bindings();
	clearwindow();
}

/****************************************************************************/

Hidden tabent savedeftab[MAXDEFS];
Hidden int nsaveharddefs= 0;
Hidden int nsavefiledefs= 0;


Visible Procedure saveharddefs() {
	tabent *d, *h;
	
	for (d= deftab, h= savedeftab; d < deftab+nharddefs; d++) {
		if (Val(d->name) && Val(d->def)) {
			h->code= d->code;
			h->name= d->name;
			h->def= d->def;
			h->rep= d->rep;
			h++;
		}
	}
	nsaveharddefs= h-savedeftab;
}

Visible Procedure savefiledefs() {
	tabent *d, *h;
	
	d= deftab + nharddefs;
	h= savedeftab + nsaveharddefs;
	for (; d < deftab + ndefs; d++) {
		if (Val(d->name) && Val(d->def)) {
			h->code= d->code;
			h->name= d->name;
			h->def= d->def;
			h->rep= d->rep;
			h++;
		}
	}
	nsavefiledefs= h-savedeftab;
}

Hidden bool a_harddef(d) tabent *d; {
	tabent *h;

	if (!Val(d->def))
		return No;
	for (h= savedeftab; h < savedeftab+nsaveharddefs; h++) {
		if (Equal(d->def, h->def) && 
			Equal(d->rep, h->rep) &&	/* TODO: needed ? */
			(d->code == h->code ||
			 d->code == IGNORE ||
			 d->code == DELBIND
			)
		   )
			return Yes;
	}
	return No;
}

Hidden Procedure init_ignore() {
	tabent *d;
	
	for (d= deftab+nharddefs; d < deftab+ndefs; d++) {
		if (d->code == IGNORE && a_harddef(d))
			/* don't show it in the bindings window */
			d->code= DELBIND;
	}
}

#ifndef NULL_EXTENDED

Hidden string findoldrepr(def) string def; {
	tabent *h;

	h= savedeftab + nsavefiledefs - 1;
	for (; h >= savedeftab; h--) {
		if (Val(h->def) && Equal(h->def, def) && Val(h->rep))
			return h->rep;
	}
	return SNULL;
}

#endif /* ! NULL_EXTENDED */

/****************************************************************************/

FILE *keyfp;			/* fileptr for key definitions file */

Hidden Procedure putkeydefs() {
	openkeyfile();
	put_table();
	put_strings();
	closekeyfile();
}

Hidden Procedure init_newfile() {
	char *termname;
	string termfile;
	
#ifdef KEYSPREFIX
	if ((termname= getenv("TERM")) != NULL) {
		termfile= (string) getmem((unsigned) strlen(KEYSPREFIX)+strlen(termname));
		strcpy(termfile, KEYSPREFIX);
		strcat(termfile, termname);
	}
	else
#endif /*KEYSPREFIX*/
		termfile= savestr(NEWFILE);
	
	if (bwsdefault
	    && (D_exists(bwsdefault) || Mkdir(bwsdefault) == 0)
	    && F_writable(bwsdefault))
	{
		newfile= makepath(bwsdefault, termfile);
	}
	else {
		putSstr(errfile,
		"Cannot use directory \"%s\" for private keydefinitions file\n",
			bwsdefault);
		putSstr(errfile,
		"Cannot use directory \"%s\" for private keydefinitions file",
			bwsdefault);
		
		newfile= termfile;
	}
}

#define MAKE_KEYFILE "Producing key definitions file %s."

Hidden Procedure openkeyfile() {
	keyfp= fopen(newfile, "w");
	nextline();
	if (keyfp == NULL) {
		sprintf(fmtbuf, E_KEYFILE, newfile);
		putdata(fmtbuf, 0);
		keyfp= stdout;
	}
	else {
		sprintf(fmtbuf, MAKE_KEYFILE, newfile);
		putdata(fmtbuf, 0);
	}
	freemem(newfile);
}

Hidden Procedure closekeyfile() {
	fclose(keyfp);
}

Hidden Procedure put_table() {
	tabent *d;
	
	for (d= deftab+nharddefs; d < deftab+ndefs; d++) {
		if (Val(d->def)) {
			if (d->code != IGNORE) {
				if (d->code == DELBIND) {
					if (!a_harddef(d))
						continue;
				}
				else if (a_harddef(d))
					continue;
			}
			put_def(d->name, d->def, d->rep);
		}
	}
}

Hidden Procedure put_strings() {
	if (terminit != DNULL && Val(terminit->def)) {
		string rep= terminit->rep;
		put_def(S_TERMINIT, terminit->def, Val(rep) ? rep : "");
	}
	else put_def(S_TERMINIT, "", "");

	if (termdone != DNULL && Val(termdone->def)) {
		string rep= termdone->rep;
		put_def(S_TERMDONE, termdone->def, Val(rep) ? rep : "");
	}
	else put_def(S_TERMDONE, "", "");
}

#define NAMESPACE 15 /* TODO: e1getc.c accepts until 20 */

Hidden Procedure put_def(name, def, rep) string name, def, rep; {
	int i;
	string s;

	i= 0;
	for (s= name; *s; s++) {
		putchr(keyfp, *s);
		i++;
	}
	while (i < NAMESPACE) {
		putchr(keyfp, ' ');
		i++;
	}
	putstr(keyfp, " = ");
	putchr(keyfp, '"');
	for (s= def; *s != '\0'; ++s) {
		if (*s == '"')
			putchr(keyfp, '\\');
		if (Printable(*s))
			putchr(keyfp, *s);
		else
			putDstr(keyfp, "\\%03o", (int) (*s&0377));
	}
	putchr(keyfp, '"');
	putSstr(keyfp, " = \"%s\"\n", rep);
}

/****************************************************************************/

#define HELP_PROMPT	"Press [SPACE] to continue, [RETURN] to exit help" 

Hidden Procedure help() {
	clearwindow();
	shorthelp();
	if (morehelp()) {
		clearwindow();
		longhelp();
	}
	else
		clearwindow();
}

Hidden Procedure shorthelp() {
	putdata(" name: (re)define binding for \"name\",", 0);
	putdata("-name: remove all the bindings for \"name\"", 0);
	putdata("=name: show all the bindings for \"name\"", 0);
	putdata(" quit: exit this program, saving the changes", 0);
	putdata("-quit: exit this program", 0);
	putdata(" init: send term-init string to screen", 0);
}

Hidden bool morehelp() {
	int c;
	int y= win.y+1;
	string prompt= mkstandout(HELP_PROMPT);
	bool ans;

	if (y < win.ylast)
		y++;
	trmputdata(y, y, 0, prompt);
	trmsync(y, strlen(prompt));

	for (;;) {
		c= Inchar();
		if (c == EOF || CRLF(c))
			{ ans= No; break; }
		else if (Cspace(c))
			{ ans= Yes; break; }
		else
			trmbell();
	}
	trmputdata(y, y, 0, "");
	return ans;
}

Hidden Procedure longhelp() {

putdata("    While (re)defining a binding, the program will ask you to enter \
a key sequence; end it with [RETURN].", 0);

putdata("If you want [RETURN] in your binding, enclose the whole binding \
with single or double quotes.", 0);

#ifndef KNOWN_KEYBOARD

putdata("It will then ask you how to represent this key in the bindings \
window; the default can be accepted with [RETURN].", 0);

#endif /* KNOWN_KEYBOARD */

putdata("    [term-init] and [term-done] are the names for the strings that \
should be sent to the screen upon startup and exit, respectively (for \
programming function keys or setting background colours etc).", 0);

sprintf(fmtbuf,
"    This program will not allow you to use your interrupt character (%s) in \
any keybinding, since the ABC system always binds this to %s.",
	representation(intr_char), S_INTERRUPT);
putdata(fmtbuf, 0);

#ifdef CANSUSPEND

if (susp_char != SNULL) {
sprintf(fmtbuf, "The same holds for your suspend character (%s), bound to %s.",
	representation(susp_char), S_SUSPEND);
putdata(fmtbuf, 0);
			}
#endif /* CANSUSPEND */

putdata("You can use this idiosyncrasy to cancel a binding while typing \
by including your interrupt character.", 0);

putdata("   The space in the window above sometimes isn't sufficient to \
show all the bindings. You will recognize this situation by a marker \
('*') after the name. Hence the option '=name'.", 0);

}
