/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1986. */

/* B editor -- read key definitions from file */

#include "b.h"
#include "feat.h"
#include "bmem.h"
#include "bobj.h"
#include "bfil.h"
#include "keys.h"
#include "getc.h"
#include "args.h"

#define ESC '\033'

/*
This file contains a little parser for key definition files.
To allow sufficient freedom in preparing such a file, a simple
grammar has been defined according to which the file is parsed.
The parsing process is extremely simple, as it can be done
top-down using recursive descent.


Lexical conventions:

- Blanks between lexical symbols are ignored.
- From '#' to end of line is comment (except inside strings).
- Strings are delimited by double quotes and
  use the same escape sequences as C strings, plus:
  \e or \E means an ESCape ('\033').
- Commandnames are like C identifiers ([a-zA-Z_][a-zA-Z0-9_]*).
  Upper/lower case distinction is significant.
- Key representations are delimited by double quotes, and may use
  any printable characters.

Syntax in modified BNF ([] mean 0 or 1, * means 0 or more, + means 1 or more):

   file: line*
   line: [def] [comment]
   def: '[' commandname ']' '=' definition  '=' representation
   definition: string


Notes:

- A definition for command "[term-init]" defines a string to be sent
  TO the terminal at initialization time, e.g. to set programmable
  function key definitions.  Similar for "[term-done]" on exiting.
- Command names are conventional editor operations.
- Some bindings are taken from tty-settings, and should not be changed.
  (interrupt and suspend).
*/

#define COMMENT '#' /* Not B-like but very UNIX-like */
#define QUOTE '"'

Hidden FILE *keysfp; /* File from which to read */
Hidden char nextc; /* Next character to be analyzed */
Hidden bool eof; /* EOF seen? */
Hidden int lcount; /* Current line number */
#ifndef KEYS
Hidden int errcount= 0; /* Number of errors detected */
#else
Visible int errcount= 0; /* Number of errors detected */
#endif

Visible int ndefs;

Hidden Procedure err1(m)
	string m;
{
	static char errbuf[MESSBUFSIZE];
		/* since putmess() below overwrites argument m via getmess() */

	sprintf(errbuf, "%s (%d): %s\n", keysfile, lcount, m);
						
	if (errcount == 0) {
		putmess(errfile, MESS(6500, "Errors in key definitions file:\n"));
	}
	++errcount;

	putstr(errfile, errbuf);
}

Hidden Procedure err(m)
	int m;
{
	err1(getmess(m));
}

Hidden Procedure adv()
{
	int c;

	if (eof)
		return;
	c= getc(keysfp);
	if (c == EOF) {
		nextc= '\n';
		eof= Yes;
	}
	else {
		nextc= c;
	}
}

Hidden Procedure skipspace()
{
	while (nextc == ' ' || nextc == '\t')
		adv();
}

Hidden int lookup(name)
	string name;
{
	int i;

	for (i= 0; i < ndefs; ++i) {
		if (deftab[i].name != NULL && strcmp(name, deftab[i].name) == 0)
			return i;
	}
	return -1;
}

/*
 * Undefine conflicting definitions, i.e. strip them from other commands.
 * Conflicts arise when a command definition is
 * an initial subsequence of another, or vice versa.
 * String definitions (code < 0) are not undefined.
 * The special commands (like interrupt) should not be undefined.
 */
Visible Procedure undefine(code, def)
	int code;
	string def;
{
	struct tabent *d, *last= deftab+ndefs;
	string p, q;

	if (code < 0) 
		return;
	for (d= deftab; d < last; ++d) {
		if (d->code > 0 && d->def != NULL) {
			for (p= def, q= d->def; *p == *q; ++p, ++q) {
				if (*p == '\0') break;
			}
			if (*p == '\0' || *q == '\0') {
				d->def= NULL;
				d->rep= NULL;
#ifdef KEYS
				bind_changed(d->code);
#endif
			}
		}
	}
}

Hidden bool store(code, name, def, rep)		/* return whether stored */
	int code;
	string name;
	string def;
	string rep;
{
	struct tabent *d, *last= deftab+ndefs;
	char *pc;

	if (code < 0) {
		/* find the place matching name to replace definition */
	        for (d= deftab; d < last; ++d) {
			if (strcmp(name, d->name) == 0)
                        	break;
		}
	}
	else {
		/* Check for illegal definition:
		   If a command definition starts with a printable character
		   OR it contains one of the special chars that are, or
	   	   must be handled as signals (like interrupt, suspend, quit).
	 	*/
		if (isascii(*def) && (isprint(*def) || *def==' ')) {
			sprintf(messbuf,
		GMESS(6501, "Definition for command %s starts with '%c'."),
				name, *def);
			err1(messbuf);
			return No;
		}
		for (pc= def; *pc != '\0'; pc++) {
			if (is_spchar(*pc)) {
				sprintf(messbuf,
#ifdef CANSUSPEND

GMESS(6502, "Definition for command %s would produce an interrupt or suspend."),

#else

GMESS(6503, "Definition for command %s would produce an interrupt."),

#endif
				name, *def);
				err1(messbuf);
				return No;
			}
		}
		
		undefine(code, def);
		/* New definitions are added at the end, so the last one can be 
		   used in the HELP blurb. */
		d= last;
		/* Extend definition table */
		if (ndefs >= MAXDEFS) {
			err(MESS(6504, "Too many key definitions"));
			return No;
		}
		ndefs++;
	}
	d->code= code;
	d->name= name;
	d->def= def;
	d->rep= rep;
#ifdef MEMTRACE
	fixmem((ptr) name);
	fixmem((ptr) def);
	fixmem((ptr) rep);
#endif
	return Yes;
}

Hidden string getname()
{
	char buffer[20];
	string bp;
	
	if (nextc != '[') {
		err(MESS(6505, "no '[' before name"));
		return NULL;
	}
	bp= buffer;
	*bp++= nextc;
	adv();
	if (!isascii(nextc)
	    ||
	    (!isalpha(nextc) && nextc != '_' && nextc != '-')
	   ) {
		err(MESS(6506, "No name after '['"));
		return NULL;
	}
	while ((isascii(nextc) && isalnum(nextc))
	       || nextc == '_' || nextc == '-'
	      ) {
		if (bp < buffer + sizeof buffer - 1)
			*bp++= (nextc == '_' ? '-' : nextc);
		adv();
	}
	if (nextc != ']') {
		err(MESS(6507, "no ']' after name"));
		return NULL;
	}
	*bp++= nextc;
	adv();
	*bp= '\0';
	return (string) savestr(buffer);
}

Hidden string getstring()
{
	char buf[256]; /* Arbitrary limit */
	char c;
	int len= 0;

	if (nextc != QUOTE) {
		err(MESS(6508, "opening string quote not found"));
		return NULL;
	}
	adv();
	while (nextc != QUOTE) {
		if (nextc == '\n') {
			err(MESS(6509, "closing string quote not found in definition"));
			return NULL;
		}
		if (nextc != '\\') {
			c= nextc;
			adv();
		}
		else {
			adv();
			switch (nextc) {

			case 'r': c= '\r'; adv(); break;
			case 'n': c= '\n'; adv(); break;
			case 'b': c= '\b'; adv(); break;
			case 't': c= '\t'; adv(); break;
			case 'f': c= '\f'; adv(); break;

			case 'E':
			case 'e': c= ESC; adv(); break;

			case '0': case '1': case '2': case '3':
			case '4': case '5': case '6': case '7':
				c= nextc-'0';
				adv();
				if (nextc >= '0' && nextc < '8') {
					c= 8*c + nextc-'0';
					adv();
					if (nextc >= '0' && nextc < '8') {
						c= 8*c + nextc-'0';
						adv();
					}
				}
				break;

			default: c=nextc; adv(); break;

			}
		}
		if (len >= sizeof buf) {
			err(MESS(6510, "definition string too long"));
			return NULL;
		}
		buf[len++]= c;
	}
	adv();
	buf[len]= '\0';
	return (string) savestr(buf);
}

Hidden string getrep()
{
	char buf[256]; /* Arbitrary limit */
	char c;
	int len= 0;

	if (nextc != QUOTE) {
		err(MESS(6511, "opening string quote not found in representation"));
		return NULL;
	}
	adv();
	while (nextc != QUOTE) {
		if (nextc == '\\')
			adv();
		if (nextc == '\n') {
			err(MESS(6512, "closing string quote not found in representation"));
			return NULL;
		}
		c= nextc;
		adv();
		if (!isprint(c) && c != ' ') {
			err(MESS(6513, "unprintable character in representation"));
			return NULL;
		}
		if (len >= sizeof buf) {
			err(MESS(6514, "representation string too long"));
			return NULL;
		}
		buf[len++]= c;
	}
	adv();
	buf[len]= '\0';
	return savestr(buf);
}

Hidden Procedure get_definition()
{
	string name;
	int d;
	int code;
	string def;
	string rep;
	
	name= getname();
	if (name == NULL)
		return;
	skipspace();
	if (nextc != '=') {
		sprintf(messbuf, GMESS(6515, "Name %s not followed by '='"), name);
		err1(messbuf);
		freemem((ptr) name);
		return;
	}
	d = lookup(name);
	if (d < 0) {
		sprintf(messbuf,
			getmess(MESS(6516, "Unknown command name: %s")), name);
		err1(messbuf);
		freemem((ptr) name);
		return;
	}
	code = deftab[d].code;
	if (code == CANCEL || code == SUSPEND) {
		sprintf(messbuf,
			getmess(MESS(6517, "Cannot rebind %s in keysfile")), name);
		err1(messbuf);
		freemem((ptr) name);
		return;
	}

	adv();
	skipspace();
	def= getstring();
	if (def == NULL) {
		freemem((ptr) name);
		return;
	}
	
	skipspace();
	if (nextc != '=') {
		sprintf(messbuf, GMESS(6518, "No '=' after definition for name %s"), name);
		err1(messbuf);
		freemem((ptr) name);
		freemem((ptr) def);
		return;
	}

	adv();
	skipspace();
	rep= getrep();
	if (rep == NULL) {
		freemem((ptr) name);
		freemem((ptr) def);
		return;
	}
	
	if (!store(code, name, def, rep)) {
		freemem((ptr) name);
		freemem((ptr) def);
		freemem((ptr) rep);
	}
}

Hidden Procedure get_line()
{
	adv();
	skipspace();
	if (nextc != COMMENT && nextc != '\n')
		get_definition();
	while (nextc != '\n')
		adv();
}

#ifdef DUMPKEYS
Visible Procedure dumpkeys(where)
	string where;
{
	int i;
	int w;
	string s;

	putSstr(stdout, "\nDump of key definitions %s.\n\n", where);
	putstr(stdout, "Code    Name            Definition               Representation\n");
	for (i= 0; i < ndefs; ++i) {
		putDstr(stdout, "%04o    ", deftab[i].code);
		if (deftab[i].name != NULL)
			putSstr(stdout, "%-15s ", deftab[i].name);
		else
			putstr(stdout, "                ");
		s= deftab[i].def;
		w= 0;
		if (s != NULL) {
			for (; *s != '\0'; ++s) {
				if (isascii(*s) && (isprint(*s) || *s == ' ')) {
					putchr(stdout, *s);
					w++;
				}
				else {
					putDstr(stdout, "\\%03o", (int)(*s&0377));
					w+= 4;
				}
			}
		}
		else {
			putstr(stdout, "NULL");
			w= 4;
		}
		while (w++ < 25)
			putchr(stdout, ' ');
		s= deftab[i].rep;
		putSstr(stdout, "%s\n", s!=NULL ? s : "NULL");
	}
	putnewline(stdout);
	fflush(stdout);
}
#endif /* DUMPKEYS */

#ifdef KEYS
extern int nharddefs;
#endif

Visible Procedure countdefs()
{
	struct tabent *d;

	d= deftab;
	while (d->name != NULL) {
		++d;
		if (d >= deftab+MAXDEFS)
			syserr(MESS(6519, "too many predefined keys"));
	}
	ndefs= d-deftab;
#ifdef KEYS
	nharddefs= ndefs;
#endif
}

Visible Procedure rd_keysfile()
{
#ifdef KEYS
	saveharddefs();
#endif
	if (keysfile != NULL)
		keysfp= fopen(keysfile, "r");
	else
		keysfp= NULL;
	if (keysfp == NULL) {
		return;
	}
/* process: */
	errcount= 0;
	lcount= 1;
	eof= No;
	do {
		get_line();
		lcount++;
	} while (!eof);
/* */
	fclose(keysfp);
	if (errcount > 0)
		fflush(errfile);
#ifdef DUMPKEYS
	if (kflag)
		dumpkeys("after reading keysfile");
#endif
#ifdef KEYS
	savefiledefs();
#endif
}

#ifndef KEYS

/* Output a named string to the terminal */

Hidden Procedure outstring(name)
	string name;
{
	int i= lookup(name);

	if (i >= 0) {
		string def= deftab[i].def;
		if (def != NULL && *def != '\0') {
			fputs(def, errfile);
			putnewline(errfile);
			fflush(errfile);
		}
	}
}

/* Output the terminal's initialization sequence, if any. */

Visible Procedure initgetc()
{
	outstring("[term-init]");
}


/* Output a sequence, if any, to return the terminal to a 'normal' state. */

Visible Procedure endgetc()
{
	outstring("[term-done]");
}


/* Read a command from the keyboard, decoding composite key definitions. */

Visible int inchar()
{
	int c;
	struct tabent *d, *last;
	char buffer[100];
	int len;

	c= trminput();
	if (c == EOF)
		return c;
	c= cvchar(c);
	last= deftab+ndefs;
	for (d= deftab; d < last; ++d) {
		if (d->code > 0 && d->def != NULL && c == (d->def[0] & 0377))
			break;
	}
	if (d == last) {
		if (isascii(c) && (isprint(c) || c == ' '))
			return c;
		else
			return 0377;
	}
	if (d->def[1] == '\0')
		return d->code;
	buffer[0]= c;
	len= 1;
	for (;;) {
		c= trminput();
		if (c == EOF)
			return EOF;
		buffer[len]= c;
		if (len < sizeof buffer - 1)
			++len;
		for (d= deftab; d < last; ++d) {
			if (d->code > 0 && d->def != NULL
				&& strncmp(buffer, d->def, len) == 0)
				break;
		}
		if (d == last) {
			return 0377; /* Hope this rings a bell */
		}
		if (d->def[len] == '\0')
			return d->code;
	}
}
#endif /* !KEYS */
