#ifndef SoftWire_RegisterAllocator_hpp
#define SoftWire_RegisterAllocator_hpp

#undef free   // Defined by memory manager

#include "Assembler.hpp"

namespace SoftWire
{
	class Optimizer;

	class RegisterAllocator : public Assembler
	{
	protected:
		struct Allocation
		{
			Allocation()
			{
				free();
			}

			void free()
			{
				reference = 0;
				priority = 0;
				partial = 0;
				copyReference = 0;
				copyInstruction = 0;
				loadInstruction = 0;
				referenced = false;
				modified = false;
			}

			OperandREF reference;
			unsigned int priority;
			int partial;     // Number of bytes used, 0/1/2 for general-purpose, 0/4 for SSE, 0 means all

		private:
			friend Optimizer;

			OperandREF copyReference;
			Encoding *copyInstruction;   // For copy propagation
			Encoding *loadInstruction;   // For load-load elimination
			bool referenced;
			bool modified;
		};

	public:
		RegisterAllocator();

		~RegisterAllocator();

		// Register allocation
		const OperandREG8 r8(const OperandREF &ref, bool copy = true);
		const OperandR_M8 m8(const OperandREF &ref);

		const OperandREG16 r16(const OperandREF &ref, bool copy = true);
		const OperandR_M16 m16(const OperandREF &ref);

		const OperandREG32 &r32(const OperandREF &ref, bool copy = true, int partial = 0);
		const OperandR_M32 m32(const OperandREF &ref, int partial = 0);
		const OperandREG32 &allocate(const OperandREG32 &reg, const OperandREF &ref, int partial);
		const OperandREG32 &assign(const OperandREG32 &reg, const OperandREF &ref, bool copy, int partial);
		const OperandREG32 &access(const OperandREG32 &reg);
		void free(const OperandREG32 &reg);
		void spill(const OperandREG32 &reg);

		const OperandMMREG &r64(const OperandREF &ref, bool copy = true);
		const OperandR_M64 m64(const OperandREF &ref);
		const OperandMMREG &allocate(const OperandMMREG &reg, const OperandREF &ref);
		const OperandMMREG &assign(const OperandMMREG &reg, const OperandREF &ref, bool copy);
		const OperandMMREG &access(const OperandMMREG &reg);
		void free(const OperandMMREG &reg);
		void spill(const OperandMMREG &reg);

		const OperandXMMREG &r128(const OperandREF &ref, bool copy = true, bool ss = false);
		const OperandR_M128 m128(const OperandREF &ref, bool ss = false);
		const OperandXMMREG &allocate(const OperandXMMREG &reg, const OperandREF &ref, bool ss);
		const OperandXMMREG &assign(const OperandXMMREG &reg, const OperandREF &ref, bool copy, bool ss);
		const OperandXMMREG &access(const OperandXMMREG &reg);
		void free(const OperandXMMREG &reg);
		void spill(const OperandXMMREG &reg);

		const OperandXMMREG &rSS(const OperandREF &ref, bool copy = true, bool ss = true);
		const OperandXMM32 mSS(const OperandREF &ref, bool ss = true);

		void free(const OperandREF &ref);
		void spill(const OperandREF &ref);

		void freeAll();
		void spillAll();
		void spillMMX();   // Specifically for using FPU after MMX
		void spillMMXcept(const OperandMMREG &reg);   // Empty MMX state but leave one associated

	protected:
		// Get allocation data of corresponding operand
		static Allocation &X32(const OperandREG32 &r32);
		static Allocation &X64(const OperandMMREG &r64);
		static Allocation &X128(const OperandXMMREG &r128);
		static Allocation &X32(Encoding::Reg r32);
		static Allocation &X64(Encoding::Reg r64);
		static Allocation &X128(Encoding::Reg r128);

		// Get register of corresponding operand
		static const OperandREG8 &R8(Encoding::Reg r8);
		static const OperandREG16 &R16(Encoding::Reg r16);
		static const OperandREG32 &R32(Encoding::Reg r32);
		static const OperandMMREG &R64(Encoding::Reg r64);
		static const OperandXMMREG &R128(Encoding::Reg r128);

		// Current allocation data
		static Allocation EAX;
		static Allocation ECX;
		static Allocation EDX;
		static Allocation EBX;
		static Allocation ESI;
		static Allocation EDI;

		static Allocation MM0;
		static Allocation MM1;
		static Allocation MM2;
		static Allocation MM3;
		static Allocation MM4;
		static Allocation MM5;
		static Allocation MM6;
		static Allocation MM7;

		static Allocation XMM0;
		static Allocation XMM1;
		static Allocation XMM2;
		static Allocation XMM3;
		static Allocation XMM4;
		static Allocation XMM5;
		static Allocation XMM6;
		static Allocation XMM7;
	};
}

#endif   // SoftWire_RegisterAllocator_hpp
