/*	Copyright (c) 1982 Michael Landy, Yoav Cohen, and George Sperling

Disclaimer:  No guarantees of performance accompany this software,
nor is any responsibility assumed on the part of the authors.  All the
software has been tested extensively and every effort has been made to
insure its reliability.   */

/* inv.fourtr.c - performs inverse Fourier transform on a sequence.
 *
 * usage - inv.fourtr [-D] [-d] [-c] [-f] [-b] <insequence >outsequence
 *
 * The input sequence can be in byte, short int, int, float, complex, double,
 * or double complex format. For byte, short int, int, float, and complex
 * images, the calculations are done in float by default.  However, by
 * specifying -D or -d, the calculations are done in double precision.  For
 * double, * and double complex images, the calculations are always done in
 * double precision.  The output format may be in either double complex format
 * (the default for double complex calculations), complex format (the default
 * for float calculation, or with -c), double (-d), float (-f) or byte (-b).
 * The -D switch yields double precision complex output when specified alone,
 * but may be specified along with another switch in order to specify double
 * precision calculations with another output format.  When byte format is
 * selected for output, the output is clipped to the range 0-255.
 *
 * The program does not require square input pictures, but the linear
 * dimensions of the picture must be powers of 2.
 *
 * Timing: about 50 sec for computation on a single 128X128 picture.
 *		i/o adds about 50 additional secs. (from & to RK07).
 *
 * To load: cc -o inv.fourtr inv.fourtr.c  -lhipl -lm
 *
 * Yoav Cohen 5/17/82
 * modified for non-square pictures - Michael Landy - 1/13/84
 * modified for double precision - Michael Landy - 2/5/89
 * made dimension limit large - WEJohnston - 9/89
 */

#include <hipl_format.h>
#include <stdio.h>
#include <math.h>
char Progname[]="inv.fourtr";
int Dflag = 0;
int dflag = 0;
int cflag = 0;
int fflag = 0;
int bflag = 0;
char usage[] = "usage: inv.fourtr [-D] [-d] [-c] [-f] [-b]";

main(argc,argv)

int argc;
char *argv[];

{
	struct header hd;
	unsigned int size;
	int nf,nrows,ncols,pf;
	int i,j,frame,index,logrows,logcols;
	float *rvec,*ivec,*cvec,*prvec,*pivec,*pcvec;
	double *drvec,*divec,*dcvec,*pdrvec,*pdivec,*pdcvec;
	char *cpix,*pcpix;
	short int *spix,*pspix;
	int *ipix,*pipix;
	float *fpix,*pfpix;
	float *compix,*compixi,*compixo,*pcompix;
	float a,b,cc;
	double da,db,dcc;
	int clipped;

	clipped=0;
	for (i=1;i<argc;i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
			case 'D':	Dflag++; break;
			case 'd':	dflag++; break;
			case 'c':	cflag++; break;
			case 'f':	fflag++; break;
			case 'b':	bflag++; break;
			default:	perr(usage);
			}
		}
		else
			perr(usage);
	}
	read_header(&hd);
	update_header(&hd,argc,argv);
	nrows=hd.rows; ncols=hd.cols; nf=hd.num_frame;
	pf=hd.pixel_format;
	size=nrows*ncols;
	logrows=logcols= -1;
	for (i=0,j=1;i<20;i++,j+=j) {
		if (j==nrows)logrows=i;
		if (j==ncols)logcols=i;
	}
	if (logrows == -1 || logcols == -1) {
	    fprintf(stderr,"inv.fourtr: array must be of dimensions which\n");
	    perr("	are powers of 2 ");
	}
	if (pf==PFBYTE || pf==PFSHORT || pf==PFINT || pf==PFFLOAT ||
	    pf==PFCOMPLEX) {
		hd.pixel_format=PFCOMPLEX; hd.bits_per_pixel=64;
	}
	else if (pf==PFDOUBLE || pf==PFDBLCOM) {
		Dflag++;
		hd.pixel_format=PFDBLCOM; hd.bits_per_pixel=128;
	}
	else
		perr("invalid input image format");
	if (Dflag && (dflag+cflag+fflag+bflag == 0)) {
		hd.pixel_format=PFDBLCOM; hd.bits_per_pixel=128;
	}
	if (dflag) {
		Dflag++;
		hd.pixel_format=PFDOUBLE; hd.bits_per_pixel=64;
	}
	if (cflag) {
		hd.pixel_format=PFCOMPLEX; hd.bits_per_pixel=64;
	}
	if (fflag) {
		hd.pixel_format=PFFLOAT; hd.bits_per_pixel=32;
	}
	if (bflag) {
		hd.pixel_format=PFBYTE; hd.bits_per_pixel=8;
	}
	if (Dflag) {
		drvec = (double *) halloc(size,sizeof(double));
		divec = (double *) halloc(size,sizeof(double));
		if (pf==PFDBLCOM || (dflag+cflag+fflag+bflag == 0))
			dcvec = (double *) halloc(2*size,sizeof(double));
		cpix = (char *) divec;
		spix = (short int *) divec;
		ipix = (int *) divec;
		fpix = (float *) divec;
		compixi = (float *) divec;
		compixo = (float *) drvec;
		write_header(&hd);
		for (frame=0;frame<nf;frame++) {
			switch (pf) {
			case PFBYTE:	if (pread(0,cpix,size*sizeof(char))
					    != size*sizeof(char))
						perr("error during read");
					pdrvec=drvec; pcpix=cpix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pcpix++ & 0377;
					break;
			case PFSHORT:	if (pread(0,spix,size*sizeof(short int))
					    != size*sizeof(short int))
						perr("error during read");
					pdrvec=drvec; pspix=spix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pspix++;
					break;
			case PFINT:	if (pread(0,ipix,size*sizeof(int))
					    != size*sizeof(int))
						perr("error during read");
					pdrvec=drvec; pipix=ipix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pipix++;
					break;
			case PFFLOAT:	if (pread(0,fpix,size*sizeof(float))
					    != size*sizeof(float))
						perr("error during read");
					pdrvec=drvec; pfpix=fpix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pfpix++;
					break;
			case PFCOMPLEX:	if (pread(0,compixi,2*size*sizeof(float))
					    != 2*size*sizeof(float))
						perr("error during read");
					pdrvec=drvec;pdivec=divec;
					pcompix=compixi;
					for (i=0;i<size;i++) {
						*pdrvec++ = *pcompix++;
						*pdivec++ = *pcompix++;
					}
					break;
			case PFDOUBLE:	if (pread(0,drvec,size*sizeof(double))
					    != size*sizeof(double))
						perr("error during read");
					break;
			case PFDBLCOM:	if (pread(0,dcvec,2*size*sizeof(double))
					    != 2*size*sizeof(double))
						perr("error during read");
					pdrvec=drvec;pdivec=divec;pdcvec=dcvec;
					for (i=0;i<size;i++) {
						*pdrvec++ = *pdcvec++;
						*pdivec++ = *pdcvec++;
					}
			} /* end switch(pf) */
			if (pf!=PFCOMPLEX && pf!=PFDBLCOM) {
				pdivec = divec;
				for (i=0;i<size;i++)
					*pdivec++ = 0.0;
			}
			else {
				pdivec = divec;
				for (i=0;i<size;i++) {
					*pdivec = -(*pdivec);
					pdivec++;
				}
			}
			dfft_2dgen(drvec,divec,logrows,logcols);
			if (bflag) {
				pdrvec=drvec;pcpix=cpix;
				for (i=0;i<size;i++) {
					cc = *pdrvec++;
					if (cc<0.) {
						j=0;clipped++;
					}
					else if (cc>255.) {
						j=255;clipped++;
					}
					else
						j=cc;
					*pcpix++ = j;
				}
				if (write(1,cpix,size*sizeof(char)) !=
				    size*sizeof(char))
					perr("error during write");
				if (clipped) {
					fprintf(stderr,"inv.fourtr: \
						frame %d, %d values clipped\n",
						frame,clipped);
					clipped=0;
				}
			}
			else if (fflag) {
				pfpix=fpix; pdrvec=drvec;
				for (i=0;i<size;i++)
					*pfpix++ = *pdrvec++;
				if (write(1,fpix,size*sizeof(float)) !=
				    size*sizeof(float))
					perr("error during write");
			}
			else if (cflag) {
				pdrvec=drvec;pdivec=divec;pcompix=compixo;
				for (i=0;i<size;i++) {
					*pcompix++ = *pdrvec++;
					*pcompix++ = -(*pdivec++);
				}
				if (write(1,compixo,2*size*sizeof(float)) !=
				    2*size*sizeof(float))
					perr("error during write");
			}
			else if (dflag) {
				if (write(1,drvec,size*sizeof(double)) !=
				    size*sizeof(double))
					perr("error during write");
			}
			else {
				pdrvec=drvec;pdivec=divec;pdcvec=dcvec;
				for (i=0;i<size;i++) {
					*pdcvec++ = *pdrvec++;
					*pdcvec++ = -(*pdivec++);
				}
				if (write(1,dcvec,2*size*sizeof(double)) !=
				    2*size*sizeof(double))
					perr("error during write");
			}
		}
	}
	else {
		rvec = (float *) halloc(size,sizeof(float));
		ivec = (float *) halloc(size,sizeof(float));
		if (pf==PFCOMPLEX || cflag)
			cvec = (float *) halloc(2*size,sizeof(float));
		cpix = (char *) ivec;
		spix = (short int *) ivec;
		ipix = (int *) ivec;
		write_header(&hd);
		for (frame=0;frame<nf;frame++) {
			switch (pf) {
			case PFBYTE:	if (pread(0,cpix,size*sizeof(char))
					    != size*sizeof(char))
						perr("error during read");
					prvec=rvec; pcpix=cpix;
					for (i=0;i<size;i++)
						*prvec++ = *pcpix++ & 0377;
					break;
			case PFSHORT:	if (pread(0,spix,size*sizeof(short int))
					    != size*sizeof(short int))
						perr("error during read");
					prvec=rvec; pspix=spix;
					for (i=0;i<size;i++)
						*prvec++ = *pspix++;
					break;
			case PFINT:	if (pread(0,ipix,size*sizeof(int))
					    != size*sizeof(int))
						perr("error during read");
					prvec=rvec; pipix=ipix;
					for (i=0;i<size;i++)
						*prvec++ = *pipix++;
					break;
			case PFFLOAT:	if (pread(0,rvec,size*sizeof(float))
					    != size*sizeof(float))
						perr("error during read");
					break;
			case PFCOMPLEX:	if (pread(0,cvec,2*size*sizeof(float))
					    != 2*size*sizeof(float))
						perr("error during read");
					prvec=rvec;pivec=ivec;pcvec=cvec;
					for (i=0;i<size;i++) {
						*prvec++ = *pcvec++;
						*pivec++ = *pcvec++;
					}
			} /* end switch(pf) */
			if (pf!=PFCOMPLEX) {
				pivec = ivec;
				for (i=0;i<size;i++)
					*pivec++ = 0.0;
			}
			else {
				pivec = ivec;
				for (i=0;i<size;i++) {
					*pivec = -(*pivec);
					pivec++;
				}
			}
			fft_2dgen(rvec,ivec,logrows,logcols);
			if (bflag) {
				prvec=rvec;pcpix=cpix;
				for (i=0;i<size;i++) {
					cc = *prvec++;
					if (cc<0.) {
						j=0;clipped++;
					}
					else if (cc>255.) {
						j=255;clipped++;
					}
					else
						j=cc;
					*pcpix++ = j;
				}
				if (write(1,cpix,size*sizeof(char)) !=
				    size*sizeof(char))
					perr("error during write");
				if (clipped) {
					fprintf(stderr,"inv.fourtr: \
						frame %d, %d values clipped\n",
						frame,clipped);
					clipped=0;
				}
			}
			else if (fflag) {
				if (write(1,rvec,size*sizeof(float)) !=
				    size*sizeof(float))
					perr("error during write");
			}
			else {
				prvec=rvec;pivec=ivec;pcvec=cvec;
				for (i=0;i<size;i++) {
					*pcvec++ = *prvec++;
					*pcvec++ = -(*pivec++);
				}
				if (write(1,cvec,2*size*sizeof(float)) !=
				    2*size*sizeof(float))
					perr("error during write");
			}
		}
	}
	return(0);
}
