/*	Copyright (c) 1982 Michael Landy, Yoav Cohen, and George Sperling

Disclaimer:  No guarantees of performance accompany this software,
nor is any responsibility assumed on the part of the authors.  All the
software has been tested extensively and every effort has been made to
insure its reliability.   */

/*
 * fourtr.c - performs Fourier transform on a sequence.
 *
 * usage - fourtr [-d] [-s] <insequence >outsequence
 *
 * The input sequence can be in byte, short int, int,
 * float, complex, double, or double complex format. The output is a Fourier
 * transform by default, or a Fourier spectrum if -s is specified.  The
 * default output format depends on the input format and whether -s was
 * specified.  By default, byte, short, int, float, and complex images produce
 * complex transforms and float spectra.  By default, double and double
 * complex images produce double complex transforms and double spectra.
 * Specification of -d produces double complex transforms and double spectra.
 *
 * For spectra, the coefficients are shifted, so that the (0,0) coefficient is
 * near the center of the output picture.
 *
 * Timing: about 50 sec for computation on a single 128X128 picture.
 *		i/o adds about 50 additional secs. (from & to RK07).
 *
 *
 * To load: cc -o fourtr fourtr.c  -lhipl -lm
 *
 * Yoav Cohen 5/14/82
 * modified for non-square images - Michael Landy - 1/13/84
 * added double precision - Michael Landy - 2/5/89
 *
 * The program does not require square input pictures, but the linear
 * dimensions must both be powers of 2.
 * made dimension limit large - WEJohnston - 9/89
 *
 */

#include <hipl_format.h>
#include <stdio.h>
#include <math.h>
char *Progname;
int sflag = 0;
int dflag = 0;
char usage[] = "usage: fourtr [-d] [-s]";

main(argc,argv)

int argc;
char *argv[];

{
	struct header hd;
	unsigned int size;
	int nf,nrows,ncols,pf;
	int i,j,frame,index,logrows,logcols,hnrows,hncols;
	float *rvec,*ivec,*cvec;
	float *prvec,*pivec,*pcvec;
	float *spvec,*pspvec;
	float *compix,*pcompix,*fpix,*pfpix;
	double *drvec,*divec,*dcvec;
	double *pdrvec,*pdivec,*pdcvec;
	double *dspvec,*pdspvec;
	char *cpix,*pcpix;
	short int *spix,*pspix;
	int *ipix,*pipix;
	float a,b;
	double da,db;

	Progname = strsave(*argv);
	read_header(&hd);
	for (i=1;i<argc;i++) {
		if (argv[i][0]=='-') {
			switch(argv[i][1]) {
			case 'd':	dflag++; break;
			case 's':	sflag++; break;
			case 'D':	break;
			default:	perr(usage);
			}
		}
		else
			perr(usage);
	}
	update_header(&hd,argc,argv);
	
	nrows=hd.rows; ncols=hd.cols; nf=hd.num_frame;
	pf=hd.pixel_format;

	size=nrows*ncols;
	hnrows=nrows/2;
	hncols=ncols/2;
	logrows=logcols= -1;
	for (i=0,j=1;i<20;i++,j+=j) {
		if (j==nrows)logrows=i;
		if (j==ncols)logcols=i;
	}
	if (logrows == -1 || logcols == -1) {
		fprintf(stderr,"fourtr: array must be of dimensions which\n");
		perr("	are powers of 2 and < 512");
	}
	if (pf==PFBYTE || pf==PFSHORT || pf==PFINT || pf==PFFLOAT
	    || pf==PFCOMPLEX) {
		if (dflag) {
			hd.pixel_format=(sflag)?PFDOUBLE:PFDBLCOM;
			hd.bits_per_pixel=(sflag)?64:128;
		}
		else {
			hd.pixel_format=(sflag)?PFFLOAT:PFCOMPLEX;
			hd.bits_per_pixel=(sflag)?32:64;
		}
	}
	else if (pf==PFDOUBLE || pf==PFDBLCOM) {
		dflag++;
		hd.pixel_format=(sflag)?PFDOUBLE:PFDBLCOM;
		hd.bits_per_pixel=(sflag)?64:128;
	}
	else
		perr("illegal input image format");

	if (dflag) {
		drvec = (double *) halloc(size,sizeof(double));
		divec = (double *) halloc(size,sizeof(double));
		if (pf == PFDBLCOM || sflag == 0)
			dcvec = (double *) halloc(2*size,sizeof(double));
		if (sflag != 0) {
			if (pf == PFDBLCOM)
				dspvec = dcvec;
			else
				dspvec = (double *) halloc(size,sizeof(double));
		}

		cpix = (char *) divec;
		spix = (short int *) divec;
		ipix = (int *) divec;
		fpix = (float *) divec;
		compix = sflag ? ((float *) dspvec) : ((float *) dcvec);
		
		write_header(&hd);

		for (frame=0; frame<nf; frame++) {
			switch (pf) {
			case PFBYTE:	if (pread(0,cpix,size*sizeof(char)) !=
					    size*sizeof(char))
						perr("error during read");
					pdrvec=drvec; pcpix=cpix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pcpix++ & 0377;
					break;
			case PFSHORT:	if (pread(0,spix,size*sizeof(short int))
					    != size*sizeof(short int))
						perr("error during read");
					pdrvec=drvec; pspix=spix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pspix++;
					break;
			case PFINT:	if (pread(0,ipix,size*sizeof(int))
					    != size*sizeof(int))
						perr("error during read");
					pdrvec=drvec; pipix=ipix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pipix++;
					break;
			case PFFLOAT:	if (pread(0,fpix,size*sizeof(float))
					    != size*sizeof(float))
						perr("error during read");
					pdrvec=drvec; pfpix=fpix;
					for (i=0;i<size;i++)
						*pdrvec++ = *pfpix++;
					break;
			case PFCOMPLEX:	if (pread(0,compix,2*size*sizeof(float))
					    != 2*size*sizeof(float))
						perr("error during read");
					pcompix=compix; pdrvec=drvec;
					pdivec=divec;
					for (i=0;i<size;i++) {
						*pdrvec++ = *pcompix++;
						*pdivec++ = *pcompix++;
					}
					break;
			case PFDOUBLE:	if (pread(0,drvec,size*sizeof(double))
					    != size*sizeof(double))
						perr("error during read");
					break;
			case PFDBLCOM:	if (pread(0,dcvec,2*size*sizeof(double))
					    != 2*size*sizeof(double))
						perr("error during read");
					pdcvec=dcvec;pdrvec=drvec;
					pdivec=divec;
					for (i=0;i<size;i++) {
						*pdrvec++ = *pdcvec++;
						*pdivec++ = *pdcvec++;
					}
					break;
			} /* switch(pf) */

			if (pf!=PFCOMPLEX && pf!=PFDBLCOM) {
				pdivec = divec;
				for (i=0;i<size;i++)
					*pdivec++ = 0.0;
			}
			dfft_2dgen(drvec,divec,logrows,logcols);

			if (sflag==0) {
				pdcvec=dcvec; pdrvec=drvec; pdivec=divec;
				for (i=0;i<size;i++) {
					*pdcvec++ = *pdrvec++ /((double) size);
					*pdcvec++ = *pdivec++ /((double) size);
				}
				if (write(1,dcvec,2*size*sizeof(double)) !=
				    2*size*sizeof(double))
					perr("error during write");
			}
			else {
				pdspvec = dspvec;
				for (i=0;i<nrows;i++)
					for (j=0;j<ncols;j++) {
					    index=((i+hnrows)%nrows)*ncols
						+((j+hncols)%ncols);
					    da = *(divec+index);
					    db = *(drvec+index);
					    *pdspvec++ = sqrt(da*da+db*db)/
						((double) size);
					}
				if (write(1,dspvec,size*sizeof(double)) !=
				    size*sizeof(double))
					perr("error during write");
			}
		}
	}
	else {
		rvec = (float *) halloc(size,sizeof(float));
		ivec = (float *) halloc(size,sizeof(float));
		if (pf == PFCOMPLEX || sflag == 0)
			cvec = (float *) halloc(2*size,sizeof(float));
		if (sflag != 0) {
			if (pf == PFCOMPLEX)
				spvec = cvec;
			else
				spvec = (float *) halloc(size,sizeof(float));
		}

		cpix = (char *) ivec;
		spix = (short int *) ivec;
		ipix = (int *) ivec;
		
		write_header(&hd);

		for (frame=0; frame<nf; frame++) {
			switch (pf) {
			case PFBYTE:	if (pread(0,cpix,size*sizeof(char))
					    != size*sizeof(char))
						perr("error during read");
					prvec=rvec; pcpix=cpix;
					for (i=0;i<size;i++)
						*prvec++ = *pcpix++ & 0377;
					break;
			case PFSHORT:	if (pread(0,spix,size*sizeof(short int))
					    != size*sizeof(short int))
						perr("error during read");
					prvec=rvec; pspix=spix;
					for (i=0;i<size;i++)
						*prvec++ = *pspix++;
					break;
			case PFINT:	if (pread(0,ipix,size*sizeof(int))
					    != size*sizeof(int))
						perr("error during read");
					prvec=rvec; pipix=ipix;
					for (i=0;i<size;i++)
						*prvec++ = *pipix++;
					break;
			case PFFLOAT:	if (pread(0,rvec,size*sizeof(float))
					    != size*sizeof(float))
						perr("error during read");
					break;
			case PFCOMPLEX:	if (pread(0,cvec,2*size*sizeof(float))
					    != 2*size*sizeof(float))
						perr("error during read");
					pcvec=cvec;prvec=rvec;pivec=ivec;
					for (i=0;i<size;i++) {
						*prvec++ = *pcvec++;
						*pivec++ = *pcvec++;
					}
			} /* switch(pf) */

			if (pf!=PFCOMPLEX) {
				pivec = ivec;
				for (i=0;i<size;i++)
					*pivec++ = 0.0;
			}
			fft_2dgen(rvec,ivec,logrows,logcols);

			if (sflag==0) {
				pcvec=cvec; prvec=rvec; pivec=ivec;
				for (i=0;i<size;i++) {
					*pcvec++ = *prvec++ /((float) size);
					*pcvec++ = *pivec++ /((float) size);
				}
				if (write(1,cvec,2*size*sizeof(float)) !=
				    2*size*sizeof(float))
					perr("error during write");
			}
			else {
				pspvec = spvec;
				for (i=0;i<nrows;i++)
					for (j=0;j<ncols;j++) {
					    index=((i+hnrows)%nrows)*ncols
						+((j+hncols)%ncols);
					    a= *(ivec+index); b= *(rvec+index);
					    *pspvec++ = (float)(sqrt(a*a+b*b)/
						((float) size));
					}
				if (write(1,spvec,size*sizeof(float)) !=
				    size*sizeof(float))
					perr("error during write");
			}
		}
	}
	return(0);
}
