/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball, Josh MacDonald, 
 * Copyright (C) 1997-1998 Jay Painter <jpaint@serv.net><jpaint@gimp.org>  
 *
 * GtkCTree widget for GTK+
 * Copyright (C) 1998 Lars Hamann and Stefan Jeske
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include "gtkctree.h"

#define PM_SIZE 8
#define CELL_SPACING 1
#define CLIST_OPTIMUM_SIZE 512

#define ROW_TOP_YPIXEL(clist, row) (((clist)->row_height * (row)) + \
				    (((row) + 1) * CELL_SPACING) + \
				    (clist)->voffset)

static void gtk_ctree_class_init        (GtkCTreeClass  *klass);
static void gtk_ctree_init              (GtkCTree       *ctree);
static gint gtk_ctree_button_press      (GtkWidget      *widget,
					 GdkEventButton *event);

static GtkCListColumn *columns_new      (GtkCList       *clist);
static void draw_row                    (GtkCList       *clist,
					 GdkRectangle   *area,
					 gint            row,
					 GtkCListRow   *clist_row);
static void cell_empty                  (GtkCList      *clist,
					 GtkCListRow   *clist_row,
					 gint           column);
static void cell_set_text               (GtkCList      *clist,
					 GtkCListRow   *clist_row,
					 gint           column,
					 gchar         *text);
static void cell_set_pixmap            (GtkCList      *clist,
					GtkCListRow   *clist_row,
					gint           column,
					GdkPixmap     *pixmap,
					GdkBitmap     *mask);
static void cell_set_pixtext           (GtkCList      *clist,
					GtkCListRow   *clist_row,
					gint           column,
					gchar         *text,
					guint8         spacing,
					GdkPixmap     *pixmap,
					GdkBitmap     *mask);

static GtkCTreeRow *row_new            (GtkCTree      *ctree);
static void row_delete                 (GtkCTree      *ctree,
					GtkCTreeRow   *ctree_row);
static void tree_delete                (GtkCTree      *ctree, 
					GList         *list, 
					gpointer       data);
static void tree_select                (GtkCTree      *ctree, 
					GList         *list, 
					gpointer       data);
static void tree_unselect              (GtkCTree      *ctree, 
					GList         *list, 
				        gpointer       data);
static void real_select_row            (GtkCTree      *ctree,
					GList         *row,
					gint           column);
static void real_unselect_row          (GtkCTree      *ctree,
					GList         *row,
					gint           column);
static void tree_toggle_row            (GtkCTree      *ctree, 
					GList         *row, 
				        gint           column);
static void gtk_ctree_marshal_signal   (GtkObject     *object,
					GtkSignalFunc  func,
					gpointer       func_data,
					GtkArg        *args);

enum
{
  TREE_SELECT_ROW,
  TREE_UNSELECT_ROW,
  LAST_SIGNAL
};

typedef void (*GtkCTreeSignal1) (GtkObject * object,
				 gint arg1,
				 gint arg2,
				 gpointer data);


static GtkContainerClass *parent_class = NULL;
static guint ctree_signals[LAST_SIGNAL] = {0};


/* ----------------------------------------------------------------- */

guint
gtk_ctree_get_type ()
{
  static guint ctree_type = 0;

  if (!ctree_type)
    {
      GtkTypeInfo ctree_info =
      {
	"GtkCTree",
	sizeof (GtkCTree),
	sizeof (GtkCTreeClass),
	(GtkClassInitFunc) gtk_ctree_class_init,
	(GtkObjectInitFunc) gtk_ctree_init,
	(GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL,
      };

      ctree_type = gtk_type_unique (gtk_clist_get_type (), &ctree_info);
    }

  return ctree_type;
}

static void
gtk_ctree_class_init (GtkCTreeClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkCListClass *clist_class;

  object_class = (GtkObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;
  container_class = (GtkContainerClass *) klass;
  clist_class = (GtkCListClass *) klass;

  parent_class = gtk_type_class (gtk_clist_get_type ());

  ctree_signals[TREE_SELECT_ROW] =
    gtk_signal_new ("tree_select_row",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkCTreeClass, tree_select_row),
		    gtk_ctree_marshal_signal,
		    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);
  ctree_signals[TREE_UNSELECT_ROW] =
    gtk_signal_new ("tree_unselect_row",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkCTreeClass, tree_unselect_row),
		    gtk_ctree_marshal_signal,
		    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);

  gtk_object_class_add_signals (object_class, ctree_signals, LAST_SIGNAL);

  widget_class->button_press_event = gtk_ctree_button_press;

  clist_class->select_row = NULL;
  clist_class->unselect_row = NULL;
  clist_class->click_column = NULL;
  clist_class->draw_row = draw_row;
  clist_class->cell_empty = cell_empty;

  klass->tree_select_row = real_select_row;
  klass->tree_unselect_row = real_unselect_row;
}

static void
gtk_ctree_init (GtkCTree *ctree)
{
  ctree->tree_indent = 20;
  ctree->tree_column = 0;
  ctree->selection_last = NULL;
  ctree->draw_lines     = TRUE;
}

void
gtk_ctree_construct (GtkCTree *ctree,
		     gint      columns, 
		     gint      tree_column,
		     gchar    *titles[])
{
  GtkCList *clist;
  int i;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (GTK_IS_CTREE (ctree));
  g_return_if_fail (GTK_CLIST(ctree)->row_mem_chunk == NULL);

  clist = GTK_CLIST (ctree);

  clist->row_mem_chunk = g_mem_chunk_new ("ctree row mem chunk",
					  sizeof (GtkCTreeRow),
					  sizeof (GtkCTreeRow)
					  * CLIST_OPTIMUM_SIZE, 
					  G_ALLOC_AND_FREE);

  clist->cell_mem_chunk = g_mem_chunk_new ("ctree cell mem chunk",
					   sizeof (GtkCell) * columns,
					   sizeof (GtkCell) * columns
					   * CLIST_OPTIMUM_SIZE, 
					   G_ALLOC_AND_FREE);

  clist->columns = columns;
  clist->column = columns_new (clist);
  ctree->tree_column = tree_column;
  gtk_clist_column_button_create (clist, 0);
  gtk_clist_create_scrollbars (clist);

  if (titles)
    {
      GTK_CLIST_SET_FLAGS (clist, CLIST_SHOW_TITLES);
      for (i = 0; i < columns; i++)
	gtk_clist_set_column_title (clist, i, titles[i]);
    }
  else
    {
      GTK_CLIST_UNSET_FLAGS (clist, CLIST_SHOW_TITLES);
    }
}

GtkWidget*
gtk_ctree_new_with_titles (gint   columns, 
			   gint   tree_column,
			   gchar *titles[])
{
  GtkWidget *widget;

  g_return_val_if_fail (columns > 0, NULL);
  g_return_val_if_fail (tree_column >= 0 && tree_column < columns, NULL);

  widget = gtk_type_new (gtk_ctree_get_type ());
  gtk_ctree_construct (GTK_CTREE (widget), columns, tree_column, titles);
  return widget;
}

GtkWidget*
gtk_ctree_new (gint columns, 
	       gint tree_column)
{
  return gtk_ctree_new_with_titles (columns, tree_column, NULL);
}

static void
gtk_ctree_marshal_signal (GtkObject     *object,
			  GtkSignalFunc  func,
			  gpointer       func_data,
			  GtkArg        *args)
{
  GtkCTreeSignal1 rfunc;

  rfunc = (GtkCTreeSignal1) func;

  (*rfunc) (object, GTK_VALUE_INT (args[0]),
	    GTK_VALUE_INT (args[1]), func_data);
}

static gint
gtk_ctree_button_press (GtkWidget      *widget,
			GdkEventButton *event)
{
  GtkCTree *ctree;
  GtkCList *clist;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_CTREE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  ctree = GTK_CTREE (widget);
  clist = GTK_CLIST (widget);

  if (event->window == clist->clist_window)
    {
      GList *work;
      gint x;
      gint y;
      gint row;
      gint column;

      x = event->x;
      y = event->y;
      if (gtk_clist_get_selection_info (GTK_CLIST (clist), 
					x, y, &row, &column))
	{
	  work = g_list_nth (clist->row_list, row);
	  
	  if (GTK_CTREE_ROW (work)->children &&
	      (event->type == GDK_2BUTTON_PRESS ||
	       (gtk_ctree_is_hot_spot (ctree, GTK_CTREE_ROW (work), 
				       row, x, y)
		&& event->button == 1)))
	    {
	      if (GTK_CTREE_ROW (work)->expanded)
		gtk_ctree_collapse (ctree, work);
	      else
		gtk_ctree_expand (ctree, work);
	    }
	  else
	    tree_toggle_row (ctree, work, column);
	}
      return FALSE;
    }
  return 
    (* GTK_WIDGET_CLASS (parent_class)->button_press_event) (widget, event);
}

static void
draw_row (GtkCList     *clist,
	  GdkRectangle *area,
	  gint          row,
	  GtkCListRow  *clist_row)
{
  GtkWidget *widget;
  GtkCTree  *ctree;
  GdkGC *fg_gc; 
  GdkGC *bg_gc;
  GdkRectangle row_rectangle;
  GdkRectangle cell_rectangle; 
  GdkRectangle clip_rectangle;
  GdkRectangle intersect_rectangle;
  GdkRectangle *rect;

  gint i, offset = 0, width, height, pixmap_width = 0;
  gint xsrc, ysrc, xdest, ydest;

  g_return_if_fail (clist != NULL);

  /* bail now if we arn't drawable yet */
  if (!GTK_WIDGET_DRAWABLE (clist))
    return;

  if (row < 0 || row >= clist->rows)
    return;

  widget = GTK_WIDGET (clist);
  ctree  = GTK_CTREE  (clist);

  /* if the function is passed the pointer to the row instead of null,
   * it avoids this expensive lookup */
  if (!clist_row)
    clist_row = (g_list_nth (clist->row_list, row))->data;

  /* rectangle of the entire row */
  row_rectangle.x = 0;
  row_rectangle.y = ROW_TOP_YPIXEL (clist, row);
  row_rectangle.width = clist->clist_window_width;
  row_rectangle.height = clist->row_height;

  /* rectangle of the cell spacing above the row */
  cell_rectangle.x = 0;
  cell_rectangle.y = row_rectangle.y - CELL_SPACING;
  cell_rectangle.width = row_rectangle.width;
  cell_rectangle.height = CELL_SPACING;

  /* rectangle used to clip drawing operations, it's y and height
   * positions only need to be set once, so we set them once here. 
   * the x and width are set withing the drawing loop below once per
   * column */
  clip_rectangle.y = row_rectangle.y;
  clip_rectangle.height = row_rectangle.height;

  /* select GC for background rectangle */
  if (clist_row->state == GTK_STATE_SELECTED)
    {
      fg_gc = widget->style->fg_gc[GTK_STATE_SELECTED];
      bg_gc = widget->style->bg_gc[GTK_STATE_SELECTED];
    }
  else
    {
      if (clist_row->fg_set)
	{
	  gdk_gc_set_foreground (clist->fg_gc, &clist_row->foreground);
	  fg_gc = clist->fg_gc;
	}
      else
	fg_gc = widget->style->fg_gc[GTK_STATE_NORMAL];
	
      if (clist_row->bg_set)
	{
	  gdk_gc_set_foreground (clist->bg_gc, &clist_row->background);
	  bg_gc = clist->bg_gc;
	}
      else
	bg_gc = widget->style->bg_gc[GTK_STATE_PRELIGHT];
    }

  /* draw the cell borders and background */
  if (area)
    {
      if (gdk_rectangle_intersect (area, &cell_rectangle, &intersect_rectangle))
	gdk_draw_rectangle (clist->clist_window,
			    widget->style->white_gc,
			    TRUE,
			    intersect_rectangle.x,
			    intersect_rectangle.y,
			    intersect_rectangle.width,
			    intersect_rectangle.height);

      /* the last row has to clear it's bottom cell spacing too */
      if (clist_row == clist->row_list_end->data)
	{
	  cell_rectangle.y += clist->row_height + CELL_SPACING;

	  if (gdk_rectangle_intersect (area, &cell_rectangle, &intersect_rectangle))
	    gdk_draw_rectangle (clist->clist_window,
				widget->style->white_gc,
				TRUE,
				intersect_rectangle.x,
				intersect_rectangle.y,
				intersect_rectangle.width,
				intersect_rectangle.height);
	}

      if (!gdk_rectangle_intersect (area, &row_rectangle, &intersect_rectangle))
	return;

      if (clist_row->state == GTK_STATE_SELECTED || clist_row->fg_set)
	gdk_draw_rectangle (clist->clist_window,
			    bg_gc,
			    TRUE,
			    intersect_rectangle.x,
			    intersect_rectangle.y,
			    intersect_rectangle.width,
			    intersect_rectangle.height);
      else
	gdk_window_clear_area (clist->clist_window,
			       intersect_rectangle.x,
			       intersect_rectangle.y,
			       intersect_rectangle.width,
			       intersect_rectangle.height);
    }
  else
    {
      gdk_draw_rectangle (clist->clist_window,
			  widget->style->white_gc,
			  TRUE,
			  cell_rectangle.x,
			  cell_rectangle.y,
			  cell_rectangle.width,
			  cell_rectangle.height);

      /* the last row has to clear it's bottom cell spacing too */
      if (clist_row == clist->row_list_end->data)
	{
	  cell_rectangle.y += clist->row_height + CELL_SPACING;

	  gdk_draw_rectangle (clist->clist_window,
			      widget->style->white_gc,
			      TRUE,
			      cell_rectangle.x,
			      cell_rectangle.y,
			      cell_rectangle.width,
			      cell_rectangle.height);     
	}	  

      if (clist_row->state == GTK_STATE_SELECTED || clist_row->fg_set)
	gdk_draw_rectangle (clist->clist_window,
			    bg_gc,
			    TRUE,
			    row_rectangle.x,
			    row_rectangle.y,
			    row_rectangle.width,
			    row_rectangle.height);
      else
	gdk_window_clear_area (clist->clist_window,
			       row_rectangle.x,
			       row_rectangle.y,
			       row_rectangle.width,
			       row_rectangle.height);
    }

  /* iterate and draw all the columns (row cells) and draw their contents */
  for (i = 0; i < clist->columns; i++)
    {
      clip_rectangle.x = clist->column[i].area.x + clist->hoffset;
      clip_rectangle.width = clist->column[i].area.width;

      /* calculate clipping region clipping region */
      if (!area)
	{
	  rect = &clip_rectangle;
	}
      else
	{
	  if (!gdk_rectangle_intersect (area, &clip_rectangle, &intersect_rectangle))
	    continue;
	  rect = &intersect_rectangle;
	}

      /* calculate real width for column justification */
      switch (clist_row->cell[i].type)
	{
	case GTK_CELL_EMPTY:
	  continue;
	  break;

	case GTK_CELL_TEXT:
	  width = gdk_string_width (GTK_WIDGET (clist)->style->font,
				    GTK_CELL_TEXT (clist_row->cell[i])->text);
	  break;

	case GTK_CELL_PIXMAP:
	  gdk_window_get_size (GTK_CELL_PIXMAP (clist_row->cell[i])->pixmap, &width, &height);
	  pixmap_width = width;
	  break;

	case GTK_CELL_PIXTEXT:
	  gdk_window_get_size (GTK_CELL_PIXTEXT (clist_row->cell[i])->pixmap, &width, &height);
	  pixmap_width = width;
	  width += GTK_CELL_PIXTEXT (clist_row->cell[i])->spacing;
	  width += gdk_string_width (GTK_WIDGET (clist)->style->font,
				     GTK_CELL_PIXTEXT (clist_row->cell[i])->text);
	  break;

	case GTK_CELL_WIDGET:
	  /* unimplimented */
	  continue;
	  break;


	default:
	  continue;
	  break;
	}

      if (i == ctree->tree_column)
	width += ((GtkCTreeRow *)clist_row)->level * ctree->tree_indent;
      
      switch (clist->column[i].justification)
	{
	case GTK_JUSTIFY_LEFT:
	  offset = clip_rectangle.x;
	  break;

	case GTK_JUSTIFY_RIGHT:
	  offset = (clip_rectangle.x + clip_rectangle.width) - width;
	  break;

	case GTK_JUSTIFY_CENTER:
	  offset = (clip_rectangle.x + (clip_rectangle.width / 2)) - (width / 2);
	  break;

	case GTK_JUSTIFY_FILL:
	  offset = (clip_rectangle.x + (clip_rectangle.width / 2)) - (width / 2);
	  break;

	default:
	  offset = 0;
	  break;
	};

      /* Draw Text or Pixmap */

      if (i == ctree->tree_column)
	{
	  GList *work;
	  gint xoffset;
	  gint yoffset;
	  gint xcenter;
	  gint ycenter;
	  
	  xsrc = 0;
	  ysrc = 0;
	  xdest = offset + clist_row->cell[i].horizontal + 
	    ((GtkCTreeRow *)clist_row)->level * ctree->tree_indent;
	  ydest = (clip_rectangle.y + (clip_rectangle.height / 2)) - 
	    height / 2 + clist_row->cell[i].vertical;

	  
	  if (xdest < rect->x + rect->width)
	    {
	      if (((GtkCTreeRow *)clist_row)->expanded)
		{
		  gdk_gc_set_clip_mask 
		    (fg_gc, GTK_CELL_PIXTEXT (clist_row->cell[i])->mask);
		  gdk_gc_set_clip_origin (fg_gc, xdest, ydest);
		  gdk_draw_pixmap (clist->clist_window, fg_gc,
				   GTK_CELL_PIXTEXT 
				   (clist_row->cell[i])->pixmap,
				   xsrc, ysrc, xdest, ydest,
				   pixmap_width, height);
		}
	      else
		{
		  gdk_gc_set_clip_mask 
		    (fg_gc, GTK_CELL_PIXTEXT (clist_row->cell[i])->mask);
		  gdk_gc_set_clip_origin (fg_gc, xdest, ydest);
		  gdk_draw_pixmap (clist->clist_window, fg_gc,
				   GTK_CELL_PIXTEXT 
				   (clist_row->cell[i])->pixmap,
				   xsrc, ysrc, xdest, ydest,
				   pixmap_width, height);
		}
	      if (xdest + pixmap_width > rect->x + rect->width)
		{
		  gdk_draw_rectangle (clist->clist_window, bg_gc, TRUE,
				      rect->x + rect->width, ydest+1,
				      xdest + pixmap_width - 
				      (rect->x + rect->width), height-1);
		  
		}
	    }
	  
	  gdk_gc_set_clip_origin (fg_gc, 0, 0);
	  (rect->y)--;
	  (rect->height)++;
	  gdk_gc_set_clip_rectangle (fg_gc, rect);
	  xdest -= ctree->tree_indent;

	  yoffset = (clip_rectangle.height - PM_SIZE) / 2;
	  xoffset = (ctree->tree_indent - PM_SIZE) / 2;
	  ycenter = clip_rectangle.y + (clip_rectangle.height / 2);
	  xcenter = xdest + (ctree->tree_indent / 2);
	    
	  if (ctree->draw_lines)
	    {
	      gdk_draw_line (clist->clist_window, fg_gc, 
			     xcenter,  clip_rectangle.y - 1, xcenter,
			     (((GtkCTreeRow *)clist_row)->sibling) ?
			     clip_rectangle.y + clip_rectangle.height : 
			     ycenter);
	  
	      gdk_draw_line (clist->clist_window, fg_gc, xcenter, ycenter, 
			     xcenter + PM_SIZE / 2 + 2, ycenter);
	    }
	  
	  if (((GtkCTreeRow *)clist_row)->children)
	    {
	      GdkGC *cgc;
	      GdkGC *tgc;
	      
	      if (clist_row->state == GTK_STATE_SELECTED)
		{
		  if (clist_row->fg_set)
		    tgc = clist->fg_gc;
		  else
		    tgc = widget->style->fg_gc[GTK_STATE_NORMAL];
		  cgc = tgc;
		}
	      else 
		{
		  cgc = GTK_WIDGET(clist)->style->fg_gc[GTK_STATE_SELECTED];
		  tgc = fg_gc;
		}
	      
	      gdk_gc_set_clip_rectangle (cgc, rect);
	      
	      gdk_draw_rectangle (clist->clist_window, 
				  GTK_WIDGET(clist)->style->
				  fg_gc[GTK_STATE_SELECTED], TRUE, 
				  xdest + xoffset, clip_rectangle.y + yoffset,
				  PM_SIZE, PM_SIZE);
		
	      gdk_draw_rectangle (clist->clist_window, tgc, FALSE,
				  xdest + xoffset, clip_rectangle.y + yoffset,
				  PM_SIZE, PM_SIZE);
		
	      gdk_draw_line (clist->clist_window, tgc, xdest + xoffset + 2, 
			     ycenter, xdest + xoffset + PM_SIZE - 2, ycenter);
		
	      if (!((GtkCTreeRow *)clist_row)->expanded)
		{
		  gdk_draw_line (clist->clist_window, tgc, xcenter, 
				 clip_rectangle.y + yoffset + 2, xcenter, 
				 clip_rectangle.y + yoffset + PM_SIZE - 2);
		}
	      gdk_gc_set_clip_rectangle (cgc, NULL);
	    }
	  
	  work = ((GtkCTreeRow *)clist_row)->parent;
	  while (work)
	    {
	      xcenter -= ctree->tree_indent;
	      if (ctree->draw_lines && GTK_CTREE_ROW(work)->sibling)
		gdk_draw_line (clist->clist_window, fg_gc, xcenter, 
			       clip_rectangle.y - 1, xcenter,
			       clip_rectangle.y + clip_rectangle.height);
	      work = GTK_CTREE_ROW (work)->parent;
	    }
      
	  offset += pixmap_width + 
	    GTK_CELL_PIXTEXT (clist_row->cell[i])->spacing;
	  (rect->y)++;
	  (rect->height)--;
	  gdk_gc_set_clip_rectangle (fg_gc, rect);
	  
	  /* draw the string */

	  gdk_draw_string (clist->clist_window, 
			   widget->style->font, fg_gc,
			   offset + clist_row->cell[i].horizontal +
			   ((GtkCTreeRow *)clist_row)->level * ctree->tree_indent,
			   row_rectangle.y + clist->row_center_offset + 
			   clist_row->cell[i].vertical,
			   GTK_CELL_PIXTEXT (clist_row->cell[i])->text);
	  
	  gdk_gc_set_clip_rectangle (fg_gc, NULL);
	}
      else
	{

	  switch (clist_row->cell[i].type)
	    {
	    case GTK_CELL_EMPTY:
	      continue;
	      break;
	      
	    case GTK_CELL_TEXT:
	      gdk_gc_set_clip_rectangle (fg_gc, rect);
	      
	      gdk_draw_string (clist->clist_window, 
			       widget->style->font,
			       fg_gc,
			       offset + clist_row->cell[i].horizontal,
			       row_rectangle.y + clist->row_center_offset + 
			       clist_row->cell[i].vertical,
			       GTK_CELL_TEXT (clist_row->cell[i])->text);
	      
	      gdk_gc_set_clip_rectangle (fg_gc, NULL);
	      break;

	    case GTK_CELL_PIXMAP:
	      xsrc = 0;
	      ysrc = 0;
	      xdest = offset + clist_row->cell[i].horizontal;
	      ydest = (clip_rectangle.y + (clip_rectangle.height / 2)) - height / 2 +
		clist_row->cell[i].vertical;

	      if (GTK_CELL_PIXMAP (clist_row->cell[i])->mask)
		{
		  gdk_gc_set_clip_mask (fg_gc, GTK_CELL_PIXMAP (clist_row->cell[i])->mask);
		  gdk_gc_set_clip_origin (fg_gc, xdest, ydest);
		}
	      gdk_draw_pixmap (clist->clist_window,
			       fg_gc,
			       GTK_CELL_PIXMAP (clist_row->cell[i])->pixmap,
			       xsrc, ysrc,
			       xdest,
			       ydest,
			       pixmap_width, height);

	      if (GTK_CELL_PIXMAP (clist_row->cell[i])->mask)
		{
		  gdk_gc_set_clip_origin (fg_gc, 0, 0);
		  gdk_gc_set_clip_mask (fg_gc, NULL);
		}
	      break;

	    case GTK_CELL_PIXTEXT:
	      /* draw the pixmap */
	      xsrc = 0;
	      ysrc = 0;
	      xdest = offset + clist_row->cell[i].horizontal;
	      ydest = (clip_rectangle.y + (clip_rectangle.height / 2)) - height / 2 +
		clist_row->cell[i].vertical;
	      
	      if (GTK_CELL_PIXTEXT (clist_row->cell[i])->mask)
		{
		  gdk_gc_set_clip_mask (fg_gc, GTK_CELL_PIXTEXT (clist_row->cell[i])->mask);
		  gdk_gc_set_clip_origin (fg_gc, xdest, ydest);
		}
              
	      gdk_draw_pixmap (clist->clist_window,
			       fg_gc,
			       GTK_CELL_PIXTEXT (clist_row->cell[i])->pixmap,
			       xsrc, ysrc,
			       xdest,
			       ydest,
			       pixmap_width, height);
	      
	      gdk_gc_set_clip_origin (fg_gc, 0, 0);
	      
	      offset += pixmap_width + GTK_CELL_PIXTEXT (clist_row->cell[i])->spacing;
	  
	      /* draw the string */
	      gdk_gc_set_clip_rectangle (fg_gc, rect);

	      gdk_draw_string (clist->clist_window, 
			       widget->style->font,
			       fg_gc,
			       offset + clist_row->cell[i].horizontal,
			       row_rectangle.y + clist->row_center_offset + 
			       clist_row->cell[i].vertical,
			       GTK_CELL_PIXTEXT (clist_row->cell[i])->text);
	      
	      gdk_gc_set_clip_rectangle (fg_gc, NULL);
	      
	      break;

	    case GTK_CELL_WIDGET:
	      /* unimplimented */
	      continue;
	      break;

	    default:
	      continue;
	      break;
	    }
	}
    }
}

/* CList copies */
static GtkCListColumn*
columns_new (GtkCList *clist)
{
  gint i;
  GtkCListColumn *column;

  column = g_new (GtkCListColumn, clist->columns);

  for (i = 0; i < clist->columns; i++)
    {
      column[i].area.x = 0;
      column[i].area.y = 0;
      column[i].area.width = 0;
      column[i].area.height = 0;
      column[i].title = NULL;
      column[i].button = NULL;
      column[i].window = NULL;
      column[i].width = 0;
      column[i].width_set = FALSE;
      column[i].justification = GTK_JUSTIFY_LEFT;
    }

  return column;
}

GList* 
gtk_ctree_last (GtkCTree *ctree,
		GList    *list)
{
  if (!list) 
    return NULL;

  while (GTK_CTREE_ROW (list)->sibling)
    list = GTK_CTREE_ROW (list)->sibling;
  
  if (GTK_CTREE_ROW (list)->children)
    return gtk_ctree_last (ctree, GTK_CTREE_ROW (list)->children);
  
  return list;
}

gint 
gtk_ctree_find (GtkCTree *ctree,
		GList    *list,
		GList    *child)
{
  while (list)
    {
      if (list == child) 
	return 1;
      if (GTK_CTREE_ROW (list)->children)
	{
	  if (gtk_ctree_find (ctree, GTK_CTREE_ROW (list)->children,
			      child))
	    {
	      if (GTK_CTREE_ROW (list)->expanded) 
		return 1;
	      else
		return -1;
	    }
	}
      list = GTK_CTREE_ROW (list)->sibling;
    }
  return 0;
}

GList*
gtk_ctree_find_row (GtkCTree    *ctree,
		    GList       *list,
		    GtkCTreeRow *row)
{
  GList *work;
  
  while (list)
    {
      if (list->data == (gpointer) row) 
	return list;
      if (GTK_CTREE_ROW (list)->children &&
	  (work = gtk_ctree_find_row 
	   (ctree, GTK_CTREE_ROW (list)->children, row)))
	return work;
      list = GTK_CTREE_ROW (list)->sibling;
    }
  return NULL;
}

GList*
gtk_ctree_last_visible (GtkCTree *ctree,
			GList    *list)
{
  GList * sib;
  
  if (!list)
    return NULL;

  sib = GTK_CTREE_ROW (list)->children;

  if (!sib || !GTK_CTREE_ROW (list)->expanded)
    return list;

  while (GTK_CTREE_ROW (sib)->sibling)
    sib = GTK_CTREE_ROW (sib)->sibling;

  return gtk_ctree_last_visible (ctree, sib);
}

GList* 
gtk_ctree_insert (GtkCTree  *ctree,
		  GList     *parent, 
		  GList     *sibling,
		  gchar     *text[],
		  guint8     spacing,
		  GdkPixmap *pixmap_closed,
		  GdkBitmap *mask_closed,
		  GdkPixmap *pixmap_opened,
		  GdkBitmap *mask_opened,
		  guint      expanded)
{
  GtkCList *clist;
  GtkCTreeRow *new_row;
  GList *work;
  GList *new_list;
  gint visible = FALSE;
  gint i;

  g_return_val_if_fail (ctree != NULL, NULL);
  
  clist = GTK_CLIST (ctree);

  /* create the row */
  new_row = row_new (ctree);
  new_list = g_list_alloc ();
  new_list->data = new_row;

  work = clist->row_list;

  if (parent)
    {
      visible = gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (parent));
      new_row->level = GTK_CTREE_ROW (parent)->level + 1;
      if (visible && GTK_CTREE_ROW (parent)->expanded)
	clist->rows++;
    }
  else
    {
      clist->rows++;
      new_row->level = 1;
    }
  
  new_row->parent = parent;
  new_row->sibling = sibling;

  if (sibling)
    {
      if (sibling == clist->row_list)
	clist->row_list = new_list;
      if (sibling->prev && sibling->prev->next == sibling)
	sibling->prev->next = new_list;
      new_list->prev = sibling->prev;
      new_list->next = sibling;
      sibling->prev = new_list;
      if (parent && GTK_CTREE_ROW (parent)->children == sibling)
	GTK_CTREE_ROW (parent)->children = new_list;
    }
  else
    {
      if (parent)
	work = GTK_CTREE_ROW (parent)->children;
      else
	work = clist->row_list;

      if (work)
	{
	  /* find sibling */
	  while (GTK_CTREE_ROW (work)->sibling)
	    work = GTK_CTREE_ROW (work)->sibling;
	  GTK_CTREE_ROW (work)->sibling = new_list;
	  
	  /* find last visible child of sibling */
	  work = gtk_ctree_last_visible (ctree, work);
	  
	  new_list->next = work->next;
	  if (work->next)
	    work->next->prev = new_list;
	  work->next = new_list;
	  new_list->prev = work;
	}
      else
	{
	  if (parent)
	    {
	      GTK_CTREE_ROW (parent)->children = new_list;
	      new_list->prev = parent;
	      if (GTK_CTREE_ROW (parent)->expanded)
		{
		  new_list->next = parent->next;
		  if (parent->next)
		    parent->next->prev = new_list;
		  parent->next = new_list;
		}
	      else
		new_list->next = NULL;
	    }
	  else
	    {
	      clist->row_list = new_list;
	      new_list->prev = NULL;
	      new_list->next = NULL;
	    }
	}
    }

  if (clist->row_list_end == NULL || clist->row_list_end->next == new_list)
    clist->row_list_end = new_list;

  if (pixmap_closed && mask_closed)
    {
      new_row->pixmap_closed = gdk_pixmap_ref (pixmap_closed);
      new_row->mask_closed   = gdk_pixmap_ref (mask_closed);
    }

  if (pixmap_opened && mask_opened)
    {
      new_row->pixmap_opened = gdk_pixmap_ref (pixmap_opened);
      new_row->mask_opened   = gdk_pixmap_ref (mask_opened);
    }

  if (text)
    for (i = 0; i < clist->columns; i++)
      if (text[i])
	cell_set_text (clist, &(new_row->row), i, text[i]);

  new_row->row.cell[ctree->tree_column].type = GTK_CELL_PIXTEXT;

  GTK_CELL_PIXTEXT (new_row->row.cell[ctree->tree_column])->spacing = spacing;
  new_row->expanded = expanded;

  if (expanded && pixmap_opened)
    {
      GTK_CELL_PIXTEXT (new_row->row.cell[ctree->tree_column])->pixmap = 
	pixmap_opened;
      GTK_CELL_PIXTEXT (new_row->row.cell[ctree->tree_column])->mask = 
	mask_opened;
    }
  else if (!expanded && pixmap_closed)
    {
      GTK_CELL_PIXTEXT (new_row->row.cell[ctree->tree_column])->pixmap = 
	pixmap_closed;
      GTK_CELL_PIXTEXT (new_row->row.cell[ctree->tree_column])->mask = 
	mask_closed;
    }

  if (!GTK_CLIST_FROZEN (clist) && visible)
    gtk_clist_thaw (clist);

  return new_list;
}

gint
gtk_ctree_is_visible (GtkCTree    *ctree, 
		      GtkCTreeRow *ctree_row)
{ 
  GtkCTreeRow *work;

  work = ctree_row;
  while (work->parent && GTK_CTREE_ROW (work->parent)->expanded)
    work = GTK_CTREE_ROW (work->parent);

  if (!work->parent)
    return TRUE;
  return FALSE;
}

void 
gtk_ctree_collapse (GtkCTree *ctree,
		    GList    *list)
{
  GList *work;
  gint level;

  if (!GTK_CTREE_ROW (list)->expanded)
    return;

  GTK_CTREE_ROW (list)->expanded = FALSE;
  level =  GTK_CTREE_ROW (list)->level;

  if (GTK_CTREE_ROW (list)->pixmap_closed)
    {
      GTK_CELL_PIXTEXT 
	(GTK_CTREE_ROW (list)->row.cell[ctree->tree_column])->pixmap = 
	GTK_CTREE_ROW (list)->pixmap_closed;
      GTK_CELL_PIXTEXT 
	(GTK_CTREE_ROW (list)->row.cell[ctree->tree_column])->mask = 
	GTK_CTREE_ROW (list)->mask_closed;
    }

  work = GTK_CTREE_ROW (list)->children;
  if (work)
    {
      gint tmp = 0;

      while (work && GTK_CTREE_ROW (work)->level > level)
	{
	  work = work->next;
	  tmp++;
	}

      if (work)
	{
	  list->next = work;
	  work->prev->next = NULL;
	  work->prev = list;
	}
      else
	{
	  list->next = NULL;
	  GTK_CLIST (ctree)->row_list_end = list;
	}

      if (gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (list)))
	{
	  GTK_CLIST (ctree)->rows -= tmp;
	  if (!GTK_CLIST_FROZEN (ctree))
	    gtk_clist_thaw (GTK_CLIST (ctree));
	}
    }
}

void 
gtk_ctree_expand (GtkCTree *ctree,
		  GList    *list)
{
  GList *work;
  gint level;

  if (GTK_CTREE_ROW (list)->expanded)
    return;

  GTK_CTREE_ROW (list)->expanded = TRUE;
  level =  GTK_CTREE_ROW (list)->level;

  if (GTK_CTREE_ROW (list)->pixmap_opened)
    {
      GTK_CELL_PIXTEXT 
	(GTK_CTREE_ROW (list)->row.cell[ctree->tree_column])->pixmap = 
	GTK_CTREE_ROW (list)->pixmap_opened;
      GTK_CELL_PIXTEXT 
	(GTK_CTREE_ROW (list)->row.cell[ctree->tree_column])->mask = 
	GTK_CTREE_ROW (list)->mask_opened;
    }

  work = GTK_CTREE_ROW (list)->children;
  if (work)
    {
      gint tmp = 0;

      while (work->next)
	{
	  work = work->next;
	  tmp++;
	}

      work->next = list->next;

      if (list->next)
	list->next->prev = work;
      else
	GTK_CLIST (ctree)->row_list_end = work;

      list->next = GTK_CTREE_ROW (list)->children;
      
      if (gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (list)))
	{
	  GTK_CLIST (ctree)->rows += tmp + 1;
	  if (!GTK_CLIST_FROZEN (ctree))
	    gtk_clist_thaw (GTK_CLIST (ctree));
	}
    }
}

void 
gtk_ctree_expand_recursive (GtkCTree *ctree,
			    GList    *list)
{
  GList *work;

  if (!GTK_CTREE_ROW (list)->children)
    return;

  work = GTK_CTREE_ROW (list)->children;
  while (work)
    {
      gtk_ctree_expand_recursive (ctree, work);
      work = GTK_CTREE_ROW (work)->sibling;
    }
  if (!GTK_CTREE_ROW (list)->expanded)
    gtk_ctree_expand (ctree,list);
}

void 
gtk_ctree_collapse_recursive (GtkCTree *ctree,
			      GList    *list)
{
  GList *work;

  if (!GTK_CTREE_ROW (list)->children)
    return;

  if (GTK_CTREE_ROW (list)->expanded)
    gtk_ctree_collapse (ctree,list);

  work = GTK_CTREE_ROW (list)->children;
  while (work)
    {
      gtk_ctree_collapse_recursive (ctree, work);
      work = GTK_CTREE_ROW (work)->sibling;
    }
}

static void
cell_set_text (GtkCList    *clist,
	       GtkCListRow *clist_row,
	       gint         column,
	       gchar       *text)
{
  (GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->cell_empty)
    (clist, clist_row, column);

  if (text)
    {
      if (column == GTK_CTREE (clist)->tree_column) 
	{
	  clist_row->cell[column].type = GTK_CELL_PIXTEXT;
	  GTK_CELL_PIXTEXT (clist_row->cell[column])->text = g_strdup (text);
	}
      else
	{
	  clist_row->cell[column].type = GTK_CELL_TEXT;
	  GTK_CELL_TEXT (clist_row->cell[column])->text = g_strdup (text);
	}
    }
}

static void
cell_set_pixmap (GtkCList    *clist,
		 GtkCListRow *clist_row,
		 gint         column,
		 GdkPixmap   *pixmap,
		 GdkBitmap   *mask)
{
  (GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->cell_empty)
    (clist, clist_row, column);

  if (pixmap)
    {
      if (column == GTK_CTREE (clist)->tree_column) 
	{
	  clist_row->cell[column].type = GTK_CELL_PIXTEXT;
	  GTK_CELL_PIXTEXT (clist_row->cell[column])->pixmap = pixmap;
	  GTK_CELL_PIXTEXT (clist_row->cell[column])->mask = mask;
	}
      else
	{
	  clist_row->cell[column].type = GTK_CELL_PIXMAP;
	  GTK_CELL_PIXMAP (clist_row->cell[column])->pixmap = pixmap;
	  GTK_CELL_PIXMAP (clist_row->cell[column])->mask = mask;
	}
    }
}

static void
cell_set_pixtext (GtkCList    *clist,
		  GtkCListRow *clist_row,
		  gint         column,
		  gchar       *text,
		  guint8       spacing,
		  GdkPixmap   *pixmap,
		  GdkBitmap   *mask)
{
  (GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->cell_empty)
    (clist, clist_row, column);

  if (text && pixmap)
    {
      clist_row->cell[column].type = GTK_CELL_PIXTEXT;
      GTK_CELL_PIXTEXT (clist_row->cell[column])->text = g_strdup (text);
      GTK_CELL_PIXTEXT (clist_row->cell[column])->spacing = spacing;
      GTK_CELL_PIXTEXT (clist_row->cell[column])->pixmap = pixmap;
      GTK_CELL_PIXTEXT (clist_row->cell[column])->mask = mask;
    }
}

void 
gtk_ctree_set_text (GtkCTree *ctree,
		    GList    *child,
		    gint      column,
		    gchar    *text)
{
  GtkCList *clist;
  GtkCTreeRow *ctree_row;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (child != NULL);

  clist = GTK_CLIST (ctree);

  if (column < 0 || column >= clist->columns)
    return;

  ctree_row = child->data;

  /* if text is null, then the cell is empty */
  cell_set_text (clist, &(ctree_row->row), column, text);

  /* redraw the list if it's not frozen */
  if (!GTK_CLIST_FROZEN (clist) && 
      gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (child)))
    {
      GList *work;
      gint row = 0;
      
      work = clist->row_list;
      while (work != child)
	{
	  work = work->next;
	  row++;
	}
      if (gtk_clist_row_is_visible (clist, row) != GTK_VISIBILITY_NONE)
	(GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->draw_row) 
	  (clist, NULL, row, (GtkCListRow *)ctree_row);
    }
}

void 
gtk_ctree_set_pixmap (GtkCTree  *ctree,
		      GList     *child,
		      gint       column,
		      GdkPixmap *pixmap,
		      GdkBitmap *mask)
{
  GtkCList *clist;
  GtkCTreeRow *ctree_row;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (child != NULL);
  g_return_if_fail (pixmap != NULL);

  clist = GTK_CLIST (ctree);

  if (column < 0 || column >= clist->columns)
    return;

  ctree_row = child->data;

  /* if text is null, then the cell is empty */
  gdk_pixmap_ref (pixmap);
  if (mask) 
    gdk_pixmap_ref (mask);

  cell_set_pixmap (clist, &(ctree_row->row), column, pixmap, mask);

  /* redraw the list if it's not frozen */
  if (!GTK_CLIST_FROZEN (ctree) && 
      gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (child)))
    {
      GList *work;
      gint row = 0;

      work = clist->row_list;
      while (work != child)
	{
	  work = work->next;
	  row++;
	}
      if (gtk_clist_row_is_visible (clist, row) != GTK_VISIBILITY_NONE)
	(GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->draw_row) 
	  (clist, NULL, row, (GtkCListRow *)ctree_row);
    }
}

void 
gtk_ctree_set_pixtext (GtkCTree  *ctree,
		       GList     *child,
		       gint       column,
		       gchar     *text,
		       guint8     spacing,
		       GdkPixmap *pixmap,
		       GdkBitmap *mask)
{
  GtkCList *clist;
  GtkCTreeRow *ctree_row;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (child != NULL);
  g_return_if_fail (pixmap != NULL);

  clist = GTK_CLIST (ctree);

  if (column < 0 || column >= clist->columns)
    return;

  ctree_row = child->data;

  /* if text is null, then the cell is empty */
  gdk_pixmap_ref (pixmap);
  if (mask) 
    gdk_pixmap_ref (mask);

  cell_set_pixtext (clist, &(ctree_row->row), column, text, spacing, 
		    pixmap, mask);

  /* redraw the list if it's not frozen */
  if (!GTK_CLIST_FROZEN (ctree) &&
      gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (child)))
    {
      GList *work;
      gint row = 0;
      
      work = clist->row_list;
      while (work != child)
	{
	  work = work->next;
	  row++;
	}
      if (gtk_clist_row_is_visible (clist, row) != GTK_VISIBILITY_NONE)
	(GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->draw_row) 
	  (clist, NULL, row, (GtkCListRow *)ctree_row);
    }
}

void
gtk_ctree_remove (GtkCTree *ctree, 
		  GList    *list)
{
  GtkCList *clist;
  gint rows;
  gint level;
  gint visible;
  GList *work;
  GList *parent;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (GTK_IS_CTREE (ctree));
  g_return_if_fail (list != NULL);

  clist = GTK_CLIST (ctree);
  
  visible = gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (list));

  if (list->next == NULL)
    clist->row_list_end = list->prev;

  /* Korrigiere Liste */
  rows = 0;
  level = GTK_CTREE_ROW (list)->level;
  work = list->next;
  while (work && GTK_CTREE_ROW (work)->level > level)
    {
      work = work->next;
      rows++;
    }

  if (work)
    work->prev = list->prev;

  if (list->prev && list->prev->next == list)
    list->prev->next = work;

  /* Korrigiere Tree */
  parent = GTK_CTREE_ROW (list)->parent;
  if (parent)
    {
      if (GTK_CTREE_ROW (parent)->children == list)
	{
	  GTK_CTREE_ROW (parent)->children = GTK_CTREE_ROW (list)->sibling;
	  if (!GTK_CTREE_ROW (parent)->children && 
	      GTK_CTREE_ROW (parent)->pixmap_closed)
	    {
	      GTK_CTREE_ROW (parent)->expanded = FALSE;
	      GTK_CELL_PIXTEXT 
		(GTK_CTREE_ROW(parent)->row.cell[ctree->tree_column])->pixmap =
		GTK_CTREE_ROW (parent)->pixmap_closed;
	      GTK_CELL_PIXTEXT 
		(GTK_CTREE_ROW (parent)->row.cell[ctree->tree_column])->mask = 
		GTK_CTREE_ROW (parent)->mask_closed;
	    }
	}
      else
	{
	  GList *sibling;

	  sibling = GTK_CTREE_ROW (parent)->children;
	  while (GTK_CTREE_ROW (sibling)->sibling != list)
	    sibling = GTK_CTREE_ROW (sibling)->sibling;
	  GTK_CTREE_ROW (sibling)->sibling = GTK_CTREE_ROW (list)->sibling;
	}
    }
  else
    {
      if (clist->row_list == list)
	clist->row_list = GTK_CTREE_ROW (list)->sibling;
      else
	{
	  GList *sibling;
	  
	  sibling = clist->row_list;
	  while (GTK_CTREE_ROW (sibling)->sibling != list)
	    sibling = GTK_CTREE_ROW (sibling)->sibling;
	  GTK_CTREE_ROW (sibling)->sibling = GTK_CTREE_ROW (list)->sibling;
	}
    }

  gtk_ctree_recursive (ctree, list, GTK_CTREE_FUNC (tree_delete), NULL);

  if (visible)
    {
      clist->rows -= (rows + 1);

      if (!GTK_CLIST_FROZEN (clist))
	gtk_clist_thaw (clist);
    }
}

void
gtk_ctree_select (GtkCTree *ctree, 
		  GList    *list)
{
  g_return_if_fail (ctree != NULL);
  g_return_if_fail (list != NULL);

  gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_SELECT_ROW], list);
}

void
gtk_ctree_unselect (GtkCTree *ctree, 
		    GList    *list)
{
  g_return_if_fail (ctree != NULL);
  g_return_if_fail (list != NULL);

  gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_UNSELECT_ROW], list);
}

void
gtk_ctree_select_recursive (GtkCTree *ctree, 
			    GList    *list)
{
  gtk_ctree_real_select_recursive (ctree, list, TRUE);
}

void
gtk_ctree_unselect_recursive (GtkCTree *ctree, 
			      GList    *list)
{
  gtk_ctree_real_select_recursive (ctree, list, FALSE);
}

void
gtk_ctree_real_select_recursive (GtkCTree *ctree, 
				 GList    *list, 
				 gint      state)
{
  GtkCList *clist;
  gboolean thaw = FALSE;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (list != NULL);

  clist = GTK_CLIST (ctree);

  if (clist->selection_mode !=  GTK_SELECTION_MULTIPLE)
    return;

  if (gtk_ctree_is_visible (ctree, GTK_CTREE_ROW (list)) && 
      !GTK_CLIST_FROZEN (clist))
    {
      gtk_clist_freeze (clist);
      thaw = TRUE;
    }

  if (state)
    gtk_ctree_recursive (ctree, list, GTK_CTREE_FUNC (tree_select), NULL);
  else 
    gtk_ctree_recursive (ctree, list, GTK_CTREE_FUNC (tree_unselect), NULL);
  
  if (thaw)
    gtk_clist_thaw (clist);
}

static void
tree_delete (GtkCTree *ctree, 
	     GList    *list, 
	     gpointer  data)
{
  GtkCList *clist;
  GList *work;

  clist = GTK_CLIST (ctree);

  work = g_list_find (clist->selection, list);
  if (work)
    {
      if (ctree->selection_last && ctree->selection_last == work)
	ctree->selection_last = ctree->selection_last->prev;
      clist->selection = g_list_remove (clist->selection, list);
    }
      
  row_delete (ctree, GTK_CTREE_ROW (list));
  g_list_free_1 (list);
}

static void
tree_select (GtkCTree *ctree, 
	     GList    *list, 
	     gpointer  data)
{
  gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_SELECT_ROW], 
		   list, data);
}

static void
tree_unselect (GtkCTree *ctree, 
	       GList    *list, 
	       gpointer  data)
{
  gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_UNSELECT_ROW], 
		   list, data);
}

void
gtk_ctree_recursive (GtkCTree     *ctree, 
		     GList        *list,
		     GtkCTreeFunc  func,
		     gpointer      data)
{
  GList *work;
  GList *tmp;

  work = GTK_CTREE_ROW (list)->children;
  while (work)
    {
      tmp = GTK_CTREE_ROW (work)->sibling;
      gtk_ctree_recursive (ctree, work, func, data);
      work = tmp;
    }
  (* func) (ctree, list, data);
}

static GtkCTreeRow*
row_new (GtkCTree *ctree)
{
  GtkCList *clist;
  GtkCTreeRow *ctree_row;
  int i;

  clist = GTK_CLIST (ctree);
  ctree_row = g_chunk_new (GtkCTreeRow, clist->row_mem_chunk);
  ctree_row->row.cell = g_chunk_new (GtkCell, clist->cell_mem_chunk);

  for (i = 0; i < clist->columns; i++)
    {
      ctree_row->row.cell[i].type = GTK_CELL_EMPTY;
      ctree_row->row.cell[i].vertical = 0;
      ctree_row->row.cell[i].horizontal = 0;
    }

  ctree_row->row.fg_set  = FALSE;
  ctree_row->row.bg_set  = FALSE;
  ctree_row->row.state   = GTK_STATE_NORMAL;
  ctree_row->row.data    = NULL;
  ctree_row->row.destroy = NULL;

  ctree_row->level         = 0;
  ctree_row->expanded      = FALSE;
  ctree_row->parent        = NULL;
  ctree_row->sibling       = NULL;
  ctree_row->children      = NULL;
  ctree_row->pixmap_closed = NULL;
  ctree_row->mask_closed   = NULL;
  ctree_row->pixmap_opened = NULL;
  ctree_row->mask_opened   = NULL;
  
  return ctree_row;
}

static void
row_delete (GtkCTree    *ctree,
	    GtkCTreeRow *ctree_row)
{
  GtkCList *clist;
  gint i;

  clist = GTK_CLIST (ctree);

  for (i = 0; i < clist->columns; i++)
    (GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->cell_empty)
      (clist, &(ctree_row->row), i);

  if (ctree_row->row.destroy)
    ctree_row->row.destroy (ctree_row->row.data);

  if (ctree_row->pixmap_closed)
    {
      gdk_pixmap_unref (ctree_row->pixmap_closed);
      gdk_bitmap_unref (ctree_row->mask_closed);
    }

  if (ctree_row->pixmap_opened)
    {
      gdk_pixmap_unref (ctree_row->pixmap_opened);
      gdk_bitmap_unref (ctree_row->mask_opened);
    }

  g_mem_chunk_free (clist->cell_mem_chunk, ctree_row->row.cell);
  g_mem_chunk_free (clist->row_mem_chunk, ctree_row);
}

static void
cell_empty (GtkCList    *clist,
	    GtkCListRow *clist_row,
	    gint          column)
{
  switch (clist_row->cell[column].type)
    {
    case GTK_CELL_EMPTY:
      break;
      
    case GTK_CELL_TEXT:
      g_free (GTK_CELL_TEXT (clist_row->cell[column])->text);
      break;
      
    case GTK_CELL_PIXMAP:
      gdk_pixmap_unref (GTK_CELL_PIXMAP (clist_row->cell[column])->pixmap);
      if (GTK_CELL_PIXMAP (clist_row->cell[column])->mask)
          gdk_bitmap_unref (GTK_CELL_PIXMAP (clist_row->cell[column])->mask);
      break;
      
    case GTK_CELL_PIXTEXT:
      g_free (GTK_CELL_PIXTEXT (clist_row->cell[column])->text);
      if (GTK_CTREE (clist)->tree_column == column)
	break;
      gdk_pixmap_unref (GTK_CELL_PIXTEXT (clist_row->cell[column])->pixmap);
      if (GTK_CELL_PIXTEXT (clist_row->cell[column])->mask)
	gdk_bitmap_unref (GTK_CELL_PIXTEXT (clist_row->cell[column])->mask);
      break;

    case GTK_CELL_WIDGET:
      /* unimplimented */
      break;
      
    default:
      break;
    }

  clist_row->cell[column].type = GTK_CELL_EMPTY;
}

gint
gtk_ctree_is_hot_spot (GtkCTree    *ctree, 
		       GtkCTreeRow *tree_row,
		       gint         row, 
		       gint         x, 
		       gint         y)
{
  GtkCellPixText *cell;
  gint xl;
  gint yu;
  
  cell = GTK_CELL_PIXTEXT(tree_row->row.cell[ctree->tree_column]);
  xl = GTK_CLIST (ctree)->column[ctree->tree_column].area.x + 
    GTK_CLIST (ctree)->hoffset 
    + cell->horizontal + (tree_row->level - 1) * ctree->tree_indent 
    + (ctree->tree_indent - PM_SIZE) / 2;
  
  yu = ROW_TOP_YPIXEL (GTK_CLIST (ctree), row) + 
    (GTK_CLIST (ctree)->row_height - PM_SIZE) / 2;

  if (x >= xl && x <= xl + PM_SIZE && y >= yu && y <= yu + PM_SIZE)
    return TRUE;

  return FALSE;
}

static void
real_select_row (GtkCTree *ctree,
		 GList    *row,
                 gint      column)
{
  GtkCTreeRow *ctree_row;
  GtkCList *clist;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (row != NULL);

  clist = GTK_CLIST (ctree);
  ctree_row = row->data;

  if (ctree_row->row.state == GTK_STATE_NORMAL)
    {
      GList *selection;

      ctree_row->row.state = GTK_STATE_SELECTED;

      selection = g_list_alloc ();
      selection->data = row;

      if (ctree->selection_last)
        {
          ctree->selection_last->next = selection;
          selection->prev = ctree->selection_last;
        }

      if (!(ctree->selection_last))
        clist->selection = selection;
      ctree->selection_last = selection;

      if (!GTK_CLIST_FROZEN (clist) && gtk_ctree_is_visible (ctree, ctree_row))
	{
	  GList *work;
	  gint num = 0;
      
	  work = clist->row_list;
	  while (work != row)
	    {
	      work = work->next;
	      num++;
	    }
	  if (gtk_clist_row_is_visible (clist, num) != GTK_VISIBILITY_NONE)
	    (GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->draw_row) 
	      (clist, NULL, num, (GtkCListRow *)ctree_row);
	}
    }
}

static void
real_unselect_row (GtkCTree *ctree,
		   GList    *row,
                   gint      column)
{
  GtkCList *clist;
  GtkCTreeRow *ctree_row;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (row != NULL);

  clist = GTK_CLIST (ctree);
  ctree_row = row->data;

  if (ctree_row->row.state == GTK_STATE_SELECTED)
    {
      ctree_row->row.state = GTK_STATE_NORMAL;
      
      if (ctree->selection_last && ctree->selection_last->data == row)
	ctree->selection_last = ctree->selection_last->prev;
      clist->selection = g_list_remove (clist->selection, row);

      if (!GTK_CLIST_FROZEN (clist) && gtk_ctree_is_visible (ctree, ctree_row))
	{
	  GList *work;
	  gint num = 0;
      
	  work = clist->row_list;
	  while (work != row)
	    {
	      work = work->next;
	      num++;
	    }
	  if (gtk_clist_row_is_visible (clist, num) != GTK_VISIBILITY_NONE)
	    (GTK_CLIST_CLASS (GTK_OBJECT (clist)->klass)->draw_row) 
	      (clist, NULL, num, (GtkCListRow *)ctree_row);
	}
    }
}

static void
tree_toggle_row (GtkCTree *ctree,
		 GList    *row,
		 gint      column)
{
  GtkCList *clist;
  GList *list;
  GList *sel_row;
  GList *selected_row;

  clist = GTK_CLIST (ctree);

  list = clist->selection;
  selected_row = NULL;

  switch (clist->selection_mode)
    {
    case GTK_SELECTION_SINGLE:
      while (list)
	{
	  sel_row = list->data;
	  list = list->next;

	  if (row != sel_row && 
	      GTK_CTREE_ROW (sel_row)->row.state == GTK_STATE_SELECTED)
	    gtk_signal_emit (GTK_OBJECT (ctree), 
			     ctree_signals[TREE_UNSELECT_ROW], sel_row, 
			     column);
	}

      if (row && GTK_CTREE_ROW (row)->row.state == GTK_STATE_SELECTED)
	gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_UNSELECT_ROW], 
			 row, column);
      else
	gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_SELECT_ROW], 
			 row, column);
      break;

    case GTK_SELECTION_BROWSE:
      while (list)
	{
	  sel_row = list->data;
	  list = list->next;

	  if (row  != sel_row && 
	      GTK_CTREE_ROW (sel_row)->row.state == GTK_STATE_SELECTED)
	    gtk_signal_emit (GTK_OBJECT (ctree), 
			     ctree_signals[TREE_UNSELECT_ROW], 
			     sel_row, column);
	}

      gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_SELECT_ROW], 
		       row, column);
      break;


    case GTK_SELECTION_MULTIPLE:
      if (GTK_CTREE_ROW (row)->row.state == GTK_STATE_SELECTED)
	gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_UNSELECT_ROW], 
			 row, column);
      else
	gtk_signal_emit (GTK_OBJECT (ctree), ctree_signals[TREE_SELECT_ROW], 
			 row, column);
      break;


    case GTK_SELECTION_EXTENDED:
      break;

    default:
      break;
    }
}

void 
gtk_ctree_set_draw_lines (GtkCTree *ctree, 
			  gint      draw_lines)
{
  gint old;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (GTK_IS_CTREE (ctree));

  old = ctree->draw_lines;
  ctree->draw_lines = (draw_lines != 0);

  if (old != ctree->draw_lines)
    gtk_widget_queue_resize (GTK_WIDGET (ctree));
}
