/*****************************************************************************/
/* module growlib.c							     */
/*									     */
/* Author: Rene W. Lutz							     */
/*	   Labo Image							     */
/*	   Computing Science Center					     */
/*	   University of Geneva, Switzerland				     */
/* Date:   September 1990						     */
/* Copyright (c) A. Jacot-Descombes, T. Pun, C. Pellegrini, Uni. of Geneva   */
/* (This copyright notice should appear).				     */
/*									     */
/*****************************************************************************/

/*******************************************************************/
/*                                                                 */
/*	      Methode de segmentation par region growing	   */
/*		 (version 1.0 destinee au LaboImage)        	   */
/*								   */
/*******************************************************************/


#include <suntool/sunview.h>
#include <suntool/panel.h>
#include <suntool/canvas.h>
#include <suntool/scrollbar.h>
#include <pixrect/pixrect_hs.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <stdio.h>
#include <math.h>

#include "define.h"
#include "structure.h"
#include "global.h"

extern char *paneltabs[];
extern char *mastertabs[];
extern char *labeltabs[];
extern char *filetabs[];
extern char *canvastabs[];
extern Frame frame;


/********************************************************

		     macros et constantes

********************************************************/


#define DIFF(a,b) fabs ((double) a - (double) b )

#define GLOBAL 0
#define LOCAL 1
#define MOYENNE 2
#define SIGMA 3
#define OFF 0
#define ON 1
#define CON4 0
#define CON8 1
#define OFF 0
#define RI 0
#define IC 1
#define RC 2
#define AND && 


/********************************************************

    procedures externes au module, lien avec LaboImage

********************************************************/


extern void hproc_region_grow();

extern ouverture() ;
extern fermeture() ;
extern Menu     mcurs, mcoord ;
extern int	write_master();
extern int	write_erreur();
extern void	fromto();
extern int	statis() ;
extern void	information() ;
extern int Confirm();


/********************************************************

	procedures du module de segmentation

********************************************************/


/*
	procedures de lien avec LaboImage
*/

caddr_t		init_main_grow_frame() ;
static void	proc_quit_grow() ;


/*	
	procedures de gestions des objets SunView
*/

static void	grow_frame_init() ;
static void	grow_panel_init() ;
static void	grow_canvases_init() ;
static int	trace_off() ;
static int	get_mode() ;
static int	get_connexity() ;
static int	get_display() ;
static int	get_tolerance() ;
static void	store_c_proc() ;
static void	store_r_proc() ;
static void	load_proc() ;
static void	show_image_in_pixwin() ;
static void	clear_screen() ;
static void	start_proc() ;
static void	display_proc() ;
static void	write_message() ;


/*
	procedures de segmentation
*/

static void	fix_parameters() ;
static void	start_segmentation() ;
static void	marquer() ;
static void	contour() ;
static void	segmente_global_8() ;
static void	segmente_global_4() ;
static void	segmente_local_8() ;
static void	segmente_local_4() ;
static void	segmente_moyenne_8() ;
static void	segmente_moyenne_4() ;
static void	segmente_min_max_8() ;
static void	segmente_min_max_4() ;

static void	segmente_sigma_8() ;	/* pas operationnel */


/********************************************************

		variables globales

********************************************************/


/*
	Lien avec LaboImage
*/

extern int flag_break, flag_creer, flag_exec ;

Menu main_grow_menu;

int flag_grow_win_open = FALSE;
int flag_in_grow = FALSE;
int flag_conv, flag_autoconvert ;


/*
	objets SunView
*/

static Frame	    grow_frame;
static Panel	    grow_panel ;
static Canvas	    left_canvas, right_canvas ;
static Pixwin	    *lpw, *rpw, *ppw, *fpw;
static Panel_item   grow_mode_item, grow_tol, grow_trace_item,
		    grow_show_item, grow_store_C_button, 
		    grow_store_R_button, grow_load_button,
		    grow_connexity_item, grow_quit_button, 
		    grow_start_button, grow_message_item;


/*
	table de couleur
*/

static unsigned char red[256] ;
static unsigned char green[256] ;
static unsigned char blue[256] ;


/*
	variables de calcul
*/

int *I, *R, *C ;		/* plans I-mage, R-egion (% 256), C-ontour */
int *Reg ;		        /* plan Reg-ion (regions etiquettees) */

int taille_image ;
int width, height ;
int swidth,bwidth, lastline ;   /* s(b)-width = width -(+) 1 ; pour le
				   calcul des bords ainsi que lastline */

int seuil ;			/* seuil de tolerance */
int init_val ;			/* valeur du pixel initial (seg. moyenne) */
int color;			/* colour d'affichage (% 256) */
int Reg_color ;			/* etiquette de region [1,2,...etc] */

int trace_val ;			/* variable d'etat des toggles */
int d_mode ;

int nombre_tot ;		/* calculs de moyennes et d'ecart-types */
int nombre ;
int mmin, mmax ;		/* pour les segmentation min_max */



/*******************************************************************/
/*								   */
/*		 Procedures de lien avec LaboImage		   */
/*								   */
/*******************************************************************/


/*******************************************************************/
/*								   */
/* nom      : init_main_grow_frame				   */
/*								   */
/* fonction : procedure appellee par le menu de LaboImage qui      */
/*            initialise et ouvre l'outil                          */
/*                                                                 */
/* entrees  : Menu	m                                          */
/*            Menu_item	mi                                         */
/*                                                                 */
/* globales : int flag_bother, flag_help, flag_in_grow		   */
/*	      Frame grow_frame					   */
/*	      Menu	main_grow_menu				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : proc_quit_grow, hproc_region_grow, grow_frame_init   */
/*	      grow_panel_init, grow_canvases_init		   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

caddr_t init_main_grow_frame(m,mi)
Menu	m ;
Menu_item mi ;
{
	static void proc_quit_grow();

	if (flag_bother)
	{
		hproc_region_grow();
		return;
	}
	if (flag_help) 
		hproc_region_grow();
	flag_grow_win_open = TRUE;
	if (flag_in_grow) 
		return ;
	flag_in_grow = TRUE;


	grow_frame_init() ;
	main_grow_menu = (Menu) window_get(grow_frame, WIN_MENU, 0);
	menu_set(main_grow_menu,
			MENU_REMOVE, 1,
			MENU_ACTION_ITEM, "quit*", proc_quit_grow,
			0);
	window_set(grow_frame, WIN_MENU, main_grow_menu,0);
	grow_panel_init() ;
	grow_canvases_init() ;
	window_fit(grow_frame) ;
	window_set(grow_frame, WIN_SHOW, TRUE, 0) ;
    
}


/*******************************************************************/
/*								   */
/* nom      : proc_quit_grow				           */
/*								   */
/* fonction : termine la session avec et quitte l'outil	           */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales : int *R, *reg, *C, flag_grow_win_open, flag_in_grow.  */
/*	      Frame grow_frame.					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : Confirm()						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
proc_quit_grow()
{
	if (Confirm("quit Tool ?") == 1)
	{
		window_destroy(grow_frame);
		flag_grow_win_open = FALSE;
		flag_in_grow = FALSE;
	}
}



/*******************************************************************/
/*								   */
/* nom      : grow_frame_init					   */
/*								   */
/* fonction : initialisartion du cadre de la fenetre		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Frame frame, grow_frame				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
grow_frame_init()
{
    grow_frame =  window_create  (frame , FRAME,
			    FRAME_LABEL, labeltabs[7],
			    WIN_X, 412, 
			    WIN_Y, 100, 
			    FRAME_SHOW_LABEL, TRUE, 
			    0) ;

}


/*******************************************************************/
/*								   */
/* nom      : grow_panel_init					   */
/*								   */
/* fonction : creation du panneau d'application			   */
/*                                                                 */
/* entrees  : ---				                   */
/*                                                                 */
/* globales : Frame grow_frame; Panel grow_panel;		   */
/*	      Panel_item verbose/smooth/masque/step_mode_item,     */
/*			 next_button, message_item ,		   */
/*			 threshold_slider, number_min ;		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
grow_panel_init()
{
    grow_panel = window_create  (grow_frame, PANEL,
			    WIN_SHOW, TRUE,
			    0) ;

    grow_mode_item = panel_create_item(grow_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(0),
			    PANEL_LABEL_STRING, paneltabs[570],
			    PANEL_CHOICE_STRINGS, paneltabs[571], 
			    paneltabs[572], paneltabs[573], paneltabs[574],0,
			    PANEL_LABEL_BOLD, TRUE,
			    0);

    grow_connexity_item = panel_create_item(grow_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(1),
			    PANEL_LABEL_STRING, paneltabs[575],
			    PANEL_CHOICE_STRINGS, "4", "8", 0,
			    PANEL_LABEL_BOLD, TRUE,
			    0);

    grow_tol = panel_create_item(grow_panel, PANEL_TEXT,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(2),
			    PANEL_LABEL_BOLD, TRUE,
                            PANEL_LABEL_STRING, "Tolerance (+/-) :",
			    PANEL_VALUE_DISPLAY_LENGTH, 3,
			    PANEL_VALUE, "10",
			    0) ;

    grow_trace_item = panel_create_item(grow_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(3),
			    PANEL_LABEL_STRING, paneltabs[576],
			    PANEL_CHOICE_STRINGS, paneltabs[577], 
			    paneltabs[578], 0,
			    PANEL_LABEL_BOLD, TRUE,
			    0) ;

    grow_show_item = panel_create_item(grow_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(4),
			    PANEL_LABEL_STRING, paneltabs[579],
			    PANEL_CHOICE_STRINGS, paneltabs[580], 
			    paneltabs[581], paneltabs[582], 0,
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_NOTIFY_PROC, display_proc,
			    0) ;

    grow_store_C_button = panel_create_item(grow_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, 515 ,
			    PANEL_LABEL_Y, ATTR_ROW(0),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(grow_panel,paneltabs[583],9,0),
			    PANEL_NOTIFY_PROC, store_c_proc,

			    0) ;

    grow_store_R_button = panel_create_item(grow_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, 515,
			    PANEL_LABEL_Y, ATTR_ROW(1),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(grow_panel,paneltabs[584],9,0),
			    PANEL_NOTIFY_PROC, store_r_proc,
			    0) ;

    grow_load_button = panel_create_item(grow_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, 515,
			    PANEL_LABEL_Y, ATTR_ROW(2),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(grow_panel,paneltabs[585],9,0),
			    PANEL_NOTIFY_PROC, load_proc,
			    0) ;   		    

    grow_start_button = panel_create_item(grow_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, 515,
			    PANEL_LABEL_Y, ATTR_ROW(3),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(grow_panel,paneltabs[586],9,0),
			    PANEL_NOTIFY_PROC, start_proc,
			    0) ;

    grow_quit_button = panel_create_item(grow_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, 515,
			    PANEL_LABEL_Y, ATTR_ROW(4),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(grow_panel,paneltabs[587],9,0),
			    PANEL_NOTIFY_PROC, proc_quit_grow,
			    0) ;

    grow_message_item = panel_create_item(grow_panel, PANEL_TEXT,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(5),
			    PANEL_VALUE_DISPLAY_LENGTH, 60,
			    0) ;
    window_fit_height(grow_panel) ;

}


/*******************************************************************/
/*								   */
/* nom      : grow_canvases_init				   */
/*								   */
/* fonction : initialise le canevas et les couleurs		   */
/*                                                                 */
/* entrees  : ---				                   */
/*                                                                 */
/* globales : Pixwin *fpw, *lpw, *rpw, *ppw, ; Frame grow_frame;   */
/*	      unsigned char red[], green[], blue[]                 */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
grow_canvases_init()
{
    Scrollbar lbarvert, lbarhoriz,
	      rbarvert, rbarhoriz ;

    int i ;  
  
    left_canvas = window_create (grow_frame, CANVAS,
			    WIN_BELOW, grow_panel,
			    WIN_X, 0, 
			    WIN_WIDTH, 300, 
			    WIN_HEIGHT, 300,
                            CANVAS_AUTO_SHRINK, FALSE,
                            CANVAS_AUTO_CLEAR, TRUE,
                            CANVAS_FIXED_IMAGE, FALSE,
			    CANVAS_RETAINED, TRUE,
	                    WIN_VERTICAL_SCROLLBAR, lbarvert = scrollbar_create(0),
			    WIN_HORIZONTAL_SCROLLBAR, lbarhoriz = scrollbar_create(0),			    0) ;

    window_fit(left_canvas) ;
    
    right_canvas = window_create (grow_frame, CANVAS,
			    WIN_BELOW, grow_panel,
			    WIN_WIDTH, 300, 
			    WIN_HEIGHT, 300, 
                            CANVAS_AUTO_SHRINK, FALSE,
                            CANVAS_AUTO_CLEAR, TRUE,
                            CANVAS_FIXED_IMAGE, FALSE,
			    CANVAS_RETAINED, TRUE,
                            WIN_VERTICAL_SCROLLBAR, rbarvert = scrollbar_create(0),
			    WIN_HORIZONTAL_SCROLLBAR, rbarhoriz = scrollbar_create(0),
			    0) ;

    window_fit(right_canvas) ;

    scrollbar_paint_clear (lbarvert);
    scrollbar_paint_clear (lbarhoriz); 
    scrollbar_paint_clear (rbarvert);
    scrollbar_paint_clear (rbarhoriz); 

    for (i=0 ; i<256 ; i++)
	    red[i] = green[i] = blue[i] = i ;

    lpw = (Pixwin *) canvas_pixwin(left_canvas) ;
    pw_setcmsname(lpw, "LEFTCANVAS") ;
    pw_putcolormap(lpw, 0, 256, red, green, blue) ;
    rpw = (Pixwin *) canvas_pixwin(right_canvas) ;
    pw_setcmsname(rpw, "RIGHTCANVAS") ;
    pw_putcolormap(rpw, 0, 256, red, green, blue) ;
    ppw = (Pixwin *) window_get(grow_panel, WIN_PIXWIN) ;
    pw_setcmsname(ppw, "PANEL") ;
    pw_putcolormap(ppw, 0, 256, red, green, blue) ;
    fpw = (Pixwin *) window_get(grow_frame, WIN_PIXWIN) ;
    pw_setcmsname(fpw, "FRAME") ;
    pw_putcolormap(fpw, 0, 256, red, green, blue) ;
    
}


/*******************************************************************/
/*								   */
/* nom      : trace_off						   */
/*								   */
/* fonction : determine si le masque est affiche		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item masque_item				   */
/*								   */
/* return   : TRUE si "OFF", "FALSE" si "ON"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static int
trace_off()
{
    return( (int) panel_get_value(grow_trace_item)) ;
}


/*******************************************************************/
/*								   */
/* nom      : get_mode  					   */
/*								   */
/* fonction : determine si le lissage du masque doit etre fait	   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item smooth_item				   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/


static int
get_mode()
{
    return( (int) panel_get_value(grow_mode_item)) ;
}


/*******************************************************************/
/*								   */
/* nom      : get_connexity  					   */
/*								   */
/* fonction : determine si le lissage du masque doit etre fait	   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item smooth_item				   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static int
get_connexity()
{
    return((int) panel_get_value(grow_connexity_item)) ;
}


/*******************************************************************/
/*								   */
/* nom      : get_display					   */
/*								   */
/* fonction : determine si le mode verbeux est actif		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item verbose_mode_item			   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/


static int
get_display()
{
    return((int) panel_get_value(grow_show_item)) ;
}


/*******************************************************************/
/*								   */
/* nom      : get_tolerance					   */
/*								   */
/* fonction : determine si le mode pas a pas est actif		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item step_mode_item				   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/


static int
get_tolerance()
{
	return(atoi((char *) panel_get(grow_tol, PANEL_VALUE))) ;
}


/*******************************************************************/
/*								   */
/* nom      : store_c_proc				           */
/*								   */
/* fonction : sauve le plan des contours dans un plan image        */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : int flag_break, *C				   */
/*	      dir_image[], dir_desc[], index_image[]		   */
/*								   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : fromto(), interruption(), statis()		   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/
 
static void
store_c_proc()
{

	fromto(TO, DEFAUT) ;
	if (flag_break) 
	{
		interruption() ;
		return;
	}

	if (dir_image[index_image[1]].image != NULL) 
		free (dir_image[index_image[1]]);

	dir_image[index_image[1]].image =  (unsigned char *) C ;
	dir_desc[index_image[1]] = dir_desc[index_image[0]];
	dir_desc[index_image[1]].type = 2;

	statis (dir_image[index_image[1]].image,
			dir_desc[index_image[1]].type, 
			dir_desc[index_image[1]].nligne,
			dir_desc[index_image[1]].ncolonne,
			&(dir_desc[index_image[1]].mmin),
			&(dir_desc[index_image[1]].mmax), 
			&(dir_desc[index_image[1]].mu),
			&(dir_desc[index_image[1]].ecart));

}


/*******************************************************************/
/*								   */
/* nom      : store_r_proc				           */
/*								   */
/* fonction : sauve le plan des regions dans un plan image         */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : int flag_break, *Reg				   */
/*	      dir_image[], dir_desc[], index_image[]		   */
/*								   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : fromto(), interruption(), statis()		   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
store_r_proc()
{

	fromto(TO, DEFAUT) ;
	if (flag_break) 
	{
		interruption() ;
		return;
	}

	if (dir_image[index_image[1]].image != NULL) 
		free (dir_image[index_image[1]]);

	dir_image[index_image[1]].image = (unsigned char *) Reg ;
	dir_desc[index_image[1]] = dir_desc[index_image[0]];
	dir_desc[index_image[1]].type = 2;

	statis (dir_image[index_image[1]].image,
			dir_desc[index_image[1]].type, 
			dir_desc[index_image[1]].nligne,
			dir_desc[index_image[1]].ncolonne,
			&(dir_desc[index_image[1]].mmin),
			&(dir_desc[index_image[1]].mmax), 
			&(dir_desc[index_image[1]].mu),
			&(dir_desc[index_image[1]].ecart));

}


/*******************************************************************/
/*								   */
/* nom      : load_proc					           */
/*								   */
/* fonction : charcg un plan image			           */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales : int *I, *R, *Reg					   */
/*	      int flag_break, flag_conv, flag_autoconvert	   */
/*	      dir_image[], dir_desc[], index_image[]		   */
/*	      int height, width, taille_image			   */
/*	      Canvas left_canvas, right_canvas			   */
/*	      Pixwin *lpw, *rpw					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : fromto(), interruption(), statis()		   */
/*	      clear_screen(), show_image_in_pixwin(),get_display() */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
load_proc()
{

	int i, *tmp_R, *tmp_C, *tmp_Reg ;

	if ( (I != NULL) && ( (flag_conv == TRUE) || (flag_autoconvert == TRUE)))
	    free(I) ;

	fromto(FROM, DEFAUT) ;
	 if (flag_break) 
	{
		interruption() ;
		return;
	}

	sprintf (buf,"%d --> \n", index_image[0]);
	write_master (buf);

	switch (dir_desc[index_image[0]].type)
	{
		case -1:
		case 0:
		case 1:	flag_conv = TRUE;
			I = (int *) conv_int (index_image[0]);
			break;
		case 2: flag_conv = FALSE;
			I = (int *)dir_image[index_image[0]].image;
			break;
		case 3:
		case 4:
		case 5: flag_autoconvert = TRUE;
			autoconvert_integer(index_image[0]);
			I = (int *) buffer_image[0].image;
			break;
	}

	height = dir_desc[index_image[0]].nligne;
	width = dir_desc[index_image[0]].ncolonne;
	taille_image = width*height ;

	if (R != NULL) 
		R = NULL ;
	if (Reg != NULL) 
		Reg = NULL ;
	if (C != NULL) 
		C = NULL ;

	R = (int *) calloc (taille_image, sizeof(int)) ;
	Reg = (int *) calloc (taille_image, sizeof(int)) ;
	C = (int *) calloc (taille_image, sizeof(int)) ;

	tmp_R = R ;
	tmp_C = C ;
	tmp_Reg = Reg ;

	for(i=0 ; i < taille_image ; i++)
	{
		*tmp_R = *tmp_C = *tmp_Reg = 0 ;
		tmp_R++ ; tmp_C++ ; tmp_Reg++ ;
	}

	window_set(left_canvas, CANVAS_HEIGHT, height, 0) ;
	window_set(left_canvas, CANVAS_WIDTH, width, 0) ;
	window_set(right_canvas, CANVAS_HEIGHT, height, 0) ;
	window_set(right_canvas, CANVAS_WIDTH, width, 0) ;

    switch(get_display())
    {

	case RI :
		clear_screen(lpw) ;
		show_image_in_pixwin(rpw,I) ;
		break ;

	case IC :
		clear_screen(rpw) ;
		show_image_in_pixwin(lpw,I) ;
		break ;

	default : 
		clear_screen(lpw) ;
		clear_screen(rpw) ;
		break ; 
    }			
}


/*******************************************************************/
/*								   */
/* nom      : show_image_in_pixwin			           */
/*								   */
/* fonction : affiche l'image (*image) dans le pixrect (*pix_ptr)  */
/*                                                                 */
/* entrees  : struct_pixwin *pix_ptr				   */
/*            int	    *image                                 */
/*                                                                 */
/* globales : int taille_image, width, height 	  	           */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : write_master(), clear_screen()		           */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
show_image_in_pixwin(pix_ptr,image) 
struct pixwin *pix_ptr ;
int *image ;
{
    unsigned char *d;
    register i=0 ;
    struct pixrect *pix ;

    if(image!=NULL)
    {	
	if ((pix = mem_create (width,height,8)) == NULL)
	{
	    write_master(" --- erreur de creation de masque ---") ;
	    return ;
	} 

	d = (unsigned char *) mpr_d(pix)-> md_image ;

	while (i < taille_image)
	{
		*d = (unsigned char) *image;
		d++ ; 
                image++ ;
		i++ ; 
	}

        pw_rop(pix_ptr, 0, 0, width, height, PIX_SRC, pix, 0, 0) ;
        pr_close(pix) ;
    }
    else
	clear_screen(pix_ptr) ;
}


/*******************************************************************/
/*								   */
/* nom      : clear_screen				           */
/*								   */
/* fonction : efface le pixwin (*pix_ptr)		           */
/*                                                                 */
/* entrees  : struct pixwin *pix_ptr                               */
/*                                                                 */
/* globales : ---						   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
clear_screen(pix_ptr)
struct pixwin *pix_ptr ;
{
    pw_writebackground(pix_ptr, 0, 0, 1024, 1024, PIX_SRC | PIX_COLOR(0)) ;
}


/*******************************************************************/
/*								   */
/* nom      : start_proc				           */
/*								   */
/* fonction : lance la segmentation			           */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales : ---						   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : fix_parameters(), start_segmentation		   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
start_proc()
{
	fix_parameters() ;
	start_segmentation() ;
}


/*******************************************************************/
/*								   */
/* nom      : display_proc				           */
/*								   */
/* fonction : affiche les plans image selon le mode d'affichage    */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales : pixwin	*lpw, *rpw				   */
/*	      int	*C, *R, *I				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : show_image_in_pixwin(), clear_screen()		   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/
 
static void
display_proc()
{
    switch(get_display())
    {
	case RC :
		show_image_in_pixwin(lpw,R) ;
		show_image_in_pixwin(rpw,C) ;
		break ;

	case RI :
		show_image_in_pixwin(lpw,R) ;
		show_image_in_pixwin(rpw,I) ;
		break ;

	case IC :
		show_image_in_pixwin(lpw,I) ;
		show_image_in_pixwin(rpw,C) ;
		break ;

	default : 
		clear_screen(lpw) ;
		clear_screen(rpw) ;
		break ; 
    }			

}


/*******************************************************************/
/*								   */
/* nom      : write_message				           */
/*								   */
/* fonction : ecrit le message "str" dans le panneau principal     */
/*                                                                 */
/* entrees  : char *str                                            */
/*                                                                 */
/* globales : Panel_item message_item;				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
write_message(str) 
char *str ;
{
    panel_set(grow_message_item, PANEL_VALUE, str, 0);
}


/*******************************************************************/
/*								   */
/*			Procedures de segmentation		   */
/*								   */
/*******************************************************************/


/*******************************************************************/
/*								   */
/* nom      : fix_parameters				           */
/*								   */
/* fonction : effectue les initialisations		           */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales : int seuil, width, swidth, bwidth, height, lastline   */
/*                taille_image, d_mode, trace_val                  */
/*                *R, *C                                           */
/*                                                                 */
/* return   : ---						   */
/*								   */
/* routines : get_display(), get_tolerance(), trace_off()          */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
fix_parameters()
{
	int i, *tmp_R, *tmp_C ;

	seuil = get_tolerance() + 1 ;
	swidth = width - 1 ;
	bwidth = width + 1 ;
	lastline = taille_image - width ;
	d_mode = get_display() ;

	R = (int *) calloc (taille_image, sizeof(int));
	C = (int *) calloc (taille_image, sizeof(int));

	tmp_R = R ;
	tmp_C = C ;

	for(i=0 ; i < taille_image ; i++)
	{
		*tmp_R = *tmp_C = 0 ;
		tmp_R++ ; tmp_C++ ;
	}
	
	trace_val = !trace_off();
}


/*******************************************************************/
/*								   */
/* nom      : start_segmentation			           */
/*								   */
/* fonction : appelle les routines de segmentation	           */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales : int Reg_color, taille_image, init_val, mmin, mmax	   */
/*		  color, trace_val 				   */
/* return   : ---						   */
/*								   */
/* routines : get_mode(), get_connexity(), display_proc()          */
/*	      segmente_global_4/8(), segmente_local_4/8()	   */
/*	      segmente_moyenne_4/8(), segmente_min_max_4/8()	   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
start_segmentation()
{
	int i = 0 ,
	    mode, connex8,
	    old ;

	mode = get_mode() ;
	connex8 = get_connexity() ;
	Reg_color = 0 ;

        if(mode == GLOBAL)
	{
	     while(i<taille_image)
	     {
	        old = TRUE ;
		while((i < taille_image) && (old))
		    old = R[i++] ;
 
	        init_val = I[i - 1] ;
		Reg_color += 1 ;
		color = 100 + (rand() % 156) ;
		if (connex8)
			segmente_global_8(i - 1) ;
		else
			segmente_global_4(i - 1) ;
	    }
	}
	else if(mode == LOCAL)
	{
	     while(i<taille_image)
	     {
	        old = TRUE ;
		while((i < taille_image) && (old))
		    old = R[i++] ;

		Reg_color += 1 ;
		color = 100 + rand() % 156 ;
		if (connex8)    
			segmente_local_8(i - 1) ;
		else
			segmente_local_4(i - 1) ;

	     }		
	}
	else if (mode == MOYENNE)
	{
	
	     while(i<taille_image)
	     {
	        old = TRUE ;
		while((i < taille_image) && (old))	
		    old = R[i++] ;
		Reg_color += 1 ;
		color = 100 + rand() % 156 ;    
		nombre_tot = I[i-1] ;
		nombre = 1 ;
		if (connex8)
			segmente_moyenne_8(i-1) ;
		else
			segmente_moyenne_4(i-1) ;
	     }
	}
	else
        {
	     while(i<taille_image)
	     {
	        old = TRUE ;
		while((i < taille_image) && (old))	
		    old = R[i++] ;
		mmin = mmax = I[i-1] ;

		color = 100 + rand() % 156 ;    
		if (connex8)				
			segmente_min_max_8(i-1) ;		
		else	
			segmente_min_max_4(i-1) ;
	     }
	}

	if (!trace_val)
		display_proc();
}


/*******************************************************************/
/*								   */
/* nom      : marquer					           */
/*								   */
/* fonction : marque le nouveau point accepte dans la region       */
/*                                                                 */
/* entrees  : int pos (position dans l'image)	                   */
/*                                                                 */
/* globales : int *R, *Reg					   */
/*		  Reg_color, color, trace_val, d_mode, width	   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
marquer(pos)
int pos ;
{
	R[pos] = color ;
	Reg[pos] = Reg_color ;
	if(trace_val && ((d_mode == RC) || (d_mode == RI)))
		pw_put(lpw, pos % width, (int) pos/width,color) ; 
}


/*******************************************************************/
/*								   */
/* nom      : contour					           */
/*								   */
/* fonction : marque le point dont aucun voisin n'est "merge-able" */
/*                                                                 */
/* entrees  : int pos (position dans l'image)	                   */
/*                                                                 */
/* globales : int *R, *C					   */
/*		  trace_val, d_mode, width			   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
contour(pos)
int pos ;
{
	if(!R[pos])
	{
	    C[pos] = 255;
	    if(trace_val && ((d_mode == IC) || (d_mode == RC)))
		pw_put(rpw, pos % width, (int) pos/width,255) ; 
	}
}


/*******************************************************************/
/*								   */
/* nom      : segmente_global_8					   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 8 et un seuil de tolerance fixe par	   */
/*	      rapport au point initial				   */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          init_val, swidth, bwidth, seuil 		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
segmente_global_8(pos)
int pos ;
{
	int i,j,
	    nofirst, nolast,
	    p[50000],r ;

	p[0] = pos ;
	r = 0 ;
        i = 0 ;
	
    while(r>-1)
    {
	if (pos > swidth)
	{
	    j = pos - bwidth ;
	
	    if((nolast = ((j % width) != swidth)) &&
	          (!R[j]) &&
	          (DIFF(init_val,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
		(DIFF(init_val,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    j++ ;

	    if( (nofirst = ((j % width) != 0)) &&
	        (!R[j]) &&	
	        (DIFF(init_val,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	}
	    
	j = pos - 1 ;

	if( (nolast) &&
	    (!R[j]) &&	
	    (DIFF(init_val,I[j]) < seuil))
	{
	   p[r++] = j ;
	   marquer(j) ;
	}

	j+=2 ;

	if( (nofirst) &&
	    (!R[j]) &&	
	    (DIFF(init_val,I[j]) < seuil))
	{
	    p[r++] = j ;
	    marquer(j) ;
	}

	if(pos < lastline)
	{
	    j = pos + swidth ;

	    if( (nolast) &&
	        (!R[j]) &&
	        (DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    
	    j++ ;
	    if(	(!R[j]) &&
	        (DIFF(init_val,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    j++ ;

	    if( (nofirst) &&
	        (!R[j]) &&	
	        (DIFF(init_val,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }	    	    
	}
    
        if (i==r)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	r-- ;
	i = r ;
	pos = p[r] ;
    } /* while */
}


/*******************************************************************/
/*								   */
/* nom      : segmente_global_4					   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 4 et un seuil de tolerance fixe par	   */
/*	      rapport au point initial				   */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          init_val, swidth, bwidth, seuil 		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
segmente_global_4(pos)
int pos ;
{
	int i,j,
	    p[50000],r ;

	p[0] = pos ;
	r = 0 ;
        i = 0 ;
	
    while(r>-1)
    {
	if (pos > swidth)
	{
	    j = pos - width ;

	    if( (!R[j]) &&
		(DIFF(init_val,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	}

	j = pos - 1  ;

	if( ((j % width) != swidth) &&
	    (!R[j]) &&
	    (DIFF(init_val,I[j]) < seuil))
	{
	    p[r++] = j ;
	    marquer(j) ;
	}

	j+= 2 ;
	    
	if( ((j % width) != 0) &&
	     (!R[j]) &&	
	     (DIFF(init_val,I[j]) < seuil))
	{
		p[r++] = j ;
		marquer(j) ;
	}
	    
	if(pos < lastline)
	{
	    j = pos + width ;

	    if(	(!R[j]) &&
	        (DIFF(init_val,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	}
    
        if (i==r)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	r-- ;
	i = r ;
	pos = p[r] ;
    } /* while */
}


/*******************************************************************/
/*								   */
/* nom      : segmente_local_8					   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 8 et un seuil local (gradient)		   */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          swidth, bwidth, seuil				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
segmente_local_8(pos)
int pos ;
{
	int i,j,
	    nofirst, nolast,
	    p[50000],r ;

	p[0] = pos ;
	r = 0 ;
        i = 0 ;
	
    while(r>-1)
    {	
	if (pos > swidth)
	{
	    j = pos - bwidth ;
	
	    if( (nolast = ((j % width) != swidth)) &&
	        (!R[j]) &&
	        (DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    
	    j++ ;
	    if(	(!R[j]) &&
	        (DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    j++ ;

	    if( (nofirst = ((j % width) != 0)) &&
	        (!R[j]) &&	
	        (DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	}
	    
	j = pos - 1 ;

	if( (nolast) &&
	    (!R[j]) &&
	    (DIFF(I[pos],I[j]) < seuil))
	{
	   p[r++] = j ;
	   marquer(j) ;
	}

	j+=2 ;

	if( (nofirst) &&
	    (!R[j]) &&	
	    (DIFF(I[pos],I[j]) < seuil))
	{
	    p[r++] = j ;
	    marquer(j) ;
	}

	if(pos < lastline)
	{
	    j = pos + swidth ;

	    if( (nolast) &&
	        (!R[j]) &&
	        (DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
		(DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	    j++ ;

	    if( (nofirst) &&
	        (!R[j]) &&	
	        (DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }	    	    
	}

        if (i==r)
	{
	    contour(pos);
	    marquer(pos) ;
	}
        r-- ;
	i = r ;
        pos = p[r] ;
    } /* while */
}


/*******************************************************************/
/*								   */
/* nom      : segmente_local_4					   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 4 et un seuil local (gradient)		   */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          swidth, bwidth, seuil				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
segmente_local_4(pos)
int pos ;
{
	int i,j,
	    p[50000],r ;

	p[0] = pos ;
	r = 0 ;
        i = 0 ;
	
    while(r>-1)
    {
	if (pos > swidth)
	{
	    j = pos - width ;

	    if( (!R[j]) &&
		(DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	}

	j = pos - 1  ;

	if( ((j % width) != swidth) &&
	    (!R[j]) &&
	    (DIFF(I[pos],I[j]) < seuil))
	{
	    p[r++] = j ;
	    marquer(j) ;
	}

	j+= 2 ;
	    
	if( ((j % width) != 0) &&
	     (!R[j]) &&	
	     (DIFF(I[pos],I[j]) < seuil))
	{
		p[r++] = j ;
		marquer(j) ;
	}
	    
	if(pos < lastline)
	{
	    j = pos + width ;

	    if(	(!R[j]) &&
	        (DIFF(I[pos],I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
	    }
	}
    
        if (i==r)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	r-- ;
	i = r ;
	pos = p[r] ;
    } /* while */
}


/*******************************************************************/
/*								   */
/* nom      : segmente_moyenne_8				   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 8 et un seuil de tolerance fixe sur	   */
/*	      la moyenne (mise a jour a chaque ajout de point)	   */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          nombre_tot, nombre, swidth, bwidth, seuil 	   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/


static void
segmente_moyenne_8(pos)
int pos ;
{
	int i,j,
	    nofirst, nolast,
	    p[50000],r ;

	p[0] = pos ;
	r = 0 ;
        i = 0 ;
	
    while(r>-1)
    {	
	if (pos > swidth)
	{
	    j = pos - bwidth ;
	
	    if( (nolast = ((j % width) != swidth)) &&
	        (!R[j]) &&
	        (DIFF(nombre_tot/nombre , I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
		(DIFF( nombre_tot/nombre,I[j]) < seuil))
	    {
		p[r] = j ;
		marquer(j) ;
	        r++ ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }
	    j++ ;

	    if( (nofirst = ((j % width) != 0)) &&
	        (!R[j]) &&	
	        (DIFF( nombre_tot/nombre,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }
	}
	    
	j = pos - 1 ;

	if( (nolast) &&
	    (!R[j]) &&
	    (DIFF( nombre_tot/nombre,I[j]) < seuil))
	{
	   p[r++] = j ;
	   marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	}

	j+=2 ;

	if( (nofirst) &&
	    (!R[j]) &&	
	    (DIFF( nombre_tot/nombre,I[j]) < seuil))
	{
	    p[r++] = j ;
	    marquer(j) ;
	    nombre++ ;
	    nombre_tot+=I[j] ;
	}

	if(pos < lastline)
	{
	    j = pos + swidth ;

	    if( (nolast) &&
	        (!R[j]) &&
	        (DIFF( nombre_tot/nombre,I[j]) < seuil))
	    {
		p[r] = j ;
		marquer(j) ;
	        r++ ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
		(DIFF( nombre_tot/nombre,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }
	    j++ ;

	    if( (nofirst) &&
	        (!R[j]) &&	
	        (DIFF( nombre_tot/nombre,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }	    	    
	}

        if (i==r)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	r-- ;
	i = r ;
	pos = p[r] ;
    } /* while */
}


/*******************************************************************/
/*								   */
/* nom      : segmente_moyenne_4				   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 4 et un seuil de tolerance fixe sur	   */
/*	      la moyenne (mise a jour a chaque ajout de point)	   */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          nombre_tot, nombre, swidth, bwidth, seuil 	   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
segmente_moyenne_4(pos)
int pos ;
{
	int i,j,
	    p[50000],r ;

	p[0] = pos ;
	r = 0 ;
        i = 0 ;
	
    while(r>-1)
    {
	if (pos > swidth)
	{
	    j = pos - width ;

	    if( (!R[j]) &&
	        (DIFF( nombre_tot/nombre,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }
	}

	j = pos - 1  ;

	if( ((j % width) != swidth) &&
	    (!R[j]) &&
	    (DIFF( nombre_tot/nombre,I[j]) < seuil))
	{
	    p[r++] = j ;
	    marquer(j) ;
	    nombre++ ;
	    nombre_tot+=I[j] ;
	}

	j+= 2 ;
	    
	if( ((j % width) != 0) &&
	     (!R[j]) &&	
	     (DIFF( nombre_tot/nombre,I[j]) < seuil))
	{
		p[r++] = j ;
		marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	}
	    
	if(pos < lastline)
	{
	    j = pos + width ;

	    if(	(!R[j]) &&
	        (DIFF( nombre_tot/nombre,I[j]) < seuil))
	    {
		p[r++] = j ;
		marquer(j) ;
		nombre++ ;
		nombre_tot+=I[j] ;
	    }
	}
    
        if (i==r)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	r-- ;
	i = r ;
	pos = p[r] ;
    } /* while */
}


/*******************************************************************/
/*								   */
/* nom      : segmente_min_max_8				   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 8 et un seuil de tolerance fixe sur	   */
/*	      la difference maximum entre le min et le max de	   */
/*            la region                                            */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          mmin, mmax, swidth, bwidth, seuil 		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
segmente_min_max_8(pos)
int pos ;
{
	int i,j,
	    nofirst, nolast,
	    p[50000],k, nb ;

    p[0] = pos ;
    k = 0 ;
    i = 0 ;
	
    while(k>-1)
    {	
	if (pos > swidth)
	{
	    j = pos - bwidth ;
	
	    if( (nolast = ((j % width) != swidth)) &&
	        (!R[j]) &&
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }
	    j++ ;

	    if( (nofirst = ((j % width) != 0)) &&
	        (!R[j]) &&	
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }
	}
	    
	j = pos - 1 ;

	if( (nolast) &&
	    (!R[j]) &&
	    ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	{
	   p[k++] = j ;
	   marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	}

	j+=2 ;

	if( (nofirst) &&
	    (!R[j]) &&	
	    ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	{
	    p[k++] = j ;
	    marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	}

	if(pos < lastline)
	{
	    j = pos + swidth ;

	    if( (nolast) &&
	        (!R[j]) &&
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }
	    j++ ;

	    if( (nofirst) &&
	        (!R[j]) &&	
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }	    	    
	}

        if (i==k)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	k-- ;
	i = k ;
	pos = p[k] ;
    }
}


/*******************************************************************/
/*								   */
/* nom      : segmente_min_max_4				   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 4 et un seuil de tolerance fixe sur	   */
/*	      la difference maximum entre le min et le max de	   */
/*            la region                                            */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *R, *I					   */
/*	          mmin, mmax, swidth, bwidth, seuil 		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
segmente_min_max_4(pos)
int pos ;
{
	int i,j,
	    p[50000],k ;

	p[0] = pos ;
	k = 0 ;
        i = 0 ;
	
    while(k>-1)
    {
	if (pos > swidth)
	{
	    j = pos - width ;

	    if( (!R[j]) &&
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <=
		seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }
	}

	j = pos - 1  ;

	if( ((j % width) != swidth) &&
	    (!R[j]) &&
	    ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	{
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	}

	j+= 2 ;
	    
	if( ((j % width) != 0) &&
	     (!R[j]) &&	
	     ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	{
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	}
	    
	if(pos < lastline)
	{
	    j = pos + width ;

	    if(	(!R[j]) &&
	        ( (DIFF(mmin , I[j]) <= seuil) && (DIFF(mmax, I[j]) <= seuil)))
	    {
		p[k++] = j ;
		marquer(j) ;

		if (I[j] < mmin)
			mmin = I[j] ;
		else if (I[j] > mmax)
			mmax = I[j] ;
	    }
	}
    
        if (i==k)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	k-- ;
	i = k ;
	pos = p[k] ;
    } /* while */
}


/*******************************************************************/
/*								   */
/* nom      : segmente_sigma_8					   */
/*								   */
/* fonction : segmentation par croissance de regions avec une	   */
/*	      connexite 8 et une tolerance sur l'ecart-type	   */
/*                                                                 */
/*            Cette procedure n'est pas encore operationnelle      */
/*                                                                 */
/* entrees  : int pos (position du point de depart)	           */
/*                                                                 */
/* globales : int *I						   */
/*		  width, swidth,				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : contour(), marquer()				   */
/*								   */
/* Made by Rene W. LUTZ 24.09.1990				   */
/*******************************************************************/

static void
segmente_sigma_8(pos)
int pos ;
{
	int i,j,
	    nofirst, nolast,
	    p[50000],k, nb ;

	float	t,q,r,mu, tol ;
	double testval ;

	t = 0.0 ;
	mu = 0.0;
	nb = 1 ;
	p[0] = pos ;
	k = 0 ;
        i = 0 ;

	q = (float) I[pos] - mu ;
        r = q/nb ;
	mu += r ;
	t = 2;
	nb++;
	
    while(k>-1)
    {	
	if (pos > swidth)
	{
	    j = pos - bwidth ;
	
	    if( (nolast = ((j % width) != swidth)) &&
	        (!R[j]) &&
	        (DIFF(mu , I[j]) <= tol * sqrt((t/(nb - 1)))))
	    {
		p[k++] = j ;
		marquer(j) ;

		q = (float) I[j] - mu ;
                r = q/nb ;
		mu += r ;
		t += (nb - 1) * q * r ;
		nb++;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
	        (DIFF(mu , I[j]) <= tol * sqrt(t/(nb - 1))))
	    {
		p[k++] = j ;
		marquer(j) ;

		q = (float) I[j] - mu ;
                r = q/nb ;
		mu += r ;
		t += (nb - 1) * q * r ;
		nb++;
	    }
	    j++ ;

	    if( (nofirst = ((j % width) != 0)) &&
	        (!R[j]) &&	
	        (DIFF(mu , I[j]) <= tol *sqrt(t/(nb - 1))))
	    {
		p[k++] = j ;
		marquer(j) ;

		q = (float) I[j] - mu ;
                r = q/nb ;
		mu += r ;
		t += (nb - 1) * q * r ;
		nb++;
	    }
	}
	    
	j = pos - 1 ;

	if( (nolast) &&
	    (!R[j]) &&
	    (DIFF(mu , I[j]) <= tol * sqrt(t/(nb - 1))))
	{
	   p[k++] = j ;
	   marquer(j) ;

	    q = (float) I[j] - mu ;
            r = q/nb ;
	    mu += r ;
	    t += (nb - 1) * q * r ;
	    nb++;
	}

	j+=2 ;

	if( (nofirst) &&
	    (!R[j]) &&	
	    (DIFF(mu , I[j]) <= tol * sqrt(t/(nb - 1))))
	{
	    p[k++] = j ;
	    marquer(j) ;

	    q = (float) I[j] - mu ;
            r = q/nb ;
	    mu += r ;
	    t += (nb - 1) * q * r ;
	    nb++;
	}

	if(pos < lastline)
	{
	    j = pos + swidth ;

	    if( (nolast) &&
	        (!R[j]) &&
	        (DIFF(mu , I[j]) <= tol * sqrt(t/(nb - 1))))
	    {
		p[k++] = j ;
		marquer(j) ;

		q = (float) I[j] - mu ;
                r = q/nb ;
		mu += r ;
		t += (nb - 1) * q * r ;
		nb++;
	    }
	    
	    j++ ;
	    if( (!R[j]) &&
	    (DIFF(mu , I[j]) <= tol * sqrt(t/(nb - 1))))
	    {
		p[k++] = j ;
		marquer(j) ;

		q = (float) I[j] - mu ;
                r = q/nb ;
		mu += r ;
		t += (nb - 1) * q * r ;
		nb++;
	    }
	    j++ ;

	    if( (nofirst) &&
	        (!R[j]) &&	
	        (DIFF(mu , I[j]) <= tol * sqrt(t/(nb - 1))))
	    {
		p[k++] = j ;
		marquer(j) ;

		q = (float) I[j] - mu ;
                r = q/nb ;
		mu += r ;
		t += (nb - 1) * q * r ;
		nb++;
	    }	    	    
	}

        if (i==k)
	{
	    contour(pos);
	    marquer(pos) ;
	}
	k-- ;
	pos = p[k] ;
    }
}
