%{
/***************************************************************
* Copyright (c) 1992      Technical Research Centre of Finland (VTT)
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose and without fee is hereby granted, provided
* that this notice and the reference to this notice appearing in each software
* module be retained unaltered, and that the name of any contributors shall not
* be used in advertising or publicity pertaining to distribution of the software
* without specific written prior permission.  No contributor makes any
* representations about the suitability of this software for any purpose.
* It is provided "as is" without any express or limited warranty.
*
*			NO WARRANTY
*
* ALL CONTRIBUTORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
* INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS.  IN NO
* EVENT SHALL ANY CONTRIBUTOR BE LIABLE FOR ANY SPECIAL, PUNITIVE, INDIRECT OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
* DATA, OR PROFITS, WHETHER IN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
* TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH, THE USE OR PERFORMANCE
* OF THIS SOFTWARE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THIS
* SOFTWARE IS WITH YOU.  SHOULD THIS SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
* As used above, "contributor" includes, but is not limited to :
*        The Technical Research Centre of Finland
***************************************************************/


/**********************************************************************
* NAME
*	prepro.l
*
* PURPOSE
*	lex specification for C++ language extensions in OTSO
*	protocol modules.
*
* USAGE
*	Used as lex analyzer for prepro.y.
*
* MODIFICATIONS
* BUGS
*  - AUTOMATON, PROCESSES, OBJECTS commented even inside comments => crash
***********************************************************************
*/

/* ---- LEX DEFINITIONS SECTION ---- */

/* -- external definitions, includes -- */
#include "ctype.h"	/* islower(), etc */	/*was: #include CTYPE_HDR*/
#include "stdio.h"	/* fopen(...) */	/*was: #include STDIO_HDR*/

/* -- local functions -- */
#ifdef yywrap
#undef yywrap	/* defined by flex, I define my own yywrap() */
#endif
int	yywrap ();
void	copyIt ();
void	commentOut();
void	handleBlockComment();
void	handleIdentifier();
void	handleOperator();
void	handleCppLineComments();
#define echo copyIt()

/* -- local variables -- */
static char	buffer1[64], *bp1,	/* general purpose */
		buffer2[64], *bp2,
		*pbyte;
%}

%Start multiLineDirective
%p 6000
%n 1000
%a 7000
%o 8000
/* ---- LEX RULES/ACTIONS SECTION ---- */

%%
ASN1			{commentOut(); return ASN1_;}
AUTOMATON		{if (inComment) 
			   {echo; return IDENTIFIER;}
			 else {
			   automaton = true; cout << "/***** ";
			   echo; return AUTOMATON;
			 }
			}
class			{echo; defaultAccess = privateAccess; return CLASS;}
COD			{commentOut(); return COD;}
const			{echo; return CONST;}
ENUM			{echo; return ENUM_uppercase;}
enum			{echo; return ENUM_lowercase;}
friend			{echo; return FRIEND;}
implementation		{commentOut(); return IMPLEMENTATION;}
interface		{commentOut(); return INTERFACE;}
OBJECTS			{
			cout << "/***** "; echo;
			return OBJECTS;
			}
\/\*$OTSO_ON[ \t\f\n]*\*\/	{PreproBase::isOtsoOn = true;
				 echo; return yylex();}
\/\*$OTSO_OFF[ \t\f\n]*\*\/	{PreproBase::isOtsoOn = false;
				 echo; return yylex();}
private			{echo; return PRIVATE;}
Process			{echo; 
			 if (processEnabled) return PROCESS;
		         else {
			   handleIdentifier();
			   return IDENTIFIER;
		        }}
PROCESSES		{
			cout << "/***** "; echo;
			return PROCESSES;
			}
protected		{echo; return PROTECTED;}
public 			{echo; return PUBLIC;}
redefined		{echo; return REDEFINED;}
[a-zA-Z_][a-zA-Z0-9_]*_interface  {
                         echo; return SOME_INTERFACE;}
static			{echo; return STATIC;}
struct			{echo; ::defaultAccess = publicAccess; 
	                 return CLASS; /*!?*/}
THIS_IS			{return THIS_IS;}
TST			{commentOut(); return TST;}
union			{echo; return UNION;}
unsigned		{echo; return UNSIGNED;}
VIRTUAL			{echo; return VIRTUAL_;}
virtual			{echo; return VIRTUAL_;}
operator[ \t\n]*\,	{echo; handleOperator(); return IDENTIFIER;}
operator[ \t\n]*\([ \t\n]*\) {echo; handleOperator(); return IDENTIFIER;}
operator[ \t\n]*[^()",;.\n]+ {echo;
                         //This is a slightly risky rule!?
                         //yacc handles this lexical symbol as a constructor.
                         handleOperator();
	 		 return IDENTIFIER;
			}
"(-)"			{if (automaton && braceLevel == 1 || /*AUTO input*/
			     braceLevel == 0) {
                           /*file scope function prototype*/
                           strcpy(yytext, "_in");
                         }
			 //else if (automaton && braceLevel > 1)
			 else 	//AUTO actions, function body
			   strcpy(yytext, "_out");
			 /* manipulate parenthesLevel to copy (-) into fsa */
			 parenthesLevel++; echo; parenthesLevel--;
			 return ABBREVIATION;
 		        }
[a-zA-Z_][a-zA-Z0-9_]*	{echo;
			 handleIdentifier();
			 return IDENTIFIER;
			}
\:\:[ \t\n]*[a-zA-Z_][a-zA-Z0-9_]* {echo; /*this rule requires big tables %a %o*/
			 handleIdentifier();
			 return IDENTIFIER;
                        }
[a-zA-Z_][a-zA-Z0-9_]*[ \t\n]*\:\:[ \t\n]*[a-zA-Z_][a-zA-Z0-9_]* {echo;
			 handleIdentifier(); /*this rule requires big tables %a %o*/
			 return IDENTIFIER;
                        }
-?[0-9]+                {	
			echo; yylval = atoi(yytext);
                        return NUMBER;
			}
#INCLUDE[ \t]+[\"\<][^\">\n]+[\"\>] {
			/* generate #ifndef wrapper for #include's */
			commentOut();
			*(bp1 = buffer1) = NULL;	/* wrapper */
			*(bp2 = buffer2) = NULL;	/* filename */
			if ((pbyte = index (yytext, '\"')) == NULL)
			    pbyte = index (yytext, '<');
			*bp2++ = *pbyte++;
			for (/*pbyte*/;
			     *pbyte && *pbyte != '\"' && *pbyte != '>';
			     pbyte++
			) {
			    *bp2++ = *pbyte;
			    if (islower(*pbyte)) *bp1++ = *pbyte + 'A' - 'a';
			    else if (*pbyte == '.') *bp1++ = '_';
			    else *bp1++ = *pbyte;
			}
			*bp2++ = *pbyte; *bp2 = NULL;
			*bp1 = NULL;
			cout << "#ifndef " << buffer1 <<" \n";
			cout << "#include " << buffer2 << "\n";
			cout << "#endif";
                        return yylex();
			}
\'\"\'			{echo; return SQUOTE_DQUOTE_SQUOTE;}
\\\\			{echo; return BACKSLASH_BACKSLASH;}
\\\"			{echo; return BACKSLASH_DQUOTE;}
\"			{echo; 
                         if (inComment) 
                           return DQUOTE;
                         else {
                           if (::inString) {
                             ::inString = false;
                             return DQUOTE; 
                           }
                           ::inString = true;
                           String s(200);
                           int i;
                           while (::inString && (i = yylex()) != DQUOTE) {
                             if (i == SQUOTE_DQUOTE_SQUOTE) {
                               s += "'";
                               break;		//?
                             }
                             else
                               s += yytext;
                           }
                           ::string = s;
                         }
                         return STRING;
                        }
"("			{parenthesLevel++; echo; return '(';}
")"			{echo; parenthesLevel--; return ')';}
","	|
";"     |
":"     |
"="     |
"&"	|
"."	|
"~"	|
"["     |
"]"     |
"*"			{echo; return yytext[0] ;}
"{"	                {echo; if (!::inComment) braceLevel++; 
			 return yytext[0];}
"}"	                {if (!::inComment) braceLevel--; 
			 echo; return yytext[0];}
"//".*			{echo;
			 if (inString) {
			   yywarning("Sorry, not implemented: \"//\" in string.  Rest of line skipped, string terminated."); //?
			   inString = false;
			   return yylex();
			 }
			 if (yytext[2]=='*')
			   yywarning("slash-slash-asterisk starting a comment");
                         if (inComment) {
			   //copy comments including slashes
                           //?: *Commented::lastComment += yytext;
			   for (sint32 i = 0; i < yyleng; i++) 
			     if (yytext[i]=='*' && yytext[i+1]=='/') {
			       //End of comment.  But rest of line not parsed.
			       if (i+1 < yyleng-1) {
				 //(note: warning not printed if inComment)
				 inComment = false; 
				 yywarning("Sorry, not implemented: \"//\" in the last line of block comment /*...*/.  Comment terminated, rest of line skipped.");
			       }
			       else 
				 inComment = false;
			       }
			   return 1; //?, something else than yylex() !
		         }
                         else if (Commented::current) { 
                           String tmp = &yytext[2]; //exclude slashes
                           //  */  --> **
                           for (sint32 i = 0; i < tmp.length() - 1; i++)
                             if ((tmp[i]=='*') && (tmp[i+1]=='/'))
                               tmp[i+1] = '*';	
			   //empty line (not first) => new paragraph
                           if (tmp.isEmpty() &&
			       !Commented::current->lineComments.isEmpty()) 
                             Commented::current->lineComments += "\n\n";
                           else
                             Commented::current->lineComments += tmp + " ";
                         }
                         return yylex();
                        }
\/\*+                   {if (inString)
                           echo; 
			 else {
			   inComment = true; echo;
			   handleBlockComment();
			 }
			 return yylex();
		        }
\*+\/			{if (inString) {
                           echo; return yylex();
			 }
			 else {
			   echo; inComment = false; 
			   *Commented::lastComment += " ";
			   return 1;
			 }
		        }
\/\*\*+\/		{echo; return yylex();
                         /*fixes problem caused by the previous comment rules'
			   extra '+' */
		        }
\\\n			{echo; lineCount++; 
			 if (inComment) 
                           *Commented::lastComment += yytext;
                         return yylex();
                        }
\n			{echo; lineCount++; 
			 if (inComment) 
                           *Commented::lastComment += yytext;
                         if (::inString) {
                           yywarning("Newline in string? String terminated.");
                           ::inString = false;
                         }
                         return yylex();
                        }
[ \t\f]+		{echo; 
			 if (inComment) 
                           *Commented::lastComment += yytext;
			 return yylex();}
^#.*                    {echo;
			 boolean wasInComment = inComment;
			 handleCppLineComments();
			 if (yytext[yyleng-1] == '\\') 
			   BEGIN multiLineDirective;
			 else if (inComment && !wasInComment)
			   handleBlockComment();
                         return yylex();}
<multiLineDirective>.*  {echo;
			 boolean wasInComment = inComment;
			 handleCppLineComments();
                         if (yytext[yyleng-1] == '\\') 
			   BEGIN multiLineDirective;
			 else {
			   BEGIN 0;
			   if (inComment && !wasInComment)
			     handleBlockComment();
			 }
                         return yylex();}
.			{echo; return STUFF; /*prepro does not care*/}
%%



/* ---- LEX PROGRAMMER ROUTINES SECTIONS ---- */

void commentOut() {            //Enclose yytext in comments
  if (!inComment) cout << "/*";
  echo;
  if (!inComment) cout << "*/";
}

void includeFrom(char* filename) {
  /* It was too difficult to figure out how to open a file in the middle
     of another with flex, lex does not seem to care at all.
     flex just kept on reading the first file (stdin) */
  original_yyin = yyin;
  originalLineCount = lineCount;
  yyin = fopen(filename, "r");
  if (yyin) {
    echoOn = false;
    lineCount = 1;
    cout << "#line 1 \"" << filename << "\"\n";
  }
  else {
    OTSO_ERROR( "Can't open " << filename );
    yyin = original_yyin;
  }
}

int yywrap() { // called at end of file
  if (yyin != original_yyin) {
    yyin = original_yyin;
    lineCount = originalLineCount;
           //Here I rely on the assumption that file.cx becomes file.cxx
    cout << "#line " << lineCount << " \"" << (char*)originalFileName << "x\"\n";
    return 0;	// continue parsing
  } 
  else {
    endOfFile = true;
    if (braceLevel != 0)
      yywarning("Mismatching braces (somewhere above)");
    return 1;	// stop parsing
  }
}

/* echo macro used in LEX RULES section */
void copyIt() {
  if (options->parseDebug) cerr << yytext;

  if (automaton && (braceLevel > 1	// inside actions
		    || braceLevel==1 && parenthesLevel > 0) // func parameters
      )
    *fsa << yytext;

  if (automaton) {	//  /*comments*/ => **comments**
    for (sint32 i = 0; i < strlen(yytext) - 1; i++) {
      if ((yytext[i]=='*') && (yytext[i+1]=='/'))
	yytext[i+1] = '*';	
      else if ((yytext[i]=='/') && (yytext[i+1]=='*'))
	yytext[i] = '*';
    }
  }

  if (inDefaultValue) //parsing function parameter default value
    defaultValueChars += yytext;
  String keepText(yytext);	// to keep Cfront happy
  if (inBrackets && keepText != "]" && keepText != ";")
    textInBrackets += yytext;

  if (echoOn)
    cout << yytext;
}

void handleIdentifier() {
  if (strlen(yytext) > maxIdentifierLength) {
    yytext[maxIdentifierLength-1] = 0;
    yyerror("Too long identifier");
  }
  if (!inComment) {
    previousIdentifier = identifier;
    identifier = yytext;
  }
}

void handleOperator() {
  if (!inComment) {
    previousIdentifier = identifier;
    String op = &yytext[8]; //the rest after "operator"
    String opww = "operator " + op.withoutWhite();
    if (opww.length() > maxIdentifierLength) {
      opww[maxIdentifierLength-1] = 0;
      yyerror("Too long identifier");
    }
    identifier = opww;
  }
}

/**********************************************************************
Check if a C preprocessor line (#...) has comments, and set '::inComment' 
accordingly.  Warning given, if there is a line comment (//...) 
on a preprocessor line.
Comment characters in strings not handled correctly.
***********************************************************************/
void handleCppLineComments() {
  char* cp;
  for (int i = 0; i < yyleng; i++) {
    cp = &yytext[i];
    if (*cp=='/' && *(cp+1)=='/' && !::inComment)
      yywarning("Don't use C++ line comments (//...) on C preprocessor lines; use block comments (/*...*/)");
    if (*cp=='/' && *(cp+1)=='*')
      ::inComment = true;
    if (*cp=='*' && *(cp+1)=='/')
      ::inComment = false;
  }
}

void handleBlockComment() {
  Commented::current = 0;		//stop collecting line comments
  *Commented::lastComment = "";		//?
  yylex();
  while (inComment && !endOfFile) {
    *Commented::lastComment += yytext;
    yylex();
  }
}
