/***********************************************************************
 * Copyright (c) 1993 Technical Research Centre of Finland
 * All rights reserved.
 *
 * This software is provided ``as is'' and without any express or
 * implied warranties, including, without limitation, the implied
 * warranties of merchantibility and fitness for a particular purpose.
 **********************************************************************/

// File: cerial.h

#include "env_sel.h"

#if CERIAL_DEBUGGING
#define CERIAL_DEBUGGING_FREE_STORE 1
#define CERIAL_PRETTY_PRINT 1
#define CERIAL_MEMBER_NAME(CERIAL,NAME) CERIAL.setMemberName(NAME)
#else
#define CERIAL_DEBUGGING_FREE_STORE 0
#define CERIAL_PRETTY_PRINT 0
#define CERIAL_MEMBER_NAME(CERIAL,NAME) /*empty*/
#endif /*CERIAL_DEBUGGING*/

#define COMPILER_GPLUS 0	/* gcc/g++ not used for now */
#define SSTREAMS 1 		/* sstreams used*/


#define CERIAL_ASSERT(EXPR) \
  if (!(EXPR)) printf("***** ERROR: ASSERT failed in file %s on line %d\n", __FILE__, __LINE__);

#if CERIAL_DEBUGGING_FREE_STORE
extern char* gFileName;
extern int gLineNumber;

#define NEW(TYPE_ARGS) \
  (gFileName = __FILE__, gLineNumber = __LINE__, new TYPE_ARGS)
#define DELETE(PTR) \
  gFileName = __FILE__, gLineNumber = __LINE__, delete (PTR) /*?*/

#else
#define NEW(TYPE_ARGS) new TYPE_ARGS
#define DELETE(PTR) delete (PTR)
#endif


/**********************************************************************
We need a few new types for ASN.1 encoding.
These are used in the generated serialize() functions.
**********************************************************************/

class SEQUENCE_t {
  friend ostream& operator<<(ostream&,SEQUENCE_t);
  char* typeName;
public:
  SEQUENCE_t(char* p): typeName(p) {}
};

extern ostream& operator<<(ostream&,SEQUENCE_t); //prints typeName

class SEQUENCEOF_t: public SEQUENCE_t {
public:
  SEQUENCEOF_t(char* p): SEQUENCE_t(p) {}
};

class EOC_t {
public:
  EOC_t(char*) {}
};

typedef int Tag;
class cerial;
class StreamFamily;
class acTag;
class SerializePointerArguments;

/**********************************************************************
Some types needed in cerial::serialize_pointer():
**********************************************************************/

typedef void (*CerialSerializeFP) (cerial&, ...);	// ... = TYPE&
typedef void* (*CerialNewFP)();				//returns new TYPE
typedef void (*CerialDeleteFP) (void*);			//deletes TYPE*
typedef int (*CerialWhichTypeFP) (cerial&, SerializePointerArguments&);

struct SerializePointerArguments {
  void** pp;			//TYPE**
  CerialNewFP newFP;		//returns new TYPE
  CerialDeleteFP deleteFP;	//calls delete (TYPE*)*pp
  CerialSerializeFP serializeFP;//serialize(cerial&,TYPE&)
  Tag tag;			//
  CerialWhichTypeFP whichTypeFP;//Needed if the exact type of the value is 
                                //not known at compile time, e.g. when
                                //serializing a base class pointer.

  SerializePointerArguments(void** p1, CerialNewFP p2, 
				 CerialDeleteFP p3, CerialSerializeFP p4, 
				 Tag t = 0,
				 CerialWhichTypeFP p5=0);
};

/**********************************************************************
List of virtual functions of class cerial that must be defined in derived
classes.  This macro is useful in defining classes derived from cerial.
**********************************************************************/
#define cerial_interface				\
  virtual void serialize (char* p);			\
  virtual void serialize (u_char* p);			\
  virtual void serialize (short* p);			\
  virtual void serialize (u_short* p);			\
  virtual void serialize (int* p);			\
  virtual void serialize (u_int* p);			\
  virtual void serialize (long* p);			\
  virtual void serialize (u_long* p);			\
  virtual void serialize (float* p);			\
  virtual void serialize (double* p);			\
  virtual void serialize_bool (bool_t*);		\
  virtual void serialize_opaque (char*, u_int); 	\
  virtual void serialize_string (char**);		\
  virtual void serialize (SEQUENCE_t);			\
  virtual void serialize (SEQUENCEOF_t);		\
  virtual void serialize (EOC_t);			\
  virtual void serialize_explicit (acTag*);  		\
  virtual bool_t  serialize_presence (SerializePointerArguments& a); \
  virtual cerial::TransferSyntax transferSyntax() const


#define CERIAL_ENCODE XDR_ENCODE
#define CERIAL_DECODE XDR_DECODE
#define CERIAL_FREE   XDR_FREE

/**********************************************************************
Class cerial is an Abstract C Language Serializer.
The encoding rules are defined in derived classes.
Each of serialize() functions can take care of encoding and decoding,
depending on the object's "mode" (CERIAL_ENCODE or CERIAL_DECODE).
The functions can also take care of deallocating free store memory 
allocated during decoding (mode CERIAL_FREE).

.SH BUGS
**********************************************************************/

class cerial {
public:

/**********
.SH Serialize C language basic types
***********/
  virtual void serialize (char* c)=0;	//encode or decode c
  virtual void serialize (u_char*)=0;	//
  virtual void serialize (short*)=0;		//
  virtual void serialize (u_short*)=0;	//
  virtual void serialize (int*)=0;		//
  virtual void serialize (u_int*)=0;		//
  virtual void serialize (long*)=0;		//
  virtual void serialize (u_long*)=0;	//
  virtual void serialize (float*)=0;		//
  virtual void serialize (double*)=0;	//

/**********
.SH Serialize some other types and constructs
***********/
  virtual void serialize_bool (bool_t*)=0;
                                        //Boolean value.
	                                //Can't overload: bool_t is int
  virtual void serialize (SEQUENCE_t)=0;
                                        //Begin structure. SEQUENCE_t may carry
                                        //the name of the particular
                                        //structure type which is usually
                                        //not used.
  virtual void serialize (SEQUENCEOF_t)=0;
                                        //Begin array; SEQUENCEOF_t may carry
                                        //name of the type.
  virtual void serialize_explicit (acTag*)=0;  
                                        //tagged type, explicit tag
  virtual void serialize (EOC_t)=0;
                                        //end structure, array or explicit tag.
  virtual void serialize_opaque (char*, u_int)=0;
	                                //Opaque data, octet values encoded
                                        //as such.
  virtual void serialize_string (char** pp)	=0;  
                                        //When encoding, *pp should be 0 or 
                                        //it should point to a 0-terminated
                                        //string.
                                        //
                                        //When decoding, *pp should point
                                        //to a space big enough to store
                                        //the incoming string.  If *pp is 0,
                                        //free store space is allocated for 
                                        //the incoming string (based on the
                                        //transfer syntax specific length
                                        //indication).
                                        //
                                        //When CERIAL_FREE, deallocates *pp.
                                        //Thus, when CERIAL_FREE,
                                        // *pp must be 0 if it is not allocated
                                        //from free store!!!
                                        //
                                        //pp must be not be 0 (mainly because
                                        //xdr_string() requires that).
  virtual void serialize_pointer (SerializePointerArguments& a); 
                                        //Serializes pointers in a relatively
                                        //generic way.  0 pointer in encoding
                                        //means no value to encode.
                                        //0 pointer in decoding means that
                                        //space is allocated for the incoming
                                        //object.
                                        //
					//Warning: If a pointer chain contains
                                        //a loop, this routine will stay
                                        //in the same loop recursively and
                                        //infinitely.
  virtual bool_t serialize_presence(SerializePointerArguments& a) =0; 
                                        //Serializes information that tells
                                        //if an optional element is present
                                        //or not.  Returns
                                        //TRUE if the element is present.
                                        //
                                        //XDR: serializes tag.
                                        //
                                        //ASN.1 encoding:
					//nothing encoded.
					//
					//ASN.1 decoding:
					//'a.tag' input value is the tag value
					//expected.  Does not extract the tag,
                                        //only looks ahead to be able to 
					//compare the next incoming tag and
					//the expected tag.
/**********
.SH Miscellaneous
***********/
public:
#if CERIAL_PRETTY_PRINT
  virtual void setMemberName(char* memberName);
                                        //Tells the name of the member to be
                                        //serialized next.  
                                        //Used only by readable_serializer.
                                        //
#endif
  int operator ! () {return !isOk;}	//
  xdr_op mode() const			{return xop;}
			                //CERIAL_ENCODE, CERIAL_DECODE, 
                                        //or CERIAL_FREE
  enum TransferSyntax {			
    xdrTransferSyntax,				
    asn1BerTransferSyntax,
    readableTransferSyntax
            //new values should be added here for new transfer syntaxes.
  };
  virtual cerial::TransferSyntax transferSyntax() const = 0;
                                        //Usage discouraged.
                                        //Used for optimization purposes.

protected:
  cerial(xdr_op op, StreamFamily* stream);
                                        //op is CERIAL_ENCODE, CERIAL_DECODE, 
                                        //or CERIAL_FREE.
                                        //stream is a pointer to the 
                                        //StreamFamily instance used with this
                                        //serializer.
  StreamFamily* stream_;		// :(
  virtual ~cerial();
  bool_t isOk;                          //Stream status: initially true,
                                        //false if some en/decoding has failed.
                                        //Updated by serialize() functions.
  xdr_op	xop;		        //selected operation 
                                        //(CERIAL_ENCODE, CERIAL_DECODE, 
                                        //or CERIAL_FREE)
};


/**********************************************************************
The generated serialize() functions call eventually these functions:
**********************************************************************/

#define SERIALIZE_BUILTIN(TYPE,NEWFN,DELETEFN)				\
  inline void serialize(cerial& l, TYPE& r) 				\
    {l.serialize(&r);} 							\
  inline void* NEWFN () {return NEW(TYPE);} 				\
  inline void DELETEFN (void* p) {DELETE((TYPE*)p);} 			\
  extern /*?*/ void serialize(cerial& l, TYPE*& p, Tag tag = 0);

SERIALIZE_BUILTIN(char,cerialNew_char,cerialDelete_char)
SERIALIZE_BUILTIN(u_char,cerialNew_u_char,cerialDelete_u_char)
SERIALIZE_BUILTIN(short,cerialNew_short,cerialDelete_short)
SERIALIZE_BUILTIN(u_short,cerialNew_u_short,cerialDelete_u_short)
SERIALIZE_BUILTIN(int,cerialNew_int,cerialDelete_int)
SERIALIZE_BUILTIN(u_int,cerialNew_u_int,cerialDelete_u_int)
SERIALIZE_BUILTIN(long,cerialNew_long,cerialDelete_long)
SERIALIZE_BUILTIN(u_long,cerialNew_u_long,cerialDelete_u_long)
SERIALIZE_BUILTIN(float,cerialNew_float,cerialDelete_float)
SERIALIZE_BUILTIN(double,cerialNew_double,cerialDelete_double)

inline void serialize_string(cerial& l, char** r) {l.serialize_string(r);}

inline void serialize_pointer(cerial& l, SerializePointerArguments& a) {l.serialize_pointer(a);}

inline void serialize(cerial& l, SEQUENCE_t& r) {l.serialize(r);}
inline void serialize(cerial& l, SEQUENCEOF_t& r) {l.serialize(r);}
inline void serialize(cerial& l, EOC_t& r) {l.serialize(r);}


/******************************************************************
A base class of a family of stream classes 
that store/transmit byte arrays.
******************************************************************/

class StreamFamily {
  friend class AsnBerCerial;	//touches xdrstrm
  friend class XdrCerial;	//touches xdrstrm
public:
  XDR* xdrs()			{ return xdrstrm; }
				// allow fine grain manipulations of the
				// underlying XDR stream by sophisticated users
protected:
  StreamFamily(xdr_op op);	// default filestreams are stdin/stdout
 ~StreamFamily();

  XDR*		xdrstrm;	// underlying XDR stream 
  xdr_op 	mode() const	{return mode_;}
  bool_t	xinitOk(xdr_op op) const;

private:
  xdr_op	mode_;		//duplicate!?
};


/******************************************************************
FileStream writes bytes to a file, or reads bytes from a file.
******************************************************************/

class FileStream: public StreamFamily {
protected:
  FILE*		xfp;		//file pointer
  FileStream(xdr_op op, FILE* fp);
};

/******************************************************************
MemoryStream puts bytes to to an array, or gets bytes from an array.
******************************************************************/

class MemoryStream: public StreamFamily {
public:
  virtual void print (ostream& os);
  unsigned int getpos();		//actual number of bytes in use

protected:
  char*		x_buffer;		//buffered bytes
  unsigned int  x_buflen; 		//max length of buffer
  bool_t	deleteBuffer;		//true iff x_buffer allocated by new()
  MemoryStream(xdr_op op, u_int size, char* addr);
 ~MemoryStream();
};

/******************************************************************
SocketStream sends/receives bytes to/from a (remote) host through a socket.
******************************************************************/

class SocketStream: public sstream, public StreamFamily {
public:
  void endofrecord(bool_t now=TRUE);
        		// Note: holdover from using Sun XDR libs: sender
			// marks the end of current msg + send now/later

protected:
  SocketStream(xdr_op op, char* host =0, u_int port =0);
 			//Client tpc/ip connection attempted for port@host
                        //if host is given.
  int readit (char* buf, int length);
                        // when the XDR* buffer needs more bytes,
                        // read bytes into buffer from the socket stream.
  int writeit(char* buf, int length);
                        // when the XDR* buffer needs to flush out bytes,
                        // write flushed bytes out to the socket stream.
};

/////////////////////////////////////

struct MBlockHeader {
  char* fileName;
  int  lineNumber;
  int  userBlockSize;
  void* pointer;
  MBlockHeader* next;
  MBlockHeader* prev;
  int isOnNow;
  static MBlockHeader* last;
  static MBlockHeader* first;
  static int isOn;
  MBlockHeader();
  ~MBlockHeader();
};

