/***********************************************************************
 * Copyright (c) 1993 Technical Research Centre of Finland
 * All rights reserved.
 *
 * This software is provided ``as is'' and without any express or
 * implied warranties, including, without limitation, the implied
 * warranties of merchantibility and fitness for a particular purpose.
 **********************************************************************/

//cerial.cc

#include "env_sel.h"
#include "iostream.h"
#include <stdio.h>
#include <rpc/rpc.h>
#if ! CERIAL_CPLUSPLUS_INCLUDE
#include "xdr.ext"
#include "str.ext"
#endif
#include "sstream.h"
#include "cerial.h"
//#include "xdrcerial.h" /*?*/
#include "asntl.h"


/***************** local defines/constants **************************/

#ifndef DEBUG
#define DEBUG 0
#endif

static const int ci_xdrbufferlength = 4024;
	// JFR: xdr overflow code seems to work better if we
	//	force length of buffer used in underlying buffers
	//	see xdrrec_create() below.




#define SERIALIZE_BUILTIN_CODE(TYPE,NEWFN,DELETEFN)			\
void serialize(cerial& l, TYPE*& p, Tag tag)				\
    {if (!p &&								\
         l.mode() == CERIAL_ENCODE &&					\
         l.transferSyntax() == cerial::asn1BerTransferSyntax)		\
           return; /*an ASN.1 encoding optimization*/			\
     void (*sfp) (cerial&, TYPE&) = serialize;				\
  SerializePointerArguments a((void**)&p, NEWFN, DELETEFN, (CerialSerializeFP)sfp, tag); \
     l.serialize_pointer(a);						\
    }

SERIALIZE_BUILTIN_CODE(char,cerialNew_char,cerialDelete_char)
SERIALIZE_BUILTIN_CODE(u_char,cerialNew_u_char,cerialDelete_u_char)
SERIALIZE_BUILTIN_CODE(short,cerialNew_short,cerialDelete_short)
SERIALIZE_BUILTIN_CODE(u_short,cerialNew_u_short,cerialDelete_u_short)
SERIALIZE_BUILTIN_CODE(int,cerialNew_int,cerialDelete_int)
SERIALIZE_BUILTIN_CODE(u_int,cerialNew_u_int,cerialDelete_u_int)
SERIALIZE_BUILTIN_CODE(long,cerialNew_long,cerialDelete_long)
SERIALIZE_BUILTIN_CODE(u_long,cerialNew_u_long,cerialDelete_u_long)
SERIALIZE_BUILTIN_CODE(float,cerialNew_float,cerialDelete_float)
SERIALIZE_BUILTIN_CODE(double,cerialNew_double,cerialDelete_double)


SerializePointerArguments::SerializePointerArguments(void** p1, 
						     CerialNewFP p2, 
						     CerialDeleteFP p3, 
						     CerialSerializeFP p4, 
						     Tag t,
						     CerialWhichTypeFP p5
						    )
    : pp(p1)
    , newFP(p2)
    , deleteFP(p3)
    , serializeFP(p4)
    , tag(t)
    , whichTypeFP(p5) 
{}

cerial::cerial(xdr_op op, StreamFamily* stream)
  : isOk(op==CERIAL_ENCODE || op==CERIAL_DECODE)
  , xop(op) 
  , stream_(stream)
{}

cerial::~cerial() {}

//serialize_pointer() is HEAVILY modified 
//from xdr_pointer() + xdr_reference() source code.

void cerial::serialize_pointer(SerializePointerArguments& a) {

                                        // If CERIAL_FREE, then destruct 
				        // pointer members and return .
  if (this->mode() == CERIAL_FREE) {
    if (*a.pp) {	     		//even if !isOK 
      (*a.serializeFP)(*this, *a.pp);	//serialize(DerivedCerial&, TYPE&)
                                        //recursively
      (*a.deleteFP)(*a.pp);
      *a.pp = 0;
    }
  }

  if (!isOk) return;

                                        // Check if member is present
  bool_t isPresent;
  					//default pointer handling
  isPresent = this->serialize_presence(a); //virtual
  if (!isOk) return;

  if (!isPresent) {			//nothing to en/decode
      *a.pp = 0;
      return;
  }
                                        //something to en/decode

  acTag t(acTag::contextSpecific | acTag::constructed, a.tag);
					//The generated tags are always
  serialize_explicit(&t);		//context specific and explicit:
                                        //TYPE* tp --> tp [1] TYPE OPTIONAL

  if (a.whichTypeFP)			//If type not known at compile time,
    (*a.whichTypeFP)(*this, a);		//serialize some type id.
	                                //By default, a.whichTypeFP == 0.

  if (*a.pp == 0)			
      if (this->mode() == CERIAL_DECODE) { //allocate space and construct
	  *a.pp = (*a.newFP)();		//new TYPE
	  if (*a.pp == 0) {
	      cerr << "serialize_pointer(): out of memory\n";
	      isOk = FALSE;
	      return;
	  }
      }

  (*a.serializeFP)(*this, *a.pp);	  //serialize(DerivedCerial&, TYPE&)
  serialize(EOC_t("explicit tag"));

}

#if CERIAL_PRETTY_PRINT
void cerial::setMemberName(char*) {
  /*do nothing*/
}
#endif

#if CERIAL_DEBUGGING_FREE_STORE

#endif

/************* StreamFamily classes ***************************/

bool_t StreamFamily::xinitOk(xdr_op op) const {
  return (op == CERIAL_ENCODE || op == CERIAL_DECODE);
}

StreamFamily::StreamFamily(xdr_op op): mode_(op) {
  this->xdrstrm = NEW(XDR);
}

StreamFamily::~StreamFamily() {
  if (this->mode() == CERIAL_FREE)
    ; //cerr << "XDR not deallocated in ~StreamFamily (CERIAL_FREE)\n";
  else
    XDR_DESTROY(this->xdrstrm);
  DELETE(this -> xdrstrm);
}

FileStream::FileStream(xdr_op op, FILE* fp)
  : StreamFamily(op) 
{
  this->xfp = NULL;
  if (this -> xinitOk(op)) {
    if (fp != 0) {
      this -> xfp = fp; 
    }
    else {
      if (op == CERIAL_ENCODE) { this -> xfp = stdout; }
      else { this -> xfp = stdin; }
    }
    ::xdrstdio_create (this -> xdrstrm, this -> xfp, op);
  }
}

MemoryStream::MemoryStream(xdr_op op, u_int size, char* addr)
  : StreamFamily(op) 
{
    if (this -> xinitOk(op) && size != 0) {
      if (addr != 0) {
	x_buffer = addr;
	deleteBuffer = FALSE;
	x_buflen = size;
	::xdrmem_create (this -> xdrstrm, addr, size, op);
      } else {
	x_buffer = NEW(char[size]);
	deleteBuffer = TRUE;
	x_buflen = size;
        ::xdrmem_create (this -> xdrstrm, x_buffer, size, op);
      }
    }
    if (size == 0) {
      x_buffer = 0;
      deleteBuffer = FALSE;
      x_buflen = 0;
    }
}

MemoryStream::~MemoryStream() {
  if (deleteBuffer) 
    DELETE(x_buffer);
}

static char* nib2hex [0x10] = {
  "0", "1", "2", "3", "4", "5", "6", "7",
  "8", "9", "A", "B", "C", "D", "E", "F"
};

void MemoryStream::print (ostream& os) {
  unsigned int	index	= 0;
  char		printNL	= 0;	// flag to print newline (if 1)
  unsigned int	handyLength	= this->getpos();
	// this should be the "real" number of bytes in the buffer
	// and not the buffer's maximum length

  if (handyLength <= 0) {
    os << "(nil)\n";
	// nothing has been put into buffer
  } else {
    while (index < handyLength) {
        os << nib2hex[(x_buffer[index] & 0xf0) >> 4]
           << nib2hex[(x_buffer[index] & 0x0f)     ] << " "; 

      if (printNL++ == 15) {
        os << "\n";
        printNL=0;
      }
      index++;	// don't increment index until finished using it !
	        // Juha: not with XDR ???
    }
    if (printNL != 0) { os << "\n"; }
  }
}

unsigned int MemoryStream::getpos() {
  //
  // xdrmem_getpos is statically declared for use in XDR library
  // but returns u_int as below...
  // return (xdrmem_getpos(this->xdrs()));
  //
  return ((u_int)this->xdrs()->x_private - (u_int)this->xdrs()->x_base);
}


SocketStream::SocketStream(xdr_op op, char* host, u_int port)
  : StreamFamily(op) 
{
  // Record-oriented model ctor
  if (this -> xinitOk(op)) {
    ::xdrrec_create (
        this -> xdrstrm,
        ci_xdrbufferlength,
        ci_xdrbufferlength,
        (char*)(this),	// first param of readit or writeit
			// in our case 'this' cause it's first param
			// of a C++ fn call.
        (int (*)()) (&SocketStream::readit),
        (int (*)()) (&SocketStream::writeit)
    );
  }

  if (host) {
    this -> open(host, port);
    if( !*this ) {
      cerr << "could not connect to " << host << " " << port <<"\n";
      exit(2);
    }
    cerr << "client " << this->sd() << " connected to " << this->tohost() 
         << " " << this->toport() << "\n";
  }
}


// event handler for filling XDR* buffer
int SocketStream::readit (char* buf, int len)
{
 
#if DEBUG
  fprintf (stderr,
    "DBUG: readit((addr)0x%x, (addr)0x%x, (len)%x);\n",
    (unsigned int)(this), (unsigned int)buf, len
  );
#endif

  // similar to writeit, but for reading

  this -> read (buf,len);

  if (!*this) {
    cerr << "bad this read\n";
    return 0;
  } else {
    return 1;
  }
}

// event handler for flushing XDR* buffer
int SocketStream::writeit (char* buf, int len)
{
  // Called by an XDR* stream during "overflow" when it wants to flush out
  // its internal buffer (see xdrrec_create above).

#if DEBUG
  fprintf (stderr,
    "DBUG: SocketStream::writeit((addr)0x%x, (addr)0x%x, (len)%x);\n",
    (unsigned int)(this), (unsigned int)buf, len
  );
#endif

  this -> write (buf,len);
  this -> flush();		// JFR: force the flush
 
  if (!*this) {
    cerr << "bad this write\n";
    return 0;
  } else {
    return 1;
  }
}

void SocketStream::endofrecord (bool_t now) {
  ::xdrrec_endofrecord(this->xdrs(), now);
}

/****************************************************************/

#if CERIAL_DEBUGGING_FREE_STORE

char* gFileName = 0;
int gLineNumber = -1;

MBlockHeader* MBlockHeader::first = 0;
MBlockHeader* MBlockHeader::last = 0;
int MBlockHeader::isOn = 0;

MBlockHeader listsExistingMBlocksAtExit;

MBlockHeader::MBlockHeader() {
  isOn = 1;
}

MBlockHeader::~MBlockHeader() {
  int isFirst = 1;
  for (MBlockHeader* h = MBlockHeader::first; h; h = h->next) {
    if (isFirst) {
      fprintf(stderr, "\n*******************************\n");
      fprintf(stderr, "Existing memory blocks at exit:\n");
      fprintf(stderr, "*******************************\n");
      isFirst = 0;
    }
    if (h->isOnNow && h->lineNumber >= 0) {
      fprintf(stderr, "%11d (%6d bytes); allocated in file %s line %d\n", 
	     (char*)h + sizeof(MBlockHeader) /*?*/, 
	     h->userBlockSize, 
  	     h->fileName, 
	     h->lineNumber);
    }
    else if (h->isOnNow)
      fprintf(stderr, "%11d (%6d bytes); Use NEW() to see file name and line number\n",
	     (char*)h + sizeof(MBlockHeader) /*?*/, 
	     h->userBlockSize);
    else
      fprintf(stderr, "%11d (%6d bytes); allocated when constructing static objects ?\n",
	     (char*)h + sizeof(MBlockHeader) /*?*/, 
	     h->userBlockSize);
  }
}

void* operator new (size_t s) {
  void* block = malloc(s + sizeof(MBlockHeader));
  MBlockHeader* header = (MBlockHeader*)block;
  void* ret = (char*)block + sizeof(MBlockHeader); //?

//  printf("@ Allocating %d (%d bytes)\n", ret, s);

  if (gFileName) {
    header->fileName = (char*)malloc(strlen(gFileName)+1);
    strcpy(header->fileName, gFileName);
    gFileName = 0;
  }
  else 
    header->fileName = 0;
  header->lineNumber = gLineNumber;
  gLineNumber = -1;
  header->userBlockSize = s;
  header->pointer = ret;
  header->isOnNow = MBlockHeader::isOn;
  if (!MBlockHeader::first) {
    MBlockHeader::first = header;
    MBlockHeader::last = header;
    header->prev = 0;
    header->next = 0;      
  }
  else {
    MBlockHeader::last->next = header;
    header->prev = MBlockHeader::last;
    header->next = 0;      
    MBlockHeader::last = header;
  }      

  return ret;
}

void operator delete (void* p) {
//  printf("@ Deleting %d\n", p);

  if (p == 0) {
    gFileName = 0;
    gLineNumber = -1;
    return;
  }

  MBlockHeader* header = (MBlockHeader*)((char*)p - sizeof(MBlockHeader)); //?
  void* freed = (void*)header;

  for (MBlockHeader* h = MBlockHeader::first; h; h = h->next)
    if (h->pointer == p) break;
  if (!h) {
    if (gFileName && gLineNumber >= 0)
      fprintf(stderr, "*** ERROR: delete(%d) in file %s on line %d: Already deleted or bad address, deleted anyway\n", p, gFileName, gLineNumber);
    else {
      fprintf(stderr, "*** ERROR: delete(%d): Already deleted or bad address, deleted anyway.\n", p);
      fprintf(stderr, "  Use DELETE() to see file name and line number\n");
    }
  }
  else {
    CERIAL_ASSERT(header == h);
    free(header->fileName);
    header->pointer = 0;
    if (MBlockHeader::first == header)
      MBlockHeader::first = header->next;
    if (MBlockHeader::last == header)
      MBlockHeader::last = header->prev;
    if (header->next)
      header->next->prev = header->prev;
    if (header->prev)
      header->prev->next = header->next;
  }

  gFileName = 0;
  gLineNumber = -1;

  free(freed);
}

#endif /*CERIAL_DEBUGGING_FREE_STORE*/
