/***********************************************************************
 * Copyright (c) 1993 Technical Research Centre of Finland
 * All rights reserved.
 *
 * This software is provided ``as is'' and without any express or
 * implied warranties, including, without limitation, the implied
 * warranties of merchantibility and fitness for a particular purpose.
 **********************************************************************/

/* @(#)asntl.h	1.8 92/11/16 */

/*****************************************************************
* NAME	asntl.h
*
* PURPOSE
*	ASN.1 Tag and Length serializer routines
*
* DESCRIPTION
*  An ASN.1 encoding has 3 or 4 parts in the Basic Encoding Rules (BER):
*	- Identifier Octets	(class acTag)
*	- Length Octets		(class acLen)
*	- Contents Octets
*	- End-Of-Contents Octets
*
*
*  TAG FIELD
*  ---------
*
*  The first octet of the Identifier (Tag) has 3 special bits (b8, b7, b6).
*  	- Class				Bit 8	Bit 7
*	  universal			0	0
*	  application			0	1
*	  context-specfic		1	0
*	  private use			1	1
*
*	- Primitive/Constructed Type	Bit 6
*	  constructed			1	
*	  primitive			0
*
*  The remaining 5 bits (bit1-bit5) can indicate the value of the tag
*  id.  But if they are all set (11111): they indicate that additional octets
*  contain the tag id (high tag number format).
*
*  Identifier Octets can be encoded in two formats in BER:
*	- Low Tag Number Format
*		encoded as above, in one octet (Id = 0 .. 30 [0x00 .. 0x1E])
*	- High Tag Number Format
*		encoded in two or more octets.  First octet is encoded
*		as above, with Id = 11111 (0x1F).  Second and following
*		octets give the tag number base 128, Most Significant Digit
*		first .. Least Significant Digit last.  As few digits as
*		possible are used, and each octet has bit 8 set to "1" except
*		the last one (LSD).
*
* LENGTH FIELD
* ------------
*
* In BER, there are three length formats:
*	1) short definite,   for lengths 0 to 127
*	2) long definite,    for lengths 0 to ((256^126)-1)
*                                   0 to (2^1008) - 1)
*	3) indefinite length, for unknown lengths
*
*  Short Definite form:
*	- 1 octet. Bit 8 has value of 0.  Bits 7-1 give length.
*
*  Indefinite form:
*	- 1 octet. Bits 1-8 have value of 1 (0xFF).
*
*  Long Definite form:
*	- 2 to 127 octets.  First octet: Bit 8 has value of 1
*       and bits 7-1 give the number of following length octets. The length
*       octets follow the first octet and give the length, base 256,
*       Most Significant Digit ... Least Significant Digit.  There can be
*       1 to 126 length octets following the first octet.  A value of 0xFF
*       in the first octet is forbidden as this would conflict with the
*       Indefinite Length format (thus 126 (0x7E) octets is the max).
*
* Note:	This implementation is more restrictive in that we assume
*	definite lengths are represented internally by a 32 bit integer
*	(MAXUINT32 = (2^32) = 4294967296 = 4 GigaByte Length possible).
*	Any lengths longer than this, we represent using the indefinite length
*	format.  This Maximum value can be represented using 5 octets using
*	the long definite format described above.
*
* REFERENCES:
*	[1] The Open Book - A Practical Perspective on OSI, Marshall Rose,
*	    Chapter 8 Abstract Syntax, 1990
*	[2] A Layman's Guide to a Subset of ASN.1, BER, and DER,
*	    Burton S. Kaliski Jr, RSA Data Security Inc, Redwood City, CA, 1991
*	[3] CCITT, ISO/IEC ASN.1 Transfer and Abstract Syntax standards
*	[4] External Data Representation: Sun Technical Notes, Network
*	    Programmer's Guide, Chapter 4, Sun Microsystems Inc
*****************************************************************/



typedef  unsigned long  uint32_t;
	// unsigned 32 bit integer (change when porting !)

class asnserializer;	// extern "asncerial.h"



#define INDEFINITE_LENGTH	0x80	/* Indefinite Length */

#define MAX_TAG_OCTETS	6
	// increase if necessary (see acTagEncode)

class acTag {
  friend class AsnBerCerial;

 public:
  enum TType {
    universal		= 0x00,		// 00bb bbbb
    application		= 0x40,		// 01bb bbbb
    contextSpecific	= 0x80,		// 10bb bbbb
    privateUse		= 0xC0,		// 11bb bbbb
    constructed		= 0x20,		// bb1b bbbb
    highTagNumberForm	= 0x1F		// bbb1 1111
  };

  virtual void	print (ostream& os);
  virtual void	ask   (istream& is);

  acTag (TType tt, uint32_t tv);
	// create (encode) tag from local types
  acTag (char* octp);
	// create (decode) tag from octet stream

  bool_t isConstructed();
  bool_t isPrimitive();
  bool_t isHighTagNumberFormat();
  unsigned char maskClass();

  bool_t operator==(const acTag& r);
  uint32_t noctets() { return (acLastTOctetIndex+1); }
  uint32_t value() { return acTagValue; }

 protected:

 private:
   TType	acTagType;
   uint32_t	acTagValue;
   bool_t	acHighTagFormat;
   char		acTOctets[MAX_TAG_OCTETS];	// 6 octets
   int		acLastTOctetIndex;		// 0 - 5 (marks LSD position)
   void		acTagEncode();			// converts acTagValue to 
						// encoded tag in acTOctets
};

// shorthand macros
#define UNIV	acTag::universal
#define APPL	acTag::application
#define CTXT	acTag::contextSpecific
#define PRIV	acTag::privateUse
#define CTOR	acTag::constructed

/*

*/

#define MAX_LEN_OCTETS	5
	// increase if necessary (see acLenEncode)

class acLen {
  friend class AsnBerCerial;

public:
  enum LType {
    shortDefinite,
    longDefinite,
    indefinite
  };

  virtual void	print (ostream& os);
  virtual void	ask   (istream& is);

  acLen (uint32_t lv, LType lt=acLen::shortDefinite);
	// create (encode) Length from local types
  acLen (char* octp);
	// create (decode) Length from octet stream

  bool_t isShortDefinite();
  bool_t isLongDefinite();
  bool_t isIndefinite();

  uint32_t noctets() { return (acLastLOctetIndex+1); }
  uint32_t value() { return acLenValue; }

 protected:

 private:
   uint32_t	acLenValue;
   LType	acLenType;
   char		acLOctets[MAX_LEN_OCTETS]; // 5 octets
   uint32_t	acLastLOctetIndex;	// 0 - 5 (marks LSD position)
   void		acLenEncode();		// converts acTagValue to 
					// encoded tag in acLOctets
};
