/***********************************************************************
 * Copyright (c) 1993 Technical Research Centre of Finland
 * All rights reserved.
 *
 * This software is provided ``as is'' and without any express or
 * implied warranties, including, without limitation, the implied
 * warranties of merchantibility and fitness for a particular purpose.
 **********************************************************************/

/* @(#)asntl.cc	1.7 92/11/16 */

/*
 * NOTE:
 *  This code is NOT written for efficiency of encoding/decoding,
 *  but hopefully it is relatively easy to understand by someone familiar
 *  with the ASN.1 BER encoding rules.  A future modification would be to make
 *  the code more efficient (and understandable ?)	-- jfr
 */

#include <stream.h>	/* for ostream, istream */
#include <rpc/types.h>	/* for bool_t */

#include "asntl.h"	/* ASN.1 Tag & Length types */

/*************
 * Tag Field *
 *************/

acTag::acTag (TType tagtype, uint32_t tagvalue) {
  acTagType = tagtype;
  acTagValue	= tagvalue;
  acHighTagFormat = FALSE;	// Default: low tag format used

  acLastTOctetIndex = 0;
  acTOctets[0] = (unsigned char)(acTagType);
  acTagEncode ();
}

/*
 * Construct (decode) a Tag from a sequence of octets
 */
acTag::acTag (char* octp) {
  char byte;
  int  ix=0;

  if (octp) {
    acTagType = (TType)(octp[0] & 0xE0); // mask 3 high order bits (Class, Ctor)
    if ((byte = octp[0] & 0x1F) == 0x1F) {
      /* high tag number format */
      acHighTagFormat = TRUE;
      acTOctets[0] = octp[0];

      for (ix=1; ix < MAX_TAG_OCTETS; ix++) {
        acTOctets[ix] = octp[ix];
        if (octp[ix] & 0x80) {          
           /* Last Tag Octet (LSD) has high bit (8) set */
           break;
        }
      }
      if (!(octp[ix-1] & 0x80) && ix == MAX_TAG_OCTETS) {
        cerr << "Error creating tag (LSD not found within "
             << int(MAX_TAG_OCTETS) << " octets)\n";
      } else {
        acTagValue = 0;
        acLastTOctetIndex = ix-1;
          // how many octets are used to encode the tag value (1..5)

        for (int i=1; i<=acLastTOctetIndex; i++) {
          /* base 128 arithmetic : */
          /*   [n*128^4] [n*128^3] .. [n*128^0] */
          /*    octp[1]   octp[2]  ..  octp[5]  */
          acTagValue += (uint32_t)(octp[i] & 0x7F) * 128^(MAX_TAG_OCTETS-1 - i);
            /* only 7 bits (base 128) are used */
            /* Remember: bit 8 is set to mark last octet, thus mask by 0x7F */        
        }
      }
    } else {
      /* low tag number format */
      acTagValue = (uint32_t)byte;
      acHighTagFormat = FALSE;
      acLastTOctetIndex = 0;
      acTOctets[0] = byte;
    }
  }
}

bool_t acTag::operator==(const acTag& r) {
  return (acTagValue == r.acTagValue &&
	  acTagType  == r.acTagType
	  //prim?
	 );
}

void acTag::print(ostream& os) {
  os << "[ ";

  switch (this -> maskClass()) {
  case acTag::universal:		os << "UNIV, "; break;
  case acTag::application:	os << "APPL, "; break;
  case acTag::contextSpecific:	os << "CTXT, "; break;
  case acTag::privateUse:		os << "PRIV, "; break;
  }

  if (this -> isConstructed()) {
    os << "CTOR, ";
  } else {
    os << "PRIM, ";
  }

  os << acTagValue;	// should be unsigned int  32 bit

  os << " [Hex Octets: ";
  for (int ix = 0; ix <= acLastTOctetIndex; ix++) {
    os << hex(((uint32_t)(acTOctets[ix])) & 0xFF) << " ";
  }

  os << "] ] ";
}

void acTag::ask (istream&) {}

bool_t acTag::isConstructed () {
  if (acTagType & acTag::constructed) {
    return TRUE;
  } else {
    return FALSE;
  }
}

bool_t acTag::isHighTagNumberFormat() {
  if ((acTagType & acTag::highTagNumberForm) == highTagNumberForm) {
    return TRUE;
  } else {
    return FALSE;
  } 
}

unsigned char acTag::maskClass() {
  return  (((unsigned char)acTagType) & 0xC0);
}

bool_t acTag::isPrimitive() { return !isConstructed(); }


/*
* It is assumed that argument "tagvalue" is simply the numeric value of
* the explicit tag. It has a minimum value of 0 and a maximum value
* of 0xFFFFFFFF (maximum 32 bit uint32_t = 2^32 -1).
*
* the second argument is assumed to be a pointer to character array
* of at most 6 bytes in length (1 tag + 5 optional highTagNumberFormat octets).
* The number 5 comes from the assumption of a maximum value of 0xFFFFFFFF
* encoded as :
*   octets=     [bbb1 1111] [0x0f]   [0x7F]   [0x7F]   [0x7F]   [0xFF]
*   description  ^^^tag^^^^  ^^MSD..high tag number form octets..LSD^^
*   index=           0         1        2        3        4        5
* (see ASN.1 *tag* encoding rules if this unclear)
*
* if low tag number format is used (i.e. tagvalue < 31) then no extra tag octets
* are needed (only inital tag octet), and it is encoded as (VVVVV=tag value<31):
* 	[bbbV VVVV]
*	^^^tag^^^^ 
*/
#if 1
#define MAXUINT32       4294967295      /* 0xffffffff */
#define QUADRUPLED128    268435456      /* 128^4 */
#define CUBED128           2097152      /* 128^3 */
#define SQUARED128           16384      /* 128^2 */
 
void acTag::acTagEncode() {
 
  uint32_t n=acTagValue, nthis=0;
 
  acLastTOctetIndex = 0;
 
  if (n < 31) {
    /* *** LOW TAG NUMBER FORMAT USED *** */
    acHighTagFormat = FALSE;
    acTOctets[0] |= (unsigned char) n;
  } else {
    /* *** HIGH TAG NUMBER FORMAT USED *** */
    acHighTagFormat = TRUE;
    acTOctets[0] |= 0x1f;       /* encode it too */
 
    if (n >= QUADRUPLED128) {
      nthis = n/QUADRUPLED128;  /* remainder rounded off */
      n -= (nthis*QUADRUPLED128);
      acTOctets [++acLastTOctetIndex] = ((unsigned char)(nthis) & 0x7F);
    }
    if (n >= CUBED128) {
      nthis = n/CUBED128;       /* remainder rounded off */
      n -= (nthis*CUBED128);
      acTOctets [++acLastTOctetIndex] = ((unsigned char)(nthis) & 0x7F);
    }
    if (n >= SQUARED128) {
      nthis = n/SQUARED128;     /* remainder rounded off */
      n -= (nthis*SQUARED128);
      acTOctets [++acLastTOctetIndex] = ((unsigned char)(nthis) & 0x7F);
    }
    if (n >= 128) {
      nthis = n/128;            /* remainder rounded off */
      n -= (nthis*128);
      acTOctets [++acLastTOctetIndex] = ((unsigned char)(nthis) & 0x7F);
    }
    if (n >= 0) {
      acTOctets [++acLastTOctetIndex] = ((unsigned char)(n) | 0x80);
        /* last octet (LSD) must have high bit = 1 */
        /* others (MSD,...) must have high bit = 0 */
    }
  }
}
#else
 /* UGGH, THERE IS A BUG BELOW!!! this only works for low tag number format */
 /* JFR must find it, but old brute force code works fine for now ... */
void acTag::acTagEncode() {

  uint32_t n=acTagValue, nthis=0;
  
  acLastTOctetIndex = 0;

  if (n < 31) {	/* not '11111' binary */
	
    /* Low Tag Number Format used */

    acHighTagFormat = FALSE;
    acTOctets[0] |= (unsigned char) n;

  } else {

    /* High Tag Number Format used */

    acHighTagFormat = TRUE;
    acTOctets[0] |= 0x1f;	/* encode it too */

    int maxpow = MAX_TAG_OCTETS - 2;
      /* maxpow: 6 - 2 = 4; octs: [tag][128^4][128^3][128^2][128^1][128^0] */

    for (int pow=maxpow; pow>=0; pow--) {
      nthis = n/(128^pow);	/* integer division: remainder rounded off */
      n -= nthis * (128^pow);	/* Length encoded base 128 */
      if (pow != 0) {
        acTOctets [++acLastTOctetIndex] = ((unsigned char)(nthis) & 0x7f);
	 /* most digits (MSD,...) must have high bit set to 0 */
      } else {
       acTOctets [++acLastTOctetIndex] = ((unsigned char)(n) | 0x80);
	/* last octet (LSD) must have high bit set to 1 */
      }
    }

  }
}
#endif

/*

*/

/****************
 * Length Field *
 ****************/

/*
 * Construct (decode) the Length field from the physical octets
 */
acLen::acLen (char* octp) {
  int  ix=0;

  if (octp) {
    if ((unsigned char)octp[0] == INDEFINITE_LENGTH) {	/* Indefinite Length */
      acLenType		= indefinite;
      acLenValue		= 0;
      acLOctets[0]		= INDEFINITE_LENGTH;
      acLastLOctetIndex	= 0;
    } else if (!(octp[0] & 0x80)) {	/* Short Definite Length */
      acLenType		= shortDefinite;
      acLenValue		= (uint32_t)(octp[0]);
      acLOctets[0]		= octp[0];
      acLastLOctetIndex	= 0;
    } else {					/* Long Definite Length */
      acLenType		= longDefinite;
      acLastLOctetIndex	= (uint32_t)(octp[0] & 0x7F);
          // how many octets are used to encode length value (1..4)
      acLOctets[0]		= octp[0];

      for (int ix=1; ix<=acLastLOctetIndex; ix++) {
        /* base 256 arithmetic : */
        /*   [n*256^3] [n*256^3] .. [n*256^0] */
        /*    octp[1]   octp[2]  ..  octp[4]  */
        acLenValue += (uint32_t)(octp[ix]) * 256^(MAX_LEN_OCTETS-1 - ix);
          /* all 8 bits (base 256) are used */
      }

    }
  }
}

acLen::acLen (uint32_t lenval, LType lt) {
  acLenType = lt;
  acLenValue = lenval;
  acLastLOctetIndex = 0;
  acLenEncode ();
	// may modify Length type based on lenval
}

void acLen::print (ostream& os) {
  os << "[ ";

  if (acLenType == indefinite) {
    os << "(indefinite length)";
  } else {
    os << int(acLenValue);
  }

  os << " [Hex Octets: ";
  for (int ix = 0; ix <= acLastLOctetIndex; ix++) {
    os << hex(((uint32_t)(acLOctets[ix])) & 0xFF) << " ";
  }

  os << "] ] ";
}

void acLen::ask (istream&) {}

bool_t acLen::isShortDefinite() { 
  if (acLenType == shortDefinite) return TRUE;
  else return FALSE;
}

bool_t acLen::isLongDefinite() { 
  if (acLenType == longDefinite) return TRUE;
  else return FALSE;
}

bool_t acLen::isIndefinite() { 
  if (acLenType == indefinite) return TRUE;
  else return FALSE;
}


/*
* It is assumed that argument "lenval" is simply the numeric value of
* the explicit len. It has a minimum value of 0 and a maximum value
* of 0xFFFFFFFF (maximum 32 bit uint32_t = 2^32).
*
* one modified class member is assumed to be a pointer to character array
* of at most 5 bytes in length (1 len + 4 optional Long Definite Length octets).
* The number 5 comes from the assumption of a maximum value of 0xFFFFFFFF
* encoded as :
*   octets=     [1 0000100] [0xFF][0xFF][0xFF][0xFF]
*   description    ^len=4^  ^^MSD.. base 256 ..LSD^^
*   index=           0         1     2     3     4
* (see ASN.1 *len* encoding rules if this unclear)
*
* if short definite format is used (i.e. 0 <= lenval <= 127) then only one
* octet is needed:
*		[0bbbbbbb]
* bit 8 = 0 => short format
* bits 1-7 (bbbbbbb) have value of 0..127
*
* if indefinite length format is used, then only one length octet is needed:
*	[1111 1111] = 0xFF (?)
*/

void acLen::acLenEncode() {

  uint32_t n=acLenValue, nthis=0;
  
  acLastLOctetIndex = 0;

  if (acLenType == indefinite) {
    acLOctets[0] = INDEFINITE_LENGTH;
    return;
  } else if (n < 128) { /* short definite form */
    acLenType = shortDefinite;
    acLOctets[0] = ((unsigned char)n & 0x7F);	/* clear bit 8 */
  } else { /* long definite form */
    acLenType = longDefinite;

    int maxpow = MAX_LEN_OCTETS - 2;
      /* maxpow: 5 - 2 = 3, octs: [nocts][256^3][256^2][256^1][256^0] */

    for (int pow=maxpow; pow >= 0; pow--) {
      nthis = n/(256^pow);	/* remainder rounded off */
      n -= (nthis * (256^pow));
      if (pow != 0) {
        acLOctets [++acLastLOctetIndex] = (unsigned char)(nthis);
      } else {
        acLOctets [++acLastLOctetIndex] = (unsigned char)(n);
      }
    }

    // acLastLOctetIndex in theory should have value between: 0 .. 126
    // but in practice (i.e. this implementation) it never exceeds 4
    // which is well within the theoretical limit

    acLOctets[0] = (unsigned char)(acLastLOctetIndex) | 0x80;
	/* first octet contains # len octets, with high bit (8) set to 1 */
  }
}
