/***********************************************************************
 * Copyright (c) 1993 Technical Research Centre of Finland
 * All rights reserved.
 *
 * This software is provided ``as is'' and without any express or
 * implied warranties, including, without limitation, the implied
 * warranties of merchantibility and fitness for a particular purpose.
 **********************************************************************/

/* @(#)asncerial.cc	1.10 92/11/17 */

#include "env_sel.h"
#include <stdio.h>
#include <rpc/rpc.h>
#include <rpc/xdr.h>

#if ! CERIAL_CPLUSPLUS_INCLUDE
#include "xdr.ext"
#include "str.ext"
#endif
#include "sstream.h"	/* socket stream */
#include "cerial.h"	/* abstract "C" serializer */
#include "asntl.h"	/* asn.1 tag & length types */
#include "asncerial.h"	/* asn.1 serializer */

#include "asn1ber.ext"	/* extern "C" ASN.1 encoder/decoder fns */

////////Tag Defines (UPT=Universal Primitive Type)/////////
#define UPT_BOOLEAN		0x01
#define UPT_INTEGER		0x02
#define UPT_OCTETSTRING		0x04
#define UPT_VOID		0x05
#define UPT_PRINTABLESTRING	0x13

///////Length Defines (SDL=Short Definite Length) ///////
#define SDL_BOOLEAN		0x01
#define SDL_INTEGER		0x04
//#define SDL_OCTETSTRING	(unknown in advance)	
#define SDL_VOID		0x00
//#define SDL_PRINTABLESTRING	(unknown in advance)

////////Tag Defines (UCT=Universal Constructed Types)////
#define UCT_SEQUENCE		0x30	/* ctor + id=16 */

////// Local Types ///////

union uint32poke_t {
  u_long value;
  char  octets[4];
};


///////////////// hacker routines ///////////////////////////

#if CERIAL_DEBUGGING
bool_t AsnBerCerial::putbytes(char* bufp, u_long buflen) {
  if (!isOk || !bufp || buflen<=0)
    return FALSE;
  return (*putBytesFn_)(stream_->xdrstrm, bufp, buflen);
}

bool_t AsnBerCerial::getbytes(char* bufp, u_long buflen) {
  if (!isOk || !bufp || buflen<=0)
    return FALSE;
  if (valueInReadBuffer)
    getbytesFromReadBuffer(bufp, buflen);
  else
    return getBytesFn_(stream_->xdrstrm, bufp, buflen);
}
#endif /* CERIAL_DEBUGGING */
   
bool_t AsnBerCerial::getbytesFromReadBuffer(char* bufp, u_long buflen)	{
  *bufp++ = readBuffer;
  valueInReadBuffer = FALSE;
  if (--buflen == 0)
    return TRUE;
  return (*((XBYTFN)getBytesFn_))(stream_->xdrstrm, bufp, buflen);
}

char AsnBerCerial::readbyte () {
  if (!valueInReadBuffer)
    valueInReadBuffer = getbytes(&readBuffer, 1);
  //what if failed ?
  return readBuffer;
}

///////////////// AsnBerCerial ////////////////////////////////////

//
// Questions
//   - char*, uchar* serialized as octet or printable string ?
//   - char** (printable string, null terminated)
//
// ToDo:
//   - encoding for float/doubles
//   - support for indefinite length strings ?
//   - better error handling ?
//
// WARNING:
//   - Explicit tagging support only, no support for Implicit tags...
//


/*
 * "isOk" is a flag to indicate whether or not it is valid to
 * continue encoding/decoding, we don't encode/decode if it's
 * no longer valid to do so...
 */

void AsnBerCerial::serialize(char* arg) {
  if (isOk) {
#if 1	/*transmit as INTEGER*/
    int tmp = (int)*arg;
    this -> serialize (&tmp);
    *arg = (char)tmp;
#else	/*transmit as 1 byte OCTETSTRING*/
    serialize_opaque(arg, 1);
#endif
  }
}

void AsnBerCerial::serialize(u_char* arg)	{
  if (isOk) {
#if 1	/*transmit as INTEGER*/
    int tmp = (int)*arg;
    this -> serialize (&tmp);
    *arg = (char)tmp;
#else	/*transmit as 1 byte OCTETSTRING*/
    serialize_opaque((char*)arg, 1);	/*?*/
#endif
  }
}

#define MAXUINT32       4294967295 	/* 2^32 */ 

void AsnBerCerial::serialize_string(char** arg)	{

  // serialized as PRINTABLESTRING 

  if (arg == 0) {
    cerr << "***** Error: char** must not be 0 in AsnBerCerial::serialize(char**).  Ignored.\n";
    return ;
  }

  static char* zeroLengthString = "";	//?
  if (*arg == 0 && xop == CERIAL_ENCODE)	//If nothing to encode
    *arg = zeroLengthString;		//send a 0-length string. ?

  char	enc = UPT_PRINTABLESTRING;
  char	dec[MAX_LEN_OCTETS] ;	// definite length max = 5 bytes

  u_long decodedLengthValue=0;
 
  if (!isOk && mode()==CERIAL_FREE && arg && *arg) {
     DELETE(*arg);
     *arg = 0;
  }

  if (isOk) {
 
    switch (this -> xop) {
 
    case CERIAL_ENCODE:
     {
      char*	bufp = *arg;
      int	buflen = ::strlen(bufp);
      acLen	lenFld (buflen);	// encode the length field
      /* Id */
      if ((isOk = this -> putbytes (&enc, 1))) {
        /* Length */
        if ((isOk = this -> putbytes (lenFld.acLOctets, lenFld.noctets()))) {
           /* Contents */
           isOk = this -> putbytes (bufp, buflen);
        }
      }
     }
      break;
 
    case CERIAL_DECODE:
      /* Id */
      if ((isOk = this -> getbytes (dec, 1))) {
        if (dec[0] == enc) {
          /* Length */
          if ((isOk = this -> getbytes (dec, 1))) {
            if ((dec[0] & 0x80)) {
              if ((((unsigned char)dec[0]) == INDEFINITE_LENGTH)) {
                /* Indefinite Length format */
                isOk = FALSE;
                cerr << "Sorry: No support for decoding indefinite"
                     << "length strings\n";
                /* JFR ToDo: add support for indefinite length strings */
              } else {
                /* Long format */
                if ((isOk = this -> getbytes(&dec[1], (uint32_t)dec[0]))) {
                  acLen tmpLen(dec);
                  decodedLengthValue = tmpLen.value();
                }
              }
            } else {
              /* Short format */
              decodedLengthValue = (u_long)(dec[0]&0x7f);
            }

            if (!isOk) return;	// some decoding error occurred already

            /* Contents */
	    if (!*arg) {
		*arg = NEW(char[decodedLengthValue+1]);
	      //cerr << "@ allocated " << decodedLengthValue+1 << " chars\n";
	    }
	    //else hopefully enough space in *arg
	    
            char* bufp = *arg;
            if ((isOk = this -> getbytes (bufp, decodedLengthValue))) {
              bufp[decodedLengthValue] = 0; // null terminate printable string
            } else {
              isOk = FALSE;
            }

          } else {
            isOk = FALSE;
          }

        } else {
          isOk = FALSE;
        }

      }
      break;
    }
  }
}

/*************************************************************************
*
*       ASN.1 BER Encoding/Decoding primitive routines
*
*************************************************************************/
 
/*
 * Note:
 *  UBC SNACC MinBuf C libs used for ASN.1 BER encoding/decoding
 */
 
 
/*
*       Signed Integer Types
*/

static char pcScratchBuf[64];
 /* JFRBUG:
  *  I tried to use: char *pcBuf = (char*)malloc(7); with a free(pcBuf)
  *  at end of method, but free gets SEGMENTATION VIOLATION, why ?
  */
 
void AsnBerCerial::serialize (long* lArg) {
  int   enclen=0;                                       // # encoded bytes
  unsigned long declen=0;                               // # decoded bytes
  char  *pcBuf = (char*)::pcScratchBuf;                 // scratch char buffer
 
  if (isOk) {
    if (this -> xop == CERIAL_ENCODE) {
      enclen = BerEncodeInt (&pcBuf, lArg);             // encode into pcBuf
      isOk   = this -> putbytes (pcBuf, enclen);        // write pcBuf to XDR*
    } else if (this -> xop == CERIAL_DECODE) {
      isOk   = this -> getbytes (pcBuf, 2);             // get tag + length
      isOk   = this -> getbytes (&pcBuf[2], (0xff&pcBuf[1]));
                                                        // get contents bytes
      BerDecodeInt (&pcBuf, lArg, &declen);
    }
  }
}

void AsnBerCerial::serialize (int* iArg) {
  if (isOk) {
    long lTmp = (long)(*iArg);          // encoder
    this->serialize(&lTmp);
    *iArg = (int)lTmp;                  // decoder
  }
}
 
void AsnBerCerial::serialize (short* sArg) {
  if (isOk) {
    long lTmp = (long)(*sArg);          // encoder
    this->serialize(&lTmp);
    *sArg = (short)lTmp;                        // decoder
  }
}




/*
*       Unigned Integer Types
*/
 
void AsnBerCerial::serialize(u_long* ulArg)       {
  int   enclen=0;                                       // # encoded bytes
  unsigned long declen=0;                               // # decoded bytes
  char  *pcBuf = (char*)::pcScratchBuf;                 // scratch char buffer
 
  if (isOk) {
    if (this -> xop == CERIAL_ENCODE) {
      enclen = BerEncodeUint (&pcBuf, ulArg);           // encode into pcBuf
      isOk   = this -> putbytes (pcBuf, enclen);        // write pcBuf to XDR*
    } else if (this -> xop == CERIAL_DECODE) {
      isOk   = this -> getbytes (pcBuf, 2);             // get tag + length
      isOk   = this -> getbytes (&pcBuf[2], (0xff&pcBuf[1]));
                                                        // get contents bytes
      BerDecodeUint (&pcBuf, ulArg, &declen);
    }
  }
}

void AsnBerCerial::serialize(u_int* uiArg) {
  if (isOk) {
    u_long ulTmp = (u_long)(*uiArg);    // encoder
    this->serialize(&ulTmp);
    *uiArg = (u_int)ulTmp;              // decoder
  }
}
 
void AsnBerCerial::serialize(u_short* usArg) {
  if (isOk) {
    u_long ulTmp = (u_long)(*usArg);    // encoder
    this->serialize(&ulTmp);
    *usArg = (u_short)ulTmp;            // decoder
  }
}


/*
*       Real Types
*/
 
void AsnBerCerial::serialize(double* dArg) {
  int   enclen=0;                                       // # encoded bytes
  unsigned long declen=0;                               // # decoded bytes
  char  *pcBuf = (char*)::pcScratchBuf;                 // scratch char buffer
 
  if (isOk) {
    if (this -> xop == CERIAL_ENCODE) {
      enclen = BerEncodeReal (&pcBuf, dArg);            // encode into pcBuf
      isOk   = this -> putbytes (pcBuf, enclen);        // write pcBuf to XDR*
    } else if (this -> xop == CERIAL_DECODE) {
      isOk   = this -> getbytes (pcBuf, 2);             // get tag + length
      isOk   = this -> getbytes (&pcBuf[2], (0xff&pcBuf[1]));
                                                        // get contents bytes
      BerDecodeReal (&pcBuf, dArg, &declen);
    }
  }
}
 
void AsnBerCerial::serialize(float* fArg) {
  if (isOk) {
    double dTmp = (double)(*fArg);      // encoder
    this->serialize(&dTmp);
    *fArg = (float)dTmp;                        // decoder
  }
}

/* MORE TYPES */

#if 0
void AsnBerCerial::serialize_enum(enum_t* arg)    {
  if (isOk) {
    int tmp = (long)(*arg);
    this -> serialize (&tmp);
    *arg = (enum_t)tmp;
  }
}
#endif

void AsnBerCerial::serialize_bool(bool_t* arg)	{
  // serialized as BOOLEAN 

  char enc[3], dec[3];

  if (isOk) {
    enc[0] = UPT_BOOLEAN;
    enc[1] = SDL_BOOLEAN;

    switch (this -> xop) {

    case CERIAL_ENCODE:
      enc[2]= (char)(*arg ? TRUE : FALSE) ;
      isOk = this -> putbytes (enc, 3);
      break;

    case CERIAL_DECODE:
      if ((isOk = this -> getbytes (dec, 3))) {
        if((dec[0] == enc[0]) && (dec[1] == enc[1])) {
          *arg = ((bool_t)dec[2] == FALSE) ? FALSE : TRUE;
        } else {
          isOk = FALSE;
        }
      }
      break;
    }
  }
}

#if 0
void AsnBerCerial::serialize(void* arg)	{
  char enc[2], dec[2];
 
  if (isOk) {
    enc[0] = UPT_VOID;
    enc[1] = SDL_VOID;
 
    switch (this -> xop) {
 
    case CERIAL_ENCODE:
      isOk = this -> putbytes (enc, 2);
      break;
 
    case CERIAL_DECODE:
      if ((isOk = this -> getbytes (dec, 2))) {
        if(!((dec[0] == enc[0]) && (dec[1] == enc[1]))) {
          isOk = FALSE;
        }
      }
      break;
    }
  }
}
#endif

void AsnBerCerial::serialize_opaque(char* opqp, u_int opqlen) {
  // serialized as OCTETSTRING 

  char enc = UPT_OCTETSTRING;
  char dec[MAX_LEN_OCTETS] ;	// definite length max = 5 bytes

  acLen lenFld (opqlen);	// encode the length field
 
  if (isOk) {
 
    switch (this -> xop) {
 
    case CERIAL_ENCODE:
      /* Id */
      if ((isOk = this -> putbytes (&enc, 1))) {
        /* Length */
        if ((isOk = this -> putbytes (lenFld.acLOctets, lenFld.noctets()))) {
           /* Contents */
           isOk = this -> putbytes (opqp, opqlen);
        }
      }
      break;
 
    case CERIAL_DECODE:
      /* Id */
      if ((isOk = this -> getbytes (dec, 1))) {
        if (dec[0] == enc) {
          /* Length */
          if ((isOk = this -> getbytes (dec, lenFld.noctets()))) {
            for (int ix=0; ix < lenFld.noctets(); ix++) {
              if (dec[ix] != lenFld.acLOctets[ix]) {
                /* if decoded and expected lengths don't match */
                isOk = FALSE;
                return ;
              }
            }

            /* Otherwise length fields matched, get contents */

            /* Contents */
            if (!(isOk = this -> getbytes (opqp, opqlen))) {
              isOk = FALSE;
            }
          } else { isOk = FALSE; }
        } else {
          isOk = FALSE;
        }
      }
      break;
    }
  }
}


  //for performance - must not be changed
static char asnBerCerialSerializeSEQUENCE_t_enc[2] = {
  UCT_SEQUENCE,
  INDEFINITE_LENGTH
};
 
void AsnBerCerial::serialize(SEQUENCE_t) {
  if (isOk) {
    switch (this -> xop) {
 
    case CERIAL_ENCODE:
      isOk = this -> putbytes (asnBerCerialSerializeSEQUENCE_t_enc, 2);
      break;
 
    case CERIAL_DECODE: {
      char dec[2];
      if ((isOk = this -> getbytes(dec, 2))) {
        if (!((dec[0] == UCT_SEQUENCE) &&
              (dec[1] == asnBerCerialSerializeSEQUENCE_t_enc[1])))
           isOk = FALSE;
      }
      }
      break;
    }
  }
}

void AsnBerCerial::serialize(SEQUENCEOF_t name) {
  //?SEQUENCEOF tag?
  serialize(SEQUENCE_t(name));	//?
}

  //for performance - must not be changed
static char asnBerCerialSerializeEOC_t_eoc[2] = {
  0x00,
  0x00
};
 
void AsnBerCerial::serialize(EOC_t) {
  if (isOk) {
    switch (this -> xop) {
 
    case CERIAL_ENCODE:
      isOk = this -> putbytes (asnBerCerialSerializeEOC_t_eoc, 2);
      break;
 
    case CERIAL_DECODE: {
      char dec[2];
      if ((isOk = this -> getbytes(dec, 2))) {
        if (!((dec[0] == 0x00) && (dec[1] == 0x00)))
           isOk = FALSE;
      }
      }
      break;
    }
  }
}

void AsnBerCerial::serialize(acTag* ptag) {
  char buf[MAX_TAG_OCTETS];
  int ix = 0;


  if (isOk) {
    switch (this -> xop) {
 
    case CERIAL_ENCODE:
      isOk = this -> putbytes (ptag -> acTOctets, ptag -> noctets());
      break;

    case CERIAL_DECODE:
      if ((isOk = this -> getbytes (buf, 1))) {
	if ((buf[0] & 0x1F) == 0x1F) {
          /* High Tag Number Format */
          ix = 1;
          while ((isOk = this -> getbytes(&buf[ix], 1))) {
             if (!(buf[ix] & 0x80)) {
               /* LSD (last digit) found */
               break;
             }
          }
        } else {
          /* Low Tag Number Format */
        }
      }

      {
	/* KEEP CFRONT HAPPY */
        /* sorry, not implemented: non trivial declaration in switch	*/
        /* statement (try enclosing it in a block)			*/
        acTag tmpTag(buf);
        *ptag = tmpTag;
      }

      break;
    }
  }

}

void AsnBerCerial::serialize(acLen* plen) {
  char buf[MAX_LEN_OCTETS];

  if (isOk) {
    switch (this -> xop) {
 
    case CERIAL_ENCODE:
      isOk = this -> putbytes (plen -> acLOctets, plen -> noctets());

      break;

    case CERIAL_DECODE:
      if ((isOk = this -> getbytes (buf, 1))) {
        if (((unsigned char)buf[0] == INDEFINITE_LENGTH) || (!(buf[0] & 0x80))){
          /* 1byte: Indefinite Length or Short Definite Length*/
        } else {
	  /* Long Definite Length */
          uint noctets = (uint)(buf[0] & 0x7F);
          this -> getbytes (&buf[1], noctets);
        }
      }

      {
        /* KEEP CFRONT HAPPY */
        /* sorry, not implemented: non trivial declaration in switch    */
        /* statement (try enclosing it in a block)                      */
        acLen tmpLen(buf);
        *plen = tmpLen;
      }

      break;
    }
  }
}

void AsnBerCerial::serialize_explicit (acTag* ptag) {

    if (isOk) {
      switch (this -> xop) {

      case CERIAL_ENCODE: {
        static acLen lenField(0, acLen::indefinite); // indefinite length
	this -> serialize (ptag);
	if (isOk) { this -> serialize (&lenField); }
	break; 
      }

      case CERIAL_DECODE: {
        acLen lenField(0, acLen::indefinite); // indefinite length
	this -> serialize (ptag);
	if (isOk) { this -> serialize (&lenField); }
	if (!lenField.isIndefinite()) { isOk = FALSE; }
	break;
      }
      }
    }
}

bool_t AsnBerCerial::serialize_presence(SerializePointerArguments& a) {
  if (this->mode() == CERIAL_ENCODE) {
    bool_t isPresent = (a.pp && *a.pp);
    return isPresent;
  }
  else if (this->mode() == CERIAL_DECODE) {
    char incomingByte = readbyte();
    if ((incomingByte & 0x1F) == 0x1F) {
      cerr << "***** Sorry, not supported: optional tag values > 30\n";
      return (isOk = FALSE);
    }
    acTag incomingTag = &incomingByte; 
      //The generated tags are always context specific 
      //and explicit (and thus constructed)
    acTag expectedTag(acTag::contextSpecific | acTag::constructed, a.tag);
    //cerr << "@ incoming  = "; incomingTag.print(cerr); cerr << "\n";
    //cerr << "@ expecting = "; expectedTag.print(cerr); cerr << "\n";
    return (expectedTag == incomingTag);
  }
}

cerial::TransferSyntax AsnBerCerial::transferSyntax() const {
  return cerial::asn1BerTransferSyntax;
}

AsnBerCerial::AsnBerCerial(xdr_op op, StreamFamily* stream)
  : cerial(op, stream)
  , valueInReadBuffer(FALSE)
{
  if (!stream->xdrstrm)
    cerr << "Error: StreamFamily base class should be initialized before AsnBerCerial\n";
  if (stream->xdrstrm->x_ops) {
    putBytesFn_ = (XBYTFN)stream->xdrstrm->x_ops->x_putbytes;
    getBytesFn_ = (XBYTFN)stream->xdrstrm->x_ops->x_getbytes;
  }
  //else CERIAL_FREE mode
}

AsnBerFileCerial::AsnBerFileCerial(xdr_op op, FILE* fp)
  : FileStream(op, fp)
  , AsnBerCerial(op, this)
{}

#if SSTREAMS
AsnBerSocketCerial::AsnBerSocketCerial(xdr_op op, char* host, u_int port)
  : SocketStream(op, host, port)
  , AsnBerCerial(op, this)
{}
#endif

AsnBerMemoryCerial::AsnBerMemoryCerial(xdr_op op, u_int size, char* addr) 
  : MemoryStream(op, size, addr)
  , AsnBerCerial(op, this)
{}


