# flake8: noqa
"""
The following contains the S22 and s26 databases of weakly interacting dimers and complexes

S22 geometry data are from
P. Jurecka, J. Sponer, J. Cerny, P. Hobza; Phys Chem Chem Phys 2006, 8 (17), 1985-1993.
See http://www.begdb.com/index.php?action=106a6c241b8797f52e1e77317b96a201 for the original files.
All geometries are optimized at either the CCSD(T) or MP2 level except for the methyl amide dimers 
where only the hydrogen position is optimized at the DFT level (the precise optimization is written as a comment).

The S22 interaction energies are all calculated using both CCSD(T)/CBS counter poised corrected (CP) and MP2 /CBS CP.
The original S22 interaction energies are listed in the above references.
The S22 energies used here are from
Takatani, T. et al., J. Chem. Phys., 132, 144104 (2010)
where a large and more complete basis set has been used for all database members.

The original S22 set has been expanded with an extra 4 single hydrogen bonded complexes.
The expanded set is called S26. Data for the 4 extra dimers are from
Riley, K.E., Hobza, P., J. Chem. Phys. A, 111(33), 8257-8263 (2007).
Geometry optimizations: MP2/cc-pVTZ CP or DFT TPSS/TZVP noCP
Interaction energies: CCSD(T)/CBS CP or MP2/cc-pVDZ CP

The original s22 has also been expanded with 4 non-equilibrium structures for each system.
This defines the s22x5 database containing one shortened and three elongated structures:
0.9, 1.0, 1.2, 1.5 and 2.0 times the original intermolecular distance.
CCSD(T)/CBS interaction energies are consistent with the original s22 work.
Reference: L. Grafova, M. Pitonak, P. Hobza, J. Chem. Theo. Comput., 2010, ASAP article.
"""

from ase.atoms import Atoms

s22 = ['Ammonia_dimer','Water_dimer','Formic_acid_dimer','Formamide_dimer',
       'Uracil_dimer_h-bonded','2-pyridoxine_2-aminopyridine_complex',
       'Adenine-thymine_Watson-Crick_complex','Methane_dimer','Ethene_dimer',
       'Benzene-methane_complex','Benzene_dimer_parallel_displaced','Pyrazine_dimer',
       'Uracil_dimer_stack','Indole-benzene_complex_stack',
       'Adenine-thymine_complex_stack','Ethene-ethyne_complex','Benzene-water_complex',
       'Benzene-ammonia_complex','Benzene-HCN_complex','Benzene_dimer_T-shaped',
       'Indole-benzene_T-shape_complex','Phenol_dimer']

s26 = s22 + ['Methanol_dimer','Methanol-formaldehyde_complex',
             'Methyl_amide_dimer_alpha','Methyl_amide_dimer_beta'] 

s22x5 = ['Ammonia_dimer_0.9','Ammonia_dimer_1.0','Ammonia_dimer_1.2','Ammonia_dimer_1.5','Ammonia_dimer_2.0',
         'Water_dimer_0.9','Water_dimer_1.0','Water_dimer_1.2','Water_dimer_1.5','Water_dimer_2.0',
         'Formic_acid_dimer_0.9','Formic_acid_dimer_1.0','Formic_acid_dimer_1.2','Formic_acid_dimer_1.5','Formic_acid_dimer_2.0',
         'Formamide_dimer_0.9','Formamide_dimer_1.0','Formamide_dimer_1.2','Formamide_dimer_1.5','Formamide_dimer_2.0',
         'Uracil_dimer_h-bonded_0.9','Uracil_dimer_h-bonded_1.0','Uracil_dimer_h-bonded_1.2','Uracil_dimer_h-bonded_1.5','Uracil_dimer_h-bonded_2.0',
         '2-pyridoxine_2-aminopyridine_complex_0.9','2-pyridoxine_2-aminopyridine_complex_1.0',
         '2-pyridoxine_2-aminopyridine_complex_1.2','2-pyridoxine_2-aminopyridine_complex_1.5','2-pyridoxine_2-aminopyridine_complex_2.0',
         'Adenine-thymine_Watson-Crick_complex_0.9','Adenine-thymine_Watson-Crick_complex_1.0',
         'Adenine-thymine_Watson-Crick_complex_1.2','Adenine-thymine_Watson-Crick_complex_1.5','Adenine-thymine_Watson-Crick_complex_2.0',
         'Methane_dimer_0.9','Methane_dimer_1.0','Methane_dimer_1.2','Methane_dimer_1.5','Methane_dimer_2.0',
         'Ethene_dimer_0.9','Ethene_dimer_1.0','Ethene_dimer_1.2','Ethene_dimer_1.5','Ethene_dimer_2.0',
         'Benzene-methane_complex_0.9','Benzene-methane_complex_1.0','Benzene-methane_complex_1.2','Benzene-methane_complex_1.5','Benzene-methane_complex_2.0',
         'Benzene_dimer_parallel_displaced_0.9','Benzene_dimer_parallel_displaced_1.0',
         'Benzene_dimer_parallel_displaced_1.2','Benzene_dimer_parallel_displaced_1.5','Benzene_dimer_parallel_displaced_2.0',
         'Pyrazine_dimer_0.9','Pyrazine_dimer_1.0','Pyrazine_dimer_1.2','Pyrazine_dimer_1.5','Pyrazine_dimer_2.0',
         'Uracil_dimer_stack_0.9','Uracil_dimer_stack_1.0','Uracil_dimer_stack_1.2','Uracil_dimer_stack_1.5','Uracil_dimer_stack_2.0',
         'Indole-benzene_complex_stack_0.9','Indole-benzene_complex_stack_1.0',
         'Indole-benzene_complex_stack_1.2','Indole-benzene_complex_stack_1.5','Indole-benzene_complex_stack_2.0',
         'Adenine-thymine_complex_stack_0.9','Adenine-thymine_complex_stack_1.0',
         'Adenine-thymine_complex_stack_1.2','Adenine-thymine_complex_stack_1.5','Adenine-thymine_complex_stack_2.0',
         'Ethene-ethyne_complex_0.9','Ethene-ethyne_complex_1.0','Ethene-ethyne_complex_1.2','Ethene-ethyne_complex_1.5','Ethene-ethyne_complex_2.0',
         'Benzene-water_complex_0.9','Benzene-water_complex_1.0','Benzene-water_complex_1.2','Benzene-water_complex_1.5','Benzene-water_complex_2.0',
         'Benzene-ammonia_complex_0.9','Benzene-ammonia_complex_1.0','Benzene-ammonia_complex_1.2','Benzene-ammonia_complex_1.5','Benzene-ammonia_complex_2.0',
         'Benzene-HCN_complex_0.9','Benzene-HCN_complex_1.0','Benzene-HCN_complex_1.2','Benzene-HCN_complex_1.5','Benzene-HCN_complex_2.0',
         'Benzene_dimer_T-shaped_0.9','Benzene_dimer_T-shaped_1.0','Benzene_dimer_T-shaped_1.2','Benzene_dimer_T-shaped_1.5','Benzene_dimer_T-shaped_2.0',
         'Indole-benzene_T-shape_complex_0.9','Indole-benzene_T-shape_complex_1.0',
         'Indole-benzene_T-shape_complex_1.2','Indole-benzene_T-shape_complex_1.5','Indole-benzene_T-shape_complex_2.0',
         'Phenol_dimer_0.9','Phenol_dimer_1.0','Phenol_dimer_1.2','Phenol_dimer_1.5','Phenol_dimer_2.0']

data = {
# --- s22 and s22x5 ---#
'2-pyridoxine_2-aminopyridine_complex': {
    'description': "Complex, S22, S26, 2 h-bond, double h-bond, nucleic base model",
    'name': "2-pyridoxine_2-aminopyridine_complex",
    's26_number': "06",
    'interaction energy CC': -0.7372,
    'interaction energies s22x5': [-0.6561,-0.7242,-0.6041,-0.3547,-0.1414],
    'offset': 0.0130,
    'symbols': 'ONCCCCCHHHHHNCCCCCHHHHNHH',
    'magmoms': None,
    'dimer atoms': [12,13],
    # Optimisation level:  MP2/cc-pVTZ 
    'positions':[[ -1.3976213, -1.8858368, -0.3673061],
                 [ -1.4642550,  0.3641828,  0.0192301],
                 [ -4.1857398,  0.3696669,  0.0360960],
                 [ -3.4832598,  1.5783111,  0.2500752],
                 [ -2.1179502,  1.5307048,  0.2338383],
                 [ -2.0773833, -0.8637492, -0.1899414],
                 [ -3.5156032, -0.8051950, -0.1757585],
                 [ -5.2678045,  0.3707428,  0.0411419],
                 [ -3.9920334,  2.5127560,  0.4214414],
                 [ -1.4929196,  2.3984096,  0.3885018],
                 [ -4.0401226, -1.7348452, -0.3379269],
                 [ -0.4265266,  0.3612127,  0.0073538],
                 [  1.4327616,  0.3639703, -0.0159508],
                 [  2.1154200, -0.7803450,  0.1681099],
                 [  3.5237586, -0.8016096,  0.1545027],
                 [  4.2185897,  0.3735783, -0.0525929],
                 [  3.5099708,  1.5615014, -0.2449763],
                 [  2.1280138,  1.4953324, -0.2175374],
                 [  4.0459206, -1.7361356,  0.3076883],
                 [  5.2999426,  0.3666009, -0.0663349],
                 [  4.0110923,  2.5024313, -0.4130052],
                 [  1.5339878,  2.3893837, -0.3670565],
                 [  1.3883123, -1.9083038,  0.4198149],
                 [  1.8694714, -2.7812773,  0.2940385],
                 [  0.4089067, -1.9079942,  0.1300860]],
    'positions 0.9':[[  -0.969652624 ,  -2.245611164 ,  -0.386822525 ],
                     [  -1.037789793 ,  0.004508753 ,  -0.001131127 ],
                     [  -3.759261297 ,  0.014028068 ,  -0.018375760 ],
                     [  -3.057727058 ,  1.221631156 ,  0.204402100 ],
                     [  -1.692392879 ,  1.172000703 ,  0.205277859 ],
                     [  -1.650068007 ,  -1.222514751 ,  -0.217981663 ],
                     [  -3.088264390 ,  -1.161828225 ,  -0.221825966 ],
                     [  -4.841300764 ,  0.016708498 ,  -0.026892047 ],
                     [  -3.567221821 ,  2.156831083 ,  0.369386687 ],
                     [  -1.068064568 ,  2.038779450 ,  0.367771502 ],
                     [  -3.612088503 ,  -2.090701001 ,  -0.390563867 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  1.673493386 ,  0.000000000 ,  0.000000000 ],
                     [  2.352093429 ,  -1.145324213 ,  0.192591910 ],
                     [  3.760459273 ,  -1.168677470 ,  0.196637005 ],
                     [  4.459573002 ,  0.005477083 ,  -0.001723239 ],
                     [  3.755182987 ,  1.194447664 ,  -0.202961469 ],
                     [  2.372894041 ,  1.130328028 ,  -0.192845808 ],
                     [  4.279274134 ,  -2.103975233 ,  0.356345736 ],
                     [  5.541001766 ,  -0.003103367 ,  -0.001911235 ],
                     [  4.259765167 ,  2.134632052 ,  -0.364687797 ],
                     [  1.782114958 ,  2.025258423 ,  -0.349790900 ],
                     [  1.620216197 ,  -2.272201547 ,  0.435153550 ],
                     [  2.101618920 ,  -3.145888174 ,  0.315408858 ],
                     [  0.644520940 ,  -2.270442069 ,  0.133172072 ]],
    'positions 1.0':[[  -0.969652624000000 ,  -2.245611164000000 ,  -0.386822525000000 ],
                     [  -1.037789793000000 ,  0.004508753000000 ,  -0.001131127000000 ],
                     [  -3.759261297000000 ,  0.014028068000000 ,  -0.018375760000000 ],
                     [  -3.057727058000000 ,  1.221631156000000 ,  0.204402100000000 ],
                     [  -1.692392879000000 ,  1.172000703000000 ,  0.205277859000000 ],
                     [  -1.650068007000000 ,  -1.222514751000000 ,  -0.217981663000000 ],
                     [  -3.088264390000000 ,  -1.161828225000000 ,  -0.221825966000000 ],
                     [  -4.841300764000000 ,  0.016708498000000 ,  -0.026892047000000 ],
                     [  -3.567221821000000 ,  2.156831083000000 ,  0.369386687000000 ],
                     [  -1.068064568000000 ,  2.038779450000000 ,  0.367771502000000 ],
                     [  -3.612088503000000 ,  -2.090701001000000 ,  -0.390563867000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  1.859437095454546 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  2.538037138454545 ,  -1.145324213000000 ,  0.192591910000000 ],
                     [  3.946402982454545 ,  -1.168677470000000 ,  0.196637005000000 ],
                     [  4.645516711454546 ,  0.005477083000000 ,  -0.001723239000000 ],
                     [  3.941126696454545 ,  1.194447664000000 ,  -0.202961469000000 ],
                     [  2.558837750454545 ,  1.130328028000000 ,  -0.192845808000000 ],
                     [  4.465217843454545 ,  -2.103975233000000 ,  0.356345736000000 ],
                     [  5.726945475454546 ,  -0.003103367000000 ,  -0.001911235000000 ],
                     [  4.445708876454546 ,  2.134632052000000 ,  -0.364687797000000 ],
                     [  1.968058667454545 ,  2.025258423000000 ,  -0.349790900000000 ],
                     [  1.806159906454545 ,  -2.272201547000000 ,  0.435153550000000 ],
                     [  2.287562629454545 ,  -3.145888174000000 ,  0.315408858000000 ],
                     [  0.830464649454546 ,  -2.270442069000000 ,  0.133172072000000 ]],
    'positions 1.2':[[  -0.969652624 ,  -2.245611164 ,  -0.386822525 ],
                     [  -1.037789793 ,  0.004508753 ,  -0.001131127 ],
                     [  -3.759261297 ,  0.014028068 ,  -0.018375760 ],
                     [  -3.057727058 ,  1.221631156 ,  0.204402100 ],
                     [  -1.692392879 ,  1.172000703 ,  0.205277859 ],
                     [  -1.650068007 ,  -1.222514751 ,  -0.217981663 ],
                     [  -3.088264390 ,  -1.161828225 ,  -0.221825966 ],
                     [  -4.841300764 ,  0.016708498 ,  -0.026892047 ],
                     [  -3.567221821 ,  2.156831083 ,  0.369386687 ],
                     [  -1.068064568 ,  2.038779450 ,  0.367771502 ],
                     [  -3.612088503 ,  -2.090701001 ,  -0.390563867 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  2.231324514 ,  0.000000000 ,  0.000000000 ],
                     [  2.909924557 ,  -1.145324213 ,  0.192591910 ],
                     [  4.318290401 ,  -1.168677470 ,  0.196637005 ],
                     [  5.017404130 ,  0.005477083 ,  -0.001723239 ],
                     [  4.313014115 ,  1.194447664 ,  -0.202961469 ],
                     [  2.930725169 ,  1.130328028 ,  -0.192845808 ],
                     [  4.837105262 ,  -2.103975233 ,  0.356345736 ],
                     [  6.098832894 ,  -0.003103367 ,  -0.001911235 ],
                     [  4.817596295 ,  2.134632052 ,  -0.364687797 ],
                     [  2.339946086 ,  2.025258423 ,  -0.349790900 ],
                     [  2.178047325 ,  -2.272201547 ,  0.435153550 ],
                     [  2.659450048 ,  -3.145888174 ,  0.315408858 ],
                     [  1.202352068 ,  -2.270442069 ,  0.133172072 ]],
    'positions 1.5':[[  -0.969652624 ,  -2.245611164 ,  -0.386822525 ],
                     [  -1.037789793 ,  0.004508753 ,  -0.001131127 ],
                     [  -3.759261297 ,  0.014028068 ,  -0.018375760 ],
                     [  -3.057727058 ,  1.221631156 ,  0.204402100 ],
                     [  -1.692392879 ,  1.172000703 ,  0.205277859 ],
                     [  -1.650068007 ,  -1.222514751 ,  -0.217981663 ],
                     [  -3.088264390 ,  -1.161828225 ,  -0.221825966 ],
                     [  -4.841300764 ,  0.016708498 ,  -0.026892047 ],
                     [  -3.567221821 ,  2.156831083 ,  0.369386687 ],
                     [  -1.068064568 ,  2.038779450 ,  0.367771502 ],
                     [  -3.612088503 ,  -2.090701001 ,  -0.390563867 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  2.789155642 ,  0.000000000 ,  0.000000000 ],
                     [  3.467755685 ,  -1.145324213 ,  0.192591910 ],
                     [  4.876121529 ,  -1.168677470 ,  0.196637005 ],
                     [  5.575235258 ,  0.005477083 ,  -0.001723239 ],
                     [  4.870845243 ,  1.194447664 ,  -0.202961469 ],
                     [  3.488556297 ,  1.130328028 ,  -0.192845808 ],
                     [  5.394936390 ,  -2.103975233 ,  0.356345736 ],
                     [  6.656664022 ,  -0.003103367 ,  -0.001911235 ],
                     [  5.375427423 ,  2.134632052 ,  -0.364687797 ],
                     [  2.897777214 ,  2.025258423 ,  -0.349790900 ],
                     [  2.735878453 ,  -2.272201547 ,  0.435153550 ],
                     [  3.217281176 ,  -3.145888174 ,  0.315408858 ],
                     [  1.760183196 ,  -2.270442069 ,  0.133172072 ]],
    'positions 2.0':[[  -0.969652624 ,  -2.245611164 ,  -0.386822525 ],
                     [  -1.037789793 ,  0.004508753 ,  -0.001131127 ],
                     [  -3.759261297 ,  0.014028068 ,  -0.018375760 ],
                     [  -3.057727058 ,  1.221631156 ,  0.204402100 ],
                     [  -1.692392879 ,  1.172000703 ,  0.205277859 ],
                     [  -1.650068007 ,  -1.222514751 ,  -0.217981663 ],
                     [  -3.088264390 ,  -1.161828225 ,  -0.221825966 ],
                     [  -4.841300764 ,  0.016708498 ,  -0.026892047 ],
                     [  -3.567221821 ,  2.156831083 ,  0.369386687 ],
                     [  -1.068064568 ,  2.038779450 ,  0.367771502 ],
                     [  -3.612088503 ,  -2.090701001 ,  -0.390563867 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  3.718874190 ,  0.000000000 ,  0.000000000 ],
                     [  4.397474233 ,  -1.145324213 ,  0.192591910 ],
                     [  5.805840077 ,  -1.168677470 ,  0.196637005 ],
                     [  6.504953806 ,  0.005477083 ,  -0.001723239 ],
                     [  5.800563791 ,  1.194447664 ,  -0.202961469 ],
                     [  4.418274845 ,  1.130328028 ,  -0.192845808 ],
                     [  6.324654938 ,  -2.103975233 ,  0.356345736 ],
                     [  7.586382570 ,  -0.003103367 ,  -0.001911235 ],
                     [  6.305145971 ,  2.134632052 ,  -0.364687797 ],
                     [  3.827495762 ,  2.025258423 ,  -0.349790900 ],
                     [  3.665597001 ,  -2.272201547 ,  0.435153550 ],
                     [  4.146999724 ,  -3.145888174 ,  0.315408858 ],
                     [  2.689901744 ,  -2.270442069 ,  0.133172072 ]]},

'Adenine-thymine_complex_stack': {
    'description': "Complex, S22, S26, stack, dispersion bonded, nucleic base",
    'name': "Adenine-thymine_complex_stack",
    's26_number': "15",
    'interaction energy CC': -0.5056,
    'interaction energies s22x5':[-0.3465,-0.5299,-0.3569,-0.1409,-0.0399],
    'offset': -0.0243,
    'symbols': 'NCHNCCNHHNCHNCHNCHCCHHHCONHCOH',
    'magmoms': None,
    'dimer atoms': [15,15],
    # Optimisation level:  MP2/cc-pVTZ
    'positions':[[  0.2793014,  2.4068393, -0.6057517],
                 [ -1.0848570,  2.4457461, -0.5511608],
                 [ -1.6594403,  3.0230294, -1.2560905],
                 [ -1.5977117,  1.7179877,  0.4287543],
                 [ -0.4897255,  1.1714358,  1.0301910],
                 [ -0.3461366,  0.2914710,  2.1172343],
                 [ -1.4187090, -0.1677767,  2.8101441],
                 [ -1.2388750, -0.9594802,  3.4047578],
                 [ -2.2918734, -0.1788223,  2.3073619],
                 [  0.8857630, -0.0700763,  2.4919494],
                 [  1.9352348,  0.4072878,  1.7968022],
                 [  2.9060330,  0.0788414,  2.1458181],
                 [  1.9409775,  1.2242019,  0.7402202],
                 [  0.6952186,  1.5779858,  0.4063984],
                 [  0.8610073,  2.8298045, -1.3104502],
                 [  1.2754606, -0.6478993, -1.9779104],
                 [  1.4130533, -1.5536850, -0.9550667],
                 [  2.4258769, -1.8670780, -0.7468778],
                 [  0.3575976, -2.0239499, -0.2530575],
                 [  0.4821292, -3.0179494,  0.8521221],
                 [  0.1757705, -2.5756065,  1.7986281],
                 [ -0.1601691, -3.8770412,  0.6639498],
                 [  1.5112443, -3.3572767,  0.9513659],
                 [ -0.9684711, -1.5298112, -0.5939792],
                 [ -2.0029280, -1.8396957, -0.0199453],
                 [ -0.9956916, -0.6383870, -1.6720420],
                 [ -1.9014057, -0.2501720, -1.8985760],
                 [  0.0684702, -0.1191762, -2.3763759],
                 [ -0.0397875,  0.7227006, -3.2531083],
                 [  2.0853289, -0.2760176, -2.4454577]],
    'positions 0.9':[[  0.067390759 ,  1.213806097 ,  -1.171192513 ],
                     [  -0.034440687 ,  0.160916029 ,  -2.035179690 ],
                     [  -0.037909102 ,  0.307694674 ,  -3.102311444 ],
                     [  -0.122286497 ,  -1.014214485 ,  -1.431659388 ],
                     [  -0.061278153 ,  -0.690156063 ,  -0.097738525 ],
                     [  -0.083866474 ,  -1.480006435 ,  1.065121981 ],
                     [  -0.207551291 ,  -2.830167865 ,  1.008466281 ],
                     [  0.020236002 ,  -3.318294510 ,  1.858492777 ],
                     [  0.100823981 ,  -3.261839820 ,  0.151791829 ],
                     [  -0.015107287 ,  -0.872886238 ,  2.254820437 ],
                     [  0.095534438 ,  0.468473589 ,  2.286592142 ],
                     [  0.148443656 ,  0.902433537 ,  3.277055537 ],
                     [  0.150791629 ,  1.330817541 ,  1.268232413 ],
                     [  0.061278153 ,  0.690156063 ,  0.097738525 ],
                     [  0.213123816 ,  2.178532043 ,  -1.420082564 ],
                     [  2.995457244 ,  1.318912569 ,  0.115169333 ],
                     [  3.033773997 ,  0.544134785 ,  1.248235461 ],
                     [  3.166936649 ,  1.084216460 ,  2.174491246 ],
                     [  2.913123372 ,  -0.802036026 ,  1.213306349 ],
                     [  2.965573998 ,  -1.664227788 ,  2.429380731 ],
                     [  2.009790775 ,  -2.161867438 ,  2.585037720 ],
                     [  3.726416066 ,  -2.435033978 ,  2.315487569 ],
                     [  3.189128467 ,  -1.070628980 ,  3.313538183 ],
                     [  2.718644614 ,  -1.440326451 ,  -0.080379664 ],
                     [  2.558245305 ,  -2.640081851 ,  -0.255033817 ],
                     [  2.729839539 ,  -0.560837886 ,  -1.168484485 ],
                     [  2.554150647 ,  -0.977998743 ,  -2.072617562 ],
                     [  2.814781928 ,  0.814169728 ,  -1.152798148 ],
                     [  2.732113465 ,  1.513854058 ,  -2.149163262 ],
                     [  3.033823338 ,  2.322516737 ,  0.179118562 ]],
    'positions 1.0':[[  0.067390759000000 ,  1.213806097000000 ,  -1.171192513000000 ],
                     [  -0.034440687000000 ,  0.160916029000000 ,  -2.035179690000000 ],
                     [  -0.037909102000000 ,  0.307694674000000 ,  -3.102311444000000 ],
                     [  -0.122286497000000 ,  -1.014214485000000 ,  -1.431659388000000 ],
                     [  -0.061278153000000 ,  -0.690156063000000 ,  -0.097738525000000 ],
                     [  -0.083866474000000 ,  -1.480006435000000 ,  1.065121981000000 ],
                     [  -0.207551291000000 ,  -2.830167865000000 ,  1.008466281000000 ],
                     [  0.020236002000000 ,  -3.318294510000000 ,  1.858492777000000 ],
                     [  0.100823981000000 ,  -3.261839820000000 ,  0.151791829000000 ],
                     [  -0.015107287000000 ,  -0.872886238000000 ,  2.254820437000000 ],
                     [  0.095534438000000 ,  0.468473589000000 ,  2.286592142000000 ],
                     [  0.148443656000000 ,  0.902433537000000 ,  3.277055537000000 ],
                     [  0.150791629000000 ,  1.330817541000000 ,  1.268232413000000 ],
                     [  0.061278153000000 ,  0.690156063000000 ,  0.097738525000000 ],
                     [  0.213123816000000 ,  2.178532043000000 ,  -1.420082564000000 ],
                     [  3.314050951181818 ,  1.318912569000000 ,  0.115169333000000 ],
                     [  3.352367704181818 ,  0.544134785000000 ,  1.248235461000000 ],
                     [  3.485530356181818 ,  1.084216460000000 ,  2.174491246000000 ],
                     [  3.231717079181818 ,  -0.802036026000000 ,  1.213306349000000 ],
                     [  3.284167705181818 ,  -1.664227788000000 ,  2.429380731000000 ],
                     [  2.328384482181818 ,  -2.161867438000000 ,  2.585037720000000 ],
                     [  4.045009773181818 ,  -2.435033978000000 ,  2.315487569000000 ],
                     [  3.507722174181819 ,  -1.070628980000000 ,  3.313538183000000 ],
                     [  3.037238321181818 ,  -1.440326451000000 ,  -0.080379664000000 ],
                     [  2.876839012181818 ,  -2.640081851000000 ,  -0.255033817000000 ],
                     [  3.048433246181818 ,  -0.560837886000000 ,  -1.168484485000000 ],
                     [  2.872744354181818 ,  -0.977998743000000 ,  -2.072617562000000 ],
                     [  3.133375635181818 ,  0.814169728000000 ,  -1.152798148000000 ],
                     [  3.050707172181818 ,  1.513854058000000 ,  -2.149163262000000 ],
                     [  3.352417045181818 ,  2.322516737000000 ,  0.179118562000000 ]],
    'positions 1.2':[[  0.067390759 ,  1.213806097 ,  -1.171192513 ],
                     [  -0.034440687 ,  0.160916029 ,  -2.035179690 ],
                     [  -0.037909102 ,  0.307694674 ,  -3.102311444 ],
                     [  -0.122286497 ,  -1.014214485 ,  -1.431659388 ],
                     [  -0.061278153 ,  -0.690156063 ,  -0.097738525 ],
                     [  -0.083866474 ,  -1.480006435 ,  1.065121981 ],
                     [  -0.207551291 ,  -2.830167865 ,  1.008466281 ],
                     [  0.020236002 ,  -3.318294510 ,  1.858492777 ],
                     [  0.100823981 ,  -3.261839820 ,  0.151791829 ],
                     [  -0.015107287 ,  -0.872886238 ,  2.254820437 ],
                     [  0.095534438 ,  0.468473589 ,  2.286592142 ],
                     [  0.148443656 ,  0.902433537 ,  3.277055537 ],
                     [  0.150791629 ,  1.330817541 ,  1.268232413 ],
                     [  0.061278153 ,  0.690156063 ,  0.097738525 ],
                     [  0.213123816 ,  2.178532043 ,  -1.420082564 ],
                     [  3.951238365 ,  1.318912569 ,  0.115169333 ],
                     [  3.989555118 ,  0.544134785 ,  1.248235461 ],
                     [  4.122717770 ,  1.084216460 ,  2.174491246 ],
                     [  3.868904493 ,  -0.802036026 ,  1.213306349 ],
                     [  3.921355119 ,  -1.664227788 ,  2.429380731 ],
                     [  2.965571896 ,  -2.161867438 ,  2.585037720 ],
                     [  4.682197187 ,  -2.435033978 ,  2.315487569 ],
                     [  4.144909588 ,  -1.070628980 ,  3.313538183 ],
                     [  3.674425735 ,  -1.440326451 ,  -0.080379664 ],
                     [  3.514026426 ,  -2.640081851 ,  -0.255033817 ],
                     [  3.685620660 ,  -0.560837886 ,  -1.168484485 ],
                     [  3.509931768 ,  -0.977998743 ,  -2.072617562 ],
                     [  3.770563049 ,  0.814169728 ,  -1.152798148 ],
                     [  3.687894586 ,  1.513854058 ,  -2.149163262 ],
                     [  3.989604459 ,  2.322516737 ,  0.179118562 ]],
    'positions 1.5':[[  0.067390759 ,  1.213806097 ,  -1.171192513 ],
                     [  -0.034440687 ,  0.160916029 ,  -2.035179690 ],
                     [  -0.037909102 ,  0.307694674 ,  -3.102311444 ],
                     [  -0.122286497 ,  -1.014214485 ,  -1.431659388 ],
                     [  -0.061278153 ,  -0.690156063 ,  -0.097738525 ],
                     [  -0.083866474 ,  -1.480006435 ,  1.065121981 ],
                     [  -0.207551291 ,  -2.830167865 ,  1.008466281 ],
                     [  0.020236002 ,  -3.318294510 ,  1.858492777 ],
                     [  0.100823981 ,  -3.261839820 ,  0.151791829 ],
                     [  -0.015107287 ,  -0.872886238 ,  2.254820437 ],
                     [  0.095534438 ,  0.468473589 ,  2.286592142 ],
                     [  0.148443656 ,  0.902433537 ,  3.277055537 ],
                     [  0.150791629 ,  1.330817541 ,  1.268232413 ],
                     [  0.061278153 ,  0.690156063 ,  0.097738525 ],
                     [  0.213123816 ,  2.178532043 ,  -1.420082564 ],
                     [  4.907019487 ,  1.318912569 ,  0.115169333 ],
                     [  4.945336240 ,  0.544134785 ,  1.248235461 ],
                     [  5.078498892 ,  1.084216460 ,  2.174491246 ],
                     [  4.824685615 ,  -0.802036026 ,  1.213306349 ],
                     [  4.877136241 ,  -1.664227788 ,  2.429380731 ],
                     [  3.921353018 ,  -2.161867438 ,  2.585037720 ],
                     [  5.637978309 ,  -2.435033978 ,  2.315487569 ],
                     [  5.100690710 ,  -1.070628980 ,  3.313538183 ],
                     [  4.630206857 ,  -1.440326451 ,  -0.080379664 ],
                     [  4.469807548 ,  -2.640081851 ,  -0.255033817 ],
                     [  4.641401782 ,  -0.560837886 ,  -1.168484485 ],
                     [  4.465712890 ,  -0.977998743 ,  -2.072617562 ],
                     [  4.726344171 ,  0.814169728 ,  -1.152798148 ],
                     [  4.643675708 ,  1.513854058 ,  -2.149163262 ],
                     [  4.945385581 ,  2.322516737 ,  0.179118562 ]],
    'positions 2.0':[[  0.067390759 ,  1.213806097 ,  -1.171192513 ],
                     [  -0.034440687 ,  0.160916029 ,  -2.035179690 ],
                     [  -0.037909102 ,  0.307694674 ,  -3.102311444 ],
                     [  -0.122286497 ,  -1.014214485 ,  -1.431659388 ],
                     [  -0.061278153 ,  -0.690156063 ,  -0.097738525 ],
                     [  -0.083866474 ,  -1.480006435 ,  1.065121981 ],
                     [  -0.207551291 ,  -2.830167865 ,  1.008466281 ],
                     [  0.020236002 ,  -3.318294510 ,  1.858492777 ],
                     [  0.100823981 ,  -3.261839820 ,  0.151791829 ],
                     [  -0.015107287 ,  -0.872886238 ,  2.254820437 ],
                     [  0.095534438 ,  0.468473589 ,  2.286592142 ],
                     [  0.148443656 ,  0.902433537 ,  3.277055537 ],
                     [  0.150791629 ,  1.330817541 ,  1.268232413 ],
                     [  0.061278153 ,  0.690156063 ,  0.097738525 ],
                     [  0.213123816 ,  2.178532043 ,  -1.420082564 ],
                     [  6.499988023 ,  1.318912569 ,  0.115169333 ],
                     [  6.538304776 ,  0.544134785 ,  1.248235461 ],
                     [  6.671467428 ,  1.084216460 ,  2.174491246 ],
                     [  6.417654151 ,  -0.802036026 ,  1.213306349 ],
                     [  6.470104777 ,  -1.664227788 ,  2.429380731 ],
                     [  5.514321554 ,  -2.161867438 ,  2.585037720 ],
                     [  7.230946845 ,  -2.435033978 ,  2.315487569 ],
                     [  6.693659246 ,  -1.070628980 ,  3.313538183 ],
                     [  6.223175393 ,  -1.440326451 ,  -0.080379664 ],
                     [  6.062776084 ,  -2.640081851 ,  -0.255033817 ],
                     [  6.234370318 ,  -0.560837886 ,  -1.168484485 ],
                     [  6.058681426 ,  -0.977998743 ,  -2.072617562 ],
                     [  6.319312707 ,  0.814169728 ,  -1.152798148 ],
                     [  6.236644244 ,  1.513854058 ,  -2.149163262 ],
                     [  6.538354117 ,  2.322516737 ,  0.179118562 ]]},

'Adenine-thymine_Watson-Crick_complex': {
    'description': "Complex, S22, S26, 2 h-bonds, double h-bond, nucleic base",
    'name': "Adenine-thymine_Watson-Crick_complex",
    's26_number': "07",
    'interaction energy CC':-0.7259,
    'interaction energies s22x5':[-0.6513,-0.7099,-0.5767,-0.3222,-0.1123],
    'offset': 0.0160,
    'symbols': 'NCCCNCNCNNHHHHHNCCCNCCOOHHHHHH',
    'magmoms': None,
    'dimer atoms': [15,15],
    # Optimisation level: MP2/cc-pVTZ 
    'positions':[[  0.9350155, -0.0279801, -0.3788916],
                 [  1.6739638, -0.0357766,  0.7424316],
                 [  3.0747955, -0.0094480,  0.5994562],
                 [  3.5646109,  0.0195446, -0.7059872],
                 [  2.8531510,  0.0258031, -1.8409596],
                 [  1.5490760,  0.0012569, -1.5808009],
                 [  4.0885824, -0.0054429,  1.5289786],
                 [  5.1829921,  0.0253971,  0.7872176],
                 [  4.9294871,  0.0412404, -0.5567274],
                 [  1.0716177, -0.0765366,  1.9391390],
                 [  0.8794435,  0.0050260, -2.4315709],
                 [  6.1882591,  0.0375542,  1.1738824],
                 [  5.6035368,  0.0648755, -1.3036811],
                 [  0.0586915, -0.0423765,  2.0039181],
                 [  1.6443796, -0.0347395,  2.7619159],
                 [ -3.9211729, -0.0009646, -1.5163659],
                 [ -4.6136833,  0.0169051, -0.3336520],
                 [ -3.9917387,  0.0219348,  0.8663338],
                 [ -2.5361367,  0.0074651,  0.8766724],
                 [ -1.9256484, -0.0110593, -0.3638948],
                 [ -2.5395897, -0.0149474, -1.5962357],
                 [ -4.7106131,  0.0413373,  2.1738637],
                 [ -1.8674730,  0.0112093,  1.9120833],
                 [ -1.9416783, -0.0291878, -2.6573783],
                 [ -4.4017172, -0.0036078, -2.4004924],
                 [ -0.8838255, -0.0216168, -0.3784269],
                 [ -5.6909220,  0.0269347, -0.4227183],
                 [ -4.4439282, -0.8302573,  2.7695655],
                 [ -4.4267056,  0.9186178,  2.7530256],
                 [ -5.7883971,  0.0505530,  2.0247280]],
    'positions 0.9':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.738685058 ,  -0.157889771 ,  1.110355410 ],
                     [  -2.139452884 ,  -0.168053559 ,  0.964712563 ],
                     [  -2.629497187 ,  -0.008665792 ,  -0.331201352 ],
                     [  -1.918309833 ,  0.152634753 ,  -1.454844039 ],
                     [  -0.614262216 ,  0.143659867 ,  -1.193547121 ],
                     [  -3.152980999 ,  -0.310697201 ,  1.883518666 ],
                     [  -4.247466012 ,  -0.237200328 ,  1.144874976 ],
                     [  -3.994250734 ,  -0.056604504 ,  -0.187030096 ],
                     [  -0.136179412 ,  -0.289433845 ,  2.300428025 ],
                     [  0.055161346 ,  0.265959015 ,  -2.035655088 ],
                     [  -5.252585445 ,  -0.308958331 ,  1.525406574 ],
                     [  -4.668404863 ,  0.026245320 ,  -0.929656824 ],
                     [  0.876876426 ,  -0.329105732 ,  2.359811410 ],
                     [  -0.708581316 ,  -0.452407073 ,  3.108240602 ],
                     [  4.674076612 ,  0.155627547 ,  -1.128075158 ],
                     [  5.366947235 ,  -0.031573530 ,  0.039652507 ],
                     [  4.745331442 ,  -0.213180550 ,  1.225999310 ],
                     [  3.289690418 ,  -0.205459536 ,  1.237959001 ],
                     [  2.678823212 ,  -0.008913767 ,  0.013109028 ],
                     [  3.292432779 ,  0.176239188 ,  -1.205417098 ],
                     [  5.464603172 ,  -0.419950938 ,  2.517000917 ],
                     [  2.621308338 ,  -0.362031655 ,  2.261654302 ],
                     [  2.694203350 ,  0.342506569 ,  -2.253367774 ],
                     [  5.154382378 ,  0.288458351 ,  -2.002300903 ],
                     [  1.636966971 ,  0.000000000 ,  0.000000000 ],
                     [  6.444191927 ,  -0.024779868 ,  -0.049650000 ],
                     [  5.195022957 ,  0.354841198 ,  3.233018736 ],
                     [  5.183915029 ,  -1.373098243 ,  2.962397530 ],
                     [  6.542374655 ,  -0.403617008 ,  2.368385087 ]],
    'positions 1.0':[[  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -0.738685058000000 ,  -0.157889771000000 ,  1.110355410000000 ],
                     [  -2.139452884000000 ,  -0.168053559000000 ,  0.964712563000000 ],
                     [  -2.629497187000000 ,  -0.008665792000000 ,  -0.331201352000000 ],
                     [  -1.918309833000000 ,  0.152634753000000 ,  -1.454844039000000 ],
                     [  -0.614262216000000 ,  0.143659867000000 ,  -1.193547121000000 ],
                     [  -3.152980999000000 ,  -0.310697201000000 ,  1.883518666000000 ],
                     [  -4.247466012000000 ,  -0.237200328000000 ,  1.144874976000000 ],
                     [  -3.994250734000000 ,  -0.056604504000000 ,  -0.187030096000000 ],
                     [  -0.136179412000000 ,  -0.289433845000000 ,  2.300428025000000 ],
                     [  0.055161346000000 ,  0.265959015000000 ,  -2.035655088000000 ],
                     [  -5.252585445000000 ,  -0.308958331000000 ,  1.525406574000000 ],
                     [  -4.668404863000000 ,  0.026245320000000 ,  -0.929656824000000 ],
                     [  0.876876426000000 ,  -0.329105732000000 ,  2.359811410000000 ],
                     [  -0.708581316000000 ,  -0.452407073000000 ,  3.108240602000000 ],
                     [  4.855961831000000 ,  0.155627547000000 ,  -1.128075158000000 ],
                     [  5.548832453999999 ,  -0.031573530000000 ,  0.039652507000000 ],
                     [  4.927216661000000 ,  -0.213180550000000 ,  1.225999310000000 ],
                     [  3.471575637000000 ,  -0.205459536000000 ,  1.237959001000000 ],
                     [  2.860708431000000 ,  -0.008913767000000 ,  0.013109028000000 ],
                     [  3.474317998000000 ,  0.176239188000000 ,  -1.205417098000000 ],
                     [  5.646488391000000 ,  -0.419950938000000 ,  2.517000917000000 ],
                     [  2.803193557000000 ,  -0.362031655000000 ,  2.261654302000000 ],
                     [  2.876088569000000 ,  0.342506569000000 ,  -2.253367774000000 ],
                     [  5.336267597000000 ,  0.288458351000000 ,  -2.002300903000000 ],
                     [  1.818852190000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  6.626077146000000 ,  -0.024779868000000 ,  -0.049650000000000 ],
                     [  5.376908176000000 ,  0.354841198000000 ,  3.233018736000000 ],
                     [  5.365800247999999 ,  -1.373098243000000 ,  2.962397530000000 ],
                     [  6.724259873999999 ,  -0.403617008000000 ,  2.368385087000000 ]],
    'positions 1.2':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.738685058 ,  -0.157889771 ,  1.110355410 ],
                     [  -2.139452884 ,  -0.168053559 ,  0.964712563 ],
                     [  -2.629497187 ,  -0.008665792 ,  -0.331201352 ],
                     [  -1.918309833 ,  0.152634753 ,  -1.454844039 ],
                     [  -0.614262216 ,  0.143659867 ,  -1.193547121 ],
                     [  -3.152980999 ,  -0.310697201 ,  1.883518666 ],
                     [  -4.247466012 ,  -0.237200328 ,  1.144874976 ],
                     [  -3.994250734 ,  -0.056604504 ,  -0.187030096 ],
                     [  -0.136179412 ,  -0.289433845 ,  2.300428025 ],
                     [  0.055161346 ,  0.265959015 ,  -2.035655088 ],
                     [  -5.252585445 ,  -0.308958331 ,  1.525406574 ],
                     [  -4.668404863 ,  0.026245320 ,  -0.929656824 ],
                     [  0.876876426 ,  -0.329105732 ,  2.359811410 ],
                     [  -0.708581316 ,  -0.452407073 ,  3.108240602 ],
                     [  5.219732269 ,  0.155627547 ,  -1.128075158 ],
                     [  5.912602892 ,  -0.031573530 ,  0.039652507 ],
                     [  5.290987099 ,  -0.213180550 ,  1.225999310 ],
                     [  3.835346075 ,  -0.205459536 ,  1.237959001 ],
                     [  3.224478869 ,  -0.008913767 ,  0.013109028 ],
                     [  3.838088436 ,  0.176239188 ,  -1.205417098 ],
                     [  6.010258829 ,  -0.419950938 ,  2.517000917 ],
                     [  3.166963995 ,  -0.362031655 ,  2.261654302 ],
                     [  3.239859007 ,  0.342506569 ,  -2.253367774 ],
                     [  5.700038035 ,  0.288458351 ,  -2.002300903 ],
                     [  2.182622628 ,  0.000000000 ,  0.000000000 ],
                     [  6.989847584 ,  -0.024779868 ,  -0.049650000 ],
                     [  5.740678614 ,  0.354841198 ,  3.233018736 ],
                     [  5.729570686 ,  -1.373098243 ,  2.962397530 ],
                     [  7.088030312 ,  -0.403617008 ,  2.368385087 ]],
    'positions 1.5':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.738685058 ,  -0.157889771 ,  1.110355410 ],
                     [  -2.139452884 ,  -0.168053559 ,  0.964712563 ],
                     [  -2.629497187 ,  -0.008665792 ,  -0.331201352 ],
                     [  -1.918309833 ,  0.152634753 ,  -1.454844039 ],
                     [  -0.614262216 ,  0.143659867 ,  -1.193547121 ],
                     [  -3.152980999 ,  -0.310697201 ,  1.883518666 ],
                     [  -4.247466012 ,  -0.237200328 ,  1.144874976 ],
                     [  -3.994250734 ,  -0.056604504 ,  -0.187030096 ],
                     [  -0.136179412 ,  -0.289433845 ,  2.300428025 ],
                     [  0.055161346 ,  0.265959015 ,  -2.035655088 ],
                     [  -5.252585445 ,  -0.308958331 ,  1.525406574 ],
                     [  -4.668404863 ,  0.026245320 ,  -0.929656824 ],
                     [  0.876876426 ,  -0.329105732 ,  2.359811410 ],
                     [  -0.708581316 ,  -0.452407073 ,  3.108240602 ],
                     [  5.765387926 ,  0.155627547 ,  -1.128075158 ],
                     [  6.458258549 ,  -0.031573530 ,  0.039652507 ],
                     [  5.836642756 ,  -0.213180550 ,  1.225999310 ],
                     [  4.381001732 ,  -0.205459536 ,  1.237959001 ],
                     [  3.770134526 ,  -0.008913767 ,  0.013109028 ],
                     [  4.383744093 ,  0.176239188 ,  -1.205417098 ],
                     [  6.555914486 ,  -0.419950938 ,  2.517000917 ],
                     [  3.712619652 ,  -0.362031655 ,  2.261654302 ],
                     [  3.785514664 ,  0.342506569 ,  -2.253367774 ],
                     [  6.245693692 ,  0.288458351 ,  -2.002300903 ],
                     [  2.728278285 ,  0.000000000 ,  0.000000000 ],
                     [  7.535503241 ,  -0.024779868 ,  -0.049650000 ],
                     [  6.286334271 ,  0.354841198 ,  3.233018736 ],
                     [  6.275226343 ,  -1.373098243 ,  2.962397530 ],
                     [  7.633685969 ,  -0.403617008 ,  2.368385087 ]],
    'positions 2.0':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.738685058 ,  -0.157889771 ,  1.110355410 ],
                     [  -2.139452884 ,  -0.168053559 ,  0.964712563 ],
                     [  -2.629497187 ,  -0.008665792 ,  -0.331201352 ],
                     [  -1.918309833 ,  0.152634753 ,  -1.454844039 ],
                     [  -0.614262216 ,  0.143659867 ,  -1.193547121 ],
                     [  -3.152980999 ,  -0.310697201 ,  1.883518666 ],
                     [  -4.247466012 ,  -0.237200328 ,  1.144874976 ],
                     [  -3.994250734 ,  -0.056604504 ,  -0.187030096 ],
                     [  -0.136179412 ,  -0.289433845 ,  2.300428025 ],
                     [  0.055161346 ,  0.265959015 ,  -2.035655088 ],
                     [  -5.252585445 ,  -0.308958331 ,  1.525406574 ],
                     [  -4.668404863 ,  0.026245320 ,  -0.929656824 ],
                     [  0.876876426 ,  -0.329105732 ,  2.359811410 ],
                     [  -0.708581316 ,  -0.452407073 ,  3.108240602 ],
                     [  6.674814021 ,  0.155627547 ,  -1.128075158 ],
                     [  7.367684644 ,  -0.031573530 ,  0.039652507 ],
                     [  6.746068851 ,  -0.213180550 ,  1.225999310 ],
                     [  5.290427827 ,  -0.205459536 ,  1.237959001 ],
                     [  4.679560621 ,  -0.008913767 ,  0.013109028 ],
                     [  5.293170188 ,  0.176239188 ,  -1.205417098 ],
                     [  7.465340581 ,  -0.419950938 ,  2.517000917 ],
                     [  4.622045747 ,  -0.362031655 ,  2.261654302 ],
                     [  4.694940759 ,  0.342506569 ,  -2.253367774 ],
                     [  7.155119787 ,  0.288458351 ,  -2.002300903 ],
                     [  3.637704380 ,  0.000000000 ,  0.000000000 ],
                     [  8.444929336 ,  -0.024779868 ,  -0.049650000 ],
                     [  7.195760366 ,  0.354841198 ,  3.233018736 ],
                     [  7.184652438 ,  -1.373098243 ,  2.962397530 ],
                     [  8.543112064 ,  -0.403617008 ,  2.368385087 ]]},

'Ammonia_dimer': {
    'description': "Complex, S22, S26, 2 h-bonds",
    'name': "Ammonia_dimer",
    's26_number': "01",
    'interaction energy CC':-0.1375,
    'interaction energies s22x5':[-0.1045,-0.1362,-0.1023,-0.0481,-0.0156],
    'offset': 0.0013,
    'symbols': 'NHHHNHHH',
    'magmoms': None,
    'dimer atoms': [4,4],
    # Optimisation level: CCSD(T)/cc-pVQZ 
    'positions':[[ -1.578718, -0.046611,  0.000000],
                 [ -2.158621,  0.136396, -0.809565],
                 [ -2.158621,  0.136396,  0.809565],
                 [ -0.849471,  0.658193,  0.000000],
                 [  1.578718,  0.046611,  0.000000],
                 [  2.158621, -0.136396, -0.809565],
                 [  0.849471, -0.658193,  0.000000],
                 [  2.158621, -0.136396,  0.809565]],
    'positions 0.9':[[  -0.535020551 ,  -0.861570006 ,  0.000000000 ],
                     [  -1.142058700 ,  -0.825740733 ,  -0.809565000 ],
                     [  -1.142058700 ,  -0.825740733 ,  0.809565000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  2.253621272 ,  0.000000000 ,  0.000000000 ],
                     [  2.860659421 ,  -0.035829274 ,  -0.809565000 ],
                     [  1.718600721 ,  -0.861570006 ,  0.000000000 ],
                     [  2.860659421 ,  -0.035829274 ,  0.809565000 ]],
    'positions 1.0':[[  -0.535020551000000 ,  -0.861570006000000 ,  0.000000000000000 ],
                     [  -1.142058700000000 ,  -0.825740733000000 ,  -0.809565000000000 ],
                     [  -1.142058700000000 ,  -0.825740733000000 ,  0.809565000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  2.504023635454546 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  3.111061784454545 ,  -0.035829274000000 ,  -0.809565000000000 ],
                     [  1.969003084454545 ,  -0.861570006000000 ,  0.000000000000000 ],
                     [  3.111061784454545 ,  -0.035829274000000 ,  0.809565000000000 ]],
    'positions 1.2':[[  -0.535020551 ,  -0.861570006 ,  0.000000000 ],
                     [  -1.142058700 ,  -0.825740733 ,  -0.809565000 ],
                     [  -1.142058700 ,  -0.825740733 ,  0.809565000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  3.004828362 ,  0.000000000 ,  0.000000000 ],
                     [  3.611866511 ,  -0.035829274 ,  -0.809565000 ],
                     [  2.469807811 ,  -0.861570006 ,  0.000000000 ],
                     [  3.611866511 ,  -0.035829274 ,  0.809565000 ]],
    'positions 1.5':[[  -0.535020551 ,  -0.861570006 ,  0.000000000 ],
                     [  -1.142058700 ,  -0.825740733 ,  -0.809565000 ],
                     [  -1.142058700 ,  -0.825740733 ,  0.809565000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  3.756035452 ,  0.000000000 ,  0.000000000 ],
                     [  4.363073601 ,  -0.035829274 ,  -0.809565000 ],
                     [  3.221014901 ,  -0.861570006 ,  0.000000000 ],
                     [  4.363073601 ,  -0.035829274 ,  0.809565000 ]],
    'positions 2.0':[[  -0.535020551 ,  -0.861570006 ,  0.000000000 ],
                     [  -1.142058700 ,  -0.825740733 ,  -0.809565000 ],
                     [  -1.142058700 ,  -0.825740733 ,  0.809565000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  5.008047270 ,  0.000000000 ,  0.000000000 ],
                     [  5.615085419 ,  -0.035829274 ,  -0.809565000 ],
                     [  4.473026719 ,  -0.861570006 ,  0.000000000 ],
                     [  5.615085419 ,  -0.035829274 ,  0.809565000 ]]},

'Benzene-methane_complex': {
    'description': "Complex, S22, S26, stack, dispersion bonded",
    'name': "Benzene-methane_complex",
    's26_number': "10",
    'interaction energy CC':-0.0629,
    'interaction energies s22x5':[-0.0473,-0.0650,-0.0490,-0.0208,-0.0052],
    'offset': -0.0021,
    'symbols': 'CCCCCCHHHHHHCHHHH',
    'magmoms': None,
    'dimer atoms': [12,5],
    # Optimisation level:  MP2/cc-pVTZ
    'positions':[[  1.3932178,  0.0362913, -0.6332803],
                 [  0.7280364, -1.1884015, -0.6333017],
                 [ -0.6651797, -1.2247077, -0.6332803],
                 [ -1.3932041, -0.0362972, -0.6333017],
                 [ -0.7280381,  1.1884163, -0.6332803],
                 [  0.6651677,  1.2246987, -0.6333017],
                 [  2.4742737,  0.0644484, -0.6317240],
                 [  1.2929588, -2.1105409, -0.6317401],
                 [ -1.1813229, -2.1750081, -0.6317240],
                 [ -2.4742614, -0.0644647, -0.6317401],
                 [ -1.2929508,  2.1105596, -0.6317240],
                 [  1.1813026,  2.1750056, -0.6317401],
                 [  0.0000000,  0.0000000,  3.0826195],
                 [  0.5868776,  0.8381742,  3.4463772],
                 [ -1.0193189,  0.0891638,  3.4463772],
                 [  0.0000000,  0.0000000,  1.9966697],
                 [  0.4324413, -0.9273380,  3.446377]],
    'positions 0.9':[[  0.000011002 ,  0.036291078 ,  -1.393218002 ],
                     [  -0.000011075 ,  -1.188401879 ,  -0.728035925 ],
                     [  0.000010922 ,  -1.224707791 ,  0.665180078 ],
                     [  -0.000011002 ,  -0.036296745 ,  1.393204002 ],
                     [  0.000011075 ,  1.188416213 ,  0.728037925 ],
                     [  -0.000010922 ,  1.224699125 ,  -0.665168078 ],
                     [  0.001567004 ,  0.064448010 ,  -2.474274004 ],
                     [  0.001550866 ,  -2.110540915 ,  -1.292958866 ],
                     [  0.001566862 ,  -2.175007759 ,  1.181323138 ],
                     [  0.001550996 ,  -0.064464677 ,  2.474261004 ],
                     [  0.001567134 ,  2.110560249 ,  1.292950866 ],
                     [  0.001551138 ,  2.175006092 ,  -1.181303138 ],
                     [  3.452913900 ,  -0.000000069 ,  0.000000000 ],
                     [  3.816671953 ,  0.838173871 ,  -0.586878053 ],
                     [  3.816671906 ,  0.089163973 ,  1.019318994 ],
                     [  2.366964900 ,  0.000000000 ,  0.000000000 ],
                     [  3.816671841 ,  -0.927338119 ,  -0.432440941 ]],
    'positions 1.0':[[  0.000011002000000 ,  0.036291078000000 ,  -1.393218002000000 ],
                     [  -0.000011075000000 ,  -1.188401879000000 ,  -0.728035925000000 ],
                     [  0.000010922000000 ,  -1.224707791000000 ,  0.665180078000000 ],
                     [  -0.000011002000000 ,  -0.036296745000000 ,  1.393204002000000 ],
                     [  0.000011075000000 ,  1.188416213000000 ,  0.728037925000000 ],
                     [  -0.000010922000000 ,  1.224699125000000 ,  -0.665168078000000 ],
                     [  0.001567004000000 ,  0.064448010000000 ,  -2.474274004000000 ],
                     [  0.001550866000000 ,  -2.110540915000000 ,  -1.292958866000000 ],
                     [  0.001566862000000 ,  -2.175007759000000 ,  1.181323138000000 ],
                     [  0.001550996000000 ,  -0.064464677000000 ,  2.474261004000000 ],
                     [  0.001567134000000 ,  2.110560249000000 ,  1.292950866000000 ],
                     [  0.001551138000000 ,  2.175006092000000 ,  -1.181303138000000 ],
                     [  3.715910000000000 ,  -0.000000069000000 ,  0.000000000000000 ],
                     [  4.079668053000000 ,  0.838173871000000 ,  -0.586878053000000 ],
                     [  4.079668005999999 ,  0.089163973000000 ,  1.019318994000000 ],
                     [  2.629961000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  4.079667940999999 ,  -0.927338119000000 ,  -0.432440941000000 ]],
    'positions 1.2':[[  0.000011002 ,  0.036291078 ,  -1.393218002 ],
                     [  -0.000011075 ,  -1.188401879 ,  -0.728035925 ],
                     [  0.000010922 ,  -1.224707791 ,  0.665180078 ],
                     [  -0.000011002 ,  -0.036296745 ,  1.393204002 ],
                     [  0.000011075 ,  1.188416213 ,  0.728037925 ],
                     [  -0.000010922 ,  1.224699125 ,  -0.665168078 ],
                     [  0.001567004 ,  0.064448010 ,  -2.474274004 ],
                     [  0.001550866 ,  -2.110540915 ,  -1.292958866 ],
                     [  0.001566862 ,  -2.175007759 ,  1.181323138 ],
                     [  0.001550996 ,  -0.064464677 ,  2.474261004 ],
                     [  0.001567134 ,  2.110560249 ,  1.292950866 ],
                     [  0.001551138 ,  2.175006092 ,  -1.181303138 ],
                     [  4.241902200 ,  -0.000000069 ,  0.000000000 ],
                     [  4.605660253 ,  0.838173871 ,  -0.586878053 ],
                     [  4.605660206 ,  0.089163973 ,  1.019318994 ],
                     [  3.155953200 ,  0.000000000 ,  0.000000000 ],
                     [  4.605660141 ,  -0.927338119 ,  -0.432440941 ]],
    'positions 1.5':[[  0.000011002 ,  0.036291078 ,  -1.393218002 ],
                     [  -0.000011075 ,  -1.188401879 ,  -0.728035925 ],
                     [  0.000010922 ,  -1.224707791 ,  0.665180078 ],
                     [  -0.000011002 ,  -0.036296745 ,  1.393204002 ],
                     [  0.000011075 ,  1.188416213 ,  0.728037925 ],
                     [  -0.000010922 ,  1.224699125 ,  -0.665168078 ],
                     [  0.001567004 ,  0.064448010 ,  -2.474274004 ],
                     [  0.001550866 ,  -2.110540915 ,  -1.292958866 ],
                     [  0.001566862 ,  -2.175007759 ,  1.181323138 ],
                     [  0.001550996 ,  -0.064464677 ,  2.474261004 ],
                     [  0.001567134 ,  2.110560249 ,  1.292950866 ],
                     [  0.001551138 ,  2.175006092 ,  -1.181303138 ],
                     [  5.030890500 ,  -0.000000069 ,  0.000000000 ],
                     [  5.394648553 ,  0.838173871 ,  -0.586878053 ],
                     [  5.394648506 ,  0.089163973 ,  1.019318994 ],
                     [  3.944941500 ,  0.000000000 ,  0.000000000 ],
                     [  5.394648441 ,  -0.927338119 ,  -0.432440941 ]],
    'positions 2.0':[[  0.000011002 ,  0.036291078 ,  -1.393218002 ],
                     [  -0.000011075 ,  -1.188401879 ,  -0.728035925 ],
                     [  0.000010922 ,  -1.224707791 ,  0.665180078 ],
                     [  -0.000011002 ,  -0.036296745 ,  1.393204002 ],
                     [  0.000011075 ,  1.188416213 ,  0.728037925 ],
                     [  -0.000010922 ,  1.224699125 ,  -0.665168078 ],
                     [  0.001567004 ,  0.064448010 ,  -2.474274004 ],
                     [  0.001550866 ,  -2.110540915 ,  -1.292958866 ],
                     [  0.001566862 ,  -2.175007759 ,  1.181323138 ],
                     [  0.001550996 ,  -0.064464677 ,  2.474261004 ],
                     [  0.001567134 ,  2.110560249 ,  1.292950866 ],
                     [  0.001551138 ,  2.175006092 ,  -1.181303138 ],
                     [  6.345871000 ,  -0.000000069 ,  0.000000000 ],
                     [  6.709629053 ,  0.838173871 ,  -0.586878053 ],
                     [  6.709629006 ,  0.089163973 ,  1.019318994 ],
                     [  5.259922000 ,  0.000000000 ,  0.000000000 ],
                     [  6.709628941 ,  -0.927338119 ,  -0.432440941 ]]},

'Benzene-ammonia_complex': {
    'description': "Complex, S22, S26",
    'name': "Benzene-ammonia_complex",
    's26_number': "18",
    'interaction energy CC':-0.1006,
    'interaction energies s22x5':[-0.0885,-0.1019,-0.0759,-0.0369,-0.0121],
    'offset': -0.0013,
    'symbols': 'CCCCCCHHHHHHNHHH',
    'magmoms': None,
    'dimer atoms': [12,4],
    # Optimisation level:   MP2/cc-pVTZ
    'positions':[[ -0.7392810,  0.5158785, -1.2071079],
                 [ -1.4261442,  0.3965455,  0.0000000],
                 [ -0.7392810,  0.5158785,  1.2071079],
                 [  0.6342269,  0.7546398,  1.2070735],
                 [  1.3210434,  0.8737566,  0.0000000],
                 [  0.6342269,  0.7546398, -1.2070735],
                 [ -1.2719495,  0.4206316, -2.1432894],
                 [ -2.4902205,  0.2052381,  0.0000000],
                 [ -1.2719495,  0.4206316,  2.1432894],
                 [  1.1668005,  0.8474885,  2.1436950],
                 [  2.3863585,  1.0596312,  0.0000000],
                 [  1.1668005,  0.8474885, -2.1436950],
                 [  0.1803930, -2.9491231,  0.0000000],
                 [  0.7595495, -3.1459477, -0.8060729],
                 [  0.7595495, -3.1459477,  0.8060729],
                 [  0.0444167, -1.9449399,  0.0000000]],
    'positions 0.9':[[  0.000000000 ,  0.000000000 ,  -1.207108000 ],
                     [  -0.094723910 ,  -0.690687169 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207108000 ],
                     [  0.189293052 ,  1.381194838 ,  1.207073000 ],
                     [  0.284209467 ,  2.071771374 ,  0.000000000 ],
                     [  0.189293052 ,  1.381194838 ,  -1.207073000 ],
                     [  -0.070884435 ,  -0.536454706 ,  -2.143289000 ],
                     [  -0.235335157 ,  -1.762640796 ,  0.000000000 ],
                     [  -0.070884435 ,  -0.536454706 ,  2.143289000 ],
                     [  0.262434233 ,  1.916830087 ,  2.143695000 ],
                     [  0.430373810 ,  3.143257869 ,  0.000000000 ],
                     [  0.262434233 ,  1.916830087 ,  -2.143695000 ],
                     [  3.322432676 ,  -0.175158455 ,  0.000000000 ],
                     [  3.685723470 ,  0.316960994 ,  -0.806073000 ],
                     [  3.685723470 ,  0.316960994 ,  0.806073000 ],
                     [  2.324338249 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.0':[[  0.000000000000000 ,  0.000000000000000 ,  -1.207108000000000 ],
                     [  -0.094723910000000 ,  -0.690687169000000 ,  0.000000000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  1.207108000000000 ],
                     [  0.189293052000000 ,  1.381194838000000 ,  1.207073000000000 ],
                     [  0.284209467000000 ,  2.071771374000000 ,  0.000000000000000 ],
                     [  0.189293052000000 ,  1.381194838000000 ,  -1.207073000000000 ],
                     [  -0.070884435000000 ,  -0.536454706000000 ,  -2.143289000000000 ],
                     [  -0.235335157000000 ,  -1.762640796000000 ,  0.000000000000000 ],
                     [  -0.070884435000000 ,  -0.536454706000000 ,  2.143289000000000 ],
                     [  0.262434233000000 ,  1.916830087000000 ,  2.143695000000000 ],
                     [  0.430373810000000 ,  3.143257869000000 ,  0.000000000000000 ],
                     [  0.262434233000000 ,  1.916830087000000 ,  -2.143695000000000 ],
                     [  3.580692481363636 ,  -0.175158455000000 ,  0.000000000000000 ],
                     [  3.943983275363637 ,  0.316960994000000 ,  -0.806073000000000 ],
                     [  3.943983275363637 ,  0.316960994000000 ,  0.806073000000000 ],
                     [  2.582598054363637 ,  0.000000000000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  0.000000000 ,  0.000000000 ,  -1.207108000 ],
                     [  -0.094723910 ,  -0.690687169 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207108000 ],
                     [  0.189293052 ,  1.381194838 ,  1.207073000 ],
                     [  0.284209467 ,  2.071771374 ,  0.000000000 ],
                     [  0.189293052 ,  1.381194838 ,  -1.207073000 ],
                     [  -0.070884435 ,  -0.536454706 ,  -2.143289000 ],
                     [  -0.235335157 ,  -1.762640796 ,  0.000000000 ],
                     [  -0.070884435 ,  -0.536454706 ,  2.143289000 ],
                     [  0.262434233 ,  1.916830087 ,  2.143695000 ],
                     [  0.430373810 ,  3.143257869 ,  0.000000000 ],
                     [  0.262434233 ,  1.916830087 ,  -2.143695000 ],
                     [  4.097212092 ,  -0.175158455 ,  0.000000000 ],
                     [  4.460502886 ,  0.316960994 ,  -0.806073000 ],
                     [  4.460502886 ,  0.316960994 ,  0.806073000 ],
                     [  3.099117665 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.5':[[  0.000000000 ,  0.000000000 ,  -1.207108000 ],
                     [  -0.094723910 ,  -0.690687169 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207108000 ],
                     [  0.189293052 ,  1.381194838 ,  1.207073000 ],
                     [  0.284209467 ,  2.071771374 ,  0.000000000 ],
                     [  0.189293052 ,  1.381194838 ,  -1.207073000 ],
                     [  -0.070884435 ,  -0.536454706 ,  -2.143289000 ],
                     [  -0.235335157 ,  -1.762640796 ,  0.000000000 ],
                     [  -0.070884435 ,  -0.536454706 ,  2.143289000 ],
                     [  0.262434233 ,  1.916830087 ,  2.143695000 ],
                     [  0.430373810 ,  3.143257869 ,  0.000000000 ],
                     [  0.262434233 ,  1.916830087 ,  -2.143695000 ],
                     [  4.871991508 ,  -0.175158455 ,  0.000000000 ],
                     [  5.235282302 ,  0.316960994 ,  -0.806073000 ],
                     [  5.235282302 ,  0.316960994 ,  0.806073000 ],
                     [  3.873897081 ,  0.000000000 ,  0.000000000 ]],
    'positions 2.0':[[  0.000000000 ,  0.000000000 ,  -1.207108000 ],
                     [  -0.094723910 ,  -0.690687169 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207108000 ],
                     [  0.189293052 ,  1.381194838 ,  1.207073000 ],
                     [  0.284209467 ,  2.071771374 ,  0.000000000 ],
                     [  0.189293052 ,  1.381194838 ,  -1.207073000 ],
                     [  -0.070884435 ,  -0.536454706 ,  -2.143289000 ],
                     [  -0.235335157 ,  -1.762640796 ,  0.000000000 ],
                     [  -0.070884435 ,  -0.536454706 ,  2.143289000 ],
                     [  0.262434233 ,  1.916830087 ,  2.143695000 ],
                     [  0.430373810 ,  3.143257869 ,  0.000000000 ],
                     [  0.262434233 ,  1.916830087 ,  -2.143695000 ],
                     [  6.163290535 ,  -0.175158455 ,  0.000000000 ],
                     [  6.526581329 ,  0.316960994 ,  -0.806073000 ],
                     [  6.526581329 ,  0.316960994 ,  0.806073000 ],
                     [  5.165196108 ,  0.000000000 ,  0.000000000 ]]},

'Benzene_dimer_parallel_displaced': {
    'description': "Complex, S22, S26, stack, dispersion bonded",
    'name': "Benzene_dimer_parallel_displaced",
    's26_number': "11",
    'interaction energy CC':-0.1136,
    'interaction energies s22x5':[-0.0065,-0.1219,-0.0833,-0.0230,-0.0030],
    'offset': -0.0083,
    'symbols': 'CCCCCCHHHHHHCCCCCCHHHHHH',
    'magmoms': None,
    'dimer atoms': [12,12],
    # Optimisation level:   MP2/cc-pVTZ
    'positions':[[ -1.0478252, -1.4216736,  0.0000000],
                 [ -1.4545034, -0.8554459,  1.2062048],
                 [ -1.4545034, -0.8554459, -1.2062048],
                 [ -2.2667970,  0.2771610,  1.2069539],
                 [ -2.6714781,  0.8450211,  0.0000000],
                 [ -2.2667970,  0.2771610, -1.2069539],
                 [ -1.1338534, -1.2920593, -2.1423150],
                 [ -2.5824943,  0.7163066, -2.1437977],
                 [ -3.3030422,  1.7232700,  0.0000000],
                 [ -2.5824943,  0.7163066,  2.1437977],
                 [ -1.1338534, -1.2920593,  2.1423150],
                 [ -0.4060253, -2.2919049,  0.0000000],
                 [  1.0478252,  1.4216736,  0.0000000],
                 [  1.4545034,  0.8554459, -1.2062048],
                 [  1.4545034,  0.8554459,  1.2062048],
                 [  2.2667970, -0.2771610, -1.2069539],
                 [  2.6714781, -0.8450211,  0.0000000],
                 [  2.2667970, -0.2771610,  1.2069539],
                 [  0.4060253,  2.2919049,  0.0000000],
                 [  1.1338534,  1.2920593,  2.1423150],
                 [  2.5824943, -0.7163066,  2.1437977],
                 [  3.3030422, -1.7232700,  0.0000000],
                 [  2.5824943, -0.7163066, -2.1437977],
                 [  1.1338534,  1.2920593, -2.1423150]],
    'positions 0.9':[[  0.629051507 ,  -1.244058476 ,  0.000000000 ],
                     [  0.314072291 ,  -0.622134657 ,  1.206205000 ],
                     [  0.314072291 ,  -0.622134657 ,  -1.206205000 ],
                     [  -0.314813547 ,  0.621699240 ,  1.206954000 ],
                     [  -0.627568995 ,  1.244929310 ,  0.000000000 ],
                     [  -0.314813547 ,  0.621699240 ,  -1.206954000 ],
                     [  0.563930576 ,  -1.102778154 ,  -2.142315000 ],
                     [  -0.559388819 ,  1.104085746 ,  -2.143798000 ],
                     [  -1.116894124 ,  2.209685917 ,  0.000000000 ],
                     [  -0.559388819 ,  1.104085746 ,  2.143798000 ],
                     [  0.563930576 ,  -1.102778154 ,  2.142315000 ],
                     [  1.129721711 ,  -2.202462660 ,  0.000000000 ],
                     [  2.759649224 ,  1.244058476 ,  0.000000000 ],
                     [  3.074628440 ,  0.622134657 ,  -1.206205000 ],
                     [  3.074628440 ,  0.622134657 ,  1.206205000 ],
                     [  3.703514278 ,  -0.621699240 ,  -1.206954000 ],
                     [  4.016269727 ,  -1.244929310 ,  0.000000000 ],
                     [  3.703514278 ,  -0.621699240 ,  1.206954000 ],
                     [  2.258979020 ,  2.202462660 ,  0.000000000 ],
                     [  2.824770156 ,  1.102778154 ,  2.142315000 ],
                     [  3.948089550 ,  -1.104085746 ,  2.143798000 ],
                     [  4.505594855 ,  -2.209685917 ,  0.000000000 ],
                     [  3.948089550 ,  -1.104085746 ,  -2.143798000 ],
                     [  2.824770156 ,  1.102778154 ,  -2.142315000 ]],
    'positions 1.0':[[  0.629051507000000 ,  -1.244058476000000 ,  0.000000000000000 ],
                     [  0.314072291000000 ,  -0.622134657000000 ,  1.206205000000000 ],
                     [  0.314072291000000 ,  -0.622134657000000 ,  -1.206205000000000 ],
                     [  -0.314813547000000 ,  0.621699240000000 ,  1.206954000000000 ],
                     [  -0.627568995000000 ,  1.244929310000000 ,  0.000000000000000 ],
                     [  -0.314813547000000 ,  0.621699240000000 ,  -1.206954000000000 ],
                     [  0.563930576000000 ,  -1.102778154000000 ,  -2.142315000000000 ],
                     [  -0.559388819000000 ,  1.104085746000000 ,  -2.143798000000000 ],
                     [  -1.116894124000000 ,  2.209685917000000 ,  0.000000000000000 ],
                     [  -0.559388819000000 ,  1.104085746000000 ,  2.143798000000000 ],
                     [  0.563930576000000 ,  -1.102778154000000 ,  2.142315000000000 ],
                     [  1.129721711000000 ,  -2.202462660000000 ,  0.000000000000000 ],
                     [  3.136171527545454 ,  1.244058476000000 ,  0.000000000000000 ],
                     [  3.451150743545455 ,  0.622134657000000 ,  -1.206205000000000 ],
                     [  3.451150743545455 ,  0.622134657000000 ,  1.206205000000000 ],
                     [  4.080036581545454 ,  -0.621699240000000 ,  -1.206954000000000 ],
                     [  4.392792030545455 ,  -1.244929310000000 ,  0.000000000000000 ],
                     [  4.080036581545454 ,  -0.621699240000000 ,  1.206954000000000 ],
                     [  2.635501323545455 ,  2.202462660000000 ,  0.000000000000000 ],
                     [  3.201292459545455 ,  1.102778154000000 ,  2.142315000000000 ],
                     [  4.324611853545455 ,  -1.104085746000000 ,  2.143798000000000 ],
                     [  4.882117158545454 ,  -2.209685917000000 ,  0.000000000000000 ],
                     [  4.324611853545455 ,  -1.104085746000000 ,  -2.143798000000000 ],
                     [  3.201292459545455 ,  1.102778154000000 ,  -2.142315000000000 ]],
    'positions 1.2':[[  0.629051507 ,  -1.244058476 ,  0.000000000 ],
                     [  0.314072291 ,  -0.622134657 ,  1.206205000 ],
                     [  0.314072291 ,  -0.622134657 ,  -1.206205000 ],
                     [  -0.314813547 ,  0.621699240 ,  1.206954000 ],
                     [  -0.627568995 ,  1.244929310 ,  0.000000000 ],
                     [  -0.314813547 ,  0.621699240 ,  -1.206954000 ],
                     [  0.563930576 ,  -1.102778154 ,  -2.142315000 ],
                     [  -0.559388819 ,  1.104085746 ,  -2.143798000 ],
                     [  -1.116894124 ,  2.209685917 ,  0.000000000 ],
                     [  -0.559388819 ,  1.104085746 ,  2.143798000 ],
                     [  0.563930576 ,  -1.102778154 ,  2.142315000 ],
                     [  1.129721711 ,  -2.202462660 ,  0.000000000 ],
                     [  3.889216135 ,  1.244058476 ,  0.000000000 ],
                     [  4.204195351 ,  0.622134657 ,  -1.206205000 ],
                     [  4.204195351 ,  0.622134657 ,  1.206205000 ],
                     [  4.833081189 ,  -0.621699240 ,  -1.206954000 ],
                     [  5.145836638 ,  -1.244929310 ,  0.000000000 ],
                     [  4.833081189 ,  -0.621699240 ,  1.206954000 ],
                     [  3.388545931 ,  2.202462660 ,  0.000000000 ],
                     [  3.954337067 ,  1.102778154 ,  2.142315000 ],
                     [  5.077656461 ,  -1.104085746 ,  2.143798000 ],
                     [  5.635161766 ,  -2.209685917 ,  0.000000000 ],
                     [  5.077656461 ,  -1.104085746 ,  -2.143798000 ],
                     [  3.954337067 ,  1.102778154 ,  -2.142315000 ]],
    'positions 1.5':[[  0.629051507 ,  -1.244058476 ,  0.000000000 ],
                     [  0.314072291 ,  -0.622134657 ,  1.206205000 ],
                     [  0.314072291 ,  -0.622134657 ,  -1.206205000 ],
                     [  -0.314813547 ,  0.621699240 ,  1.206954000 ],
                     [  -0.627568995 ,  1.244929310 ,  0.000000000 ],
                     [  -0.314813547 ,  0.621699240 ,  -1.206954000 ],
                     [  0.563930576 ,  -1.102778154 ,  -2.142315000 ],
                     [  -0.559388819 ,  1.104085746 ,  -2.143798000 ],
                     [  -1.116894124 ,  2.209685917 ,  0.000000000 ],
                     [  -0.559388819 ,  1.104085746 ,  2.143798000 ],
                     [  0.563930576 ,  -1.102778154 ,  2.142315000 ],
                     [  1.129721711 ,  -2.202462660 ,  0.000000000 ],
                     [  5.018783046 ,  1.244058476 ,  0.000000000 ],
                     [  5.333762262 ,  0.622134657 ,  -1.206205000 ],
                     [  5.333762262 ,  0.622134657 ,  1.206205000 ],
                     [  5.962648100 ,  -0.621699240 ,  -1.206954000 ],
                     [  6.275403549 ,  -1.244929310 ,  0.000000000 ],
                     [  5.962648100 ,  -0.621699240 ,  1.206954000 ],
                     [  4.518112842 ,  2.202462660 ,  0.000000000 ],
                     [  5.083903978 ,  1.102778154 ,  2.142315000 ],
                     [  6.207223372 ,  -1.104085746 ,  2.143798000 ],
                     [  6.764728677 ,  -2.209685917 ,  0.000000000 ],
                     [  6.207223372 ,  -1.104085746 ,  -2.143798000 ],
                     [  5.083903978 ,  1.102778154 ,  -2.142315000 ]],
    'positions 2.0':[[  0.629051507 ,  -1.244058476 ,  0.000000000 ],
                     [  0.314072291 ,  -0.622134657 ,  1.206205000 ],
                     [  0.314072291 ,  -0.622134657 ,  -1.206205000 ],
                     [  -0.314813547 ,  0.621699240 ,  1.206954000 ],
                     [  -0.627568995 ,  1.244929310 ,  0.000000000 ],
                     [  -0.314813547 ,  0.621699240 ,  -1.206954000 ],
                     [  0.563930576 ,  -1.102778154 ,  -2.142315000 ],
                     [  -0.559388819 ,  1.104085746 ,  -2.143798000 ],
                     [  -1.116894124 ,  2.209685917 ,  0.000000000 ],
                     [  -0.559388819 ,  1.104085746 ,  2.143798000 ],
                     [  0.563930576 ,  -1.102778154 ,  2.142315000 ],
                     [  1.129721711 ,  -2.202462660 ,  0.000000000 ],
                     [  6.901394563 ,  1.244058476 ,  0.000000000 ],
                     [  7.216373779 ,  0.622134657 ,  -1.206205000 ],
                     [  7.216373779 ,  0.622134657 ,  1.206205000 ],
                     [  7.845259617 ,  -0.621699240 ,  -1.206954000 ],
                     [  8.158015066 ,  -1.244929310 ,  0.000000000 ],
                     [  7.845259617 ,  -0.621699240 ,  1.206954000 ],
                     [  6.400724359 ,  2.202462660 ,  0.000000000 ],
                     [  6.966515495 ,  1.102778154 ,  2.142315000 ],
                     [  8.089834889 ,  -1.104085746 ,  2.143798000 ],
                     [  8.647340194 ,  -2.209685917 ,  0.000000000 ],
                     [  8.089834889 ,  -1.104085746 ,  -2.143798000 ],
                     [  6.966515495 ,  1.102778154 ,  -2.142315000 ]]},

'Benzene_dimer_T-shaped': {
    'description': "Complex, S22, S26",
    'name': "Benzene_dimer_T-shaped",
    's26_number': "20",
    'interaction energy CC':-0.1175,
    'interaction energies s22x5':[-0.0954,-0.1214,-0.0976,-0.0486,-0.0152],
    'offset': -0.0039,
    'symbols': 'CCCCCCHHHHHHCCCCCCHHHHHH',
    'magmoms': None,
    'dimer atoms': [12,12],
    # Optimisation level:   MP2/cc-pVTZ
    'positions':[[  0.0000000,  0.0000000,  1.0590353],
                 [  0.0000000, -1.2060084,  1.7576742],
                 [  0.0000000, -1.2071767,  3.1515905],
                 [  0.0000000,  0.0000000,  3.8485751],
                 [  0.0000000,  1.2071767,  3.1515905],
                 [  0.0000000,  1.2060084,  1.7576742],
                 [  0.0000000,  0.0000000, -0.0215805],
                 [  0.0000000, -2.1416387,  1.2144217],
                 [  0.0000000, -2.1435657,  3.6929953],
                 [  0.0000000,  0.0000000,  4.9301499],
                 [  0.0000000,  2.1435657,  3.6929953],
                 [  0.0000000,  2.1416387,  1.2144217],
                 [ -1.3940633,  0.0000000, -2.4541524],
                 [ -0.6970468,  1.2072378, -2.4546277],
                 [  0.6970468,  1.2072378, -2.4546277],
                 [  1.3940633,  0.0000000, -2.4541524],
                 [  0.6970468, -1.2072378, -2.4546277],
                 [ -0.6970468, -1.2072378, -2.4546277],
                 [ -2.4753995,  0.0000000, -2.4503221],
                 [ -1.2382321,  2.1435655, -2.4536764],
                 [  1.2382321,  2.1435655, -2.4536764],
                 [  2.4753995,  0.0000000, -2.4503221],
                 [  1.2382321, -2.1435655, -2.4536764],
                 [ -1.2382321, -2.1435655, -2.4536764]],
    'positions 0.9':[[  -1.080615000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.779254000 ,  -1.206008000 ,  0.000000000 ],
                     [  -3.173171000 ,  -1.207177000 ,  0.000000000 ],
                     [  -3.870155000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.173171000 ,  1.207177000 ,  0.000000000 ],
                     [  -1.779254000 ,  1.206008000 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.236002000 ,  -2.141639000 ,  0.000000000 ],
                     [  -3.714575000 ,  -2.143566000 ,  0.000000000 ],
                     [  -4.951730000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.714575000 ,  2.143566000 ,  0.000000000 ],
                     [  -1.236002000 ,  2.141639000 ,  0.000000000 ],
                     [  2.189283067 ,  0.000000000 ,  -1.394063000 ],
                     [  2.189759067 ,  1.207238000 ,  -0.697047000 ],
                     [  2.189759067 ,  1.207238000 ,  0.697047000 ],
                     [  2.189283067 ,  0.000000000 ,  1.394063000 ],
                     [  2.189759067 ,  -1.207238000 ,  0.697047000 ],
                     [  2.189759067 ,  -1.207238000 ,  -0.697047000 ],
                     [  2.185453067 ,  0.000000000 ,  -2.475399000 ],
                     [  2.188807067 ,  2.143565000 ,  -1.238232000 ],
                     [  2.188807067 ,  2.143565000 ,  1.238232000 ],
                     [  2.185453067 ,  0.000000000 ,  2.475399000 ],
                     [  2.188807067 ,  -2.143565000 ,  1.238232000 ],
                     [  2.188807067 ,  -2.143565000 ,  -1.238232000 ]],
    'positions 1.0':[[  -1.080615000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -1.779254000000000 ,  -1.206008000000000 ,  0.000000000000000 ],
                     [  -3.173171000000000 ,  -1.207177000000000 ,  0.000000000000000 ],
                     [  -3.870155000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -3.173171000000000 ,  1.207177000000000 ,  0.000000000000000 ],
                     [  -1.779254000000000 ,  1.206008000000000 ,  0.000000000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -1.236002000000000 ,  -2.141639000000000 ,  0.000000000000000 ],
                     [  -3.714575000000000 ,  -2.143566000000000 ,  0.000000000000000 ],
                     [  -4.951730000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -3.714575000000000 ,  2.143566000000000 ,  0.000000000000000 ],
                     [  -1.236002000000000 ,  2.141639000000000 ,  0.000000000000000 ],
                     [  2.432572000272727 ,  0.000000000000000 ,  -1.394063000000000 ],
                     [  2.433048000272727 ,  1.207238000000000 ,  -0.697047000000000 ],
                     [  2.433048000272727 ,  1.207238000000000 ,  0.697047000000000 ],
                     [  2.432572000272727 ,  0.000000000000000 ,  1.394063000000000 ],
                     [  2.433048000272727 ,  -1.207238000000000 ,  0.697047000000000 ],
                     [  2.433048000272727 ,  -1.207238000000000 ,  -0.697047000000000 ],
                     [  2.428742000272727 ,  0.000000000000000 ,  -2.475399000000000 ],
                     [  2.432096000272727 ,  2.143565000000000 ,  -1.238232000000000 ],
                     [  2.432096000272727 ,  2.143565000000000 ,  1.238232000000000 ],
                     [  2.428742000272727 ,  0.000000000000000 ,  2.475399000000000 ],
                     [  2.432096000272727 ,  -2.143565000000000 ,  1.238232000000000 ],
                     [  2.432096000272727 ,  -2.143565000000000 ,  -1.238232000000000 ]],
    'positions 1.2':[[  -1.080615000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.779254000 ,  -1.206008000 ,  0.000000000 ],
                     [  -3.173171000 ,  -1.207177000 ,  0.000000000 ],
                     [  -3.870155000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.173171000 ,  1.207177000 ,  0.000000000 ],
                     [  -1.779254000 ,  1.206008000 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.236002000 ,  -2.141639000 ,  0.000000000 ],
                     [  -3.714575000 ,  -2.143566000 ,  0.000000000 ],
                     [  -4.951730000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.714575000 ,  2.143566000 ,  0.000000000 ],
                     [  -1.236002000 ,  2.141639000 ,  0.000000000 ],
                     [  2.919149867 ,  0.000000000 ,  -1.394063000 ],
                     [  2.919625867 ,  1.207238000 ,  -0.697047000 ],
                     [  2.919625867 ,  1.207238000 ,  0.697047000 ],
                     [  2.919149867 ,  0.000000000 ,  1.394063000 ],
                     [  2.919625867 ,  -1.207238000 ,  0.697047000 ],
                     [  2.919625867 ,  -1.207238000 ,  -0.697047000 ],
                     [  2.915319867 ,  0.000000000 ,  -2.475399000 ],
                     [  2.918673867 ,  2.143565000 ,  -1.238232000 ],
                     [  2.918673867 ,  2.143565000 ,  1.238232000 ],
                     [  2.915319867 ,  0.000000000 ,  2.475399000 ],
                     [  2.918673867 ,  -2.143565000 ,  1.238232000 ],
                     [  2.918673867 ,  -2.143565000 ,  -1.238232000 ]],
    'positions 1.5':[[  -1.080615000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.779254000 ,  -1.206008000 ,  0.000000000 ],
                     [  -3.173171000 ,  -1.207177000 ,  0.000000000 ],
                     [  -3.870155000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.173171000 ,  1.207177000 ,  0.000000000 ],
                     [  -1.779254000 ,  1.206008000 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.236002000 ,  -2.141639000 ,  0.000000000 ],
                     [  -3.714575000 ,  -2.143566000 ,  0.000000000 ],
                     [  -4.951730000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.714575000 ,  2.143566000 ,  0.000000000 ],
                     [  -1.236002000 ,  2.141639000 ,  0.000000000 ],
                     [  3.649016667 ,  0.000000000 ,  -1.394063000 ],
                     [  3.649492667 ,  1.207238000 ,  -0.697047000 ],
                     [  3.649492667 ,  1.207238000 ,  0.697047000 ],
                     [  3.649016667 ,  0.000000000 ,  1.394063000 ],
                     [  3.649492667 ,  -1.207238000 ,  0.697047000 ],
                     [  3.649492667 ,  -1.207238000 ,  -0.697047000 ],
                     [  3.645186667 ,  0.000000000 ,  -2.475399000 ],
                     [  3.648540667 ,  2.143565000 ,  -1.238232000 ],
                     [  3.648540667 ,  2.143565000 ,  1.238232000 ],
                     [  3.645186667 ,  0.000000000 ,  2.475399000 ],
                     [  3.648540667 ,  -2.143565000 ,  1.238232000 ],
                     [  3.648540667 ,  -2.143565000 ,  -1.238232000 ]],
    'positions 2.0':[[  -1.080615000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.779254000 ,  -1.206008000 ,  0.000000000 ],
                     [  -3.173171000 ,  -1.207177000 ,  0.000000000 ],
                     [  -3.870155000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.173171000 ,  1.207177000 ,  0.000000000 ],
                     [  -1.779254000 ,  1.206008000 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -1.236002000 ,  -2.141639000 ,  0.000000000 ],
                     [  -3.714575000 ,  -2.143566000 ,  0.000000000 ],
                     [  -4.951730000 ,  0.000000000 ,  0.000000000 ],
                     [  -3.714575000 ,  2.143566000 ,  0.000000000 ],
                     [  -1.236002000 ,  2.141639000 ,  0.000000000 ],
                     [  4.865461333 ,  0.000000000 ,  -1.394063000 ],
                     [  4.865937333 ,  1.207238000 ,  -0.697047000 ],
                     [  4.865937333 ,  1.207238000 ,  0.697047000 ],
                     [  4.865461333 ,  0.000000000 ,  1.394063000 ],
                     [  4.865937333 ,  -1.207238000 ,  0.697047000 ],
                     [  4.865937333 ,  -1.207238000 ,  -0.697047000 ],
                     [  4.861631333 ,  0.000000000 ,  -2.475399000 ],
                     [  4.864985333 ,  2.143565000 ,  -1.238232000 ],
                     [  4.864985333 ,  2.143565000 ,  1.238232000 ],
                     [  4.861631333 ,  0.000000000 ,  2.475399000 ],
                     [  4.864985333 ,  -2.143565000 ,  1.238232000 ],
                     [  4.864985333 ,  -2.143565000 ,  -1.238232000 ]]},

'Benzene-HCN_complex': {
    'description': "Complex, S22, S26",
    'name': "Benzene-HCN_complex",
    's26_number': "19",
    'interaction energy CC':-0.1973,
    'interaction energies s22x5':[-0.1743,-0.1960,-0.1596,-0.0906,-0.0369],
    'offset': 0.0013,
    'symbols': 'CCCCCCHHHHHHNCH',
    'magmoms': None,
    'dimer atoms': [12,3],
    # Optimisation level:   MP2/cc-pVTZ
    'positions':[[ -0.7097741, -0.9904230,  1.2077018],
                 [ -1.4065340, -0.9653529,  0.0000000],
                 [ -0.7097741, -0.9904230, -1.2077018],
                 [  0.6839651, -1.0405105, -1.2078652],
                 [  1.3809779, -1.0655522,  0.0000000],
                 [  0.6839651, -1.0405105,  1.2078652],
                 [ -1.2499482, -0.9686280,  2.1440507],
                 [ -2.4869197, -0.9237060,  0.0000000],
                 [ -1.2499482, -0.9686280, -2.1440507],
                 [  1.2242882, -1.0580753, -2.1442563],
                 [  2.4615886, -1.1029818,  0.0000000],
                 [  1.2242882, -1.0580753,  2.1442563],
                 [ -0.0034118,  3.5353926,  0.0000000],
                 [  0.0751963,  2.3707040,  0.0000000],
                 [  0.1476295,  1.3052847,  0.0000000]],
    'positions 0.9':[[  -0.023100946 ,  0.696978594 ,  1.207702000 ],
                     [  -0.046160335 ,  1.393808033 ,  0.000000000 ],
                     [  -0.023100946 ,  0.696978594 ,  -1.207702000 ],
                     [  0.023085816 ,  -0.696895106 ,  -1.207865000 ],
                     [  0.046190594 ,  -1.393975010 ,  0.000000000 ],
                     [  0.023085816 ,  -0.696895106 ,  1.207865000 ],
                     [  -0.038624622 ,  1.237369182 ,  2.144051000 ],
                     [  -0.079148681 ,  2.474493071 ,  0.000000000 ],
                     [  -0.038624622 ,  1.237369182 ,  -2.144051000 ],
                     [  0.042839694 ,  -1.237142510 ,  -2.144256000 ],
                     [  0.083401415 ,  -2.474593580 ,  0.000000000 ],
                     [  0.042839694 ,  -1.237142510 ,  2.144256000 ],
                     [  4.308034683 ,  0.304536859 ,  0.000000000 ],
                     [  3.151543935 ,  0.145763954 ,  0.000000000 ],
                     [  2.093660645 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.0':[[  -0.023100946000000 ,  0.696978594000000 ,  1.207702000000000 ],
                     [  -0.046160335000000 ,  1.393808033000000 ,  0.000000000000000 ],
                     [  -0.023100946000000 ,  0.696978594000000 ,  -1.207702000000000 ],
                     [  0.023085816000000 ,  -0.696895106000000 ,  -1.207865000000000 ],
                     [  0.046190594000000 ,  -1.393975010000000 ,  0.000000000000000 ],
                     [  0.023085816000000 ,  -0.696895106000000 ,  1.207865000000000 ],
                     [  -0.038624622000000 ,  1.237369182000000 ,  2.144051000000000 ],
                     [  -0.079148681000000 ,  2.474493071000000 ,  0.000000000000000 ],
                     [  -0.038624622000000 ,  1.237369182000000 ,  -2.144051000000000 ],
                     [  0.042839694000000 ,  -1.237142510000000 ,  -2.144256000000000 ],
                     [  0.083401415000000 ,  -2.474593580000000 ,  0.000000000000000 ],
                     [  0.042839694000000 ,  -1.237142510000000 ,  2.144256000000000 ],
                     [  4.540663643636363 ,  0.304536859000000 ,  0.000000000000000 ],
                     [  3.384172895636364 ,  0.145763954000000 ,  0.000000000000000 ],
                     [  2.326289605636364 ,  0.000000000000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  -0.023100946 ,  0.696978594 ,  1.207702000 ],
                     [  -0.046160335 ,  1.393808033 ,  0.000000000 ],
                     [  -0.023100946 ,  0.696978594 ,  -1.207702000 ],
                     [  0.023085816 ,  -0.696895106 ,  -1.207865000 ],
                     [  0.046190594 ,  -1.393975010 ,  0.000000000 ],
                     [  0.023085816 ,  -0.696895106 ,  1.207865000 ],
                     [  -0.038624622 ,  1.237369182 ,  2.144051000 ],
                     [  -0.079148681 ,  2.474493071 ,  0.000000000 ],
                     [  -0.038624622 ,  1.237369182 ,  -2.144051000 ],
                     [  0.042839694 ,  -1.237142510 ,  -2.144256000 ],
                     [  0.083401415 ,  -2.474593580 ,  0.000000000 ],
                     [  0.042839694 ,  -1.237142510 ,  2.144256000 ],
                     [  5.005921565 ,  0.304536859 ,  0.000000000 ],
                     [  3.849430817 ,  0.145763954 ,  0.000000000 ],
                     [  2.791547527 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.5':[[  -0.023100946 ,  0.696978594 ,  1.207702000 ],
                     [  -0.046160335 ,  1.393808033 ,  0.000000000 ],
                     [  -0.023100946 ,  0.696978594 ,  -1.207702000 ],
                     [  0.023085816 ,  -0.696895106 ,  -1.207865000 ],
                     [  0.046190594 ,  -1.393975010 ,  0.000000000 ],
                     [  0.023085816 ,  -0.696895106 ,  1.207865000 ],
                     [  -0.038624622 ,  1.237369182 ,  2.144051000 ],
                     [  -0.079148681 ,  2.474493071 ,  0.000000000 ],
                     [  -0.038624622 ,  1.237369182 ,  -2.144051000 ],
                     [  0.042839694 ,  -1.237142510 ,  -2.144256000 ],
                     [  0.083401415 ,  -2.474593580 ,  0.000000000 ],
                     [  0.042839694 ,  -1.237142510 ,  2.144256000 ],
                     [  5.703808447 ,  0.304536859 ,  0.000000000 ],
                     [  4.547317699 ,  0.145763954 ,  0.000000000 ],
                     [  3.489434409 ,  0.000000000 ,  0.000000000 ]],
    'positions 2.0':[[  -0.023100946 ,  0.696978594 ,  1.207702000 ],
                     [  -0.046160335 ,  1.393808033 ,  0.000000000 ],
                     [  -0.023100946 ,  0.696978594 ,  -1.207702000 ],
                     [  0.023085816 ,  -0.696895106 ,  -1.207865000 ],
                     [  0.046190594 ,  -1.393975010 ,  0.000000000 ],
                     [  0.023085816 ,  -0.696895106 ,  1.207865000 ],
                     [  -0.038624622 ,  1.237369182 ,  2.144051000 ],
                     [  -0.079148681 ,  2.474493071 ,  0.000000000 ],
                     [  -0.038624622 ,  1.237369182 ,  -2.144051000 ],
                     [  0.042839694 ,  -1.237142510 ,  -2.144256000 ],
                     [  0.083401415 ,  -2.474593580 ,  0.000000000 ],
                     [  0.042839694 ,  -1.237142510 ,  2.144256000 ],
                     [  6.866953250 ,  0.304536859 ,  0.000000000 ],
                     [  5.710462502 ,  0.145763954 ,  0.000000000 ],
                     [  4.652579212 ,  0.000000000 ,  0.000000000 ]]},

'Benzene-water_complex': {
    'description': "Complex, S22, S26",
    'name': "Benzene-water_complex",
    's26_number': "17",
    'interaction energy CC':-0.1427,
    'interaction energies s22x5':[-0.1305,-0.1418,-0.1071,-0.0564,-0.0212],
    'offset': 0.0009,
    'symbols': 'CCCCCCHHHHHHOHH',
    'magmoms': None,
    'dimer atoms': [12,3],
    # Optimisation level:   MP2/cc-pVTZ
    'positions':[[  0.7806117, -0.6098875, -1.2075426],
                 [  0.4784039,  0.7510406, -1.2079040],
                 [  0.3276592,  1.4318573,  0.0000000],
                 [  0.4784039,  0.7510406,  1.2079040],
                 [  0.7806117, -0.6098875,  1.2075426],
                 [  0.9321510, -1.2899614,  0.0000000],
                 [  0.8966688, -1.1376051, -2.1441482],
                 [  0.3573895,  1.2782091, -2.1440546],
                 [  0.0918593,  2.4871407,  0.0000000],
                 [  0.3573895,  1.2782091,  2.1440546],
                 [  0.8966688, -1.1376051,  2.1441482],
                 [  1.1690064, -2.3451668,  0.0000000],
                 [ -2.7885270, -0.2744854,  0.0000000],
                 [ -2.6229114, -1.2190831,  0.0000000],
                 [ -1.9015103,  0.0979110,  0.0000000]],
    'positions 0.9':[[  0.068736158 ,  1.392383840 ,  -1.207543000 ],
                     [  0.000000000 ,  0.000000000 ,  -1.207904000 ],
                     [  -0.034807303 ,  -0.696435878 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207904000 ],
                     [  0.068736158 ,  1.392383840 ,  1.207543000 ],
                     [  0.102581137 ,  2.088313342 ,  0.000000000 ],
                     [  0.096477114 ,  1.931999350 ,  -2.144148000 ],
                     [  -0.022815407 ,  -0.540397951 ,  -2.144055000 ],
                     [  -0.086694943 ,  -1.776497744 ,  0.000000000 ],
                     [  -0.022815407 ,  -0.540397951 ,  2.144055000 ],
                     [  0.096477114 ,  1.931999350 ,  2.144148000 ],
                     [  0.153430751 ,  3.168579194 ,  0.000000000 ],
                     [  3.175061618 ,  0.124369730 ,  0.000000000 ],
                     [  3.265337861 ,  1.079117991 ,  0.000000000 ],
                     [  2.221117117 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.0':[[  0.068736158000000 ,  1.392383840000000 ,  -1.207543000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  -1.207904000000000 ],
                     [  -0.034807303000000 ,  -0.696435878000000 ,  0.000000000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  1.207904000000000 ],
                     [  0.068736158000000 ,  1.392383840000000 ,  1.207543000000000 ],
                     [  0.102581137000000 ,  2.088313342000000 ,  0.000000000000000 ],
                     [  0.096477114000000 ,  1.931999350000000 ,  -2.144148000000000 ],
                     [  -0.022815407000000 ,  -0.540397951000000 ,  -2.144055000000000 ],
                     [  -0.086694943000000 ,  -1.776497744000000 ,  0.000000000000000 ],
                     [  -0.022815407000000 ,  -0.540397951000000 ,  2.144055000000000 ],
                     [  0.096477114000000 ,  1.931999350000000 ,  2.144148000000000 ],
                     [  0.153430751000000 ,  3.168579194000000 ,  0.000000000000000 ],
                     [  3.421852408818182 ,  0.124369730000000 ,  0.000000000000000 ],
                     [  3.512128651818182 ,  1.079117991000000 ,  0.000000000000000 ],
                     [  2.467907907818182 ,  0.000000000000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  0.068736158 ,  1.392383840 ,  -1.207543000 ],
                     [  0.000000000 ,  0.000000000 ,  -1.207904000 ],
                     [  -0.034807303 ,  -0.696435878 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207904000 ],
                     [  0.068736158 ,  1.392383840 ,  1.207543000 ],
                     [  0.102581137 ,  2.088313342 ,  0.000000000 ],
                     [  0.096477114 ,  1.931999350 ,  -2.144148000 ],
                     [  -0.022815407 ,  -0.540397951 ,  -2.144055000 ],
                     [  -0.086694943 ,  -1.776497744 ,  0.000000000 ],
                     [  -0.022815407 ,  -0.540397951 ,  2.144055000 ],
                     [  0.096477114 ,  1.931999350 ,  2.144148000 ],
                     [  0.153430751 ,  3.168579194 ,  0.000000000 ],
                     [  3.915433991 ,  0.124369730 ,  0.000000000 ],
                     [  4.005710234 ,  1.079117991 ,  0.000000000 ],
                     [  2.961489490 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.5':[[  0.068736158 ,  1.392383840 ,  -1.207543000 ],
                     [  0.000000000 ,  0.000000000 ,  -1.207904000 ],
                     [  -0.034807303 ,  -0.696435878 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207904000 ],
                     [  0.068736158 ,  1.392383840 ,  1.207543000 ],
                     [  0.102581137 ,  2.088313342 ,  0.000000000 ],
                     [  0.096477114 ,  1.931999350 ,  -2.144148000 ],
                     [  -0.022815407 ,  -0.540397951 ,  -2.144055000 ],
                     [  -0.086694943 ,  -1.776497744 ,  0.000000000 ],
                     [  -0.022815407 ,  -0.540397951 ,  2.144055000 ],
                     [  0.096477114 ,  1.931999350 ,  2.144148000 ],
                     [  0.153430751 ,  3.168579194 ,  0.000000000 ],
                     [  4.655806363 ,  0.124369730 ,  0.000000000 ],
                     [  4.746082606 ,  1.079117991 ,  0.000000000 ],
                     [  3.701861862 ,  0.000000000 ,  0.000000000 ]],
    'positions 2.0':[[  0.068736158 ,  1.392383840 ,  -1.207543000 ],
                     [  0.000000000 ,  0.000000000 ,  -1.207904000 ],
                     [  -0.034807303 ,  -0.696435878 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  1.207904000 ],
                     [  0.068736158 ,  1.392383840 ,  1.207543000 ],
                     [  0.102581137 ,  2.088313342 ,  0.000000000 ],
                     [  0.096477114 ,  1.931999350 ,  -2.144148000 ],
                     [  -0.022815407 ,  -0.540397951 ,  -2.144055000 ],
                     [  -0.086694943 ,  -1.776497744 ,  0.000000000 ],
                     [  -0.022815407 ,  -0.540397951 ,  2.144055000 ],
                     [  0.096477114 ,  1.931999350 ,  2.144148000 ],
                     [  0.153430751 ,  3.168579194 ,  0.000000000 ],
                     [  5.889760317 ,  0.124369730 ,  0.000000000 ],
                     [  5.980036560 ,  1.079117991 ,  0.000000000 ],
                     [  4.935815816 ,  0.000000000 ,  0.000000000 ]]},

'Ethene_dimer': {
    'description': "Complex, S22, S26, stack, dispersion bonded",
    'name': "Ethene_dimer",
    's26_number': "09",
    'interaction energy CC':-0.0650,
    'interaction energies s22x5':[-0.0295,-0.0642,-0.0351,-0.0087,-0.0013],
    'offset': 0.0008,
    'symbols': 'CCHHHHCCHHHH',
    'magmoms': None,
    'dimer atoms': [6,6],
    # Optimisation level: CCSD(T)/cc-pVQZ
    'positions':[[ -0.471925, -0.471925, -1.859111],
                 [  0.471925,  0.471925, -1.859111],
                 [ -0.872422, -0.872422, -0.936125],
                 [  0.872422,  0.872422, -0.936125],
                 [ -0.870464, -0.870464, -2.783308],
                 [  0.870464,  0.870464, -2.783308],
                 [ -0.471925,  0.471925,  1.859111],
                 [  0.471925, -0.471925,  1.859111],
                 [ -0.872422,  0.872422,  0.936125],
                 [  0.872422, -0.872422,  0.936125],
                 [ -0.870464,  0.870464,  2.783308],
                 [  0.870464, -0.870464,  2.783308]],
     'positions 0.9':[[  0.000000000 ,  -0.471925000 ,  0.471925000 ],
                      [  0.000000000 ,  0.471925000 ,  -0.471925000 ],
                      [  0.922986000 ,  -0.872422000 ,  0.872422000 ],
                      [  0.922986000 ,  0.872422000 ,  -0.872422000 ],
                      [  -0.924197000 ,  -0.870464000 ,  0.870464000 ],
                      [  -0.924197000 ,  0.870464000 ,  -0.870464000 ],
                      [  3.346399800 ,  0.471925000 ,  0.471925000 ],
                      [  3.346399800 ,  -0.471925000 ,  -0.471925000 ],
                      [  2.423413800 ,  0.872422000 ,  0.872422000 ],
                      [  2.423413800 ,  -0.872422000 ,  -0.872422000 ],
                      [  4.270596800 ,  0.870464000 ,  0.870464000 ],
                      [  4.270596800 ,  -0.870464000 ,  -0.870464000 ]],
    'positions 1.0':[[  0.000000000000000 ,  -0.471925000000000 ,  0.471925000000000 ],
                     [  0.000000000000000 ,  0.471925000000000 ,  -0.471925000000000 ],
                     [  0.922986000000000 ,  -0.872422000000000 ,  0.872422000000000 ],
                     [  0.922986000000000 ,  0.872422000000000 ,  -0.872422000000000 ],
                     [  -0.924197000000000 ,  -0.870464000000000 ,  0.870464000000000 ],
                     [  -0.924197000000000 ,  0.870464000000000 ,  -0.870464000000000 ],
                     [  3.718222000000000 ,  0.471925000000000 ,  0.471925000000000 ],
                     [  3.718222000000000 ,  -0.471925000000000 ,  -0.471925000000000 ],
                     [  2.795236000000000 ,  0.872422000000000 ,  0.872422000000000 ],
                     [  2.795236000000000 ,  -0.872422000000000 ,  -0.872422000000000 ],
                     [  4.642418999999999 ,  0.870464000000000 ,  0.870464000000000 ],
                     [  4.642418999999999 ,  -0.870464000000000 ,  -0.870464000000000 ]],
    'positions 1.2':[[  0.000000000 ,  -0.471925000 ,  0.471925000 ],
                     [  0.000000000 ,  0.471925000 ,  -0.471925000 ],
                     [  0.922986000 ,  -0.872422000 ,  0.872422000 ],
                     [  0.922986000 ,  0.872422000 ,  -0.872422000 ],
                     [  -0.924197000 ,  -0.870464000 ,  0.870464000 ],
                     [  -0.924197000 ,  0.870464000 ,  -0.870464000 ],
                     [  4.461866400 ,  0.471925000 ,  0.471925000 ],
                     [  4.461866400 ,  -0.471925000 ,  -0.471925000 ],
                     [  3.538880400 ,  0.872422000 ,  0.872422000 ],
                     [  3.538880400 ,  -0.872422000 ,  -0.872422000 ],
                     [  5.386063400 ,  0.870464000 ,  0.870464000 ],
                     [  5.386063400 ,  -0.870464000 ,  -0.870464000 ]],
    'positions 1.5':[[  0.000000000 ,  -0.471925000 ,  0.471925000 ],
                     [  0.000000000 ,  0.471925000 ,  -0.471925000 ],
                     [  0.922986000 ,  -0.872422000 ,  0.872422000 ],
                     [  0.922986000 ,  0.872422000 ,  -0.872422000 ],
                     [  -0.924197000 ,  -0.870464000 ,  0.870464000 ],
                     [  -0.924197000 ,  0.870464000 ,  -0.870464000 ],
                     [  5.577333000 ,  0.471925000 ,  0.471925000 ],
                     [  5.577333000 ,  -0.471925000 ,  -0.471925000 ],
                     [  4.654347000 ,  0.872422000 ,  0.872422000 ],
                     [  4.654347000 ,  -0.872422000 ,  -0.872422000 ],
                     [  6.501530000 ,  0.870464000 ,  0.870464000 ],
                     [  6.501530000 ,  -0.870464000 ,  -0.870464000 ]],
    'positions 2.0':[[  0.000000000 ,  -0.471925000 ,  0.471925000 ],
                     [  0.000000000 ,  0.471925000 ,  -0.471925000 ],
                     [  0.922986000 ,  -0.872422000 ,  0.872422000 ],
                     [  0.922986000 ,  0.872422000 ,  -0.872422000 ],
                     [  -0.924197000 ,  -0.870464000 ,  0.870464000 ],
                     [  -0.924197000 ,  0.870464000 ,  -0.870464000 ],
                     [  7.436444000 ,  0.471925000 ,  0.471925000 ],
                     [  7.436444000 ,  -0.471925000 ,  -0.471925000 ],
                     [  6.513458000 ,  0.872422000 ,  0.872422000 ],
                     [  6.513458000 ,  -0.872422000 ,  -0.872422000 ],
                     [  8.360641000 ,  0.870464000 ,  0.870464000 ],
                     [  8.360641000 ,  -0.870464000 ,  -0.870464000 ]]},

'Ethene-ethyne_complex': {
    'description': "Complex, S22, S26",
    'name': "Ethene-ethyne_complex",
    's26_number': "16",
    'interaction energy CC':-0.0655,
    'interaction energies s22x5':[-0.0507,-0.0646,-0.0468,-0.0212,-0.0065],
    'offset': 0.0009,
    'symbols': 'CCHHHHCCHH',
    'magmoms': None,
    'dimer atoms': [6,4],
    # Optimisation level: CCSD(T)/cc-pVQZ
    'positions':[[  0.000000, -0.667578, -2.124659],
                 [  0.000000,  0.667578, -2.124659],
                 [  0.923621, -1.232253, -2.126185],
                 [ -0.923621, -1.232253, -2.126185],
                 [ -0.923621,  1.232253, -2.126185],
                 [  0.923621,  1.232253, -2.126185],
                 [  0.000000,  0.000000,  2.900503],
                 [  0.000000,  0.000000,  1.693240],
                 [  0.000000,  0.000000,  0.627352],
                 [  0.000000,  0.000000,  3.963929]],
    'positions 0.9':[[  0.000000000 ,  -0.667578000 ,  0.000000000 ],
                     [  0.000000000 ,  0.667578000 ,  0.000000000 ],
                     [  -0.001526000 ,  -1.232253000 ,  -0.923621000 ],
                     [  -0.001526000 ,  -1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  -0.923621000 ],
                     [  4.749960900 ,  0.000000000 ,  0.000000000 ],
                     [  3.542697900 ,  0.000000000 ,  0.000000000 ],
                     [  2.476809900 ,  0.000000000 ,  0.000000000 ],
                     [  5.813386900 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.0':[[  0.000000000000000 ,  -0.667578000000000 ,  0.000000000000000 ],
                     [  0.000000000000000 ,  0.667578000000000 ,  0.000000000000000 ],
                     [  -0.001526000000000 ,  -1.232253000000000 ,  -0.923621000000000 ],
                     [  -0.001526000000000 ,  -1.232253000000000 ,  0.923621000000000 ],
                     [  -0.001526000000000 ,  1.232253000000000 ,  0.923621000000000 ],
                     [  -0.001526000000000 ,  1.232253000000000 ,  -0.923621000000000 ],
                     [  5.025162000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  3.817899000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  2.752011000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  6.088588000000001 ,  0.000000000000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  0.000000000 ,  -0.667578000 ,  0.000000000 ],
                     [  0.000000000 ,  0.667578000 ,  0.000000000 ],
                     [  -0.001526000 ,  -1.232253000 ,  -0.923621000 ],
                     [  -0.001526000 ,  -1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  -0.923621000 ],
                     [  5.575564200 ,  0.000000000 ,  0.000000000 ],
                     [  4.368301200 ,  0.000000000 ,  0.000000000 ],
                     [  3.302413200 ,  0.000000000 ,  0.000000000 ],
                     [  6.638990200 ,  0.000000000 ,  0.000000000 ]],
    'positions 1.5':[[  0.000000000 ,  -0.667578000 ,  0.000000000 ],
                     [  0.000000000 ,  0.667578000 ,  0.000000000 ],
                     [  -0.001526000 ,  -1.232253000 ,  -0.923621000 ],
                     [  -0.001526000 ,  -1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  -0.923621000 ],
                     [  6.401167500 ,  0.000000000 ,  0.000000000 ],
                     [  5.193904500 ,  0.000000000 ,  0.000000000 ],
                     [  4.128016500 ,  0.000000000 ,  0.000000000 ],
                     [  7.464593500 ,  0.000000000 ,  0.000000000 ]],
    'positions 2.0':[[  0.000000000 ,  -0.667578000 ,  0.000000000 ],
                     [  0.000000000 ,  0.667578000 ,  0.000000000 ],
                     [  -0.001526000 ,  -1.232253000 ,  -0.923621000 ],
                     [  -0.001526000 ,  -1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  0.923621000 ],
                     [  -0.001526000 ,  1.232253000 ,  -0.923621000 ],
                     [  7.777173000 ,  0.000000000 ,  0.000000000 ],
                     [  6.569910000 ,  0.000000000 ,  0.000000000 ],
                     [  5.504022000 ,  0.000000000 ,  0.000000000 ],
                     [  8.840599000 ,  0.000000000 ,  0.000000000 ]]},

'Formamide_dimer': {
    'description': "Complex, S22, S26, 2 h-bonds, double h-bond",
    'name': "Formamide_dimer",
    's26_number': "04",
    'interaction energy CC':-0.6990,
    'interaction energies s22x5':[-0.6132,-0.6917,-0.5811,-0.3512,-0.1522],
    'offset': 0.0073,
    'symbols': 'CONHHHCONHHH',
    'magmoms': None,
    'dimer atoms': [6,6],
    # Optimisation level: MP2/cc-pVTZ
    'positions':[[ -2.018649,  0.052883,  0.000000],
                 [ -1.452200,  1.143634,  0.000000],
                 [ -1.407770, -1.142484,  0.000000],
                 [ -1.964596, -1.977036,  0.000000],
                 [ -0.387244, -1.207782,  0.000000],
                 [ -3.117061, -0.013701,  0.000000],
                 [  2.018649, -0.052883,  0.000000],
                 [  1.452200, -1.143634,  0.000000],
                 [  1.407770,  1.142484,  0.000000],
                 [  1.964596,  1.977036,  0.000000],
                 [  0.387244,  1.207782,  0.000000],
                 [  3.117061,  0.013701,  0.000000]],
    'positions 0.9':[[  -0.604120150 ,  -1.070346233 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.035273679 ,  -2.286277608 ,  0.000000000 ],
                     [  -0.620847527 ,  -3.100915874 ,  0.000000000 ],
                     [  0.982356530 ,  -2.387103713 ,  0.000000000 ],
                     [  -1.704185444 ,  -1.098607493 ,  0.000000000 ],
                     [  3.242982655 ,  -1.316757480 ,  0.000000000 ],
                     [  2.638862505 ,  -2.387103713 ,  0.000000000 ],
                     [  2.674136184 ,  -0.100826104 ,  0.000000000 ],
                     [  3.259710032 ,  0.713812161 ,  0.000000000 ],
                     [  1.656505975 ,  0.000000000 ,  0.000000000 ],
                     [  4.343047949 ,  -1.288496220 ,  0.000000000 ]],
    'positions 1.0':[[  -0.604120150000000 ,  -1.070346233000000 ,  0.000000000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -0.035273679000000 ,  -2.286277608000000 ,  0.000000000000000 ],
                     [  -0.620847527000000 ,  -3.100915874000000 ,  0.000000000000000 ],
                     [  0.982356530000000 ,  -2.387103713000000 ,  0.000000000000000 ],
                     [  -1.704185444000000 ,  -1.098607493000000 ,  0.000000000000000 ],
                     [  3.427038874545455 ,  -1.316757480000000 ,  0.000000000000000 ],
                     [  2.822918724545455 ,  -2.387103713000000 ,  0.000000000000000 ],
                     [  2.858192403545455 ,  -0.100826104000000 ,  0.000000000000000 ],
                     [  3.443766251545455 ,  0.713812161000000 ,  0.000000000000000 ],
                     [  1.840562194545454 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  4.527104168545454 ,  -1.288496220000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  -0.604120150 ,  -1.070346233 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.035273679 ,  -2.286277608 ,  0.000000000 ],
                     [  -0.620847527 ,  -3.100915874 ,  0.000000000 ],
                     [  0.982356530 ,  -2.387103713 ,  0.000000000 ],
                     [  -1.704185444 ,  -1.098607493 ,  0.000000000 ],
                     [  3.795151314 ,  -1.316757480 ,  0.000000000 ],
                     [  3.191031164 ,  -2.387103713 ,  0.000000000 ],
                     [  3.226304843 ,  -0.100826104 ,  0.000000000 ],
                     [  3.811878691 ,  0.713812161 ,  0.000000000 ],
                     [  2.208674634 ,  0.000000000 ,  0.000000000 ],
                     [  4.895216608 ,  -1.288496220 ,  0.000000000 ]],
    'positions 1.5':[[  -0.604120150 ,  -1.070346233 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.035273679 ,  -2.286277608 ,  0.000000000 ],
                     [  -0.620847527 ,  -3.100915874 ,  0.000000000 ],
                     [  0.982356530 ,  -2.387103713 ,  0.000000000 ],
                     [  -1.704185444 ,  -1.098607493 ,  0.000000000 ],
                     [  4.347319973 ,  -1.316757480 ,  0.000000000 ],
                     [  3.743199823 ,  -2.387103713 ,  0.000000000 ],
                     [  3.778473502 ,  -0.100826104 ,  0.000000000 ],
                     [  4.364047350 ,  0.713812161 ,  0.000000000 ],
                     [  2.760843293 ,  0.000000000 ,  0.000000000 ],
                     [  5.447385267 ,  -1.288496220 ,  0.000000000 ]],
    'positions 2.0':[[  -0.604120150 ,  -1.070346233 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.035273679 ,  -2.286277608 ,  0.000000000 ],
                     [  -0.620847527 ,  -3.100915874 ,  0.000000000 ],
                     [  0.982356530 ,  -2.387103713 ,  0.000000000 ],
                     [  -1.704185444 ,  -1.098607493 ,  0.000000000 ],
                     [  5.267601070 ,  -1.316757480 ,  0.000000000 ],
                     [  4.663480920 ,  -2.387103713 ,  0.000000000 ],
                     [  4.698754599 ,  -0.100826104 ,  0.000000000 ],
                     [  5.284328447 ,  0.713812161 ,  0.000000000 ],
                     [  3.681124390 ,  0.000000000 ,  0.000000000 ],
                     [  6.367666364 ,  -1.288496220 ,  0.000000000 ]]},

'Formic_acid_dimer': {
    'description': "Complex, S22, S26, 2 h-bonds, double h-bond",
    'name': "Formic_acid_dimer",
    's26_number': "03",
    'interaction energy CC':-0.8152,
    'interaction energies s22x5':[-0.7086,-0.8061,-0.6773,-0.4007,-0.1574],
    'offset': 0.0091,
    'symbols': 'COOHHCOOHH',
    'magmoms': None,
    'dimer atoms': [5,5],
    # Optimisation level:  CCSD(T)/cc-pVTZ
    'positions':[[ -1.888896, -0.179692,  0.000000],
                 [ -1.493280,  1.073689,  0.000000],
                 [ -1.170435, -1.166590,  0.000000],
                 [ -2.979488, -0.258829,  0.000000],
                 [ -0.498833,  1.107195,  0.000000],
                 [  1.888896,  0.179692,  0.000000],
                 [  1.493280, -1.073689,  0.000000],
                 [  1.170435,  1.166590,  0.000000],
                 [  2.979488,  0.258829,  0.000000],
                 [  0.498833, -1.107195,  0.000000]],
    'positions 0.9':[[  -1.434944263 ,  -1.236643950 ,  0.000000000 ],
                     [  -0.995009531 ,  0.001876693 ,  0.000000000 ],
                     [  -0.752030700 ,  -2.248465543 ,  0.000000000 ],
                     [  -2.527660580 ,  -1.276950582 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  2.186205474 ,  -1.011821594 ,  0.000000000 ],
                     [  1.746270742 ,  -2.250342236 ,  0.000000000 ],
                     [  1.503291911 ,  0.000000000 ,  0.000000000 ],
                     [  3.278921791 ,  -0.971514961 ,  0.000000000 ],
                     [  0.751261211 ,  -2.248465543 ,  0.000000000 ]],
    'positions 1.0':[[  -1.434944263000000 ,  -1.236643950000000 ,  0.000000000000000 ],
                     [  -0.995009531000000 ,  0.001876693000000 ,  0.000000000000000 ],
                     [  -0.752030700000000 ,  -2.248465543000000 ,  0.000000000000000 ],
                     [  -2.527660580000000 ,  -1.276950582000000 ,  0.000000000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  2.353237908636364 ,  -1.011821594000000 ,  0.000000000000000 ],
                     [  1.913303176636364 ,  -2.250342236000000 ,  0.000000000000000 ],
                     [  1.670324345636364 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  3.445954225636364 ,  -0.971514961000000 ,  0.000000000000000 ],
                     [  0.918293645636364 ,  -2.248465543000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  -1.434944263 ,  -1.236643950 ,  0.000000000 ],
                     [  -0.995009531 ,  0.001876693 ,  0.000000000 ],
                     [  -0.752030700 ,  -2.248465543 ,  0.000000000 ],
                     [  -2.527660580 ,  -1.276950582 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  2.687302778 ,  -1.011821594 ,  0.000000000 ],
                     [  2.247368046 ,  -2.250342236 ,  0.000000000 ],
                     [  2.004389215 ,  0.000000000 ,  0.000000000 ],
                     [  3.780019095 ,  -0.971514961 ,  0.000000000 ],
                     [  1.252358515 ,  -2.248465543 ,  0.000000000 ]],
    'positions 1.5':[[  -1.434944263 ,  -1.236643950 ,  0.000000000 ],
                     [  -0.995009531 ,  0.001876693 ,  0.000000000 ],
                     [  -0.752030700 ,  -2.248465543 ,  0.000000000 ],
                     [  -2.527660580 ,  -1.276950582 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  3.188400082 ,  -1.011821594 ,  0.000000000 ],
                     [  2.748465350 ,  -2.250342236 ,  0.000000000 ],
                     [  2.505486519 ,  0.000000000 ,  0.000000000 ],
                     [  4.281116399 ,  -0.971514961 ,  0.000000000 ],
                     [  1.753455819 ,  -2.248465543 ,  0.000000000 ]],
    'positions 2.0':[[  -1.434944263 ,  -1.236643950 ,  0.000000000 ],
                     [  -0.995009531 ,  0.001876693 ,  0.000000000 ],
                     [  -0.752030700 ,  -2.248465543 ,  0.000000000 ],
                     [  -2.527660580 ,  -1.276950582 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  4.023562255 ,  -1.011821594 ,  0.000000000 ],
                     [  3.583627523 ,  -2.250342236 ,  0.000000000 ],
                     [  3.340648692 ,  0.000000000 ,  0.000000000 ],
                     [  5.116278572 ,  -0.971514961 ,  0.000000000 ],
                     [  2.588617992 ,  -2.248465543 ,  0.000000000 ]]},

'Indole-benzene_complex_stack': {
    'description': "Complex, S22, S26, stack, dispersion bonded",
    'name': "Indole-benzene_complex_stack",
    's26_number': "14",
    'interaction energy CC':-0.1990,
    'interaction energies s22x5':[-0.0924,-0.2246,-0.1565,-0.0468,-0.0043],
    'offset': -0.0256,
    'symbols': 'CCCCCCHHHHHHHCCHCCHCNCCHCHHH',
    'magmoms': None,
    'dimer atoms': [12,16],
    # Optimisation level:  MP2/cc-pVTZ
    'positions':[[ -0.0210742,  1.5318615, -1.3639345],
                 [ -1.2746794,  0.9741030, -1.6074097],
                 [ -1.3783055, -0.2256981, -2.3084154],
                 [ -0.2289426, -0.8664053, -2.7687944],
                 [  1.0247882, -0.3035171, -2.5312410],
                 [  1.1289996,  0.8966787, -1.8299830],
                 [  0.0600740,  2.4565627, -0.8093957],
                 [ -2.1651002,  1.4654521, -1.2405676],
                 [ -2.3509735, -0.6616122, -2.4926698],
                 [ -0.3103419, -1.7955762, -3.3172704],
                 [  1.9165847, -0.7940845, -2.8993942],
                 [  2.1000347,  1.3326757, -1.6400420],
                 [ -2.9417647,  0.8953834,  2.2239054],
                 [ -2.0220674,  0.4258540,  1.9013549],
                 [ -0.8149418,  1.0740453,  2.1066982],
                 [ -0.7851529,  2.0443812,  2.5856086],
                 [  0.3704286,  0.4492852,  1.6847458],
                 [  1.7508619,  0.8038935,  1.7194004],
                 [  2.1870108,  1.6998281,  2.1275903],
                 [  2.4451359, -0.2310742,  1.1353313],
                 [  1.5646462, -1.2137812,  0.7555384],
                 [  0.2861214, -0.8269486,  1.0618752],
                 [ -0.9284667, -1.4853121,  0.8606937],
                 [ -0.9729200, -2.4554847,  0.3834013],
                 [ -2.0792848, -0.8417668,  1.2876443],
                 [ -3.0389974, -1.3203846,  1.1468400],
                 [  1.8075741, -2.0366963,  0.2333038],
                 [  3.5028794, -0.3485344,  0.969523]],
    'positions 0.9':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.044485647 ,  -1.177978626 ,  0.743160105 ],
                     [  -0.010824638 ,  -2.411208517 ,  0.095333145 ],
                     [  0.064150773 ,  -2.466933785 ,  -1.295623602 ],
                     [  0.100950904 ,  -1.287437054 ,  -2.038959973 ],
                     [  0.067356799 ,  -0.053500209 ,  -1.391376263 ],
                     [  -0.013797739 ,  0.956881587 ,  0.503348328 ],
                     [  -0.091346970 ,  -1.134458005 ,  1.822398921 ],
                     [  -0.039754009 ,  -3.325680275 ,  0.672358669 ],
                     [  0.085389531 ,  -3.424849020 ,  -1.798373823 ],
                     [  0.146442780 ,  -1.330172544 ,  -3.119514770 ],
                     [  0.100852832 ,  0.862456237 ,  -1.964945566 ],
                     [  2.717766027 ,  -0.578056849 ,  3.494904751 ],
                     [  2.793508398 ,  -0.571969873 ,  2.415753956 ],
                     [  2.753054336 ,  0.633650134 ,  1.734349558 ],
                     [  2.645935858 ,  1.567038531 ,  2.272036098 ],
                     [  2.855804852 ,  0.624347564 ,  0.333339655 ],
                     [  2.845637545 ,  1.633662034 ,  -0.673499279 ],
                     [  2.762013625 ,  2.698030593 ,  -0.533251753 ],
                     [  2.976224608 ,  0.992808148 ,  -1.884517470 ],
                     [  3.081930238 ,  -0.360086596 ,  -1.675422891 ],
                     [  2.997750328 ,  -0.624347564 ,  -0.333339655 ],
                     [  3.046288127 ,  -1.839842986 ,  0.351754941 ],
                     [  3.153106953 ,  -2.780217935 ,  -0.172940228 ],
                     [  2.941516868 ,  -1.796211682 ,  1.733036170 ],
                     [  2.973148444 ,  -2.718261443 ,  2.297634930 ],
                     [  3.103876306 ,  -1.056446212 ,  -2.398978775 ],
                     [  3.012441631 ,  1.398036276 ,  -2.881807744 ]],
    'positions 1.0':[[  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -0.044485647000000 ,  -1.177978626000000 ,  0.743160105000000 ],
                     [  -0.010824638000000 ,  -2.411208517000000 ,  0.095333145000000 ],
                     [  0.064150773000000 ,  -2.466933785000000 ,  -1.295623602000000 ],
                     [  0.100950904000000 ,  -1.287437054000000 ,  -2.038959973000000 ],
                     [  0.067356799000000 ,  -0.053500209000000 ,  -1.391376263000000 ],
                     [  -0.013797739000000 ,  0.956881587000000 ,  0.503348328000000 ],
                     [  -0.091346970000000 ,  -1.134458005000000 ,  1.822398921000000 ],
                     [  -0.039754009000000 ,  -3.325680275000000 ,  0.672358669000000 ],
                     [  0.085389531000000 ,  -3.424849020000000 ,  -1.798373823000000 ],
                     [  0.146442780000000 ,  -1.330172544000000 ,  -3.119514770000000 ],
                     [  0.100852832000000 ,  0.862456237000000 ,  -1.964945566000000 ],
                     [  3.042963537000000 ,  -0.578056849000000 ,  3.494904751000000 ],
                     [  3.118705908000000 ,  -0.571969873000000 ,  2.415753956000000 ],
                     [  3.078251846000000 ,  0.633650134000000 ,  1.734349558000000 ],
                     [  2.971133368000000 ,  1.567038531000000 ,  2.272036098000000 ],
                     [  3.181002362000000 ,  0.624347564000000 ,  0.333339655000000 ],
                     [  3.170835055000000 ,  1.633662034000000 ,  -0.673499279000000 ],
                     [  3.087211135000000 ,  2.698030593000000 ,  -0.533251753000000 ],
                     [  3.301422118000000 ,  0.992808148000000 ,  -1.884517470000000 ],
                     [  3.407127748000000 ,  -0.360086596000000 ,  -1.675422891000000 ],
                     [  3.322947838000000 ,  -0.624347564000000 ,  -0.333339655000000 ],
                     [  3.371485637000000 ,  -1.839842986000000 ,  0.351754941000000 ],
                     [  3.478304463000000 ,  -2.780217935000000 ,  -0.172940228000000 ],
                     [  3.266714378000000 ,  -1.796211682000000 ,  1.733036170000000 ],
                     [  3.298345954000000 ,  -2.718261443000000 ,  2.297634930000000 ],
                     [  3.429073816000000 ,  -1.056446212000000 ,  -2.398978775000000 ],
                     [  3.337639141000000 ,  1.398036276000000 ,  -2.881807744000000 ]],
    'positions 1.2':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.044485647 ,  -1.177978626 ,  0.743160105 ],
                     [  -0.010824638 ,  -2.411208517 ,  0.095333145 ],
                     [  0.064150773 ,  -2.466933785 ,  -1.295623602 ],
                     [  0.100950904 ,  -1.287437054 ,  -2.038959973 ],
                     [  0.067356799 ,  -0.053500209 ,  -1.391376263 ],
                     [  -0.013797739 ,  0.956881587 ,  0.503348328 ],
                     [  -0.091346970 ,  -1.134458005 ,  1.822398921 ],
                     [  -0.039754009 ,  -3.325680275 ,  0.672358669 ],
                     [  0.085389531 ,  -3.424849020 ,  -1.798373823 ],
                     [  0.146442780 ,  -1.330172544 ,  -3.119514770 ],
                     [  0.100852832 ,  0.862456237 ,  -1.964945566 ],
                     [  3.693358557 ,  -0.578056849 ,  3.494904751 ],
                     [  3.769100928 ,  -0.571969873 ,  2.415753956 ],
                     [  3.728646866 ,  0.633650134 ,  1.734349558 ],
                     [  3.621528388 ,  1.567038531 ,  2.272036098 ],
                     [  3.831397382 ,  0.624347564 ,  0.333339655 ],
                     [  3.821230075 ,  1.633662034 ,  -0.673499279 ],
                     [  3.737606155 ,  2.698030593 ,  -0.533251753 ],
                     [  3.951817138 ,  0.992808148 ,  -1.884517470 ],
                     [  4.057522768 ,  -0.360086596 ,  -1.675422891 ],
                     [  3.973342858 ,  -0.624347564 ,  -0.333339655 ],
                     [  4.021880657 ,  -1.839842986 ,  0.351754941 ],
                     [  4.128699483 ,  -2.780217935 ,  -0.172940228 ],
                     [  3.917109398 ,  -1.796211682 ,  1.733036170 ],
                     [  3.948740974 ,  -2.718261443 ,  2.297634930 ],
                     [  4.079468836 ,  -1.056446212 ,  -2.398978775 ],
                     [  3.988034161 ,  1.398036276 ,  -2.881807744 ]],
    'positions 1.5':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.044485647 ,  -1.177978626 ,  0.743160105 ],
                     [  -0.010824638 ,  -2.411208517 ,  0.095333145 ],
                     [  0.064150773 ,  -2.466933785 ,  -1.295623602 ],
                     [  0.100950904 ,  -1.287437054 ,  -2.038959973 ],
                     [  0.067356799 ,  -0.053500209 ,  -1.391376263 ],
                     [  -0.013797739 ,  0.956881587 ,  0.503348328 ],
                     [  -0.091346970 ,  -1.134458005 ,  1.822398921 ],
                     [  -0.039754009 ,  -3.325680275 ,  0.672358669 ],
                     [  0.085389531 ,  -3.424849020 ,  -1.798373823 ],
                     [  0.146442780 ,  -1.330172544 ,  -3.119514770 ],
                     [  0.100852832 ,  0.862456237 ,  -1.964945566 ],
                     [  4.668951087 ,  -0.578056849 ,  3.494904751 ],
                     [  4.744693458 ,  -0.571969873 ,  2.415753956 ],
                     [  4.704239396 ,  0.633650134 ,  1.734349558 ],
                     [  4.597120918 ,  1.567038531 ,  2.272036098 ],
                     [  4.806989912 ,  0.624347564 ,  0.333339655 ],
                     [  4.796822605 ,  1.633662034 ,  -0.673499279 ],
                     [  4.713198685 ,  2.698030593 ,  -0.533251753 ],
                     [  4.927409668 ,  0.992808148 ,  -1.884517470 ],
                     [  5.033115298 ,  -0.360086596 ,  -1.675422891 ],
                     [  4.948935388 ,  -0.624347564 ,  -0.333339655 ],
                     [  4.997473187 ,  -1.839842986 ,  0.351754941 ],
                     [  5.104292013 ,  -2.780217935 ,  -0.172940228 ],
                     [  4.892701928 ,  -1.796211682 ,  1.733036170 ],
                     [  4.924333504 ,  -2.718261443 ,  2.297634930 ],
                     [  5.055061366 ,  -1.056446212 ,  -2.398978775 ],
                     [  4.963626691 ,  1.398036276 ,  -2.881807744 ]],
    'positions 2.0':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.044485647 ,  -1.177978626 ,  0.743160105 ],
                     [  -0.010824638 ,  -2.411208517 ,  0.095333145 ],
                     [  0.064150773 ,  -2.466933785 ,  -1.295623602 ],
                     [  0.100950904 ,  -1.287437054 ,  -2.038959973 ],
                     [  0.067356799 ,  -0.053500209 ,  -1.391376263 ],
                     [  -0.013797739 ,  0.956881587 ,  0.503348328 ],
                     [  -0.091346970 ,  -1.134458005 ,  1.822398921 ],
                     [  -0.039754009 ,  -3.325680275 ,  0.672358669 ],
                     [  0.085389531 ,  -3.424849020 ,  -1.798373823 ],
                     [  0.146442780 ,  -1.330172544 ,  -3.119514770 ],
                     [  0.100852832 ,  0.862456237 ,  -1.964945566 ],
                     [  6.294938637 ,  -0.578056849 ,  3.494904751 ],
                     [  6.370681008 ,  -0.571969873 ,  2.415753956 ],
                     [  6.330226946 ,  0.633650134 ,  1.734349558 ],
                     [  6.223108468 ,  1.567038531 ,  2.272036098 ],
                     [  6.432977462 ,  0.624347564 ,  0.333339655 ],
                     [  6.422810155 ,  1.633662034 ,  -0.673499279 ],
                     [  6.339186235 ,  2.698030593 ,  -0.533251753 ],
                     [  6.553397218 ,  0.992808148 ,  -1.884517470 ],
                     [  6.659102848 ,  -0.360086596 ,  -1.675422891 ],
                     [  6.574922938 ,  -0.624347564 ,  -0.333339655 ],
                     [  6.623460737 ,  -1.839842986 ,  0.351754941 ],
                     [  6.730279563 ,  -2.780217935 ,  -0.172940228 ],
                     [  6.518689478 ,  -1.796211682 ,  1.733036170 ],
                     [  6.550321054 ,  -2.718261443 ,  2.297634930 ],
                     [  6.681048916 ,  -1.056446212 ,  -2.398978775 ],
                     [  6.589614241 ,  1.398036276 ,  -2.881807744 ]]},

'Indole-benzene_T-shape_complex': {
    'description': "Complex, S22, S26",
    'name': "Indole-benzene_T-shape_complex",
    's26_number': "21",
    'interaction energy CC':-0.2437,
    'interaction energies s22x5':[-0.2164,-0.2489,-0.2116,-0.1214,-0.0477],
    'offset': -0.0052,
    'symbols': 'CCCCCCHHHHHHHNCCCCCCCCHHHHHH',
    'magmoms': None,
    'dimer atoms': [12,16],
    # Optimisation level:  MP2/cc-pVTZ
    'positions':[[  2.5118997,  1.6250148,  0.0000000],
                 [  2.7130094,  0.9578537, -1.2082918],
                 [  3.1177821, -0.3767436, -1.2083647],
                 [  3.3213848, -1.0437307,  0.0000000],
                 [  3.1177821, -0.3767436,  1.2083647],
                 [  2.7130094,  0.9578537,  1.2082918],
                 [  2.2024038,  2.6611358,  0.0000000],
                 [  2.5511760,  1.4736908, -2.1445900],
                 [  3.2702999, -0.8951406, -2.1448379],
                 [  3.6368139, -2.0781521,  0.0000000],
                 [  3.2702999, -0.8951406,  2.1448379],
                 [  2.5511760,  1.4736908,  2.1445900],
                 [  0.8065245, -0.4358866,  0.0000000],
                 [ -0.1442408, -0.7686927,  0.0000000],
                 [ -0.5161122, -2.0893220,  0.0000000],
                 [ -1.8898755, -2.1814495,  0.0000000],
                 [ -2.3932317, -0.8470830,  0.0000000],
                 [ -1.2640653,  0.0195887,  0.0000000],
                 [ -1.3896004,  1.4117668,  0.0000000],
                 [ -2.6726501,  1.9366450,  0.0000000],
                 [ -3.8054511,  1.0974790,  0.0000000],
                 [ -3.6798167, -0.2817209,  0.0000000],
                 [  0.2310024, -2.8653173,  0.0000000],
                 [ -2.4585759, -3.0956052,  0.0000000],
                 [ -0.5188733,  2.0539520,  0.0000000],
                 [ -2.8077570,  3.0097859,  0.0000000],
                 [ -4.7905991,  1.5439372,  0.0000000],
                 [ -4.5580187, -0.9142916,  0.0000000]],
    'positions 0.9':[[  -0.052652077 ,  -1.393225783 ,  0.000000000 ],
                     [  -0.025543347 ,  -0.696940104 ,  -1.208292000 ],
                     [  0.026348254 ,  0.696724226 ,  -1.208365000 ],
                     [  0.051042263 ,  1.393657541 ,  0.000000000 ],
                     [  0.026348254 ,  0.696724226 ,  1.208365000 ],
                     [  -0.025543347 ,  -0.696940104 ,  1.208292000 ],
                     [  -0.097430661 ,  -2.473655966 ,  0.000000000 ],
                     [  -0.040509756 ,  -1.237360068 ,  -2.144590000 ],
                     [  0.050955575 ,  1.236531293 ,  -2.144838000 ],
                     [  0.089657645 ,  2.474412421 ,  0.000000000 ],
                     [  0.050955575 ,  1.236531293 ,  2.144838000 ],
                     [  -0.040509756 ,  -1.237360068 ,  2.144590000 ],
                     [  2.007797424 ,  0.000000000 ,  0.000000000 ],
                     [  3.015114828 ,  0.005056388 ,  0.000000000 ],
                     [  3.796769012 ,  1.132604937 ,  0.000000000 ],
                     [  5.125653739 ,  0.772354616 ,  0.000000000 ],
                     [  5.167047225 ,  -0.653193161 ,  0.000000000 ],
                     [  3.817202589 ,  -1.104920876 ,  0.000000000 ],
                     [  3.482542920 ,  -2.462094972 ,  0.000000000 ],
                     [  4.524735226 ,  -3.376178892 ,  0.000000000 ],
                     [  5.869058665 ,  -2.951641292 ,  0.000000000 ],
                     [  6.199398544 ,  -1.606705567 ,  0.000000000 ],
                     [  3.343074787 ,  2.109594763 ,  0.000000000 ],
                     [  5.961043541 ,  1.451489921 ,  0.000000000 ],
                     [  2.450153978 ,  -2.785730808 ,  0.000000000 ],
                     [  4.303017780 ,  -4.434822780 ,  0.000000000 ],
                     [  6.655123584 ,  -3.694570139 ,  0.000000000 ],
                     [  7.235724321 ,  -1.294593877 ,  0.000000000 ]],
    'positions 1.0':[[  -0.052652077000000 ,  -1.393225783000000 ,  0.000000000000000 ],
                     [  -0.025543347000000 ,  -0.696940104000000 ,  -1.208292000000000 ],
                     [  0.026348254000000 ,  0.696724226000000 ,  -1.208365000000000 ],
                     [  0.051042263000000 ,  1.393657541000000 ,  0.000000000000000 ],
                     [  0.026348254000000 ,  0.696724226000000 ,  1.208365000000000 ],
                     [  -0.025543347000000 ,  -0.696940104000000 ,  1.208292000000000 ],
                     [  -0.097430661000000 ,  -2.473655966000000 ,  0.000000000000000 ],
                     [  -0.040509756000000 ,  -1.237360068000000 ,  -2.144590000000000 ],
                     [  0.050955575000000 ,  1.236531293000000 ,  -2.144838000000000 ],
                     [  0.089657645000000 ,  2.474412421000000 ,  0.000000000000000 ],
                     [  0.050955575000000 ,  1.236531293000000 ,  2.144838000000000 ],
                     [  -0.040509756000000 ,  -1.237360068000000 ,  2.144590000000000 ],
                     [  2.230886026727273 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  3.238203430727273 ,  0.005056388000000 ,  0.000000000000000 ],
                     [  4.019857614727273 ,  1.132604937000000 ,  0.000000000000000 ],
                     [  5.348742341727273 ,  0.772354616000000 ,  0.000000000000000 ],
                     [  5.390135827727273 ,  -0.653193161000000 ,  0.000000000000000 ],
                     [  4.040291191727273 ,  -1.104920876000000 ,  0.000000000000000 ],
                     [  3.705631522727273 ,  -2.462094972000000 ,  0.000000000000000 ],
                     [  4.747823828727273 ,  -3.376178892000000 ,  0.000000000000000 ],
                     [  6.092147267727273 ,  -2.951641292000000 ,  0.000000000000000 ],
                     [  6.422487146727273 ,  -1.606705567000000 ,  0.000000000000000 ],
                     [  3.566163389727273 ,  2.109594763000000 ,  0.000000000000000 ],
                     [  6.184132143727273 ,  1.451489921000000 ,  0.000000000000000 ],
                     [  2.673242580727273 ,  -2.785730808000000 ,  0.000000000000000 ],
                     [  4.526106382727273 ,  -4.434822780000000 ,  0.000000000000000 ],
                     [  6.878212186727272 ,  -3.694570139000000 ,  0.000000000000000 ],
                     [  7.458812923727273 ,  -1.294593877000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  -0.052652077 ,  -1.393225783 ,  0.000000000 ],
                     [  -0.025543347 ,  -0.696940104 ,  -1.208292000 ],
                     [  0.026348254 ,  0.696724226 ,  -1.208365000 ],
                     [  0.051042263 ,  1.393657541 ,  0.000000000 ],
                     [  0.026348254 ,  0.696724226 ,  1.208365000 ],
                     [  -0.025543347 ,  -0.696940104 ,  1.208292000 ],
                     [  -0.097430661 ,  -2.473655966 ,  0.000000000 ],
                     [  -0.040509756 ,  -1.237360068 ,  -2.144590000 ],
                     [  0.050955575 ,  1.236531293 ,  -2.144838000 ],
                     [  0.089657645 ,  2.474412421 ,  0.000000000 ],
                     [  0.050955575 ,  1.236531293 ,  2.144838000 ],
                     [  -0.040509756 ,  -1.237360068 ,  2.144590000 ],
                     [  2.677063232 ,  0.000000000 ,  0.000000000 ],
                     [  3.684380636 ,  0.005056388 ,  0.000000000 ],
                     [  4.466034820 ,  1.132604937 ,  0.000000000 ],
                     [  5.794919547 ,  0.772354616 ,  0.000000000 ],
                     [  5.836313033 ,  -0.653193161 ,  0.000000000 ],
                     [  4.486468397 ,  -1.104920876 ,  0.000000000 ],
                     [  4.151808728 ,  -2.462094972 ,  0.000000000 ],
                     [  5.194001034 ,  -3.376178892 ,  0.000000000 ],
                     [  6.538324473 ,  -2.951641292 ,  0.000000000 ],
                     [  6.868664352 ,  -1.606705567 ,  0.000000000 ],
                     [  4.012340595 ,  2.109594763 ,  0.000000000 ],
                     [  6.630309349 ,  1.451489921 ,  0.000000000 ],
                     [  3.119419786 ,  -2.785730808 ,  0.000000000 ],
                     [  4.972283588 ,  -4.434822780 ,  0.000000000 ],
                     [  7.324389392 ,  -3.694570139 ,  0.000000000 ],
                     [  7.904990129 ,  -1.294593877 ,  0.000000000 ]],
    'positions 1.5':[[  -0.052652077 ,  -1.393225783 ,  0.000000000 ],
                     [  -0.025543347 ,  -0.696940104 ,  -1.208292000 ],
                     [  0.026348254 ,  0.696724226 ,  -1.208365000 ],
                     [  0.051042263 ,  1.393657541 ,  0.000000000 ],
                     [  0.026348254 ,  0.696724226 ,  1.208365000 ],
                     [  -0.025543347 ,  -0.696940104 ,  1.208292000 ],
                     [  -0.097430661 ,  -2.473655966 ,  0.000000000 ],
                     [  -0.040509756 ,  -1.237360068 ,  -2.144590000 ],
                     [  0.050955575 ,  1.236531293 ,  -2.144838000 ],
                     [  0.089657645 ,  2.474412421 ,  0.000000000 ],
                     [  0.050955575 ,  1.236531293 ,  2.144838000 ],
                     [  -0.040509756 ,  -1.237360068 ,  2.144590000 ],
                     [  3.346329040 ,  0.000000000 ,  0.000000000 ],
                     [  4.353646444 ,  0.005056388 ,  0.000000000 ],
                     [  5.135300628 ,  1.132604937 ,  0.000000000 ],
                     [  6.464185355 ,  0.772354616 ,  0.000000000 ],
                     [  6.505578841 ,  -0.653193161 ,  0.000000000 ],
                     [  5.155734205 ,  -1.104920876 ,  0.000000000 ],
                     [  4.821074536 ,  -2.462094972 ,  0.000000000 ],
                     [  5.863266842 ,  -3.376178892 ,  0.000000000 ],
                     [  7.207590281 ,  -2.951641292 ,  0.000000000 ],
                     [  7.537930160 ,  -1.606705567 ,  0.000000000 ],
                     [  4.681606403 ,  2.109594763 ,  0.000000000 ],
                     [  7.299575157 ,  1.451489921 ,  0.000000000 ],
                     [  3.788685594 ,  -2.785730808 ,  0.000000000 ],
                     [  5.641549396 ,  -4.434822780 ,  0.000000000 ],
                     [  7.993655200 ,  -3.694570139 ,  0.000000000 ],
                     [  8.574255937 ,  -1.294593877 ,  0.000000000 ]],
    'positions 2.0':[[  -0.052652077 ,  -1.393225783 ,  0.000000000 ],
                     [  -0.025543347 ,  -0.696940104 ,  -1.208292000 ],
                     [  0.026348254 ,  0.696724226 ,  -1.208365000 ],
                     [  0.051042263 ,  1.393657541 ,  0.000000000 ],
                     [  0.026348254 ,  0.696724226 ,  1.208365000 ],
                     [  -0.025543347 ,  -0.696940104 ,  1.208292000 ],
                     [  -0.097430661 ,  -2.473655966 ,  0.000000000 ],
                     [  -0.040509756 ,  -1.237360068 ,  -2.144590000 ],
                     [  0.050955575 ,  1.236531293 ,  -2.144838000 ],
                     [  0.089657645 ,  2.474412421 ,  0.000000000 ],
                     [  0.050955575 ,  1.236531293 ,  2.144838000 ],
                     [  -0.040509756 ,  -1.237360068 ,  2.144590000 ],
                     [  4.461772054 ,  0.000000000 ,  0.000000000 ],
                     [  5.469089458 ,  0.005056388 ,  0.000000000 ],
                     [  6.250743642 ,  1.132604937 ,  0.000000000 ],
                     [  7.579628369 ,  0.772354616 ,  0.000000000 ],
                     [  7.621021855 ,  -0.653193161 ,  0.000000000 ],
                     [  6.271177219 ,  -1.104920876 ,  0.000000000 ],
                     [  5.936517550 ,  -2.462094972 ,  0.000000000 ],
                     [  6.978709856 ,  -3.376178892 ,  0.000000000 ],
                     [  8.323033295 ,  -2.951641292 ,  0.000000000 ],
                     [  8.653373174 ,  -1.606705567 ,  0.000000000 ],
                     [  5.797049417 ,  2.109594763 ,  0.000000000 ],
                     [  8.415018171 ,  1.451489921 ,  0.000000000 ],
                     [  4.904128608 ,  -2.785730808 ,  0.000000000 ],
                     [  6.756992410 ,  -4.434822780 ,  0.000000000 ],
                     [  9.109098214 ,  -3.694570139 ,  0.000000000 ],
                     [  9.689698951 ,  -1.294593877 ,  0.000000000 ]]},

'Methane_dimer': {
    'description': "Complex, S22, S26, dispersion bonded",
    'name': "Methane_dimer",
    's26_number': "08",
    'interaction energy CC':-0.0230,
    'interaction energies s22x5':[-0.0147,-0.0230,-0.0108,-0.0026,-0.0004],
    'offset': 0.0000,
    'symbols': 'CHHHHCHHHH',
    'magmoms': None,
    'dimer atoms': [5,5],
    # Optimisation level:  CCSD(T)/cc-pVTZ
    'positions':[[  0.000000, -0.000140,  1.859161],
                 [ -0.888551,  0.513060,  1.494685],
                 [  0.888551,  0.513060,  1.494685],
                 [  0.000000, -1.026339,  1.494868],
                 [  0.000000,  0.000089,  2.948284],
                 [  0.000000,  0.000140, -1.859161],
                 [  0.000000, -0.000089, -2.948284],
                 [ -0.888551, -0.513060, -1.494685],
                 [  0.888551, -0.513060, -1.494685],
                 [  0.000000,  1.026339, -1.494868]],
    'positions 0.9':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  0.364514644 ,  0.513239461 ,  -0.888512354 ],
                     [  0.364514644 ,  0.513105641 ,  0.888589641 ],
                     [  0.364215723 ,  -1.026226426 ,  -0.000077278 ],
                     [  -1.089122980 ,  0.000311014 ,  0.000000023 ],
                     [  3.346489810 ,  0.000000000 ,  0.000000000 ],
                     [  4.435612789 ,  -0.000311014 ,  -0.000000023 ],
                     [  2.981975165 ,  -0.513105641 ,  -0.888589641 ],
                     [  2.981975165 ,  -0.513239461 ,  0.888512354 ],
                     [  2.982274086 ,  1.026226426 ,  0.000077278 ]],
    'positions 1.0':[[  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  0.364514644000000 ,  0.513239461000000 ,  -0.888512354000000 ],
                     [  0.364514644000000 ,  0.513105641000000 ,  0.888589641000000 ],
                     [  0.364215723000000 ,  -1.026226426000000 ,  -0.000077278000000 ],
                     [  -1.089122980000000 ,  0.000311014000000 ,  0.000000023000000 ],
                     [  3.718322011090909 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  4.807444990090909 ,  -0.000311014000000 ,  -0.000000023000000 ],
                     [  3.353807366090909 ,  -0.513105641000000 ,  -0.888589641000000 ],
                     [  3.353807366090909 ,  -0.513239461000000 ,  0.888512354000000 ],
                     [  3.354106287090909 ,  1.026226426000000 ,  0.000077278000000 ]],
    'positions 1.2':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  0.364514644 ,  0.513239461 ,  -0.888512354 ],
                     [  0.364514644 ,  0.513105641 ,  0.888589641 ],
                     [  0.364215723 ,  -1.026226426 ,  -0.000077278 ],
                     [  -1.089122980 ,  0.000311014 ,  0.000000023 ],
                     [  4.461986413 ,  0.000000000 ,  0.000000000 ],
                     [  5.551109392 ,  -0.000311014 ,  -0.000000023 ],
                     [  4.097471768 ,  -0.513105641 ,  -0.888589641 ],
                     [  4.097471768 ,  -0.513239461 ,  0.888512354 ],
                     [  4.097770689 ,  1.026226426 ,  0.000077278 ]],
    'positions 1.5':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  0.364514644 ,  0.513239461 ,  -0.888512354 ],
                     [  0.364514644 ,  0.513105641 ,  0.888589641 ],
                     [  0.364215723 ,  -1.026226426 ,  -0.000077278 ],
                     [  -1.089122980 ,  0.000311014 ,  0.000000023 ],
                     [  5.577483016 ,  0.000000000 ,  0.000000000 ],
                     [  6.666605995 ,  -0.000311014 ,  -0.000000023 ],
                     [  5.212968371 ,  -0.513105641 ,  -0.888589641 ],
                     [  5.212968371 ,  -0.513239461 ,  0.888512354 ],
                     [  5.213267292 ,  1.026226426 ,  0.000077278 ]],
    'positions 2.0':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  0.364514644 ,  0.513239461 ,  -0.888512354 ],
                     [  0.364514644 ,  0.513105641 ,  0.888589641 ],
                     [  0.364215723 ,  -1.026226426 ,  -0.000077278 ],
                     [  -1.089122980 ,  0.000311014 ,  0.000000023 ],
                     [  7.436644022 ,  0.000000000 ,  0.000000000 ],
                     [  8.525767001 ,  -0.000311014 ,  -0.000000023 ],
                     [  7.072129377 ,  -0.513105641 ,  -0.888589641 ],
                     [  7.072129377 ,  -0.513239461 ,  0.888512354 ],
                     [  7.072428298 ,  1.026226426 ,  0.000077278 ]]},

'Phenol_dimer': {
    'description': "Complex, S22, S26",
    'name': "Phenol_dimer",
    's26_number': "22",
    'interaction energy CC':-0.3075,
    'interaction energies s22x5':[-0.2784,-0.3057,-0.2511,-0.1479,-0.0598],
    'offset': 0.0018,
    'symbols': 'COHCCCCCHHHHHOCHCCCCCHHHHH',
    'magmoms': None,
    'dimer atoms': [13,13],
    # Optimisation level: MP2/cc-pVTZ
    'positions':[[ -2.0071056,  0.7638459, -0.1083509],
                 [ -1.3885044,  1.9298523, -0.4431206],
                 [ -0.5238121,  1.9646519, -0.0064609],
                 [ -1.4630807, -0.1519120,  0.7949930],
                 [ -2.1475789, -1.3295094,  1.0883677],
                 [ -3.3743208, -1.6031427,  0.4895864],
                 [ -3.9143727, -0.6838545, -0.4091028],
                 [ -3.2370496,  0.4929609, -0.7096126],
                 [ -0.5106510,  0.0566569,  1.2642563],
                 [ -1.7151135, -2.0321452,  1.7878417],
                 [ -3.9024664, -2.5173865,  0.7197947],
                 [ -4.8670730, -0.8822939, -0.8811319],
                 [ -3.6431662,  1.2134345, -1.4057590],
                 [  1.3531168,  1.9382724,  0.4723133],
                 [  2.0369747,  0.7865043,  0.1495491],
                 [  1.7842846,  2.3487495,  1.2297110],
                 [  1.5904026,  0.0696860, -0.9574153],
                 [  2.2417367, -1.1069765, -1.3128110],
                 [  3.3315674, -1.5665603, -0.5748636],
                 [  3.7696838, -0.8396901,  0.5286439],
                 [  3.1224836,  0.3383498,  0.8960491],
                 [  0.7445512,  0.4367983, -1.5218583],
                 [  1.8921463, -1.6649726, -2.1701843],
                 [  3.8330227, -2.4811537, -0.8566666],
                 [  4.6137632, -1.1850101,  1.1092635],
                 [  3.4598854,  0.9030376,  1.7569489]],
    'positions 0.9':[[  -1.445967355 ,  -1.221065858 ,  0.265808750 ],
                     [  -0.945229913 ,  -0.047318091 ,  -0.209467563 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.683142700 ,  -2.127785201 ,  1.005109011 ],
                     [  -1.257798399 ,  -3.314090975 ,  1.456540663 ],
                     [  -2.590627730 ,  -3.605427919 ,  1.179051667 ],
                     [  -3.348500619 ,  -2.695116849 ,  0.443286115 ],
                     [  -2.782549405 ,  -1.509701903 ,  -0.013287247 ],
                     [  0.352786431 ,  -1.905463972 ,  1.224781047 ],
                     [  -0.656349187 ,  -4.009576034 ,  2.026231320 ],
                     [  -3.032993188 ,  -4.526384329 ,  1.531085059 ],
                     [  -4.385512900 ,  -2.907317436 ,  0.221017935 ],
                     [  -3.357888956 ,  -0.796017014 ,  -0.586234960 ],
                     [  1.743489077 ,  0.000000000 ,  0.000000000 ],
                     [  2.341981491 ,  -1.142898789 ,  -0.483732445 ],
                     [  2.342838533 ,  0.417604441 ,  0.628041164 ],
                     [  1.645485086 ,  -1.867622674 ,  -1.447211527 ],
                     [  2.204739700 ,  -3.035912794 ,  -1.954567993 ],
                     [  3.449296078 ,  -3.479350313 ,  -1.509647408 ],
                     [  4.136609561 ,  -2.744696418 ,  -0.547410307 ],
                     [  3.584309534 ,  -1.574952605 ,  -0.029436748 ],
                     [  0.681454799 ,  -1.513028491 ,  -1.784467064 ],
                     [  1.661729182 ,  -3.600082357 ,  -2.699896207 ],
                     [  3.877956013 ,  -4.387511286 ,  -1.908204233 ],
                     [  5.102623102 ,  -3.077497147 ,  -0.194005162 ],
                     [  4.116289930 ,  -1.004251641 ,  0.722333197 ]],
    'positions 1.0':[[  -1.445967355000000 ,  -1.221065858000000 ,  0.265808750000000 ],
                     [  -0.945229913000000 ,  -0.047318091000000 ,  -0.209467563000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -0.683142700000000 ,  -2.127785201000000 ,  1.005109011000000 ],
                     [  -1.257798399000000 ,  -3.314090975000000 ,  1.456540663000000 ],
                     [  -2.590627730000000 ,  -3.605427919000000 ,  1.179051667000000 ],
                     [  -3.348500619000000 ,  -2.695116849000000 ,  0.443286115000000 ],
                     [  -2.782549405000000 ,  -1.509701903000000 ,  -0.013287247000000 ],
                     [  0.352786431000000 ,  -1.905463972000000 ,  1.224781047000000 ],
                     [  -0.656349187000000 ,  -4.009576034000000 ,  2.026231320000000 ],
                     [  -3.032993188000000 ,  -4.526384329000000 ,  1.531085059000000 ],
                     [  -4.385512900000000 ,  -2.907317436000000 ,  0.221017935000000 ],
                     [  -3.357888956000000 ,  -0.796017014000000 ,  -0.586234960000000 ],
                     [  1.937210085636364 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  2.535702499636364 ,  -1.142898789000000 ,  -0.483732445000000 ],
                     [  2.536559541636364 ,  0.417604441000000 ,  0.628041164000000 ],
                     [  1.839206094636364 ,  -1.867622674000000 ,  -1.447211527000000 ],
                     [  2.398460708636364 ,  -3.035912794000000 ,  -1.954567993000000 ],
                     [  3.643017086636364 ,  -3.479350313000000 ,  -1.509647408000000 ],
                     [  4.330330569636364 ,  -2.744696418000000 ,  -0.547410307000000 ],
                     [  3.778030542636364 ,  -1.574952605000000 ,  -0.029436748000000 ],
                     [  0.875175807636364 ,  -1.513028491000000 ,  -1.784467064000000 ],
                     [  1.855450190636364 ,  -3.600082357000000 ,  -2.699896207000000 ],
                     [  4.071677021636363 ,  -4.387511286000000 ,  -1.908204233000000 ],
                     [  5.296344110636364 ,  -3.077497147000000 ,  -0.194005162000000 ],
                     [  4.310010938636363 ,  -1.004251641000000 ,  0.722333197000000 ]],
    'positions 1.2':[[  -1.445967355 ,  -1.221065858 ,  0.265808750 ],
                     [  -0.945229913 ,  -0.047318091 ,  -0.209467563 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.683142700 ,  -2.127785201 ,  1.005109011 ],
                     [  -1.257798399 ,  -3.314090975 ,  1.456540663 ],
                     [  -2.590627730 ,  -3.605427919 ,  1.179051667 ],
                     [  -3.348500619 ,  -2.695116849 ,  0.443286115 ],
                     [  -2.782549405 ,  -1.509701903 ,  -0.013287247 ],
                     [  0.352786431 ,  -1.905463972 ,  1.224781047 ],
                     [  -0.656349187 ,  -4.009576034 ,  2.026231320 ],
                     [  -3.032993188 ,  -4.526384329 ,  1.531085059 ],
                     [  -4.385512900 ,  -2.907317436 ,  0.221017935 ],
                     [  -3.357888956 ,  -0.796017014 ,  -0.586234960 ],
                     [  2.324652103 ,  0.000000000 ,  0.000000000 ],
                     [  2.923144517 ,  -1.142898789 ,  -0.483732445 ],
                     [  2.924001559 ,  0.417604441 ,  0.628041164 ],
                     [  2.226648112 ,  -1.867622674 ,  -1.447211527 ],
                     [  2.785902726 ,  -3.035912794 ,  -1.954567993 ],
                     [  4.030459104 ,  -3.479350313 ,  -1.509647408 ],
                     [  4.717772587 ,  -2.744696418 ,  -0.547410307 ],
                     [  4.165472560 ,  -1.574952605 ,  -0.029436748 ],
                     [  1.262617825 ,  -1.513028491 ,  -1.784467064 ],
                     [  2.242892208 ,  -3.600082357 ,  -2.699896207 ],
                     [  4.459119039 ,  -4.387511286 ,  -1.908204233 ],
                     [  5.683786128 ,  -3.077497147 ,  -0.194005162 ],
                     [  4.697452956 ,  -1.004251641 ,  0.722333197 ]],
    'positions 1.5':[[  -1.445967355 ,  -1.221065858 ,  0.265808750 ],
                     [  -0.945229913 ,  -0.047318091 ,  -0.209467563 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.683142700 ,  -2.127785201 ,  1.005109011 ],
                     [  -1.257798399 ,  -3.314090975 ,  1.456540663 ],
                     [  -2.590627730 ,  -3.605427919 ,  1.179051667 ],
                     [  -3.348500619 ,  -2.695116849 ,  0.443286115 ],
                     [  -2.782549405 ,  -1.509701903 ,  -0.013287247 ],
                     [  0.352786431 ,  -1.905463972 ,  1.224781047 ],
                     [  -0.656349187 ,  -4.009576034 ,  2.026231320 ],
                     [  -3.032993188 ,  -4.526384329 ,  1.531085059 ],
                     [  -4.385512900 ,  -2.907317436 ,  0.221017935 ],
                     [  -3.357888956 ,  -0.796017014 ,  -0.586234960 ],
                     [  2.905815129 ,  0.000000000 ,  0.000000000 ],
                     [  3.504307543 ,  -1.142898789 ,  -0.483732445 ],
                     [  3.505164585 ,  0.417604441 ,  0.628041164 ],
                     [  2.807811138 ,  -1.867622674 ,  -1.447211527 ],
                     [  3.367065752 ,  -3.035912794 ,  -1.954567993 ],
                     [  4.611622130 ,  -3.479350313 ,  -1.509647408 ],
                     [  5.298935613 ,  -2.744696418 ,  -0.547410307 ],
                     [  4.746635586 ,  -1.574952605 ,  -0.029436748 ],
                     [  1.843780851 ,  -1.513028491 ,  -1.784467064 ],
                     [  2.824055234 ,  -3.600082357 ,  -2.699896207 ],
                     [  5.040282065 ,  -4.387511286 ,  -1.908204233 ],
                     [  6.264949154 ,  -3.077497147 ,  -0.194005162 ],
                     [  5.278615982 ,  -1.004251641 ,  0.722333197 ]],
    'positions 2.0':[[  -1.445967355 ,  -1.221065858 ,  0.265808750 ],
                     [  -0.945229913 ,  -0.047318091 ,  -0.209467563 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.683142700 ,  -2.127785201 ,  1.005109011 ],
                     [  -1.257798399 ,  -3.314090975 ,  1.456540663 ],
                     [  -2.590627730 ,  -3.605427919 ,  1.179051667 ],
                     [  -3.348500619 ,  -2.695116849 ,  0.443286115 ],
                     [  -2.782549405 ,  -1.509701903 ,  -0.013287247 ],
                     [  0.352786431 ,  -1.905463972 ,  1.224781047 ],
                     [  -0.656349187 ,  -4.009576034 ,  2.026231320 ],
                     [  -3.032993188 ,  -4.526384329 ,  1.531085059 ],
                     [  -4.385512900 ,  -2.907317436 ,  0.221017935 ],
                     [  -3.357888956 ,  -0.796017014 ,  -0.586234960 ],
                     [  3.874420172 ,  0.000000000 ,  0.000000000 ],
                     [  4.472912586 ,  -1.142898789 ,  -0.483732445 ],
                     [  4.473769628 ,  0.417604441 ,  0.628041164 ],
                     [  3.776416181 ,  -1.867622674 ,  -1.447211527 ],
                     [  4.335670795 ,  -3.035912794 ,  -1.954567993 ],
                     [  5.580227173 ,  -3.479350313 ,  -1.509647408 ],
                     [  6.267540656 ,  -2.744696418 ,  -0.547410307 ],
                     [  5.715240629 ,  -1.574952605 ,  -0.029436748 ],
                     [  2.812385894 ,  -1.513028491 ,  -1.784467064 ],
                     [  3.792660277 ,  -3.600082357 ,  -2.699896207 ],
                     [  6.008887108 ,  -4.387511286 ,  -1.908204233 ],
                     [  7.233554197 ,  -3.077497147 ,  -0.194005162 ],
                     [  6.247221025 ,  -1.004251641 ,  0.722333197 ]]},

'Pyrazine_dimer': {
    'description': "Complex, S22, S26, dispersion bonded",
    'name': "Pyrazine_dimer",
    's26_number': "12",
    'interaction energy CC':-0.1821,
    'interaction energies s22x5':[-0.0733,-0.1956,-0.1310,-0.0425,-0.0082],
    'offset': -0.0135,
    'symbols': 'CCNCCNHHHHCCNCCNHHHH',
    'magmoms': None,
    'dimer atoms': [10,10],
    # Optimisation level: MP2/cc-pVTZ
    'positions':[[ -1.2471894, -1.1718212, -0.6961388],
                 [ -1.2471894, -1.1718212,  0.6961388],
                 [ -0.2589510, -1.7235771,  1.4144796],
                 [  0.7315327, -2.2652221,  0.6967288],
                 [  0.7315327, -2.2652221, -0.6967288],
                 [ -0.2589510, -1.7235771, -1.4144796],
                 [ -2.0634363, -0.7223199, -1.2472797],
                 [ -2.0634363, -0.7223199,  1.2472797],
                 [  1.5488004, -2.7128282,  1.2475604],
                 [  1.5488004, -2.7128282, -1.2475604],
                 [ -0.3380031,  2.0800608,  1.1300452],
                 [  0.8540254,  1.3593471,  1.1306308],
                 [  1.4701787,  0.9907598,  0.0000000],
                 [  0.8540254,  1.3593471, -1.1306308],
                 [ -0.3380031,  2.0800608, -1.1300452],
                 [ -0.9523059,  2.4528836,  0.0000000],
                 [ -0.8103758,  2.3643033,  2.0618643],
                 [  1.3208583,  1.0670610,  2.0623986],
                 [  1.3208583,  1.0670610, -2.0623986],
                 [ -0.8103758,  2.3643033, -2.0618643]],
    'positions 0.9':[[  0.395653045 ,  1.059432142 ,  -0.696139000 ],
                     [  0.395653045 ,  1.059432142 ,  0.696139000 ],
                     [  -0.003263357 ,  0.000227377 ,  1.414480000 ],
                     [  -0.391847355 ,  -1.059697307 ,  0.696729000 ],
                     [  -0.391847355 ,  -1.059697307 ,  -0.696729000 ],
                     [  -0.003263357 ,  0.000227377 ,  -1.414480000 ],
                     [  0.718983381 ,  1.933370245 ,  -1.247280000 ],
                     [  0.718983381 ,  1.933370245 ,  1.247280000 ],
                     [  -0.713152254 ,  -1.934362753 ,  1.247560000 ],
                     [  -0.713152254 ,  -1.934362753 ,  -1.247560000 ],
                     [  3.398538200 ,  0.643131999 ,  1.130045000 ],
                     [  2.862793235 ,  -0.642689433 ,  1.130631000 ],
                     [  2.589772167 ,  -1.306738847 ,  0.000000000 ],
                     [  2.862793235 ,  -0.642689433 ,  -1.130631000 ],
                     [  3.398538200 ,  0.643131999 ,  -1.130045000 ],
                     [  3.676023139 ,  1.305979850 ,  0.000000000 ],
                     [  3.609496345 ,  1.152471205 ,  2.061864000 ],
                     [  2.643057716 ,  -1.147744338 ,  2.062399000 ],
                     [  2.643057716 ,  -1.147744338 ,  -2.062399000 ],
                     [  3.609496345 ,  1.152471205 ,  -2.061864000 ]],
    'positions 1.0':[[  0.395653045000000 ,  1.059432142000000 ,  -0.696139000000000 ],
                     [  0.395653045000000 ,  1.059432142000000 ,  0.696139000000000 ],
                     [  -0.003263357000000 ,  0.000227377000000 ,  1.414480000000000 ],
                     [  -0.391847355000000 ,  -1.059697307000000 ,  0.696729000000000 ],
                     [  -0.391847355000000 ,  -1.059697307000000 ,  -0.696729000000000 ],
                     [  -0.003263357000000 ,  0.000227377000000 ,  -1.414480000000000 ],
                     [  0.718983381000000 ,  1.933370245000000 ,  -1.247280000000000 ],
                     [  0.718983381000000 ,  1.933370245000000 ,  1.247280000000000 ],
                     [  -0.713152254000000 ,  -1.934362753000000 ,  1.247560000000000 ],
                     [  -0.713152254000000 ,  -1.934362753000000 ,  -1.247560000000000 ],
                     [  3.746481288363636 ,  0.643131999000000 ,  1.130045000000000 ],
                     [  3.210736323363636 ,  -0.642689433000000 ,  1.130631000000000 ],
                     [  2.937715255363636 ,  -1.306738847000000 ,  0.000000000000000 ],
                     [  3.210736323363636 ,  -0.642689433000000 ,  -1.130631000000000 ],
                     [  3.746481288363636 ,  0.643131999000000 ,  -1.130045000000000 ],
                     [  4.023966227363637 ,  1.305979850000000 ,  0.000000000000000 ],
                     [  3.957439433363636 ,  1.152471205000000 ,  2.061864000000000 ],
                     [  2.991000804363636 ,  -1.147744338000000 ,  2.062399000000000 ],
                     [  2.991000804363636 ,  -1.147744338000000 ,  -2.062399000000000 ],
                     [  3.957439433363636 ,  1.152471205000000 ,  -2.061864000000000 ]],
    'positions 1.2':[[  0.395653045 ,  1.059432142 ,  -0.696139000 ],
                     [  0.395653045 ,  1.059432142 ,  0.696139000 ],
                     [  -0.003263357 ,  0.000227377 ,  1.414480000 ],
                     [  -0.391847355 ,  -1.059697307 ,  0.696729000 ],
                     [  -0.391847355 ,  -1.059697307 ,  -0.696729000 ],
                     [  -0.003263357 ,  0.000227377 ,  -1.414480000 ],
                     [  0.718983381 ,  1.933370245 ,  -1.247280000 ],
                     [  0.718983381 ,  1.933370245 ,  1.247280000 ],
                     [  -0.713152254 ,  -1.934362753 ,  1.247560000 ],
                     [  -0.713152254 ,  -1.934362753 ,  -1.247560000 ],
                     [  4.442367465 ,  0.643131999 ,  1.130045000 ],
                     [  3.906622500 ,  -0.642689433 ,  1.130631000 ],
                     [  3.633601432 ,  -1.306738847 ,  0.000000000 ],
                     [  3.906622500 ,  -0.642689433 ,  -1.130631000 ],
                     [  4.442367465 ,  0.643131999 ,  -1.130045000 ],
                     [  4.719852404 ,  1.305979850 ,  0.000000000 ],
                     [  4.653325610 ,  1.152471205 ,  2.061864000 ],
                     [  3.686886981 ,  -1.147744338 ,  2.062399000 ],
                     [  3.686886981 ,  -1.147744338 ,  -2.062399000 ],
                     [  4.653325610 ,  1.152471205 ,  -2.061864000 ]],
    'positions 1.5':[[  0.395653045 ,  1.059432142 ,  -0.696139000 ],
                     [  0.395653045 ,  1.059432142 ,  0.696139000 ],
                     [  -0.003263357 ,  0.000227377 ,  1.414480000 ],
                     [  -0.391847355 ,  -1.059697307 ,  0.696729000 ],
                     [  -0.391847355 ,  -1.059697307 ,  -0.696729000 ],
                     [  -0.003263357 ,  0.000227377 ,  -1.414480000 ],
                     [  0.718983381 ,  1.933370245 ,  -1.247280000 ],
                     [  0.718983381 ,  1.933370245 ,  1.247280000 ],
                     [  -0.713152254 ,  -1.934362753 ,  1.247560000 ],
                     [  -0.713152254 ,  -1.934362753 ,  -1.247560000 ],
                     [  5.486196730 ,  0.643131999 ,  1.130045000 ],
                     [  4.950451765 ,  -0.642689433 ,  1.130631000 ],
                     [  4.677430697 ,  -1.306738847 ,  0.000000000 ],
                     [  4.950451765 ,  -0.642689433 ,  -1.130631000 ],
                     [  5.486196730 ,  0.643131999 ,  -1.130045000 ],
                     [  5.763681669 ,  1.305979850 ,  0.000000000 ],
                     [  5.697154875 ,  1.152471205 ,  2.061864000 ],
                     [  4.730716246 ,  -1.147744338 ,  2.062399000 ],
                     [  4.730716246 ,  -1.147744338 ,  -2.062399000 ],
                     [  5.697154875 ,  1.152471205 ,  -2.061864000 ]],
    'positions 2.0':[[  0.395653045 ,  1.059432142 ,  -0.696139000 ],
                     [  0.395653045 ,  1.059432142 ,  0.696139000 ],
                     [  -0.003263357 ,  0.000227377 ,  1.414480000 ],
                     [  -0.391847355 ,  -1.059697307 ,  0.696729000 ],
                     [  -0.391847355 ,  -1.059697307 ,  -0.696729000 ],
                     [  -0.003263357 ,  0.000227377 ,  -1.414480000 ],
                     [  0.718983381 ,  1.933370245 ,  -1.247280000 ],
                     [  0.718983381 ,  1.933370245 ,  1.247280000 ],
                     [  -0.713152254 ,  -1.934362753 ,  1.247560000 ],
                     [  -0.713152254 ,  -1.934362753 ,  -1.247560000 ],
                     [  7.225912172 ,  0.643131999 ,  1.130045000 ],
                     [  6.690167207 ,  -0.642689433 ,  1.130631000 ],
                     [  6.417146139 ,  -1.306738847 ,  0.000000000 ],
                     [  6.690167207 ,  -0.642689433 ,  -1.130631000 ],
                     [  7.225912172 ,  0.643131999 ,  -1.130045000 ],
                     [  7.503397111 ,  1.305979850 ,  0.000000000 ],
                     [  7.436870317 ,  1.152471205 ,  2.061864000 ],
                     [  6.470431688 ,  -1.147744338 ,  2.062399000 ],
                     [  6.470431688 ,  -1.147744338 ,  -2.062399000 ],
                     [  7.436870317 ,  1.152471205 ,  -2.061864000 ]]},

'Uracil_dimer_h-bonded': {
    'description': "Complex, S22, S26, 2 h-bonds, double h-bond, nucleic base",
    'name': "Uracil_dimer_h-bonded",
    's26_number': "05",
    'interaction energy CC':-0.8972,
    'interaction energies s22x5':[-0.8122,-0.8872,-0.7441,-0.4536,-0.1986],
    'offset': 0.0100,
    'symbols': 'OCNCCCNOHHHHOCNCCCNOHHHH',
    'magmoms': None,
    'dimer atoms': [12,12],
    # Optimisation level: MP2/cc-pVTZ
    'positions':[[ -1.4663316,  1.0121693,  0.0000000],
                 [ -0.6281464,  1.9142678,  0.0000000],
                 [  0.7205093,  1.6882688,  0.0000000],
                 [  1.6367290,  2.7052764,  0.0000000],
                 [  1.2769036,  4.0061763,  0.0000000],
                 [ -0.1286005,  4.3621549,  0.0000000],
                 [ -0.9777230,  3.2396433,  0.0000000],
                 [ -0.5972229,  5.4864066,  0.0000000],
                 [  2.0103504,  4.7938642,  0.0000000],
                 [  1.0232515,  0.7061820,  0.0000000],
                 [ -1.9700268,  3.4323850,  0.0000000],
                 [  2.6690620,  2.3883417,  0.0000000],
                 [  1.4663316, -1.0121693,  0.0000000],
                 [  0.6281464, -1.9142678,  0.0000000],
                 [ -0.7205093, -1.6882688,  0.0000000],
                 [ -1.6367290, -2.7052764,  0.0000000],
                 [ -1.2769036, -4.0061763,  0.0000000],
                 [  0.1286005, -4.3621549,  0.0000000],
                 [  0.9777230, -3.2396433,  0.0000000],
                 [  0.5972229, -5.4864066,  0.0000000],
                 [ -2.0103504, -4.7938642,  0.0000000],
                 [ -1.0232515, -0.7061820,  0.0000000],
                 [  1.9700268, -3.4323850,  0.0000000],
                 [ -2.6690620, -2.3883417,  0.0000000]],
    'positions 0.9':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.664243938 ,  1.036879148 ,  0.000000000 ],
                     [  -0.108663437 ,  2.286389518 ,  0.000000000 ],
                     [  -0.864691937 ,  3.427521953 ,  0.000000000 ],
                     [  -2.214231597 ,  3.403909532 ,  0.000000000 ],
                     [  -2.909869859 ,  2.131803891 ,  0.000000000 ],
                     [  -2.034924624 ,  1.029301194 ,  0.000000000 ],
                     [  -4.115521524 ,  1.958733959 ,  0.000000000 ],
                     [  -2.793840332 ,  4.310799346 ,  0.000000000 ],
                     [  0.917908194 ,  2.334329905 ,  0.000000000 ],
                     [  -2.469325804 ,  0.116551326 ,  0.000000000 ],
                     [  -0.300037631 ,  4.348024043 ,  0.000000000 ],
                     [  2.515009084 ,  2.334329905 ,  0.000000000 ],
                     [  3.179253022 ,  1.297450757 ,  0.000000000 ],
                     [  2.623672521 ,  0.047940387 ,  0.000000000 ],
                     [  3.379701020 ,  -1.093192048 ,  0.000000000 ],
                     [  4.729240680 ,  -1.069579627 ,  0.000000000 ],
                     [  5.424878943 ,  0.202526014 ,  0.000000000 ],
                     [  4.549933708 ,  1.305028711 ,  0.000000000 ],
                     [  6.630530608 ,  0.375595946 ,  0.000000000 ],
                     [  5.308849416 ,  -1.976469441 ,  0.000000000 ],
                     [  1.597100890 ,  0.000000000 ,  0.000000000 ],
                     [  4.984334888 ,  2.217778579 ,  0.000000000 ],
                     [  2.815046715 ,  -2.013694138 ,  0.000000000 ]],
    'positions 1.0':[[  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  -0.664243938000000 ,  1.036879148000000 ,  0.000000000000000 ],
                     [  -0.108663437000000 ,  2.286389518000000 ,  0.000000000000000 ],
                     [  -0.864691937000000 ,  3.427521953000000 ,  0.000000000000000 ],
                     [  -2.214231597000000 ,  3.403909532000000 ,  0.000000000000000 ],
                     [  -2.909869859000000 ,  2.131803891000000 ,  0.000000000000000 ],
                     [  -2.034924624000000 ,  1.029301194000000 ,  0.000000000000000 ],
                     [  -4.115521524000000 ,  1.958733959000000 ,  0.000000000000000 ],
                     [  -2.793840332000000 ,  4.310799346000000 ,  0.000000000000000 ],
                     [  0.917908194000000 ,  2.334329905000000 ,  0.000000000000000 ],
                     [  -2.469325804000000 ,  0.116551326000000 ,  0.000000000000000 ],
                     [  -0.300037631000000 ,  4.348024043000000 ,  0.000000000000000 ],
                     [  2.692464738545454 ,  2.334329905000000 ,  0.000000000000000 ],
                     [  3.356708676545455 ,  1.297450757000000 ,  0.000000000000000 ],
                     [  2.801128175545454 ,  0.047940387000000 ,  0.000000000000000 ],
                     [  3.557156674545455 ,  -1.093192048000000 ,  0.000000000000000 ],
                     [  4.906696334545455 ,  -1.069579627000000 ,  0.000000000000000 ],
                     [  5.602334597545455 ,  0.202526014000000 ,  0.000000000000000 ],
                     [  4.727389362545455 ,  1.305028711000000 ,  0.000000000000000 ],
                     [  6.807986262545454 ,  0.375595946000000 ,  0.000000000000000 ],
                     [  5.486305070545455 ,  -1.976469441000000 ,  0.000000000000000 ],
                     [  1.774556544545455 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  5.161790542545455 ,  2.217778579000000 ,  0.000000000000000 ],
                     [  2.992502369545454 ,  -2.013694138000000 ,  0.000000000000000 ]],
    'positions 1.2':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.664243938 ,  1.036879148 ,  0.000000000 ],
                     [  -0.108663437 ,  2.286389518 ,  0.000000000 ],
                     [  -0.864691937 ,  3.427521953 ,  0.000000000 ],
                     [  -2.214231597 ,  3.403909532 ,  0.000000000 ],
                     [  -2.909869859 ,  2.131803891 ,  0.000000000 ],
                     [  -2.034924624 ,  1.029301194 ,  0.000000000 ],
                     [  -4.115521524 ,  1.958733959 ,  0.000000000 ],
                     [  -2.793840332 ,  4.310799346 ,  0.000000000 ],
                     [  0.917908194 ,  2.334329905 ,  0.000000000 ],
                     [  -2.469325804 ,  0.116551326 ,  0.000000000 ],
                     [  -0.300037631 ,  4.348024043 ,  0.000000000 ],
                     [  3.047376048 ,  2.334329905 ,  0.000000000 ],
                     [  3.711619986 ,  1.297450757 ,  0.000000000 ],
                     [  3.156039485 ,  0.047940387 ,  0.000000000 ],
                     [  3.912067984 ,  -1.093192048 ,  0.000000000 ],
                     [  5.261607644 ,  -1.069579627 ,  0.000000000 ],
                     [  5.957245907 ,  0.202526014 ,  0.000000000 ],
                     [  5.082300672 ,  1.305028711 ,  0.000000000 ],
                     [  7.162897572 ,  0.375595946 ,  0.000000000 ],
                     [  5.841216380 ,  -1.976469441 ,  0.000000000 ],
                     [  2.129467854 ,  0.000000000 ,  0.000000000 ],
                     [  5.516701852 ,  2.217778579 ,  0.000000000 ],
                     [  3.347413679 ,  -2.013694138 ,  0.000000000 ]],
    'positions 1.5':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.664243938 ,  1.036879148 ,  0.000000000 ],
                     [  -0.108663437 ,  2.286389518 ,  0.000000000 ],
                     [  -0.864691937 ,  3.427521953 ,  0.000000000 ],
                     [  -2.214231597 ,  3.403909532 ,  0.000000000 ],
                     [  -2.909869859 ,  2.131803891 ,  0.000000000 ],
                     [  -2.034924624 ,  1.029301194 ,  0.000000000 ],
                     [  -4.115521524 ,  1.958733959 ,  0.000000000 ],
                     [  -2.793840332 ,  4.310799346 ,  0.000000000 ],
                     [  0.917908194 ,  2.334329905 ,  0.000000000 ],
                     [  -2.469325804 ,  0.116551326 ,  0.000000000 ],
                     [  -0.300037631 ,  4.348024043 ,  0.000000000 ],
                     [  3.579743012 ,  2.334329905 ,  0.000000000 ],
                     [  4.243986950 ,  1.297450757 ,  0.000000000 ],
                     [  3.688406449 ,  0.047940387 ,  0.000000000 ],
                     [  4.444434948 ,  -1.093192048 ,  0.000000000 ],
                     [  5.793974608 ,  -1.069579627 ,  0.000000000 ],
                     [  6.489612871 ,  0.202526014 ,  0.000000000 ],
                     [  5.614667636 ,  1.305028711 ,  0.000000000 ],
                     [  7.695264536 ,  0.375595946 ,  0.000000000 ],
                     [  6.373583344 ,  -1.976469441 ,  0.000000000 ],
                     [  2.661834818 ,  0.000000000 ,  0.000000000 ],
                     [  6.049068816 ,  2.217778579 ,  0.000000000 ],
                     [  3.879780643 ,  -2.013694138 ,  0.000000000 ]],
    'positions 2.0':[[  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  -0.664243938 ,  1.036879148 ,  0.000000000 ],
                     [  -0.108663437 ,  2.286389518 ,  0.000000000 ],
                     [  -0.864691937 ,  3.427521953 ,  0.000000000 ],
                     [  -2.214231597 ,  3.403909532 ,  0.000000000 ],
                     [  -2.909869859 ,  2.131803891 ,  0.000000000 ],
                     [  -2.034924624 ,  1.029301194 ,  0.000000000 ],
                     [  -4.115521524 ,  1.958733959 ,  0.000000000 ],
                     [  -2.793840332 ,  4.310799346 ,  0.000000000 ],
                     [  0.917908194 ,  2.334329905 ,  0.000000000 ],
                     [  -2.469325804 ,  0.116551326 ,  0.000000000 ],
                     [  -0.300037631 ,  4.348024043 ,  0.000000000 ],
                     [  4.467021284 ,  2.334329905 ,  0.000000000 ],
                     [  5.131265222 ,  1.297450757 ,  0.000000000 ],
                     [  4.575684721 ,  0.047940387 ,  0.000000000 ],
                     [  5.331713220 ,  -1.093192048 ,  0.000000000 ],
                     [  6.681252880 ,  -1.069579627 ,  0.000000000 ],
                     [  7.376891143 ,  0.202526014 ,  0.000000000 ],
                     [  6.501945908 ,  1.305028711 ,  0.000000000 ],
                     [  8.582542808 ,  0.375595946 ,  0.000000000 ],
                     [  7.260861616 ,  -1.976469441 ,  0.000000000 ],
                     [  3.549113090 ,  0.000000000 ,  0.000000000 ],
                     [  6.936347088 ,  2.217778579 ,  0.000000000 ],
                     [  4.767058915 ,  -2.013694138 ,  0.000000000 ]]},

'Uracil_dimer_stack': {
    'description': "Complex, S22, S26, stack, dispersion bonded, nucleic base",
    'name': "Uracil_dimer_stack",
    's26_number': "13",
    'interaction energy CC':-0.4224,
    'interaction energies s22x5':[-0.2931,-0.4280,-0.2715,-0.1049,-0.0299],
    'offset': -0.0056,
    'symbols': 'NCHCHCONHCOHNCHCHCONHCOH',
    'magmoms': None,
    'dimer atoms': [12,12],
    # Optimisation level: MP2/cc-pVTZ
    'positions':[[  2.0113587, -1.2132073, -0.0980673],
                 [  2.0257076, -0.6971797, -1.3644029],
                 [  2.2975208, -1.3910592, -2.1456459],
                 [  1.7145226,  0.5919651, -1.6124892],
                 [  1.7272873,  0.9908466, -2.6120050],
                 [  1.3089605,  1.4575340, -0.5205890],
                 [  0.9205926,  2.6110864, -0.6260457],
                 [  1.3768885,  0.8397454,  0.7346356],
                 [  1.0518040,  1.3862229,  1.5233710],
                 [  1.6459909, -0.4852113,  1.0187267],
                 [  1.5611090, -0.9718061,  2.1298059],
                 [  2.1294635, -2.2015046,  0.0568134],
                 [ -2.0113587,  1.2132073, -0.0980673],
                 [ -2.0257076,  0.6971797, -1.3644029],
                 [ -2.2975208,  1.3910592, -2.1456459],
                 [ -1.7145226, -0.5919651, -1.6124892],
                 [ -1.7272873, -0.9908466, -2.6120050],
                 [ -1.3089605, -1.4575340, -0.5205890],
                 [ -0.9205926, -2.6110864, -0.6260457],
                 [ -1.3768885, -0.8397454,  0.7346356],
                 [ -1.0518040, -1.3862229,  1.5233710],
                 [ -1.6459909,  0.4852113,  1.0187267],
                 [ -1.5611090,  0.9718061,  2.1298059],
                 [ -2.1294635,  2.2015046,  0.0568134]],
    'positions 0.9':[[  -0.277905006 ,  1.293679543 ,  0.176141970 ],
                     [  -0.313143400 ,  0.778657200 ,  -1.090194030 ],
                     [  -0.556628453 ,  1.482976305 ,  -1.871437030 ],
                     [  -0.054429325 ,  -0.522034140 ,  -1.338280030 ],
                     [  -0.083339176 ,  -0.920071815 ,  -2.337796030 ],
                     [  0.315741834 ,  -1.403319766 ,  -0.246380030 ],
                     [  0.657066634 ,  -2.571655559 ,  -0.351837030 ],
                     [  0.272892517 ,  -0.783286382 ,  1.008844970 ],
                     [  0.575575188 ,  -1.342483138 ,  1.797579970 ],
                     [  0.057676398 ,  0.551482081 ,  1.292935970 ],
                     [  0.162197796 ,  1.034239706 ,  2.404014970 ],
                     [  -0.355882042 ,  2.285950208 ,  0.331021970 ],
                     [  3.306699593 ,  -1.293679543 ,  0.176141970 ],
                     [  3.341937987 ,  -0.778657200 ,  -1.090194030 ],
                     [  3.585423040 ,  -1.482976305 ,  -1.871437030 ],
                     [  3.083223911 ,  0.522034140 ,  -1.338280030 ],
                     [  3.112133763 ,  0.920071815 ,  -2.337796030 ],
                     [  2.713052753 ,  1.403319766 ,  -0.246380030 ],
                     [  2.371727953 ,  2.571655559 ,  -0.351837030 ],
                     [  2.755902070 ,  0.783286382 ,  1.008844970 ],
                     [  2.453219399 ,  1.342483138 ,  1.797579970 ],
                     [  2.971118189 ,  -0.551482081 ,  1.292935970 ],
                     [  2.866596791 ,  -1.034239706 ,  2.404014970 ],
                     [  3.384676629 ,  -2.285950208 ,  0.331021970 ]],
    'positions 1.0':[[  -0.277905006000000 ,  1.293679543000000 ,  0.176141970000000 ],
                     [  -0.313143400000000 ,  0.778657200000000 ,  -1.090194030000000 ],
                     [  -0.556628453000000 ,  1.482976305000000 ,  -1.871437030000000 ],
                     [  -0.054429325000000 ,  -0.522034140000000 ,  -1.338280030000000 ],
                     [  -0.083339176000000 ,  -0.920071815000000 ,  -2.337796030000000 ],
                     [  0.315741834000000 ,  -1.403319766000000 ,  -0.246380030000000 ],
                     [  0.657066634000000 ,  -2.571655559000000 ,  -0.351837030000000 ],
                     [  0.272892517000000 ,  -0.783286382000000 ,  1.008844970000000 ],
                     [  0.575575188000000 ,  -1.342483138000000 ,  1.797579970000000 ],
                     [  0.057676398000000 ,  0.551482081000000 ,  1.292935970000000 ],
                     [  0.162197796000000 ,  1.034239706000000 ,  2.404014970000000 ],
                     [  -0.355882042000000 ,  2.285950208000000 ,  0.331021970000000 ],
                     [  3.643232324909091 ,  -1.293679543000000 ,  0.176141970000000 ],
                     [  3.678470718909091 ,  -0.778657200000000 ,  -1.090194030000000 ],
                     [  3.921955771909091 ,  -1.482976305000000 ,  -1.871437030000000 ],
                     [  3.419756642909091 ,  0.522034140000000 ,  -1.338280030000000 ],
                     [  3.448666494909091 ,  0.920071815000000 ,  -2.337796030000000 ],
                     [  3.049585484909091 ,  1.403319766000000 ,  -0.246380030000000 ],
                     [  2.708260684909091 ,  2.571655559000000 ,  -0.351837030000000 ],
                     [  3.092434801909091 ,  0.783286382000000 ,  1.008844970000000 ],
                     [  2.789752130909091 ,  1.342483138000000 ,  1.797579970000000 ],
                     [  3.307650920909091 ,  -0.551482081000000 ,  1.292935970000000 ],
                     [  3.203129522909091 ,  -1.034239706000000 ,  2.404014970000000 ],
                     [  3.721209360909091 ,  -2.285950208000000 ,  0.331021970000000 ]],
    'positions 1.2':[[  -0.277905006 ,  1.293679543 ,  0.176141970 ],
                     [  -0.313143400 ,  0.778657200 ,  -1.090194030 ],
                     [  -0.556628453 ,  1.482976305 ,  -1.871437030 ],
                     [  -0.054429325 ,  -0.522034140 ,  -1.338280030 ],
                     [  -0.083339176 ,  -0.920071815 ,  -2.337796030 ],
                     [  0.315741834 ,  -1.403319766 ,  -0.246380030 ],
                     [  0.657066634 ,  -2.571655559 ,  -0.351837030 ],
                     [  0.272892517 ,  -0.783286382 ,  1.008844970 ],
                     [  0.575575188 ,  -1.342483138 ,  1.797579970 ],
                     [  0.057676398 ,  0.551482081 ,  1.292935970 ],
                     [  0.162197796 ,  1.034239706 ,  2.404014970 ],
                     [  -0.355882042 ,  2.285950208 ,  0.331021970 ],
                     [  4.316297789 ,  -1.293679543 ,  0.176141970 ],
                     [  4.351536183 ,  -0.778657200 ,  -1.090194030 ],
                     [  4.595021236 ,  -1.482976305 ,  -1.871437030 ],
                     [  4.092822107 ,  0.522034140 ,  -1.338280030 ],
                     [  4.121731959 ,  0.920071815 ,  -2.337796030 ],
                     [  3.722650949 ,  1.403319766 ,  -0.246380030 ],
                     [  3.381326149 ,  2.571655559 ,  -0.351837030 ],
                     [  3.765500266 ,  0.783286382 ,  1.008844970 ],
                     [  3.462817595 ,  1.342483138 ,  1.797579970 ],
                     [  3.980716385 ,  -0.551482081 ,  1.292935970 ],
                     [  3.876194987 ,  -1.034239706 ,  2.404014970 ],
                     [  4.394274825 ,  -2.285950208 ,  0.331021970 ]],
    'positions 1.5':[[  -0.277905006 ,  1.293679543 ,  0.176141970 ],
                     [  -0.313143400 ,  0.778657200 ,  -1.090194030 ],
                     [  -0.556628453 ,  1.482976305 ,  -1.871437030 ],
                     [  -0.054429325 ,  -0.522034140 ,  -1.338280030 ],
                     [  -0.083339176 ,  -0.920071815 ,  -2.337796030 ],
                     [  0.315741834 ,  -1.403319766 ,  -0.246380030 ],
                     [  0.657066634 ,  -2.571655559 ,  -0.351837030 ],
                     [  0.272892517 ,  -0.783286382 ,  1.008844970 ],
                     [  0.575575188 ,  -1.342483138 ,  1.797579970 ],
                     [  0.057676398 ,  0.551482081 ,  1.292935970 ],
                     [  0.162197796 ,  1.034239706 ,  2.404014970 ],
                     [  -0.355882042 ,  2.285950208 ,  0.331021970 ],
                     [  5.325895984 ,  -1.293679543 ,  0.176141970 ],
                     [  5.361134378 ,  -0.778657200 ,  -1.090194030 ],
                     [  5.604619431 ,  -1.482976305 ,  -1.871437030 ],
                     [  5.102420302 ,  0.522034140 ,  -1.338280030 ],
                     [  5.131330154 ,  0.920071815 ,  -2.337796030 ],
                     [  4.732249144 ,  1.403319766 ,  -0.246380030 ],
                     [  4.390924344 ,  2.571655559 ,  -0.351837030 ],
                     [  4.775098461 ,  0.783286382 ,  1.008844970 ],
                     [  4.472415790 ,  1.342483138 ,  1.797579970 ],
                     [  4.990314580 ,  -0.551482081 ,  1.292935970 ],
                     [  4.885793182 ,  -1.034239706 ,  2.404014970 ],
                     [  5.403873020 ,  -2.285950208 ,  0.331021970 ]],
    'positions 2.0':[[  -0.277905006 ,  1.293679543 ,  0.176141970 ],
                     [  -0.313143400 ,  0.778657200 ,  -1.090194030 ],
                     [  -0.556628453 ,  1.482976305 ,  -1.871437030 ],
                     [  -0.054429325 ,  -0.522034140 ,  -1.338280030 ],
                     [  -0.083339176 ,  -0.920071815 ,  -2.337796030 ],
                     [  0.315741834 ,  -1.403319766 ,  -0.246380030 ],
                     [  0.657066634 ,  -2.571655559 ,  -0.351837030 ],
                     [  0.272892517 ,  -0.783286382 ,  1.008844970 ],
                     [  0.575575188 ,  -1.342483138 ,  1.797579970 ],
                     [  0.057676398 ,  0.551482081 ,  1.292935970 ],
                     [  0.162197796 ,  1.034239706 ,  2.404014970 ],
                     [  -0.355882042 ,  2.285950208 ,  0.331021970 ],
                     [  7.008559644 ,  -1.293679543 ,  0.176141970 ],
                     [  7.043798038 ,  -0.778657200 ,  -1.090194030 ],
                     [  7.287283091 ,  -1.482976305 ,  -1.871437030 ],
                     [  6.785083962 ,  0.522034140 ,  -1.338280030 ],
                     [  6.813993814 ,  0.920071815 ,  -2.337796030 ],
                     [  6.414912804 ,  1.403319766 ,  -0.246380030 ],
                     [  6.073588004 ,  2.571655559 ,  -0.351837030 ],
                     [  6.457762121 ,  0.783286382 ,  1.008844970 ],
                     [  6.155079450 ,  1.342483138 ,  1.797579970 ],
                     [  6.672978240 ,  -0.551482081 ,  1.292935970 ],
                     [  6.568456842 ,  -1.034239706 ,  2.404014970 ],
                     [  7.086536680 ,  -2.285950208 ,  0.331021970 ]]},

'Water_dimer': {
    'description': "Complex, S22, S26, 1 h-bond, OH-O",
    'name': "Water_dimer",
    's26_number': "02",
    'interaction energy CC':-0.2177,
    'interaction energies s22x5':[-0.1873,-0.2155,-0.1752,-0.0993,-0.0416],
    'offset': 0.0022,
    'symbols': 'OHHOHH',
    'magmoms': None,
    'dimer atoms': [3,3],
    # Optimisation level: CCSD(T)/cc-pVQZ
    'positions':[[ -1.551007, -0.114520,  0.000000],
                 [ -1.934259,  0.762503,  0.000000],
                 [ -0.599677,  0.040712,  0.000000],
                 [  1.350625,  0.111469,  0.000000],
                 [  1.680398, -0.373741, -0.758561],
                 [  1.680398, -0.373741,  0.758561]],
    'positions 0.9':[[  -0.956332646 ,  -0.120638358 ,  0.000000000 ],
                     [  -1.307535174 ,  0.769703274 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  1.756426600 ,  0.000000000 ,  0.000000000 ],
                     [  2.068390928 ,  -0.496847294 ,  -0.758561000 ],
                     [  2.068390928 ,  -0.496847294 ,  0.758561000 ]],
    'positions 1.0':[[  -0.956332646000000 ,  -0.120638358000000 ,  0.000000000000000 ],
                     [  -1.307535174000000 ,  0.769703274000000 ,  0.000000000000000 ],
                     [  0.000000000000000 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  1.951585111090909 ,  0.000000000000000 ,  0.000000000000000 ],
                     [  2.263549439090909 ,  -0.496847294000000 ,  -0.758561000000000 ],
                     [  2.263549439090909 ,  -0.496847294000000 ,  0.758561000000000 ]],
    'positions 1.2':[[  -0.956332646 ,  -0.120638358 ,  0.000000000 ],
                     [  -1.307535174 ,  0.769703274 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  2.341902133 ,  0.000000000 ,  0.000000000 ],
                     [  2.653866461 ,  -0.496847294 ,  -0.758561000 ],
                     [  2.653866461 ,  -0.496847294 ,  0.758561000 ]],
    'positions 1.5':[[  -0.956332646 ,  -0.120638358 ,  0.000000000 ],
                     [  -1.307535174 ,  0.769703274 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  2.927377666 ,  0.000000000 ,  0.000000000 ],
                     [  3.239341994 ,  -0.496847294 ,  -0.758561000 ],
                     [  3.239341994 ,  -0.496847294 ,  0.758561000 ]],
    'positions 2.0':[[  -0.956332646 ,  -0.120638358 ,  0.000000000 ],
                     [  -1.307535174 ,  0.769703274 ,  0.000000000 ],
                     [  0.000000000 ,  0.000000000 ,  0.000000000 ],
                     [  3.903170222 ,  0.000000000 ,  0.000000000 ],
                     [  4.215134550 ,  -0.496847294 ,  -0.758561000 ],
                     [  4.215134550 ,  -0.496847294 ,  0.758561000 ]]},

# --- s26 ---#
'Methanol_dimer': {
    'description': "1 h-bond, OH-O, S26",
    'name': "Methanol_dimer",
    's26_number': "23",
    'interaction energy MP2':-0.1947,
    'interaction energy CC':-0.2472,
    'symbols': 'COHHHHCOHHHH',
    'magmoms': None,
    # Optimisation level: MP2/cc-pVTZ
    'positions':[[ -2.114335, -0.445120,  0.221169],
                 [ -1.298032,  0.687432, -0.091609],
                 [ -1.514720, -1.087407,  0.858397],
                 [ -2.389026, -0.999598, -0.675819],
                 [ -3.014036, -0.146131,  0.758353],
                 [ -1.779011,  1.249219, -0.706289],
                 [  2.245711,  0.159561,  0.329180],
                 [  1.285289, -0.472004, -0.501635],
                 [  3.156806, -0.431037,  0.275178],
                 [  1.921474,  0.200114,  1.371809],
                 [  2.472512,  1.174527, -0.005695],
                 [  0.459691,  0.030236, -0.432082]]},
'Methanol-formaldehyde_complex': {
    'description': "1 h-bond, OH-O, S26",
    's26_number': "24",
    'name': "Methanol-formaldehyde_complex",
    'interaction energy MP2':-0.1375,
    'interaction energy CC':-0.2303,
    'symbols': 'COHHHHCOHH',
    'magmoms': None,
    # Optimisation level: MP2/cc-pVTZ
    'positions':[[  1.4073776162,  1.0401758064,  2.0396751091],
                 [  0.9349167370,  0.2900025037,  0.9338944612],
                 [  2.1022348002,  0.4092302046,  2.5857336738],
                 [  0.6031517696,  1.3305232490,  2.7201012084],
                 [  1.9382206717,  1.9424443037,  1.7274684180],
                 [  0.2386426835,  0.8096239461,  0.5150020113],
                 [ -2.0809868810, -0.1309834084,  0.2601720974],
                 [ -1.6206107677,  0.9480216819, -0.1003790153],
                 [ -3.1316901290, -0.3840062180,  0.0820343467],
                 [ -1.4275985002, -0.8637260692,  0.7543476894]]},
'Methyl_amide_dimer_alpha': {
    'description': "1 h-bond, NH-O, S26",
    's26_number': "25",
    'name': "Methyl_amide_dimer_alpha",
    'interaction energy MP2':-0.2068,
    'interaction energy CC':-0.2901,
    'symbols': 'CCOHHHNHHCCOHHHNHH',
    'magmoms': None,
    # Optimisation level: DFT TPSS/TZVP (hydrogen positions optimized)
    'positions':[[  5.575000,  7.306000, -12.014000],
                 [  4.318000,  8.065000, -12.345000],
                 [  4.212000,  9.236000, -11.986000],
                 [  6.072000,  7.809000, -11.186000],
                 [  6.246000,  7.323000, -12.882000],
                 [  5.392000,  6.256000, -11.755000],
                 [  3.378000,  7.446000, -13.058000],
                 [  3.468000,  6.488000, -13.367000],
                 [  2.561000,  7.968000, -13.350000],
                 [  0.768000,  8.395000, -9.9890000],
                 [  1.666000,  9.133000, -8.9870000],
                 [  1.355000,  9.267000, -7.8060000],
                 [ -0.014000,  9.085000, -10.326000],
                 [  0.289000,  7.561000, -9.4730000],
                 [  1.315000,  8.032000, -10.865000],
                 [  2.798000,  9.666000, -9.4430000],
                 [  3.139000,  9.599000, -10.401000],
                 [  3.350000,  10.195000, -8.779000]]},
'Methyl_amide_dimer_beta': {
    'description': "1 h-bond, NH-O, S26",
    'name': "Methyl_amide_dimer_beta",
    's26_number': "26",
    'interaction energy MP2':-0.2342,
    'interaction energy CC':-0.3317,
    'symbols': 'CCOHHHNHHCCOHHHNHH',
    'magmoms': None,
    # Optimisation level: DFT TPSS/TZVP (hydrogen positions optimized)
    'positions':[[  0.300000, -7.945000, -4.8440000],
                 [ -1.133000, -7.581000, -4.4840000],
                 [ -1.612000, -7.787000, -3.3770000],
                 [  0.650000, -7.434000, -5.7440000],
                 [  0.351000, -9.028000, -5.0100000],
                 [  0.952000, -7.712000, -3.9990000],
                 [ -1.811000, -7.075000, -5.4730000],
                 [ -2.781000, -6.832000, -5.3080000],
                 [ -1.403000, -6.863000, -6.3820000],
                 [ -0.931000, -6.425000, -10.105000],
                 [  0.041000, -6.447000, -8.9820000],
                 [ -0.356000, -6.488000, -7.8210000],
                 [ -0.492000, -6.635000, -11.086000],
                 [ -1.398000, -5.434000, -10.143000],
                 [ -1.724000, -7.150000, -9.9060000],
                 [  1.318000, -6.364000, -9.3020000],
                 [  1.636000, -6.336000, -10.260000],
                 [  2.015000, -6.339000, -8.5670000]]},
}


def create_s22_system(name, dist=None, **kwargs):
    """Create S22/S26/s22x5 system.
    """
    s22_,s22x5_,s22_name,dist = identify_s22_sys(name,dist)
    if s22_ is True:
        d = data[s22_name]
        return Atoms(d['symbols'], d['positions'], **kwargs)
    elif s22x5_ is True:
        d = data[s22_name]
        pos = 'positions '+dist
        return Atoms(d['symbols'], d[pos], **kwargs)
    else:
        raise NotImplementedError('s22/s26/s22x5 creation failed')


def identify_s22_sys(name,dist=None):
    s22_ = False
    s22x5_ = False
    if (name in s22 or name in s26) and dist == None:
        s22_name = name
        s22_ = True
    elif name in s22x5 and dist == None:
        s22_name, dist = get_s22x5_id(name)
        s22x5_ = True
    elif name in s22 and dist != None:
        dist_ = str(dist)
        if dist_ not in ['0.9','1.0','1.2','1.5','2.0']:
            raise KeyError('Bad s22x5 distance specified: %s' % dist_) 
        else:
            s22_name = name
            dist = dist_
            s22x5_ = True
    if s22_ is False and s22x5_ is False:
        raise KeyError('s22 combination %s %s not in database' %(name,str(dist)))
    return s22_, s22x5_, s22_name, dist


def get_s22x5_id(name):
    """Get main name and relative separation distance of an S22x5 system.
    """
    s22_name = name[:-4]
    dist = name[-3:]
    return s22_name, dist


def get_s22_number(name,dist=None):
    """Returns the S22/S26 database number of a system as a string.
    """
    s22_,s22x5_,s22_name,dist_ = identify_s22_sys(name,dist)
    return data[s22_name]['s26_number']


def get_interaction_energy_cc(name,dist=None):
    """Returns the S22/S26 CCSD(T)/CBS CP interaction energy in eV.
    """
    s22_,s22x5_,s22_name,dist_ = identify_s22_sys(name,dist)
    return data[s22_name]['interaction energy CC']


def get_interaction_energy_s22(name,dist=None):
    """Returns the S22/S26 CCSD(T)/CBS CP interaction energy in eV.
    """
    s22_,s22x5_,s22_name,dist_ = identify_s22_sys(name,dist)
    e = get_interaction_energy_cc(s22_name)
    return e


def get_interaction_energy_s22x5(name, dist=None, correct_offset=True):
    """Returns the S22x5 CCSD(T)/CBS CP interaction energy in eV.
    """
    s22_,s22x5_,s22_name,dist_ = identify_s22_sys(name,dist)
    if dist_ == '0.9':
        i = 0
    elif dist_ == '1.0':
        i = 1
    elif dist_ == '1.2':
        i = 2
    elif dist_ == '1.5':
        i = 3
    elif dist_ == '2.0':
        i = 4
    else:
        raise KeyError('error, mate!')
    e = data[s22_name]['interaction energies s22x5'][i]
    if correct_offset == True:
        e *= data[s22_name]['interaction energy CC']/data[s22_name]['interaction energies s22x5'][1]
    return e


def get_name(name,dist=None):
    """Returns the database name of an s22 system
    """
    s22_,s22x5_,s22_name,dist_ = identify_s22_sys(name,dist)
    if s22x5_ is True:
        raise KeyError('System may not be in s22x5')    
    return data[name]['name']


def get_number_of_dimer_atoms(name,dist=None):
    """Returns the number of atoms in each s22 dimer as a list; [x,y].
    """
    s22_,s22x5_,s22_name,dist_ = identify_s22_sys(name,dist)
    return data[s22_name]['dimer atoms']


def get_s22x5_distance(name, dist=None):
    """Returns the relative intermolecular distance in angstroms.
       Values are in Angstrom and are relative to the original s22 distance.
    """
    s22_,s22x5_,s22_name,dist_ = identify_s22_sys(name,dist)
    if s22_ is True:
        raise KeyError('System must be in s22x5')    
    else:
        x00 = data[s22_name]['positions 1.0'][0][0]
        x01 = data[s22_name]['positions 1.0'][-1][0]
        x10 = data[s22_name]['positions '+dist_][0][0]
        x11 = data[s22_name]['positions '+dist_][-1][0]
        d0 = x01 - x00
        d1 = x11 - x10
        return d1-d0
