/* copyright 1989 by Phil Andrews, Pittsburgh Supercomputing Center */
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include "defs.h"
#define byte_size 8
#define out_ln(instr) {out_string(instr); fb();}

/* module for QMS specific functions, note that we expect the device
   resolution is pxl_per_in, set by the calling program. All dimensions are
   in terms of device pixels
*/   

#define word_size (sizeof(int) * byte_size)
static int pxl_per_in;
static char string_buffer[max_str + 1];
static int str_index;

/* store these pointers to allow lookups into the CGM data structures */
static struct one_opt 		*popt;	/* the command line options, in only */
static struct mf_d_struct 	*pc1;	/* the class 1 elements, in only */
static struct pic_d_struct 	*pc2;	/* the class 2 elements, in only */
static struct control_struct	*pc3;	/* the class 3 elements, in only */
static struct attrib_struct	*pc5;	/* the class 5 elements, in only */
static struct info_struct 	*dptr;	/* device info, in and out */

/* for hex strings */
static hex_char[16] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
	'A', 'B', 'C', 'D', 'E', 'F'};

static int old_l_type;			/* line type presently set */
static int old_l_width;			/* line width presently set */
static float old_r, old_g, old_b;	/* colour presently set */
static int state_level;
static int rot_page;
/* macro to convert from r, g, b to grey_scale, NTSC standard */
#define col_grey(r, g, b)  (1.0 - (0.3 * r + 0.59 * g + 0.11 * b))
static int glyph_height = 77;	/* magic number from old GTEX code */
/* now define the useful QMS command language commands */
/* note that we tend to use similar commands to Talaris, would */
/* like to maintain compatability as far as possible */
#define ctrl_char '^'		/* the QMS control character */
#define quote_string "^$"	/* next char comes in hex */
#define qms_status "^ISTATUS"	/* ask for status report */
#define qms_rule "^LS"		/* tell QMS to set rule */
#define quic_on "^PY^-"		/* get everything going */
#define set_ori "^IO"		/* set orientation */
#define free_format "^F"	/* ignore <cr> etc. */
#define qms_lmargin "^IT"	/* where to set left margin */
#define qms_tmargin "^IJ"	/* where to set top margin */
#define char_spacing "^IC"	/* character spacing */
#define our_syntax "^ISYNTAX00010" /* the syntax we will use */
#define def_font "^IS"		/* select default font */
#define hor_pos "^T"		/* set position from left margin */
#define ver_pos "^J"		/* set position from original top */
#define quic_off "^-^PN^-"	/* back to line printer mode */
#define form_feed "\014"	/* note string */
#define qms_p_page "^,"		/* end pass and print page */
#define do_copies "^DCC"	/* make mutliple copies */
#define no_free "^O"		/* turn off free format */
#define start_flag "^IGV"	/* start vector mode */
#define end_flag "^IGE"		/* end vector mode */
#define pen_up "^U"		/* raise pen */
#define pen_down "^D"		/* lower pen */
#define p_fill_b "^PF"		/* begin filled polygon */
#define p_fill_e "^PS"		/* end filled polygon */
#define pen_width "^PW"		/* set pen width */
#define sel_line_pattern "^V"	/* select line pattern */
#define quic_char "^M"		/* quic characters coming */
#define pass_term "^-"		/* terminate the pass */
#define gen_term "^G"		/* terminate optional commands */
#define abs_ver "^IV"		/* absolute vertical positioning */
#define abs_hor "^IH"		/* absolute horizontal positioning */
#define rel_hor "^TD"		/* relative horizontal positioning */
#define rel_ver "^JD"		/* relative vertical positioning */
#define RESET_STRING "ReSeTrEsEtReSeT" 	/* reset to power up defaults */
#define port_mode "^IOP"	/* portrait mode */
#define land_mode "^IOL"	/* landscape mode */
#define dl_font "^DF"		/* load a font */
#define pxl_data "^P"		/* pixel data coming */
#define repeat_on "^B"		/* repeat on byte */
#define repeat_byte "^C"	/* repeat arbitrary byte */
#define repeat_off "^D"		/* repeat off byte */
#define pxl_size "^IP"		/* bigger pixels */
/* handy macros */
#define out_qvec(x,y) {out_qx(x);outc(':');out_qy(y);}
static enum {qms_t, qms_v, qms_u} qms_m = qms_u;
#define vec_mode (qms_m == qms_v)
#define text_mode (qms_m == qms_t)
#define vec_on {out_string(start_flag); qms_m = qms_v;}
#define vec_off {out_string(end_flag);qms_m = qms_t;}
extern outgint();	/* in IO.C */
static int out_qx(x)
int x;
{
	return(outgint(x, 4));
}
static int out_qy(y)
int y;
{
	y = (dptr->y_size) * dptr->ypxl_in - y;
	return(outgint(y, 4));
}
/* set the line type and width */
static void set_l_type(l_type, line_width)
int line_width;
enum line_enum l_type;
{
int dash_size, dot_size;

	if ((int) l_type !=  old_l_type) {
	    old_l_type = (int) l_type;
	    out_string(sel_line_pattern);
	    switch (l_type) {
case solid_l:	outc('0'); break;
case dash:	outc('1'); break;
case dot_l:	outc('2'); break;
case dash_dot:	outc('7'); break;
case dash_d_d:	outc('9'); break;
default:	fprintf(stderr, "unknown line type [%d]!\n", l_type);
	    }
	}
	if (line_width != old_l_width) {
	    out_string(pen_width);
	    if (line_width < 1) line_width = 1;
	    if (!line_width % 2) ++line_width;
	    outgint(line_width, 2);
	    old_l_width = line_width;
	}

	return;
}

/* set up the printer */
qms_begin(comment, file_name, prog_name)
char *comment, *file_name, *prog_name;
{
long time();
char *ctime(), *cuserid(), out_date[40];
int now, slen;
float rotation;
static char buffer[max_str];

	++state_level;
	if (state_level > 0) return(1);	/* already done everything */
	pxl_per_in = dptr->pxl_in;	/* might have been changed */
	fb();			/* QMS often needs a blank line here */
	out_string(quic_on);	/* QUIC command mode */
	out_string(free_format);	/* ignore carriage returns */
	fb();
        out_string(quic_on);    /* Quic back on for quirks in some qms's */
        out_string(port_mode);  /* port Mode for another quic setup quirk */
	out_string(free_format);	/* ignore carriage returns */   
	out_string(qms_lmargin);	/* left margin */
	out_string(qms_tmargin);	/* top margin */
	out_string(char_spacing); out_string("00");	/* charspacing */

	rotation = (popt[(int) degrees].set) ? popt[(int) degrees].val.r :
	    0.0;
	if ((rotation > 89.0) && (rotation < 91.0)) { /* can do it */
	    rot_page = 1;
	    out_string(port_mode);
	} else {
	    rot_page = 0;
	    out_string(land_mode); 
	}
	out_ln(our_syntax);	/* use dot addressability, needs new line ! */
	return(1);
}



/* finish up */
qms_end(pages_done)
int pages_done;
{
	--state_level;
	if (state_level >= 0) return(1);
	out_string(no_free);
	out_string(quic_off);
	fb();	/* flush the buffer */
	return(1);
}
/* start the page */
qms_bpage(pic_name, xoffset, yoffset, rotation, rb, gb, bb, page_no,
    xsize, ysize)
char *pic_name;
float rotation;		/* how much to rotate the page */
int xoffset, yoffset;	/* offset in pixels */
int page_no;		/* page number */
float rb, gb, bb;	/* background colour values */
int xsize, ysize;
{
int max_x, max_y;
char buffer[max_str];

	if (rot_page) {
	    out_string(port_mode);
	} else {
	    out_string(land_mode); 
	}
	old_l_type = -1;
	old_l_width = -1;
	dptr->c_height = dptr->c_width = pc5->c_height;

	return(1);
}
/* end the page */
qms_epage(no_copies)
int no_copies;
{
	
	out_string(pass_term);		/* for safety */
	if (state_level == 0) {
	    if (no_copies > 0) {
	    	out_string(do_copies);
	    	outgint(no_copies, 4);
	    }
	    out_string(qms_p_page);
	}
	return(1);
}


/* set text on a QMS device */
qms_text(x, y, final, buffer)
int x, y, final;
char *buffer;
{
int str_l;

	if (!text_mode) vec_off;
	qms_abs_both(x, y + pc5->c_height);
	out_string(quic_char);
	outgint(pc5->c_height, 4);
	outgint(pc5->c_height, 4);
	out_string(gen_term);
	while (*buffer) {
	    if ( iscntrl(*buffer) || (*buffer == ctrl_char )) {
	    	out_string(quote_string);
	    	outc(hex_char[(*buffer >> 4) & 15]);
	    	outc(hex_char[*buffer & 15]);
	    } else outc(*buffer);
	    ++buffer;
	}
	out_string(pass_term);
	return(1);
}
/* plot a set of lines */
qms_pline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;

	if ( no_pairs <= 1 ) return(1);
	if (!vec_mode) vec_on;
	set_l_type(pc5->line_type, pc5->line_width.i);
	out_string(pen_up);
	out_qvec(x1_ptr[0], y1_ptr[0]);
	for (i=1;i<no_pairs;++i) {
	    out_string(pen_down);
	    out_qvec(x1_ptr[i], y1_ptr[i]);
	}
	vec_off;
	return(1);
}
/* plot a set of lines between alternate points */
qms_dpline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;

	if ( no_pairs <= 1 ) return(1);

	if (!vec_mode) vec_on;
	set_l_type(pc5->line_type, pc5->line_width.i);
	for (i=0; i < no_pairs ; i+=2) {
	    out_string(pen_up);
	    out_qvec(x1_ptr[i], y1_ptr[i]);
	    out_string(pen_down);
	    out_qvec(x1_ptr[i+1], y1_ptr[i+1]);
	}
	return(1);
}
/* put up a series of markers, characters are set at bottom left corner */
qms_pmarker(no_pairs, xptr, yptr)
int no_pairs, *xptr, *yptr;
{
int i, x_adjust = 0, y_adjust = 0;

	switch (pc5->mk_type) {
case 1 : y_adjust = - 0.05 * pc5->mk_size.i; break;
case 2 : y_adjust = -0.28 * pc5->mk_size.i; break;
case 3 : y_adjust = -0.45 * pc5->mk_size.i; break;
case 4 : y_adjust = -0.23 * pc5->mk_size.i; break;
case 5 : y_adjust = -0.23 * pc5->mk_size.i; break;
default: fprintf(stderr, "illegal marker type ! = %d\n",  pc5->mk_type);
	   y_adjust = pc5->mk_size.i / 2; 
}

	if (!text_mode) vec_off;
	out_string(quic_char);
	outgint(pc5->mk_size.i, 4);
	outgint(pc5->mk_size.i, 4);
	out_string(gen_term);

	for (i=0;i<no_pairs;++i) {
	    qms_abs_both(xptr[i], yptr[i] + pc5->mk_size.i + y_adjust);
	    outc(c_mark[pc5->mk_type]);
	}
	out_string(pass_term);
	return(1);
}
/* do a polygon */
qms_pgon(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;
char do_border, fill_string[3];
	if ( no_pairs <= 1 ) return(1);

	switch (pc5->int_style) {
	case hollow :	break;	/* don't try to fill these */
	case empty :	break;
	
	case pattern :		/* no pattern fill yet */
	case hatch :		/* no hatch fill yet */
	case solid_i :		
				break;
	}

	if (!vec_mode) vec_on;
	/* may want visible edge */
	do_border = ((pc5->edge_vis == on) || (pc5->int_style == hollow)) ?
	    '1' : '0';

	set_l_type(pc5->edge_type, pc5->edge_width.i);
	set_fill(&pc5->fill_colour, fill_string);
	fill_string[2] = 0;

	out_string(p_fill_b);
	outc(do_border);
	out_string(fill_string);
	out_string(pen_up);
	out_qvec(x1_ptr[0], y1_ptr[0]);
	for (i=1;i<no_pairs;++i) {
	    out_string(pen_down);
	    out_qvec(x1_ptr[i], y1_ptr[i]);
	}
	out_string(p_fill_e);
	vec_off;	
	return(1);
}
/* function to return closest fill string */
static set_fill(cptr, fill_string)
struct rgbi_struct *cptr;
char *fill_string;
{
float grey_col(), grey_val;
unsigned int fill_num, mode;
#define max_fill_char 18
static char *fill_char[max_fill_char] = { "00", "01", "02", "03", "04",
"05", "06", "07", "08", "09", "10", "11", "12", "13", "14", "15", "16", "20"};

	grey_val = grey_col(cptr);
 	fill_num = max_fill_char * grey_val;
	if (fill_num >= max_fill_char) fill_num = max_fill_char - 1;
	strcpy(fill_string, fill_char[fill_num]);
	return(1);
}
#undef max_fill_char

/* return a real value for the grey associated with a colour (0->light) */
static float grey_col(cptr)
struct rgbi_struct *cptr;
{
float r, g, b;

	if (pc2->c_s_mode == i_c_mode) {	/* indexed */
	    r = pc5->ctab[cptr->ind * 3];
	    g = pc5->ctab[cptr->ind * 3 + 1];
	    b = pc5->ctab[cptr->ind * 3 + 2];
	}
	else if (pc2->c_s_mode == d_c_mode) {	/* direct colour */
	    r = cptr->red;
	    g = cptr->green;
	    r = cptr->blue;
	}
	return(col_grey(r, g, b));
}

/* cgm cell array */
qms_carray(cp, cq, cr, nx, ny, col_prec, dat_ptr, rep_mode, no_bytes)
int *cp;	/* first corner */
int *cq;	/* the diagonal to cp */
int *cr;	/* first row is from cp to cr */
int nx;		/* number of elements in x rows (i.e., parallel to cp-cr) */
int ny;		/* number of rows */
int col_prec;	/* the colour precision (how many possible colour values */
unsigned char *dat_ptr;	/* pointer to the body of information */
int rep_mode;	/* the representation mode; 0, run length encoded, 1 normal */
int no_bytes;	/* total number of bytes */
/* without compression there should be ny * row_size bytes after dat_ptr, 
   the first nx * c_size bits of each row should contain information */
{
int i, j, repeat_count, row_size, hmul, vmul, start[2];
unsigned char hold_byte;

	if ((nx <= 0) || (ny <= 0)) return(2);
	if (!text_mode) vec_off;

	/* these checks should be unnecessary */
	start[0] = (cp[0] < cr[0]) ? cp[0] : cr[0];
	start[1] = (cp[1] > cq[1]) ? cp[1] : cq[1];
	if (start[0] < 0) start[0] = 0;
	if (start[1] < 0) start[1] = 0;
	/* assume direction is right, the qms does l to r, t to b */
	qms_abs_both(start[0], start[1]); fb();
	hmul = (cr[0] - cp[0]) / nx;
	vmul = (cq[1] - cr[1]) / ny;
	if (hmul < 0) hmul = -hmul;
	if (vmul < 0) vmul = -vmul;
	hmul = (hmul) ? hmul + 0.5 : 1;
	vmul = (vmul) ? vmul + 0.5 : 1;
	if (hmul > 99) hmul = 99;
	if (vmul > 99) vmul = 99;

	if ((hmul > 1) || (vmul > 1)) {
	    out_string(pxl_size);
	    outgint(vmul, 2);
	    outgint(hmul, 2);
	    fb();
	}

	row_size = (nx + byte_size - 1) / byte_size;
	/* should have a precision 1, indexed cell array */
	fb();	/* new line */
	out_string(pxl_data);	
	outgint(nx, 4); fb();
/* now the big loop, not that we don't compress accross line bdrys */
	for (i=0; i<ny; ++i) {
	    repeat_count = 0;
	    hold_byte = *dat_ptr;
	    for (j = 0; j < row_size; ++j) {
		if ((*dat_ptr != hold_byte) || (repeat_count == 999)) {
		    if (repeat_count == 1) {
			outc(hex_char[(hold_byte >> 4) & 15]);
			outc(hex_char[hold_byte & 15]);
		    } else 
/*		    if (hold_byte == 0) {
			out_string(repeat_off);
			outgint(repeat_count, 3);
		    } else
		    if (hold_byte == 255) {
			out_string(repeat_on);
			outgint(repeat_count, 3);
		    } else
*/		    {
			out_string(repeat_byte);
			outgint(repeat_count, 3);
			outc(hex_char[(hold_byte >> 4) & 15]);
			outc(hex_char[hold_byte & 15]);
		    }
		    hold_byte = *dat_ptr;
		    repeat_count = 1;
		} else ++repeat_count;
		++dat_ptr;
	    }
/* have to flush out what we've been storing */
	if (repeat_count == 1) {
	    outc(hex_char[(hold_byte >> 4) & 15]);
	    outc(hex_char[hold_byte & 15]);
/*	} else 
	if (hold_byte == 0) {
	    out_string(repeat_off);
	    outgint(repeat_count, 3);
	} else
	if (hold_byte == 255) {
	    out_string(repeat_on);
	    outgint(repeat_count, 3);
*/	} else {
	    out_string(repeat_byte);
	    outgint(repeat_count, 3);
	    outc(hex_char[(hold_byte >> 4) & 15]);
	    outc(hex_char[hold_byte & 15]);
	}
	repeat_count = 0;
	    fb();
	}
	out_ln(gen_term);
	return(1);
}
/* set a rectangle */
qms_rectangle(x1, y1, x2, y2)
int x1, x2, y1, y2;
{

	qms_abs_both(x1, y1);
	out_string(qms_rule);
	outgint(x2 - x1 + 1, 4);
	return(outgint(y2 - y1 + 1, 4));
}
/* the attributes */
/* set the character height */
qms_cheight(size)
int size;
{
	dptr->c_height = dptr->c_width = size;	/* show we have done it */
	return(1);
}


	
/* the TeX setting routines */
/* set the current font */
qms_s_font(font_no)
int font_no;
{
char buffer[max_str];

	out_string(def_font);
	return(outgint(font_no, 5));
}
/* set a rule */
qms_s_rule(h, v, w_pxl, h_pxl)
int h, v, w_pxl, h_pxl;
{
	qms_abs_both(h, v + h_pxl - 1);
	out_string(qms_rule);
	outgint(w_pxl, 4);
	return(outgint(h_pxl, 4));
}
/* set both the absolute vertical and horizontal positions */
qms_abs_both(hpos, vpos)
int hpos, vpos;
{
	qms_abs_h(hpos);
	qms_abs_v(vpos);
	return(1);
}
/* set the absolute horizontal position */
qms_abs_h(pos)
int pos;
{
	out_string(abs_hor);
	out_qx(pos);
	return(1);
}
/* set the absolute vertical position */
qms_abs_v(pos)
int pos;
{
	out_string(abs_ver);
	out_qy(pos);
	return(1);
}
/* move horizontally, relatively */
qms_rel_h(shift)
int shift;
{
	out_string(rel_hor);
	return(outgint(shift, 4));
}
/* set a char */
qms_s_char(char_no)
int char_no;
{
	if ( iscntrl(char_no) || (char_no == ctrl_char )) {
	    out_string(quote_string);
	    outc(hex_char[(char_no >> 4) & 15]);
	    outc(hex_char[char_no & 15]);
	} else outc(char_no);

	return(1);
}
/* start a QUIC font definition */
qms_st_font(font_no, finfo)
int font_no;
struct font *finfo;
{
static char version = '5';
char orientation;
int i;
	if (((dptr->capability & rot1_font) && (!rot_page)) ||
	    ((dptr->capability & rot2_font) && (rot_page))) {
	    orientation = 'Y';
	} else {
	    orientation = 'X';
	}

	fb();	/* this is important, so be careful */
	out_string(dl_font);
	doint(font_no);
	outc(orientation);
	outc(version);
	for (i=0; (i<4) && finfo->name[i]; ++i) outc(finfo->name[i]);
	for (;i<4;++i) outc(' ');
	outgint(glyph_height, 3);
	outc('T');
	outc(',');
	return(1);
}
/* end the font definition */
qms_e_font(font_no)
int font_no;
{
	out_string(gen_term);
	return(1);
}
/* make a QUIC character definition */
qms_dev_char(in_ptr, w_bytes, height, width, char_no, 
	tfm_width, pxl_width, char_array, bytes_found, h_dots, v_dots)
unsigned char *in_ptr;
char *char_array;
int w_bytes, height, char_no, width, tfm_width,pxl_width,
*bytes_found, h_dots, v_dots;
{
int i, no_bytes = 0;
unsigned char *cur_ptr;
char v_sign, h_sign;

	if ( (height == 0) || (width == 0) ) {	/* invisible, or illegal */
	    /* put in an empty character or it gets upset */
	    sprintf(char_array, "\n%02X%03d%03d%03d%c%03d%c%03d",
	    	char_no, pxl_width, 1, 1, '+', 1, '+', 1);
	    no_bytes += strlen(char_array);
	    sprintf(char_array + no_bytes, "\n%04X,", 0);
	    no_bytes += 4;
	    *bytes_found += 2;
	    return(no_bytes);
	}
	/* now real characters */
	v_sign = (v_dots >= 0) ? '+' : '-';
	h_sign = (h_dots >= 0) ? '+' : '-';
	sprintf(char_array, "\n%02X%03d%03d%03d%c%03d%c%03d",
	    char_no, pxl_width, height, width, v_sign, abs(v_dots),
	    h_sign, abs(h_dots));
	no_bytes += strlen(char_array);
	cur_ptr = (unsigned char *) in_ptr;
	for (i = 0; i<height * w_bytes; ++i){
	    char_array[no_bytes++] = hex_char[(*cur_ptr >> 4) & 15];
	    char_array[no_bytes++] = hex_char[*cur_ptr & 15];
	    ++cur_ptr;
	}
	char_array[no_bytes++] = ',';
	*bytes_found += height * w_bytes;
	return(no_bytes);
}

/* suspend things gracefully */
qms_suspend()
{
	out_string(gen_term);	/* for safety */
	out_string(pass_term);
	return(1);
}
/* and start things up again */
qms_restart()
{
	out_string(gen_term);	/* for safety */
	out_string(pass_term);
	vec_off;
	return(1);
}
/* this is the routine that sets everything up */
/* the initialising routine for the QSM module */
void qms_setup(opt, dev_info, c1, c2, c3, c5, delim, mfdesc, pdesc, mfctrl,
	gprim, attr, escfun, extfun, ctrl, pargc, argv)

struct one_opt 		*opt;	/* the command line options, in only */
struct info_struct *dev_info;	/* device info to fill out, out only */
struct mf_d_struct 	*c1;	/* the class 1 elements, in only */
struct pic_d_struct 	*c2;	/* the class 2 elements, in only */
struct control_struct	*c3;	/* the class 3 elements, in only */
struct attrib_struct	*c5;	/* the class 5 elements, in only */
int (*delim[])();    		/* delimiter functions */
int (*mfdesc[])();     		/* metafile descriptor functions */
int (*pdesc[])();      		/* page descriptor functions */
int (*mfctrl[])();     		/* controller functions */
int (*gprim[])();      		/* graphical primitives */
int (*attr[])();       		/* the attribute functions */
int (*escfun[])();     		/* the escape functions */
int (*extfun[])();     		/* the external functions */
int (*ctrl[])();       		/* controller functions */
int *pargc;			/* pointer to argc */
char **argv;			/* the main argv */	
{
#define dev_length 20
char dev_name[dev_length + 1];

	/* store the command line argument pointer */
	popt = opt;
	/* store the device info pointer */
	dptr = dev_info;
	/* and the cgm data strucures */
	pc1 = c1;
	pc2 = c2;
	pc3 = c3;
	pc5 = c5;
	/* have to figure out what type we're on */
	if (opt[(int) device].set) 
	    strncpy(dev_name, opt[(int) device].val.str, dev_length);
	else strncpy(dev_name, "qms", dev_length);
	dev_name[dev_length - 1] = 0;	/* safety */


	/* fill out the device info structure */
	dev_info->pxl_in 	= 300.0;
	dev_info->ypxl_in 	= 300.0;
	dev_info->x_size 	= 10.5;
	dev_info->y_size 	= 8.0;
	dev_info->x_offset	= 0.5;
	dev_info->y_offset	= 0.5;
	dev_info->c_height	= 10 * dev_info->pxl_in / 72;	/* 10 pt */
	dev_info->c_width	= 10 * dev_info->pxl_in / 72;	/* 10 pt */
	dev_info->d_l_width	= 3;				
	dev_info->d_e_width	= 3;				
	dev_info->d_m_size	= dev_info->c_height;	/* marker size */
	sprintf(dev_info->out_name, ".%s", dev_name);
	dev_info->capability	= port_land + brk_ok + stroke_text
				+ char_text + string_text  
				+ no_def_calls + pad_rows;
	/* have to deal with different scan line directions */
	if ((dev_name[3] == '1') || (dev_name[3] == '2')) {
	    /* qms1200 or qms2400 */
	    dev_info->capability	|= rot2_font;
	} else { /* qms 800 default */
	    dev_info->capability	|= rot1_font;
	}
	dev_info->rec_size	= 80;

	/* store the CGM data structure pointers */
	pc1 = c1;
	pc3 = c3;
	pc5 = c5;

	/* now fill out the function pointer arrays for CGM */
	/* the delimiter functions */
	delim[(int) B_Mf] 	= qms_begin;
	delim[(int) E_Mf]	= qms_end;
	delim[(int) B_Pic_Body]	= qms_bpage;
	delim[(int) E_Pic]	= qms_epage;

	/* the graphical primitives */
	gprim[(int) PolyLine]	= qms_pline;
	gprim[(int) Dis_Poly]	= qms_dpline;
	gprim[(int) PolyMarker]	= qms_pmarker;
	gprim[(int) Text]	= qms_text;
	gprim[(int) Polygon]	= qms_pgon;
	gprim[(int) Cell_Array]	= qms_carray;
	gprim[(int) Rectangle]	= qms_rectangle;

	/* the attributes */
	state_level = -1;	/* just starting */
	attr[(int) CHeight]	= qms_cheight;
	return;
}
/* the following function optimises our TeX implementation and is 
   unnecessary for CGM support only */
void qms_gtex_setup(opt, gtex_calls)	/* function pointers, out only */
struct one_opt 		*opt;	/* the command line options, in only */
int (*gtex_calls[])();
{
	/* assign TeX-specific functions */
	gtex_calls[(int) S_Font] 	= qms_s_font;
	gtex_calls[(int) Suspend]	= qms_suspend;
	gtex_calls[(int) Restart]	= qms_restart;
	gtex_calls[(int) S_Rule]	= qms_s_rule;
	gtex_calls[(int) S_Abs_Both]	= qms_abs_both;
	gtex_calls[(int) S_Abs_H]	= qms_abs_h;
	gtex_calls[(int) S_Abs_V]	= qms_abs_v;
	gtex_calls[(int) M_Rel_H]	= qms_rel_h;
	gtex_calls[(int) S_Char]	= qms_s_char;
	gtex_calls[(int) St_Font]	= qms_st_font;
	gtex_calls[(int) M_Dev_Char]	= qms_dev_char;
	gtex_calls[(int) E_Font]	= qms_e_font;

	return;
}

