/* splwin.cpp:  spell window class procedures

    Copyright (C) 1993, 1994 John-Marc Chandonia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "spl.hpp"
#include "general.hpp"
#include "spldlg.h"
#include "bookwin.hpp"
#include "splwin.hpp"
#include "spldlgs.hpp"

// stuff defined in the main program
extern HAB hab;
extern HINI hini;
extern BYTE xfer_buffer[10240];
extern int windows;
extern char changed_file_name[256];
extern HWND hwndspellhelp;

spellwindow *firstspell=NULL;

spellwindow::spellwindow(spelllist *spll) {    
    char *windowname;

    ULONG flFlags = FCF_TITLEBAR |
	FCF_SIZEBORDER|
	FCF_MINMAX|
	FCF_SYSMENU|
	FCF_MENU|
	FCF_ACCELTABLE|
	FCF_TASKLIST;

    // add new window as the first one
    next=firstspell;
    if (firstspell!=NULL) firstspell->prev=this;
    firstspell=this;
    prev=NULL;

    readonly=true;
    sl=spll;

    if (sl) windowname=sl->s->name;
    else windowname="Spell";

    // create the window 
    hwndframe = WinCreateStdWindow(HWND_DESKTOP,
				   WS_VISIBLE,
				   &flFlags,
				   (PSZ)splclass,
				   (PSZ)windowname,
				   WS_VISIBLE,
				   0,
				   SPELLMENU,
				   NULL);

    // add F1-sensitive help
    if (hwndspellhelp)
	WinAssociateHelpInstance(hwndspellhelp,hwndframe);
}

spellwindow::~spellwindow() {
    // remove from linked list
    if (this==firstspell) firstspell=next;
    if (next!=NULL) next->prev=prev;
    if (prev!=NULL) prev->next=next;

    // remove from display
    WinDestroyWindow(hwndframe);
}

// empty out mle and show a spell in it
void mle_show_spell(HWND hwndmle, spell *s) {
    IPT insertion_point=0;

    // remove old spell
    IPT text_length;
    text_length=(IPT)WinSendMsg(hwndmle,
				MLM_QUERYTEXTLENGTH,
				0,0);

    // hide window for speed:
    WinShowWindow(hwndmle,FALSE);

    WinSendMsg(hwndmle,
	       MLM_DELETE,
	       0,
	       (MPARAM)text_length);

    // set up transfer buffer
    WinSendMsg(hwndmle,
	       MLM_SETIMPORTEXPORT,
	       (MPARAM)(&xfer_buffer),
	       (MPARAM)10240);

    WinSendMsg(hwndmle,
	       MLM_FORMAT,
	       (MPARAM)MLFIE_CFTEXT,
	       0);

    s->s_print((char *)xfer_buffer);

    WinSendMsg(hwndmle,
	       MLM_IMPORT,
	       (MPARAM)(&insertion_point),
	       (MPARAM)strlen((char *)xfer_buffer));

    // show window again
    WinShowWindow(hwndmle,TRUE);
}

void edit_save_changes(spellwindow *mywin) {
    IPT insertion_point;
    ULONG text_length;
    char *tmpfilename;
    FILE *tmpfile;
    spell *mod_spell, *old_spell;
    char *bptr;
    int i;

    WinSendMsg(mywin->hwndmle,MLM_SETREADONLY,(MPARAM)TRUE,0);
    mywin->readonly=true;

	    
    // set up transfer buffer
    text_length=(ULONG)WinSendMsg(mywin->hwndmle,
				  MLM_QUERYTEXTLENGTH,
				  0,0);
    WinSendMsg(mywin->hwndmle,
	       MLM_SETIMPORTEXPORT,
	       (MPARAM)(&xfer_buffer),
	       (MPARAM)10240);
    WinSendMsg(mywin->hwndmle,
	       MLM_FORMAT,
	       (MPARAM)MLFIE_CFTEXT,
	       0);
    
    // get spell into transfer buffer
    insertion_point=0;

    WinSendMsg(mywin->hwndmle,
	       MLM_EXPORT,
	       (MPARAM)(&insertion_point),
	       (MPARAM)(&text_length));
    
    // write spell to file
    tmpfilename=tmpnam(NULL);
    tmpfile=fopen(tmpfilename,"w");
    bptr=(char *)xfer_buffer;
    i=0;
    while (*bptr!=(char)0) {
	if (*bptr==(char)13) {
	    fprintf(tmpfile,"\n");
	    bptr++;
	    i=0;
	}
	else if ((i>65) && (*bptr==' ')) {
	    fprintf(tmpfile,"\n");
	    i=0;
	}
	else fprintf(tmpfile,"%c",*bptr);
	bptr++;
	i++;
    }
    fprintf(tmpfile,"\n");
    fclose(tmpfile);
    
    // read it back in, deleting tmp file
    tmpfile=fopen(tmpfilename,"r");
    if (mywin->sl->s->type=='M') mod_spell=new magespell;
    else mod_spell=new priestspell;
    mod_spell->f_read(tmpfile);
    fclose(tmpfile);
    //unlink(tmpfilename);

    old_spell=mywin->sl->s;

    // change spell in all windows
    change_all(old_spell,mod_spell);

    // write the spell to the end of "changed"
    tmpfile=fopen(changed_file_name,"a");
    fprintf(tmpfile,"-----\n");
    mod_spell->f_print(tmpfile);
    fclose(tmpfile);

    // show the new spell window
    find_and_show_spell(mywin->parent, mod_spell);

    // remove old spell, and my window
    delete old_spell;
    WinSendMsg(mywin->hwnd,
	       WM_CLOSE,
	       0,
	       0); 
}

// change mle to given spellist, if everything's OK.
void mle_set_spell(spellwindow *mywin, spelllist *sl) {
    if (sl!=NULL) {
	if (!mywin->readonly) {
	    ULONG response = 
		WinMessageBox(HWND_DESKTOP,
			      HWND_DESKTOP,
			      (PSZ)"You are in the middle of editing.  Any changes will be lost.  Are you sure?",
			      (PSZ)"Warning!",
			      102,
			      MB_YESNO|MB_ICONEXCLAMATION);
	    if (response==MBID_NO) return;
	    mywin->readonly=true;
	}
	mywin->sl=sl;
	WinSetWindowText(mywin->hwndframe,(PSZ)mywin->sl->s->name);
	mle_show_spell(mywin->hwndmle,mywin->sl->s);
    }
}

// window function for spell window:
MRESULT EXPENTRY spell_window_func(HWND hwnd, ULONG msg, 
				   MPARAM param1, MPARAM param2) {
    spellwindow *mywin;
    int cx,cy;
    SWP winpos,swp;
    ULONG ultemp;
    SHORT w,h;

    switch (msg) {
    case WM_ERASEBACKGROUND:
	return (MRESULT)TRUE;

    case WM_CREATE:
	// save handle 
	firstspell->hwnd=hwnd;

	// save pointer to my class in window word
	WinSetWindowPtr(hwnd,
			0,
			firstspell);

	// resize to default spellbook size
	WinQueryTaskSizePos(hab,0,&swp);
	WinQueryTaskSizePos(hab,0,&winpos);
	ultemp=sizeof(winpos);
	PrfQueryProfileData(hini,
			    "spellbook",
			    "newspellpos",
			    &winpos,
			    &ultemp);
	swp.cx = winpos.cx;
	swp.cy = winpos.cy;

	// make sure we aren't off screen.
	w = WinQuerySysValue(HWND_DESKTOP,SV_CXSCREEN);
	h = WinQuerySysValue(HWND_DESKTOP,SV_CYSCREEN);
	if (swp.x+swp.cx > w) swp.x = w - swp.cx;
	if (swp.y+swp.cy > h) swp.y = h - swp.cy;

	firstspell->hwndframe = WinQueryWindow(hwnd,QW_PARENT);
	WinSetWindowPos(firstspell->hwndframe,
			HWND_TOP,
			swp.x,swp.y,
			swp.cx,swp.cy,
			SWP_SIZE|SWP_SHOW|SWP_MOVE|SWP_ZORDER);

	// find out my size 
	WinQueryWindowPos(hwnd,&winpos);
	cx=winpos.cx;
	cy=winpos.cy;

	// create multi line entry filling whole window 
	firstspell->hwndmle=
	    WinCreateWindow(
			    hwnd,
			    WC_MLE,
			    (PSZ)NULL,
			    MLS_READONLY | MLS_WORDWRAP | MLS_VSCROLL,
    			    (LONG)0,
			    (LONG)0,
			    (LONG)cx,
			    (LONG)cy,
			    hwnd,
			    HWND_TOP,
			    windows++,
			    NULL,
			    NULL);

	// fill mle with spell 
	mle_show_spell(firstspell->hwndmle, firstspell->sl->s);

	// and activate it 
	WinShowWindow(firstspell->hwndmle,TRUE);

	break;

    case WM_SIZE:
	mywin=(spellwindow *)WinQueryWindowPtr(hwnd,
					       0);

	cx=SHORT1FROMMP(param2);
	cy=SHORT2FROMMP(param2);
	// tell the MLE to change its size 
	WinSetWindowPos(mywin->hwndmle,0,0,0,cx,cy,SWP_SIZE);

	break;


    // menu is activated
    case WM_INITMENU:
	mywin=(spellwindow *)WinQueryWindowPtr(hwnd,
					       0);
	switch(SHORT1FROMMP(param1)) {
	case EDIT_SUBMENU:
	    MRESULT mr1, mr2;
	    BOOL enable;

	    // see if text is selected in MLE
	    mr1 = WinSendMsg(mywin->hwndmle, MLM_QUERYSEL,MPFROMSHORT(MLFQS_MINSEL), NULL);
	    mr2 = WinSendMsg(mywin->hwndmle, MLM_QUERYSEL,MPFROMSHORT(MLFQS_MAXSEL), NULL);
	    if (mr1 != mr2) enable = TRUE;
	    else enable = FALSE;

	    enable_menu_item(HWNDFROMMP(param2), EDIT_COPY, enable);

	    if (mywin->readonly) enable=FALSE;
	    enable_menu_item(HWNDFROMMP(param2), EDIT_CUT, enable);
	    enable_menu_item(HWNDFROMMP(param2), EDIT_CLEAR, enable);

	    // if undoable, and !readonly, enable undo.
	    if (!mywin->readonly) {
		mr1 =  WinSendMsg(mywin->hwndmle, MLM_QUERYUNDO, NULL, NULL);
		if (mr1 != 0) enable = TRUE;
		else enable = FALSE;
	    }
	    enable_menu_item(HWNDFROMMP(param2), EDIT_UNDO, enable);

	    // if clipboard has text, and !readonly, enable paste.
	    if (!mywin->readonly) {
		if(WinOpenClipbrd(hab))
		{
		    ULONG ulFmtInfo;
		    if (WinQueryClipbrdFmtInfo(hab, CF_TEXT, &ulFmtInfo))
			enable = TRUE;
		    else
			enable = FALSE;
		    WinCloseClipbrd(hab);
		}
		else
		    enable = TRUE;
	    }
	    enable_menu_item(HWNDFROMMP(param2), EDIT_PASTE, enable);

	    // enable other menus depending on readonly status
	    if (mywin->readonly) enable=TRUE;
	    else enable=FALSE;
	    enable_menu_item(HWNDFROMMP(param2),EDIT_START,enable);
	    enable_menu_item(HWNDFROMMP(param2),EDIT_SAVECHANGES,!enable);
	    enable_menu_item(HWNDFROMMP(param2),EDIT_DISCARDCHANGES,!enable);

	    break;
	}
	break;
	
    // command from menu 
    case WM_COMMAND:
	mywin=(spellwindow *)WinQueryWindowPtr(hwnd,
					       0);
	
	switch(SHORT1FROMMP(param1)) {
	case NEXTSPELL:
	    mle_set_spell(mywin,mywin->sl->next);
	    break;

	case PREVSPELL:
	    mle_set_spell(mywin,mywin->sl->prev);
	    break;

	case EDIT_UNDO:
	    WinSendMsg(mywin->hwndmle,MLM_UNDO,0,0);
	    break;

	case EDIT_CUT:
	    WinSendMsg(mywin->hwndmle,MLM_CUT,0,0);
	    break;

	case EDIT_CLEAR:
	    WinSendMsg(mywin->hwndmle,MLM_CLEAR,0,0);
	    break;

	case EDIT_COPY:
	    WinSendMsg(mywin->hwndmle,MLM_COPY,0,0);
	    break;

	case EDIT_PASTE:
	    WinSendMsg(mywin->hwndmle,MLM_PASTE,0,0);
	    break;

	case EDIT_START:
	    WinSendMsg(mywin->hwndmle,MLM_SETREADONLY,(MPARAM)FALSE,0);
	    mywin->readonly=false;
	    break;

	case EDIT_SAVECHANGES:
	    edit_save_changes(mywin);
	    break;

	case EDIT_DISCARDCHANGES:
	    WinSendMsg(mywin->hwndmle,MLM_SETREADONLY,(MPARAM)TRUE,0);
	    mywin->readonly=true;
	    mle_show_spell(mywin->hwndmle,mywin->sl->s);
	    break;

	case ABOUT:
	    WinDlgBox(HWND_DESKTOP,
		      hwnd,
		      about_dlg_func,
		      0L,
		      ABOUT_DLG,
		      NULL);
	    break;

	case HELP_INDEX:
	    WinSendMsg(hwndspellhelp,
		       HM_HELP_INDEX,
		       0,0);
	    break;

	case HELP_GENERAL:
	    WinSendMsg(hwndspellhelp,
		       HM_DISPLAY_HELP,
		       MPFROMSHORT(38),
		       HM_RESOURCEID);
	    break;

	case HELP_HELP:
	    WinSendMsg(hwndspellhelp,
		       HM_DISPLAY_HELP,
		       0,0);
	    break;
	}
	break;

    case WM_CLOSE:
	mywin=(spellwindow *)WinQueryWindowPtr(hwnd,0);
	delete mywin;
	return (MRESULT)TRUE;
	break;

    case WM_DESTROY:
	mywin=(spellwindow *)WinQueryWindowPtr(hwnd,0);
	WinQueryWindowPos(mywin->hwndframe,&winpos);
	PrfWriteProfileData(hini,
			    "spellbook",
			    "newspellpos",
			    &winpos,
			    sizeof(winpos));
	break;
    }
    return WinDefWindowProc(hwnd, msg, param1, param2);
}



