/*
 * OS/2 PM File Finder Utility - SOURCE CODE
 *
 * LANGUAGE      : Microsoft C5.1
 * MODEL         : small
 * ENVIRONMENT   : Microsoft OS/2 PM SDK
 * STATUS        : operational
 *
 * Adapted from KPW's Windows 2.1 version of 07/20/88.
 *
 * 11/01/88 1.00 - Jerry Weldon - initial creation.
 *
 */

#define INCL_DOSPROCESS
#define INCL_DOSNLS
#define INCL_WINWINDOWMGR
#define INCL_WINDIALOGS
#define INCL_WINBUTTONS
#define INCL_WINENTRYFIELDS
#define INCL_WINLISTBOXES
#define INCL_WINMENUS
#define INCL_WINFRAMEMGR
#define INCL_WINSYS
#define INCL_WINPOINTERS
#define INCL_GPIPRIMITIVES
#define INCL_GPILCIDS

#include <os2.h>
#include <stdio.h>
#include <string.h>

#include "finder.h"

/* global variables */
FATTRS      fat;                                  /* font attributes  */

/**/

/*
 * main( VOID ) : VOID;
 *
 * This function is the system entry point for the application
 * and is responsible for defining the appropriate window 
 * classes and for processing all the messages.  Note how
 * the dialog box manager is responsible for the operation of
 * the file finder window.
 *
 */
 
VOID main( VOID )
{
  HAB         hab;                        /* handle to anchor block   */
  HMQ         hmq;                        /* handle to message queue  */
  HWND        hwnd;                       /* handle to main window    */
  HPOINTER    hptr;                       /* handle to window's icon  */

  /* initialize thread & create message queue */
  hab = WinInitialize( 0 );
  hmq = WinCreateMsgQueue( hab, DEFAULT_QUEUE_SIZE );

  /* load the main dialog */
  hwnd = WinLoadDlg(
    HWND_DESKTOP,
    HWND_DESKTOP,
    FinderDlgProc,
    NULL,
    ID_FINDER,
    NULL
  );

  /* set an icon for the dialog */
  hptr = WinLoadPointer( HWND_DESKTOP, NULL, ID_FINDER );
  WinSendMsg( hwnd, WM_SETICON, (MPARAM)hptr, 0L );

  /* process the dialog */
  WinProcessDlg( hwnd );

  /* destroy the dialog and its icon */
  WinDestroyWindow( hwnd );
  WinDestroyPointer( hptr );

  /* destroy message queue & terminate thread */
  WinDestroyMsgQueue( hmq );
  WinTerminate( hab );
}

/**/

/*
 * FinderDlgProc( hwnd, usMsg, mp1, mp2 ) : MRESULT;
 *
 *    hwnd          handle to main dialog window
 *    usMsg         message number
 *    mp1           message parameter 1
 *    mp2           message parameter 2
 *
 * This function is responsible for processing all the messages
 * which relate to the file finder dialog box.  This mainly
 * involves the definition and retrieval of the various
 * events generated by the user.
 *
 */

MRESULT EXPENTRY FinderDlgProc(
  HWND        hwnd,
  USHORT      usMsg,
  MPARAM      mp1,
  MPARAM      mp2 )
{
  MRESULT     mresRtnVal;                 /* function return value    */
  BOOL        fPassToDef;                 /* pass to def window proc? */

  mresRtnVal = FALSE;
  fPassToDef = FALSE;
  
  switch ( usMsg ) {

    case WM_INITDLG:
      {
        HWND        hwndSysMenu;          /* system menu handle       */
        USHORT      idSysMenu;            /* system menu id           */
        MENUITEM    miSysMenu;            /* system menu item info    */
        MENUITEM    miAbout;              /* About menu item info     */
        USHORT      usCurDrive;           /* current drive number     */
        ULONG       flDrives;             /* available logical drives */
        CHAR        szDrive[6];           /* temporary drive string   */
        USHORT      iItem;                /* temporary loop variable  */
        USHORT      cszList;              /* # of selected drives     */
        HWND        hwndListBox;          /* handle to list box       */

        /* add About item to system menu */
        if ( hwndSysMenu = WinWindowFromID( hwnd, FID_SYSMENU ) ) {

          /* get handle of system submenu */
          idSysMenu = SHORT1FROMMR( WinSendMsg( hwndSysMenu, MM_ITEMIDFROMPOSITION, MPFROMSHORT( 0 ), 0L ) );
          WinSendMsg( hwndSysMenu, MM_QUERYITEM, MPFROM2SHORT( idSysMenu, FALSE ), MPFROMP( &miSysMenu ) );
          hwndSysMenu = miSysMenu.hwndSubMenu;

          /* add menu separator */
          miAbout.iPosition = MIT_END;
          miAbout.afStyle = MIS_SEPARATOR;
          miAbout.afAttribute = 0;
          miAbout.id = 0;
          miAbout.hwndSubMenu = NULL;
          miAbout.hItem = NULL;
          WinSendMsg( hwndSysMenu, MM_INSERTITEM, MPFROMP( &miAbout ), NULL );

          /* add About item */
          miAbout.afStyle = MIS_TEXT;
          miAbout.id = IDM_ABOUT;
          WinSendMsg( hwndSysMenu, MM_INSERTITEM, MPFROMP( &miAbout ), MPFROMP( "A~bout..." ) );

        }

        /* center dialog box on screen */
        CenterPopup( hwnd, HWND_DESKTOP );
        
        /* define search pattern */
        WinSetDlgItemText( hwnd, IDD_PATTERN, "*.*" );

        /* initialize drive list */
        DosQCurDisk( &usCurDrive, &flDrives );
        szDrive[0] = '[';
        szDrive[1] = '-';
        szDrive[3] = '-';
        szDrive[4] = ']';
        szDrive[5] = '\0';
        for ( szDrive[2] = 'A'; szDrive[2] <= 'Z'; szDrive[2]++ ) {
          if ( flDrives & 1 )
            WinSendDlgItemMsg(
              hwnd,
              IDD_DRIVES,
              LM_INSERTITEM,
              MPFROMSHORT( LIT_END ),
              MPFROMP( szDrive )
            );
          flDrives >>= 1;
        }
    
        /* select all fixed drives */           
        hwndListBox = WinWindowFromID( hwnd, IDD_DRIVES );
        cszList = SHORT1FROMMR( WinSendMsg(
          hwndListBox,
          LM_QUERYITEMCOUNT,
          0L,
          0L
        ) );
        for ( iItem = 0; iItem < cszList; iItem++ ) {
          WinSendMsg(
            hwndListBox,
            LM_QUERYITEMTEXT,
            MPFROM2SHORT( iItem, sizeof(szDrive) ),
            MPFROMP( szDrive )
          );
          if ( szDrive[2] >= 'C' )
            WinSendMsg(
              hwndListBox,
              LM_SELECTITEM,
              MPFROMSHORT( iItem ),
              MPFROMSHORT( TRUE )
            );
        }
      }
      break;

    case WM_CONTROL:
      switch ( SHORT1FROMMP( mp1 ) ) {

        case IDD_PATTERN:
          if ( SHORT2FROMMP( mp1 ) == EN_CHANGE ) 
            /* enable/disable search button */
            WinEnableWindow(
              WinWindowFromID( hwnd, IDD_SEARCH ),
              WinQueryWindowTextLength(
                WinWindowFromID( hwnd, IDD_PATTERN )
              )
            );
          else
            fPassToDef = TRUE;
          break;

        case IDD_FILES:
          switch ( SHORT2FROMMP( mp1 ) ) {

            case LN_SELECT: /* an item was selected */
              {
                USHORT      iSelection;     /* selected file index    */
                CHAR        szFile[64];     /* name of selected file  */
                USHORT      cchFile;        /* file name length       */
                BOOL        fExecable;      /* is file executable?    */

                /* get name of selected file */
                fExecable = FALSE;
                iSelection = SHORT1FROMMR( WinSendDlgItemMsg(
                  hwnd,
                  IDD_FILES,
                  LM_QUERYSELECTION,
                  MPFROMSHORT( LIT_FIRST ),
                  0L
                ) );
                if ( iSelection != LIT_NONE ) {
                  WinSendDlgItemMsg(
                    hwnd,
                    IDD_FILES,
                    LM_QUERYITEMTEXT,
                    MPFROM2SHORT( iSelection, sizeof(szFile) ),
                    MPFROMP( szFile )
                  );

                  /* is file executable? */
                  if ( szFile[0] == ' ' ) {
                    cchFile = strlen( strtok( &szFile[2], " " ) );
                    if ( cchFile > 4
                      && strcmp( &szFile[2+cchFile-4], ".EXE" ) == 0 )
                      fExecable = TRUE;
                  }
                }

                /* if file is executable, enable run button */
                WinEnableWindow(
                  WinWindowFromID( hwnd, IDD_RUN ),
                  fExecable
                );
              }
              break;

            case LN_ENTER: /* equivalent to run button clicked */
              WinSendDlgItemMsg( hwnd, IDD_RUN, BM_CLICK, 0L, 0L );
              break;

            default:
              fPassToDef = TRUE;
              break;

          }
          break;

        default:
          fPassToDef = TRUE;
          break;

      }
      break;

    case WM_COMMAND:
      switch ( COMMANDMSG( &usMsg )->cmd ) {

        case IDD_SEARCH: /* search using path */
        case DID_OK:
          {
            HPOINTER  hptrOld;          /* handle to old cursor       */
            USHORT    iItem;            /* temporary loop variable    */
            HWND      hwndListBox;      /* handle to list box         */
            BOOL      fContinue;        /* boolean search flag        */
            CHAR      szDrive[8];       /* drive specification string */
            CHAR      szPattern[64];    /* search pattern string      */
            CHAR      szFileSpec[64];   /* file specification string  */
      
            /* disable run button & change cursor to hourglass */
            WinEnableWindow( WinWindowFromID( hwnd, IDD_RUN ), FALSE );
            hptrOld = WinQueryPointer( HWND_DESKTOP );
            WinSetPointer(
              HWND_DESKTOP,
              WinQuerySysPointer( HWND_DESKTOP, SPTR_WAIT, FALSE )
            );
        
            /* retrieve search pattern & erase current file list */
            WinQueryDlgItemText(
              hwnd,
              IDD_PATTERN,
              sizeof(szPattern),
              szPattern
            );
            WinSendDlgItemMsg( hwnd, IDD_FILES, LM_DELETEALL, 0L, 0L );

            /* perform search on each selected drive */
            fContinue = TRUE;
            hwndListBox = WinWindowFromID( hwnd, IDD_DRIVES );
            iItem = LIT_FIRST;
            while (
              (iItem = SHORT1FROMMR( WinSendMsg(
                hwndListBox,
                LM_QUERYSELECTION,
                MPFROMSHORT( iItem ),
                0L ) )) != LIT_NONE
              && fContinue
            ) {

              /* define file specification */
              WinSendMsg(
                hwndListBox,
                LM_QUERYITEMTEXT,
                MPFROM2SHORT( iItem, sizeof(szDrive) ),
                MPFROMP( szDrive )
              );
              sprintf( szFileSpec, "%c:\\%s", szDrive[2], szPattern );

              /* create directory listing using pattern */
              fContinue = Directory(
                szFileSpec,
                FILE_NORMAL,
                WinWindowFromID( hwnd, IDD_FILES )
              );
          
            }
        
            /* check for null search operation */
            if ( !WinSendDlgItemMsg(
              hwnd,
              IDD_FILES,
              LM_QUERYITEMCOUNT,
              0L,
              0L
            ) )
              WinMessageBox(
                HWND_DESKTOP,
                hwnd,
                "No files found matching criteria!",
                "File Finder",
                0,
                MB_OK | MB_ICONHAND
              );
          
            /* restore cursor */
            WinSetPointer( HWND_DESKTOP, hptrOld );
          }
          break;

        case IDD_RUN:
          {
            USHORT      iSelection;       /* selected file index      */
            CHAR        szFile[64];       /* name of executable file  */
            CHAR        szDir[64];        /* name of directory        */
            CHAR        szPath[64];       /* full path name of file   */
            CHAR        szFail[64];       /* if DosExecPgm fails      */
            RESULTCODES resc;             /* result of child process  */
            USHORT      usErr;            /* error code               */

            /* get name of selected file */
            iSelection = SHORT1FROMMR( WinSendDlgItemMsg(
              hwnd,
              IDD_FILES,
              LM_QUERYSELECTION,
              MPFROMSHORT( LIT_FIRST ),
              0L
            ) );
            if ( iSelection != LIT_NONE ) {
              WinSendDlgItemMsg(
                hwnd,
                IDD_FILES,
                LM_QUERYITEMTEXT,
                MPFROM2SHORT( iSelection, sizeof(szFile) ),
                MPFROMP( szFile )
              );

              /* define full path name */
              do {
                WinSendDlgItemMsg(
                  hwnd,
                  IDD_FILES,
                  LM_QUERYITEMTEXT,
                  MPFROM2SHORT( --iSelection, sizeof(szDir) ),
                  MPFROMP( szDir )
                );
              } while ( szDir[0] == ' ' );
              sprintf(
                szPath,
                "%s\\%s",
                szDir,
                strtok( &szFile[2], " " )
              );

              /* run it */
              usErr = DosExecPgm(
                szFail,
                sizeof(szFail),
                EXEC_ASYNC,
                NULL,
                NULL,
                &resc,
                szPath
              );
              if ( usErr )
                WinMessageBox(
                  HWND_DESKTOP,
                  hwnd,
                  "Unable to execute file!",
                  "File Finder",
                  0,
                  MB_OK | MB_ICONHAND
                );
              else
                WinSendMsg(
                  hwnd,
                  WM_SYSCOMMAND,
                  MPFROMSHORT( SC_MINIMIZE ),
                  MPFROMSHORT( CMDSRC_OTHER )
                );
            }
          }
          break;

        case IDD_QUIT:
          WinDismissDlg( hwnd, TRUE );
          break;

        case IDM_ABOUT:
          WinDlgBox( HWND_DESKTOP, hwnd, AboutDlgProc, NULL, IDD_ABOUTDLG, NULL );
          break;

        case DID_CANCEL:
          /* don't quit the program when escape key is pressed */
          break;

        default:
          fPassToDef = TRUE;
          break;

      }
      break;

    case WM_MEASUREITEM:
      if ( SHORT1FROMMP( mp1 ) == IDD_FILES ) {

        HPS         hps;                  /* handle to window PS      */
        FONTMETRICS afm[14];              /* info of available fonts  */
        LONG        cfm;                  /* # of available fonts     */
        LONG        i;                    /* font index               */
        LONG        lErr;                 /* error code               */
        BOOL        fErr;                 /* error code               */

        /* find a monospace font for list boxes */
        hps = WinGetPS( hwnd );
        cfm = 14;
        GpiQueryFonts(
          hps,
          QF_PUBLIC,
          "Courier",
          &cfm,
          (LONG)sizeof(afm[0]),
          afm
        );
        for ( i = 0; i < cfm && afm[i].fsDefn & 0x0001; i++ )
          /* skip proportional fonts */;

        /* set up for using it */
        fat.usRecordLength = sizeof(fat);
        fat.fsSelection = afm[i].fsSelection;
        fat.lMatch = afm[i].lMatch;
        strcpy( fat.szFacename, afm[i].szFacename );
        fat.idRegistry = afm[i].idRegistry;
        fat.usCodePage = 850;
        fat.lMaxBaselineExt = afm[i].lMaxBaselineExt;
        fat.lAveCharWidth = afm[i].lAveCharWidth;
        fat.fsType = FATTR_TYPE_FIXED;
        fat.fsFontUse = 0;
        lErr = GpiCreateLogFont( hps, NULL, 1L, &fat );
        fErr = GpiSetCharSet( hps, 1L );

        /* return the height of a listbox item */
        mresRtnVal = MRFROMLONG( fat.lMaxBaselineExt+2 );
        WinReleasePS( hps );

      }
      break;

    case WM_DRAWITEM:
      if ( SHORT1FROMMP( mp1 ) == IDD_FILES ) {

        POWNERITEM  poi;                          /* item to be drawn */
        LONG        lErr;                         /* error code       */
        BOOL        fErr;                         /* error code       */

        /* change the font to our monospace one if it's not already */
        poi = PVOIDFROMMP( mp2 );
        if ( GpiQueryCharSet( poi->hps ) != 1L ) {
          lErr = GpiCreateLogFont( poi->hps, NULL, 1L, &fat );
          fErr = GpiSetCharSet( poi->hps, 1L );
        }

      }
      break;

    case WM_MINMAXFRAME:
      {
        PSWP        pswp;                     /* pos change structure */

        /* hide list box when minimized so it doesn't overwrite icon */
        pswp = PVOIDFROMMP( mp1 );
        WinShowWindow(
          WinWindowFromID( hwnd, IDD_FILES ),
          !(pswp->fs & SWP_MINIMIZE)
        );
      }
      break;

    default:
      fPassToDef = TRUE;
      break;

  } 

  /* pass to def dialog proc if needed */
  if ( fPassToDef )
    mresRtnVal = WinDefDlgProc( hwnd, usMsg, mp1, mp2 );
  
  return mresRtnVal;
}

/**/

/*
 * AboutDlgProc( hwndDlg, usMsg, mp1, mp2 ) : MRESULT;
 *
 *    hwndDlg        handle to dialog box
 *    usMsg          message number
 *    mp1            message parameter 1
 *    mp2            message parameter 2
 *
 * This is the dialog procedure for the About dialog box.
 *
 */

MRESULT EXPENTRY AboutDlgProc( HWND hwndDlg, USHORT usMsg, MPARAM mp1, MPARAM mp2 )
{
  MRESULT       mresRtnVal;                 /* function return value  */
  BOOL          fPassToDef;                 /* pass to def dlg proc?  */

  mresRtnVal = FALSE;
  fPassToDef = FALSE;

  switch ( usMsg ) {

    case WM_INITDLG:
      CenterPopup( hwndDlg, HWNDOWNER( hwndDlg ) );
      break;

    case WM_COMMAND:
      switch ( COMMANDMSG( &usMsg )->cmd ) {
        case DID_OK:
          WinDismissDlg( hwndDlg, TRUE );
          break;
        default:
          fPassToDef = TRUE;
          break;
      }
      break;

    default:
      fPassToDef = TRUE;
      break;

  }

  if ( fPassToDef )
    mresRtnVal = WinDefDlgProc( hwndDlg, usMsg, mp1, mp2 );

  return mresRtnVal;
}

/**/

/*
 * Directory( szFileSpec, fsAttributes, hwndListBox ) : BOOL;
 *
 *    szFileSpec     file search specification
 *    fsAttributes   file attributes to search for
 *    hwndListBox    handle to listbox for directory
 *
 * This function searches the disk for files of the specified type
 * and appends the results found to the list box whose handle is
 * provided.  An error is generated and the process aborted if
 * insufficient memory is available for the search.
 *
 * A value of TRUE is returned if the directory operation was
 * successful.
 *
 */

static BOOL Directory(
  NPSZ        szFileSpec,
  USHORT      fsAttributes,
  HWND        hwndListBox )
{
  BOOL        fContinue;              /* boolean search flag          */
  BOOL        fDirOutput;             /* boolean directory flag       */
  USHORT      iChar;                  /* index into char string       */
  CHAR        szPath[64];             /* path specification string    */
  CHAR        szSpec[64];             /* search specification string  */
  HDIR        hdirSearch;             /* handle to search directory   */
  USHORT      usSearchCount;          /* # of files to search for     */
  FILEFINDBUF findbuf;                /* search result buffer         */
  CHAR        szDate[16];             /* date string                  */
  CHAR        szTime[16];             /* time string                  */
  CHAR        szEntry[64];            /* current file name            */
  SHORT       sErr;                   /* error code                   */

  /* initialization */
  fContinue = TRUE;
  fDirOutput = FALSE;

  /* separate file spec into path and wildcards */
  for ( iChar=strlen(szFileSpec)-1; szFileSpec[iChar]!='\\'; iChar-- );
  strcpy( szPath, szFileSpec );
  szPath[iChar] = '\0';
  strcpy( szSpec, &szFileSpec[iChar+1] );
    
  /* perform search for normal files */
  hdirSearch = HDIR_CREATE;
  usSearchCount = 1;
  sErr = DosFindFirst(
    szFileSpec,
    &hdirSearch,
    fsAttributes,
    &findbuf,
    sizeof(findbuf),
    &usSearchCount,
    0L
  );
  if ( !sErr ) {
  
    /* repeat until all entries exhausted */
    do {
    
      /* output current directory if not done */
      if ( !fDirOutput ) {
        fDirOutput = TRUE;
        sErr = SHORT1FROMMR( WinSendMsg(
          hwndListBox,
          LM_INSERTITEM,
          MPFROMSHORT( LIT_END ),
          szPath
        ) );
        switch ( sErr ) {

          case LIT_MEMERROR:
            WinMessageBox(
              HWND_DESKTOP,
              WinQueryWindow( hwndListBox, QW_PARENT, FALSE ),
              "Insufficient Memory!",
              "File Finder",
              0,
              MB_OK | MB_ICONHAND
            );
            fContinue = FALSE;
            break;

          case LIT_ERROR:
            WinMessageBox(
              HWND_DESKTOP,
              WinQueryWindow( hwndListBox, QW_PARENT, FALSE ),
              "Unable to Add Item to List Box!",
              "File Finder",
              0,
              MB_OK | MB_ICONHAND
            );
            fContinue = FALSE;
            break;

          default:
            break;

        }
      }  

      if ( fContinue ) {

        /* output current file name */
        MakeDateString( findbuf.fdateLastWrite, szDate );
        MakeTimeString( findbuf.ftimeLastWrite, szTime );
        sprintf(
          szEntry, 
          "  %-12s %7ld %s %s",
          findbuf.achName,
          findbuf.cbFile,
          szDate,
          szTime
        );
          
        /* add entry to file list box */
        sErr = SHORT1FROMMR( WinSendMsg(
          hwndListBox,
          LM_INSERTITEM,
          MPFROMSHORT( LIT_END ),
          szEntry
        ) );
        switch ( sErr ) {

          case LIT_MEMERROR:
            WinMessageBox(
              HWND_DESKTOP,
              WinQueryWindow( hwndListBox, QW_PARENT, FALSE ),
              "Insufficient Memory!",
              "File Finder",
              0,
              MB_OK | MB_ICONHAND
            );
            fContinue = FALSE;
            break;

          case LIT_ERROR:
            WinMessageBox(
              HWND_DESKTOP,
              WinQueryWindow( hwndListBox, QW_PARENT, FALSE ),
              "Unable to Add Item to List Box!",
              "File Finder",
              0,
              MB_OK | MB_ICONHAND
            );
            fContinue = FALSE;
            break;

          default:
            break;

        }

      }

      /* get next entry */
      if ( fContinue )
        sErr = DosFindNext(
          hdirSearch,
          &findbuf,
          sizeof(findbuf),
          &usSearchCount
        );
      
    } while ( fContinue && !sErr );
    
  } 
  
  if ( fContinue ) {

    /* perform search for sub-directories */
    sprintf( szEntry, "%s\\*.*", szPath );
    hdirSearch = HDIR_CREATE;
    usSearchCount = 1;
    sErr = DosFindFirst(
      szEntry,
      &hdirSearch,
      FILE_DIRECTORY,
      &findbuf,
      sizeof(findbuf),
      &usSearchCount,
      0L
    );
    if ( !sErr ) {
  
      /* repeat until all entries exhausted */
      do {
    
        /* eliminate special directory entries */
        if ( findbuf.attrFile == FILE_DIRECTORY
        && findbuf.achName[0] != '.' ) {
          sprintf(
            szEntry,
            "%s\\%s\\%s",
            szPath,
            findbuf.achName,
            szSpec
          );
          fContinue = Directory( szEntry, fsAttributes, hwndListBox );
        }
      
        /* get next entry */
        if ( fContinue )
          sErr = DosFindNext(
            hdirSearch,
            &findbuf,
            sizeof(findbuf),
            &usSearchCount
          );

      } while ( fContinue && !sErr );
    
    }   

  }
  
  /* return final result */
  return fContinue;
}

/**/

/*
 * CenterPopup( hwndPopup, hwndCenter ) : VOID;
 *
 *    hwndPopup      handle to popup window to center
 *    hwndCenter     handle to window within which to center
 *
 * This function centers the popup window within the given window.
 * To center within the entire desktop, pass HWND_DESKTOP as the
 * hwndCenter parameter.  If the centered popup would be partially
 * or completely off the edges of the screen, it is moved to be
 * within the borders of the screen.
 *
 */

static VOID CenterPopup(
  HWND        hwndPopup,
  HWND        hwndCenter )
{
  LONG        cxPopup;                            /* popup width      */
  LONG        cyPopup;                            /* popup height     */
  LONG        cxCenter;                           /* parent width     */
  LONG        cyCenter;                           /* parent height    */
  LONG        cxScreen;                           /* screen width     */
  LONG        cyScreen;                           /* screen height    */
  RECTL       rclCenter;                          /* parent rectangle */
  RECTL       rclPopup;                           /* popup rectangle  */

  /* get window rects for popup and center */
  if ( !hwndCenter || (hwndCenter == HWND_DESKTOP) ) {
    rclCenter.xLeft = 0;
    rclCenter.yBottom = 0;
    rclCenter.xRight = WinQuerySysValue( HWND_DESKTOP, SV_CXSCREEN );
    rclCenter.yTop = WinQuerySysValue( HWND_DESKTOP, SV_CYSCREEN );
  } else
    WinQueryWindowRect( hwndCenter, &rclCenter );
  WinQueryWindowRect( hwndPopup, &rclPopup );

  /* convert to screen coordinates */
  MAPWINDOWRECTS( hwndPopup, HWND_DESKTOP, &rclPopup, 1 );
  MAPWINDOWRECTS( hwndCenter, HWND_DESKTOP, &rclCenter, 1 );

  /* calculate window widths */
  cxPopup = ABS(rclPopup.xRight - rclPopup.xLeft);
  cyPopup = ABS(rclPopup.yTop - rclPopup.yBottom);
  cxCenter = ABS(rclCenter.xRight - rclCenter.xLeft);
  cyCenter = ABS(rclCenter.yTop - rclCenter.yBottom);

  /* center popup within parent */
  rclPopup.xLeft = rclCenter.xLeft + (cxCenter-cxPopup)/2;
  rclPopup.yBottom = rclCenter.yBottom + (cyCenter-cyPopup)/2;

  /* adjust position if hanging off on right or left */
  cxScreen = WinQuerySysValue( HWND_DESKTOP, SV_CXSCREEN );
  if ( rclPopup.xLeft+cxPopup >= cxScreen )
    rclPopup.xLeft = (cxScreen - 1) - cxPopup;
  if ( rclPopup.xLeft < 0 )
    rclPopup.xLeft = 0;

  /* adjust position if hanging off on top or bottom */
  cyScreen = WinQuerySysValue( HWND_DESKTOP, SV_CYSCREEN );
  if ( rclPopup.yBottom+cyPopup >= cyScreen )
    rclPopup.yBottom = (cyScreen - 1) - cyPopup;
  if ( rclPopup.yBottom < 0 )
    rclPopup.yBottom = 0;

  /* convert to parent coordinates */
  rclPopup.xRight = rclPopup.yTop = 0;  /* don't care about topright */
  MAPWINDOWRECTS( HWND_DESKTOP, HWNDPARENT( hwndPopup ), &rclPopup, 1 );

  /* place popup window using parent coordinates */
  WinSetWindowPos(
    hwndPopup,
    NULL,
    LOUSHORT( rclPopup.xLeft ),
    LOUSHORT( rclPopup.yBottom ),
    0,
    0,
    SWP_MOVE
  );
}

/**/

/*
 * MakeDateString( fdate, szDate ) : VOID;
 *
 *    fdate          date to format
 *    szDateTime     date string buffer
 *
 * This function formats a given date into a string form,
 * taking into account the current user preference settings.
 *
 */

static VOID MakeDateString(
  FDATE       fdate,
  NPSZ        szDate )
{
  COUNTRYCODE ctryc;                  /* country code struct          */
  COUNTRYINFO ctryi;                  /* country info for date        */
  USHORT      cbCtryInfo;             /* # of bytes returned in ctryi */

  /* get country-specific info */
  ctryc.country = 0;
  ctryc.codepage = 0;
  DosGetCtryInfo( sizeof(ctryi), &ctryc, &ctryi, &cbCtryInfo );
  
  /* format date accordingly */
  sprintf(
    szDate,
    "%2u%s%02u%s%02u",
    ctryi.fsDateFmt == DATEFMT_MM_DD_YY
      ? fdate.month
      : ctryi.fsDateFmt == DATEFMT_DD_MM_YY ? fdate.day : fdate.year+80,
    ctryi.szDateSeparator,
    ctryi.fsDateFmt == DATEFMT_MM_DD_YY ? fdate.day : fdate.month,
    ctryi.szDateSeparator,
    ctryi.fsDateFmt == DATEFMT_YY_MM_DD ? fdate.day : fdate.year+80
  );
}

/**/

/*
 * MakeTimeString( ftime, szTime ) : VOID;
 *
 *    ftime          time to format
 *    szTimeTime     time string buffer
 *
 * This function formats a given time into a string form,
 * taking into account the current user preference settings.
 *
 */

static VOID MakeTimeString(
  FTIME       ftime,
  NPSZ        szTime )
{
  COUNTRYCODE ctryc;                  /* country code struct          */
  COUNTRYINFO ctryi;                  /* country info for time        */
  USHORT      cbCtryInfo;             /* # of bytes returned in ctryi */

  /* get country-specific info */
  ctryc.country = 0;
  ctryc.codepage = 0;
  DosGetCtryInfo( sizeof(ctryi), &ctryc, &ctryi, &cbCtryInfo );
  
  /* format time accordingly */
  if ( ctryi.fsTimeFmt )
    sprintf(
      szTime,
      "%02u%s%02u",
      ftime.hours,
      ctryi.szTimeSeparator,
      ftime.minutes
    );
  else
    sprintf(
      szTime,
      "%2u%s%02u%c",
      ftime.hours > 12
        ? ftime.hours-12
        : ftime.hours ? ftime.hours : 12,
      ctryi.szTimeSeparator,
      ftime.minutes,
      ftime.hours < 12 ? 'a' : 'p'
    );
}
