/* Copyright 1990 GROUPE BULL -- See licence conditions in file COPYRIGHT */
/* XWrPixF.c:
 *
 *  XPM
 *  Write utility for XPM file format
 *  Developed by Arnaud Le Hors
 */

#include "xpmP.h"
#ifdef SYSV
#include <string.h>
#else
#include <strings.h>
#endif

#define MAXPRINTABLE 93			/* number of printable ascii chars 
					   minus \ and " for string compat. */

static char *printable = 
" .XoO+@#$%&*=-;:?>,<1234567890qwertyuipasdfghjklzxcvbnmMNBVCZ\
ASDFGHJKLPIUYTREWQ!~^/()_`'][{}|" ;
					/* printable begin with a space, so
					   in most case, due to my algorythm, 
					   when the number of different colors 
					   is less than MAXPRINTABLE, it will 
					   give a char follow by "nothing" 
					   (a space) in the readable xpm file
					 */

int storePixel(pixel, pixels, indexsize, ncolors, index_return)
Pixel pixel, **pixels;
unsigned int *indexsize, *ncolors, *index_return;
{
    unsigned int a;
    for (a = 0; a < *ncolors; a++)
	if ((*pixels)[a] == pixel) break;
    if (a == *ncolors) {
	if (*ncolors > *indexsize) {
	    Pixel *p;
	    *indexsize *= 2;
	    p = (Pixel *) realloc(*pixels, sizeof(Pixel) * *indexsize);
	    if (!p) return(1);
	    *pixels = p;
	}
	(*pixels)[*ncolors] = pixel;
	(*ncolors)++;
    }
    *index_return = a;
    return 0;
}

#undef RETURN
#define RETURN(status) \
  { if (ximage) XDestroyImage(ximage);\
    if (pixelindex) free(pixelindex); \
    if (pixels) free(pixels); \
    if (xcolors) free(xcolors); \
    if (colorStrings) { \
	for (a = 0; a < ncolors; a++) \
	    if (colorStrings[a]) \
		free(colorStrings[a]); \
	free(colorStrings); \
    } \
    mclose(mdata); \
    return(status); }

int XWritePixmapFile(display, colormap, filename, pixmap, width, 
		     height, type, cppm, infos)
     Display *display;
     Colormap colormap;
     char *filename;
     Pixmap pixmap;
     unsigned int width, height;
     char *type;
     unsigned int cppm;
     XpmInfo *infos;

{ MData *mdata;
  XImage *ximage = NULL;
  unsigned int pixgap = 0;	/* length of pixels gap */
  unsigned int *iptr, *pixelindex = NULL;
  Pixel *pixels = NULL;		/* pixels colors */
  unsigned int indexsize = 256;	/* should be enough most of the time */
  XColor *xcolors = NULL;	/* used colors */
  char **colorStrings = NULL;	/* character strings associated to colors */
  unsigned int ncolors = 0;	/* number of colors */
  unsigned int cpp;             /* chars per pixel */
  char *name;
  unsigned int a, b, c, x, y, n = 0, key;
  Pixel pixel;
  rgb_names rgbn[MAX_RGBNAMES];
  int rgbn_max = 0;
  char *colorname;
  int ErrorStatus;
  Boolean defaultCase;

  if ((ErrorStatus = mwriteopen(filename, &mdata)) != PixmapSuccess)
      return(ErrorStatus);

  if (filename) {
      if (!(name = rindex(filename, '/')))
	  name = filename;
      else
	  name++;
  } else
      name = "pixmap_name";

  /* force output type to the C syntax */
  n = 1;

  /*
   * read the image data
   */

  ximage = XGetImage(display, pixmap, 0, 0, width, height, AllPlanes, ZPixmap);
  defaultCase = False;
  pixelindex = (unsigned int *)malloc(sizeof(unsigned int)*width*height);
  if (!pixelindex) RETURN(PixmapNoMemory);
  pixels = (Pixel *)malloc(sizeof(Pixel) * indexsize);
  if (!pixels) RETURN(PixmapNoMemory);
  switch (ximage->depth) {
  case 8:
      {
	  byte *ip;
	  pixgap = ximage->bytes_per_line - width;
	  ip = (byte *) ximage->data;
	  iptr = pixelindex;
	  for (y = 0; y < height; y++) {
	      for (x = 0; x < width; x++, iptr++) {
		  pixel = (Pixel) *ip++;
		  if (storePixel(pixel, &pixels, &indexsize, &ncolors, iptr))
		      RETURN(PixmapNoMemory);
	      }
	      for (b = 0; b < pixgap; b++) ip++; /* skip extra pixels */
	  }
	  break;
      }
  case 1:
      {
	  byte  *destline, *destptr, destmask, mbyte;
	  int  bperline;
	  bperline = ximage->bytes_per_line;
	  iptr = pixelindex; destline = (byte *)ximage->data;
	  if (ximage->bitmap_bit_order == MSBFirst)
	      for (y = 0; y < height; y++, destline += bperline) {
		  destmask = 0x80; destptr = destline;
		  for (x = 0; x < width; x++, iptr++) {
		      if (*destptr & destmask)
			  mbyte |= 1;
		      else
			  mbyte &= ~1;
		      if (!(destmask >>= 1)) {
			  destmask = 0x80;
			  destptr++;
		      }
		      pixel = (Pixel) mbyte;
		      if (storePixel(pixel, &pixels, &indexsize, 
				     &ncolors, iptr))
			  RETURN(PixmapNoMemory);
		  }
	      }
	  else
	      for (y = 0; y < height; y++, destline += bperline) {
		  destmask = 1; destptr = destline;
		  for (x = 0; x < width; x++, iptr++) {
		      if (*destptr & destmask)
			  mbyte |= 1;
		      else
			  mbyte &= ~1;
		      if (!(destmask <<= 1)) {
			  destmask = 1;
			  destptr++;
		      }
		      pixel = (Pixel) mbyte;
		      if (storePixel(pixel, &pixels, &indexsize, 
				     &ncolors, iptr))
			  RETURN(PixmapNoMemory);
		  }
	      }
	  break;
      }
  case 4:
      {
	  byte *ip, *lip;
	  int  bperline, half;
	  bperline = ximage->bytes_per_line;
	  if (ximage->bits_per_pixel == 4) {
	      pixgap = ximage->bytes_per_line - width/2;
	      iptr = pixelindex; lip = (byte *) ximage->data;
	      for (y = 0; y < height; y++, lip += bperline) {
		  for (x = 0, ip = lip, half = 0; 
		       x < width; x++, iptr++, half++) {
		      if (half & 1)
			  pixel = *ip++>>4 & 0x0f;
		      else
			  pixel = *ip & 0x0f;
		      if (storePixel(pixel, &pixels, &indexsize, 
				     &ncolors, iptr))
			  RETURN(PixmapNoMemory);
		  }
	      for (b = 0; b < pixgap; b++) ip++; /* skip extra pixels */
	      }
	  } else if (ximage->bits_per_pixel == 8) {
	      pixgap = ximage->bytes_per_line - width;
	      iptr = pixelindex; ip = (byte *) ximage->data;
	      for (y = 0; y < height; y++) {
		  for (x = 0; x < width; x++, iptr++) {
		      pixel = (Pixel) *ip++;
		      if (storePixel(pixel, &pixels, &indexsize, 
				     &ncolors, iptr))
			  RETURN(PixmapNoMemory);
		  }
	      for (b = 0; b < pixgap; b++) ip++; /* skip extra pixels */
	      }
	  } else {
	      defaultCase = True;
	  }
	  break;
      }
  case 6:
      {
	  byte *ip;
	  if (ximage->bits_per_pixel == 8) {
	      pixgap = ximage->bytes_per_line - width;
	      ip = (byte *) ximage->data;
	      iptr = pixelindex;
	      for (y = 0; y < height; y++) {
		  for (x = 0; x < width; x++, iptr++) {
		      pixel = (Pixel) *ip++;
		      if (storePixel(pixel, &pixels, &indexsize, 
				     &ncolors, iptr))
			  RETURN(PixmapNoMemory);
		  }
		  for (b = 0; b < pixgap; b++) ip++; /* skip extra pixels */
	      }
	  } else {
	      defaultCase = True;
	  }
	  break;
      }
  case 24:
  case 32:
      {
	  byte *ip;
	  Pixel mpixel;
	  pixgap = ximage->bytes_per_line - 4*width;
	  ip = (byte *) ximage->data;
	  iptr = pixelindex;
	  if (ximage->byte_order == MSBFirst)
	      for (y = 0; y < height; y++) {
		  for (x = 0; x < width; x++, iptr++) {
		      pixel = 0; ip++;	/* *ip should be 0 */
		      pixel |= *ip++; pixel <<= 8;
		      pixel |= *ip++; pixel <<= 8;
		      pixel |= *ip++;
		      if (storePixel(pixel, &pixels, &indexsize, 
				     &ncolors, iptr))
			  RETURN(PixmapNoMemory);
		  }
		  for (b = 0; b < pixgap; b++) ip++; /* skip extra pixels */
	      }
	  else
	      for (y = 0; y < height; y++) {
		  for (x = 0; x < width; x++, iptr++) {
		      pixel = *ip++;
		      mpixel = *ip++;
		      pixel |= mpixel<<8;
		      mpixel = *ip++;
		      pixel |= mpixel<<16;
		      ip++;		/* *ip should be 0 */
		      if (storePixel(pixel, &pixels, &indexsize, 
				     &ncolors, iptr))
			  RETURN(PixmapNoMemory);
		  }
		  for (b = 0; b < pixgap; b++) ip++; /* skip extra pixels */
	      }
	  break;
      }
  default:
      defaultCase = True;
  }
  if (defaultCase) {
      iptr = pixelindex;
      for (y = 0; y < height; y++)
	  for (x = 0; x < width; x++, iptr++) {
	      pixel = XGetPixel(ximage, x, y);
	      if (storePixel(pixel, &pixels, &indexsize, &ncolors, iptr))
		  RETURN(PixmapNoMemory);
	  }
  }

  /* get rgb values and a string of char for each color
   */
  xcolors = (XColor *) malloc(sizeof(XColor) * ncolors);
  if (!xcolors) RETURN(PixmapNoMemory);
  colorStrings = (char **) calloc(ncolors, sizeof(char *));
  if (!colorStrings) RETURN(PixmapNoMemory);

  for (cpp = 1, c = MAXPRINTABLE; ncolors > c; cpp++) c *= MAXPRINTABLE;
  if (cpp < cppm) cpp = cppm;

  for (a = 0; a < ncolors; a++) {
      if (! (colorStrings[a] = (char *)malloc(cpp)))
	  RETURN(PixmapNoMemory);
      *colorStrings[a] = printable[c = a % MAXPRINTABLE];
      for (b = 1; b < cpp; b++)
	  colorStrings[a][b] = printable[c= ((a-c)/MAXPRINTABLE)%MAXPRINTABLE];
      xcolors[a].pixel = pixels[a];
  }
  XQueryColors(display, colormap, xcolors, ncolors);

  /*
   * read the rgb file if any was specified
   */
  if (infos && infos->rgb_fname)
      rgbn_max = read_rgb_names(infos->rgb_fname, rgbn);

  /*
   * print the pixmap file
   */

  /* print the header line 
   */
  fprintf(mdata->stream.file, "%s XPM %s", DataTypes[n].Bcmt,
	  DataTypes[n].Ecmt);
  if (n != 0)			/* print the assignment line */
      fprintf(mdata->stream.file, "%s %s %s", DataTypes[n].Dec, name, 
	      DataTypes[n].Boa);

  /* print the hints 
   */
  if (infos && infos->hints_cmt) /* print hints comment line */
      fprintf(mdata->stream.file, "%s%s%s", DataTypes[n].Bcmt, 
	      infos->hints_cmt, DataTypes[n].Ecmt);

  if (DataTypes[n].Bos) fprintf(mdata->stream.file, "%c", DataTypes[n].Bos);
  fprintf(mdata->stream.file, "%d %d %d %d", width, height, ncolors, cpp);
  if (DataTypes[n].Eos) fprintf(mdata->stream.file, "%c", DataTypes[n].Eos);
  fprintf(mdata->stream.file, DataTypes[n].Strs);

  /* print colors 
   */
  if (infos && infos->colors_cmt) /* print colors comment line */
      fprintf(mdata->stream.file, "%s%s%s", DataTypes[n].Bcmt, 
	      infos->colors_cmt, DataTypes[n].Ecmt);

  for (a = 0; a < ncolors; a++) {
      if (DataTypes[n].Bos)
	  fprintf(mdata->stream.file, "%c", DataTypes[n].Bos);
      for (b = 0; b < cpp; b++)
	  fprintf(mdata->stream.file, "%c", colorStrings[a][b]);
      c = 1;
      if (infos && infos->pixels) {
	  for (b = 0; b < infos->ncolors; b++)
	      if (infos->pixels[b] == xcolors[a].pixel) break;
	  if (b != infos->ncolors) {
	      c = 0;
	      for (key = 1; key < NKEYS + 1; key++) {
		  if (infos->colorTable[b][key])
		      fprintf(mdata->stream.file, " %s %s", 
			      ColorKeys[key - 1], infos->colorTable[b][key]);
	      }
	  }
      }
      if (c) {
	  colorname = NULL;
	  if (rgbn_max) 
	      colorname = get_color_name(rgbn, rgbn_max, xcolors[a].red, 
					 xcolors[a].green, xcolors[a].blue);
	  if (colorname)
	      fprintf(mdata->stream.file, " c %s", colorname);
	  else
	      fprintf(mdata->stream.file, " c #%04X%04X%04X", 
		      xcolors[a].red, xcolors[a].green, xcolors[a].blue);
      }
      if (DataTypes[n].Eos)
	  fprintf(mdata->stream.file, "%c", DataTypes[n].Eos);
      fprintf(mdata->stream.file, DataTypes[n].Strs);
  }

  /* print pixels
   */
  if (infos && infos->pixels_cmt) /* print pixels comment line */
      fprintf(mdata->stream.file,  "%s%s%s", DataTypes[n].Bcmt, 
	      infos->pixels_cmt, DataTypes[n].Ecmt);

  iptr = pixelindex;
  for (y = 0; y < height; y++) {
      if (DataTypes[n].Bos)
	  fprintf(mdata->stream.file, "%c", DataTypes[n].Bos);
      for (x = 0; x < width; x++, iptr++)
	  for (b = 0; b < cpp; b++)
	      fprintf(mdata->stream.file, "%c", colorStrings[*iptr][b]);
      if (DataTypes[n].Eos)
	  fprintf(mdata->stream.file, "%c", DataTypes[n].Eos);
      if (y < height - 1) 
	  fprintf(mdata->stream.file, DataTypes[n].Strs);
  }
  fprintf(mdata->stream.file, DataTypes[n].Eoa);

  free_rgbn(rgbn, rgbn_max);

  RETURN(PixmapSuccess);
}
