/* 1405, Fri 16 Oct 92

   FLOWTREE.C:  AU Meter, using a height-balanced tree

   Copyright (C) 1992 by Nevil Brownlee,
   Computer Centre,  University of Auckland */

#define DEBUG
#define noDECNET

#include <stdio.h>
#include <alloc.h>
#include <mem.h>

#ifdef DEBUG
#include <dos.h>
#endif

#include "pktsnap.h"

#define EXTFLOW
#include "flowtree.h"
#include "decnet.h"

#define keyp(n)  &n->k.addr_type

#define keycpy(d,s)   addrcpy(\
   (unsigned char far *)d,(unsigned char far *)s, sizeof(struct key) )

#ifdef DEBUG
void paddr(unsigned char far *a);
void daddr(unsigned char far *a);
void pkey(char *msg,node_p n);
#endif

FILE *log;

#define FBLKSZ 500
#define BBLKSZ 50

node_p fl;
struct f_node far *fa;  int nfnodes;
node_p bl;
struct b_node far *ba;  int nbnodes;

node_p getnode(t)  /* Get node from free list, or get a new one */
int t;
{
   node_p q;
   if (t == N_FLOW) {
      if (fl != NULL) {
	 q = fl;  fl = fl->h.rlink;
	 }
      else {
	 if (nfnodes == 0) {
	    fa = farmalloc(sizeof(struct f_node)*FBLKSZ);
	    if (fa == NULL) {
	       printf("No memory for new flows!\n");
	       return (node_p)NULL;
	       }
	    nfnodes = FBLKSZ-1;
	    }
	 else {
	    ++fa;  --nfnodes;
	    }
	 q = (node_p)fa;
	 }
      }
   else if (t == N_BACK_FLOW) {
      if (bl != NULL) {
	 q = bl;  bl = bl->h.rlink;
	 }
      else {
	 if (nbnodes == 0) {
	    ba = farmalloc(sizeof(struct b_node)*BBLKSZ);
	    if (ba == NULL) {
	       printf("No memory for new back_flows!\n");
	       return (node_p)NULL;
	       }
	    nbnodes = BBLKSZ-1;
	    }
	 else {
	    ++ba;  --nbnodes;
	    }
	 q = (node_p)ba;
	 }
      }
   return q;
   }

void freenode(q)  /* Add node to head of free list */
node_p q;
{
   if (q->h.type == N_FLOW) {
      q->h.rlink = fl;
      fl = q;
      }
   else {
      q->h.rlink = bl;
      bl = q;
      }
   }

node_p new_tree()  /* Initialise a new tree */
{
   node_p q;
   if ((q = getnode(N_FLOW)) != NULL) {
      q->h.llink = q->h.rlink = NULL;
      q->h.balance = 0;  q->h.type = N_ROOT;
      q->i.r.height = q->i.r.nbr_nodes =
	 q->i.r.compares = q->i.r.searches = 0;
      }
   return q;
   }

node_p new_node(t)  /* Space for new node, type t */
int t;
{
   node_p q;
   if ((q = getnode(t)) != NULL) {
      q->h.llink = q->h.rlink = NULL;
      q->h.balance = 0;  q->h.type = t;
      }
   return q;
   }

/* Height-balanced tree algorithm from Knuth (1973) Vol 3 */

node_p hbtree(node_p n, int (*compfn)(node_p n, node_p p),
   int type, node_p (*add_new_node)(int type, node_p n, node_p tree),
   node_p tree);

#define link(a,p)  (a == -1 ? p->h.llink : p->h.rlink)
#define setlink(a,p, v)  if (a == -1) p->h.llink = v; else p->h.rlink = v

node_p hbtree(n,compfn, type,add_new_node, tree)  /* Find node n in tree */
node_p n;
int (*compfn)(node_p n, node_p p);  /* Compare two nodes */
int type;  /* Type of node to add */
node_p (*add_new_node)  /* Add new node, initialise it */
   (int type, node_p n, node_p tree);
node_p tree;
{
   node_p t,  /* Father of s */
      s,  /* Place where rebalancing may be neccessary */
      p, q,r;
   char a;
   t = tree;  /* A1: Initialise */
   s = p = t->h.rlink;
   tree->i.r.searches += 1;
   if (p == NULL) {  /* Empty tree */
      if ((q = add_new_node(type,n,tree)) == NULL) return q;
      tree->h.rlink = q;
      tree->i.r.nbr_nodes = tree->i.r.height = 1;
      return q;
      }
   for (;;) {  /* A2: Compare */
      tree->i.r.compares += 1;
      if ((a = compfn(n,p)) == 0) return p;  /* Success */
      if (a < 0) {
	 if ((q = p->h.llink) == NULL) {  /* A3: Move left */
	    if ((q = add_new_node(type,n,tree)) == NULL) return q;
	    p->h.llink = q;
	    break;
	    }
	 }
      else {  /* A4: Move right */
	 if ((q = p->h.rlink) == NULL) {
	    if ((q = add_new_node(type,n,tree)) == NULL) return q;
	    p->h.rlink = q;
	    break;
	    }
	 }
      if (q->h.balance != 0) {
	 t = p;  s = q;
	 }
      p = q;
      }
   tree->i.r.nbr_nodes += 1;  /* A5: Insert */
   tree->i.r.compares += 1;
   if (compfn(n,s) < 0)  /* A6: Adjust balance factors */
      r = p = s->h.llink;
   else
      r = p = s->h.rlink;
   while (p != q) {
      tree->i.r.compares += 1;
      if (compfn(n,p) < 0) {
	 p->h.balance = -1;  p = p->h.llink;
	 }
      else {
	 p->h.balance = 1;  p = p->h.rlink;
	 }
      }
   tree->i.r.compares += 1;
   a = compfn(n,s) < 0 ? -1 : 1;  /* A7: Balancing act */
   if (s->h.balance == 0) {  /* Tree has grown higher */
      s->h.balance = a;
      tree->i.r.height +=1;
      return q;
      }
   if (s->h.balance == -a) {  /* Tree balance has improved */
      s->h.balance = 0;
      return q;
      }
   /* Tree has become unbalanced */
   if (r->h.balance == a) {  /* A8: Single rotation */
      p = r;
      setlink(a,s, link(-a,r));
      setlink(-a,r, s);
      s->h.balance = r->h.balance = 0;
      }
   else {  /* A9: Double rotation */
      p = link(-a,r);
      setlink(-a,r, link(a,p));
      setlink(a,p, r);
      setlink(a,s, link(-a,p));
      setlink(-a,p, s);
      if (p->h.balance == a) {
	 s->h.balance = -a;  r->h.balance = 0;
	 }
      else if (p->h.balance == 0) {
	 s->h.balance = r->h.balance = 0;
	 }
      else {
	 s->h.balance = 0;  r->h.balance = a;
	 }
      p->h.balance = 0;
      }
   if (s == t->h.rlink) t->h.rlink = p;  /* A10: Finishing touch */
   else t->h.llink = p;
   return q;
   }

void d_tree_walk(t,d, f)  /* Symmetric walk, d = current depth */
node_p t;
int d;
void (*f)();
{
   if (t->h.llink) d_tree_walk(t->h.llink,d+1, f);
   f(t,d);
   if (t->h.rlink) d_tree_walk(t->h.rlink,d+1, f);
   }

void tree_walk(t, f)  /* Depth-first walk of tree */
node_p t;
void (*f)();
{
   if (t->h.llink) tree_walk(t->h.llink, f);
   if (t->h.rlink) tree_walk(t->h.rlink, f);
   f(t);
   }

int log_nbr = 1;

void open_log()
{
   char fn[30];
   sprintf(fn,"flows.%03d",log_nbr++);
   log = fopen(fn, "w");
   }

#ifdef DEBUG
void paddr(a)
unsigned char far *a;
{
   fprintf(log,"%d.%d.%d.%d ", a[0],a[1],a[2],a[3]);
   }

void daddr(a)
unsigned char far *a;
{
   printf("%d.%d.%d.%d ", a[0],a[1],a[2],a[3]);
   }

void pkey(msg,n)
char *msg;
node_p n;
{
   if (!log) open_log();
   fprintf(log,"%s: %04x:%04x ",msg,FP_SEG(n),FP_OFF(n));
   paddr(n->k.from_value); paddr(n->k.from_mask); paddr(n->k.from_tally);
   paddr(n->k.to_value); paddr(n->k.to_mask); paddr(n->k.to_tally);
   fprintf(log,"\n");
   }
#endif

int compfn(n,p)
node_p n;  node_p p;
{
   unsigned char j, d,v;
   if (n->k.addr_type < p->k.addr_type) return -1;
   if (n->k.addr_type > p->k.addr_type) return 1;
   for (j = 0;  j != ADDR_LEN;  ++j) {
      d = n->k.from_value[j] & p->k.from_mask[j];
      v = p->k.from_value[j];
      if (d < v) return -1;
      if (d > v) return 1;
      }
   for (j = 0;  j != ADDR_LEN;  ++j) {
      d = n->k.to_value[j] & p->k.to_mask[j];
      v = p->k.to_value[j];
      if (d < v) return -1;
      if (d > v) return 1;
      }
   return 0;
   }

int tally_compfn(n,p)
node_p n;  node_p p;
{
   unsigned char j, d,v;
   if (n->k.addr_type < p->k.addr_type) return -1;
   if (n->k.addr_type > p->k.addr_type) return 1;
   for (j = 0;  j != ADDR_LEN;  ++j) {
      d = n->k.from_value[j] & p->k.from_tally[j];
      v = p->k.from_value[j];
      if (d < v) return -1;
      if (d > v) return 1;
      }
   for (j = 0;  j != ADDR_LEN;  ++j) {
      d = n->k.to_value[j] & p->k.to_tally[j];
      v = p->k.to_value[j];
      if (d < v) return -1;
      if (d > v) return 1;
      }
   return 0;
   }

unsigned char new_node_added;

node_p no_flow_node(type,n,tree)  /* Never makes new flow */
int type;
node_p n;  node_p tree;
{
   return NULL;
   }

node_p new_flow_node(type,n,tree)  /* Always makes new flow, */
int type;                          /*    using masks from n */
node_p n;  node_p tree;
{
   unsigned char j, d,v;
   node_p q;
   if ((q = new_node(type)) == NULL) return NULL;
   keycpy(keyp(q),keyp(n));
   new_node_added = 1;
   return q;
   }

node_p tally_flow_node(type,n,tree)  /* Only makes new tally flows */
int type;
node_p n;  node_p tree;
{
   unsigned char j, d,v;
   node_p t,q;
   t = hbtree(n,tally_compfn, N_FLOW,no_flow_node, tree);
   if (t == NULL) return NULL;  /* No matching tally flow in tree */
   if ((q = new_node(type)) == NULL) {
      scpos(0,24);  /* !!! */
      printf("Failed to get new node, type %d\n", type);
      return NULL;
      }
   keycpy(keyp(q),keyp(t));  /* Copy masks and addr_type from tally flow */
   for (j = 0; j != ADDR_LEN; ++j) {  /* Now copy the values from n */
      q->k.from_value[j] = n->k.from_value[j] & t->k.from_mask[j];
      q->k.to_value[j] = n->k.to_value[j] & t->k.to_mask[j];
      }
   new_node_added = 1;
   return q;
   }

node_p data;  /* Work space for building tree nodes */

node_p rule_table;

unsigned char null_addr[ADDR_LEN] = {0,0,0,0};

void init_rule_table()
{
   crp = (struct node *)&current_rule;
   }

void add_rule()
{
   node_p rp, rule;
   rule = getnode(N_BACK_FLOW);
   addrcpy((unsigned char far *)rule, (unsigned char far *)&current_rule,
      sizeof(current_rule));
   rule->h.rlink = NULL;
   if (nrules == 0) rule_table = rule;
   else {
      for (rp = rule_table; rp->h.rlink != NULL; rp = rp->h.rlink) ;
      rp->h.rlink = rule;
      }
   ++nrules;
   }

node_p find_rule(n)  /* n is 1-org */
int n;
{
   node_p rp;  int j;
   if (nrules == 0 || n < 1 || n > nrules) return NULL;
   for (rp = rule_table, j = 1;  j != n;  rp = rp->h.rlink, ++j) ;
   return rp;
   }

void clear_rule_table()
{
   node_p lrp, rp;
   rp = rule_table;
   while (rp != NULL) {
      lrp = rp;  rp = rp->h.rlink;
      freenode(lrp);
      }
   nrules = 0;
   }

struct flow_seed {  /* Default rules */
   unsigned char tree_nbr;
   struct key k;
   };

struct flow_seed flow_info[] = {
   {0, AT_IP,        0,0,0,0,  255,255,255,0,   0,0,0,0,
		     0,0,0,0,  255,255,255,0,   0,0,0,0},
   {0, AT_NOVELL,    0,0,0,0,  255,255,255,255, 0,0,0,0,
		     0,0,0,0,  255,255,255,255, 0,0,0,0},
   {0, AT_DECNET,    0,0,0,0,  255,0,0,0,       0,0,0,0,
		     0,0,0,0,  255,0,0,0,       0,0,0,0},
   {0, AT_ETHERTALK, 0,0,0,0,  255,255,255,0,   0,0,0,0,
		     0,0,0,0,  255,255,255,0,   0,0,0,0},
   {0, AT_IGNORE},
   };

void add_flow(node, tree_nbr)
node_p node;
unsigned char tree_nbr;
{
   unsigned char j, s_addr[ADDR_LEN];
   node_p tree, f;

   for (j = 0; j != ADDR_LEN; ++j) {
      if (node->k.from_value[j] != node->k.to_value[j]) {
	 if (node->k.from_value[j] > node->k.to_value[j]) {
	    addrcpy(s_addr,node->k.from_value,ADDR_LEN);  /* Swap addresses, */
	    addrcpy(node->k.from_value,node->k.to_value,ADDR_LEN);
	    addrcpy(node->k.to_value,s_addr,ADDR_LEN);
	    addrcpy(s_addr,node->k.from_mask,ADDR_LEN);   /* masks */
	    addrcpy(node->k.from_mask,node->k.to_mask,ADDR_LEN);
	    addrcpy(node->k.to_mask,s_addr,ADDR_LEN);
	    addrcpy(s_addr,node->k.from_tally,ADDR_LEN);  /* and tallies */
	    addrcpy(node->k.from_tally,node->k.to_tally,ADDR_LEN);
	    addrcpy(node->k.to_tally,s_addr,ADDR_LEN);
	    }
	 break;
	 }
      }
   if ((tree = forest[tree_nbr]) == NULL)
      tree = forest[tree_nbr] = new_tree();
   f = hbtree(node,compfn, N_FLOW,new_flow_node, tree);
   if (f != NULL) init_flow(tree,f);
   }

void setup_flows()
{
   node_p rp, tree, f;
   struct flow_seed *fp;

   if (nrules == 0) {  /* Use the default rule table */
      for (fp = flow_info;  fp->k.addr_type != AT_IGNORE;  ++fp) {
	 keycpy(keyp(data),(unsigned char far *)&fp->k);
	 add_flow(data, fp->tree_nbr);
	 }
      }
   else {  /* Manager has set up some rules */
      for (rp = rule_table;  rp != NULL;  rp = rp->h.rlink) {
	 keycpy(keyp(data),(unsigned char far *)&rp->k);
	 add_flow(data, rp->i.tree_nbr);
	 }
      }
   }

void init_flow(tree,f)
node_p tree;  node_p f;
{
   number_flow(f);
   f->i.f.fwd_packets = f->i.f.fwd_bytes =  /* Initialise flow */
      f->i.f.back_packets = f->i.f.back_bytes =
   f->i.f.s_fwd_packets = f->i.f.s_fwd_bytes =
      f->i.f.s_back_packets = f->i.f.s_back_bytes = 0;
   f->i.f.s_active = 0;
   }

node_p flow_ix[MXFLOWS];  /* Flow_nbr index */

void number_flow(n)  /* Allocate (and remember) a flow_nbr */
node_p n;
{
   if (nflows == MXFLOWS)
      printf("Too many flows!\n");
   else {
      flow_ix[nflows] = n;
      n->i.f.flow_nbr = nflows++;  /* 0-org */
      }
   }

void free_flow(n)  /* Forget a numbered flow */
node_p n;
{
   flow_ix[n->i.f.flow_nbr] = NULL;
   }

node_p find_flow(x)  /* Find flow with flow_nbr x */
int x;
{
   if (x >= MXFLOWS || x < 0) {
      printf("Invalid flow number!\n");
      return NULL;
      }
   return flow_ix[x];
   }

void t_clear_flows(t)
node_p t;
{
   if (t->h.type == N_FLOW) free_flow(t);
   freenode(t);
   }

void clear_flows()
{
   int j;  node_p tree;
   for (j = 0;  (tree = forest[j]) != NULL;  ++j) {
      tree_walk(tree->h.rlink, t_clear_flows);
      freenode(tree);  forest[j] = NULL;
      }
   nflows = 0;  /* Start again from scratch */
   }


#define MAXPKTLEN  1526

#ifdef DECNET
int dn_node(dn_addr)
unsigned char far *dn_addr;
{
   return dn_addr[1]<<8 | dn_addr[0];
   }

#define dn_area(node) (node >> 10)
#define dn_host(node) (node & 0x3FF)
#endif

void unpack_dn_node(ap, dn_addr)
unsigned char far *ap;
unsigned char far *dn_addr;
{
   unsigned char dl = dn_addr[1];
   ap[0] = dl >> 2;    /* DECnet area */
   ap[1] = dl & 0x03;  ap[2] = dn_addr[0];  /* DECnet host */
   ap[3] = 0;
   }

void pkt_monitor(pp,len,type)
unsigned char far *pp;
int len;
unsigned char type;
{
   unsigned char j, dtype,
      high_to_low, s_addr[ADDR_LEN];
   union decnet far *dp;
   node_p tree, n,f;
#ifdef DECNET
   unsigned int dlen, snode, dnode, area, host;
#endif
   if (type == AT_IP) {
      data->k.addr_type = AT_IP;
      addrcpy(data->k.from_value,&pp[26-SNAPFROM],ADDR_LEN);  /* IP source */
      addrcpy(data->k.to_value,&pp[30-SNAPFROM],ADDR_LEN);  /* IP dest */
      }
   else if (type == AT_NOVELL) {
      data->k.addr_type = AT_NOVELL;
      addrcpy(data->k.from_value,&pp[20-SNAPFROM],ADDR_LEN);  /* IPX soure net */
      addrcpy(data->k.to_value,&pp[32-SNAPFROM],ADDR_LEN);  /* IPX dest net */
      }
   else if (type == AT_DECNET) {
      data->k.addr_type = AT_DECNET;
      addrcpy(data->k.to_value,null_addr,ADDR_LEN);
      dtype = pp[16-SNAPFROM];  /* DECnet packet type */
      if (dtype != 0x81) dp = (union decnet far *)&pp[17-SNAPFROM];
      else {
	 dtype = pp[17-SNAPFROM];
	 dp = (union decnet far *)&pp[18-SNAPFROM];
	 }
      switch (dtype) {
      case 0x07:  /* Level 1 routing */
	 unpack_dn_node(data->k.from_value, dp->l1r.src_dn_addr);
#ifdef DECNET
	 snode = dn_node(dp->l1r.src_dn_addr);
	 fprintf(log,"Level 1 routing from %d.%d\n",
	    dn_area(snode),dn_host(snode));
#endif
	 break;
      case 0x09:  /* Level 2 routing */
	 unpack_dn_node(data->k.from_value, dp->l2r.src_dn_addr);
#ifdef DECNET
	 snode = dn_node(dp->l2r.src_dn_addr);
	 fprintf(log,"Level 2 routing from %d.%d\n",
	    dn_area(snode),dn_host(snode));
#endif
	 break;
      case 0x0B:  /* Router hello */
	 unpack_dn_node(data->k.from_value, dp->rh.src_dn_addr);
#ifdef DECNET
	 snode = dn_node(dp->rh.src_dn_addr);
	 dnode = dn_node(dp->rh.rtr_dn_addr);
	 fprintf(log,"Router hello from %d.%d, other router %d.%d\n",
	    dn_area(snode),dn_host(snode),
	    dn_area(dnode),dn_host(dnode) );
#endif
	 break;
      case 0x0D:  /* Endnode hello */
	 unpack_dn_node(data->k.from_value, dp->eh.src_dn_addr);
#ifdef DECNET
	 snode = dn_node(dp->eh.src_dn_addr);
	 dnode = dn_node(dp->eh.rtr_dn_addr);
	 fprintf(log,"Endnode hello from %d.%d, designated router %d.%d\n",
	    dn_area(snode),dn_host(snode),
	    dn_area(dnode),dn_host(dnode) );
#endif
	 break;
      case 0x06:  /* Don't know what 06 or 0E are, but */
      case 0x0E:  /*    they both look like data packets! */
      case 0x26:  /* Data */
      case 0x2E:  /* Data */
	 unpack_dn_node(data->k.from_value, dp->d.src_dn_addr);
	 unpack_dn_node(data->k.to_value, dp->d.dest_dn_addr);
#ifdef DECNET
	 dnode = dn_node(dp->d.dest_dn_addr);
	 snode = dn_node(dp->d.src_dn_addr);
	 fprintf(log,"Data to %d.%d from %d.%d\n",
	    dn_area(dnode),dn_host(dnode),
	    dn_area(snode),dn_host(snode) );
#endif
	 break;
      default:  /* Unknown DECnet type */
	 scpos(0,24);
	 printf("\nDN pkt type %02x: ", dtype);
	 for (j=17; j != 34; ++j) fprintf(log," %02x",pp[j-SNAPFROM]);
	 printf("\n");
	 return;  /* Ignore it */
	 }
      }

   high_to_low = 0;
   for (j = 0; j != ADDR_LEN; ++j) {
      if (data->k.from_value[j] != data->k.to_value[j]) {
	 if (data->k.from_value[j] > data->k.to_value[j]) {
	    high_to_low = 1;
	    addrcpy(s_addr,data->k.from_value,ADDR_LEN);  /* Swap addresses */
	    addrcpy(data->k.from_value,data->k.to_value,ADDR_LEN);
	    addrcpy(data->k.to_value,s_addr,ADDR_LEN);
	    }
	 break;
	 }
      }

   new_node_added = 0;
   for (j = 0;  (tree = forest[j]) != NULL;  ++j) {

      n = hbtree(data,compfn, N_FLOW,tally_flow_node, tree);
      if (n == NULL) continue;  /* Try next tree */

      if (new_node_added) init_flow(tree,n);
      if (high_to_low) {
	 n->i.f.back_packets += 1;  n->i.f.back_bytes += len;
	 }
      else {
	 n->i.f.fwd_packets += 1;  n->i.f.fwd_bytes += len;
	 }
      return;  /* Matched and counted */

      }  /* Ignore packet if not matched in forest */
   }

void save_counts()
{
   int j;  node_p t;
   p_s_chktime = s_chktime;
   s_chktime = uptime();
   s_nflows = nflows;
   s_act_flows = s_first_flow = 0;
   for (j = 0;  j != nflows;  ++j) {
      if ((t = find_flow(j)) != NULL) {
	 t->i.f.s_active =
	    t->i.f.fwd_bytes != t->i.f.s_fwd_bytes ||
	    t->i.f.back_bytes != t->i.f.s_back_bytes;
	 if (t->i.f.s_active) {
	    if (s_act_flows == 0) s_first_flow = j;
	    ++s_act_flows;
	    t->i.f.s_fwd_packets = t->i.f.fwd_packets;
	    t->i.f.s_fwd_bytes = t->i.f.fwd_bytes;
	    t->i.f.s_back_packets = t->i.f.back_packets;
	    t->i.f.s_back_bytes = t->i.f.back_bytes;
	    }
	 }
      }
   s_tod_h = tod_h;  s_tod_m = tod_m;  s_tod_s = tod_s;
   elapsed_sec = 0;
   }

void write_flows()
{
   int j;
   node_p t;
   if (!log) open_log();
   for (j = 0; j != nflows; ++j) {
      if ((t = find_flow(j)) == NULL) continue;
      fprintf(log, "%3d %1d %3u.%3u.%3u->%3u.%3u.%3u %6lu %8lu %6lu %8lu\n",
	 t->i.f.flow_nbr, t->k.addr_type,
	 t->k.from_value[0],t->k.from_value[1],t->k.from_value[2],
	 t->k.to_value[0],t->k.to_value[1],t->k.to_value[2],
	 t->i.f.fwd_packets, t->i.f.fwd_bytes,
	 t->i.f.back_packets, t->i.f.back_bytes);
      }
   fclose(log);  log = NULL;
   }

void t_write_log(t,d)
node_p t;
int d;
{
   if (t->h.type == N_BACK_FLOW) return;
   fprintf(log, "%d %d  %d  %3u.%3u.%3u%3u -> %3u.%3u.%3u.%3u\n",
      d, t->h.type, t->k.addr_type,
      t->k.from_value[0],t->k.from_value[1],t->k.from_value[2],t->k.from_value[3],
      t->k.to_value[0],t->k.to_value[1],t->k.to_value[2],t->k.to_value[3]);
   }

void write_log()
{
   int j;  node_p tree;
   if (!log) open_log();
   for (j = 0;  (tree = forest[j]) != NULL;  ++j) {
      fprintf(log, "Tree %d: %u nodes, %u lvls, %lu cmps/srch\n\n",
	 j, tree->i.r.nbr_nodes,tree->i.r.height,
	 tree->i.r.compares/tree->i.r.searches);
      d_tree_walk(tree->h.rlink,1, t_write_log);
      fprintf(log,"\n");
      }
   fclose(log);  log = NULL;
   }

void tree_stats(searches,compares)
unsigned long *searches;
unsigned long *compares;
{
   int j;  node_p tree;
   unsigned long s,c;
   s = c = 0;
   for (j = 0;  (tree = forest[j]) != NULL;  ++j) {
      s += tree->i.r.searches;
      c += tree->i.r.compares;
      }
   *searches = s;  *compares = c;
   }

void show_stats(t)
node_p t;
{
   unsigned long s,c;
   tree_stats(&s,&c);
   scpos(0,24);
   printf("Stats: %u flows, %lu cmps/srch",
      nflows, c/s);
   }

void show_time()
{
   scpos(0,24);
   printf("%lu seconds since %02d%02d:%02d",
      elapsed_sec, s_tod_h,s_tod_m,s_tod_s);
   }

void init_monitor()
{
   data = getnode(N_BACK_FLOW);  data->h.type = N_VOID;
   init_rule_table();
   setup_flows();
   boot_time = biosticks();
   }

void show_help()
{
   w_clear(0,7, 40,24);  /* Clear window */
   scpos(0,7);
   printf("Copyright (C) 1992 by Nevil Brownlee\n");
   printf("Computer Centre, University of Auckland\n\n");
   printf("Keyboard commands ..\n\n");
   printf("  b: show Bad packet counts\n");
   printf("  m: show Memory usage\n");
   printf("  s: show Statistics\n");
   printf("  v: show meter Version\n");
   if (kb_enabled) printf("\nEsc: stop metering, exit to DOS\n\n");
   }

void handle_kb(ch)
int ch;
{
   switch (tolower(ch)) {
   case 's':
      show_stats(forest[0]);
      break;
   case 't':
      show_time();
      break;
   case 'w':
      if (kb_enabled) write_log();
      break;
   case 'x':
      if (kb_enabled) write_flows();
      break;
   case 'z':
      if (kb_enabled) save_counts();
      break;
   case '?':
      show_help();
      break;
   default:
      break;
      }
   }
