#include "kernel.h"
#include <minix/com.h>
#include "ether.h"

/* #define DEBUG 1 */

#if (ETHERNET == ETH_3C509)

/* I/O macros - port must within the current window (see setwin()) */
#define inb(port)		in_byte(io_addr+port)
#define inw(port)		in_word(io_addr+port)
#define outb(port, value)	out_byte(io_addr+port, value)
#define outw(port, value)	out_word(io_addr+port, value)

/* LOCAL VARIABLES */
PRIVATE unsigned id_port = 0x110;	/* board initialization port */
   /* if you have problems, try other values: 0x100, 0x120, 0x130 ... 0x1f0 */
PRIVATE unsigned addr_config;	/* address configuration */
PRIVATE unsigned res_config;	/* resource configuration */
PRIVATE unsigned io_addr;	/* port */
PRIVATE unsigned int_no;	/* irq */
PRIVATE unsigned is_bnc;	/* 1 -> BNC transceiver */
PRIVATE unsigned is_tp;		/* 1 -> Twisted Pair transceiver */
#define ETHER_BUFF_LEN ((6+6+2)+2)	/* must be even number of words */
PRIVATE unsigned char ether_buff[ETHER_BUFF_LEN];  /* buffer for ethernet header */
  
/* LOCAL FUNCTIONS */
FORWARD void setwin();
FORWARD void checksum();
FORWARD void write_id_pat();
FORWARD void read_eaddr();
FORWARD unsigned read_ee();
FORWARD void delay275();
FORWARD void notfound();

/*===========================================================================*
 *				recv_pkt				     * 
 *===========================================================================*/
PUBLIC int recv_pkt()
{
/* Receive packet routine.
 * Returns OK if packet received.
 */
  int res;
  res = EAGAIN;		/* process can't get frame whenever it wants */
  return res;		/* it has to wait for one (i.e. block in driver) */
}


/*===========================================================================*
 *				send_pkt				     * 
 *===========================================================================*/
PUBLIC int send_pkt(pkt_buf, pkt_length)
phys_bytes pkt_buf;	/* packet buffer */
vir_bytes pkt_length;	/* packet length */
{
/* Transmit packet routine.
 * Returns OK if packet sent or error code otherwise.
 */
  unsigned val, length4, nwords, txfree;
  int cnt;
  phys_bytes dst_addr;

  if (pkt_length > GIANT)	/* is this packet too large? */
    return(EFBIG);
  if (pkt_length < RUNT)	/* is this frame long enough? */
    pkt_length=RUNT;		/* stretch frame to minimum allowed */
  
  val = inb(PORT_TxStatus);	/* get the previous transmit status */
  if (val & (TXS_UNDERRUN | TXS_JABBERERROR))	/* underrun or jabber error? */
    outw(PORT_CmdStatus, CMD_TXRESET);		/* reset transmitter */
  if (val & TXS_ERRTYPE)	/* any errors at all? */    
    outw(PORT_CmdStatus, CMD_TXENABLE);		/* re-enable transmitter */
    
#if DEBUG
printf("\r\nsend_pkt: TxStatus = %02x (previous transmit status)\r\n", val);
#endif
    
  length4 = (pkt_length + 3) & ~3;	/* round up to dwords */
  
#if DEBUG
printf("pkt_length = %d, length4 = %d\r\n", pkt_length, length4);
#endif

  for(cnt = 18 ; cnt > 0 ; cnt--)	/* wait for enough bytes in transmit buffer */
  {
    if((txfree=inw(PORT_TxFree)) >= length4)
      break;
    delay275(); 
  }
  if (cnt == 0)		/* can't send frame now, try later */
    return(EAGAIN); 
    
#if DEBUG
printf("timeout loop terminated at cnt = %d, TxFree = %d\r\n", cnt, txfree);
#endif

  outw(PORT_TxFIFO, pkt_length);	/* output the count (no interrupts) */
  outw(PORT_TxFIFO, pkt_length);	/* output the second reserved word */
  
  /* now output the rest of the frame (word by word) */
  dst_addr=numap(ETHER, (vir_bytes) &val, (vir_bytes) 2);
  nwords = (pkt_length + 1) >> 1;	/* number of full data words */
  
#if DEBUG
printf("nwords = %d\r\n", nwords);  
#endif

  while(nwords--)
  {
    phys_copy(pkt_buf, dst_addr, (phys_bytes) 2);	/* copy one word */
    pkt_buf += 2;
    outw(PORT_TxFIFO, val);
  }
  
  nwords = (pkt_length + 1) >> 1;	/* number of full data words */
  if (nwords % 2)		/* odd number of words? */
    outw(PORT_TxFIFO, 0);	/* insert dummy word into transmit buffer */

  return OK;
}

/*===========================================================================*
 *				reset_board				     * 
 *===========================================================================*/
PUBLIC void reset_board()
{
  outw(PORT_CmdStatus, CMD_GLOBALRESET);
}


/*===========================================================================*
 *				recv					     * 
 *===========================================================================*/
PUBLIC void recv()
{
  unsigned val, size, nwords;
  unsigned pkt_type, pkt_class;
  int cnt, loop;
  phys_bytes src_addr, dst_addr, stored_dst_addr;
  
  for(loop = 1 ; ; ++loop)
  {
    val = inw(PORT_CmdStatus);
    outw(PORT_CmdStatus, val | CMD_ACKNOWLEDGE);
#if DEBUG  
printf("\r\nrecv (loop = %d): CmdStatus=%04x\r\n", loop, val);
#endif
    if ((val & INT_RXCOMPLETE) == 0)
      break;
    size = inw(PORT_RxStatus);	/* get frame size and error flags */
#if DEBUG  
printf("RxStatus=%04x, frm size=%d bytes, error=%04x, error type=%04x\r\n",
       size, size & RXS_LENGTH, size & RXS_ERROR, size & (RXS_ERRTYPE));
#endif
    if (size & RXS_ERROR)	/* any errors? */
    {
      if (size & RXS_ERRTYPE != RXS_DRIBBLE) 
        goto _discard; /* can only discard bad frame */
      /* 3Com says DRIBBLE is just a warning, so let's belive him */      
    }
    size &= RXS_LENGTH;		/* get the real size */
  
    /* read header */
    for(cnt = 0 ; cnt < ETHER_BUFF_LEN ; cnt+=2)
      *((unsigned *) (ether_buff + cnt)) = inw(PORT_RxFIFO); 
    
#if DEBUG
  printf("DA: %02x:%02x:%02x:%02x:%02x:%02x   SA: %02x:%02x:%02x:%02x:%02x:%02x\r\n",
         ether_buff[0],  ether_buff[1],  ether_buff[2], 
         ether_buff[3],  ether_buff[4],  ether_buff[5], 
         ether_buff[6],  ether_buff[7],  ether_buff[8], 
         ether_buff[9],  ether_buff[10], ether_buff[11] );
  printf("ETHERNET_II TYPE: %02x%02x   IEEE802.3 TYPE: %02x%02x\r\n",
         ether_buff[12],  ether_buff[13], ether_buff[14], ether_buff[15] );
#endif  

    /* determine packet type and class */
    pkt_type = ((unsigned)ether_buff[12] << 8) + ether_buff[13];
    pkt_class = BLUEBOOK;			
#if ALLOW_IEEE8023  
    if (pkt_type <= 1500)
    {
      pkt_type = ((unsigned)ether_buff[14] << 8) + ether_buff[15];
      pkt_class = IEEE8023;			
    }
#endif
  
    /* see if anybody wants this frame */
    dst_addr = recv_find((unsigned) size, 
		         (unsigned) pkt_type,
		         (unsigned) pkt_class);
    stored_dst_addr = dst_addr;  /* store it (original value is needed later) */
  
    if (dst_addr != DISCARD_PKT && dst_addr != LEAVE_PKT)   /* copy frame? */
    {
      /* first copy header */
      src_addr = numap(ETHER, (vir_bytes) ether_buff, (vir_bytes) ETHER_BUFF_LEN);
      phys_copy(src_addr, dst_addr, (phys_bytes) ETHER_BUFF_LEN);
      dst_addr += ETHER_BUFF_LEN;
      size -= ETHER_BUFF_LEN;
      
      /* now copy the rest of the frame */
      src_addr = numap(ETHER, (vir_bytes) &val, (vir_bytes) 2);
      nwords = (size + 1) >> 1;
      while(nwords--)
      {
        val = inw(PORT_RxFIFO);
        phys_copy(src_addr, dst_addr, (phys_bytes) 2);	/* copy one word */
        dst_addr += 2;
      }
      nwords = (size + 1) >> 1;
      if (nwords % 2)	/* odd number of words? */
        (void) inw(PORT_RxFIFO);	/* get (and discard) the pad word */
      
      size += ETHER_BUFF_LEN;		/* restore real size */
      
      /* tell them that frame was copied */
      recv_copy(stored_dst_addr, size);
    }
  
_discard:
    /* discard unwanted or bad frame */
    outw(PORT_CmdStatus, CMD_RXDISCARD);
    while(inw(PORT_CmdStatus) & ST_BUSY)	/* wait for command to finish */
      ;
  } /* and go to next loop */
}


/*===========================================================================*
 *				etopen					     * 
 *===========================================================================*/
PUBLIC void etopen()		/* initialize interface */
{
  unsigned val;
  int cnt;
  
  id_port &= 0x1F0;	/* only these bits can be used */
  write_id_pat();
  out_byte(id_port, ID_GLOBAL_RESET);	/* reset adapter */
  delay275();
  write_id_pat();
  out_byte(id_port, SET_TAG_REGISTER);
  checksum(); 		/* check if board is present */
  read_eaddr();		/* get ethernet address */
  
  /* get address and resource configurations */
  addr_config = read_ee(EE_ADDR_CONFIGURATION);	
  res_config = read_ee(EE_RESOURCE_CONFIGURATION);
  out_byte(id_port, ACTIVATE_VULCAN);	/* activate the board */
  
  io_addr = addr_config;
  io_addr &= 0x1F;
  io_addr <<= 4;
  io_addr += MIN_IO_BASE_ADDR;
  if (io_addr != ETHER_PORT)
  {
    printf("WARNING: 3C509 uses port number %x rather than %x.\r\n",
           io_addr, ETHER_PORT);
    printf("         You should redefine ETHER_PORT.\r\n");
  }           
  
  val = addr_config & (BNC_XCVR | TP_XCVR | AUI_XCVR);
  if (val = BNC_XCVR)	/* is it BNC transceiver? */
    is_bnc = 1;
  if (val = TP_XCVR)	/* is it Twisted Pair transceiver? */
    is_tp = 1;
    
  int_no = (res_config >> 12);
  if (int_no != ETHER_IRQ)
  {
    printf("FATAL: 3C509 uses irq %d, not %d", int_no, ETHER_IRQ);
    panic("You must redefine ETHER_IRQ and ETHER_MASK", NO_NUM);
  } 
  
#if DEBUG
  printf("io_addr = %x, int_no = %d\r\n", io_addr, int_no);
  printf("is_bnc = %d, is_tp = %d\r\n", is_bnc, is_tp);
#endif 
  
  setwin(WNO_DIAGNOSTICS);   	/* select window 1 */
  val = inw(PORT_MediaStatus);   
  val |= (MEDIA_LBEATENABLE | MEDIA_JABBERENABLE);
  outw(PORT_MediaStatus, val);   
  
  setwin(WNO_SETUP);   		/* select window 0 */
  
  outw(PORT_CmdStatus, CMD_TXRESET);		/* reset the transmitter */
  outw(PORT_CmdStatus, CMD_TXENABLE);		/* enable the transmitter */
  outw(PORT_CmdStatus, CMD_RXENABLE);		/* enable the receiver */
  
  /* enable RX interrupts */
  outw(PORT_CmdStatus, CMD_SETINTMASK + INT_RXCOMPLETE);
  
  /* enable all the status bits */ 
  outw(PORT_CmdStatus, CMD_SETRZMASK + 0xFE);
  
  /* start transmitting after this many bytes */
  outw(PORT_CmdStatus, CMD_SETTXSTART + 0);	
  
  /* receive after this many bytes */
  outw(PORT_CmdStatus, CMD_SETRXEARLY + RXEARLY_DISABLED);	
  
  if (is_bnc)	/* coax? */
  {
    outw(PORT_CmdStatus, CMD_STARTINTXCVR);	/* start internal transceiver */
    delay275();
  }
  
  outb(PORT_CfgControl, ENABLE_ADAPTER);	/* enable the adapter */ 
  
  /* receive individual address & broadcast */
  outw(PORT_CmdStatus, CMD_SETRXFILTER + FILTER_INDIVIDUAL + FILTER_BROADCAST);
  
  /* set "my own" address */
  setwin(WNO_STATIONADDRESS);
  for(cnt=0; cnt < 6; cnt++)
    outb(PORT_SA0_1 + cnt, curr_hw_addr.ea[cnt]);
  
  /* ready to operate */
  setwin(WNO_OPERATING); 
}


/*===========================================================================*
 *				checksum				     * 
 *===========================================================================*/
PRIVATE void checksum()	/* check 3C509's checksum */
{
  unsigned retval, addr, cnt, cksum;
  unsigned char lo, hi;
  
#if DEBUG  
printf("id_port = %x, 3C509 checksum:\r\n", id_port);
#endif
  
  cksum = 0;
  for(cnt = 16, addr = 0 ; cnt ; cnt--, addr++)
  {
    retval = read_ee(addr);
#if DEBUG  
printf("  loop %d  -> res = %x, chsum before = %x\r\n", cnt, retval, cksum);
#endif
    switch(addr)
    {
      case 3: if (retval & 0xF0FF != 0x9050) notfound(); 
      	      goto _default;
      case 7: if (retval != 0x6D50) notfound();
      	      goto _default;
      case 8:
      case 9:
      case 13:
               lo = retval & BYTE;
               hi = (retval >> 8) & BYTE;
               lo ^= hi;
               hi = 0; 
               retval = ((unsigned)hi << 8) + lo;
               break;
      case 15: 
               break;
      default:
     _default: 
               lo = retval & BYTE;
               hi = (retval >> 8) & BYTE;
               hi ^= lo; 
               lo = 0;
               retval = ((unsigned)hi << 8) + lo;
     	       break;
    }
    cksum ^= retval;
  }
  if (cksum != 0) notfound();
#if DEBUG  
printf("checksum ok...\r\n");
#endif
}


/*===========================================================================*
 *				write_id_pat				     * 
 *===========================================================================*/
PRIVATE void notfound()
{
  panic("3C509 board not found", NO_NUM);
}


/*===========================================================================*
 *				write_id_pat				     * 
 *===========================================================================*/
PRIVATE void write_id_pat()	/* write the 3C509 pattern to the ID port */
{
  register unsigned pat, cnt;
  out_byte(id_port, 0);		/* select the ID port */
  out_byte(id_port, 0);		/* reset hardware pattern generator */
  pat = 0xFF;
  cnt = 0xFF; 
#if DEBUG  
printf("pattern writing: ");
#endif
  for ( ; cnt > 0 ; cnt-- )
    {
      out_byte(id_port, pat);
#if DEBUG  
printf("%x ", pat);
#endif
      pat <<= 1;
      if (pat & 0x0100) /* carry? */
      {
        pat ^= 0xCF;	/* XOR */
        pat &= 0xFF;	/* only low byte */
      }
    }
#if DEBUG  
printf("\r\n");
#endif
}


/*===========================================================================*
 *				read_eaddr				     * 
 *===========================================================================*/
PRIVATE void read_eaddr()	/* read board's address from EEPROM */
{
  unsigned u;
  u = read_ee(EE_TCOM_NODE_ADDR_WORD0);
  curr_hw_addr.ea[0] = (u >> 8) & BYTE;
  curr_hw_addr.ea[1] = u & BYTE;
  u = read_ee(EE_TCOM_NODE_ADDR_WORD1);
  curr_hw_addr.ea[2] = (u >> 8) & BYTE;
  curr_hw_addr.ea[3] = u & BYTE;
  u = read_ee(EE_TCOM_NODE_ADDR_WORD2);
  curr_hw_addr.ea[4] = (u >> 8) & BYTE;
  curr_hw_addr.ea[5] = u & BYTE;
}


/*===========================================================================*
 *				read_ee					     * 
 *===========================================================================*/
PRIVATE unsigned read_ee(addr)	/* read EEPROM */
unsigned addr;	/* EEPROM address to read */
{
  register unsigned result, i;
  int cnt;
  
#if DEBUG  
printf("READING %x: ",addr);
#endif
  addr |= READ_EEPROM;
  out_byte(id_port, addr);
  delay275();
  result = 0;
  for(cnt = 16 ; cnt > 0 ; cnt--) 
   {
      result <<= 1;	/* shift left */
      if ((i=in_byte(id_port)) & 0x01)	/* LSbit set ? */
        result |= 1;	/* then put it into result */
#if DEBUG  
printf("%x ", i);
#endif
   }
#if DEBUG  
   printf("\r\n");
#endif
  return result;
}


/*===========================================================================*
 *				delay275				     * 
 *===========================================================================*/
PRIVATE void delay275()	/* delay 27.5 ms (or maybe more?) */
{
  unsigned d;
  for(d = 50000; d--; );
}


/*===========================================================================*
 *				setwin					     * 
 *===========================================================================*/
PRIVATE void setwin(win)	/* set current window */
unsigned win;
{
  outw(PORT_CmdStatus, CMD_SELECTWINDOW + win); 
}

#endif /* ETHERNET - no code after this line ! */
