-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: parseprocnet.p
-- Author: Andy Lowry
-- SCCS Info: @(#)parseprocnet.p	1.3 3/11/91

-- Process to parse a tokenized command line and produce structures
-- describing a network of communicating processes that can be invoked
-- by the plumber utility.  The command line is assumed nonempty.

parseProcNet:
using (pshell, tokenize, plumbing, load)
linking (parseProc, parsePipe)

process (Q: parseProcNetQ)
  
declare
  args: parseProcNet;
  procs: procInfoSet;
  labels: procLabels;
  pipes: pipeInfoSet;
  waitingPipes: pipeInfoSet;
  pipe: pipeInfo;
  noProc: procID;
  source: procID;
  dest: procID;
  state: topState;
  stack: parseStack;
  info: parseInfo;
  toks: tokenList;
  tok: token;
  delim: char;
  parseProc: parseProcFn;
  parsePipe: parsePipeFn;
  pathLoad: load_func;

begin
  receive args from Q;
  
  -- get program loader capability
  unwrap pathLoad from args.rm.get("pathLoad", "") {init};

  -- Assume caller should wait for processes to return
  args.asynchronous <- 'false';

  -- Link to processes we'll be needing
  parseProc <- procedure of process parseProc;
  parsePipe <- procedure of process parsePipe;

  -- clear state
  new procs;
  new labels;
  new pipes;
  new waitingPipes;
  noProc <- unique;
  source := noProc;
  dest := noProc;
  state <- 'proc';
  new stack;
  toks := args.cmd;
  remove tok from toks[0];

  -- enter the state machine to parse the command line
  while state <> 'done' repeat
    if case of tok = 'delimiter' then-- get delimiter char if any
      reveal tok.delimiter;
      delim := tok.delimiter;
    else
      delim := ' ';
    end if;
    select state
    where ('proc')
      -- Here we're expecting either a process specification or a push
      -- or pop.  Anything else is a syntax error.
      select
      where (case of tok = 'word')
	-- This must be a process specification... go get it
	insert copy of tok into toks at 0;
	block begin
	  dest <- parseProc(toks, procs, labels);
	on (parseProc.syntaxError)
	  exit syntaxError;
	end block;
	-- resolve all pipes waiting for a destination
	while size of waitingPipes <> 0 repeat
	  remove pipe from waitingPipes[];
	  pipe.dest := dest;
	  insert pipe into pipes;
	end while;
	-- move on to next token and move to 'pipe' state, setting
	-- newly parsed proc as source for following pipes
	block begin
	  remove tok from toks[0];
	  source := dest;
	  state <- 'pipe';
	on (notFound)
	  state <- 'done';
	end block;
	
      where (case of tok = 'delimiter' and delim = '<')
	-- Push parse info and continue in same state, consuming token
	new info;
	info.pushedFrom <- 'proc';
	info.source := noProc;
	info.pipes <- waitingPipes;
	insert info into stack;
	new waitingPipes;
	block begin
	  remove tok from toks[0];
	on (notFound)
	  exit syntaxError;
	end block;
	
      where (case of tok = 'delimiter' and delim = '>')
        -- pop parse info from prior push, merging pushed pipes 
        block begin
	  remove info from stack[size of stack - 1];
	on (notFound)
	  exit syntaxError;
	end block;
	if (info.pushedFrom <> 'proc') then
	  exit syntaxError;
	end if;
	merge info.pipes into waitingPipes;
	-- advance to next token
	block begin
	  remove tok from toks[0];
	on (notFound)
	  exit syntaxError;
	end block;
	
      otherwise
	exit syntaxError;
      end select;
      
    where ('pipe')
      -- Here we can parse a pipe specification, a push or pop, or a
      -- semicolon or ampersand to end the pipeline.
      select
      where (case of tok = 'delimiter' and delim = '|')
	-- Go parse a pipe specification... the pipe parser assumes
	-- the vertical bar has already been stripped, so we needn't
	-- replace it
	block begin
	  pipe <- parsePipe(toks);
	on (parsePipe.syntaxError)
	  exit syntaxError; 
	end block;
	-- Fill in source and leave destination unspecified... the
	-- new pipe is still waiting for its dest
	pipe.source <- source;
	source := noProc;
	pipe.dest := noProc;
	insert pipe into waitingPipes;
	-- Advance to next token and switch state.  No more tokens
	-- is a syntax error
	block begin
	  remove tok from toks[0];
	  state <- 'proc';
	on (notFound)
	  exit syntaxError;
	end block;
	
      where (case of tok = 'delimiter' and delim = '<')
	-- push the current parse information and consume the token
	new info;
	info.pushedFrom <- 'pipe';
	info.source := source;
	new info.pipes;
	insert info into stack;
	block begin
	  remove tok from toks[0];
	on (notFound)
	  exit syntaxError;
	end block;
	
      where (case of tok = 'delimiter' and delim = '>')
	-- pop parse information... no waiting pipes to restore
	block begin
	  remove info from stack[size of stack - 1];
	on (notFound)
	  exit syntaxError;
	end block;
	if info.pushedFrom <> 'pipe' then
	  exit syntaxError;
	end if;
	source := info.source;
	-- Advance token... if none, we're done
	block begin
	  remove tok from toks[0];
	on (notFound)
	  state <- 'done';
	end block;
	
      where (case of tok = 'delimiter' and delim = ';')
	-- Here to start a fresh pipeline, which may be connected to
	-- other pipelines in the network via the use of labels in
	-- process specifications
	source := noProc;
	-- consume the token and advance to 'proc' state to get
	-- another process specification
	block begin
	  remove tok from toks[0];
	  state <- 'proc';
	on (notFound)
	  exit syntaxError;
	end block;

	where (case of tok = 'delimiter' and delim = '&')
	  -- Tie off the pipeline and set the asynchronous for the
	  -- caller
	  args.asynchronous <- 'true';
	  state <- 'done';
	      
      otherwise
	exit syntaxError;
      end select;
      
    otherwise
      exit cantHappen;
    end select;
  end while;
  
  -- Here when we finally reached the 'done' state... the token list
  -- must be empty, else we have a syntax error  
  if size of toks <> 0 then
    exit syntaxError;
  end if;
  
  -- Make sure all the process labels eventually got defined with
  -- process specifications
  if exists of label in labels where (not label.defined) then
    exit syntaxError;
  end if;
  
  -- Configure the process network from the parsed info
  block declare
    pnode: procNode;
    arc: pipeArc;
    namedPipe: namedPipe;
  begin
    -- configure each parsed procedure
    new args.procs;
    for proc in procs[] inspect
      new pnode;
      pnode.id := proc.id;
      block begin
	pnode.proc <- create of pathLoad(proc.name);
      on (others)
	args.failer := proc.name;
	return args exception procFailure;
	exit done;
      end block;
      pnode.argv := proc.args;
      pnode.terminal := args.terminal;
      new pnode.inputs;
      new pnode.outputs;
      for pipeInfo in pipes where (pipeInfo.source = proc.id) inspect
	new namedPipe;
	namedPipe.name := pipeInfo.sourceName;
	namedPipe.id := pipeInfo.id;
	block begin
	  insert namedPipe into pnode.outputs;
	on (duplicateKey)
	  exit syntaxError;
	end block;
      end for;
      for pipeInfo in pipes where (pipeInfo.dest = proc.id) inspect
	new namedPipe;
	namedPipe.name := pipeInfo.destName;
	namedPipe.id := pipeInfo.id;
	block begin
	  insert namedPipe into pnode.inputs;
	on (duplicateKey)
	  exit syntaxError;
	end block;
      end for;
      insert pnode into args.procs;
    end for;

    -- Configure all the pipes
    new args.pipes;
    for pipeInfo in pipes[] inspect
      new arc;
      arc.id := pipeInfo.id;
      arc.source := pipeInfo.source;
      arc.sink := pipeInfo.dest;
      insert arc into args.pipes;
    end for;
  end block;

  return args;
  
on exit(syntaxError)
  return args exception syntaxError;
  
on exit (cantHappen)
  print charString#"Internal error in parseProcNet";
  
on exit (done)
  
end process
